/****************************************************************************
**
*W  objfgelm.c                  GAP source                       Frank Celler
**
*
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This  file contains the  C functions for free  group elements.  There are
**  basically three different (internal) types: 8  bits, 16 bits, and 32 bits
**  for  the generator/exponent  pairs.   The  number of  bits  used for  the
**  generator numbers is determined by the number of free group generators in
**  the family.  Any object of this type looks like:
**
**    +------+-----+-------+-------+-----+-----------+
**    | TYPE | len | g1/e1 | g2/e2 | ... | glen/elen | 
**    +------+-----+-------+-------+-----+-----------+
**
**  where  <len> is a GAP integer  object and <gi>/<ei> occupies 8, 16, or 32
**  bits depending on the type.  A TYPE of such an objects looks like:
**
**    +--------+-------+--------+----------+-------+------+------+
**    | FAMILY | FLAGS | SHARED | PURETYPE | EBITS | RANK | BITS |
**    +--------+-------+--------+----------+-------+------+------+
**
**  <PURETYPE> is the kind  of a result, <EBITS>  is the number of  bits used
**  for the exponent, <RANK> is number of free group generators.  But instead
**  of accessing these entries directly you should use the following macros.
**
**
**  The file "objects.h" defines the following macros:
**
**  NEW_WORD( <result>, <kind>, <npairs> )
**    creates   a  new  objects   of   kind  <kind>  with  room  for <npairs>
**    generator/exponent pairs.
**
**  RESIZE_WORD( <word>, <npairs> ) 
**    resizes the  <word> such that  it will hold <npairs> generator/exponent
**    pairs.
**
**
**  BITS_WORD( <word> )
**    returns the number of bits as C integers
**
**  DATA_WORD( <word> )
**    returns a pointer to the beginning of the data area
**
**  EBITS_WORD( <word> )
**    returns the ebits as C integer
**
**  NPAIRS_WORD( <word> )
**    returns the number of pairs as C integer
**
**  RANK_WORD( <word> )
**    returns the rank as C integer
**
**  PURETYPE_WORD( <word> )
**    returns the result kind
**
**
**  BITS_WORDTYPE( <kind> )
**    returns the number of bits as C integers
**
**  EBITS_WORDTYPE( <kind> )
**    returns the ebits as C integer
**
**  RANK_WORDTYPE( <kind> )
**    returns the rank as C integer
**
**  PURETYPE_WORDTYPE( <kind> )
**    returns the result kind
*/
#include        <assert.h>              /* assert                          */
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */
#include        "ariths.h"              /* arithmetic macros               */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "bool.h"                /* booleans                        */

#include        "objfgelm.h"            /* objects of free groups          */


/****************************************************************************
**

*F * * * * * * * * * * * * * * * * 8 bits words * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  Func8Bits_Equal( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func8Bits_Equal (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt1 *     pl;             /* data area in <l>                        */
    libGAP_UInt1 *     pr;             /* data area in <r>                        */

    /* if <l> or <r> is the identity it is easy                            */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( nl != nr ) {
        return libGAP_False;
    }

    /* compare the generator/exponent pairs                                */
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
    pr = (libGAP_UInt1*)libGAP_DATA_WORD(r);
    for ( ;  0 < nl;  nl--, pl++, pr++ ) {
        if ( *pl != *pr ) {
            return libGAP_False;
        }
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  Func8Bits_ExponentSums3( <self>, <obj>, <start>, <end> )
*/
libGAP_Obj libGAP_Func8Bits_ExponentSums3 (
    libGAP_Obj         self,
    libGAP_Obj         obj,
    libGAP_Obj         vstart,
    libGAP_Obj         vend )
{
    libGAP_Int         start;          /* the lowest generator number             */
    libGAP_Int         end;            /* the highest generator number            */
    libGAP_Obj         sums;           /* result, the exponent sums               */
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         pos;            /* current generator number                */
    libGAP_Int         exp;            /* current exponent                        */
    libGAP_UInt1 *     ptr;            /* pointer into the data area of <obj>     */

    /* <start> must be positive                                            */
    while ( !libGAP_IS_INTOBJ(vstart) || libGAP_INT_INTOBJ(vstart) <= 0 )
        vstart = libGAP_ErrorReturnObj( "<start> must be a positive integer", 0L, 0L,
                                 "you can replace <start> via 'return <start>;'" );
    start = libGAP_INT_INTOBJ(vstart);

    /* <end> must be positive                                              */
    while ( !libGAP_IS_INTOBJ(vend) || libGAP_INT_INTOBJ(vend) <= 0 )
        vend = libGAP_ErrorReturnObj( "<end> must be a positive integer", 0L, 0L,
                               "you can replace <end> via 'return <end>;'" );
    end = libGAP_INT_INTOBJ(vend);

    /* <end> must be at least <start>                                      */
    if ( end < start ) {
        sums = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, 0 );
        libGAP_SET_LEN_PLIST( sums, 0 );
        return sums;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(obj);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the number of gen/exp pairs                                     */
    num = libGAP_NPAIRS_WORD(obj);

    /* create the zero vector                                              */
    sums = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, end-start+1 );
    libGAP_SET_LEN_PLIST( sums, end-start+1 );
    for ( i = start;  i <= end;  i++ )
        libGAP_SET_ELM_PLIST( sums, i-start+1, 0 );

    /* and unpack <obj> into <sums>                                        */
    ptr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {
        pos = ((*ptr) >> ebits)+1;
        if ( start <= pos && pos <= end ) {
            if ( (*ptr) & exps )
                exp = ((*ptr)&expm)-exps;
            else
                exp = (*ptr)&expm;

            /* this will not cause a garbage collection                    */
            exp = exp + (libGAP_Int) libGAP_ELM_PLIST( sums, pos-start+1 );
            libGAP_SET_ELM_PLIST( sums, pos-start+1, (libGAP_Obj) exp );
            assert( ptr == (libGAP_UInt1*)libGAP_DATA_WORD(obj) + (i-1) );
        }
    }

    /* convert integers into values                                        */
    for ( i = start;  i <= end;  i++ ) {
        exp = (libGAP_Int) libGAP_ELM_PLIST( sums, i-start+1 );
        libGAP_SET_ELM_PLIST( sums, i-start+1, libGAP_INTOBJ_INT(exp) );
    }

    /* return the exponent sums                                            */
    return sums;
}


/****************************************************************************
**
*F  Func8Bits_ExponentSums1( <self>, <obj> )
*/
libGAP_Obj libGAP_Func8Bits_ExponentSums1 (
    libGAP_Obj         self,
    libGAP_Obj         obj )
{
    return libGAP_Func8Bits_ExponentSums3( self, obj,
        libGAP_INTOBJ_INT(1), libGAP_INTOBJ_INT(libGAP_RANK_WORD(obj)) );
}


/****************************************************************************
**
*F  Func8Bits_ExponentSyllable( <self>, <w>, <i> )
*/
libGAP_Obj libGAP_Func8Bits_ExponentSyllable (
    libGAP_Obj         self,
    libGAP_Obj         w,
    libGAP_Obj         vi )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* integer corresponding to <vi>           */
    libGAP_UInt1       p;              /* <i>th syllable                          */

    /* check <i>                                                           */
    num = libGAP_NPAIRS_WORD(w);
    while ( !libGAP_IS_INTOBJ(vi) || libGAP_INT_INTOBJ(vi) <= 0 || num < libGAP_INT_INTOBJ(vi) )
        vi = libGAP_ErrorReturnObj( "<i> must be an integer between 1 and %d", num, 0L,
                             "you can replace <i> via 'return <i>;'" );
    i = libGAP_INT_INTOBJ(vi);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(w);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* return the <i> th exponent                                          */
    p = ((libGAP_UInt1*)libGAP_DATA_WORD(w))[i-1];
    if ( p & exps )
        return libGAP_INTOBJ_INT((p&expm)-exps);
    else
        return libGAP_INTOBJ_INT(p&expm);
}


/****************************************************************************
**
*F  Func8Bits_ExtRepOfObj( <self>, <obj> )
*/
libGAP_Obj libGAP_Func8Bits_ExtRepOfObj (
    libGAP_Obj         self,
    libGAP_Obj         obj )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Obj         kind;           /* kind of <obj>                           */
    libGAP_UInt1 *     ptr;            /* pointer into the data area of <obj>     */
    libGAP_Obj         lst;            /* result                                  */

    /* get the kind of <obj>                                               */
    kind = libGAP_TYPE_DATOBJ(obj);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the number of gen/exp pairs                                     */
    num = libGAP_NPAIRS_WORD(obj);

    /* construct a list with 2*<num> entries                               */
    lst = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*num );
    libGAP_SET_LEN_PLIST( lst, 2*num );

    /* and unpack <obj> into <lst>                                         */
    ptr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);

    /* this will not cause a garbage collection                            */
    for ( i = 1;  i <= num;  i++, ptr++ ) {
        libGAP_SET_ELM_PLIST( lst, 2*i-1, libGAP_INTOBJ_INT(((*ptr) >> ebits)+1) );
        if ( (*ptr) & exps )
            libGAP_SET_ELM_PLIST( lst, 2*i, libGAP_INTOBJ_INT(((*ptr)&expm)-exps) );
        else
            libGAP_SET_ELM_PLIST( lst, 2*i, libGAP_INTOBJ_INT((*ptr)&expm) );
        assert( ptr == (libGAP_UInt1*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(lst);

    /* return the gen/exp list                                             */
    return lst;
}


/****************************************************************************
**
*F  Func8Bits_GeneratorSyllable( <self>, <w>, <i> )
*/
libGAP_Obj libGAP_Func8Bits_GeneratorSyllable (
    libGAP_Obj         self,
    libGAP_Obj         w,
    libGAP_Obj         vi )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* integer corresponding to <vi>           */
    libGAP_UInt1       p;              /* <i>th syllable                          */

    /* check <i>                                                           */
    num = libGAP_NPAIRS_WORD(w);
    while ( !libGAP_IS_INTOBJ(vi) || libGAP_INT_INTOBJ(vi) <= 0 || num < libGAP_INT_INTOBJ(vi) )
        vi = libGAP_ErrorReturnObj( "<i> must be an integer between 1 and %d", num, 0L,
                             "you can replace <i> via 'return <i>;'" );
    i = libGAP_INT_INTOBJ(vi);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(w);

    /* return the <i> th generator                                         */
    p = ((libGAP_UInt1*)libGAP_DATA_WORD(w))[i-1];
    return libGAP_INTOBJ_INT((p >> ebits)+1);
}


/****************************************************************************
**
*F  Func8Bits_HeadByNumber( <self>, <l>, <gen> )
*/
libGAP_Obj libGAP_Func8Bits_HeadByNumber (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         sl;             /* start position in <obj>                 */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         gr;             /* value of <r>                            */
    libGAP_UInt1 *     pl;             /* data area in <l>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt1 *     po;             /* data area in <obj>                      */

    /* get the generator number to stop                                    */
    gr = libGAP_INT_INTOBJ(r) - 1;

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);


    /* get the generator mask                                              */
    genm = ((1UL << (8-ebits)) - 1) << ebits;

    /* if <l> is the identity return                                       */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )  return l;

    /* look closely at the generators                                      */
    sl = 0;
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
    while ( sl < nl && ((*pl & genm) >> ebits) < gr ) {
        sl++;  pl++;
    }
    if ( sl == nl )
        return l;

    /* create a new word                                                   */
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), sl );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
    while ( 0 < sl-- )
        *po++ = *pl++;

    return obj;
}


/****************************************************************************
**
*F  Func8Bits_Less( <self>, <l>, <r> )
**
**  This  function implements    a length-plus-lexicographic   ordering   on
**  associative words.  This ordering  is translation  invariant, therefore,
**  we can  skip common prefixes.  This  is done  in  the first loop  of the
**  function.  When  a difference in  the  two words  is  encountered, it is
**  decided which is the  lexicographically smaller.  There are several case
**  that can occur:
**
**  The syllables where the difference  occurs have different generators.  In
**  this case it is sufficient to compare the two generators.  
**  Example: x^3 < y^3.
**
**  The syllables have the same generator but one exponent is the negative of
**  the other.  In this case the word with  the negative exponent is smaller.
**  Example: x^-1 < x.
**
**  Now it suffices  to compare the  unsigned  exponents.  For the  syllable
**  with the smaller exponent we  have to take  the  next generator in  that
**  word into  account.   This  means that  we  are  discarding  the smaller
**  syllable  as a common prefix.  Note  that if this  happens at the end of
**  one of the two words, then this word (ignoring the common prefix) is the
**  empty word and we can immediately decide which word is the smaller.
**  Examples: y^3 x < y^2 z, y^3 x > y^2 x z,  x^2 < x y^-1,  x^2 < x^3.
**
*/
libGAP_Obj libGAP_Func8Bits_Less (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         exl;            /* left exponent                           */
    libGAP_Int         exr;            /* right exponent                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt1 *     pl;             /* data area in <l>                        */
    libGAP_UInt1 *     pr;             /* data area in <r>                        */
    libGAP_Obj         lexico;         /* lexicographic order of <l> and <r>      */
    libGAP_Obj         ll;             /* length of <l>                           */
    libGAP_Obj         lr;             /* length of <r>                           */

    /* if <l> or <r> is the identity it is easy                            */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( nl == 0 || nr == 0 ) {
        return ( nr != 0 ) ? libGAP_True : libGAP_False;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);
    
    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    
    /* Skip the common prefix and determine if the first word is smaller   */
    /* with respect to the lexicographic ordering.                         */
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
    pr = (libGAP_UInt1*)libGAP_DATA_WORD(r);
    for ( lexico = libGAP_False;  0 < nl && 0 < nr;  nl--, nr--, pl++, pr++ )
        if ( *pl != *pr ) {
            /* got a difference                                            */

            /* get the generator mask                                      */
            genm = ((1UL << (8-ebits)) - 1) << ebits;

            /* compare the generators                                      */
            if ( (*pl & genm) != (*pr & genm) ) {
                lexico = ( (*pl & genm) < (*pr & genm) ) ? libGAP_True : libGAP_False;
                break;
            }

            /* get the unsigned exponents                                  */
            exl = (*pl & exps) ? (exps - (*pl & expm)) : (*pl & expm);
            exr = (*pr & exps) ? (exps - (*pr & expm)) : (*pr & expm);

            /* compare the sign of the exponents                           */
            if( exl == exr && (*pl & exps) != (*pr & exps) ) {
                lexico = (*pl & exps) ? libGAP_True : libGAP_False;
                break;
            }

            /* compare the exponents, and check the next generator.  This  */
            /* amounts to stripping off the common prefix  x^|expl-expr|.  */
            if( exl > exr ) {
                if( nr > 0 ) {
                  lexico = (*pl & genm) < (*(pr+1) & genm) ? libGAP_True : libGAP_False;
                  break;
                }
                else
                    /* <r> is now essentially the empty word.             */
                    return libGAP_False;
            }
            if( nl > 0 ) {  /* exl < exr                                  */
                lexico = (*(pl+1) & genm) < (*pr & genm) ? libGAP_True : libGAP_False;
                break;
            }
            /* <l> is now essentially the empty word.                     */
            return libGAP_True;
        }

    /* compute the lengths of the rest                                    */
    for ( ll = libGAP_INTOBJ_INT(0);  0 < nl;  nl--,  pl++ ) {
        exl = (*pl & exps) ? (exps - (*pl & expm)) : (*pl & expm);
        libGAP_C_SUM_FIA(ll,ll,libGAP_INTOBJ_INT(exl));
    }
    for ( lr = libGAP_INTOBJ_INT(0);  0 < nr;  nr--,  pr++ ) {
        exr = (*pr & exps) ? (exps - (*pr & expm)) : (*pr & expm);
        libGAP_C_SUM_FIA(lr,lr,libGAP_INTOBJ_INT(exr));
    }

    if( libGAP_EQ( ll, lr ) ) return lexico;

    return libGAP_LT( ll, lr ) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  Func8Bits_AssocWord( <self>, <kind>, <data> )
*/
libGAP_Obj libGAP_Func8Bits_AssocWord (
    libGAP_Obj         self,
    libGAP_Obj         kind,
    libGAP_Obj         data )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Obj         vexp;           /* value of current exponent               */
    libGAP_Int         nexp;           /* current exponent                        */
    libGAP_Obj         vgen;           /* value of current generator              */
    libGAP_Int         ngen;           /* current generator                       */
    libGAP_Obj         obj;            /* result                                  */
    libGAP_UInt1 *     ptr;            /* pointer into the data area of <obj>     */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* construct a new object                                              */
    num = libGAP_LEN_LIST(data)/2;
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt1 pointer for eight bits                                    */
    ptr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {

        /* this will not cause a garbage collection                        */
        vgen = libGAP_ELMW_LIST( data, 2*i-1 );
        ngen = libGAP_INT_INTOBJ(vgen);
        vexp = libGAP_ELMW_LIST( data, 2*i );
        nexp = libGAP_INT_INTOBJ(vexp);
        while ( ! libGAP_IS_INTOBJ(vexp) || nexp == 0 ) {
            vexp = libGAP_ErrorReturnObj( "<exponent> must be a positive integer", 
                                   0L, 0L,
                                   "you can replace <exponent> via 'return <exponent>;'" );
            nexp = libGAP_INT_INTOBJ(vexp);
            ptr  = (libGAP_UInt1*)libGAP_DATA_WORD(obj) + (i-1);
        }
        nexp = nexp & expm;
        *ptr = ((ngen-1) << ebits) | nexp;
        assert( ptr == (libGAP_UInt1*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(obj);

    return obj;
}


/****************************************************************************
**
*F  Func8Bits_ObjByVector( <self>, <kind>, <data> )
*/
libGAP_Obj libGAP_Func8Bits_ObjByVector (
    libGAP_Obj         self,
    libGAP_Obj         kind,
    libGAP_Obj         data )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         j;              /* loop variable for exponent vector       */
    libGAP_Int         nexp;           /* current exponent                        */
    libGAP_Obj         vexp;           /* value of current exponent               */
    libGAP_Obj         obj;            /* result                                  */
    libGAP_UInt1 *     ptr;            /* pointer into the data area of <obj>     */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* count the number of non-zero entries                                */
    for ( i = libGAP_LEN_LIST(data), num = 0, j = 1;  0 < i;  i-- ) {
        vexp = libGAP_ELMW_LIST(data,i);
        while ( ! libGAP_IS_INTOBJ(vexp) ) {
            vexp = libGAP_ErrorReturnObj(
                "%d element must be integer (not a %s)",
                (libGAP_Int) i, (libGAP_Int) libGAP_TNAM_OBJ(vexp),
                "you can replace the element by <val> via 'return <val>;'" );
        }
        if ( vexp != libGAP_INTOBJ_INT(0) ) {
            j = i;
            num++;
        }
    }

    /* construct a new object                                              */
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt1 pointer for eight bits                                    */
    ptr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++, j++ ) {

        /* this will not cause a garbage collection                        */
        while ( libGAP_ELMW_LIST(data,j) == libGAP_INTOBJ_INT(0) )
            j++;
        vexp = libGAP_ELMW_LIST( data, j );
        nexp = libGAP_INT_INTOBJ(vexp) & expm;
        *ptr = ((j-1) << ebits) | nexp;
        assert( ptr == (libGAP_UInt1*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(obj);

    return obj;
}


/****************************************************************************
**
*F  Func8Bits_Power( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func8Bits_Power (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         invm;           /* mask used to invert exponent            */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         sr;             /* start position in <r>                   */
    libGAP_Int         sl;             /* start position in <obj>                 */
    libGAP_UInt1 *     pl;             /* data area in <l>                        */
    libGAP_UInt1 *     pr;             /* data area in <obj>                      */
    libGAP_UInt1 *     pe;             /* end marker                              */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         pow;            /* power to take                           */
    libGAP_Int         apw;            /* absolute value of <pow>                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    invm = (1UL<<ebits)-1;

    /* get the generator mask                                              */
    genm = ((1UL << (8-ebits)) - 1) << ebits;

    /* if <l> is the identity return <l>                                   */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )
        return l;

    /* if <pow> is zero return the identity                                */
    pow = libGAP_INT_INTOBJ(r);
    if ( pow == 0 ) {
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 0 );
        return obj;
    }

    /* if <pow> is one return <l>                                          */
    if ( pow == 1 )
        return l;

    /* if <pow> is minus one invert <l>                                    */
    if ( pow == -1 ) {
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl );
        pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt1*)libGAP_DATA_WORD(obj) + (nl-1);
        sl = nl;

        /* exponents are symmtric, so we cannot get an overflow            */
        while ( 0 < sl-- ) {
            *pr-- = ( *pl++ ^ invm ) + 1;
        }
        return obj;
    }

    /* split word into w * h * w^-1                                        */
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
    pr = pl + (nl-1);
    sl = 0;
    sr = nl-1;
    while ( (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) == (*pr&exps) )
            break;
        if ( (*pl&expm) + (*pr&expm) != exps )
            break;
        pl++;  sl++;
        pr--;  sr--;
    }

    /* special case: w * gi^n * w^-1                                       */
    if ( sl == sr ) {
        ex = (*pl&expm);
        if ( *pl & exps )  ex -= exps;
        ex = ex * pow;

        /* check that n*pow fits into the exponent                         */
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }

        /* copy <l> into <obj>                                             */
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl );
        pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
        sl = nl;
        while ( 0 < sl-- ) {
            *pr++ = *pl++;
        }

        /* and fix the exponent at position <sr>                           */
        pr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
        pr[sr] = (pr[sr] & genm) | (ex & ((1UL<<ebits)-1));
        return obj;
    }

    /* special case: w * gj^x * t * gj^y * w^-1, x != -y                   */
    if ( (*pl & genm) == (*pr & genm) ) {
        ex = (*pl&expm) + (*pr&expm);
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex -= exps;

        /* check that <ex> fits into the exponent                          */
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
        if ( 0 < pow )
            ex = ex & ((1UL<<ebits)-1);
        else
            ex = (-ex) & ((1UL<<ebits)-1);

        /* create a new word                                               */
        apw = ( pow < 0 ) ? -pow : pow;
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 2*(sl+1)+apw*(sr-sl-1)+(apw-1) );

        /* copy the beginning w * gj^x into <obj>                          */
        pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
        pe = pl+sl;
        while ( pl <= pe ) {
            *pr++ = *pl++;
        }

        /* copy t * gj^<ex> <pow> times into <obj>                         */
        if ( 0 < pow ) {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt1*)libGAP_DATA_WORD(l) + (sl+1);
                pe = pl + (sr-sl-1);
                while ( pl <= pe ) {
                    *pr++ = *pl++;
                }
                pr[-1] = (pr[-1] & genm) | ex;
            }

            /* copy tail gj^y * w^-1 into <obj>                            */
            pr[-1] = pl[-1];
            pe = (libGAP_UInt1*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr++ = *pl++;
            }
        }

        /* copy and invert t * gj^<ex> <pow> times into <obj>              */
        else {
            pr[-1] = ( pl[sr-sl-1] ^ invm ) + 1;
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt1*)libGAP_DATA_WORD(l) + (sr-1);
                pe = pl + (sl-sr+1);
                while ( pe <= pl ) {
                    *pr++ = ( *pl-- ^ invm ) + 1;
                }
                pr[-1] = (pr[-1] & genm) | ex;
            }

            /* copy tail gj^x * w^-1 into <obj>                            */
            pr[-1] = ( pl[1] ^ invm ) + 1;
            pl = (libGAP_UInt1*)libGAP_DATA_WORD(l) + (sr+1);
            pe = (libGAP_UInt1*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr ++ = *pl++;
            }
        }
        return obj;
    }

    /* general case: w * t * w^-1                                          */
    else {

        /* create a new word                                               */
        apw = ( pow < 0 ) ? -pow : pow;
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 2*sl+apw*(sr-sl+1) );

        /* copy the beginning w * gj^x into <obj>                          */
        pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
        pe = pl+sl;
        while ( pl < pe ) {
            *pr++ = *pl++;
        }

        /* copy t <pow> times into <obj>                                   */
        if ( 0 < pow ) {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt1*)libGAP_DATA_WORD(l) + sl;
                pe = pl + (sr-sl);
                while ( pl <= pe ) {
                    *pr++ = *pl++;
                }
            }

            /* copy tail w^-1 into <obj>                                   */
            pe = (libGAP_UInt1*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr++ = *pl++;
            }
        }

        /* copy and invert t <pow> times into <obj>                        */
        else {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt1*)libGAP_DATA_WORD(l) + sr;
                pe = pl + (sl-sr);
                while ( pe <= pl ) {
                    *pr++ = ( *pl-- ^ invm ) + 1;
                }
            }

            /* copy tail w^-1 into <obj>                                   */
            pl = (libGAP_UInt1*)libGAP_DATA_WORD(l) + (sr+1);
            pe = (libGAP_UInt1*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr ++ = *pl++;
            }
        }
        return obj;
    }
}


/****************************************************************************
**
*F  Func8Bits_Product( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func8Bits_Product (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_Int         sr;             /* start position in <r>                   */
    libGAP_UInt1 *     pl;             /* data area in <l>                        */
    libGAP_UInt1 *     pr;             /* data area in <r>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt1 *     po;             /* data area in <obj>                      */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         over;           /* overlap                                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the generator mask                                              */
    genm = ((1UL << (8-ebits)) - 1) << ebits;

    /* if <l> or <r> is the identity return the other                      */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )  return r;
    nr = libGAP_NPAIRS_WORD(r);
    if ( 0 == nr )  return l;

    /* look closely at the meeting point                                   */
    sr = 0;
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l)+(nl-1);
    pr = (libGAP_UInt1*)libGAP_DATA_WORD(r);
    while ( 0 < nl && sr < nr && (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) == (*pr&exps) )
            break;
        if ( (*pl&expm) + (*pr&expm) != exps )
            break;
        pr++;  sr++;
        pl--;  nl--;
    }

    /* create a new word                                                   */
    over = ( 0 < nl && sr < nr && (*pl & genm) == (*pr & genm) ) ? 1 : 0;
    if ( over ) {
        ex = ( *pl & expm ) + ( *pr & expm );
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex -= exps;
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
    }
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl+(nr-sr)-over );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
    while ( 0 < nl-- )
        *po++ = *pl++;

    /* handle the overlap                                                  */
    if ( over ) {
        po[-1] = (po[-1] & genm) | (ex & ((1UL<<ebits)-1));
        sr++;
    }

    /* copy the <r> part into the word                                     */
    pr = ((libGAP_UInt1*)libGAP_DATA_WORD(r)) + sr;
    while ( sr++ < nr )
        *po++ = *pr++;
    return obj;
}


/****************************************************************************
**
*F  Func8Bits_Quotient( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func8Bits_Quotient (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        sepm;           /* unsigned exponent mask                  */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt1 *     pl;             /* data area in <l>                        */
    libGAP_UInt1 *     pr;             /* data area in <r>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt1 *     po;             /* data area in <obj>                      */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         over;           /* overlap                                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    sepm = (1UL << ebits) - 1;

    /* get the generator mask                                              */
    genm = ((1UL << (8-ebits)) - 1) << ebits;

    /* if <r> is the identity return <l>                                   */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( 0 == nr )  return l;

    /* look closely at the meeting point                                   */
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l)+(nl-1);
    pr = (libGAP_UInt1*)libGAP_DATA_WORD(r)+(nr-1);
    while ( 0 < nl && 0 < nr && (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) != (*pr&exps) )
            break;
        if ( (*pl&expm) != (*pr&expm) )
            break;
        pr--;  nr--;
        pl--;  nl--;
    }

    /* create a new word                                                   */
    over = ( 0 < nl && 0 < nr && (*pl & genm) == (*pr & genm) ) ? 1 : 0;
    if ( over ) {
        ex = ( *pl & expm ) - ( *pr & expm );
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex += exps;
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
    }
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl+nr-over );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt1*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt1*)libGAP_DATA_WORD(l);
    while ( 0 < nl-- )
        *po++ = *pl++;

    /* handle the overlap                                                  */
    if ( over ) {
        po[-1] = (po[-1] & genm) | (ex & sepm);
        nr--;
    }

    /* copy the <r> part into the word                                     */
    pr = ((libGAP_UInt1*)libGAP_DATA_WORD(r)) + (nr-1);
    while ( 0 < nr-- ) {
        *po++ = (*pr&genm) | (exps-(*pr&expm)) | (~*pr & exps);
        pr--;
    }
    return obj;
}

/****************************************************************************
**
*F  Func8Bits_LengthWord( <self>, <w> )
*/

libGAP_Obj libGAP_Func8Bits_LengthWord (
    libGAP_Obj         self,
    libGAP_Obj         w )
{
  libGAP_UInt npairs,i,ebits,exps,expm;
  libGAP_Obj len, uexp;
  libGAP_UInt1 *data, pair;
  npairs = libGAP_NPAIRS_WORD(w);
  ebits = libGAP_EBITS_WORD(w);
  data = (libGAP_UInt1*)libGAP_DATA_WORD(w);
  
  /* get the exponent masks                                              */
  exps = 1UL << (ebits-1);
  expm = exps - 1;
  
  len = libGAP_INTOBJ_INT(0);
  for (i = 0; i < npairs; i++)
    {
      pair = data[i];
      if (pair & exps)
        uexp = libGAP_INTOBJ_INT(exps - (pair & expm));
      else
        uexp = libGAP_INTOBJ_INT(pair & expm);
      libGAP_C_SUM_FIA(len,len,uexp);
    }
  return len;
}

/****************************************************************************
**
*F * * * * * * * * * * * * * * * * 16 bits word * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  Func16Bits_Equal( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func16Bits_Equal (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt2 *     pl;             /* data area in <l>                        */
    libGAP_UInt2 *     pr;             /* data area in <r>                        */

    /* if <l> or <r> is the identity it is easy                            */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( nl != nr ) {
        return libGAP_False;
    }

    /* compare the generator/exponent pairs                                */
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
    pr = (libGAP_UInt2*)libGAP_DATA_WORD(r);
    for ( ;  0 < nl;  nl--, pl++, pr++ ) {
        if ( *pl != *pr ) {
            return libGAP_False;
        }
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  Func16Bits_ExponentSums3( <self>, <obj>, <start>, <end> )
*/
libGAP_Obj libGAP_Func16Bits_ExponentSums3 (
    libGAP_Obj         self,
    libGAP_Obj         obj,
    libGAP_Obj         vstart,
    libGAP_Obj         vend )
{
    libGAP_Int         start;          /* the lowest generator number             */
    libGAP_Int         end;            /* the highest generator number            */
    libGAP_Obj         sums;           /* result, the exponent sums               */
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         pos;            /* current generator number                */
    libGAP_Int         exp;            /* current exponent                        */
    libGAP_UInt2 *     ptr;            /* pointer into the data area of <obj>     */

    /* <start> must be positive                                            */
    while ( !libGAP_IS_INTOBJ(vstart) || libGAP_INT_INTOBJ(vstart) <= 0 )
        vstart = libGAP_ErrorReturnObj( "<start> must be a positive integer", 0L, 0L,
                                 "you can replace <start> via 'return <start>;'" );
    start = libGAP_INT_INTOBJ(vstart);

    /* <end> must be positive                                              */
    while ( !libGAP_IS_INTOBJ(vend) || libGAP_INT_INTOBJ(vend) <= 0 )
        vend = libGAP_ErrorReturnObj( "<end> must be a positive integer", 0L, 0L,
                               "you can replace <end> via 'return <end>;'" );
    end = libGAP_INT_INTOBJ(vend);

    /* <end> must be at least <start>                                      */
    if ( end < start ) {
        sums = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, 0 );
        libGAP_SET_LEN_PLIST( sums, 0 );
        return sums;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(obj);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the number of gen/exp pairs                                     */
    num = libGAP_NPAIRS_WORD(obj);

    /* create the zero vector                                              */
    sums = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, end-start+1 );
    libGAP_SET_LEN_PLIST( sums, end-start+1 );
    for ( i = start;  i <= end;  i++ )
        libGAP_SET_ELM_PLIST( sums, i-start+1, 0 );

    /* and unpack <obj> into <sums>                                        */
    ptr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {
        pos = ((*ptr) >> ebits)+1;
        if ( start <= pos && pos <= end ) {
            if ( (*ptr) & exps )
                exp = ((*ptr)&expm)-exps;
            else
                exp = (*ptr)&expm;

            /* this will not cause a garbage collection                    */
            exp = exp + (libGAP_Int) libGAP_ELM_PLIST( sums, pos-start+1 );
            libGAP_SET_ELM_PLIST( sums, pos-start+1, (libGAP_Obj) exp );
            assert( ptr == (libGAP_UInt2*)libGAP_DATA_WORD(obj) + (i-1) );
        }
    }

    /* convert integers into values                                        */
    for ( i = start;  i <= end;  i++ ) {
        exp = (libGAP_Int) libGAP_ELM_PLIST( sums, i-start+1 );
        libGAP_SET_ELM_PLIST( sums, i-start+1, libGAP_INTOBJ_INT(exp) );
    }

    /* return the exponent sums                                            */
    return sums;
}


/****************************************************************************
**
*F  Func16Bits_ExponentSums1( <self>, <obj> )
*/
libGAP_Obj libGAP_Func16Bits_ExponentSums1 (
    libGAP_Obj         self,
    libGAP_Obj         obj )
{
    return libGAP_Func16Bits_ExponentSums3( self, obj,
        libGAP_INTOBJ_INT(1), libGAP_INTOBJ_INT(libGAP_RANK_WORD(obj)) );
}


/****************************************************************************
**
*F  Func16Bits_ExponentSyllable( <self>, <w>, <i> )
*/
libGAP_Obj libGAP_Func16Bits_ExponentSyllable (
    libGAP_Obj         self,
    libGAP_Obj         w,
    libGAP_Obj         vi )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* integer corresponding to <vi>           */
    libGAP_UInt2       p;              /* <i>th syllable                          */

    /* check <i>                                                           */
    num = libGAP_NPAIRS_WORD(w);
    while ( !libGAP_IS_INTOBJ(vi) || libGAP_INT_INTOBJ(vi) <= 0 || num < libGAP_INT_INTOBJ(vi) )
        vi = libGAP_ErrorReturnObj( "<i> must be an integer between 1 and %d", num, 0L,
                             "you can replace <i> via 'return <i>;'" );
    i = libGAP_INT_INTOBJ(vi);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(w);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* return the <i> th exponent                                          */
    p = ((libGAP_UInt2*)libGAP_DATA_WORD(w))[i-1];
    if ( p & exps )
        return libGAP_INTOBJ_INT((p&expm)-exps);
    else
        return libGAP_INTOBJ_INT(p&expm);
}


/****************************************************************************
**
*F  Func16Bits_ExtRepOfObj( <self>, <obj> )
*/
libGAP_Obj libGAP_Func16Bits_ExtRepOfObj (
    libGAP_Obj         self,
    libGAP_Obj         obj )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Obj         kind;           /* kind of <obj>                           */
    libGAP_UInt2 *     ptr;            /* pointer into the data area of <obj>     */
    libGAP_Obj         lst;            /* result                                  */

    /* get the kind of <obj>                                               */
    kind = libGAP_TYPE_DATOBJ(obj);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the number of gen/exp pairs                                     */
    num = libGAP_NPAIRS_WORD(obj);

    /* construct a list with 2*<num> entries                               */
    lst = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*num );
    libGAP_SET_LEN_PLIST( lst, 2*num );

    /* and unpack <obj> into <lst>                                         */
    ptr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {

        /* this will not cause a garbage collection                        */
        libGAP_SET_ELM_PLIST( lst, 2*i-1, libGAP_INTOBJ_INT(((*ptr) >> ebits)+1) );
        if ( (*ptr) & exps )
            libGAP_SET_ELM_PLIST( lst, 2*i, libGAP_INTOBJ_INT(((*ptr)&expm)-exps) );
        else
            libGAP_SET_ELM_PLIST( lst, 2*i, libGAP_INTOBJ_INT((*ptr)&expm) );
        assert( ptr == (libGAP_UInt2*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(lst);

    /* return the gen/exp list                                             */
    return lst;
}


/****************************************************************************
**
*F  Func16Bits_GeneratorSyllable( <self>, <w>, <i> )
*/
libGAP_Obj libGAP_Func16Bits_GeneratorSyllable (
    libGAP_Obj         self,
    libGAP_Obj         w,
    libGAP_Obj         vi )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* integer corresponding to <vi>           */
    libGAP_UInt2       p;              /* <i>th syllable                          */

    /* check <i>                                                           */
    num = libGAP_NPAIRS_WORD(w);
    while ( !libGAP_IS_INTOBJ(vi) || libGAP_INT_INTOBJ(vi) <= 0 || num < libGAP_INT_INTOBJ(vi) )
        vi = libGAP_ErrorReturnObj( "<i> must be an integer between 1 and %d", num, 0L,
                             "you can replace <i> via 'return <i>;'" );
    i = libGAP_INT_INTOBJ(vi);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(w);

    /* return the <i> th generator                                         */
    p = ((libGAP_UInt2*)libGAP_DATA_WORD(w))[i-1];
    return libGAP_INTOBJ_INT((p >> ebits)+1);
}


/****************************************************************************
**
*F  Func16Bits_HeadByNumber( <self>, <l>, <gen> )
*/
libGAP_Obj libGAP_Func16Bits_HeadByNumber (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         sl;             /* start position in <obj>                 */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         gr;             /* value of <r>                            */
    libGAP_UInt2 *     pl;             /* data area in <l>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt2 *     po;             /* data area in <obj>                      */

    /* get the generator number to stop                                    */
    gr = libGAP_INT_INTOBJ(r) - 1;

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the generator mask                                              */
    genm = ((1UL << (16-ebits)) - 1) << ebits;

    /* if <l> is the identity return                                       */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )  return l;

    /* look closely at the generators                                      */
    sl = 0;
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
    while ( sl < nl && ((*pl & genm) >> ebits) < gr ) {
        sl++;  pl++;
    }
    if ( sl == nl )
        return l;

    /* create a new word                                                   */
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), sl );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
    while ( 0 < sl-- )
        *po++ = *pl++;

    return obj;
}


/****************************************************************************
**
*F  Func16Bits_Less( <self>, <l>, <r> )
**
**  For an explanation of this function, see the comments before
**  Func8Bits_Less(). 
*/
libGAP_Obj libGAP_Func16Bits_Less (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         exl;            /* left exponent                           */
    libGAP_Int         exr;            /* right exponent                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt2 *     pl;             /* data area in <l>                        */
    libGAP_UInt2 *     pr;             /* data area in <r>                        */
    libGAP_Obj         lexico;         /* lexicographic order of <l> and <r>      */
    libGAP_Obj         ll;             /* length of <l>                           */
    libGAP_Obj         lr;             /* length of <r>                           */

    /* if <l> or <r> is the identity it is easy                            */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( nl == 0 || nr == 0 ) {
        return ( nr != 0 ) ? libGAP_True : libGAP_False;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);
    
    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    
    /* Skip the common prefix and determine if the first word is smaller   */
    /* with respect to the lexicographic ordering.                         */
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
    pr = (libGAP_UInt2*)libGAP_DATA_WORD(r);
    for ( lexico = libGAP_False;  0 < nl && 0 < nr;  nl--, nr--, pl++, pr++ )
        if ( *pl != *pr ) {
            /* got a difference                                            */

            /* get the generator mask                                      */
            genm = ((1UL << (16-ebits)) - 1) << ebits;

            /* compare the generators                                      */
            if ( (*pl & genm) != (*pr & genm) ) {
                lexico = ( (*pl & genm) < (*pr & genm) ) ? libGAP_True : libGAP_False;
                break;
            }

            /* get the unsigned exponents                                  */
            exl = (*pl & exps) ? (exps - (*pl & expm)) : (*pl & expm);
            exr = (*pr & exps) ? (exps - (*pr & expm)) : (*pr & expm);

            /* compare the sign of the exponents                           */
            if( exl == exr && (*pl & exps) != (*pr & exps) ) {
                lexico = (*pl & exps) ? libGAP_True : libGAP_False;
                break;
            }

            /* compare the exponents, and check the next generator.  This  */
            /* amounts to stripping off the common prefix  x^|expl-expr|.  */
            if( exl > exr ) {
                if( nr > 0 ) {
                  lexico = (*pl & genm) < (*(pr+1) & genm) ? libGAP_True : libGAP_False;
                  break;
                }
                else
                    /* <r> is now essentially the empty word.             */
                    return libGAP_False;
            }
            if( nl > 0 ) {  /* exl < exr                                  */
                lexico = (*(pl+1) & genm) < (*pr & genm) ? libGAP_True : libGAP_False;
                break;
            }
            /* <l> is now essentially the empty word.                     */
            return libGAP_True;
        }

    /* compute the lengths of the rest                                    */
    for ( ll = libGAP_INTOBJ_INT(0);  0 < nl;  nl--,  pl++ ) {
        exl = (*pl & exps) ? (exps - (*pl & expm)) : (*pl & expm);
        libGAP_C_SUM_FIA(ll,ll,libGAP_INTOBJ_INT(exl));
    }
    for ( lr = libGAP_INTOBJ_INT(0);  0 < nr;  nr--,  pr++ ) {
        exr = (*pr & exps) ? (exps - (*pr & expm)) : (*pr & expm);
        libGAP_C_SUM_FIA(lr,lr,libGAP_INTOBJ_INT(exr));
    }

    if( libGAP_EQ( ll, lr ) ) return lexico;

    return libGAP_LT( ll, lr ) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  Func16Bits_AssocWord( <self>, <kind>, <data> )
*/
libGAP_Obj libGAP_Func16Bits_AssocWord (
    libGAP_Obj         self,
    libGAP_Obj         kind,
    libGAP_Obj         data )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Obj         vgen;           /* value of current exponent               */
    libGAP_Int         nexp;           /* current exponent                        */
    libGAP_Obj         vexp;           /* value of current generator              */
    libGAP_Int         ngen;           /* current generator                       */
    libGAP_Obj         obj;            /* result                                  */
    libGAP_UInt2 *     ptr;            /* pointer into the data area of <obj>     */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* construct a new object                                              */
    num = libGAP_LEN_LIST(data)/2;
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt2 pointer for sixteen bits                                  */
    ptr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {

        /* this will not cause a garbage collection                        */
        vgen = libGAP_ELMW_LIST( data, 2*i-1 );
        ngen = libGAP_INT_INTOBJ(vgen);
        vexp = libGAP_ELMW_LIST( data, 2*i );
        nexp = libGAP_INT_INTOBJ(vexp);
        while ( ! libGAP_IS_INTOBJ(vexp) || nexp == 0 ) {
            vexp = libGAP_ErrorReturnObj( "<exponent> must be a positive integer", 
                                   0L, 0L,
                                   "you can replace <exponent> via 'return <exponent>;'" );
            nexp = libGAP_INT_INTOBJ(vexp);
            ptr = (libGAP_UInt2*)libGAP_DATA_WORD(obj) + (i-1);
        }
        nexp = nexp & expm;
        *ptr = ((ngen-1) << ebits) | nexp;
        assert( ptr == (libGAP_UInt2*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(obj);

    return obj;
}


/****************************************************************************
**
*F  Func16Bits_ObjByVector( <self>, <kind>, <data> )
*/
libGAP_Obj libGAP_Func16Bits_ObjByVector (
    libGAP_Obj         self,
    libGAP_Obj         kind,
    libGAP_Obj         data )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         j;              /* loop variable for exponent vector       */
    libGAP_Int         nexp;           /* current exponent                        */
    libGAP_Obj         vexp;           /* value of current exponent               */
    libGAP_Obj         obj;            /* result                                  */
    libGAP_UInt2 *     ptr;            /* pointer into the data area of <obj>     */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* count the number of non-zero entries                                */
    for ( i = libGAP_LEN_LIST(data), num = 0, j = 1;  0 < i;  i-- ) {
        vexp = libGAP_ELMW_LIST(data,i);
        while ( ! libGAP_IS_INTOBJ(vexp) ) {
            vexp = libGAP_ErrorReturnObj(
                "%d element must be an integer (not a %s)",
                (libGAP_Int) i, (libGAP_Int) libGAP_TNAM_OBJ(vexp),
                "you can replace the element by <val> via 'return <val>;'" );
        }
        if ( vexp != libGAP_INTOBJ_INT(0) ) {
            j = i;
            num++;
        }
    }

    /* construct a new object                                              */
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt2 pointer for sixteen bits                                  */
    ptr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++, j++ ) {

        /* this will not cause a garbage collection                        */
        while ( libGAP_ELMW_LIST(data,j) == libGAP_INTOBJ_INT(0) )
            j++;
        vexp = libGAP_ELMW_LIST( data, j );
        nexp = libGAP_INT_INTOBJ(vexp) & expm;
        *ptr = ((j-1) << ebits) | nexp;
        assert( ptr == (libGAP_UInt2*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(obj);

    return obj;
}


/****************************************************************************
**
*F  Func16Bits_Power( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func16Bits_Power (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         invm;           /* mask used to invert exponent            */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         sr;             /* start position in <r>                   */
    libGAP_Int         sl;             /* start position in <obj>                 */
    libGAP_UInt2 *     pl;             /* data area in <l>                        */
    libGAP_UInt2 *     pr;             /* data area in <obj>                      */
    libGAP_UInt2 *     pe;             /* end marker                              */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         pow;            /* power to take                           */
    libGAP_Int         apw;            /* absolute value of <pow>                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    invm = (1UL<<ebits)-1;

    /* get the generator mask                                              */
    genm = ((1UL << (16-ebits)) - 1) << ebits;

    /* if <l> is the identity return <l>                                   */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )
        return l;

    /* if <pow> is zero return the identity                                */
    pow = libGAP_INT_INTOBJ(r);
    if ( pow == 0 ) {
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 0 );
        return obj;
    }

    /* if <pow> is one return <l>                                          */
    if ( pow == 1 )
        return l;

    /* if <pow> is minus one invert <l>                                    */
    if ( pow == -1 ) {
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl );
        pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt2*)libGAP_DATA_WORD(obj) + (nl-1);
        sl = nl;

        /* exponents are symmtric, so we cannot get an overflow            */
        while ( 0 < sl-- ) {
            *pr-- = ( *pl++ ^ invm ) + 1;
        }
        return obj;
    }

    /* split word into w * h * w^-1                                        */
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
    pr = pl + (nl-1);
    sl = 0;
    sr = nl-1;
    while ( (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) == (*pr&exps) )
            break;
        if ( (*pl&expm) + (*pr&expm) != exps )
            break;
        pl++;  sl++;
        pr--;  sr--;
    }

    /* special case: w * gi^n * w^-1                                       */
    if ( sl == sr ) {
        ex = (*pl&expm);
        if ( *pl & exps )  ex -= exps;
        ex = ex * pow;

        /* check that n*pow fits into the exponent                         */
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }

        /* copy <l> into <obj>                                             */
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl );
        pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
        sl = nl;
        while ( 0 < sl-- ) {
            *pr++ = *pl++;
        }

        /* and fix the exponent at position <sr>                           */
        pr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
        pr[sr] = (pr[sr] & genm) | (ex & ((1UL<<ebits)-1));
        return obj;
    }

    /* special case: w * gj^x * t * gj^y * w^-1, x != -y                   */
    if ( (*pl & genm) == (*pr & genm) ) {
        ex = (*pl&expm) + (*pr&expm);
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex -= exps;

        /* check that <ex> fits into the exponent                          */
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
        if ( 0 < pow )
            ex = ex & ((1UL<<ebits)-1);
        else
            ex = (-ex) & ((1UL<<ebits)-1);

        /* create a new word                                               */
        apw = ( pow < 0 ) ? -pow : pow;
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 2*(sl+1)+apw*(sr-sl-1)+(apw-1) );

        /* copy the beginning w * gj^x into <obj>                          */
        pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
        pe = pl+sl;
        while ( pl <= pe ) {
            *pr++ = *pl++;
        }

        /* copy t * gj^<ex> <pow> times into <obj>                         */
        if ( 0 < pow ) {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt2*)libGAP_DATA_WORD(l) + (sl+1);
                pe = pl + (sr-sl-1);
                while ( pl <= pe ) {
                    *pr++ = *pl++;
                }
                pr[-1] = (pr[-1] & genm) | ex;
            }

            /* copy tail gj^y * w^-1 into <obj>                            */
            pr[-1] = pl[-1];
            pe = (libGAP_UInt2*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr++ = *pl++;
            }
        }

        /* copy and invert t * gj^<ex> <pow> times into <obj>              */
        else {
            pr[-1] = ( pl[sr-sl-1] ^ invm ) + 1;
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt2*)libGAP_DATA_WORD(l) + (sr-1);
                pe = pl + (sl-sr+1);
                while ( pe <= pl ) {
                    *pr++ = ( *pl-- ^ invm ) + 1;
                }
                pr[-1] = (pr[-1] & genm) | ex;
            }

            /* copy tail gj^x * w^-1 into <obj>                            */
            pr[-1] = ( pl[1] ^ invm ) + 1;
            pl = (libGAP_UInt2*)libGAP_DATA_WORD(l) + (sr+1);
            pe = (libGAP_UInt2*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr ++ = *pl++;
            }
        }
        return obj;
    }

    /* general case: w * t * w^-1                                          */
    else {

        /* create a new word                                               */
        apw = ( pow < 0 ) ? -pow : pow;
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 2*sl+apw*(sr-sl+1) );

        /* copy the beginning w * gj^x into <obj>                          */
        pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
        pe = pl+sl;
        while ( pl < pe ) {
            *pr++ = *pl++;
        }

        /* copy t <pow> times into <obj>                                   */
        if ( 0 < pow ) {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt2*)libGAP_DATA_WORD(l) + sl;
                pe = pl + (sr-sl);
                while ( pl <= pe ) {
                    *pr++ = *pl++;
                }
            }

            /* copy tail w^-1 into <obj>                                   */
            pe = (libGAP_UInt2*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr++ = *pl++;
            }
        }

        /* copy and invert t <pow> times into <obj>                        */
        else {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt2*)libGAP_DATA_WORD(l) + sr;
                pe = pl + (sl-sr);
                while ( pe <= pl ) {
                    *pr++ = ( *pl-- ^ invm ) + 1;
                }
            }

            /* copy tail w^-1 into <obj>                                   */
            pl = (libGAP_UInt2*)libGAP_DATA_WORD(l) + (sr+1);
            pe = (libGAP_UInt2*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr ++ = *pl++;
            }
        }
        return obj;
    }
}


/****************************************************************************
**
*F  Func16Bits_Product( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func16Bits_Product (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_Int         sr;             /* start position in <r>                   */
    libGAP_UInt2 *     pl;             /* data area in <l>                        */
    libGAP_UInt2 *     pr;             /* data area in <r>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt2 *     po;             /* data area in <obj>                      */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         over;           /* overlap                                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the generator mask                                              */
    genm = ((1UL << (16-ebits)) - 1) << ebits;

    /* if <l> or <r> is the identity return the other                      */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )  return r;
    nr = libGAP_NPAIRS_WORD(r);
    if ( 0 == nr )  return l;

    /* look closely at the meeting point                                   */
    sr = 0;
    pl = ((libGAP_UInt2*)libGAP_DATA_WORD(l))+(nl-1);
    pr = (libGAP_UInt2*)libGAP_DATA_WORD(r);
    while ( 0 < nl && sr < nr && (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) == (*pr&exps) )
            break;
        if ( (*pl&expm) + (*pr&expm) != exps )
            break;
        pr++;  sr++;
        pl--;  nl--;
    }

    /* create a new word                                                   */
    over = ( 0 < nl && sr < nr && (*pl & genm) == (*pr & genm) ) ? 1 : 0;
    if ( over ) {
        ex = ( *pl & expm ) + ( *pr & expm );
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex -= exps;
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
    }
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl+(nr-sr)-over );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
    while ( 0 < nl-- )
        *po++ = *pl++;

    /* handle the overlap                                                  */
    if ( over ) {
        po[-1] = (po[-1] & genm) | (ex & ((1UL<<ebits)-1));
        sr++;
    }

    /* copy the <r> part into the word                                     */
    pr = ((libGAP_UInt2*)libGAP_DATA_WORD(r)) + sr;
    while ( sr++ < nr )
        *po++ = *pr++;
    return obj;
}


/****************************************************************************
**
*F  Func16Bits_Quotient( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func16Bits_Quotient (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        sepm;           /* unsigned exponent mask                  */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt2 *     pl;             /* data area in <l>                        */
    libGAP_UInt2 *     pr;             /* data area in <r>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt2 *     po;             /* data area in <obj>                      */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         over;           /* overlap                                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    sepm = (1UL << ebits) - 1;

    /* get the generator mask                                              */
    genm = ((1UL << (16-ebits)) - 1) << ebits;

    /* if <r> is the identity return <l>                                   */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( 0 == nr )  return l;

    /* look closely at the meeting point                                   */
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l)+(nl-1);
    pr = (libGAP_UInt2*)libGAP_DATA_WORD(r)+(nr-1);
    while ( 0 < nl && 0 < nr && (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) != (*pr&exps) )
            break;
        if ( (*pl&expm) != (*pr&expm) )
            break;
        pr--;  nr--;
        pl--;  nl--;
    }

    /* create a new word                                                   */
    over = ( 0 < nl && 0 < nr && (*pl & genm) == (*pr & genm) ) ? 1 : 0;
    if ( over ) {
        ex = ( *pl & expm ) - ( *pr & expm );
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex += exps;
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
    }
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl+nr-over );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt2*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt2*)libGAP_DATA_WORD(l);
    while ( 0 < nl-- )
        *po++ = *pl++;

    /* handle the overlap                                                  */
    if ( over ) {
        po[-1] = (po[-1] & genm) | (ex & sepm);
        nr--;
    }

    /* copy the <r> part into the word                                     */
    pr = ((libGAP_UInt2*)libGAP_DATA_WORD(r)) + (nr-1);
    while ( 0 < nr-- ) {
        *po++ = (*pr&genm) | (exps-(*pr&expm)) | (~*pr & exps);
        pr--;
    }
    return obj;
}

/****************************************************************************
**
*F  Func16Bits_LengthWord( <self>, <w> )
*/

libGAP_Obj libGAP_Func16Bits_LengthWord (
    libGAP_Obj         self,
    libGAP_Obj         w )
{
  libGAP_UInt npairs,i,ebits,exps,expm;
  libGAP_Obj len, uexp;
  libGAP_UInt2 *data, pair;
  
  npairs = libGAP_NPAIRS_WORD(w);
  ebits = libGAP_EBITS_WORD(w);
  data = (libGAP_UInt2*)libGAP_DATA_WORD(w);
  
  /* get the exponent masks                                              */
  exps = 1UL << (ebits-1);
  expm = exps - 1;
  
  len = libGAP_INTOBJ_INT(0);
  for (i = 0; i < npairs; i++)
    {
      pair = data[i];
      if (pair & exps)
        uexp = libGAP_INTOBJ_INT(exps - (pair & expm));
      else
        uexp = libGAP_INTOBJ_INT(pair & expm);
      libGAP_C_SUM_FIA(len,len,uexp);
    }
  return len;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * * *  32 bits words * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  Func32Bits_Equal( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func32Bits_Equal (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt4 *     pl;             /* data area in <l>                        */
    libGAP_UInt4 *     pr;             /* data area in <r>                        */

    /* if <l> or <r> is the identity it is easy                            */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( nl != nr ) {
        return libGAP_False;
    }

    /* compare the generator/exponent pairs                                */
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
    pr = (libGAP_UInt4*)libGAP_DATA_WORD(r);
    for ( ;  0 < nl;  nl--, pl++, pr++ ) {
        if ( *pl != *pr ) {
            return libGAP_False;
        }
    }
    return libGAP_True;
}


/****************************************************************************
**
*F  Func32Bits_ExponentSums3( <self>, <obj>, <start>, <end> )
*/
libGAP_Obj libGAP_Func32Bits_ExponentSums3 (
    libGAP_Obj         self,
    libGAP_Obj         obj,
    libGAP_Obj         vstart,
    libGAP_Obj         vend )
{
    libGAP_Int         start;          /* the lowest generator number             */
    libGAP_Int         end;            /* the highest generator number            */
    libGAP_Obj         sums;           /* result, the exponent sums               */
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         pos;            /* current generator number                */
    libGAP_Int         exp;            /* current exponent                        */
    libGAP_UInt4 *     ptr;            /* pointer into the data area of <obj>     */

    /* <start> must be positive                                            */
    while ( !libGAP_IS_INTOBJ(vstart) || libGAP_INT_INTOBJ(vstart) <= 0 )
        vstart = libGAP_ErrorReturnObj( "<start> must be a positive integer", 0L, 0L,
                                 "you can replace <start> via 'return <start>;'" );
    start = libGAP_INT_INTOBJ(vstart);

    /* <end> must be positive                                              */
    while ( !libGAP_IS_INTOBJ(vend) || libGAP_INT_INTOBJ(vend) <= 0 )
        vend = libGAP_ErrorReturnObj( "<end> must be a positive integer", 0L, 0L,
                               "you can replace <end> via 'return <end>;'" );
    end = libGAP_INT_INTOBJ(vend);

    /* <end> must be at least <start>                                      */
    if ( end < start ) {
        sums = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, 0 );
        libGAP_SET_LEN_PLIST( sums, 0 );
        return sums;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(obj);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the number of gen/exp pairs                                     */
    num = libGAP_NPAIRS_WORD(obj);

    /* create the zero vector                                              */
    sums = libGAP_NEW_PLIST( libGAP_T_PLIST_CYC, end-start+1 );
    libGAP_SET_LEN_PLIST( sums, end-start+1 );
    for ( i = start;  i <= end;  i++ )
        libGAP_SET_ELM_PLIST( sums, i-start+1, 0 );

    /* and unpack <obj> into <sums>                                        */
    ptr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {
        pos = ((*ptr) >> ebits)+1;
        if ( start <= pos && pos <= end ) {
            if ( (*ptr) & exps )
                exp = ((*ptr)&expm)-exps;
            else
                exp = (*ptr)&expm;

            /* this will not cause a garbage collection                    */
            exp = exp + (libGAP_Int) libGAP_ELM_PLIST( sums, pos-start+1 );
            libGAP_SET_ELM_PLIST( sums, pos-start+1, (libGAP_Obj) exp );
            assert( ptr == (libGAP_UInt4*)libGAP_DATA_WORD(obj) + (i-1) );
        }
    }

    /* convert integers into values                                        */
    for ( i = start;  i <= end;  i++ ) {
        exp = (libGAP_Int) libGAP_ELM_PLIST( sums, i-start+1 );
        libGAP_SET_ELM_PLIST( sums, i-start+1, libGAP_INTOBJ_INT(exp) );
    }

    /* return the exponent sums                                            */
    return sums;
}


/****************************************************************************
**
*F  Func32Bits_ExponentSums1( <self>, <obj> )
*/
libGAP_Obj libGAP_Func32Bits_ExponentSums1 (
    libGAP_Obj         self,
    libGAP_Obj         obj )
{
    return libGAP_Func32Bits_ExponentSums3( self, obj,
        libGAP_INTOBJ_INT(1), libGAP_INTOBJ_INT(libGAP_RANK_WORD(obj)) );
}


/****************************************************************************
**
*F  Func32Bits_ExponentSyllable( <self>, <w>, <i> )
*/
libGAP_Obj libGAP_Func32Bits_ExponentSyllable (
    libGAP_Obj         self,
    libGAP_Obj         w,
    libGAP_Obj         vi )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* integer corresponding to <vi>           */
    libGAP_UInt4       p;              /* <i>th syllable                          */

    /* check <i>                                                           */
    num = libGAP_NPAIRS_WORD(w);
    while ( !libGAP_IS_INTOBJ(vi) || libGAP_INT_INTOBJ(vi) <= 0 || num < libGAP_INT_INTOBJ(vi) )
        vi = libGAP_ErrorReturnObj( "<i> must be an integer between 1 and %d", num, 0L,
                             "you can replace <i> via 'return <i>;'" );
    i = libGAP_INT_INTOBJ(vi);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(w);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* return the <i> th exponent                                          */
    p = ((libGAP_UInt4*)libGAP_DATA_WORD(w))[i-1];
    if ( p & exps )
        return libGAP_INTOBJ_INT((p&expm)-exps);
    else
        return libGAP_INTOBJ_INT(p&expm);
}


/****************************************************************************
**
*F  Func32Bits_ExtRepOfObj( <self>, <obj> )
*/
libGAP_Obj libGAP_Func32Bits_ExtRepOfObj (
    libGAP_Obj         self,
    libGAP_Obj         obj )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Obj         kind;           /* kind of <obj>                           */
    libGAP_UInt4 *     ptr;            /* pointer into the data area of <obj>     */
    libGAP_Obj         lst;            /* result                                  */

    /* get the kind of <obj>                                               */
    kind = libGAP_TYPE_DATOBJ(obj);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the number of gen/exp pairs                                     */
    num = libGAP_NPAIRS_WORD(obj);

    /* construct a list with 2*<num> entries                               */
    lst = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*num );
    libGAP_SET_LEN_PLIST( lst, 2*num );

    /* and unpack <obj> into <lst>                                         */
    ptr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {

        /* this will not cause a garbage collection                    */
        libGAP_SET_ELM_PLIST( lst, 2*i-1, libGAP_INTOBJ_INT(((*ptr) >> ebits)+1) );
        if ( (*ptr) & exps )
            libGAP_SET_ELM_PLIST( lst, 2*i, libGAP_INTOBJ_INT(((*ptr)&expm)-exps) );
        else
            libGAP_SET_ELM_PLIST( lst, 2*i, libGAP_INTOBJ_INT((*ptr)&expm) );
        assert( ptr == (libGAP_UInt4*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(lst);

    /* return the gen/exp list                                             */
    return lst;
}


/****************************************************************************
**
*F  Func32Bits_GeneratorSyllable( <self>, <w>, <i> )
*/
libGAP_Obj libGAP_Func32Bits_GeneratorSyllable (
    libGAP_Obj         self,
    libGAP_Obj         w,
    libGAP_Obj         vi )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* integer corresponding to <vi>           */
    libGAP_UInt4       p;              /* <i>th syllable                          */

    /* check <i>                                                           */
    num = libGAP_NPAIRS_WORD(w);
    while ( !libGAP_IS_INTOBJ(vi) || libGAP_INT_INTOBJ(vi) <= 0 || num < libGAP_INT_INTOBJ(vi) )
        vi = libGAP_ErrorReturnObj( "<i> must be an integer between 1 and %d", num, 0L,
                             "you can replace <i> via 'return <i>;'" );
    i = libGAP_INT_INTOBJ(vi);

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(w);

    /* return the <i> th generator                                         */
    p = ((libGAP_UInt4*)libGAP_DATA_WORD(w))[i-1];
    return libGAP_INTOBJ_INT((p >> ebits)+1);
}


/****************************************************************************
**
*F  Func32Bits_HeadByNumber( <self>, <l>, <gen> )
*/
libGAP_Obj libGAP_Func32Bits_HeadByNumber (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         sl;             /* start position in <obj>                 */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         gr;             /* value of <r>                            */
    libGAP_UInt4 *     pl;             /* data area in <l>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt4 *     po;             /* data area in <obj>                      */

    /* get the generator number to stop                                    */
    gr = libGAP_INT_INTOBJ(r) - 1;

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the generator mask                                              */
    genm = ((1UL << (32-ebits)) - 1) << ebits;

    /* if <l> is the identity return                                       */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )  return l;

    /* look closely at the generators                                      */
    sl = 0;
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
    while ( sl < nl && ((*pl & genm) >> ebits) < gr ) {
        sl++;  pl++;
    }
    if ( sl == nl )
        return l;

    /* create a new word                                                   */
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), sl );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
    while ( 0 < sl-- )
        *po++ = *pl++;

    return obj;
}


/****************************************************************************
**
*F  Func32Bits_Less( <self>, <l>, <r> )
**
**  For an explanation of this function, see the comments before
**  Func8Bits_Less(). 
*/
libGAP_Obj libGAP_Func32Bits_Less (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         exl;            /* left exponent                           */
    libGAP_Int         exr;            /* right exponent                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt4 *     pl;             /* data area in <l>                        */
    libGAP_UInt4 *     pr;             /* data area in <r>                        */
    libGAP_Obj         lexico;         /* lexicographic order of <l> and <r>      */
    libGAP_Obj         ll;             /* length of <l>                           */
    libGAP_Obj         lr;             /* length of <r>                           */

    /* if <l> or <r> is the identity it is easy                            */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( nl == 0 || nr == 0 ) {
        return ( nr != 0 ) ? libGAP_True : libGAP_False;
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);
    
    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    
    /* Skip the common prefix and determine if the first word is smaller   */
    /* with respect to the lexicographic ordering.                         */
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
    pr = (libGAP_UInt4*)libGAP_DATA_WORD(r);
    for ( lexico = libGAP_False;  0 < nl && 0 < nr;  nl--, nr--, pl++, pr++ )
        if ( *pl != *pr ) {
            /* got a difference                                            */

            /* get the generator mask                                      */
            genm = ((1UL << (32-ebits)) - 1) << ebits;

            /* compare the generators                                      */
            if ( (*pl & genm) != (*pr & genm) ) {
                lexico = ( (*pl & genm) < (*pr & genm) ) ? libGAP_True : libGAP_False;
                break;
            }

            /* get the unsigned exponents                                  */
            exl = (*pl & exps) ? (exps - (*pl & expm)) : (*pl & expm);
            exr = (*pr & exps) ? (exps - (*pr & expm)) : (*pr & expm);

            /* compare the sign of the exponents                           */
            if( exl == exr && (*pl & exps) != (*pr & exps) ) {
                lexico = (*pl & exps) ? libGAP_True : libGAP_False;
                break;
            }

            /* compare the exponents, and check the next generator.  This  */
            /* amounts to stripping off the common prefix  x^|expl-expr|.  */
            if( exl > exr ) {
                if( nr > 0 ) {
                  lexico = (*pl & genm) < (*(pr+1) & genm) ? libGAP_True : libGAP_False;
                  break;
                }
                else
                    /* <r> is now essentially the empty word.             */
                    return libGAP_False;
            }
            if( nl > 0 ) {  /* exl < exr                                  */
                lexico = (*(pl+1) & genm) < (*pr & genm) ? libGAP_True : libGAP_False;
                break;
            }
            /* <l> is now essentially the empty word.                     */
            return libGAP_True;
        }

    /* compute the lengths of the rest                                    */
    for ( ll = libGAP_INTOBJ_INT(0);  0 < nl;  nl--,  pl++ ) {
        exl = (*pl & exps) ? (exps - (*pl & expm)) : (*pl & expm);
        libGAP_C_SUM_FIA(ll,ll,libGAP_INTOBJ_INT(exl));
    }
    for ( lr = libGAP_INTOBJ_INT(0);  0 < nr;  nr--,  pr++ ) {
        exr = (*pr & exps) ? (exps - (*pr & expm)) : (*pr & expm);
        libGAP_C_SUM_FIA(lr,lr,libGAP_INTOBJ_INT(exr));
    }

    if( libGAP_EQ( ll, lr ) ) return lexico;

    return libGAP_LT( ll, lr ) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  Func32Bits_AssocWord( <self>, <kind>, <data> )
*/
libGAP_Obj libGAP_Func32Bits_AssocWord (
    libGAP_Obj         self,
    libGAP_Obj         kind,
    libGAP_Obj         data )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Obj         vgen;           /* value of current exponent               */
    libGAP_Int         nexp;           /* current exponent                        */
    libGAP_Obj         vexp;           /* value of current generator              */
    libGAP_Int         ngen;           /* current generator                       */
    libGAP_Obj         obj;            /* result                                  */
    libGAP_UInt4 *     ptr;            /* pointer into the data area of <obj>     */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* construct a new object                                              */
    num = libGAP_LEN_LIST(data)/2;
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt4 pointer for eight bits                                    */
    ptr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++ ) {

        /* this will not cause a garbage collection                        */
        vgen = libGAP_ELMW_LIST( data, 2*i-1 );
        ngen = libGAP_INT_INTOBJ(vgen);
        vexp = libGAP_ELMW_LIST( data, 2*i );
        nexp = libGAP_INT_INTOBJ(vexp);
        while ( ! libGAP_IS_INTOBJ(vexp) || nexp == 0 ) {
            vexp = libGAP_ErrorReturnObj( "<exponent> must not be a positive integer",
                                   0L, 0L,
                                   "you can replace <exponent> via 'return <exponent>;'" );
            nexp = libGAP_INT_INTOBJ(vexp);
            ptr = (libGAP_UInt4*)libGAP_DATA_WORD(obj) + (i-1);
        }
        nexp = nexp & expm;
        *ptr = ((ngen-1) << ebits) | nexp;
        assert( ptr == (libGAP_UInt4*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(obj);

    return obj;
}


/****************************************************************************
**
*F  Func32Bits_ObjByVector( <self>, <kind>, <data> )
*/
libGAP_Obj libGAP_Func32Bits_ObjByVector (
    libGAP_Obj         self,
    libGAP_Obj         kind,
    libGAP_Obj         data )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* unsigned exponent mask                  */
    libGAP_Int         num;            /* number of gen/exp pairs in <data>       */
    libGAP_Int         i;              /* loop variable for gen/exp pairs         */
    libGAP_Int         j;              /* loop variable for exponent vector       */
    libGAP_Int         nexp;           /* current exponent                        */
    libGAP_Obj         vexp;           /* value of current exponent               */
    libGAP_Obj         obj;            /* result                                  */
    libGAP_UInt4 *     ptr;            /* pointer into the data area of <obj>     */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE(kind);

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* count the number of non-zero entries                                */
    for ( i = libGAP_LEN_LIST(data), num = 0, j = 1;  0 < i;  i-- ) {
        vexp = libGAP_ELMW_LIST(data,i);
        while ( ! libGAP_IS_INTOBJ(vexp) ) {
            vexp = libGAP_ErrorReturnObj(
                "%d element must be an integer (not a %s)",
                (libGAP_Int) i, (libGAP_Int) libGAP_TNAM_OBJ(vexp),
                "you can replace the element by <val> via 'return <val>;'" );
        }
        if ( vexp != libGAP_INTOBJ_INT(0) ) {
            j = i;
            num++;
        }
    }

    /* construct a new object                                              */
    libGAP_NEW_WORD( obj, kind, num );

    /* use UInt4 pointer for thirteen bits                                 */
    ptr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    for ( i = 1;  i <= num;  i++, ptr++, j++ ) {

        /* this will not cause a garbage collection                        */
        while ( libGAP_ELMW_LIST(data,j) == libGAP_INTOBJ_INT(0) )
            j++;
        vexp = libGAP_ELMW_LIST( data, j );
        nexp = libGAP_INT_INTOBJ(vexp) & expm;
        *ptr = ((j-1) << ebits) | nexp;
        assert( ptr == (libGAP_UInt4*)libGAP_DATA_WORD(obj) + (i-1) );
    }
    libGAP_CHANGED_BAG(obj);

    return obj;
}


/****************************************************************************
**
*F  Func32Bits_Power( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func32Bits_Power (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         invm;           /* mask used to invert exponent            */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         sr;             /* start position in <r>                   */
    libGAP_Int         sl;             /* start position in <obj>                 */
    libGAP_UInt4 *     pl;             /* data area in <l>                        */
    libGAP_UInt4 *     pr;             /* data area in <obj>                      */
    libGAP_UInt4 *     pe;             /* end marker                              */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         exs;            /* save <ex> for overflow test             */
    libGAP_Int         pow;            /* power to take                           */
    libGAP_Int         apw;            /* absolute value of <pow>                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    invm = (1UL<<ebits)-1;

    /* get the generator mask                                              */
    genm = ((1UL << (32-ebits)) - 1) << ebits;

    /* if <l> is the identity return <l>                                   */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )
        return l;

    /* if <pow> is zero return the identity                                */
    pow = libGAP_INT_INTOBJ(r);
    if ( pow == 0 ) {
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 0 );
        return obj;
    }

    /* if <pow> is one return <l>                                          */
    if ( pow == 1 )
        return l;

    /* if <pow> is minus one invert <l>                                    */
    if ( pow == -1 ) {
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl );
        pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt4*)libGAP_DATA_WORD(obj) + (nl-1);
        sl = nl;

        /* exponents are symmtric, so we cannot get an overflow            */
        while ( 0 < sl-- ) {
            *pr-- = ( *pl++ ^ invm ) + 1;
        }
        return obj;
    }

    /* split word into w * h * w^-1                                        */
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
    pr = pl + (nl-1);
    sl = 0;
    sr = nl-1;
    while ( (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) == (*pr&exps) )
            break;
        if ( (*pl&expm) + (*pr&expm) != exps )
            break;
        pl++;  sl++;
        pr--;  sr--;
    }

    /* special case: w * gi^n * w^-1                                       */
    if ( sl == sr ) {
        ex = (*pl&expm);
        if ( *pl & exps )  ex -= exps;
        exs = ex;
        ex  = (libGAP_Int)((libGAP_UInt)ex * (libGAP_UInt)pow);

        /* check that n*pow fits into the exponent                         */
        if ( ex/pow!=exs || (0<ex && expm<ex) || (ex<0 && expm<-ex) ) {
            return libGAP_TRY_NEXT_METHOD;
        }

        /* copy <l> into <obj>                                             */
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl );
        pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
        sl = nl;
        while ( 0 < sl-- ) {
            *pr++ = *pl++;
        }

        /* and fix the exponent at position <sr>                           */
        pr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
        pr[sr] = (pr[sr] & genm) | (ex & ((1UL<<ebits)-1));
        return obj;
    }

    /* special case: w * gj^x * t * gj^y * w^-1, x != -y                   */
    if ( (*pl & genm) == (*pr & genm) ) {
        ex = (*pl&expm) + (*pr&expm);
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex -= exps;

        /* check that <ex> fits into the exponent                          */
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
        if ( 0 < pow )
            ex = ex & ((1UL<<ebits)-1);
        else
            ex = (-ex) & ((1UL<<ebits)-1);

        /* create a new word                                               */
        apw = ( pow < 0 ) ? -pow : pow;
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 2*(sl+1)+apw*(sr-sl-1)+(apw-1) );

        /* copy the beginning w * gj^x into <obj>                          */
        pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
        pe = pl+sl;
        while ( pl <= pe ) {
            *pr++ = *pl++;
        }

        /* copy t * gj^<ex> <pow> times into <obj>                         */
        if ( 0 < pow ) {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt4*)libGAP_DATA_WORD(l) + (sl+1);
                pe = pl + (sr-sl-1);
                while ( pl <= pe ) {
                    *pr++ = *pl++;
                }
                pr[-1] = (pr[-1] & genm) | ex;
            }

            /* copy tail gj^y * w^-1 into <obj>                            */
            pr[-1] = pl[-1];
            pe = (libGAP_UInt4*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr++ = *pl++;
            }
        }

        /* copy and invert t * gj^<ex> <pow> times into <obj>              */
        else {
            pr[-1] = ( pl[sr-sl-1] ^ invm ) + 1;
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt4*)libGAP_DATA_WORD(l) + (sr-1);
                pe = pl + (sl-sr+1);
                while ( pe <= pl ) {
                    *pr++ = ( *pl-- ^ invm ) + 1;
                }
                pr[-1] = (pr[-1] & genm) | ex;
            }

            /* copy tail gj^x * w^-1 into <obj>                            */
            pr[-1] = ( pl[1] ^ invm ) + 1;
            pl = (libGAP_UInt4*)libGAP_DATA_WORD(l) + (sr+1);
            pe = (libGAP_UInt4*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr ++ = *pl++;
            }
        }
        return obj;
    }

    /* general case: w * t * w^-1                                          */
    else {

        /* create a new word                                               */
        apw = ( pow < 0 ) ? -pow : pow;
        libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), 2*sl+apw*(sr-sl+1) );

        /* copy the beginning w * gj^x into <obj>                          */
        pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
        pr = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
        pe = pl+sl;
        while ( pl < pe ) {
            *pr++ = *pl++;
        }

        /* copy t <pow> times into <obj>                                   */
        if ( 0 < pow ) {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt4*)libGAP_DATA_WORD(l) + sl;
                pe = pl + (sr-sl);
                while ( pl <= pe ) {
                    *pr++ = *pl++;
                }
            }

            /* copy tail w^-1 into <obj>                                   */
            pe = (libGAP_UInt4*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr++ = *pl++;
            }
        }

        /* copy and invert t <pow> times into <obj>                        */
        else {
            for ( ; 0 < apw;  apw-- ) {
                pl = (libGAP_UInt4*)libGAP_DATA_WORD(l) + sr;
                pe = pl + (sl-sr);
                while ( pe <= pl ) {
                    *pr++ = ( *pl-- ^ invm ) + 1;
                }
            }

            /* copy tail w^-1 into <obj>                                   */
            pl = (libGAP_UInt4*)libGAP_DATA_WORD(l) + (sr+1);
            pe = (libGAP_UInt4*)libGAP_DATA_WORD(l) + nl;
            while ( pl < pe ) {
                *pr ++ = *pl++;
            }
        }
        return obj;
    }
}


/****************************************************************************
**
*F  Func32Bits_Product( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func32Bits_Product (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_Int         sr;             /* start position in <r>                   */
    libGAP_UInt4 *     pl;             /* data area in <l>                        */
    libGAP_UInt4 *     pr;             /* data area in <r>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt4 *     po;             /* data area in <obj>                      */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         over;           /* overlap                                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;

    /* get the generator mask                                              */
    genm = ((1UL << (32-ebits)) - 1) << ebits;

    /* if <l> or <r> is the identity return the other                      */
    nl = libGAP_NPAIRS_WORD(l);
    if ( 0 == nl )  return r;
    nr = libGAP_NPAIRS_WORD(r);
    if ( 0 == nr )  return l;

    /* look closely at the meeting point                                   */
    sr = 0;
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l)+(nl-1);
    pr = (libGAP_UInt4*)libGAP_DATA_WORD(r);
    while ( 0 < nl && sr < nr && (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) == (*pr&exps) )
            break;
        if ( (*pl&expm) + (*pr&expm) != exps )
            break;
        pr++;  sr++;
        pl--;  nl--;
    }

    /* create a new word                                                   */
    over = ( 0 < nl && sr < nr && (*pl & genm) == (*pr & genm) ) ? 1 : 0;
    if ( over ) {
        ex = ( *pl & expm ) + ( *pr & expm );
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex -= exps;
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
    }
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl+(nr-sr)-over );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
    while ( 0 < nl-- )
        *po++ = *pl++;

    /* handle the overlap                                                  */
    if ( over ) {
        po[-1] = (po[-1] & genm) | (ex & ((1UL<<ebits)-1));
        sr++;
    }

    /* copy the <r> part into the word                                     */
    pr = ((libGAP_UInt4*)libGAP_DATA_WORD(r)) + sr;
    while ( sr++ < nr )
        *po++ = *pr++;
    return obj;
}


/****************************************************************************
**
*F  Func32Bits_Quotient( <self>, <l>, <r> )
*/
libGAP_Obj libGAP_Func32Bits_Quotient (
    libGAP_Obj         self,
    libGAP_Obj         l,
    libGAP_Obj         r )
{
    libGAP_Int         ebits;          /* number of bits in the exponent          */
    libGAP_UInt        expm;           /* signed exponent mask                    */
    libGAP_UInt        sepm;           /* unsigned exponent mask                  */
    libGAP_UInt        exps;           /* sign exponent mask                      */
    libGAP_UInt        genm;           /* generator mask                          */
    libGAP_Int         nl;             /* number of pairs to consider in <l>      */
    libGAP_Int         nr;             /* number of pairs in <r>                  */
    libGAP_UInt4 *     pl;             /* data area in <l>                        */
    libGAP_UInt4 *     pr;             /* data area in <r>                        */
    libGAP_Obj         obj;            /* the result                              */
    libGAP_UInt4 *     po;             /* data area in <obj>                      */
    libGAP_Int         ex = 0;         /* meeting exponent                        */
    libGAP_Int         over;           /* overlap                                 */

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORD(l);

    /* get the exponent masks                                              */
    exps = 1UL << (ebits-1);
    expm = exps - 1;
    sepm = (1UL << ebits) - 1;

    /* get the generator mask                                              */
    genm = ((1UL << (32-ebits)) - 1) << ebits;

    /* if <r> is the identity return <l>                                   */
    nl = libGAP_NPAIRS_WORD(l);
    nr = libGAP_NPAIRS_WORD(r);
    if ( 0 == nr )  return l;

    /* look closely at the meeting point                                   */
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l)+(nl-1);
    pr = (libGAP_UInt4*)libGAP_DATA_WORD(r)+(nr-1);
    while ( 0 < nl && 0 < nr && (*pl & genm) == (*pr & genm) ) {
        if ( (*pl&exps) != (*pr&exps) )
            break;
        if ( (*pl&expm) != (*pr&expm) )
            break;
        pr--;  nr--;
        pl--;  nl--;
    }

    /* create a new word                                                   */
    over = ( 0 < nl && 0 < nr && (*pl & genm) == (*pr & genm) ) ? 1 : 0;
    if ( over ) {
        ex = ( *pl & expm ) - ( *pr & expm );
        if ( *pl & exps )  ex -= exps;
        if ( *pr & exps )  ex += exps;
        if ( ( 0 < ex && expm < ex ) || ( ex < 0 && expm < -ex ) ) {
            return libGAP_TRY_NEXT_METHOD;
        }
    }
    libGAP_NEW_WORD( obj, libGAP_PURETYPE_WORD(l), nl+nr-over );

    /* copy the <l> part into the word                                     */
    po = (libGAP_UInt4*)libGAP_DATA_WORD(obj);
    pl = (libGAP_UInt4*)libGAP_DATA_WORD(l);
    while ( 0 < nl-- )
        *po++ = *pl++;

    /* handle the overlap                                                  */
    if ( over ) {
        po[-1] = (po[-1] & genm) | (ex & sepm);
        nr--;
    }

    /* copy the <r> part into the word                                     */
    pr = ((libGAP_UInt4*)libGAP_DATA_WORD(r)) + (nr-1);
    while ( 0 < nr-- ) {
        *po++ = (*pr&genm) | (exps-(*pr&expm)) | (~*pr & exps);
        pr--;
    }
    return obj;
}

/****************************************************************************
**
*F  Func32Bits_LengthWord( <self>, <w> )
*/

libGAP_Obj libGAP_Func32Bits_LengthWord (
    libGAP_Obj         self,
    libGAP_Obj         w )
{
  libGAP_UInt npairs,i,ebits,exps,expm;
  libGAP_Obj len, uexp;
  libGAP_UInt4 *data, pair;
  
  npairs = libGAP_NPAIRS_WORD(w);
  ebits = libGAP_EBITS_WORD(w);
  data = (libGAP_UInt4*)libGAP_DATA_WORD(w);
  
  /* get the exponent masks                                              */
  exps = 1UL << (ebits-1);
  expm = exps - 1;
  
  len = libGAP_INTOBJ_INT(0);
  for (i = 0; i < npairs; i++)
    {
      pair = data[i];
      if (pair & exps)
        uexp = libGAP_INTOBJ_INT(exps - (pair & expm));
      else
        uexp = libGAP_INTOBJ_INT(pair & expm);
      libGAP_C_SUM_FIA(len,len,uexp);
    }
  return len;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * * * all bits words * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  FuncNBits_NumberSyllables( <self>, <w> )
*/
libGAP_Obj libGAP_FuncNBits_NumberSyllables (
    libGAP_Obj         self,
    libGAP_Obj         w )
{
    /* return the number of syllables                                      */
    return libGAP_INTOBJ_INT( libGAP_NPAIRS_WORD(w) );
}


/**************************************************************************
* letter rep arithmetic */
/**************************************************************************
*F  FuncMultWorLettrep( <self>, <a>,<b> ) */
libGAP_Obj libGAP_FuncMultWorLettrep (
    libGAP_Obj         self,
    libGAP_Obj         a,
    libGAP_Obj         b)
{
  libGAP_UInt l,m,i,j,newlen,as,bs,ae,be;
  libGAP_Obj n;
  libGAP_Obj *p;
  libGAP_Obj *q;

  /* short check */
  while ( ! libGAP_IS_PLIST(a) ) {
      a = libGAP_ErrorReturnObj(
                "first argument must be plain list (not a %s)",
                (libGAP_Int) libGAP_TNAM_OBJ(a), 0L,
                "you can replace the element by <val> via 'return <val>;'" );
  }
  while ( ! libGAP_IS_PLIST(b) ) {
      b = libGAP_ErrorReturnObj(
                "second argument must be plain list (not a %s)",
                (libGAP_Int) libGAP_TNAM_OBJ(b), 0L, 
                "you can replace the element by <val> via 'return <val>;'" );
  }

  /* Find overlap */
  /* l:=Length(a); */
  l=libGAP_LEN_PLIST(a);
  if (l==0) {
    return b; 
  }  
  /* m:=Length(b); */
  m=libGAP_LEN_PLIST(b);
  if (m==0) {
    return a; 
  }  
  /* now we know both lists are length >0 */

  /* i:=l; */
  i=l;
  /* j:=1; */
  j=1;
  /* while i>=1 and j<=m and a[i]=-b[j] do */
  while ((i>=1)&&(j<=m)&& 
    (libGAP_INT_INTOBJ(libGAP_ELM_PLIST(a,i))==-libGAP_INT_INTOBJ(libGAP_ELM_PLIST(b,j)))) {
    /* i:=i-1; */
    i--;
    /* j:=j+1; */
    j++;
  /* od; */
  }
  /* if i=0 then */
  if (i==0) {
    /* if j>m then */
    if (j>m) {
      /* full cancellation */
      /* return false; */
      return libGAP_False;
    }
    /* fi; */
    /* a:=b{[j..m]}; */
    as=1;
    ae=0;
    bs=j;
    be=m;
    newlen=m-j+1;
  }
  /* elif j>m then */
  else {
    if (j>m) {
    /* a:=a{[1..i]}; */
      as=1;
      ae=i;
      bs=1;
      be=0;
      newlen=i;
    }
    else {
  /* else */
    /* a:=Concatenation(a{[1..i]},b{[j..m]}); */
      as=1;
      ae=i;
      bs=j;
      be=m;
      newlen=m-j+1+i;
    }
  /* fi; */
  }
  /* make the new list */
  n=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC,newlen);
  q=libGAP_ADDR_OBJ(n);
  q++;
  j=as;
  /* a[as] position */
  /* there must be a better way for giving this address ... */
  p=(libGAP_Obj*) &(libGAP_ADDR_OBJ(a)[as]);
  while (j<=ae) {
    *q++=*p++;
    j++;
  }
  j=bs;
  /* b[bs] position */
  /* there must be a better way for giving this address ... */
  p=(libGAP_Obj*) &(libGAP_ADDR_OBJ(b)[bs]);
  while (j<=be) {
    *q++=*p++;
    j++;
  }
  libGAP_SET_LEN_PLIST(n,newlen);
  libGAP_CHANGED_BAG(n);
  /* return a; */
  return n;
}

/*F  FuncMultBytLettrep( <self>, <a>,<b> ) */
libGAP_Obj libGAP_FuncMultBytLettrep (
    libGAP_Obj         self,
    libGAP_Obj         a,
    libGAP_Obj         b)
{
  libGAP_UInt l,m,i,j,newlen,as,bs,ae,be;
  libGAP_Obj n;
  libGAP_UInt1 *p,*q;
  
  /* short check, if necessary strings are compacted */
  while ( ! libGAP_IsStringConv(a) ) {
      a = libGAP_ErrorReturnObj(
                "first argument must be string (not a %s)",
                (libGAP_Int) libGAP_TNAM_OBJ(a), 0L,
                "you can replace the element by <val> via 'return <val>;'" );
  }
  while ( ! libGAP_IsStringConv(b) ) {
      b = libGAP_ErrorReturnObj(
                "second argument must be string (not a %s)",
                (libGAP_Int) libGAP_TNAM_OBJ(b), 0L, 
                "you can replace the element by <val> via 'return <val>;'" );
  }
  
  /* Find overlap */
  /* l:=Length(a); */
  l=libGAP_GET_LEN_STRING(a);
  if (l==0) {
    return b; 
  }  
  /* m:=Length(b); */
  m=libGAP_GET_LEN_STRING(b);
  if (m==0) {
    return a; 
  }  
  /* now we know both lists are length >0 */

  /* i:=l; */
  i=l;
  /* j:=1; */
  j=1;
  /* while i>=1 and j<=m and a[i]=-b[j] do */
  p=libGAP_CHARS_STRING(a);
  q=libGAP_CHARS_STRING(b);
  while ((i>=1)&&(j<=m)&&
    (libGAP_SINT_CHAR(p[i-1])==-libGAP_SINT_CHAR(q[j-1]))) {
    /* i:=i-1; */
    i--;
    /* j:=j+1; */
    j++;
  /* od; */
  }
  /* if i=0 then */
  if (i==0) {
    /* if j>m then */
    if (j>m) {
      /* full cancellation */
      /* return false; */
      return libGAP_False;
    }
    /* fi; */
    /* a:=b{[j..m]}; */
    as=1;
    ae=0;
    bs=j;
    be=m;
    newlen=m-j+1;
  }
  /* elif j>m then */
  else {
    if (j>m) {
    /* a:=a{[1..i]}; */
      as=1;
      ae=i;
      bs=1;
      be=0;
      newlen=i;
    }
    else {
  /* else */
    /* a:=Concatenation(a{[1..i]},b{[j..m]}); */
      as=1;
      ae=i;
      bs=j;
      be=m;
      newlen=m-j+1+i;
    }
  /* fi; */
  }
  /* make the new list */
  n=libGAP_NEW_STRING(newlen);
  q=libGAP_CHARS_STRING(n);
  p=libGAP_CHARS_STRING(a);
  j=as;
  /* a[as] position */
  while (j<=ae) {
    *q++=p[j-1];
    j++;
  }
  j=bs;
  p=libGAP_CHARS_STRING(b);
  /* b[bs] position */
  while (j<=be) {
    *q++=p[j-1];
    j++;
  }
  /* return a; */
  libGAP_CHANGED_BAG(n);
  return n;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "8Bits_Equal", 2, "8_bits_word, 8_bits_word",
      libGAP_Func8Bits_Equal, "src/objfgelm.c:8Bits_Equal" },

    { "8Bits_ExponentSums1", 1, "8_bits_word",
      libGAP_Func8Bits_ExponentSums1, "src/objfgelm.c:8Bits_ExponentSums1" },

    { "8Bits_ExponentSums3", 3, "8_bits_word, start, end",
      libGAP_Func8Bits_ExponentSums3, "src/objfgelm.c:8Bits_ExponentSums3" },

    { "8Bits_ExponentSyllable", 2, "8_bits_word, position",
      libGAP_Func8Bits_ExponentSyllable, "src/objfgelm.c:8Bits_ExponentSyllable" },

    { "8Bits_ExtRepOfObj", 1, "8_bits_word",
      libGAP_Func8Bits_ExtRepOfObj, "src/objfgelm.c:8Bits_ExtRepOfObj" },

    { "8Bits_GeneratorSyllable", 2, "8_bits_word, position",
      libGAP_Func8Bits_GeneratorSyllable, "src/objfgelm.c:8Bits_GeneratorSyllable" },

    { "8Bits_Less", 2, "8_bits_word, 8_bits_word",
      libGAP_Func8Bits_Less, "src/objfgelm.c:8Bits_Less" },

    { "8Bits_AssocWord", 2, "kind, data",
      libGAP_Func8Bits_AssocWord, "src/objfgelm.c:8Bits_AssocWord" },

    { "8Bits_NumberSyllables", 1, "8_bits_word",
      libGAP_FuncNBits_NumberSyllables, "src/objfgelm.c:8Bits_NumberSyllables" },

    { "8Bits_ObjByVector", 2, "kind, data",
      libGAP_Func8Bits_ObjByVector, "src/objfgelm.c:8Bits_ObjByVector" },

    { "8Bits_HeadByNumber", 2, "16_bits_word, gen_num",
      libGAP_Func8Bits_HeadByNumber, "src/objfgelm.c:8Bits_HeadByNumber" },

    { "8Bits_Power", 2, "8_bits_word, small_integer",
      libGAP_Func8Bits_Power, "src/objfgelm.c:8Bits_Power" },

    { "8Bits_Product", 2, "8_bits_word, 8_bits_word",
      libGAP_Func8Bits_Product, "src/objfgelm.c:8Bits_Product" },

    { "8Bits_Quotient", 2, "8_bits_word, 8_bits_word",
      libGAP_Func8Bits_Quotient, "src/objfgelm.c:8Bits_Quotient" },

    { "8Bits_LengthWord", 1, "8_bits_word",
      libGAP_Func8Bits_LengthWord, "src/objfgelm.c:8Bits_LengthWord" },

    { "16Bits_Equal", 2, "16_bits_word, 16_bits_word",
      libGAP_Func16Bits_Equal, "src/objfgelm.c:16Bits_Equal" },

    { "16Bits_ExponentSums1", 1, "16_bits_word",
      libGAP_Func16Bits_ExponentSums1, "src/objfgelm.c:16Bits_ExponentSums1" },

    { "16Bits_ExponentSums3", 3, "16_bits_word, start, end",
      libGAP_Func16Bits_ExponentSums3, "src/objfgelm.c:16Bits_ExponentSums3" },

    { "16Bits_ExponentSyllable", 2, "16_bits_word, position",
      libGAP_Func16Bits_ExponentSyllable, "src/objfgelm.c:16Bits_ExponentSyllable" },

    { "16Bits_ExtRepOfObj", 1, "16_bits_word",
      libGAP_Func16Bits_ExtRepOfObj, "src/objfgelm.c:16Bits_ExtRepOfObj" },

    { "16Bits_GeneratorSyllable", 2, "16_bits_word, pos",
      libGAP_Func16Bits_GeneratorSyllable, "src/objfgelm.c:16Bits_GeneratorSyllable" },

    { "16Bits_Less", 2, "16_bits_word, 16_bits_word",
      libGAP_Func16Bits_Less, "src/objfgelm.c:16Bits_Less" },

    { "16Bits_AssocWord", 2, "kind, data",
      libGAP_Func16Bits_AssocWord, "src/objfgelm.c:16Bits_AssocWord" },

    { "16Bits_NumberSyllables", 1, "16_bits_word",
      libGAP_FuncNBits_NumberSyllables, "src/objfgelm.c:16Bits_NumberSyllables" },

    { "16Bits_ObjByVector", 2, "kind, data",
      libGAP_Func16Bits_ObjByVector, "src/objfgelm.c:16Bits_ObjByVector" },

    { "16Bits_HeadByNumber", 2, "16_bits_word, gen_num",
      libGAP_Func16Bits_HeadByNumber, "src/objfgelm.c:16Bits_HeadByNumber" },

    { "16Bits_Power", 2, "16_bits_word, small_integer",
      libGAP_Func16Bits_Power, "src/objfgelm.c:16Bits_Power" },

    { "16Bits_Product", 2, "16_bits_word, 16_bits_word",
      libGAP_Func16Bits_Product, "src/objfgelm.c:16Bits_Product" },

    { "16Bits_Quotient", 2, "16_bits_word, 16_bits_word",
      libGAP_Func16Bits_Quotient, "src/objfgelm.c:16Bits_Quotient" },

    { "16Bits_LengthWord", 1, "16_bits_word",
      libGAP_Func16Bits_LengthWord, "src/objfgelm.c:16Bits_LengthWord" },

    { "32Bits_Equal", 2, "32_bits_word, 32_bits_word",
      libGAP_Func32Bits_Equal, "src/objfgelm.c:32Bits_Equal" },

    { "32Bits_ExponentSums1", 1, "32_bits_word",
      libGAP_Func32Bits_ExponentSums1, "src/objfgelm.c:32Bits_ExponentSums1" },

    { "32Bits_ExponentSums3", 3, "32_bits_word, start, end",
      libGAP_Func32Bits_ExponentSums3, "src/objfgelm.c:32Bits_ExponentSums3" },

    { "32Bits_ExponentSyllable", 2, "32_bits_word, position",
      libGAP_Func32Bits_ExponentSyllable, "src/objfgelm.c:32Bits_ExponentSyllable" },

    { "32Bits_ExtRepOfObj", 1, "32_bits_word",
      libGAP_Func32Bits_ExtRepOfObj, "src/objfgelm.c:32Bits_ExtRepOfObj" },

    { "32Bits_GeneratorSyllable", 2, "32_bits_word, pos",
      libGAP_Func32Bits_GeneratorSyllable, "src/objfgelm.c:32Bits_GeneratorSyllable" },

    { "32Bits_Less", 2, "32_bits_word, 32_bits_word",
      libGAP_Func32Bits_Less, "src/objfgelm.c:32Bits_Less" },

    { "32Bits_AssocWord", 2, "kind, data",
      libGAP_Func32Bits_AssocWord, "src/objfgelm.c:32Bits_AssocWord" },

    { "32Bits_NumberSyllables", 1, "32_bits_word",
      libGAP_FuncNBits_NumberSyllables, "src/objfgelm.c:32Bits_NumberSyllables" },

    { "32Bits_ObjByVector", 2, "kind, data",
      libGAP_Func32Bits_ObjByVector, "src/objfgelm.c:32Bits_ObjByVector" },

    { "32Bits_HeadByNumber", 2, "16_bits_word, gen_num",
      libGAP_Func32Bits_HeadByNumber, "src/objfgelm.c:32Bits_HeadByNumber" },

    { "32Bits_Power", 2, "32_bits_word, small_integer",
      libGAP_Func32Bits_Power, "src/objfgelm.c:32Bits_Power" },

    { "32Bits_Product", 2, "32_bits_word, 32_bits_word",
      libGAP_Func32Bits_Product, "src/objfgelm.c:32Bits_Product" },

    { "32Bits_Quotient", 2, "32_bits_word, 32_bits_word",
      libGAP_Func32Bits_Quotient, "src/objfgelm.c:32Bits_Quotient" },

    { "32Bits_LengthWord", 1, "32_bits_word",
      libGAP_Func32Bits_LengthWord, "src/objfgelm.c:32Bits_LengthWord" },

    { "MULT_WOR_LETTREP", 2, "list,list",
      libGAP_FuncMultWorLettrep, "src/objfgelm.c:MULT_WOR_LETTREP" },

    { "MULT_BYT_LETTREP", 2, "string,string",
      libGAP_FuncMultBytLettrep, "src/objfgelm.c:MULT_BYT_LETTREP" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* export position numbers 'AWP_SOMETHING'                             */
    libGAP_AssGVar( libGAP_GVarName( "AWP_FIRST_ENTRY" ),
             libGAP_INTOBJ_INT(libGAP_AWP_FIRST_ENTRY) );
    libGAP_AssGVar( libGAP_GVarName( "AWP_PURE_TYPE" ),
             libGAP_INTOBJ_INT(libGAP_AWP_PURE_TYPE) );
    libGAP_AssGVar( libGAP_GVarName( "AWP_NR_BITS_EXP" ),
             libGAP_INTOBJ_INT(libGAP_AWP_NR_BITS_EXP) );
    libGAP_AssGVar( libGAP_GVarName( "AWP_NR_GENS" ),
             libGAP_INTOBJ_INT(libGAP_AWP_NR_GENS) );
    libGAP_AssGVar( libGAP_GVarName( "AWP_NR_BITS_PAIR" ),
             libGAP_INTOBJ_INT(libGAP_AWP_NR_BITS_PAIR) );
    libGAP_AssGVar( libGAP_GVarName( "AWP_FUN_OBJ_BY_VECTOR" ),
             libGAP_INTOBJ_INT(libGAP_AWP_FUN_OBJ_BY_VECTOR) );
    libGAP_AssGVar( libGAP_GVarName( "AWP_FUN_ASSOC_WORD" ),
             libGAP_INTOBJ_INT(libGAP_AWP_FUN_ASSOC_WORD) );
    libGAP_AssGVar( libGAP_GVarName( "AWP_FIRST_FREE" ),
             libGAP_INTOBJ_INT(libGAP_AWP_FIRST_FREE) );

    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoFreeGroupElements() . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "objfgelm",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoFreeGroupElements ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  objfgelm.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
