/****************************************************************************
**
*W  gap.c                       GAP source                       Frank Celler
*W                                                         & Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the various read-eval-print loops and  related  stuff.
*/
#include        <stdio.h>
#include        <assert.h>
#include        <string.h>              /* memcpy */
#include        <stdlib.h>

#include        "system.h"              /* system dependent part           */

#ifdef HAVE_SYS_STAT_H
#include        <sys/stat.h>
#endif

#include <unistd.h> /* move this and wrap execvp later */

extern char * libGAP_In;

#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "read.h"                /* reader                          */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "integer.h"             /* integers                        */
#include        "rational.h"            /* rationals                       */
#include        "cyclotom.h"            /* cyclotomics                     */
#include        "finfield.h"            /* finite fields and ff elements   */

#include        "bool.h"                /* booleans                        */
#include        "macfloat.h"            /* machine doubles                 */
#include        "permutat.h"            /* permutations                    */
#include        "trans.h"               /* transformations                 */
#include        "pperm.h"               /* partial perms                   */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listoper.h"            /* operations for generic lists    */
#include        "listfunc.h"            /* functions for generic lists     */
#include        "plist.h"               /* plain lists                     */
#include        "set.h"                 /* plain sets                      */
#include        "vector.h"              /* functions for plain vectors     */
#include        "vecffe.h"              /* functions for fin field vectors */
#include        "blister.h"             /* boolean lists                   */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */
#include        "vecgf2.h"              /* functions for GF2 vectors       */
#include        "vec8bit.h"             /* functions for other compressed
                                           GF(q) vectors                   */
#include        "objfgelm.h"            /* objects of free groups          */
#include        "objpcgel.h"            /* objects of polycyclic groups    */
#include        "objscoll.h"            /* single collector                */
#include        "objccoll.h"            /* combinatorial collector         */
#include        "objcftl.h"             /* from the left collect           */

#include        "dt.h"                  /* deep thought                    */
#include        "dteval.h"              /* deep thought evaluation          */

#include        "sctable.h"             /* structure constant table        */
#include        "costab.h"              /* coset table                     */
#include        "tietze.h"              /* tietze helper functions         */

#include        "code.h"                /* coder                           */

#include        "vars.h"                /* variables                       */
#include        "exprs.h"               /* expressions                     */
#include        "stats.h"               /* statements                      */
#include        "funcs.h"               /* functions                       */

#include        "intrprtr.h"            /* interpreter                     */

#include        "compiler.h"            /* compiler                        */

#include        "compstat.h"            /* statically linked modules       */

#include        "saveload.h"            /* saving and loading              */

#include        "streams.h"             /* streams package                 */
#include        "sysfiles.h"            /* file input/output               */
#include        "weakptr.h"             /* weak pointers                   */

#ifdef GAPMPI
#include        "gapmpi.h"              /* ParGAP/MPI                      */
#endif

#include        "intfuncs.h"
#include        "iostream.h"

/****************************************************************************
**

*V  Last  . . . . . . . . . . . . . . . . . . . . . . global variable  'last'
**
**  'Last',  'Last2', and 'Last3'  are the  global variables 'last', 'last2',
**  and  'last3', which are automatically  assigned  the result values in the
**  main read-eval-print loop.
*/
libGAP_UInt libGAP_Last;


/****************************************************************************
**
*V  Last2 . . . . . . . . . . . . . . . . . . . . . . global variable 'last2'
*/
libGAP_UInt libGAP_Last2;


/****************************************************************************
**
*V  Last3 . . . . . . . . . . . . . . . . . . . . . . global variable 'last3'
*/
libGAP_UInt libGAP_Last3;


/****************************************************************************
**
*V  Time  . . . . . . . . . . . . . . . . . . . . . . global variable  'time'
**
**  'Time' is the global variable 'time', which is automatically assigned the
**  time the last command took.
*/
libGAP_UInt libGAP_Time;


/****************************************************************************
**
*F  ViewObjHandler  . . . . . . . . . handler to view object and catch errors
**
**  This is the function actually called in Read-Eval-View loops.
**  We might be in trouble if the library has not (yet) loaded and so ViewObj
**  is not yet defined, or the fallback methods not yet installed. To avoid
**  this problem, we check, and use PrintObj if there is a problem
**
**  We also install a hook to use the GAP level function 'CustomView' if
**  it exists. This can for example be used to restrict the amount of output
**  or to show long output in a pager or .....
**  
**  This function also supplies the \n after viewing.
*/
libGAP_UInt libGAP_ViewObjGVar;
libGAP_UInt libGAP_CustomViewGVar;

void libGAP_ViewObjHandler ( libGAP_Obj obj )
{
  volatile libGAP_Obj        func;
  volatile libGAP_Obj        cfunc;
  libGAP_syJmp_buf             readJmpError;

  /* get the functions                                                   */
  func = libGAP_ValAutoGVar(libGAP_ViewObjGVar);
  cfunc = libGAP_ValAutoGVar(libGAP_CustomViewGVar);

  /* if non-zero use this function, otherwise use `PrintObj'             */
  memcpy( readJmpError, libGAP_ReadJmpError, sizeof(libGAP_syJmp_buf) );
  if ( ! libGAP_READ_ERROR() ) {
    if ( cfunc != 0 && libGAP_TNUM_OBJ(cfunc) == libGAP_T_FUNCTION ) {
      libGAP_CALL_1ARGS(cfunc, obj);
    }
    else if ( func != 0 && libGAP_TNUM_OBJ(func) == libGAP_T_FUNCTION ) {
      libGAP_ViewObj(obj);
    }
    else {
      libGAP_PrintObj( obj );
    }
    libGAP_Pr( "\n", 0L, 0L );
    memcpy( libGAP_ReadJmpError, readJmpError, sizeof(libGAP_syJmp_buf) );
  }
  else {
    memcpy( libGAP_ReadJmpError, readJmpError, sizeof(libGAP_syJmp_buf) );
  }
}


/****************************************************************************
**
*F  main( <argc>, <argv> )  . . . . . . .  main program, read-eval-print loop
*/
libGAP_Obj libGAP_AtExitFunctions;

libGAP_Obj libGAP_AlternativeMainLoop;

libGAP_UInt libGAP_SaveOnExitFileGVar;

libGAP_UInt libGAP_QUITTINGGVar;

libGAP_Obj libGAP_OnGapPromptHook;

libGAP_Obj libGAP_ErrorHandler;               /* not yet settable from GAP level */


typedef struct {
    const libGAP_Char *                name;
    libGAP_Obj *                       address;
} libGAP_StructImportedGVars;

#ifndef libGAP_MAX_IMPORTED_GVARS
#define libGAP_MAX_IMPORTED_GVARS      1024
#endif

static libGAP_StructImportedGVars libGAP_ImportedGVars[libGAP_MAX_IMPORTED_GVARS];
libGAP_Int libGAP_NrImportedGVars;

static libGAP_StructImportedGVars libGAP_ImportedFuncs[libGAP_MAX_IMPORTED_GVARS];
libGAP_Int libGAP_NrImportedFuncs;

/* int restart_argc; 
   char **restart_argv; */

char *libGAP_original_argv0;
static char **libGAP_sysargv;
char **libGAP_sysenviron;

/* 
syJmp_buf SyRestartBuf;
*/

libGAP_Obj libGAP_ShellContext = 0;
libGAP_Obj libGAP_BaseShellContext = 0;
libGAP_UInt libGAP_ShellContextDepth;


libGAP_Obj libGAP_Shell ( libGAP_Obj context, 
            libGAP_UInt canReturnVoid,
            libGAP_UInt canReturnObj,
            libGAP_UInt lastDepth,
            libGAP_UInt setTime,
            libGAP_Char *prompt,
            libGAP_Obj preCommandHook,
            libGAP_UInt catchQUIT,
            libGAP_Char *inFile,
            libGAP_Char *outFile)
{
  libGAP_UInt time = 0;
  libGAP_UInt status;
  libGAP_UInt oldindent;
  libGAP_UInt oldPrintDepth;
  libGAP_Obj res;
  libGAP_Obj oldShellContext;
  libGAP_Obj oldBaseShellContext;
  libGAP_Int oldRecursionDepth;
  oldShellContext = libGAP_ShellContext;
  libGAP_ShellContext = context;
  oldBaseShellContext = libGAP_BaseShellContext;
  libGAP_BaseShellContext = context;
  libGAP_ShellContextDepth = 0;
  oldRecursionDepth = libGAP_RecursionDepth;
  
  /* read-eval-print loop                                                */
  if (!libGAP_OpenOutput(outFile))
    libGAP_ErrorMayQuit("SHELL: can't open outfile %s",(libGAP_Int)outFile,0);

  if(!libGAP_OpenInput(inFile))
    {
      libGAP_CloseOutput();
      libGAP_ErrorMayQuit("SHELL: can't open infile %s",(libGAP_Int)inFile,0);
    }
  
  oldPrintDepth = libGAP_PrintObjDepth;
  libGAP_PrintObjDepth = 0;
  oldindent = libGAP_Output->indent;
  libGAP_Output->indent = 0;

  while ( 1 ) {

    /* start the stopwatch                                             */
    if (setTime)
      time = libGAP_SyTime();

    /* read and evaluate one command                                   */
    libGAP_Prompt = prompt;
    libGAP_ClearError();
    libGAP_PrintObjDepth = 0;
    libGAP_Output->indent = 0;
    libGAP_RecursionDepth = 0;
      
    /* here is a hook: */
    if (preCommandHook) {
      if (!libGAP_IS_FUNC(preCommandHook))
        {
                  libGAP_Pr("#E CommandHook was non-function, ignoring\n",0L,0L);
        }
      else
        {
          libGAP_Call0ArgsInNewReader(preCommandHook);
          /* Recover from a potential break loop: */
          libGAP_Prompt = prompt;
          libGAP_ClearError();
        }
    }

    /* now  read and evaluate and view one command  */
    status = libGAP_ReadEvalCommand(libGAP_ShellContext);
    if (libGAP_UserHasQUIT)
      break;


    /* handle ordinary command                                         */
    if ( status == libGAP_STATUS_END && libGAP_ReadEvalResult != 0 ) {

      /* remember the value in 'last'    */
      if (lastDepth >= 3)
        libGAP_AssGVar( libGAP_Last3, libGAP_VAL_GVAR( libGAP_Last2 ) );
      if (lastDepth >= 2)
        libGAP_AssGVar( libGAP_Last2, libGAP_VAL_GVAR( libGAP_Last  ) );
      if (lastDepth >= 1)
        libGAP_AssGVar( libGAP_Last,  libGAP_ReadEvalResult   );

      /* print the result                                            */
      if ( ! libGAP_DualSemicolon ) {
        libGAP_ViewObjHandler( libGAP_ReadEvalResult );
      }
            
    }

    /* handle return-value or return-void command                      */
    else if (status & libGAP_STATUS_RETURN_VAL) 
      if(canReturnObj)
        break;
      else
        libGAP_Pr( "'return <object>' cannot be used in this read-eval-print loop\n",
            0L, 0L );

    else if (status & libGAP_STATUS_RETURN_VOID) 
      if(canReturnVoid ) 
        break;
      else
        libGAP_Pr( "'return' cannot be used in this read-eval-print loop\n",
            0L, 0L );
    
    /* handle quit command or <end-of-file>                            */
    else if ( status & (libGAP_STATUS_EOF | libGAP_STATUS_QUIT ) ) {
      libGAP_RecursionDepth = 0;
      libGAP_UserHasQuit = 1;
      break;
    }
        
    /* handle QUIT */
    else if (status & (libGAP_STATUS_QQUIT)) {
      libGAP_UserHasQUIT = 1;
      break;
    }
        
    /* stop the stopwatch                                          */
    if (setTime)
      libGAP_AssGVar( libGAP_Time, libGAP_INTOBJ_INT( libGAP_SyTime() - time ) );

    if (libGAP_UserHasQuit)
      {
        libGAP_FlushRestOfInputLine();
        libGAP_UserHasQuit = 0;        /* quit has done its job if we are here */
      }

  }
  
  libGAP_PrintObjDepth = oldPrintDepth;
  libGAP_Output->indent = oldindent;
  libGAP_CloseInput();
  libGAP_CloseOutput();
  libGAP_BaseShellContext = oldBaseShellContext;
  libGAP_ShellContext = oldShellContext;
  libGAP_RecursionDepth = oldRecursionDepth;
  if (libGAP_UserHasQUIT)
    {
      if (catchQUIT)
        {
          libGAP_UserHasQUIT = 0;
          libGAP_MakeReadWriteGVar(libGAP_QUITTINGGVar);
          libGAP_AssGVar(libGAP_QUITTINGGVar, libGAP_True);
          libGAP_MakeReadOnlyGVar(libGAP_QUITTINGGVar);
          return libGAP_Fail;
        }
      else
        libGAP_ReadEvalError();
    }

  if (status & (libGAP_STATUS_EOF | libGAP_STATUS_QUIT | libGAP_STATUS_QQUIT))
    {
      return libGAP_Fail;
    }
  if (status & libGAP_STATUS_RETURN_VOID)
    {
      res = libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY,0);
      libGAP_SET_LEN_PLIST(res,0);
      return res;
    }
  if (status & libGAP_STATUS_RETURN_VAL)
    {
      res = libGAP_NEW_PLIST(libGAP_T_PLIST_HOM,1);
      libGAP_SET_LEN_PLIST(res,1);
      libGAP_SET_ELM_PLIST(res,1,libGAP_ReadEvalResult);
      return res;
    }
  assert(0); 
  return (libGAP_Obj) 0;
}



libGAP_Obj libGAP_FuncSHELL (libGAP_Obj self, libGAP_Obj args)
{
  libGAP_Obj context = 0;
  libGAP_UInt canReturnVoid = 0;
  libGAP_UInt canReturnObj = 0;
  libGAP_Int lastDepth = 0;
  libGAP_UInt setTime = 0;
  libGAP_Obj prompt = 0;
  libGAP_Obj preCommandHook = 0;
  libGAP_Obj infile;
  libGAP_Obj outfile;
  libGAP_Obj res;
  libGAP_Char promptBuffer[81];
  libGAP_UInt catchQUIT = 0;
  
  if (!libGAP_IS_PLIST(args) || libGAP_LEN_PLIST(args) != 10)
    libGAP_ErrorMayQuit("SHELL takes 10 arguments",0,0);
  
  context = libGAP_ELM_PLIST(args,1);
  if (libGAP_TNUM_OBJ(context) != libGAP_T_LVARS)
    libGAP_ErrorMayQuit("SHELL: 1st argument should be a local variables bag",0,0);
  
  if (libGAP_ELM_PLIST(args,2) == libGAP_True)
    canReturnVoid = 1;
  else if (libGAP_ELM_PLIST(args,2) == libGAP_False)
    canReturnVoid = 0;
  else
    libGAP_ErrorMayQuit("SHELL: 2nd argument (can return void) should be true or false",0,0);

  if (libGAP_ELM_PLIST(args,3) == libGAP_True)
    canReturnObj = 1;
  else if (libGAP_ELM_PLIST(args,3) == libGAP_False)
    canReturnObj = 0;
  else
    libGAP_ErrorMayQuit("SHELL: 3rd argument (can return object) should be true or false",0,0);
  
  if (!libGAP_IS_INTOBJ(libGAP_ELM_PLIST(args,4)))
    libGAP_ErrorMayQuit("SHELL: 4th argument (last depth) should be a small integer",0,0);
  lastDepth = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(args,4));
  if (lastDepth < 0 )
    {
      libGAP_Pr("#W SHELL: negative last depth treated as zero",0,0);
      lastDepth = 0;
    }
  else if (lastDepth > 3 )
    {
      libGAP_Pr("#W SHELL: last depth greater than 3 treated as 3",0,0);
      lastDepth = 3;
    }

  if (libGAP_ELM_PLIST(args,5) == libGAP_True)
    setTime = 1;
  else if (libGAP_ELM_PLIST(args,5) == libGAP_False)
    setTime = 0;
  else
    libGAP_ErrorMayQuit("SHELL: 5th argument (set time) should be true or false",0,0);
  
  prompt = libGAP_ELM_PLIST(args,6);
  if (!libGAP_IsStringConv(prompt) || libGAP_GET_LEN_STRING(prompt) > 80)
    libGAP_ErrorMayQuit("SHELL: 6th argument (prompt) must be a string of length at most 80 characters",0,0);
  promptBuffer[0] = '\0';
  libGAP_strlcat(promptBuffer, libGAP_CSTR_STRING(prompt), sizeof(promptBuffer));

  preCommandHook = libGAP_ELM_PLIST(args,7);
 
  if (preCommandHook == libGAP_False)
    preCommandHook = 0;
  else if (!libGAP_IS_FUNC(preCommandHook))
    libGAP_ErrorMayQuit("SHELL: 7th argument (preCommandHook) must be function or false",0,0);

  
  infile = libGAP_ELM_PLIST(args,8);
  if (!libGAP_IsStringConv(infile))
    libGAP_ErrorMayQuit("SHELL: 8th argument (infile) must be a string",0,0);

  outfile = libGAP_ELM_PLIST(args,9);
  if (!libGAP_IsStringConv(infile))
    libGAP_ErrorMayQuit("SHELL: 9th argument (outfile) must be a string",0,0);

  if (libGAP_ELM_PLIST(args,10) == libGAP_True)
    catchQUIT = 1;
  else if (libGAP_ELM_PLIST(args,10) == libGAP_False)
    catchQUIT = 0;
  else
    libGAP_ErrorMayQuit("SHELL: 10th argument (catch QUIT) should be true or false",0,0);



  res =  libGAP_Shell(context, canReturnVoid, canReturnObj, lastDepth, setTime, promptBuffer, preCommandHook, catchQUIT,
               libGAP_CSTR_STRING(infile), libGAP_CSTR_STRING(outfile));

  libGAP_UserHasQuit = 0;
  return res;
}
#ifdef HAVE_REALPATH

static void libGAP_StrAppend(char **st, const char *st2)
{
    libGAP_Int len,len2;
    if (*st == NULL)
        len = 0;
    else
        len = strlen(*st);
    len2 = strlen(st2);
    *st = realloc(*st,len+len2+1);
    if (*st == NULL) {
        printf("Extremely unexpected out of memory error. Giving up.\n");
        exit(1);
    }
    memcpy(*st + len, st2, len2);
}

static void libGAP_DoFindMyself(char *myself, char **mypath, char **gappath)
{
    char *tmppath;
    char *p;

    /* First we find our own position in the filesystem: */
    *mypath = realpath(myself,NULL);
    if (*mypath == NULL) {
        printf("Could not determine my own path, giving up.\n");
        exit(-1);
    }
    tmppath = NULL;
    libGAP_StrAppend(&tmppath,*mypath);
    p = tmppath+strlen(tmppath);
    while (*p != '/') p--;
    *p = 0;
    libGAP_StrAppend(&tmppath,"/../..");
    *gappath = realpath(tmppath,NULL);
    if (*gappath == NULL) {
        printf("Could not determine GAP path, giving up.\n");
        exit(-2);
    }
    free(tmppath);
}


int libGAP_DoCreateStartupScript(int argc, char *argv[], int withws)
{
    /* This is used to create a startup shell script, possibly using
     * a saved workspace in a standard location. */
    /* We can use malloc/realloc here arbitrarily since this GAP
     * process will never start its memory manager before terminating! */
    char *mypath;
    char *gappath;
    char *tmppath;
    char *p;
    FILE *f;
    int i;

    libGAP_DoFindMyself(argv[0],&mypath,&gappath);

    /* Now write out the startup script: */
    f = fopen(argv[2],"w");
    if (f == NULL) {
        printf("Could not write startup script to\n  %s\ngiving up.\n",argv[2]);
        return -3;
    }
    fprintf(f,"#!/bin/sh\n");
    fprintf(f,"# Created by %s\n",mypath);
    fprintf(f,"GAP_DIR=\"%s\"\n",gappath);
    fprintf(f,"GAP_PRG=\"%s\"\n",mypath);
    fprintf(f,"GAP_ARCH=\"%s\"\n",SYS_ARCH);
    tmppath = NULL;
    libGAP_StrAppend(&tmppath,SYS_ARCH);
    p = tmppath;
    while (*p != 0 && *p != '/') p++;
    *p++ = 0;
    fprintf(f,"GAP_ARCH_SYS=\"%s\"\n",tmppath);
    fprintf(f,"GAP_ARCH_ABI=\"%s\"\n",p);	// FIXME: WRONG
    fprintf(f,"exec %s -l %s",mypath,gappath);
    if (withws) {
        tmppath[0] = 0;
        libGAP_StrAppend(&tmppath,mypath);
        p = tmppath+strlen(tmppath);
        while (*p != '/') p--;
        p[1] = 0;
        libGAP_StrAppend(&tmppath,"workspace.gap");
        fprintf(f," -L %s",tmppath);
    }
    for (i = 3;i < argc;i++) fprintf(f," %s",argv[i]);
    fprintf(f," \"$@\"\n");
    fclose(f);
#ifdef HAVE_CHMOD
    chmod(argv[2],S_IRUSR | S_IWUSR | S_IXUSR |
                  S_IRGRP | S_IWGRP | S_IXGRP |
                  S_IROTH | S_IXOTH);
#else
    printf("Warning: Do not have chmod to make script executable!\n");
#endif
    free(tmppath);
    free(mypath);
    free(gappath);
    return 0;
}

int libGAP_DoCreateWorkspace(char *myself)
{
    /* This is used to create an architecture-dependent saved
     * workspace in a standard location. */
    char *mypath;
    char *gappath;
    char *command;
    char *tmppath;
    char *p;
    FILE *f;

    libGAP_DoFindMyself(myself,&mypath,&gappath);

    /* Now we create a saved workspace: */
    printf("Creating workspace...\n");
    command = NULL;
    libGAP_StrAppend(&command,mypath);
    libGAP_StrAppend(&command," -N -r");
    libGAP_StrAppend(&command," -l ");
    libGAP_StrAppend(&command,gappath);

    tmppath = NULL;
    libGAP_StrAppend(&tmppath,mypath);
    p = tmppath+strlen(tmppath);
    while (*p != '/') p--;
    p[1] = 0;
    libGAP_StrAppend(&tmppath,"workspace.gap");

    /* Now to the action: */
    f = popen(command,"w");
    if (f == NULL) {
        printf("Could not start myself to save workspace, giving up.\n");
        return -6;
    }
    fprintf(f,"??blabla\n");
    fprintf(f,"SaveWorkspace(\"%s\");\n",tmppath);
    fprintf(f,"quit;\n");
    fflush(f);
    pclose(f);
    printf("\nDone creating workspace in\n  %s\n",tmppath);

    free(tmppath);
    free(command);
    free(gappath);
    free(mypath);

    return 0;
}

int libGAP_DoFixGac(char *myself)
{
    char *mypath;
    char *gappath;
    FILE *f;
    char *gacpath;
    char *gapbin;
    char *newpath;
    char *p,*q,*r;
    char *buf,*buf2;
    size_t len,written;

    libGAP_DoFindMyself(myself,&mypath,&gappath);
    gacpath = NULL;
    libGAP_StrAppend(&gacpath,mypath);
    p = gacpath + strlen(gacpath);
    while (*p != '/') p--;
    *p = 0;
    gapbin = NULL;
    libGAP_StrAppend(&gapbin,gacpath);
    libGAP_StrAppend(&gacpath,"/gac");
    newpath = NULL;
    libGAP_StrAppend(&newpath,gacpath);
    libGAP_StrAppend(&newpath,".new");
    f = fopen(gacpath,"r");
    if (f == NULL) {
        printf("Could not open gac. Giving up.\n");
        return -7;
    }
    buf = malloc(65536);
    buf2 = malloc(65536+strlen(gapbin)+10);
    if (buf == NULL || buf2 == NULL) {
        printf("Could not allocate 128kB of memory. Giving up.\n");
        return -8;
    }
    len = fread(buf,1,65534,f);
    fclose(f);

    /* Now manipulate it: */
    p = buf;
    p[len] = 0;
    p[len+1] = 0;
    q = buf2;
    while (*p) {
        if (!strncmp(p,"gap_bin=",8)) {
            while (*p != '\n' && *p != 0) p++;
            *q++ = 'g'; *q++ = 'a'; *q++ = 'p'; *q++ = '_';
            *q++ = 'b'; *q++ = 'i'; *q++ = 'n'; *q++ = '=';
            r = gapbin;
            while (*r) *q++ = *r++;
            *q++ = '\n';
        } else {
            while (*p != '\n' && *p != 0) *q++ = *p++;
            *q++ = *p++;
        }
    }
    len = q - buf2;

    f = fopen(newpath,"w");
    if (f == NULL) {
        printf("Could not open gac.new. Giving up.\n");
        return -9;
    }
    written = fwrite(buf2,1,len,f);
    if (written < len) {
        printf("Could not write gac.new. Giving up.\n");
        fclose(f);
        return -10;
    }
    if (fclose(f) < 0) {
        printf("Could not close gac.new. Giving up.\n");
        fclose(f);
        return -11;
    }
    if (rename(newpath,gacpath) < 0) {
        printf("Could not replace gac with new version. Giving up.\n");
        return -12;
    }
    return 0;
}
#endif

int libGAP_gap_main_loop (
          int                 argc,
          char *              argv [],
          char *              environ [] )
{
  libGAP_UInt                type;                   /* result of compile       */
  libGAP_Obj                 func;                   /* function (compiler)     */
  libGAP_Int4                crc;                    /* crc of file to compile  */

#ifdef HAVE_REALPATH
  if (argc >= 3 && !strcmp(argv[1],"--createstartupscript")) {
      return libGAP_DoCreateStartupScript(argc,argv,0);
  }
  if (argc >= 3 && !strcmp(argv[1],"--createstartupscriptwithws")) {
      return libGAP_DoCreateStartupScript(argc,argv,1);
  }
  if (argc >= 2 && !strcmp(argv[1],"--createworkspace")) {
      return libGAP_DoCreateWorkspace(argv[0]);
  }
  if (argc >= 2 && !strcmp(argv[1],"--fixgac")) {
      return libGAP_DoFixGac(argv[0]);
  }
#endif
  
  libGAP_original_argv0 = argv[0];
  libGAP_sysargv = argv;
  libGAP_sysenviron = environ;
  
  /* prepare for a possible restart 
  if (setjmp(SyRestartBuf))
    {
      argc = restart_argc;
      argv = restart_argv;
    }
    `*/

  /* Initialize assorted variables in this file */
  /*   BreakOnError = 1;
       ErrorCount = 0; */
  libGAP_NrImportedGVars = 0;
  libGAP_NrImportedFuncs = 0;
  libGAP_ErrorHandler = (libGAP_Obj) 0;
  libGAP_UserHasQUIT = 0;
  libGAP_UserHasQuit = 0;
    
  /* initialize everything and read init.g which runs the GAP session */
  libGAP_InitializeGap( &argc, argv );
  if (!libGAP_UserHasQUIT) {           /* maybe the user QUIT from the initial
                                   read of init.g  somehow*/
    /* maybe compile in which case init.g got skipped */
    if ( libGAP_SyCompilePlease ) {
      if ( ! libGAP_OpenInput(libGAP_SyCompileInput) ) {
        libGAP_SyExit(1);
      }
      func = libGAP_READ_AS_FUNC();
      crc  = libGAP_SyGAPCRC(libGAP_SyCompileInput);
      if (strlen(libGAP_SyCompileOptions) != 0)
        libGAP_SetCompileOpts(libGAP_SyCompileOptions);
      type = libGAP_CompileFunc(
                         libGAP_SyCompileOutput,
                         func,
                         libGAP_SyCompileName,
                         crc,
                         libGAP_SyCompileMagic1 );
      if ( type == 0 )
        libGAP_SyExit( 1 );
      libGAP_SyExit( 0 );
    }
  }
  libGAP_SyExit(0);
  return 0;
}

/****************************************************************************
**
*F  FuncRESTART_GAP( <self>, <cmdline> ) . . . . . . . .  restart gap
**
*/

libGAP_Char *libGAP_restart_argv_buffer[1000];

libGAP_Obj libGAP_FuncRESTART_GAP( libGAP_Obj self, libGAP_Obj cmdline )
{
  libGAP_Char *s, *f,  **v;
  while (!libGAP_IsStringConv(cmdline))
    {
      cmdline = libGAP_ErrorReturnObj("RESTART_GAP: <cmdline> must be a string, not a %s",
                               (libGAP_Int) libGAP_TNAM_OBJ(cmdline), (libGAP_Int) 0,
                               "You can resturn a string to continue");
    }
  s = libGAP_CSTR_STRING(cmdline);
  /* Pr("%s\n",(Int)s, 0); */
  f = s;
  v = libGAP_restart_argv_buffer;
  while (*s) {
    *v++ = s;
    while (*s && !libGAP_IsSpace(*s))
      s++;
    while (libGAP_IsSpace(*s))
      *s++ = '\0';
  }
  *v = (libGAP_Char *)0;
  /*  restart_argc = ct;
      restart_argv = restart_argv_buffer; */
  /* FinishBags(); */
  execvp(f,libGAP_restart_argv_buffer);
  /*  longjmp(SyRestartBuf,1); */
  return libGAP_Fail; /* shouldn't normally get here */
}



/****************************************************************************
**
*F  FuncID_FUNC( <self>, <val1> ) . . . . . . . . . . . . . . . return <val1>
*/
libGAP_Obj libGAP_FuncID_FUNC (
                 libGAP_Obj                 self,
                 libGAP_Obj                 val1 )
{
  return val1;
}


/****************************************************************************
**
*F  FuncRuntime( <self> ) . . . . . . . . . . . . internal function 'Runtime'
**
**  'FuncRuntime' implements the internal function 'Runtime'.
**
**  'Runtime()'
**
**  'Runtime' returns the time spent since the start of GAP in  milliseconds.
**  How much time execution of statements take is of course system dependent.
**  The accuracy of this number is also system dependent.
*/
libGAP_Obj libGAP_FuncRuntime (
                 libGAP_Obj                 self )
{
  return libGAP_INTOBJ_INT( libGAP_SyTime() );
}



libGAP_Obj libGAP_FuncRUNTIMES( libGAP_Obj     self)
{
  libGAP_Obj    res;
  res = libGAP_NEW_PLIST(libGAP_T_PLIST, 4);
  libGAP_SET_LEN_PLIST(res, 4);
  libGAP_SET_ELM_PLIST(res, 1, libGAP_INTOBJ_INT( libGAP_SyTime() ));
  libGAP_SET_ELM_PLIST(res, 2, libGAP_INTOBJ_INT( libGAP_SyTimeSys() ));
  libGAP_SET_ELM_PLIST(res, 3, libGAP_INTOBJ_INT( libGAP_SyTimeChildren() ));
  libGAP_SET_ELM_PLIST(res, 4, libGAP_INTOBJ_INT( libGAP_SyTimeChildrenSys() ));
  return res;
   
}



/****************************************************************************
**
*F  FuncSizeScreen( <self>, <args> )  . . . .  internal function 'SizeScreen'
**
**  'FuncSizeScreen'  implements  the  internal  function 'SizeScreen' to get
**  or set the actual screen size.
**
**  'SizeScreen()'
**
**  In this form 'SizeScreen'  returns the size of the  screen as a list with
**  two  entries.  The first is  the length of each line,  the  second is the
**  number of lines.
**
**  'SizeScreen( [ <x>, <y> ] )'
**
**  In this form 'SizeScreen' sets the size of the screen.  <x> is the length
**  of each line, <y> is the number of lines.  Either value may  be  missing,
**  to leave this value unaffected.  Note that those parameters can  also  be
**  set with the command line options '-x <x>' and '-y <y>'.
*/
libGAP_Obj libGAP_FuncSizeScreen (
                    libGAP_Obj                 self,
                    libGAP_Obj                 args )
{
  libGAP_Obj                 size;           /* argument and result list        */
  libGAP_Obj                 elm;            /* one entry from size             */
  libGAP_UInt                len;            /* length of lines on the screen   */
  libGAP_UInt                nr;             /* number of lines on the screen   */

  /* check the arguments                                                 */
  while ( ! libGAP_IS_SMALL_LIST(args) || 1 < libGAP_LEN_LIST(args) ) {
    args = libGAP_ErrorReturnObj(
                          "Function: number of arguments must be 0 or 1 (not %d)",
                          libGAP_LEN_LIST(args), 0L,
                          "you can replace the argument list <args> via 'return <args>;'" );
  }

  /* get the arguments                                                   */
  if ( libGAP_LEN_LIST(args) == 0 ) {
    size = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST( size, 0 );
  }

  /* otherwise check the argument                                        */
  else {
    size = libGAP_ELM_LIST( args, 1 );
    while ( ! libGAP_IS_SMALL_LIST(size) || 2 < libGAP_LEN_LIST(size) ) {
      size = libGAP_ErrorReturnObj(
                            "SizeScreen: <size> must be a list of length 2",
                            0L, 0L,
                            "you can replace <size> via 'return <size>;'" );
    }
  }

  /* extract the length                                                  */
  if ( libGAP_LEN_LIST(size) < 1 || libGAP_ELM0_LIST(size,1) == 0 ) {
    len = 0;
  }
  else {
    elm = libGAP_ELMW_LIST(size,1);
    while ( libGAP_TNUM_OBJ(elm) != libGAP_T_INT ) {
      elm = libGAP_ErrorReturnObj(
                           "SizeScreen: <x> must be an integer",
                           0L, 0L,
                           "you can replace <x> via 'return <x>;'" );
    }
    len = libGAP_INT_INTOBJ( elm );
    if ( len < 20  )  len = 20;
    if ( libGAP_MAXLENOUTPUTLINE < len )  len = libGAP_MAXLENOUTPUTLINE;
  }

  /* extract the number                                                  */
  if ( libGAP_LEN_LIST(size) < 2 || libGAP_ELM0_LIST(size,2) == 0 ) {
    nr = 0;
  }
  else {
    elm = libGAP_ELMW_LIST(size,2);
    while ( libGAP_TNUM_OBJ(elm) != libGAP_T_INT ) {
      elm = libGAP_ErrorReturnObj(
                           "SizeScreen: <y> must be an integer",
                           0L, 0L,
                           "you can replace <y> via 'return <y>;'" );
    }
    nr = libGAP_INT_INTOBJ( elm );
    if ( nr < 10 )  nr = 10;
  }

  /* set length and number                                               */
  if (len != 0)
    {
      libGAP_SyNrCols = len;
      libGAP_SyNrColsLocked = 1;
    }
  if (nr != 0)
    {
      libGAP_SyNrRows = nr;
      libGAP_SyNrRowsLocked = 1;
    }

  /* make and return the size of the screen                              */
  size = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
  libGAP_SET_LEN_PLIST( size, 2 );
  libGAP_SET_ELM_PLIST( size, 1, libGAP_INTOBJ_INT(libGAP_SyNrCols) );
  libGAP_SET_ELM_PLIST( size, 2, libGAP_INTOBJ_INT(libGAP_SyNrRows)  );
  return size;

}


/****************************************************************************
**
*F  FuncWindowCmd( <self>, <args> ) . . . . . . . .  execute a window command
*/
static libGAP_Obj libGAP_WindowCmdString;

libGAP_Obj libGAP_FuncWindowCmd (
                   libGAP_Obj              self,
                   libGAP_Obj             args )
{
  libGAP_Obj             tmp;
  libGAP_Obj               list;
  libGAP_Int             len;
  libGAP_Int             n,  m;
  libGAP_Int             i;
  libGAP_Char *          ptr;
  libGAP_Char *          qtr;

  /* check arguments                                                     */
  while ( ! libGAP_IS_SMALL_LIST(args) ) {
    args = libGAP_ErrorReturnObj( "argument list must be a list (not a %s)",
                           (libGAP_Int)libGAP_TNAM_OBJ(args), 0L,
                           "you can replace the argument list <args> via 'return <args>;'" );

  }
  tmp = libGAP_ELM_LIST(args,1);
  while ( ! libGAP_IsStringConv(tmp) || 3 != libGAP_LEN_LIST(tmp) ) {
    while ( ! libGAP_IsStringConv(tmp) ) {
      tmp = libGAP_ErrorReturnObj( "<cmd> must be a string (not a %s)",
                            (libGAP_Int)libGAP_TNAM_OBJ(tmp), 0L,
                            "you can replace <cmd> via 'return <cmd>;'" );
    }
    if ( 3 != libGAP_LEN_LIST(tmp) ) {
      tmp = libGAP_ErrorReturnObj( "<cmd> must be a string of length 3",
                            0L, 0L,
                            "you can replace <cmd> via 'return <cmd>;'" );
    }
  }

  /* compute size needed to store argument string                        */
  len = 13;
  for ( i = 2;  i <= libGAP_LEN_LIST(args);  i++ )
    {
      tmp = libGAP_ELM_LIST( args, i );
      while ( libGAP_TNUM_OBJ(tmp) != libGAP_T_INT && ! libGAP_IsStringConv(tmp) ) {
        tmp = libGAP_ErrorReturnObj(
                             "%d. argument must be a string or integer (not a %s)",
                             i, (libGAP_Int)libGAP_TNAM_OBJ(tmp),
                             "you can replace the argument <arg> via 'return <arg>;'" );
        libGAP_SET_ELM_PLIST( args, i, tmp );
      }
      if ( libGAP_TNUM_OBJ(tmp) == libGAP_T_INT )
        len += 12;
      else
        len += 12 + libGAP_LEN_LIST(tmp);
    }
  if ( libGAP_SIZE_OBJ(libGAP_WindowCmdString) <= len ) {
    libGAP_ResizeBag( libGAP_WindowCmdString, 2*len+1 );
  }

  /* convert <args> into an argument string                              */
  ptr  = (libGAP_Char*) libGAP_CSTR_STRING(libGAP_WindowCmdString);

  /* first the command name                                              */
  memcpy( ptr, libGAP_CSTR_STRING( libGAP_ELM_LIST(args,1) ), 3 + 1 );
  ptr += 3;

  /* and now the arguments                                               */
  for ( i = 2;  i <= libGAP_LEN_LIST(args);  i++ )
    {
      tmp = libGAP_ELM_LIST(args,i);

      if ( libGAP_TNUM_OBJ(tmp) == libGAP_T_INT ) {
        *ptr++ = 'I';
        m = libGAP_INT_INTOBJ(tmp);
        for ( m = (m<0)?-m:m;  0 < m;  m /= 10 )
          *ptr++ = (m%10) + '0';
        if ( libGAP_INT_INTOBJ(tmp) < 0 )
          *ptr++ = '-';
        else
          *ptr++ = '+';
      }
      else {
        *ptr++ = 'S';
        m = libGAP_LEN_LIST(tmp);
        for ( ; 0 < m;  m/= 10 )
          *ptr++ = (m%10) + '0';
        *ptr++ = '+';
        qtr = libGAP_CSTR_STRING(tmp);
        for ( m = libGAP_LEN_LIST(tmp);  0 < m;  m-- )
          *ptr++ = *qtr++;
      }
    }
  *ptr = 0;

  /* now call the window front end with the argument string              */
  qtr = libGAP_CSTR_STRING(libGAP_WindowCmdString);
  ptr = libGAP_SyWinCmd( qtr, strlen(qtr) );
  len = strlen(ptr);

  /* now convert result back into a list                                 */
  list = libGAP_NEW_PLIST( libGAP_T_PLIST, 11 );
  libGAP_SET_LEN_PLIST( list, 0 );
  i = 1;
  while ( 0 < len ) {
    if ( *ptr == 'I' ) {
      ptr++;
      for ( n=0,m=1; '0' <= *ptr && *ptr <= '9'; ptr++,m *= 10,len-- )
        n += (*ptr-'0') * m;
      if ( *ptr++ == '-' )
        n *= -1;
      len -= 2;
      libGAP_AssPlist( list, i, libGAP_INTOBJ_INT(n) );
    }
    else if ( *ptr == 'S' ) {
      ptr++;
      for ( n=0,m=1;  '0' <= *ptr && *ptr <= '9';  ptr++,m *= 10,len-- )
        n += (*ptr-'0') * m;
      ptr++; /* ignore the '+' */
      libGAP_C_NEW_STRING(tmp, n, ptr);
      ptr += n;
      len -= n+2;
      libGAP_AssPlist( list, i, tmp );
    }
    else {
      libGAP_ErrorQuit( "unknown return value '%s'", (libGAP_Int)ptr, 0 );
      return 0;
    }
    i++;
  }

  /* if the first entry is one signal an error */
  if ( libGAP_ELM_LIST(list,1) == libGAP_INTOBJ_INT(1) ) {
    libGAP_C_NEW_STRING_CONST(tmp, "window system: ");
    libGAP_SET_ELM_PLIST( list, 1, tmp );
    libGAP_SET_LEN_PLIST( list, i-1 );
    return libGAP_CALL_XARGS(libGAP_Error,list);
    /*     return FuncError( 0, list );*/
  }
  else {
    for ( m = 1;  m <= i-2;  m++ )
      libGAP_SET_ELM_PLIST( list, m, libGAP_ELM_PLIST(list,m+1) );
    libGAP_SET_LEN_PLIST( list, i-2 );
    return list;
  }
}


/****************************************************************************
**

*F * * * * * * * * * * * * * * error functions * * * * * * * * * * * * * * *
*/



/****************************************************************************
**

*F  FuncDownEnv( <self>, <level> )  . . . . . . . . .  change the environment
*/
libGAP_UInt libGAP_ErrorLevel;

libGAP_Obj  libGAP_ErrorLVars0;    
libGAP_Obj  libGAP_ErrorLVars;
libGAP_Int  libGAP_ErrorLLevel;

extern libGAP_Obj libGAP_BottomLVars;


void libGAP_DownEnvInner( libGAP_Int depth )
{
  /* if we really want to go up                                          */
  if ( depth < 0 && -libGAP_ErrorLLevel <= -depth ) {
    depth = 0;
    libGAP_ErrorLVars = libGAP_ErrorLVars0;
    libGAP_ErrorLLevel = 0;
    libGAP_ShellContextDepth = 0;
    libGAP_ShellContext = libGAP_BaseShellContext;
  }
  else if ( depth < 0 ) {
    depth = -libGAP_ErrorLLevel + depth;
    libGAP_ErrorLVars = libGAP_ErrorLVars0;
    libGAP_ErrorLLevel = 0;
    libGAP_ShellContextDepth = 0;
    libGAP_ShellContext = libGAP_BaseShellContext;
  }
  
  /* now go down                                                         */
  while ( 0 < depth
          && libGAP_ErrorLVars != libGAP_BottomLVars
          && libGAP_PTR_BAG(libGAP_ErrorLVars)[2] != libGAP_BottomLVars ) {
    libGAP_ErrorLVars = libGAP_PTR_BAG(libGAP_ErrorLVars)[2];
    libGAP_ErrorLLevel--;
    libGAP_ShellContext = libGAP_PTR_BAG(libGAP_ShellContext)[2];
    libGAP_ShellContextDepth--;
    depth--;
  }
}
  
libGAP_Obj libGAP_FuncDownEnv (
                 libGAP_Obj                 self,
                 libGAP_Obj                 args )
{
  libGAP_Int                 depth;

  if ( libGAP_LEN_LIST(args) == 0 ) {
    depth = 1;
  }
  else if ( libGAP_LEN_LIST(args) == 1 && libGAP_IS_INTOBJ( libGAP_ELM_PLIST(args,1) ) ) {
    depth = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( args, 1 ) );
  }
  else {
    libGAP_ErrorQuit( "usage: DownEnv( [ <depth> ] )", 0L, 0L );
    return 0;
  }
  if ( libGAP_ErrorLVars == 0 ) {
    libGAP_Pr( "not in any function\n", 0L, 0L );
    return 0;
  }

  libGAP_DownEnvInner( depth);

  /* return nothing                                                      */
  return 0;
}

libGAP_Obj libGAP_FuncUpEnv (
               libGAP_Obj                 self,
               libGAP_Obj                 args )
{
  libGAP_Int                 depth;
  if ( libGAP_LEN_LIST(args) == 0 ) {
    depth = 1;
  }
  else if ( libGAP_LEN_LIST(args) == 1 && libGAP_IS_INTOBJ( libGAP_ELM_PLIST(args,1) ) ) {
    depth = libGAP_INT_INTOBJ( libGAP_ELM_PLIST( args, 1 ) );
  }
  else {
    libGAP_ErrorQuit( "usage: UpEnv( [ <depth> ] )", 0L, 0L );
    return 0;
  }
  if ( libGAP_ErrorLVars == 0 ) {
    libGAP_Pr( "not in any function\n", 0L, 0L );
    return 0;
  }

  libGAP_DownEnvInner(-depth);
  return 0;
}


libGAP_Obj libGAP_FuncPrintExecutingStatement(libGAP_Obj self, libGAP_Obj context)
{
  libGAP_Obj currLVars = libGAP_CurrLVars;
  libGAP_Expr call;
  if (context == libGAP_BottomLVars)
    return (libGAP_Obj) 0;
  libGAP_SWITCH_TO_OLD_LVARS(context);
  call = libGAP_BRK_CALL_TO();
  if ( call == 0 ) {
    libGAP_Pr( "<compiled or corrupted statement> ", 0L, 0L );
  }
#if libGAP_T_PROCCALL_0ARGS
    else if ( libGAP_FIRST_STAT_TNUM <= libGAP_TNUM_STAT(call)
              && libGAP_TNUM_STAT(call)  <= libGAP_LAST_STAT_TNUM ) {
#else
     else if ( libGAP_TNUM_STAT(call)  <= libGAP_LAST_STAT_TNUM ) {
#endif
      libGAP_PrintStat( call );
    }
    else if ( libGAP_FIRST_EXPR_TNUM <= libGAP_TNUM_EXPR(call)
              && libGAP_TNUM_EXPR(call)  <= libGAP_LAST_EXPR_TNUM ) {
      libGAP_PrintExpr( call );
    }
    libGAP_SWITCH_TO_OLD_LVARS( currLVars );
    return (libGAP_Obj) 0;
}    

/****************************************************************************
**
*F  FuncCallFuncTrapError( <self>, <func> )
**
*/
  
/* syJmp_buf CatchBuffer; */
libGAP_Obj libGAP_ThrownObject = 0;

libGAP_Obj libGAP_FuncCALL_WITH_CATCH( libGAP_Obj self, libGAP_Obj func, libGAP_Obj args )
{
    libGAP_syJmp_buf readJmpError;
    libGAP_Obj plain_args;
    libGAP_Obj res;
    libGAP_Obj currLVars;
    libGAP_Obj result;
    libGAP_Int recursionDepth;
    libGAP_Stat currStat;
    if (!libGAP_IS_FUNC(func))
      libGAP_ErrorMayQuit("CALL_WITH_CATCH(<func>,<args>): <func> must be a function",0,0);
    if (!libGAP_IS_LIST(args))
      libGAP_ErrorMayQuit("CALL_WITH_CATCH(<func>,<args>): <args> must be a list",0,0);
    if (!libGAP_IS_PLIST(args))
      {
        plain_args = libGAP_SHALLOW_COPY_OBJ(args);
        libGAP_PLAIN_LIST(plain_args);
      }
    else 
      plain_args = args;
    memcpy((void *)&readJmpError, (void *)&libGAP_ReadJmpError, sizeof(libGAP_syJmp_buf));
    currLVars = libGAP_CurrLVars;
    currStat = libGAP_CurrStat;
    recursionDepth = libGAP_RecursionDepth;
    res = libGAP_NEW_PLIST(libGAP_T_PLIST_DENSE+libGAP_IMMUTABLE,2);
    if (libGAP_sySetjmp(libGAP_ReadJmpError)) {
      libGAP_SET_LEN_PLIST(res,2);
      libGAP_SET_ELM_PLIST(res,1,libGAP_False);
      libGAP_SET_ELM_PLIST(res,2,libGAP_ThrownObject);
      libGAP_CHANGED_BAG(res);
      libGAP_ThrownObject = 0;
      libGAP_CurrLVars = currLVars;
      libGAP_PtrLVars = libGAP_PTR_BAG(libGAP_CurrLVars);
      libGAP_PtrBody = (libGAP_Stat*)libGAP_PTR_BAG(libGAP_BODY_FUNC(libGAP_CURR_FUNC));
      libGAP_CurrStat = currStat;
      libGAP_RecursionDepth = recursionDepth;
    } else {
      switch (libGAP_LEN_PLIST(plain_args)) {
      case 0: result = libGAP_CALL_0ARGS(func);
        break;
      case 1: result = libGAP_CALL_1ARGS(func, libGAP_ELM_PLIST(plain_args,1));
        break;
      case 2: result = libGAP_CALL_2ARGS(func, libGAP_ELM_PLIST(plain_args,1),
                                  libGAP_ELM_PLIST(plain_args,2));
        break;
      case 3: result = libGAP_CALL_3ARGS(func, libGAP_ELM_PLIST(plain_args,1),
                                  libGAP_ELM_PLIST(plain_args,2), libGAP_ELM_PLIST(plain_args,3));
        break;
      case 4: result = libGAP_CALL_4ARGS(func, libGAP_ELM_PLIST(plain_args,1),
                                  libGAP_ELM_PLIST(plain_args,2), libGAP_ELM_PLIST(plain_args,3),
                                  libGAP_ELM_PLIST(plain_args,4));
        break;
      case 5: result = libGAP_CALL_5ARGS(func, libGAP_ELM_PLIST(plain_args,1),
                                  libGAP_ELM_PLIST(plain_args,2), libGAP_ELM_PLIST(plain_args,3),
                                  libGAP_ELM_PLIST(plain_args,4), libGAP_ELM_PLIST(plain_args,5));
        break;
      case 6: result = libGAP_CALL_6ARGS(func, libGAP_ELM_PLIST(plain_args,1),
                                  libGAP_ELM_PLIST(plain_args,2), libGAP_ELM_PLIST(plain_args,3),
                                  libGAP_ELM_PLIST(plain_args,4), libGAP_ELM_PLIST(plain_args,5),
                                  libGAP_ELM_PLIST(plain_args,6));
        break;
      default: result = libGAP_CALL_XARGS(func, plain_args);
      }
      libGAP_SET_ELM_PLIST(res,1,libGAP_True);
      if (result)
        {
          libGAP_SET_LEN_PLIST(res,2);
          libGAP_SET_ELM_PLIST(res,2,result);
          libGAP_CHANGED_BAG(res);
        }
      else
        libGAP_SET_LEN_PLIST(res,1);
    }
    memcpy((void *)&libGAP_ReadJmpError, (void *)&readJmpError, sizeof(libGAP_syJmp_buf));
    return res;
}

libGAP_Obj libGAP_FuncJUMP_TO_CATCH( libGAP_Obj self, libGAP_Obj payload)
{
  libgap_call_error_handler();
  libGAP_ThrownObject = payload;
  libGAP_syLongjmp(libGAP_ReadJmpError, 1);
  return 0;
}
  

libGAP_UInt libGAP_UserHasQuit;
libGAP_UInt libGAP_UserHasQUIT; 

libGAP_Obj libGAP_FuncSetUserHasQuit( libGAP_Obj Self, libGAP_Obj value)
{
  libGAP_UserHasQuit = libGAP_INT_INTOBJ(value);
  if (libGAP_UserHasQuit)
    libGAP_RecursionDepth = 0;
  return 0;
}

/****************************************************************************
**
*F  ErrorQuit( <msg>, <arg1>, <arg2> )  . . . . . . . . . . .  print and quit
*/

static libGAP_Obj libGAP_ErrorMessageToGAPString( 
    const libGAP_Char *        msg,
    libGAP_Int                 arg1,
    libGAP_Int                 arg2 )
{
  libGAP_Char message[120];
  libGAP_Obj Message;
  libGAP_SPrTo(message, 120, msg, arg1, arg2);
  message[119] = '\0';
  libGAP_C_NEW_STRING_DYN(Message, message);
  return Message;
}

libGAP_Obj libGAP_CallErrorInner (
    const libGAP_Char *        msg,
    libGAP_Int                 arg1,
    libGAP_Int                 arg2,
    libGAP_UInt                justQuit,
    libGAP_UInt                mayReturnVoid,
    libGAP_UInt                mayReturnObj,
    libGAP_Obj                 lateMessage,
    libGAP_UInt                printThisStatement)
{
  libGAP_Obj EarlyMsg;
  libGAP_Obj r = libGAP_NEW_PREC(0);
  libGAP_Obj l;
  EarlyMsg = libGAP_ErrorMessageToGAPString(msg, arg1, arg2);
  libGAP_AssPRec(r, libGAP_RNamName("context"), libGAP_CurrLVars);
  libGAP_AssPRec(r, libGAP_RNamName("justQuit"), justQuit? libGAP_True : libGAP_False);
  libGAP_AssPRec(r, libGAP_RNamName("mayReturnObj"), mayReturnObj? libGAP_True : libGAP_False);
  libGAP_AssPRec(r, libGAP_RNamName("mayReturnVoid"), mayReturnVoid? libGAP_True : libGAP_False);
  libGAP_AssPRec(r, libGAP_RNamName("printThisStatement"), printThisStatement? libGAP_True : libGAP_False);
  libGAP_AssPRec(r, libGAP_RNamName("lateMessage"), lateMessage);
  l = libGAP_NEW_PLIST(libGAP_T_PLIST_HOM+libGAP_IMMUTABLE, 1);
  libGAP_SET_ELM_PLIST(l,1,EarlyMsg);
  libGAP_SET_LEN_PLIST(l,1);
  libGAP_SET_BRK_CALL_TO(libGAP_CurrStat);
  return libGAP_CALL_2ARGS(libGAP_ErrorInner,r,l);  
}

void libGAP_ErrorQuit (
    const libGAP_Char *        msg,
    libGAP_Int                 arg1,
    libGAP_Int                 arg2 )
{
  libGAP_CallErrorInner(msg, arg1, arg2, 1, 0, 0, libGAP_False, 1);
}


/****************************************************************************
**
*F  ErrorQuitBound( <name> )  . . . . . . . . . . . . . . .  unbound variable
*/
void libGAP_ErrorQuitBound (
    const libGAP_Char *        name )
{
    libGAP_ErrorQuit(
        "variable '%s' must have an assigned value",
        (libGAP_Int)name, 0L );
}


/****************************************************************************
**
*F  ErrorQuitFuncResult() . . . . . . . . . . . . . . . . must return a value
*/
void libGAP_ErrorQuitFuncResult ( void )
{
    libGAP_ErrorQuit(
        "function must return a value",
        0L, 0L );
}


/****************************************************************************
**
*F  ErrorQuitIntSmall( <obj> )  . . . . . . . . . . . . . not a small integer
*/
void libGAP_ErrorQuitIntSmall (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "<obj> must be a small integer (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L );
}


/****************************************************************************
**
*F  ErrorQuitIntSmallPos( <obj> ) . . . . . . .  not a positive small integer
*/
void libGAP_ErrorQuitIntSmallPos (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "<obj> must be a positive small integer (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L );
}

/****************************************************************************
**
*F  ErrorQuitIntPos( <obj> ) . . . . . . .  not a positive small integer
*/
void libGAP_ErrorQuitIntPos (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "<obj> must be a positive integer (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L );
}


/****************************************************************************
**
*F  ErrorQuitBool( <obj> )  . . . . . . . . . . . . . . . . . . not a boolean
*/
void libGAP_ErrorQuitBool (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "<obj> must be 'true' or 'false' (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L );
}


/****************************************************************************
**
*F  ErrorQuitFunc( <obj> )  . . . . . . . . . . . . . . . . .  not a function
*/
void libGAP_ErrorQuitFunc (
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit(
        "<obj> must be a function (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L );
}


/****************************************************************************
**
*F  ErrorQuitNrArgs( <narg>, <args> ) . . . . . . . wrong number of arguments
*/
void libGAP_ErrorQuitNrArgs (
    libGAP_Int                 narg,
    libGAP_Obj                 args )
{
    libGAP_ErrorQuit(
        "Function Calls: number of arguments must be %d (not %d)",
        narg, libGAP_LEN_PLIST( args ) );
}

/****************************************************************************
**
*F  ErrorQuitRange3( <first>, <second>, <last> ) . . divisibility
*/
void libGAP_ErrorQuitRange3 (
                      libGAP_Obj                 first,
                      libGAP_Obj                 second,
                      libGAP_Obj                 last)
{
    libGAP_ErrorQuit(
        "Range expression <last>-<first> must be divisible by <second>-<first>, not %d %d",
        libGAP_INT_INTOBJ(last)-libGAP_INT_INTOBJ(first), libGAP_INT_INTOBJ(second)-libGAP_INT_INTOBJ(first) );
}


/****************************************************************************
**
*F  ErrorReturnObj( <msg>, <arg1>, <arg2>, <msg2> ) . .  print and return obj
*/
libGAP_Obj libGAP_ErrorReturnObj (
    const libGAP_Char *        msg,
    libGAP_Int                 arg1,
    libGAP_Int                 arg2,
    const libGAP_Char *        msg2 )
{
  libGAP_Obj LateMsg;
  libGAP_C_NEW_STRING_DYN(LateMsg, msg2);
  return libGAP_CallErrorInner(msg, arg1, arg2, 0, 0, 1, LateMsg, 1);
}


/****************************************************************************
**
*F  ErrorReturnVoid( <msg>, <arg1>, <arg2>, <msg2> )  . . .  print and return
*/
void libGAP_ErrorReturnVoid (
    const libGAP_Char *        msg,
    libGAP_Int                 arg1,
    libGAP_Int                 arg2,
    const libGAP_Char *        msg2 )
{
  libGAP_Obj LateMsg;
  libGAP_C_NEW_STRING_DYN(LateMsg, msg2);
  libGAP_CallErrorInner( msg, arg1, arg2, 0,1,0,LateMsg, 1);
  /*    ErrorMode( msg, arg1, arg2, (Obj)0, msg2, 'x' ); */
}

/****************************************************************************
**
*F  ErrorMayQuit( <msg>, <arg1>, <arg2> )  . . .  print and return
*/
void libGAP_ErrorMayQuit (
    const libGAP_Char *        msg,
    libGAP_Int                 arg1,
    libGAP_Int                 arg2)
{
  libGAP_CallErrorInner(msg, arg1, arg2, 0, 0,0, libGAP_False, 1);
 
}

libGAP_Obj libGAP_Error;
libGAP_Obj libGAP_ErrorInner;


/****************************************************************************
**

*F * * * * * * * * * functions for creating the init file * * * * * * * * * *
*/

/* deleted 9/5/11 */

/*************************************************************************
**

*F * * * * * * * * * functions for dynamical/static modules * * * * * * * * *
*/



/****************************************************************************
**

*F  FuncGAP_CRC( <self>, <name> ) . . . . . . . create a crc value for a file
*/
libGAP_Obj libGAP_FuncGAP_CRC (
    libGAP_Obj                 self,
    libGAP_Obj                 filename )
{
    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }

    /* compute the crc value                                               */
    return libGAP_INTOBJ_INT( libGAP_SyGAPCRC( libGAP_CSTR_STRING(filename) ) );
}


/****************************************************************************
**
*F  FuncLOAD_DYN( <self>, <name>, <crc> ) . . .  try to load a dynamic module
*/
libGAP_Obj libGAP_FuncLOAD_DYN (
    libGAP_Obj                 self,
    libGAP_Obj                 filename,
    libGAP_Obj                 crc )
{
    libGAP_InitInfoFunc        init;
    libGAP_StructInitInfo *    info;
    libGAP_Obj                 crc1;
    libGAP_Int                 res;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    while ( ! libGAP_IS_INTOBJ(crc) && crc!=libGAP_False ) {
        crc = libGAP_ErrorReturnObj(
            "<crc> must be a small integer or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(crc), 0L,
            "you can replace <crc> via 'return <crc>;'" );
    }

    /* try to read the module                                              */
    init = libGAP_SyLoadModule( libGAP_CSTR_STRING(filename) );
    if ( (libGAP_Int)init == 1 )
        libGAP_ErrorQuit( "module '%s' not found", (libGAP_Int)libGAP_CSTR_STRING(filename), 0L );
    else if ( (libGAP_Int) init == 3 )
        libGAP_ErrorQuit( "symbol 'Init_Dynamic' not found", 0L, 0L );
    else if ( (libGAP_Int) init == 5 )
        libGAP_ErrorQuit( "forget symbol failed", 0L, 0L );

    /* no dynamic library support                                          */
    else if ( (libGAP_Int) init == 7 ) {
        if ( libGAP_SyDebugLoading ) {
            libGAP_Pr( "#I  LOAD_DYN: no support for dynamical loading\n", 0L, 0L );
        }
        return libGAP_False; 
    }

    /* get the description structure                                       */
    info = (*init)();
    if ( info == 0 )
        libGAP_ErrorQuit( "call to init function failed", 0L, 0L );

    /* check the crc value                                                 */
    if ( crc != libGAP_False ) {
        crc1 = libGAP_INTOBJ_INT( info->crc );
        if ( ! libGAP_EQ( crc, crc1 ) ) {
            if ( libGAP_SyDebugLoading ) {
                libGAP_Pr( "#I  LOAD_DYN: crc values do not match, gap ", 0L, 0L );
                libGAP_PrintInt( crc );
                libGAP_Pr( ", dyn ", 0L, 0L );
                libGAP_PrintInt( crc1 );
                libGAP_Pr( "\n", 0L, 0L );
            }
            return libGAP_False;
        }
    }

    /* link and init me                                                    */
    info->isGapRootRelative = 0;
    res = (info->initKernel)(info);
    libGAP_UpdateCopyFopyInfo();

    /* Start a new executor to run the outer function of the module
       in global context */
    libGAP_ExecBegin( libGAP_BottomLVars );
    res = res || (info->initLibrary)(info);
    libGAP_ExecEnd(res ? libGAP_STATUS_ERROR : libGAP_STATUS_END);
    if ( res ) {
        libGAP_Pr( "#W  init functions returned non-zero exit code\n", 0L, 0L );
    }
    libGAP_RecordLoadedModule(info, libGAP_CSTR_STRING(filename));

    return libGAP_True;
}


/****************************************************************************
**
*F  FuncLOAD_STAT( <self>, <name>, <crc> )  . . . . try to load static module
*/
libGAP_Obj libGAP_FuncLOAD_STAT (
    libGAP_Obj                 self,
    libGAP_Obj                 filename,
    libGAP_Obj                 crc )
{
    libGAP_StructInitInfo *    info = 0;
    libGAP_Obj                 crc1;
    libGAP_Int                 k;
    libGAP_Int                 res;

    /* check the argument                                                  */
    while ( ! libGAP_IsStringConv( filename ) ) {
        filename = libGAP_ErrorReturnObj(
            "<filename> must be a string (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(filename), 0L,
            "you can replace <filename> via 'return <filename>;'" );
    }
    while ( !libGAP_IS_INTOBJ(crc) && crc!=libGAP_False ) {
        crc = libGAP_ErrorReturnObj(
            "<crc> must be a small integer or 'false' (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(crc), 0L,
            "you can replace <crc> via 'return <crc>;'" );
    }

    /* try to find the module                                              */
    for ( k = 0;  libGAP_CompInitFuncs[k];  k++ ) {
        info = (*(libGAP_CompInitFuncs[k]))();
        if ( info == 0 ) {
            continue;
        }
        if ( ! strcmp( libGAP_CSTR_STRING(filename), info->name ) ) {
            break;
        }
    }
    if ( libGAP_CompInitFuncs[k] == 0 ) {
        if ( libGAP_SyDebugLoading ) {
            libGAP_Pr( "#I  LOAD_STAT: no module named '%s' found\n",
                (libGAP_Int)libGAP_CSTR_STRING(filename), 0L );
        }
        return libGAP_False;
    }

    /* check the crc value                                                 */
    if ( crc != libGAP_False ) {
        crc1 = libGAP_INTOBJ_INT( info->crc );
        if ( ! libGAP_EQ( crc, crc1 ) ) {
            if ( libGAP_SyDebugLoading ) {
                libGAP_Pr( "#I  LOAD_STAT: crc values do not match, gap ", 0L, 0L );
                libGAP_PrintInt( crc );
                libGAP_Pr( ", stat ", 0L, 0L );
                libGAP_PrintInt( crc1 );
                libGAP_Pr( "\n", 0L, 0L );
            }
            return libGAP_False;
        }
    }

    /* link and init me                                                    */
    info->isGapRootRelative = 0;
    res = (info->initKernel)(info);
    libGAP_UpdateCopyFopyInfo();
    /* Start a new executor to run the outer function of the module
       in global context */
    libGAP_ExecBegin( libGAP_BottomLVars );
    res = res || (info->initLibrary)(info);
    libGAP_ExecEnd(res ? libGAP_STATUS_ERROR : libGAP_STATUS_END);
    if ( res ) {
        libGAP_Pr( "#W  init functions returned non-zero exit code\n", 0L, 0L );
    }
    libGAP_RecordLoadedModule(info, libGAP_CSTR_STRING(filename));

    return libGAP_True;
}


/****************************************************************************
**
*F  FuncSHOW_STAT() . . . . . . . . . . . . . . . . . . . show static modules
*/
libGAP_Obj libGAP_FuncSHOW_STAT (
    libGAP_Obj                 self )
{
    libGAP_Obj                 modules;
    libGAP_Obj                 name;
    libGAP_StructInitInfo *    info;
    libGAP_Int                 k;
    libGAP_Int                 im;

    /* count the number of install modules                                 */
    for ( k = 0,  im = 0;  libGAP_CompInitFuncs[k];  k++ ) {
        info = (*(libGAP_CompInitFuncs[k]))();
        if ( info == 0 ) {
            continue;
        }
        im++;
    }

    /* make a list of modules with crc values                              */
    modules = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*im );
    libGAP_SET_LEN_PLIST( modules, 2*im );

    for ( k = 0,  im = 1;  libGAP_CompInitFuncs[k];  k++ ) {
        info = (*(libGAP_CompInitFuncs[k]))();
        if ( info == 0 ) {
            continue;
        }
        libGAP_C_NEW_STRING_DYN(name, info->name);

        libGAP_SET_ELM_PLIST( modules, im, name );

        /* compute the crc value                                           */
        libGAP_SET_ELM_PLIST( modules, im+1, libGAP_INTOBJ_INT( info->crc ) );
        im += 2;
    }

    return modules;
}


/****************************************************************************
**
*F  FuncLoadedModules( <self> ) . . . . . . . . . . . list all loaded modules
*/
libGAP_Obj libGAP_FuncLoadedModules (
    libGAP_Obj                 self )
{
    libGAP_Int                 i;
    libGAP_StructInitInfo *    m;
    libGAP_Obj                 str;
    libGAP_Obj                 list;

    /* create a list                                                       */
    list = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_NrModules * 3 );
    libGAP_SET_LEN_PLIST( list, libGAP_NrModules * 3 );
    for ( i = 0;  i < libGAP_NrModules;  i++ ) {
        m = libGAP_Modules[i];
        if ( m->type == libGAP_MODULE_BUILTIN ) {
            libGAP_SET_ELM_PLIST( list, 3*i+1, libGAP_ObjsChar[(libGAP_Int)'b'] );
            libGAP_CHANGED_BAG(list);
            libGAP_C_NEW_STRING_DYN( str, m->name );
            libGAP_SET_ELM_PLIST( list, 3*i+2, str );
            libGAP_SET_ELM_PLIST( list, 3*i+3, libGAP_INTOBJ_INT(m->version) );
        }
        else if ( m->type == libGAP_MODULE_DYNAMIC ) {
            libGAP_SET_ELM_PLIST( list, 3*i+1, libGAP_ObjsChar[(libGAP_Int)'d'] );
            libGAP_CHANGED_BAG(list);
            libGAP_C_NEW_STRING_DYN( str, m->name );
            libGAP_SET_ELM_PLIST( list, 3*i+2, str );
            libGAP_CHANGED_BAG(list);
            libGAP_C_NEW_STRING_DYN( str, m->filename );
            libGAP_SET_ELM_PLIST( list, 3*i+3, str );
        }
        else if ( m->type == libGAP_MODULE_STATIC ) {
            libGAP_SET_ELM_PLIST( list, 3*i+1, libGAP_ObjsChar[(libGAP_Int)'s'] );
            libGAP_CHANGED_BAG(list);
            libGAP_C_NEW_STRING_DYN( str, m->name );
            libGAP_SET_ELM_PLIST( list, 3*i+2, str );
            libGAP_CHANGED_BAG(list);
            libGAP_C_NEW_STRING_DYN( str, m->filename );
            libGAP_SET_ELM_PLIST( list, 3*i+3, str );
        }
    }
    return libGAP_CopyObj( list, 0 );
}


/****************************************************************************
**


*F * * * * * * * * * * * * * * debug functions  * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  FuncGASMAN( <self>, <args> )  . . . . . . . . .  expert function 'GASMAN'
**
**  'FuncGASMAN' implements the internal function 'GASMAN'
**
**  'GASMAN( "display" | "clear" | "collect" | "message" | "partial" )'
*/
libGAP_Obj libGAP_FuncGASMAN (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 cmd;            /* argument                        */
    libGAP_UInt                i,  k;          /* loop variables                  */
    libGAP_Char                buf[41];

    /* check the argument                                                  */
    while ( ! libGAP_IS_SMALL_LIST(args) || libGAP_LEN_LIST(args) == 0 ) {
        args = libGAP_ErrorReturnObj(
            "usage: GASMAN( \"display\"|\"displayshort\"|\"clear\"|\"collect\"|\"message\"|\"partial\" )",
            0L, 0L,
            "you can replace the argument list <args> via 'return <args>;'" );
    }

    /* loop over the arguments                                             */
    for ( i = 1; i <= libGAP_LEN_LIST(args); i++ ) {

        /* evaluate and check the command                                  */
        cmd = libGAP_ELM_PLIST( args, i );
again:
        while ( ! libGAP_IsStringConv(cmd) ) {
           cmd = libGAP_ErrorReturnObj(
               "GASMAN: <cmd> must be a string (not a %s)",
               (libGAP_Int)libGAP_TNAM_OBJ(cmd), 0L,
               "you can replace <cmd> via 'return <cmd>;'" );
       }

        /* if request display the statistics                               */
        if ( strcmp( libGAP_CSTR_STRING(cmd), "display" ) == 0 ) {
            libGAP_Pr( "%40s ", (libGAP_Int)"type",  0L          );
            libGAP_Pr( "%8s %8s ",  (libGAP_Int)"alive", (libGAP_Int)"kbyte" );
            libGAP_Pr( "%8s %8s\n",  (libGAP_Int)"total", (libGAP_Int)"kbyte" );
            for ( k = 0; k < 256; k++ ) {
                if ( libGAP_InfoBags[k].name != 0 ) {
                    buf[0] = '\0';
                    libGAP_strlcat( buf, libGAP_InfoBags[k].name, sizeof(buf) );
                    libGAP_Pr("%40s ",    (libGAP_Int)buf, 0L );
                    libGAP_Pr("%8d %8d ", (libGAP_Int)libGAP_InfoBags[k].nrLive,
                                   (libGAP_Int)(libGAP_InfoBags[k].sizeLive/1024));
                    libGAP_Pr("%8d %8d\n",(libGAP_Int)libGAP_InfoBags[k].nrAll,
                                   (libGAP_Int)(libGAP_InfoBags[k].sizeAll/1024));
                }
            }
        }

        /* if request give a short display of the statistics                */
        else if ( strcmp( libGAP_CSTR_STRING(cmd), "displayshort" ) == 0 ) {
            libGAP_Pr( "%40s ", (libGAP_Int)"type",  0L          );
            libGAP_Pr( "%8s %8s ",  (libGAP_Int)"alive", (libGAP_Int)"kbyte" );
            libGAP_Pr( "%8s %8s\n",  (libGAP_Int)"total", (libGAP_Int)"kbyte" );
            for ( k = 0; k < 256; k++ ) {
                if ( libGAP_InfoBags[k].name != 0 && 
                     (libGAP_InfoBags[k].nrLive != 0 ||
                      libGAP_InfoBags[k].sizeLive != 0 ||
                      libGAP_InfoBags[k].nrAll != 0 ||
                      libGAP_InfoBags[k].sizeAll != 0) ) {
                    buf[0] = '\0';
                    libGAP_strlcat( buf, libGAP_InfoBags[k].name, sizeof(buf) );
                    libGAP_Pr("%40s ",    (libGAP_Int)buf, 0L );
                    libGAP_Pr("%8d %8d ", (libGAP_Int)libGAP_InfoBags[k].nrLive,
                                   (libGAP_Int)(libGAP_InfoBags[k].sizeLive/1024));
                    libGAP_Pr("%8d %8d\n",(libGAP_Int)libGAP_InfoBags[k].nrAll,
                                   (libGAP_Int)(libGAP_InfoBags[k].sizeAll/1024));
                }
            }
        }

        /* if request display the statistics                               */
        else if ( strcmp( libGAP_CSTR_STRING(cmd), "clear" ) == 0 ) {
            for ( k = 0; k < 256; k++ ) {
#ifdef GASMAN_CLEAR_TO_LIVE
                libGAP_InfoBags[k].nrAll    = libGAP_InfoBags[k].nrLive;
                libGAP_InfoBags[k].sizeAll  = libGAP_InfoBags[k].sizeLive;
#else
                libGAP_InfoBags[k].nrAll    = 0;
                libGAP_InfoBags[k].sizeAll  = 0;
#endif
            }
        }

        /* or collect the garbage                                          */
        else if ( strcmp( libGAP_CSTR_STRING(cmd), "collect" ) == 0 ) {
            libGAP_CollectBags(0,1);
        }

        /* or collect the garbage                                          */
        else if ( strcmp( libGAP_CSTR_STRING(cmd), "partial" ) == 0 ) {
            libGAP_CollectBags(0,0);
        }

        /* or display information about global bags                        */
        else if ( strcmp( libGAP_CSTR_STRING(cmd), "global" ) == 0 ) {
            for ( i = 0;  i < libGAP_GlobalBags.nr;  i++ ) {
                if ( *(libGAP_GlobalBags.addr[i]) != 0 ) {
                    libGAP_Pr( "%50s: %12d bytes\n", (libGAP_Int)libGAP_GlobalBags.cookie[i], 
                        (libGAP_Int)libGAP_SIZE_BAG(*(libGAP_GlobalBags.addr[i])) );
                }
            }
        }

        /* or finally toggle Gasman messages                               */
        else if ( strcmp( libGAP_CSTR_STRING(cmd), "message" ) == 0 ) {
            libGAP_SyMsgsFlagBags = (libGAP_SyMsgsFlagBags + 1) % 3;
        }

        /* otherwise complain                                              */
        else {
            cmd = libGAP_ErrorReturnObj(
                "GASMAN: <cmd> must be %s or %s",
                (libGAP_Int)"\"display\" or \"clear\" or \"global\" or ",
                (libGAP_Int)"\"collect\" or \"partial\" or \"message\"",
                "you can replace <cmd> via 'return <cmd>;'" );
            goto again;
        }
    }

    /* return nothing, this function is a procedure                        */
    return 0;
}

libGAP_Obj libGAP_FuncGASMAN_STATS(libGAP_Obj self)
{
  libGAP_Obj res;
  libGAP_Obj row;
  libGAP_Obj entry;
  libGAP_UInt i,j;
  libGAP_Int x;
  res = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT + libGAP_IMMUTABLE, 2);
  libGAP_SET_LEN_PLIST(res, 2);
  for (i = 1; i <= 2; i++)
    {
      row = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC + libGAP_IMMUTABLE, 9);
      libGAP_SET_ELM_PLIST(res, i, row);
      libGAP_CHANGED_BAG(res);
      libGAP_SET_LEN_PLIST(row, 9);
      for (j = 1; j <= 8; j++)
        {
          x = libGAP_SyGasmanNumbers[i-1][j];

          /* convert x to GAP integer. x may be too big to be a small int */
          if (x < (1L << libGAP_NR_SMALL_INT_BITS))
            entry = libGAP_INTOBJ_INT(x);
          else
            entry = libGAP_SUM( libGAP_PROD(libGAP_INTOBJ_INT(x >> (libGAP_NR_SMALL_INT_BITS/2)),
                              libGAP_INTOBJ_INT(1 << (libGAP_NR_SMALL_INT_BITS/2))),
                         libGAP_INTOBJ_INT( x % ( 1 << (libGAP_NR_SMALL_INT_BITS/2))));
          libGAP_SET_ELM_PLIST(row, j, entry);
        }
      libGAP_SET_ELM_PLIST(row, 9, libGAP_INTOBJ_INT(libGAP_SyGasmanNumbers[i-1][0]));       
    }
  return res;      
}

libGAP_Obj libGAP_FuncGASMAN_MESSAGE_STATUS( libGAP_Obj self )
{
  return libGAP_INTOBJ_INT(libGAP_SyMsgsFlagBags);
}

libGAP_Obj libGAP_FuncGASMAN_LIMITS( libGAP_Obj self )
{
  libGAP_Obj list;
  list = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, 3);
  libGAP_SET_LEN_PLIST(list,3);
  libGAP_SET_ELM_PLIST(list, 1, libGAP_INTOBJ_INT(libGAP_SyStorMin));
  libGAP_SET_ELM_PLIST(list, 2, libGAP_INTOBJ_INT(libGAP_SyStorMax));
  libGAP_SET_ELM_PLIST(list, 3, libGAP_INTOBJ_INT(libGAP_SyStorKill));
  return list;
}

/****************************************************************************
**
*F  FuncSHALLOW_SIZE( <self>, <obj> ) . . . .  expert function 'SHALLOW_SIZE'
*/
libGAP_Obj libGAP_FuncSHALLOW_SIZE (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
  if (libGAP_IS_INTOBJ(obj) || libGAP_IS_FFE(obj))
    return libGAP_INTOBJ_INT(0);
  else
    return libGAP_ObjInt_UInt( libGAP_SIZE_BAG( obj ) );
}


/****************************************************************************
**
*F  FuncTNUM_OBJ( <self>, <obj> ) . . . . . . . .  expert function 'TNUM_OBJ'
*/

libGAP_Obj libGAP_FuncTNUM_OBJ (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 res;
    libGAP_Obj                 str;
    const libGAP_Char *        cst;

    res = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( res, 2 );

    /* set the type                                                        */
    libGAP_SET_ELM_PLIST( res, 1, libGAP_INTOBJ_INT( libGAP_TNUM_OBJ(obj) ) );
    cst = libGAP_TNAM_OBJ(obj);
    libGAP_C_NEW_STRING_DYN(str, cst);
    libGAP_SET_ELM_PLIST( res, 2, str );

    /* and return                                                          */
    return res;
}

libGAP_Obj libGAP_FuncTNUM_OBJ_INT (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{

  
    return  libGAP_INTOBJ_INT( libGAP_TNUM_OBJ(obj) ) ;
}

/****************************************************************************
**
*F  FuncXTNUM_OBJ( <self>, <obj> )  . . . . . . . expert function 'XTNUM_OBJ'
*/
libGAP_Obj libGAP_FuncXTNUM_OBJ (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 res;
    libGAP_Obj                 str;

    res = libGAP_NEW_PLIST( libGAP_T_PLIST, 2 );
    libGAP_SET_LEN_PLIST( res, 2 );
    libGAP_SET_ELM_PLIST( res, 1, libGAP_Fail );
    libGAP_C_NEW_STRING_CONST(str, "xtnums abolished");
    libGAP_SET_ELM_PLIST(res, 2,str);
    /* and return                                                          */
    return res;
}


/****************************************************************************
**
*F  FuncOBJ_HANDLE( <self>, <obj> ) . . . . . .  expert function 'OBJ_HANDLE'
*/
libGAP_Obj libGAP_FuncOBJ_HANDLE (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_UInt                hand;
    libGAP_UInt                prod;
    libGAP_Obj                 rem;

    if ( libGAP_IS_INTOBJ(obj) ) {
        return (libGAP_Obj)libGAP_INT_INTOBJ(obj);
    }
    else if ( libGAP_TNUM_OBJ(obj) == libGAP_T_INTPOS ) {
        hand = 0;
        prod = 1;
        while ( libGAP_EQ( obj, libGAP_INTOBJ_INT(0) ) == 0 ) {
            rem  = libGAP_RemInt( obj, libGAP_INTOBJ_INT( 1 << 16 ) );
            obj  = libGAP_QuoInt( obj, libGAP_INTOBJ_INT( 1 << 16 ) );
            hand = hand + prod * libGAP_INT_INTOBJ(rem);
            prod = prod * ( 1 << 16 );
        }
        return (libGAP_Obj) hand;
    }
    else {
        libGAP_ErrorQuit( "<handle> must be a positive integer", 0L, 0L );
        return 0;
    }
}


/****************************************************************************
**
*F  FuncHANDLE_OBJ( <self>, <obj> ) . . . . . .  expert function 'HANDLE_OBJ'
*/
libGAP_Obj libGAP_FuncHANDLE_OBJ (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_Obj                 hnum;
    libGAP_Obj                 prod;
    libGAP_Obj                 tmp;
    libGAP_UInt                hand;

    hand = (libGAP_UInt) obj;
    hnum = libGAP_INTOBJ_INT(0);
    prod = libGAP_INTOBJ_INT(1);
    while ( 0 < hand ) {
        tmp  = libGAP_PROD( prod, libGAP_INTOBJ_INT( hand & 0xffff ) );
        prod = libGAP_PROD( prod, libGAP_INTOBJ_INT( 1 << 16 ) );
        hnum = libGAP_SUM(  hnum, tmp );
        hand = hand >> 16;
    }
    return hnum;
}

libGAP_Obj libGAP_FuncMASTER_POINTER_NUMBER(libGAP_Obj self, libGAP_Obj o)
{
    if ((void **) o >= (void **) libGAP_MptrBags && (void **) o < (void **) libGAP_OldBags) {
        return libGAP_INTOBJ_INT( ((void **) o - (void **) libGAP_MptrBags) + 1 );
    } else {
        return libGAP_INTOBJ_INT( 0 );
    }
}

libGAP_Obj libGAP_FuncFUNC_BODY_SIZE(libGAP_Obj self, libGAP_Obj f)
{
    libGAP_Obj body;
    if (libGAP_TNUM_OBJ(f) != libGAP_T_FUNCTION) return libGAP_Fail;
    body = libGAP_BODY_FUNC(f);
    if (body == 0) return libGAP_INTOBJ_INT(0);
    else return libGAP_INTOBJ_INT( libGAP_SIZE_BAG( body ) );
}

/****************************************************************************
**
*F  FuncSWAP_MPTR( <self>, <obj1>, <obj2> ) . . . . . . . swap master pointer
**
**  Never use this function unless you are debugging.
*/
libGAP_Obj libGAP_FuncSWAP_MPTR (
    libGAP_Obj                 self,
    libGAP_Obj                 obj1,
    libGAP_Obj                 obj2 )
{
    if ( libGAP_TNUM_OBJ(obj1) == libGAP_T_INT || libGAP_TNUM_OBJ(obj1) == libGAP_T_FFE ) {
        libGAP_ErrorQuit("SWAP_MPTR: <obj1> must not be an integer or ffe", 0L, 0L);
        return 0;
    }
    if ( libGAP_TNUM_OBJ(obj2) == libGAP_T_INT || libGAP_TNUM_OBJ(obj2) == libGAP_T_FFE ) {
        libGAP_ErrorQuit("SWAP_MPTR: <obj2> must not be an integer or ffe", 0L, 0L);
        return 0;
    }
        
    libGAP_SwapMasterPoint( obj1, obj2 );
    return 0;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  FillInVersion( <module>, <rev_c>, <rev_h> ) . . .  fill in version number
*/
void libGAP_FillInVersion (
    libGAP_StructInitInfo *            libGAP_module )
{
}


/****************************************************************************
**
*F  RequireModule( <calling>, <required>, <version> ) . . . .  require module
*/
void libGAP_RequireModule (
    libGAP_StructInitInfo *            libGAP_module,
    const libGAP_Char *                required,
    libGAP_UInt                        version )
{
}


/****************************************************************************
**
*F  InitBagNamesFromTable( <table> )  . . . . . . . . .  initialise bag names
*/
void libGAP_InitBagNamesFromTable (
    libGAP_StructBagNames *            tab )
{
    libGAP_Int                         i;

    for ( i = 0;  tab[i].tnum != -1;  i++ ) {
        libGAP_InfoBags[tab[i].tnum].name = tab[i].name;
    }
}


/****************************************************************************
**
*F  InitClearFiltsTNumsFromTable( <tab> ) . . .  initialise clear filts tnums
*/
void libGAP_InitClearFiltsTNumsFromTable (
    libGAP_Int *               tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i] != -1;  i += 2 ) {
        libGAP_ClearFiltsTNums[tab[i]] = tab[i+1];
    }
}


/****************************************************************************
**
*F  InitHasFiltListTNumsFromTable( <tab> )  . . initialise tester filts tnums
*/
void libGAP_InitHasFiltListTNumsFromTable (
    libGAP_Int *               tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i] != -1;  i += 3 ) {
        libGAP_HasFiltListTNums[tab[i]][tab[i+1]] = tab[i+2];
    }
}


/****************************************************************************
**
*F  InitSetFiltListTNumsFromTable( <tab> )  . . initialise setter filts tnums
*/
void libGAP_InitSetFiltListTNumsFromTable (
    libGAP_Int *               tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i] != -1;  i += 3 ) {
        libGAP_SetFiltListTNums[tab[i]][tab[i+1]] = tab[i+2];
    }
}


/****************************************************************************
**
*F  InitResetFiltListTNumsFromTable( <tab> )  initialise unsetter filts tnums
*/
void libGAP_InitResetFiltListTNumsFromTable (
    libGAP_Int *               tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i] != -1;  i += 3 ) {
        libGAP_ResetFiltListTNums[tab[i]][tab[i+1]] = tab[i+2];
    }
}


/****************************************************************************
**
*F  InitGVarFiltsFromTable( <tab> ) . . . . . . . . . . . . . . . new filters
*/
void libGAP_InitGVarFiltsFromTable (
    libGAP_StructGVarFilt *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_UInt gvar = libGAP_GVarName( tab[i].name );
        libGAP_AssGVar( gvar,
            libGAP_NewFilter( libGAP_NameGVarObj( gvar ), 1, libGAP_ArgStringToList( tab[i].argument ), tab[i].handler ) );
        libGAP_MakeReadOnlyGVar( gvar );
    }
}


/****************************************************************************
**
*F  InitGVarAttrsFromTable( <tab> ) . . . . . . . . . . . . .  new attributes
*/
void libGAP_InitGVarAttrsFromTable (
    libGAP_StructGVarAttr *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_UInt gvar = libGAP_GVarName( tab[i].name );
        libGAP_AssGVar( gvar,
            libGAP_NewAttribute( libGAP_NameGVarObj( gvar ),
                          1,
                          libGAP_ArgStringToList( tab[i].argument ),
                          tab[i].handler ) );
        libGAP_MakeReadOnlyGVar( gvar );
    }
}


/****************************************************************************
**
*F  InitGVarPropsFromTable( <tab> ) . . . . . . . . . . . . .  new properties
*/
void libGAP_InitGVarPropsFromTable (
    libGAP_StructGVarProp *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_UInt gvar = libGAP_GVarName( tab[i].name );
        libGAP_AssGVar( gvar,
            libGAP_NewProperty( libGAP_NameGVarObj( gvar ),
                        1,
                        libGAP_ArgStringToList( tab[i].argument ),
                        tab[i].handler ) );
        libGAP_MakeReadOnlyGVar( gvar );
    }
}


/****************************************************************************
**
*F  InitGVarOpersFromTable( <tab> ) . . . . . . . . . . . . .  new operations
*/
void libGAP_InitGVarOpersFromTable (
    libGAP_StructGVarOper *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_UInt gvar = libGAP_GVarName( tab[i].name );
        libGAP_AssGVar( gvar,
            libGAP_NewOperation( libGAP_NameGVarObj( gvar ),
                          tab[i].nargs,
                          libGAP_ArgStringToList( tab[i].args ),
                          tab[i].handler ) );
        libGAP_MakeReadOnlyGVar( gvar );
    }
}


/****************************************************************************
**
*F  InitGVarFuncsFromTable( <tab> ) . . . . . . . . . . . . . . new functions
*/
void libGAP_InitGVarFuncsFromTable (
    libGAP_StructGVarFunc *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_UInt gvar = libGAP_GVarName( tab[i].name );
        libGAP_AssGVar( gvar,
            libGAP_NewFunction( libGAP_NameGVarObj( gvar ),
                         tab[i].nargs,
                         libGAP_ArgStringToList( tab[i].args ),
                         tab[i].handler ) );
        libGAP_MakeReadOnlyGVar( gvar );
    }
}


/****************************************************************************
**
*F  InitHdlrFiltsFromTable( <tab> ) . . . . . . . . . . . . . . . new filters
*/
void libGAP_InitHdlrFiltsFromTable (
    libGAP_StructGVarFilt *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_InitHandlerFunc( tab[i].handler, tab[i].cookie );
        libGAP_InitFopyGVar( tab[i].name, tab[i].filter );
    }
}


/****************************************************************************
**
*F  InitHdlrAttrsFromTable( <tab> ) . . . . . . . . . . . . .  new attributes
*/
void libGAP_InitHdlrAttrsFromTable (
    libGAP_StructGVarAttr *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_InitHandlerFunc( tab[i].handler, tab[i].cookie );
        libGAP_InitFopyGVar( tab[i].name, tab[i].attribute );
    }
}


/****************************************************************************
**
*F  InitHdlrPropsFromTable( <tab> ) . . . . . . . . . . . . .  new properties
*/
void libGAP_InitHdlrPropsFromTable (
    libGAP_StructGVarProp *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_InitHandlerFunc( tab[i].handler, tab[i].cookie );
        libGAP_InitFopyGVar( tab[i].name, tab[i].property );
    }
}


/****************************************************************************
**
*F  InitHdlrOpersFromTable( <tab> ) . . . . . . . . . . . . .  new operations
*/
void libGAP_InitHdlrOpersFromTable (
    libGAP_StructGVarOper *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_InitHandlerFunc( tab[i].handler, tab[i].cookie );
        libGAP_InitFopyGVar( tab[i].name, tab[i].operation );
    }
}


/****************************************************************************
**
*F  InitHdlrFuncsFromTable( <tab> ) . . . . . . . . . . . . . . new functions
*/
void libGAP_InitHdlrFuncsFromTable (
    libGAP_StructGVarFunc *    tab )
{
    libGAP_Int                 i;

    for ( i = 0;  tab[i].name != 0;  i++ ) {
        libGAP_InitHandlerFunc( tab[i].handler, tab[i].cookie );
    }
}


/****************************************************************************
**
*F  ImportGVarFromLibrary( <name>, <address> )  . . .  import global variable
*/


void libGAP_ImportGVarFromLibrary(
    const libGAP_Char *        name,
    libGAP_Obj *               address )
{
    if ( libGAP_NrImportedGVars == 1024 ) {
        libGAP_Pr( "#W  warning: too many imported GVars\n", 0L, 0L );
    }
    else {
        libGAP_ImportedGVars[libGAP_NrImportedGVars].name    = name;
        libGAP_ImportedGVars[libGAP_NrImportedGVars].address = address;
        libGAP_NrImportedGVars++;
    }
    if ( address != 0 ) {
        libGAP_InitCopyGVar( name, address );
    }
}


/****************************************************************************
**
*F  ImportFuncFromLibrary( <name>, <address> )  . . .  import global function
*/


void libGAP_ImportFuncFromLibrary(
    const libGAP_Char *        name,
    libGAP_Obj *               address )
{
    if ( libGAP_NrImportedFuncs == 1024 ) {
        libGAP_Pr( "#W  warning: too many imported Funcs\n", 0L, 0L );
    }
    else {
        libGAP_ImportedFuncs[libGAP_NrImportedFuncs].name    = name;
        libGAP_ImportedFuncs[libGAP_NrImportedFuncs].address = address;
        libGAP_NrImportedFuncs++;
    }
    if ( address != 0 ) {
        libGAP_InitFopyGVar( name, address );
    }
}


/****************************************************************************
**
*F  FuncExportToKernelFinished( <self> )  . . . . . . . . . . check functions
*/
libGAP_Obj libGAP_FuncExportToKernelFinished (
    libGAP_Obj             self )
{
    libGAP_UInt            i;
    libGAP_Int             errs = 0;
    libGAP_Obj             val;

    libGAP_SyInitializing = 0;
    for ( i = 0;  i < libGAP_NrImportedGVars;  i++ ) {
        if ( libGAP_ImportedGVars[i].address == 0 ) {
            val = libGAP_ValAutoGVar(libGAP_GVarName(libGAP_ImportedGVars[i].name));
            if ( val == 0 ) {
                errs++;
                if ( ! libGAP_SyQuiet ) {
                    libGAP_Pr( "#W  global variable '%s' has not been defined\n",
                        (libGAP_Int)libGAP_ImportedFuncs[i].name, 0L );
                }
            }
        }
        else if ( *libGAP_ImportedGVars[i].address == 0 ) {
            errs++;
            if ( ! libGAP_SyQuiet ) {
                libGAP_Pr( "#W  global variable '%s' has not been defined\n",
                    (libGAP_Int)libGAP_ImportedGVars[i].name, 0L );
            }
        }
        else {
            libGAP_MakeReadOnlyGVar(libGAP_GVarName(libGAP_ImportedGVars[i].name));
        }
    }
    
    for ( i = 0;  i < libGAP_NrImportedFuncs;  i++ ) {
        if (  libGAP_ImportedFuncs[i].address == 0 ) {
            val = libGAP_ValAutoGVar(libGAP_GVarName(libGAP_ImportedFuncs[i].name));
            if ( val == 0 || ! libGAP_IS_FUNC(val) ) {
                errs++;
                if ( ! libGAP_SyQuiet ) {
                    libGAP_Pr( "#W  global function '%s' has not been defined\n",
                        (libGAP_Int)libGAP_ImportedFuncs[i].name, 0L );
                }
            }
        }
        else if ( *libGAP_ImportedFuncs[i].address == libGAP_ErrorMustEvalToFuncFunc
          || *libGAP_ImportedFuncs[i].address == libGAP_ErrorMustHaveAssObjFunc )
        {
            errs++;
            if ( ! libGAP_SyQuiet ) {
                libGAP_Pr( "#W  global function '%s' has not been defined\n",
                    (libGAP_Int)libGAP_ImportedFuncs[i].name, 0L );
            }
        }
        else {
            libGAP_MakeReadOnlyGVar(libGAP_GVarName(libGAP_ImportedFuncs[i].name));
        }
    }
    
    return errs == 0 ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncSleep( <self>, <secs> )
**
*/

libGAP_Obj libGAP_FuncSleep( libGAP_Obj self, libGAP_Obj secs )
{
  libGAP_Int  s;

  while( ! libGAP_IS_INTOBJ(secs) )
    secs = libGAP_ErrorReturnObj( "<secs> must be a small integer", 0L, 0L, 
                           "you can replace <secs> via 'return <secs>;'" );

  
  if ( (s = libGAP_INT_INTOBJ(secs)) > 0)
    libGAP_SySleep((libGAP_UInt)s);
  
  /* either we used up the time, or we were interrupted. */
  if (libGAP_SyIsIntr())
    {
      libGAP_ClearError(); /* The interrupt may still be pending */
      libGAP_ErrorReturnVoid("user interrupt in sleep", 0L, 0L,
                    "you can 'return;' as if the sleep was finished");
    }
  
  return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  FuncQUIT_GAP()
**
*/

libGAP_Obj libGAP_FuncQUIT_GAP( libGAP_Obj self )
{
  libGAP_UserHasQUIT = 1;
  libGAP_ReadEvalError();
  return (libGAP_Obj)0; 
}


/****************************************************************************
**
*F  KERNEL_INFO() ......................record of information from the kernel
** 
** The general idea is to put all kernel-specific info in here, and clean up
** the assortment of global variables previously used
*/

libGAP_Obj libGAP_FuncKERNEL_INFO(libGAP_Obj self) {
  libGAP_Obj res = libGAP_NEW_PREC(0);
  libGAP_UInt r,lenvec,lenstr,lenstr2;
  libGAP_Char *p;
  libGAP_Obj tmp,list,str;
  libGAP_UInt i,j;

  /* GAP_ARCHITECTURE                                                    */
  libGAP_C_NEW_STRING_DYN( tmp, libGAP_SyArchitecture );
  libGAP_RetypeBag( tmp, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(tmp)) );
  r = libGAP_RNamName("GAP_ARCHITECTURE");
  libGAP_AssPRec(res,r,tmp);
  /* KERNEL_VERSION */
  libGAP_C_NEW_STRING_DYN( tmp, libGAP_SyKernelVersion );
  libGAP_RetypeBag( tmp, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(tmp)) );
  r = libGAP_RNamName("KERNEL_VERSION");
  libGAP_AssPRec(res,r,tmp);
  /* GAP_ROOT_PATH                                                       */
  /* do we need this. Could we rebuild it from the command line in GAP
     if so, should we                                                    */
  list = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, libGAP_MAX_GAP_DIRS );
  for ( i = 0, j = 1;  i < libGAP_MAX_GAP_DIRS;  i++ ) {
    if ( libGAP_SyGapRootPaths[i][0] ) {
      libGAP_C_NEW_STRING_DYN( tmp, libGAP_SyGapRootPaths[i] );
      libGAP_RetypeBag( tmp, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(tmp)) );
      libGAP_SET_ELM_PLIST( list, j, tmp );
      j++;
    }
  }
  libGAP_SET_LEN_PLIST( list, j-1 );
  r = libGAP_RNamName("GAP_ROOT_PATHS");
  libGAP_AssPRec(res,r,list);
  /* And also the DotGapPath if available */
#if HAVE_DOTGAPRC
  libGAP_C_NEW_STRING_DYN( tmp, libGAP_DotGapPath );
  libGAP_RetypeBag( tmp, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(tmp)) );
  r = libGAP_RNamName("DOT_GAP_PATH");
  libGAP_AssPRec(res,r,tmp);
#endif
    
  /* make command line and environment available to GAP level       */
  for (lenvec=0; libGAP_SyOriginalArgv[lenvec]; lenvec++);
  tmp = libGAP_NEW_PLIST( libGAP_T_PLIST+libGAP_IMMUTABLE, lenvec );
  libGAP_SET_LEN_PLIST( tmp, lenvec );
  for (i = 0; i<lenvec; i++) {
    libGAP_C_NEW_STRING_DYN( str, libGAP_SyOriginalArgv[i] );
    libGAP_SET_ELM_PLIST(tmp, i+1, str);
    libGAP_CHANGED_BAG(tmp);
  }
  r = libGAP_RNamName("COMMAND_LINE");
  libGAP_AssPRec(res,r, tmp);

  tmp = libGAP_NEW_PREC(0);
  for (i = 0; libGAP_sysenviron[i]; i++) {
    for (p = libGAP_sysenviron[i]; *p != '='; p++)
      ;
    lenstr2 = (libGAP_UInt) (p-libGAP_sysenviron[i]);
    p++;   /* Move pointer behind = character */
    lenstr = strlen(p);
    if (lenstr2 > lenstr)
        str = libGAP_NEW_STRING(lenstr2);
    else
        str = libGAP_NEW_STRING(lenstr);
    libGAP_SyStrncat(libGAP_CSTR_STRING(str),libGAP_sysenviron[i],lenstr2);
    r = libGAP_RNamName(libGAP_CSTR_STRING(str));
    *(libGAP_CSTR_STRING(str)) = 0;
    libGAP_SyStrncat(libGAP_CSTR_STRING(str),p, lenstr);
    libGAP_SET_LEN_STRING(str, lenstr);
    libGAP_SHRINK_STRING(str);
    libGAP_AssPRec(tmp,r , str);
  }
  r = libGAP_RNamName("ENVIRONMENT");
  libGAP_AssPRec(res,r, tmp);

  /* and also the CONFIGNAME of the running  GAP kernel  */
  libGAP_C_NEW_STRING_DYN( str, CONFIGNAME );
  r = libGAP_RNamName("CONFIGNAME");
  libGAP_AssPRec(res, r, str);
  
  /* export if we want to use readline  */
  r = libGAP_RNamName("HAVE_LIBREADLINE");
  if (libGAP_SyUseReadline)
    libGAP_AssPRec(res, r, libGAP_True);
  else
    libGAP_AssPRec(res, r, libGAP_False);

  return res;
  
}


/****************************************************************************
**
*F FuncGETPID  ... export UNIX getpid to GAP level
**
*/

libGAP_Obj libGAP_FuncGETPID(libGAP_Obj self) {
  return libGAP_INTOBJ_INT(getpid());
}


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "Runtime", 0, "",
      libGAP_FuncRuntime, "src/gap.c:Runtime" },

    { "RUNTIMES", 0, "",
      libGAP_FuncRUNTIMES, "src/gap.c:RUNTIMES" },

    { "SizeScreen", -1, "args",
      libGAP_FuncSizeScreen, "src/gap.c:SizeScreen" },

    { "ID_FUNC", 1, "object",
      libGAP_FuncID_FUNC, "src/gap.c:ID_FUNC" },

    { "RESTART_GAP", 1, "cmdline",
      libGAP_FuncRESTART_GAP, "src/gap.c:RESTART_GAP" },

    { "ExportToKernelFinished", 0, "",
      libGAP_FuncExportToKernelFinished, "src/gap.c:ExportToKernelFinished" },

    { "DownEnv", -1, "args",
      libGAP_FuncDownEnv, "src/gap.c:DownEnv" },

    { "UpEnv", -1, "args",
      libGAP_FuncUpEnv, "src/gap.c:UpEnv" },

    { "GAP_CRC", 1, "filename",
      libGAP_FuncGAP_CRC, "src/gap.c:GAP_CRC" },

    { "LOAD_DYN", 2, "filename, crc",
      libGAP_FuncLOAD_DYN, "src/gap.c:LOAD_DYN" },

    { "LOAD_STAT", 2, "filename, crc",
      libGAP_FuncLOAD_STAT, "src/gap.c:LOAD_STAT" },

    { "SHOW_STAT", 0, "",
      libGAP_FuncSHOW_STAT, "src/gap.c:SHOW_STAT" },

    { "GASMAN", -1, "args",
      libGAP_FuncGASMAN, "src/gap.c:GASMAN" },

    { "GASMAN_STATS", 0, "",
      libGAP_FuncGASMAN_STATS, "src/gap.c:GASMAN_STATS" },

    { "GASMAN_MESSAGE_STATUS", 0, "",
      libGAP_FuncGASMAN_MESSAGE_STATUS, "src/gap.c:GASMAN_MESSAGE_STATUS" },

    { "GASMAN_LIMITS", 0, "",
      libGAP_FuncGASMAN_LIMITS, "src/gap.c:GASMAN_LIMITS" },

    { "SHALLOW_SIZE", 1, "object",
      libGAP_FuncSHALLOW_SIZE, "src/gap.c:SHALLOW_SIZE" },

    { "TNUM_OBJ", 1, "object",
      libGAP_FuncTNUM_OBJ, "src/gap.c:TNUM_OBJ" },

    { "TNUM_OBJ_INT", 1, "object",
      libGAP_FuncTNUM_OBJ_INT, "src/gap.c:TNUM_OBJ_INT" },

    { "XTNUM_OBJ", 1, "object",
      libGAP_FuncXTNUM_OBJ, "src/gap.c:XTNUM_OBJ" },

    { "OBJ_HANDLE", 1, "object",
      libGAP_FuncOBJ_HANDLE, "src/gap.c:OBJ_HANDLE" },

    { "HANDLE_OBJ", 1, "object",
      libGAP_FuncHANDLE_OBJ, "src/gap.c:HANDLE_OBJ" },

    { "SWAP_MPTR", 2, "obj1, obj2",
      libGAP_FuncSWAP_MPTR, "src/gap.c:SWAP_MPTR" },

    { "LoadedModules", 0, "",
      libGAP_FuncLoadedModules, "src/gap.c:LoadedModules" },

    { "WindowCmd", 1, "arg-list",
      libGAP_FuncWindowCmd, "src/gap.c:WindowCmd" },


    { "Sleep", 1, "secs",
      libGAP_FuncSleep, "src/gap.c:Sleep" },

    { "QUIT_GAP", 0, "",
      libGAP_FuncQUIT_GAP, "src/gap.c:QUIT_GAP" },


    { "SHELL", -1, "context, canReturnVoid, canReturnObj, lastDepth, setTime, prompt, promptHook, infile, outfile",
      libGAP_FuncSHELL, "src/gap.c:FuncSHELL" },

    { "CALL_WITH_CATCH", 2, "func, args",
      libGAP_FuncCALL_WITH_CATCH, "src/gap.c:CALL_WITH_CATCH" },

    { "JUMP_TO_CATCH", 1, "payload",
      libGAP_FuncJUMP_TO_CATCH, "src/gap.c:JUMP_TO_CATCH" },


    { "KERNEL_INFO", 0, "",
      libGAP_FuncKERNEL_INFO, "src/gap.c:KERNEL_INFO" },

    { "SetUserHasQuit", 1, "value",
      libGAP_FuncSetUserHasQuit, "src/gap.c:SetUserHasQuit" },

    { "GETPID", 0, "",
      libGAP_FuncGETPID, "src/gap.c:GETPID" },

    { "MASTER_POINTER_NUMBER", 1, "ob",
      libGAP_FuncMASTER_POINTER_NUMBER, "src/gap.c:MASTER_POINTER_NUMBER" },

    { "FUNC_BODY_SIZE", 1, "f",
      libGAP_FuncFUNC_BODY_SIZE, "src/gap.c:FUNC_BODY_SIZE" },

    { "PRINT_CURRENT_STATEMENT", 1, "context",
      libGAP_FuncPrintExecutingStatement, "src/gap.c:PRINT_CURRENT_STATEMENT" },

  
    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init the completion function                                        */
    libGAP_InitGlobalBag( &libGAP_ThrownObject,      "src/gap.c:ThrownObject"      );

    /* list of exit functions                                              */
    libGAP_InitGlobalBag( &libGAP_AtExitFunctions, "src/gap.c:AtExitFunctions" );
    libGAP_InitGlobalBag( &libGAP_WindowCmdString, "src/gap.c:WindowCmdString" );

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );



    /* establish Fopy of ViewObj                                           */
    libGAP_ImportFuncFromLibrary(  "ViewObj", 0L );
    libGAP_ImportFuncFromLibrary(  "Error", &libGAP_Error );
    libGAP_ImportFuncFromLibrary(  "ErrorInner", &libGAP_ErrorInner );


#if HAVE_SELECT
    libGAP_InitCopyGVar("OnCharReadHookActive",&libGAP_OnCharReadHookActive);
    libGAP_InitCopyGVar("OnCharReadHookInFds",&libGAP_OnCharReadHookInFds);
    libGAP_InitCopyGVar("OnCharReadHookInFuncs",&libGAP_OnCharReadHookInFuncs);
    libGAP_InitCopyGVar("OnCharReadHookOutFds",&libGAP_OnCharReadHookOutFds);
    libGAP_InitCopyGVar("OnCharReadHookOutFuncs",&libGAP_OnCharReadHookOutFuncs);
    libGAP_InitCopyGVar("OnCharReadHookExcFds",&libGAP_OnCharReadHookExcFds);
    libGAP_InitCopyGVar("OnCharReadHookExcFuncs",&libGAP_OnCharReadHookExcFuncs);
#endif

    /* If a package or .gaprc or file read from the command line
       sets this to a function, then we want to know                       */
    libGAP_InitCopyGVar(  "AlternativeMainLoop", &libGAP_AlternativeMainLoop );

    libGAP_InitGlobalBag(&libGAP_ErrorHandler, "gap.c: ErrorHandler");

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
      libGAP_UInt var;

    /* library name and other stuff                                        */
    var = libGAP_GVarName( "DEBUG_LOADING" );
    libGAP_MakeReadWriteGVar(var);
    libGAP_AssGVar( var, (libGAP_SyDebugLoading ? libGAP_True : libGAP_False) );
    libGAP_MakeReadOnlyGVar(var);

    /* construct the `ViewObj' variable                                    */
    libGAP_ViewObjGVar = libGAP_GVarName( "ViewObj" ); 
    libGAP_CustomViewGVar = libGAP_GVarName( "CustomView" ); 

    /* construct the last and time variables                               */
    libGAP_Last              = libGAP_GVarName( "last"  );
    libGAP_Last2             = libGAP_GVarName( "last2" );
    libGAP_Last3             = libGAP_GVarName( "last3" );
    libGAP_Time              = libGAP_GVarName( "time"  );
    libGAP_SaveOnExitFileGVar= libGAP_GVarName( "SaveOnExitFile" );
    libGAP_QUITTINGGVar      = libGAP_GVarName( "QUITTING" );
    
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{



    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* create windows command buffer                                       */
    libGAP_WindowCmdString = libGAP_NEW_STRING( 1000 );
    
    /* return success                                                      */
    return libGAP_PostRestore( libGAP_module );
}


/****************************************************************************
**
*F  InitInfoGap() . . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "gap",                              /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoGap ( void )
{
    return &libGAP_module;
}


/****************************************************************************
**

*V  InitFuncsBuiltinModules . . . . .  list of builtin modules init functions
*/
static libGAP_InitInfoFunc libGAP_InitFuncsBuiltinModules[] = {

    /* global variables                                                    */
    libGAP_InitInfoGVars,

    /* objects                                                             */
    libGAP_InitInfoObjects,

    /* scanner, reader, interpreter, coder, caller, compiler               */
    libGAP_InitInfoScanner,
    libGAP_InitInfoRead,
    libGAP_InitInfoCalls,
    libGAP_InitInfoExprs,
    libGAP_InitInfoStats,
    libGAP_InitInfoCode,
    libGAP_InitInfoVars,       /* must come after InitExpr and InitStats */
    libGAP_InitInfoFuncs,
    libGAP_InitInfoOpers,
    libGAP_InitInfoIntrprtr,
    libGAP_InitInfoCompiler,

    /* arithmetic operations                                               */
    libGAP_InitInfoAriths,
    libGAP_InitInfoInt,
    libGAP_InitInfoIntFuncs,
    libGAP_InitInfoRat,
    libGAP_InitInfoCyc,
    libGAP_InitInfoFinfield,
    libGAP_InitInfoPermutat,
    libGAP_InitInfoTrans,
    libGAP_InitInfoPPerm,
    libGAP_InitInfoBool,
    libGAP_InitInfoMacfloat,

    /* record packages                                                     */
    libGAP_InitInfoRecords,
    libGAP_InitInfoPRecord,

    /* list packages                                                       */
    libGAP_InitInfoLists,
    libGAP_InitInfoListOper,
    libGAP_InitInfoListFunc,
    libGAP_InitInfoPlist,
    libGAP_InitInfoSet,
    libGAP_InitInfoVector,
    libGAP_InitInfoVecFFE,
    libGAP_InitInfoBlist,
    libGAP_InitInfoRange,
    libGAP_InitInfoString,
    libGAP_InitInfoGF2Vec,
    libGAP_InitInfoVec8bit,

    /* free and presented groups                                           */
    libGAP_InitInfoFreeGroupElements,
    libGAP_InitInfoCosetTable,
    libGAP_InitInfoTietze,
    libGAP_InitInfoPcElements,
    libGAP_InitInfoSingleCollector,
    libGAP_InitInfoCombiCollector,
    libGAP_InitInfoPcc,
    libGAP_InitInfoDeepThought,
    libGAP_InitInfoDTEvaluation,

    /* algebras                                                            */
    libGAP_InitInfoSCTable,

    /* save and load workspace, weak pointers                              */
    libGAP_InitInfoWeakPtr,
    libGAP_InitInfoSaveLoad,

    /* input and output                                                    */
    libGAP_InitInfoStreams,
    libGAP_InitInfoSysFiles,
    libGAP_InitInfoIOStream,

    /* main module                                                         */
    libGAP_InitInfoGap,

#ifdef GAPMPI
    /* ParGAP/MPI module                                                   */
    InitInfoGapmpi,
#endif

    0
};


/****************************************************************************
**
*F  Modules . . . . . . . . . . . . . . . . . . . . . . . . . list of modules
*/
#ifndef libGAP_MAX_MODULES
#define libGAP_MAX_MODULES     1000
#endif


#ifndef libGAP_MAX_MODULE_FILENAMES
#define libGAP_MAX_MODULE_FILENAMES (libGAP_MAX_MODULES*50)
#endif

libGAP_Char libGAP_LoadedModuleFilenames[libGAP_MAX_MODULE_FILENAMES];
libGAP_Char *libGAP_NextLoadedModuleFilename = libGAP_LoadedModuleFilenames;


libGAP_StructInitInfo * libGAP_Modules [ libGAP_MAX_MODULES ];
libGAP_UInt libGAP_NrModules;
libGAP_UInt libGAP_NrBuiltinModules;


/****************************************************************************
**
*F  RecordLoadedModule( <module> )  . . . . . . . . store module in <Modules>
*/

void libGAP_RecordLoadedModule (
    libGAP_StructInitInfo *        info,
    libGAP_Char *filename )
{
    libGAP_UInt len;
    if ( libGAP_NrModules == libGAP_MAX_MODULES ) {
        libGAP_Pr( "panic: no room to record module\n", 0L, 0L );
    }
    len = strlen(filename);
    if (libGAP_NextLoadedModuleFilename + len + 1
        > libGAP_LoadedModuleFilenames+libGAP_MAX_MODULE_FILENAMES) {
      libGAP_Pr( "panic: no room for module filename\n", 0L, 0L );
    }
    *libGAP_NextLoadedModuleFilename = '\0';
    memcpy(libGAP_NextLoadedModuleFilename, filename, len+1);
    info->filename = libGAP_NextLoadedModuleFilename;
    libGAP_NextLoadedModuleFilename += len +1;
    libGAP_Modules[libGAP_NrModules++] = info;
}


/****************************************************************************
**
*F  InitializeGap() . . . . . . . . . . . . . . . . . . . . . . intialize GAP
**
**  Each module  (builtin  or compiled) exports  a sturctures  which contains
**  information about the name, version, crc, init function, save and restore
**  functions.
**
**  The init process is split into three different functions:
**
**  `InitKernel':   This function setups the   internal  data structures  and
**  tables,   registers the global bags  and   functions handlers, copies and
**  fopies.  It is not allowed to create objects, gvar or rnam numbers.  This
**  function is used for both starting and restoring.
**
**  `InitLibrary': This function creates objects,  gvar and rnam number,  and
**  does  assignments of auxillary C   variables (for example, pointers  from
**  objects, length of hash lists).  This function is only used for starting.
**
**  `PostRestore': Everything in  `InitLibrary' execpt  creating objects.  In
**  general    `InitLibrary'  will  create    all objects    and  then  calls
**  `PostRestore'.  This function is only used when restoring.
*/
extern libGAP_TNumMarkFuncBags libGAP_TabMarkFuncBags [ 256 ];

static libGAP_Obj libGAP_POST_RESTORE;

void libGAP_InitializeGap (
    int *               pargc,
    char *              argv [] )
{
  /*    UInt                type; */
    libGAP_UInt                i;
    libGAP_Int                 ret;


    /* initialize the basic system and gasman                              */
#ifdef GAPMPI
    /* ParGAP/MPI needs to call MPI_Init() first to remove command line args */
    InitGapmpi( pargc, &argv );
#endif

    libGAP_InitSystem( *pargc, argv );

    /* Initialise memory  -- have to do this here to make sure we are at top of C stack */
    libGAP_InitBags( libGAP_SyAllocBags, libGAP_SyStorMin,
              0, (libGAP_Bag*)(((libGAP_UInt)pargc/libGAP_SyStackAlign)*libGAP_SyStackAlign), libGAP_SyStackAlign,
              libGAP_SyCacheSize, 0, libGAP_SyAbortBags );
              libGAP_InitMsgsFuncBags( libGAP_SyMsgsBags ); 


    /* get info structures for the build in modules                        */
    libGAP_NrModules = 0;
    for ( i = 0;  libGAP_InitFuncsBuiltinModules[i];  i++ ) {
        if ( libGAP_NrModules == libGAP_MAX_MODULES ) {
            libGAP_FPUTS_TO_STDERR( "panic: too many builtin modules\n" );
            libGAP_SyExit(1);
        }
        libGAP_Modules[libGAP_NrModules++] = libGAP_InitFuncsBuiltinModules[i]();
#       ifdef DEBUG_LOADING
            libGAP_FPUTS_TO_STDERR( "#I  InitInfo(builtin " );
            libGAP_FPUTS_TO_STDERR( libGAP_Modules[libGAP_NrModules-1]->name );
            libGAP_FPUTS_TO_STDERR( ")\n" );
#       endif
    }
    libGAP_NrBuiltinModules = libGAP_NrModules;

    /* call kernel initialisation                                          */
    for ( i = 0;  i < libGAP_NrBuiltinModules;  i++ ) {
        if ( libGAP_Modules[i]->initKernel ) {
#           ifdef DEBUG_LOADING
                libGAP_FPUTS_TO_STDERR( "#I  InitKernel(builtin " );
                libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                libGAP_FPUTS_TO_STDERR( ")\n" );
#           endif
            ret =libGAP_Modules[i]->initKernel( libGAP_Modules[i] );
            if ( ret ) {
                libGAP_FPUTS_TO_STDERR( "#I  InitKernel(builtin " );
                libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                libGAP_FPUTS_TO_STDERR( ") returned non-zero value\n" );
            }
        }
    }

    libGAP_InitGlobalBag(&libGAP_POST_RESTORE, "gap.c: POST_RESTORE");
    libGAP_InitFopyGVar( "POST_RESTORE", &libGAP_POST_RESTORE);

    /* you should set 'COUNT_BAGS' as well                                 */
#   ifdef DEBUG_LOADING
        if ( libGAP_SyRestoring ) {
            libGAP_Pr( "#W  after setup\n", 0L, 0L );
            libGAP_Pr( "#W  %36s ", (libGAP_Int)"type",  0L          );
            libGAP_Pr( "%8s %8s ",  (libGAP_Int)"alive", (libGAP_Int)"kbyte" );
            libGAP_Pr( "%8s %8s\n",  (libGAP_Int)"total", (libGAP_Int)"kbyte" );
            for ( i = 0;  i < 256;  i++ ) {
                if ( libGAP_InfoBags[i].name != 0 && libGAP_InfoBags[i].nrAll != 0 ) {
                    char    buf[41];

                    buf[0] = '\0';
                    libGAP_strlcat( buf, libGAP_InfoBags[i].name, sizeof(buf) );
                    libGAP_Pr("#W  %36s ",    (libGAP_Int)buf, 0L );
                    libGAP_Pr("%8d %8d ", (libGAP_Int)libGAP_InfoBags[i].nrLive,
                       (libGAP_Int)(libGAP_InfoBags[i].sizeLive/1024));
                    libGAP_Pr("%8d %8d\n",(libGAP_Int)libGAP_InfoBags[i].nrAll,
                       (libGAP_Int)(libGAP_InfoBags[i].sizeAll/1024));
                }
            }
        }
#   endif

    /* and now for a special hack                                          */
    for ( i = libGAP_LAST_CONSTANT_TNUM+1; i <= libGAP_LAST_REAL_TNUM; i++ ) {
      if (libGAP_TabMarkFuncBags[i + libGAP_COPYING] == libGAP_MarkAllSubBagsDefault)
        libGAP_TabMarkFuncBags[ i+libGAP_COPYING ] = libGAP_TabMarkFuncBags[ i ];
    }

    /* if we are restoring, load the workspace and call the post restore   */
    if ( libGAP_SyRestoring ) {
       libGAP_LoadWorkspace(libGAP_SyRestoring);
        for ( i = 0;  i < libGAP_NrModules;  i++ ) {
            if ( libGAP_Modules[i]->libGAP_postRestore ) {
#               ifdef DEBUG_LOADING
                    libGAP_FPUTS_TO_STDERR( "#I  PostRestore(builtin " );
                    libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                    libGAP_FPUTS_TO_STDERR( ")\n" );
#               endif
                ret = libGAP_Modules[i]->libGAP_postRestore( libGAP_Modules[i] );
                if ( ret ) {
                    libGAP_FPUTS_TO_STDERR( "#I  PostRestore(builtin " );
                    libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                    libGAP_FPUTS_TO_STDERR( ") returned non-zero value\n" );
                }
            }
        }
        libGAP_SyRestoring = NULL;


        /* Call POST_RESTORE which is a GAP function that now takes control, 
           calls the post restore functions and then runs a GAP session */
        if (libGAP_POST_RESTORE != (libGAP_Obj) 0 &&
            libGAP_IS_FUNC(libGAP_POST_RESTORE))
          if (!libGAP_READ_ERROR())
            libGAP_CALL_0ARGS(libGAP_POST_RESTORE);
    }


    /* otherwise call library initialisation                               */
    else {
        libGAP_WarnInitGlobalBag = 1;
#       ifdef DEBUG_HANDLER_REGISTRATION
            libGAP_CheckAllHandlers();
#       endif

        libGAP_SyInitializing = 1;    
        for ( i = 0;  i < libGAP_NrBuiltinModules;  i++ ) {
            if ( libGAP_Modules[i]->initLibrary ) {
#               ifdef DEBUG_LOADING
                    libGAP_FPUTS_TO_STDERR( "#I  InitLibrary(builtin " );
                    libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                    libGAP_FPUTS_TO_STDERR( ")\n" );
#               endif
                ret = libGAP_Modules[i]->initLibrary( libGAP_Modules[i] );
                if ( ret ) {
                    libGAP_FPUTS_TO_STDERR( "#I  InitLibrary(builtin " );
                    libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                    libGAP_FPUTS_TO_STDERR( ") returned non-zero value\n" );
                }
            }
        }
        libGAP_WarnInitGlobalBag = 0;
    }

    /* check initialisation                                                */
    for ( i = 0;  i < libGAP_NrModules;  i++ ) {
        if ( libGAP_Modules[i]->checkInit ) {
#           ifdef DEBUG_LOADING
                libGAP_FPUTS_TO_STDERR( "#I  CheckInit(builtin " );
                libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                libGAP_FPUTS_TO_STDERR( ")\n" );
#           endif
            ret = libGAP_Modules[i]->checkInit( libGAP_Modules[i] );
            if ( ret ) {
                libGAP_FPUTS_TO_STDERR( "#I  CheckInit(builtin " );
                libGAP_FPUTS_TO_STDERR( libGAP_Modules[i]->name );
                libGAP_FPUTS_TO_STDERR( ") returned non-zero value\n" );
            }
        }
    }

    /* read the init files      
       this now actually runs the GAP session, we only get 
       past here when we're about to exit. 
                                           */
    if ( libGAP_SySystemInitFile[0] ) {
      if (!libGAP_READ_ERROR()) {
        if ( libGAP_READ_GAP_ROOT(libGAP_SySystemInitFile) == 0 ) {
          /*             if ( ! SyQuiet ) { */
                libGAP_Pr( "gap: hmm, I cannot find '%s' maybe",
                    (libGAP_Int)libGAP_SySystemInitFile, 0L );
                libGAP_Pr( " use option '-l <gaproot>'?\n If you ran the GAP"
                    " binary directly, try running the 'gap.sh' or 'gap.bat'"
                    " script instead.", 0L, 0L );
            }
      }
      else
        {
          libGAP_Pr("Caught error at top-most level, probably quit from library loading",0L,0L);
          libGAP_SyExit(1);
        }
        /*         } */
    }

}


/****************************************************************************
**

*E  gap.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/






