/*******************************************************************************
*
* A transformation <f> has internal representation as follows:
*
* [Obj* image set, Obj* flat kernel, Obj* external degree, entries image list]
*
* The <internal degree> of <f> is just the length of <entries image list>, this
* is accessed here using <DEG_TRANS2> and <DEG_TRANS4>, in GAP it can be accessed
* using INT_DEG_TRANS (for debugging purposes only).
*
* Transformations must always have internal degree greater than or equal to the
* largest point in <entries image list>. 
* 
* An element of <entries image list> of a transformation in T_TRANS2 must be at
* most 65535 and be UInt2. Hence the internal and external degrees of a T_TRANS2
* are at most 65536. If <f> is T_TRANS4, then the elements of <entries
* image list> must be UInt4.
*
* The <image set> and <flat kernel> are found relative to the internal degree of
* the transformation, and must not be changed after they are first found. 
*
* The <external degree> is the largest non-negative integer <n> such that
* <n^f!=n> or <i^f=n> for some <i!=n>, or equivalently the degree of a
* transformation is the least non-negative <n> such that <[n+1,n+2,...]> is
* fixed pointwise by <f>. This value is an invariant of <f>, in the sense that
* it does not depend on the internal representation. In GAP,
* <DegreeOfTransformation(f)> returns the external degree, so that if <f=g>,
* then <DegreeOfTransformation(f)=DegreeOfTransformation(g)>.
*
* In this file, the external degree of a transformation is accessed using
* <FuncDegreeOfTransformation(f)> (if it is not known if <EXT_TRANS(f)==NULL>)
* or <EXT_TRANS> (if it is known that <EXT_TRANS(f)!=NULL>).
* 
*******************************************************************************/

#include        "trans.h"               /* transformations                 */

#define libGAP_MIN(a,b)          (a<b?a:b)
#define libGAP_MAX(a,b)          (a<b?b:a)

#define libGAP_IMG_TRANS(f)       (*(libGAP_Obj*)(libGAP_ADDR_OBJ(f)))
#define libGAP_KER_TRANS(f)       (*((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+1))
#define libGAP_EXT_TRANS(f)       (*((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+2))

#define libGAP_NEW_TRANS2(deg)     libGAP_NewBag(libGAP_T_TRANS2, deg*sizeof(libGAP_UInt2)+3*sizeof(libGAP_Obj))
#define libGAP_ADDR_TRANS2(f)      ((libGAP_UInt2*)((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+3))
#define libGAP_DEG_TRANS2(f)       ((libGAP_UInt)(libGAP_SIZE_OBJ(f)-3*sizeof(libGAP_Obj))/sizeof(libGAP_UInt2))
#define libGAP_RANK_TRANS2(f)      (libGAP_IMG_TRANS(f)==NULL?libGAP_INIT_TRANS2(f):libGAP_LEN_PLIST(libGAP_IMG_TRANS(f)))

#define libGAP_NEW_TRANS4(deg)     libGAP_NewBag(libGAP_T_TRANS4, deg*sizeof(libGAP_UInt4)+3*sizeof(libGAP_Obj))
#define libGAP_ADDR_TRANS4(f)      ((libGAP_UInt4*)((libGAP_Obj*)(libGAP_ADDR_OBJ(f))+3))
#define libGAP_DEG_TRANS4(f)       ((libGAP_UInt)(libGAP_SIZE_OBJ(f)-3*sizeof(libGAP_Obj))/sizeof(libGAP_UInt4))
#define libGAP_RANK_TRANS4(f)      (libGAP_IMG_TRANS(f)==NULL?libGAP_INIT_TRANS4(f):libGAP_LEN_PLIST(libGAP_IMG_TRANS(f)))

#define libGAP_IS_TRANS(f)       (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2||libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4)
#define libGAP_RANK_TRANS(f)     (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2?libGAP_RANK_TRANS2(f):libGAP_RANK_TRANS4(f))
#define libGAP_DEG_TRANS(f)      (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2?libGAP_DEG_TRANS2(f):libGAP_DEG_TRANS4(f))

// TmpTrans is the same as TmpPerm

libGAP_Obj libGAP_TmpTrans;
libGAP_Obj libGAP_IdentityTrans;

/*******************************************************************************
** Static functions for transformations
*******************************************************************************/

static inline void libGAP_ResizeTmpTrans( libGAP_UInt len ){
  if(libGAP_SIZE_OBJ(libGAP_TmpTrans)<len*sizeof(libGAP_UInt4)){
    libGAP_ResizeBag(libGAP_TmpTrans,len*sizeof(libGAP_UInt4));
  }
}

static inline libGAP_UInt4 * libGAP_ResizeInitTmpTrans( libGAP_UInt len ){
  libGAP_UInt    i;
  libGAP_UInt4   *pttmp;

  if(libGAP_SIZE_OBJ(libGAP_TmpTrans)<len*sizeof(libGAP_UInt4)){
    libGAP_ResizeBag(libGAP_TmpTrans,len*sizeof(libGAP_UInt4));
  }
  pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  for(i=0;i<len;i++) pttmp[i]=0;
  return pttmp;
}

/* find rank, canonical trans same kernel, and img set (unsorted) */
static libGAP_UInt libGAP_INIT_TRANS2(libGAP_Obj f){ 
  libGAP_UInt    deg, rank, i, j;
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *pttmp;
  libGAP_Obj     img, ker;

  deg=libGAP_DEG_TRANS2(f);
  
  if(deg==0){//special case for degree 0
    img=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(img, 0);
    libGAP_IMG_TRANS(f)=img;
    libGAP_KER_TRANS(f)=img;
    libGAP_CHANGED_BAG(f);
    return 0;
  }

  img=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, deg);
  ker=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_NSORT+libGAP_IMMUTABLE, deg);
  libGAP_SET_LEN_PLIST(ker, (libGAP_Int) deg);

  pttmp=libGAP_ResizeInitTmpTrans(deg);
  ptf=libGAP_ADDR_TRANS2(f); 
 
  rank=0;
  for(i=0;i<deg;i++){        
    j=ptf[i];               /* f(i) */
    if(pttmp[j]==0){ 
      pttmp[j]=++rank;         
      libGAP_SET_ELM_PLIST(img, rank, libGAP_INTOBJ_INT(j+1));
    }
    libGAP_SET_ELM_PLIST(ker, i+1, libGAP_INTOBJ_INT(pttmp[j]));
  }

  libGAP_SHRINK_PLIST(img, (libGAP_Int) rank);
  libGAP_SET_LEN_PLIST(img, (libGAP_Int) rank);
  
  libGAP_IMG_TRANS(f)=img;
  libGAP_KER_TRANS(f)=ker;
  libGAP_CHANGED_BAG(f);
  return rank;
}


static libGAP_UInt libGAP_INIT_TRANS4(libGAP_Obj f){ 
  libGAP_UInt    deg, rank, i, j;
  libGAP_UInt4   *ptf;
  libGAP_UInt4   *pttmp;
  libGAP_Obj     img, ker;

  deg=libGAP_DEG_TRANS4(f);
  
  if(deg==0){//special case for degree 0
    img=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(img, 0);
    libGAP_IMG_TRANS(f)=img;
    libGAP_KER_TRANS(f)=img;
    libGAP_CHANGED_BAG(f);
    return 0;
  }

  img=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, deg);
  ker=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_NSORT+libGAP_IMMUTABLE, deg);
  libGAP_SET_LEN_PLIST(ker, (libGAP_Int) deg);

  pttmp=libGAP_ResizeInitTmpTrans(deg);
  ptf=libGAP_ADDR_TRANS4(f); 
 
  rank=0;
  for(i=0;i<deg;i++){        
    j=ptf[i];               /* f(i) */
    if(pttmp[j]==0){ 
      pttmp[j]=++rank;         
      libGAP_SET_ELM_PLIST(img, rank, libGAP_INTOBJ_INT(j+1));
    }
    libGAP_SET_ELM_PLIST(ker, i+1, libGAP_INTOBJ_INT(pttmp[j]));
  }

  libGAP_SHRINK_PLIST(img, (libGAP_Int) rank);
  libGAP_SET_LEN_PLIST(img, (libGAP_Int) rank);
  
  libGAP_IMG_TRANS(f)=img;
  libGAP_KER_TRANS(f)=ker;
  libGAP_CHANGED_BAG(f);
  return rank;
}


static libGAP_Obj libGAP_SORT_PLIST_CYC(libGAP_Obj res){
  libGAP_Obj     tmp;      
  libGAP_UInt    h, i, k, len;
  
  len=libGAP_LEN_PLIST(res); 
  h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1;
  while ( 0 < h ) {
    for ( i = h+1; i <= len; i++ ) {
      tmp = libGAP_ADDR_OBJ(res)[i];  k = i;
      while ( h < k && ((libGAP_Int)tmp < (libGAP_Int)(libGAP_ADDR_OBJ(res)[k-h])) ) {
        libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h];
        k -= h;
      }
      libGAP_ADDR_OBJ(res)[k] = tmp;
    }
    h = h / 3;
  }
  libGAP_RetypeBag(res, libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE); 
  libGAP_CHANGED_BAG(res);
  return res;
}

/*******************************************************************************
** GAP functions for transformations
*******************************************************************************/

//for debugging...
libGAP_Obj libGAP_FuncHAS_KER_TRANS( libGAP_Obj self, libGAP_Obj f ){
  if(libGAP_IS_TRANS(f)){
    return (libGAP_KER_TRANS(f)==NULL?libGAP_False:libGAP_True);
  } else {
    return libGAP_Fail;
  }
}

libGAP_Obj libGAP_FuncHAS_IMG_TRANS( libGAP_Obj self, libGAP_Obj f ){
  if(libGAP_IS_TRANS(f)){
    return (libGAP_IMG_TRANS(f)==NULL?libGAP_False:libGAP_True);
  } else {
    return libGAP_Fail;
  }
}

libGAP_Obj libGAP_FuncINT_DEG_TRANS( libGAP_Obj self, libGAP_Obj f ){
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    return libGAP_INTOBJ_INT(libGAP_DEG_TRANS2(f));
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    return libGAP_INTOBJ_INT(libGAP_DEG_TRANS4(f));
  }
  return libGAP_Fail;
}

//this only works when <f> is T_TRANS2 and deg(f)<=m<10
libGAP_Obj libGAP_FuncNUMB_TRANS_INT(libGAP_Obj self, libGAP_Obj f, libGAP_Obj m){
  libGAP_UInt                n, a, i, def;
  libGAP_UInt2               *ptf2;

  n=libGAP_INT_INTOBJ(m);  a=0;
  def=libGAP_DEG_TRANS2(f);
  ptf2=libGAP_ADDR_TRANS2(f);
  for(i=0;i<def;i++)  a=a*n+ptf2[i];
  for(;i<n;i++)       a=a*n+i;      
  return libGAP_INTOBJ_INT(a+1);  
}  
 
//this only works when <f> is T_TRANS2 and deg(f)<=m<10
libGAP_Obj libGAP_FuncTRANS_NUMB_INT(libGAP_Obj self, libGAP_Obj b, libGAP_Obj m){
  libGAP_UInt    n, a, i, q;
  libGAP_Obj     f;
  libGAP_UInt2*  ptf;

  n=libGAP_INT_INTOBJ(m);   a=libGAP_INT_INTOBJ(b)-1;
  f=libGAP_NEW_TRANS2(n);   ptf=libGAP_ADDR_TRANS2(f);
  for(i=n;0<i;i--){
    q=a/n;  ptf[i-1]=a-q*n;  a=q; 
  } 
  return f;
}

/* method for creating transformation */
libGAP_Obj libGAP_FuncTransformationNC( libGAP_Obj self, libGAP_Obj list ){ 
  libGAP_UInt    i, deg;
  libGAP_UInt2*  ptf2;
  libGAP_UInt4*  ptf4;
  libGAP_Obj     f; 
 
  deg=libGAP_LEN_LIST(list);
  
  if(deg<=65536){ 
    f=libGAP_NEW_TRANS2(deg);
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=0;i<deg;i++) ptf2[i]=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i+1))-1;
  }else{
    f=libGAP_NEW_TRANS4(deg);
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=0;i<deg;i++) ptf4[i]=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i+1))-1;
  }
  return f; 
}


libGAP_Obj libGAP_FuncTransformationListListNC( libGAP_Obj self, libGAP_Obj src, libGAP_Obj ran ){ 
  libGAP_UInt    deg, i, s, r;
  libGAP_Obj     f;
  libGAP_UInt2*  ptf2;
  libGAP_UInt4*  ptf4;

  if(!libGAP_IS_SMALL_LIST(src)){
    libGAP_ErrorQuit("usage: <src> must be a list (not a %s)", 
        (libGAP_Int)libGAP_TNAM_OBJ(src), 0L);
  }
  if(!libGAP_IS_SMALL_LIST(ran)){
    libGAP_ErrorQuit("usage: <ran> must be a list (not a %s)", 
        (libGAP_Int)libGAP_TNAM_OBJ(ran), 0L);
  }
  if(libGAP_LEN_LIST(src)!=libGAP_LEN_LIST(ran)){
    libGAP_ErrorQuit("usage: <src> and <ran> must have equal length,", 0L, 0L);
  }

  deg=0;
  for(i=libGAP_LEN_LIST(src);1<=i;i--){
    s=libGAP_INT_INTOBJ(libGAP_ELM_LIST(src, i));
    if(s>deg) deg=s;
    r=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ran, i));
    if(r>deg) deg=r;
  } 

  if(deg<=65536){ 
    f=libGAP_NEW_TRANS2(deg);
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=0;i<deg;i++) ptf2[i]=i;
    for(i=libGAP_LEN_LIST(src);1<=i;i--){
      ptf2[libGAP_INT_INTOBJ(libGAP_ELM_LIST(src, i))-1]=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ran, i))-1;
    }
  }else{
    f=libGAP_NEW_TRANS4(deg);
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=0;i<deg;i++) ptf4[i]=i;
    for(i=libGAP_LEN_LIST(src);1<=i;i--){
      ptf4[libGAP_INT_INTOBJ(libGAP_ELM_LIST(src, i))-1]=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ran, i))-1;
    }
  }
  return f; 
}

libGAP_Obj libGAP_FuncDegreeOfTransformation(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    n, i, deg;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  libGAP_Obj     ext;

  ext=libGAP_EXT_TRANS(f);
  if(ext!=NULL){
    return ext;
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){ 
    n=libGAP_DEG_TRANS2(f);
    ptf2=libGAP_ADDR_TRANS2(f);
    if(ptf2[n-1]!=n-1){
      ext=libGAP_INTOBJ_INT(n);
    } else {
      deg=0;
      for(i=0;i<n;i++){ 
        if(ptf2[i]>i&&ptf2[i]+1>deg){
          deg=ptf2[i]+1;
        } else if(ptf2[i]<i&&i+1>deg){
          deg=i+1;
        }
      }  
      ext=libGAP_INTOBJ_INT(deg);
    }
    return ext;
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    n=libGAP_DEG_TRANS4(f);
    ptf4=libGAP_ADDR_TRANS4(f);
    if(ptf4[n-1]!=n-1){
      ext=libGAP_INTOBJ_INT(n);
    } else {
      deg=0;
      for(i=0;i<n;i++){ 
        if(ptf4[i]>i&&ptf4[i]+1>deg){
          deg=ptf4[i]+1;
        } else if(ptf4[i]<i&&i+1>deg){
          deg=i+1;
        }
      }  
      ext=libGAP_INTOBJ_INT(deg);
    }
    return ext;
  }
  libGAP_ErrorQuit("usage: the argument should be a transformation,", 0L, 0L);
  return 0L;
}


/* rank of transformation */
libGAP_Obj libGAP_FuncRANK_TRANS(libGAP_Obj self, libGAP_Obj f){ 
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){ 
    return libGAP_SumInt(libGAP_INTOBJ_INT(libGAP_RANK_TRANS2(f)-libGAP_DEG_TRANS2(f)), 
          libGAP_FuncDegreeOfTransformation(self, f)); 
  } else {
    return libGAP_SumInt(libGAP_INTOBJ_INT(libGAP_RANK_TRANS4(f)-libGAP_DEG_TRANS4(f)), 
          libGAP_FuncDegreeOfTransformation(self, f)); 
  }
}

/* corank of transformation 
Obj FuncCORANK_TRANS(Obj self, Obj f){ 
  if(TNUM_OBJ(f)==T_TRANS2){ 
    return INTOBJ_INT(DEG_TRANS2(f)-RANK_TRANS2(f));
  } else {
    return INTOBJ_INT(DEG_TRANS4(f)-RANK_TRANS4(f));
  }
}*/

/* rank of transformation */
libGAP_Obj libGAP_FuncRANK_TRANS_INT(libGAP_Obj self, libGAP_Obj f, libGAP_Obj n){ 
  libGAP_UInt    rank, i, m;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *pttmp, *ptf4;

  m=libGAP_INT_INTOBJ(n);
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){ 
    if(m>=libGAP_DEG_TRANS2(f)){
      return libGAP_INTOBJ_INT(libGAP_RANK_TRANS2(f)-libGAP_DEG_TRANS2(f)+m);
    } else {
      pttmp=libGAP_ResizeInitTmpTrans(libGAP_DEG_TRANS2(f));
      ptf2=libGAP_ADDR_TRANS2(f);
      rank=0; 
      for(i=0;i<m;i++){        
        if(pttmp[ptf2[i]]==0){ 
          rank++;
          pttmp[ptf2[i]]=1;         
        }
      }
      return libGAP_INTOBJ_INT(rank);
    }
  } else {
    if(m>=libGAP_DEG_TRANS2(f)){
      return libGAP_INTOBJ_INT(libGAP_RANK_TRANS4(f)-libGAP_DEG_TRANS4(f)+m);
    } else {
      pttmp=libGAP_ResizeInitTmpTrans(libGAP_DEG_TRANS4(f));
      ptf4=libGAP_ADDR_TRANS4(f);
      rank=0; 
      for(i=0;i<m;i++){        
        if(pttmp[ptf4[i]]==0){ 
          rank++;
          pttmp[ptf4[i]]=1;         
        }
      }
      return libGAP_INTOBJ_INT(rank);
    }
  }
}

libGAP_Obj libGAP_FuncRANK_TRANS_LIST(libGAP_Obj self, libGAP_Obj f, libGAP_Obj list){ 
  libGAP_UInt    rank, i, j, len, def;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *pttmp, *ptf4;
  libGAP_Obj     pt;

  len=libGAP_LEN_LIST(list);
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){ 
    def=libGAP_DEG_TRANS2(f);
    pttmp=libGAP_ResizeInitTmpTrans(def);
    ptf2=libGAP_ADDR_TRANS2(f);
    rank=0; 
    for(i=1;i<=len;i++){
      pt=libGAP_ELM_LIST(list, i);
      if(!libGAP_TNUM_OBJ(pt)==libGAP_T_INT||libGAP_INT_INTOBJ(pt)<1){
        libGAP_ErrorQuit("usage: the second argument <list> must be a list of positive\n integers (not a %s)", (libGAP_Int)libGAP_TNAM_OBJ(pt), 0L);
      }
      j=libGAP_INT_INTOBJ(pt)-1;
      if(j<=def){
        j=ptf2[libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i))-1];
        if(pttmp[j]==0){ rank++; pttmp[j]=1; }
      } else {
        rank++;
      }
    }
  } else {
    def=libGAP_DEG_TRANS4(f);
    pttmp=libGAP_ResizeInitTmpTrans(def);
    ptf4=libGAP_ADDR_TRANS4(f);
    rank=0; 
    for(i=1;i<=len;i++){
      pt=libGAP_ELM_LIST(list, i);
      if(!libGAP_TNUM_OBJ(pt)==libGAP_T_INT||libGAP_INT_INTOBJ(pt)<1){
        libGAP_ErrorQuit("usage: the second argument <list> must be a list of positive\n integers (not a %s)", (libGAP_Int)libGAP_TNAM_OBJ(pt), 0L);
      }
      j=libGAP_INT_INTOBJ(pt)-1;
      if(j<=def){
        j=ptf4[libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i))-1];
        if(pttmp[j]==0){ rank++; pttmp[j]=1; }
      } else {
        rank++;
      }
    }
  }
  return libGAP_INTOBJ_INT(rank);
}

/* test if a transformation is the identity. */

libGAP_Obj libGAP_FuncIS_ID_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2*  ptf2=libGAP_ADDR_TRANS2(f);
  libGAP_UInt4*  ptf4=libGAP_ADDR_TRANS4(f);
  libGAP_UInt    deg, i; 

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    for(i=0;i<deg;i++){
      if(ptf2[i]!=i){
        return libGAP_False;
      }
    }
  } else {
    deg=libGAP_DEG_TRANS4(f);
    for(i=0;i<deg;i++){
      if(ptf4[i]!=i){
        return libGAP_False;
      }
    }
  }
  return libGAP_True;
}


libGAP_Obj libGAP_FuncLARGEST_MOVED_PT_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  libGAP_UInt    i;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=libGAP_DEG_TRANS2(f);1<=i;i--){
      if(ptf2[i-1]!=i-1) break;
    }
    return libGAP_INTOBJ_INT(i);
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){ 
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=libGAP_DEG_TRANS4(f);1<=i;i--){
      if(ptf4[i-1]!=i-1) break;
    }
    return libGAP_INTOBJ_INT(i);
  }
  return 0L;
}

// the largest point in [1..LargestMovedPoint(f)]^f

libGAP_Obj libGAP_FuncLARGEST_IMAGE_PT (libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  libGAP_UInt    i, max, def;
  
  if(!libGAP_IS_TRANS(f)){
    libGAP_ErrorQuit("usage: the argument should be a transformation,", 0L, 0L);
  }
 
  max=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    def=libGAP_DEG_TRANS2(f);
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=libGAP_DEG_TRANS2(f);1<=i;i--){ if(ptf2[i-1]!=i-1) break; }
    for(;1<=i;i--){ 
      if(ptf2[i-1]+1>max){
        max=ptf2[i-1]+1; 
        if(max==def) break;
      }
    }
  } else {
    def=libGAP_DEG_TRANS4(f);
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=libGAP_DEG_TRANS4(f);1<=i;i--){ if(ptf4[i-1]!=i-1) break; }
    for(;1<=i;i--){ 
      if(ptf4[i-1]+1>max){ 
        max=ptf4[i-1]+1;
        if(max==def) break;
      }
    }
  }
  return libGAP_INTOBJ_INT(max);
}

// returns the wrong answer when applied to the identity
libGAP_Obj libGAP_FuncSMALLEST_MOVED_PT_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  libGAP_UInt    i, deg;
 
  if(!libGAP_IS_TRANS(f)){
    libGAP_ErrorQuit("usage: the argument should be a transformation,", 0L, 0L);
  }
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    deg=libGAP_DEG_TRANS2(f);
    for(i=1;i<=deg;i++) if(ptf2[i-1]!=i-1) break;
  } else { 
    ptf4=libGAP_ADDR_TRANS4(f);
    deg=libGAP_DEG_TRANS4(f);
    for(i=1;i<=deg;i++) if(ptf4[i-1]!=i-1) break;
  }
  return libGAP_INTOBJ_INT(i);
}

// the smallest point in [SmallestMovedPoint..LargestMovedPoint(f)]^f
libGAP_Obj libGAP_FuncSMALLEST_IMAGE_PT (libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  libGAP_UInt    i, min, deg;
  
  if(!libGAP_IS_TRANS(f)){
    libGAP_ErrorQuit("usage: the argument should be a transformation,", 0L, 0L);
  }
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    deg=libGAP_DEG_TRANS2(f); 
    min=deg;
    for(i=0;i<deg;i++){ if(ptf2[i]!=i&&ptf2[i]<min) min=ptf2[i]; }
  } else {
    ptf4=libGAP_ADDR_TRANS4(f);
    deg=libGAP_DEG_TRANS4(f);
    min=deg;
    for(i=0;i<deg;i++){ if(ptf4[i]!=i&&ptf4[i]<min) min=ptf4[i]; }
  }
  return libGAP_INTOBJ_INT(min+1);
}

 
libGAP_Obj libGAP_FuncNR_MOVED_PTS_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    nr, i, deg;
  libGAP_UInt2*  ptf2;
  libGAP_UInt4*  ptf4;

  if(!libGAP_IS_TRANS(f)){
    libGAP_ErrorQuit("usage: the argument should be a transformation,", 0L, 0L);
  }

  nr=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    deg=libGAP_DEG_TRANS2(f);
    for(i=0;i<deg;i++){ if(ptf2[i]!=i) nr++; }
  } else {
    ptf4=libGAP_ADDR_TRANS4(f);
    deg=libGAP_DEG_TRANS4(f);
    for(i=0;i<deg;i++){ if(ptf4[i]!=i) nr++; }
  }
  return libGAP_INTOBJ_INT(nr);
}


 
libGAP_Obj libGAP_FuncMOVED_PTS_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    len, deg, i, k;
  libGAP_Obj     out, tmp;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;

  if(!libGAP_IS_TRANS(f)){
    libGAP_ErrorQuit("usage: the argument should be a transformation,", 0L, 0L);
  }

  if(libGAP_FuncIS_ID_TRANS(self, f)==libGAP_True){
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(out, 0);
    return out;
  }

  len=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg);
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=0;i<deg;i++){
      if(ptf2[i]!=i) libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
    }
  } else {
    deg=libGAP_DEG_TRANS4(f);
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg);
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=0;i<deg;i++){
      if(ptf4[i]!=i) libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
    }
  }
  
  // remove duplicates
  tmp=libGAP_ADDR_OBJ(out)[1];  k = 1;
  for(i=2;i<=len;i++){
    if(libGAP_INT_INTOBJ(tmp)!=libGAP_INT_INTOBJ(libGAP_ADDR_OBJ(out)[i])) {
      k++;
      tmp = libGAP_ADDR_OBJ(out)[i];
      libGAP_ADDR_OBJ(out)[k] = tmp;
    }
  }

  if(k<len||len<deg){
    libGAP_ResizeBag(out, (k+1)*sizeof(libGAP_Obj) );
  }
  libGAP_SET_LEN_PLIST(out, k);
  return out;
}

/* kernel of transformation */
libGAP_Obj libGAP_FuncFLAT_KERNEL_TRANS (libGAP_Obj self, libGAP_Obj f){ 

  if(libGAP_KER_TRANS(f)==NULL){
    if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
      libGAP_INIT_TRANS2(f);
    } else {
      libGAP_INIT_TRANS4(f);
    }
  }
  return libGAP_KER_TRANS(f);
} 

libGAP_Obj libGAP_FuncFLAT_KERNEL_TRANS_INT (libGAP_Obj self, libGAP_Obj f, libGAP_Obj n){
  libGAP_Obj     new, *ptnew, *ptker; 
  libGAP_UInt    deg, m, i;

  m=libGAP_INT_INTOBJ(n);
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    if(libGAP_KER_TRANS(f)==NULL) libGAP_INIT_TRANS2(f);
    deg=libGAP_DEG_TRANS2(f);
    if(m==deg){
      return libGAP_KER_TRANS(f);
    } else if(m==0){
      new=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
      libGAP_SET_LEN_PLIST(new, 0);
      return new;
    } else {
      new=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_NSORT, m);
      libGAP_SET_LEN_PLIST(new, m);
      
      ptker=libGAP_ADDR_OBJ(libGAP_KER_TRANS(f))+1;
      ptnew=libGAP_ADDR_OBJ(new)+1;

      //copy the kernel set up to minimum of m, deg
      if(m<deg){
        for(i=0;i<m;i++)      *ptnew++=*ptker++;
      } else { //m>deg
        for(i=0;i<deg;i++)    *ptnew++=*ptker++;
        //add new points
        for(i=libGAP_RANK_TRANS2(f)+1;i<=m;i++) *ptnew++=libGAP_INTOBJ_INT(i);
      }
      return new;
    }
  }else{
    if(libGAP_KER_TRANS(f)==NULL) libGAP_INIT_TRANS4(f);
    deg=libGAP_DEG_TRANS4(f);
    if(m==deg){
      return libGAP_KER_TRANS(f);
    } else if(m==0){
      new=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
      libGAP_SET_LEN_PLIST(new, 0);
      return new;
    } else {
      new=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_NSORT, m);
      libGAP_SET_LEN_PLIST(new, m);
      
      ptker=libGAP_ADDR_OBJ(libGAP_KER_TRANS(f))+1;
      ptnew=libGAP_ADDR_OBJ(new)+1;

      //copy the kernel set up to minimum of m, deg
      if(m<deg){
        for(i=0;i<m;i++)      *ptnew++=*ptker++;
      } else { //m>deg
        for(i=0;i<deg;i++)    *ptnew++=*ptker++;
        //add new points
        for(i=libGAP_RANK_TRANS4(f)+1;i<=m;i++) *ptnew++=libGAP_INTOBJ_INT(i);
      }
      return new;
    }
  }
}

/* image set of transformation */
libGAP_Obj libGAP_FuncIMAGE_SET_TRANS (libGAP_Obj self, libGAP_Obj f){ 
  if(libGAP_IMG_TRANS(f)==NULL){
    if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
      libGAP_INIT_TRANS2(f);
    } else {
      libGAP_INIT_TRANS4(f);
    }
  }
  if(!libGAP_IS_SSORT_LIST(libGAP_IMG_TRANS(f))){
    return libGAP_SORT_PLIST_CYC(libGAP_IMG_TRANS(f));
  }
  return libGAP_IMG_TRANS(f);  
} 

//the image set of <f> when applied to <1..n> 

libGAP_Obj libGAP_FuncIMAGE_SET_TRANS_INT (libGAP_Obj self, libGAP_Obj f, libGAP_Obj n){ 
  libGAP_Obj     im, new; 
  libGAP_UInt    deg, m, len, i, j, rank;
  libGAP_Obj     *ptnew, *ptim;
  libGAP_UInt4   *pttmp, *ptf4;
  libGAP_UInt2   *ptf2;

  m=libGAP_INT_INTOBJ(n);
  deg=libGAP_DEG_TRANS(f);

  if(m==deg){
    return libGAP_FuncIMAGE_SET_TRANS(self, f);
  } else if(m==0){
    new=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(new, 0);
    return new;
  } else if(m<deg){
    //JDM add a check to see if IMAGE_SET_TRANS is known
    pttmp=libGAP_ResizeInitTmpTrans(deg);
    new=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, m);
    pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
    
    if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
      ptf2=libGAP_ADDR_TRANS2(f);
      rank=0;
      for(i=0;i<m;i++){        
        j=ptf2[i];               /* f(i) */
        if(pttmp[j]==0){ 
          pttmp[j]=++rank;         
          libGAP_SET_ELM_PLIST(new, rank, libGAP_INTOBJ_INT(j+1));
        }
      }
    } else {
      ptf4=libGAP_ADDR_TRANS4(f);
      rank=0;
      for(i=0;i<m;i++){        
        j=ptf4[i];               /* f(i) */
        if(pttmp[j]==0){ 
          pttmp[j]=++rank;         
          libGAP_SET_ELM_PLIST(new, rank, libGAP_INTOBJ_INT(j+1));
        }
      }
    }
    libGAP_SHRINK_PLIST(new, (libGAP_Int) rank);
    libGAP_SET_LEN_PLIST(new, (libGAP_Int) rank);
    libGAP_SORT_PLIST_CYC(new);
  } else {    //m>deg and so m is at least 1!
    im=libGAP_FuncIMAGE_SET_TRANS(self, f);
    len=libGAP_LEN_PLIST(im);
    new=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, m-deg+len);
    libGAP_SET_LEN_PLIST(new, m-deg+len);
    
    ptnew=libGAP_ADDR_OBJ(new)+1;
    ptim=libGAP_ADDR_OBJ(im)+1;

    //copy the image set 
    for(i=0;i<len;i++)      *ptnew++=*ptim++;
    //add new points
    for(i=deg+1;i<=m;i++)   *ptnew++=libGAP_INTOBJ_INT(i);
  }
  return new;
} 

/* image list of transformation */

libGAP_Obj libGAP_FuncIMAGE_TRANS (libGAP_Obj self, libGAP_Obj f, libGAP_Obj n ){ 
  libGAP_UInt2*    ptf2;
  libGAP_UInt4*    ptf4;
  libGAP_UInt      i, deg, m;
  libGAP_Obj       out;
  
  m=libGAP_INT_INTOBJ(n);

  if(m==0){
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(out, 0);
    return out;
  }

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, m);
    ptf2=libGAP_ADDR_TRANS2(f);
    deg=libGAP_MIN(libGAP_DEG_TRANS2(f), m); 
    for(i=0;i<deg;i++){ 
      libGAP_SET_ELM_PLIST(out,i+1,libGAP_INTOBJ_INT(ptf2[i]+1));
    }
    for(;i<m;i++) libGAP_SET_ELM_PLIST(out,i+1,libGAP_INTOBJ_INT(i+1));
  }else{
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, m);
    ptf4=libGAP_ADDR_TRANS4(f);
    deg=libGAP_MIN(libGAP_DEG_TRANS4(f), m);
    for(i=0;i<deg;i++){ 
      libGAP_SET_ELM_PLIST(out,i+1,libGAP_INTOBJ_INT(ptf4[i]+1));
    }
    for(;i<m;i++) libGAP_SET_ELM_PLIST(out,i+1,libGAP_INTOBJ_INT(i+1));
  }

  libGAP_SET_LEN_PLIST(out,(libGAP_Int) m);
  return out;
} 

/* the kernel as a partition of [1..n] */

libGAP_Obj libGAP_FuncKERNEL_TRANS (libGAP_Obj self, libGAP_Obj f, libGAP_Obj n){
  libGAP_Obj     ker, flat;
  libGAP_UInt    i, j, deg, nr, m, rank, len, min;
  libGAP_UInt4*  pttmp;
   
  if(libGAP_INT_INTOBJ(n)==0){//special case for the identity
    ker=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY, 0);
    libGAP_SET_LEN_PLIST(ker, 0);
    return ker;
  }
  
  deg=libGAP_DEG_TRANS(f);
  rank=libGAP_RANK_TRANS(f);
  flat=libGAP_KER_TRANS(f);
  
  m=libGAP_INT_INTOBJ(n);
  nr=(m<=deg?rank:rank+m-deg);  // the number of classes
  len=(libGAP_UInt) deg/nr+1;          // average size of a class
  min=libGAP_MIN(m,deg);
  
  ker=libGAP_NEW_PLIST(libGAP_T_PLIST_HOM_SSORT, nr);
  pttmp=libGAP_ResizeInitTmpTrans(nr);

  nr=0;
  // read off flat kernel
  for(i=0;i<min;i++){
    /* renew the ptrs in case of garbage collection */
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(flat, i+1));
    if(pttmp[j-1]==0){
      nr++;
      libGAP_SET_ELM_PLIST(ker, j, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, len));
      libGAP_CHANGED_BAG(ker);
      pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
    }
    libGAP_AssPlist(libGAP_ELM_PLIST(ker, j), (libGAP_Int) ++pttmp[j-1], libGAP_INTOBJ_INT(i+1));
    pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  }
  
  for(i=0;i<nr;i++){
    libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(ker, i+1), (libGAP_Int) pttmp[i]);
    libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(ker, i+1), (libGAP_Int) pttmp[i]);
    /* beware maybe SHRINK_PLIST will trigger a garbage collection */
  }

  for(i=deg;i<m;i++){//add trailing singletons if there are any
    libGAP_SET_ELM_PLIST(ker, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, 1));
    libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(ker, nr), 1); 
    libGAP_SET_ELM_PLIST(libGAP_ELM_PLIST(ker, nr), 1, libGAP_INTOBJ_INT(i+1));
    libGAP_CHANGED_BAG(ker);
  }
  libGAP_SET_LEN_PLIST(ker, (libGAP_Int) nr);
  return ker;
}


libGAP_Obj libGAP_FuncPREIMAGES_TRANS_INT (libGAP_Obj self, libGAP_Obj f, libGAP_Obj pt){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4;
  libGAP_UInt    deg, nr, i, j;
  libGAP_Obj     out;

  deg=libGAP_DEG_TRANS(f);

  if(libGAP_INT_INTOBJ(pt)>deg){
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 1);
    libGAP_SET_LEN_PLIST(out, 1);
    libGAP_SET_ELM_PLIST(out, 1, pt);
    return out;
  }

  i=(libGAP_UInt) libGAP_INT_INTOBJ(pt)-1;
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg);

  /* renew the ptr in case of garbage collection */
  nr=0;
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    for(j=0;j<deg;j++) if(ptf2[j]==i) libGAP_SET_ELM_PLIST(out, ++nr, libGAP_INTOBJ_INT(j+1));
  }else{
    ptf4=libGAP_ADDR_TRANS4(f);
    for(j=0;j<deg;j++) if(ptf4[j]==i) libGAP_SET_ELM_PLIST(out, ++nr, libGAP_INTOBJ_INT(j+1));
  }

  libGAP_SET_LEN_PLIST(out, (libGAP_Int) nr);
  libGAP_SHRINK_PLIST(out, (libGAP_Int) nr);
  return out;
}

// AsTransformation for a permutation <p> and a pos int <n>. This might be
// quicker if we don't install the kernel etc, but then getting the kernel etc
// back is slower than it is from here. 
libGAP_Obj libGAP_FuncAS_TRANS_PERM_INT(libGAP_Obj self, libGAP_Obj p, libGAP_Obj deg){
  libGAP_UInt2   *ptp2, *ptf2;
  libGAP_UInt4   *ptp4, *ptf4;
  libGAP_Obj     f, img, *ptimg;
  libGAP_UInt    def, dep, i, min;
  libGAP_Int     n;
  
  n=libGAP_INT_INTOBJ(deg);
  if(n==0) return libGAP_IdentityTrans;

  //find the degree of f
  def=n;
  dep=(libGAP_TNUM_OBJ(p)==libGAP_T_PERM2?libGAP_DEG_PERM2(p):libGAP_DEG_PERM4(p));

  if(n<dep){
    min=def;
    if(libGAP_TNUM_OBJ(p)==libGAP_T_PERM2){
      ptp2=libGAP_ADDR_PERM2(p);
      for(i=0;i<n;i++){
        if(ptp2[i]+1>def) def=ptp2[i]+1;
      }
    } else {
      dep=libGAP_DEG_PERM4(p);
      ptp4=libGAP_ADDR_PERM4(p);
      for(i=0;i<n;i++){
        if(ptp4[i]+1>def) def=ptp4[i]+1;
      }
    }
  } else {
    min=dep;
  }

  img=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT+libGAP_IMMUTABLE, def);
  //create f 
  if(def<=65536){
    f=libGAP_NEW_TRANS2(def);
    ptimg=libGAP_ADDR_OBJ(img)+1;
    ptf2=libGAP_ADDR_TRANS2(f);
    
    if(libGAP_TNUM_OBJ(p)==libGAP_T_PERM2){
      ptp2=libGAP_ADDR_PERM2(p);
      for(i=0;i<min;i++){
        ptf2[i]=ptp2[i];
        ptimg[i]=libGAP_INTOBJ_INT(i+1);
      }
    } else { //TNUM_OBJ(p)==T_PERM4
      ptp4=libGAP_ADDR_PERM4(p);
      for(i=0;i<min;i++){
        ptf2[i]=ptp4[i];
        ptimg[i]=libGAP_INTOBJ_INT(i+1);
      }
    }
    for(;i<def;i++){
      ptf2[i]=i;
      ptimg[i]=libGAP_INTOBJ_INT(i+1);
    }
    libGAP_IMG_TRANS(f)=img;
    libGAP_KER_TRANS(f)=img;
    libGAP_CHANGED_BAG(f);
  } else { //def>65536
    f=libGAP_NEW_TRANS4(def);
    ptimg=libGAP_ADDR_OBJ(img)+1;
    ptf4=libGAP_ADDR_TRANS4(f);
    
    if(libGAP_TNUM_OBJ(p)==libGAP_T_PERM2){
      ptp2=libGAP_ADDR_PERM2(p);
      for(i=0;i<min;i++){
        ptf4[i]=ptp2[i];
        ptimg[i]=libGAP_INTOBJ_INT(i+1);
      }
    } else { //TNUM_OBJ(p)==T_PERM4
      ptp4=libGAP_ADDR_PERM4(p);
      for(i=0;i<min;i++){
        ptf4[i]=ptp4[i];
        ptimg[i]=libGAP_INTOBJ_INT(i+1);
      }
    }
    for(;i<def;i++){
      ptf4[i]=i;
      ptimg[i]=libGAP_INTOBJ_INT(i+1);
    }
    libGAP_IMG_TRANS(f)=img;
    libGAP_KER_TRANS(f)=img;
    libGAP_CHANGED_BAG(f);
  }
  
  libGAP_SET_LEN_PLIST(img, def);
  return f;
}

/* AsTransformation for a permutation */

libGAP_Obj libGAP_FuncAS_TRANS_PERM(libGAP_Obj self, libGAP_Obj p){
  libGAP_UInt2   *ptPerm2;
  libGAP_UInt4   *ptPerm4;
  libGAP_UInt    sup;

  //find largest moved point 
  if(libGAP_TNUM_OBJ(p)==libGAP_T_PERM2){
    ptPerm2=libGAP_ADDR_PERM2(p);
    for(sup=libGAP_DEG_PERM2(p);1<=sup;sup--) if(ptPerm2[sup-1]!=sup-1) break;
    return libGAP_FuncAS_TRANS_PERM_INT(self, p, libGAP_INTOBJ_INT(sup));
  } else { 
    ptPerm4 = libGAP_ADDR_PERM4(p);
    for ( sup = libGAP_DEG_PERM4(p); 1 <= sup; sup-- ) {
      if ( ptPerm4[sup-1] != sup-1 ) break;
    }
    return libGAP_FuncAS_TRANS_PERM_INT(self, p, libGAP_INTOBJ_INT(sup));
  }
}

/* converts transformation into permutation of its image if possible */

libGAP_Obj libGAP_FuncAS_PERM_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2, *ptp2;
  libGAP_UInt4   *ptf4, *ptp4;
  libGAP_UInt    deg, i;
  libGAP_Obj     p;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    if(libGAP_RANK_TRANS2(f)!=deg) return libGAP_Fail;
    
    p=libGAP_NEW_PERM2(deg);
    ptp2=libGAP_ADDR_PERM2(p);
    ptf2=libGAP_ADDR_TRANS2(f);
    
    for(i=0;i<deg;i++){
      ptp2[i]=ptf2[i];
    }
    return p;
  }else if (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    deg=libGAP_DEG_TRANS4(f);
    if(libGAP_RANK_TRANS4(f)!=deg) return libGAP_Fail;
    
    p=libGAP_NEW_PERM4(deg);
    ptp4=libGAP_ADDR_PERM4(p);
    ptf4=libGAP_ADDR_TRANS4(f);
    
    for(i=0;i<deg;i++){
      ptp4[i]=ptf4[i];
    }
    return p;
  }
  return libGAP_Fail;
}

libGAP_Obj libGAP_FuncPERM_IMG_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2, *ptp2;
  libGAP_UInt4   *ptf4, *ptp4, *pttmp;
  libGAP_UInt    deg, rank, i, j;
  libGAP_Obj     p, img;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    rank=libGAP_RANK_TRANS2(f);
    deg=libGAP_DEG_TRANS2(f);

    p=libGAP_NEW_PERM2(deg);
    libGAP_ResizeTmpTrans(deg); 
    
    pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
    ptp2=libGAP_ADDR_PERM2(p);
    for(i=0;i<deg;i++){ pttmp[i]=0; ptp2[i]=i; }
    
    ptf2=libGAP_ADDR_TRANS2(f);
    img=libGAP_IMG_TRANS(f);
   
    for(i=0;i<rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i+1))-1;    /* ranset(f)[i] */ 
      if(pttmp[ptf2[j]]!=0) return libGAP_Fail; 
      pttmp[ptf2[j]]=1;
      ptp2[j]=ptf2[j];
    }
    return p;
  }else if (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    rank=libGAP_RANK_TRANS4(f);
    deg=libGAP_DEG_TRANS4(f);

    p=libGAP_NEW_PERM4(deg);
    libGAP_ResizeTmpTrans(deg);

    pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
    ptp4=libGAP_ADDR_PERM4(p);
    for(i=0;i<deg;i++){ pttmp[i]=0; ptp4[i]=i; }
    
    ptf4=libGAP_ADDR_TRANS4(f);
    img=libGAP_IMG_TRANS(f);
   
    for(i=0;i<rank;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i+1))-1;    /* ranset(f)[i] */ 
      if(pttmp[ptf4[j]]!=0) return libGAP_Fail; 
      pttmp[ptf4[j]]=1;
      ptp4[j]=ptf4[j];
    }
    return p;
  }
  return libGAP_Fail;
}

/* if <g>=RESTRICTED_TRANS(f), then <g> acts like <f> on <list> and fixes every
 * other point */

libGAP_Obj libGAP_FuncRESTRICTED_TRANS(libGAP_Obj self, libGAP_Obj f, libGAP_Obj list){
  libGAP_UInt    deg, i, j, len;
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *ptf4, *ptg4;
  libGAP_Obj     libGAP_g;

  len=libGAP_LEN_LIST(list);

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    libGAP_g=libGAP_NEW_TRANS2(deg);
  
    ptf2=libGAP_ADDR_TRANS2(f);
    ptg2=libGAP_ADDR_TRANS2(libGAP_g);

    /* g fixes every point */
    for(i=0;i<deg;i++) ptg2[i]=i;

    /* g acts like f on list */
    for(i=0;i<len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i+1))-1;
      if(j<deg) ptg2[j]=ptf2[j];
    }
  }else{
    deg=libGAP_DEG_TRANS4(f);
    libGAP_g=libGAP_NEW_TRANS4(deg);
  
    ptf4=libGAP_ADDR_TRANS4(f);
    ptg4=libGAP_ADDR_TRANS4(libGAP_g);

    /* g fixes every point */
    for(i=0;i<deg;i++) ptg4[i]=i;

    /* g acts like f on list */
    for(i=0;i<len;i++){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i+1))-1;
      if(j<deg) ptg4[j]=ptf4[j];
    }
  }
  return libGAP_g;
}

// AsTransformation for a transformation <f> and a pos int <m> either restricts
// <f> to [1..m] or returns <f> depending on whether m is less than or equal
// Degree(f) or not.

// in the first form, this is similar to TRIM_TRANS except that a new
// transformation is returned. 


libGAP_Obj libGAP_FuncAS_TRANS_TRANS(libGAP_Obj self, libGAP_Obj f, libGAP_Obj m){
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *ptf4, *ptg4;
  libGAP_UInt    i, n, def;
  libGAP_Obj     libGAP_g;

  n=libGAP_INT_INTOBJ(m);
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){    // f and g are T_TRANS2
    def=libGAP_DEG_TRANS2(f);
    if(def<=n) return f;

    libGAP_g=libGAP_NEW_TRANS2(n);
    ptf2=libGAP_ADDR_TRANS2(f);
    ptg2=libGAP_ADDR_TRANS2(libGAP_g);
    for(i=0;i<n;i++){
      if(ptf2[i]>n-1) return libGAP_Fail;
      ptg2[i]=ptf2[i];
    }
    return libGAP_g;
  }else{                    // f is T_TRANS4
    def=libGAP_DEG_TRANS4(f);
    if(def<=n) return f;

    if(n>65536){            // g is T_TRANS4
      libGAP_g=libGAP_NEW_TRANS4(n);
      ptf4=libGAP_ADDR_TRANS4(f);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      for(i=0;i<n;i++){
        if(ptf4[i]>n-1) return libGAP_Fail;
        ptg4[i]=ptf4[i];
      }
    }else{  //  f is T_TRANS4 but n<=65536<def and so g will be T_TRANS2 */
      libGAP_g=libGAP_NEW_TRANS2(n);
      ptf4=libGAP_ADDR_TRANS4(f);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      for(i=0;i<n;i++){
        if(ptf4[i]>n-1) return libGAP_Fail;
        ptg2[i]=(libGAP_UInt2) ptf4[i];
      }
    }
  }
  return libGAP_g;
}

// it is assumed that f is actually a transformation of [1..m], i.e. that i^f<=m
// for all i in [1..m]
libGAP_Obj libGAP_FuncTRIM_TRANS (libGAP_Obj self, libGAP_Obj f, libGAP_Obj m){
  libGAP_UInt    deg, i;
  libGAP_UInt4   *ptf;

  if(!libGAP_IS_TRANS(f)){
    libGAP_ErrorQuit("the argument must be a transformation,", 0L, 0L);
  }

  deg=libGAP_INT_INTOBJ(m);

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){  // output is T_TRANS2
    if(deg>libGAP_DEG_TRANS2(f)) return (libGAP_Obj)0;
    libGAP_ResizeBag(f, deg*sizeof(libGAP_UInt2)+3*sizeof(libGAP_Obj));
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    if(deg>libGAP_DEG_TRANS4(f)) return (libGAP_Obj)0;
    if(deg>65536UL){          // output is T_TRANS4
      libGAP_ResizeBag(f, deg*sizeof(libGAP_UInt4)+3*sizeof(libGAP_Obj));
    } else {                  // output is T_TRANS2
      ptf=libGAP_ADDR_TRANS4(f);
      for(i=0;i<deg;i++) ((libGAP_UInt2*)ptf)[i]=(libGAP_UInt2)ptf[i];
      libGAP_RetypeBag(f, libGAP_T_TRANS2);
      libGAP_ResizeBag(f, deg*sizeof(libGAP_UInt2)+3*sizeof(libGAP_Obj));
    }
  }
  libGAP_IMG_TRANS(f)=NULL;
  libGAP_KER_TRANS(f)=NULL;
  libGAP_EXT_TRANS(f)=NULL;
  libGAP_CHANGED_BAG(f);
  return (libGAP_Obj)0;
}

libGAP_Obj libGAP_FuncHASH_FUNC_FOR_TRANS(libGAP_Obj self, libGAP_Obj f, libGAP_Obj data){
  libGAP_UInt deg;

  deg=libGAP_INT_INTOBJ(libGAP_FuncDegreeOfTransformation(self, f));
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    if(deg<=65536){
      libGAP_FuncTRIM_TRANS(self, f, libGAP_EXT_TRANS(f));
    } else {
      return libGAP_INTOBJ_INT((libGAP_HASHKEY_BAG_NC(f, (libGAP_UInt4) 255, 3*sizeof(libGAP_Obj), 
              (int) 4*deg) % (libGAP_INT_INTOBJ(data)))+1);
    }
  }

  return libGAP_INTOBJ_INT((libGAP_HASHKEY_BAG_NC(f, (libGAP_UInt4) 255, 3*sizeof(libGAP_Obj), 
          (int) 2*deg) % (libGAP_INT_INTOBJ(data)))+1);

}

/* check if the trans or list t is injective on the list l */
libGAP_Obj libGAP_FuncIS_INJECTIVE_LIST_TRANS( libGAP_Obj self, libGAP_Obj l, libGAP_Obj t){
  libGAP_UInt    n, i, j;
  libGAP_UInt2   *ptt2;
  libGAP_UInt4   *pttmp=0L;
  libGAP_UInt4   *ptt4;
  
  /* init buffer */
  n=(libGAP_IS_TRANS(t)?libGAP_DEG_TRANS(t):libGAP_LEN_LIST(t));
  pttmp=libGAP_ResizeInitTmpTrans(n);

  if(libGAP_TNUM_OBJ(t)==libGAP_T_TRANS2){/* and LEN_LIST(l), deg(f)<=65536 */
    ptt2=libGAP_ADDR_TRANS2(t);
    for(i=libGAP_LEN_LIST(l);i>=1;i--){
      j=(libGAP_UInt) libGAP_INT_INTOBJ(libGAP_ELM_LIST(l, i));
      if(j<=n){
        if(pttmp[ptt2[j-1]]!=0) return libGAP_False;
        pttmp[ptt2[j-1]]=1;
      }
    }
  } else if(libGAP_TNUM_OBJ(t)==libGAP_T_TRANS4){
    ptt4=libGAP_ADDR_TRANS4(t);
    for(i=libGAP_LEN_LIST(l);i>=1;i--){
      j=(libGAP_UInt) libGAP_INT_INTOBJ(libGAP_ELM_LIST(l, i));
      if(j<=n) {
        if(pttmp[ptt4[j-1]]!=0) return libGAP_False;
        pttmp[ptt4[j-1]]=1;
      }
    }
  }else if(n<=65536){/* t is a list */
    for(i=libGAP_LEN_LIST(l);i>=1;i--){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(l, i));    
      if(j<=n){
        if(pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(t, j))-1]!=0) return libGAP_False;
        pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(t, j))-1]=1;
      }
    }
  }else{ /* t is a list */
    for(i=libGAP_LEN_LIST(l);i>=1;i--){
      j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(l, i));    
      if(j<=n){
        if(pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(t, j))-1]!=0) return libGAP_False;
        pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(t, j))-1]=1;
      }
    }
  }
  return libGAP_True;
}

/* the perm2 of im(f) induced by f^-1*g, no checking*/
libGAP_Obj libGAP_FuncPERM_LEFT_QUO_TRANS_NC(libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g)
{ libGAP_UInt2   *ptf2, *ptg2, *ptp2;
  libGAP_UInt4   *ptf4, *ptg4, *ptp4;
  libGAP_UInt    def, deg, i;
  libGAP_Obj     perm;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS2){
    def=libGAP_DEG_TRANS2(f); 
    deg=libGAP_DEG_TRANS2(libGAP_g);
    
    if(def<=deg){
      perm=libGAP_NEW_PERM2(deg);
      ptp2=libGAP_ADDR_PERM2(perm);
      ptf2=libGAP_ADDR_TRANS2(f);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      for(i=0;i<deg;i++) ptp2[i]=i;
      for(i=0;i<def;i++) ptp2[ptf2[i]]=ptg2[i];
      for(;i<deg;i++)    ptp2[i]=ptg2[i];
    } else { //def>deg
      perm=libGAP_NEW_PERM2(def);
      ptp2=libGAP_ADDR_PERM2(perm);
      ptf2=libGAP_ADDR_TRANS2(f);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      for(i=0;i<def;i++) ptp2[i]=i;
      for(i=0;i<deg;i++) ptp2[ptf2[i]]=ptg2[i];
      for(;i<def;i++)    ptp2[ptf2[i]]=i; 
    }
    return perm;
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS4){ //def<deg
    def=libGAP_DEG_TRANS2(f);
    deg=libGAP_DEG_TRANS4(libGAP_g); 
    perm=libGAP_NEW_PERM4(deg);
    ptp4=libGAP_ADDR_PERM4(perm);
    ptf2=libGAP_ADDR_TRANS2(f);
    ptg4=libGAP_ADDR_TRANS4(libGAP_g);
    for(i=0;i<deg;i++) ptp4[i]=i;
    for(i=0;i<def;i++) ptp4[ptf2[i]]=ptg4[i];
    for(;i<deg;i++)    ptp4[i]=ptg4[i];
    return perm;
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS2){ //def>deg
    def=libGAP_DEG_TRANS4(f);
    deg=libGAP_DEG_TRANS2(libGAP_g); 
    perm=libGAP_NEW_PERM4(def);
    ptp4=libGAP_ADDR_PERM4(perm);
    ptf4=libGAP_ADDR_TRANS4(f);
    ptg2=libGAP_ADDR_TRANS2(libGAP_g);
    for(i=0;i<def;i++) ptp4[i]=i;
    for(i=0;i<deg;i++) ptp4[ptf4[i]]=ptg2[i];
    for(;i<def;i++)    ptp4[ptf4[i]]=i; 
    return perm;
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4&&libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS4){
    def=libGAP_DEG_TRANS4(f); 
    deg=libGAP_DEG_TRANS4(libGAP_g);
    if(def<=deg){
      perm=libGAP_NEW_PERM4(deg);
      ptp4=libGAP_ADDR_PERM4(perm);
      ptf4=libGAP_ADDR_TRANS4(f);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      for(i=0;i<deg;i++) ptp4[i]=i;
      for(i=0;i<def;i++) ptp4[ptf4[i]]=ptg4[i];
      for(;i<deg;i++)    ptp4[i]=ptg4[i];
    } else { //def>deg
      perm=libGAP_NEW_PERM4(def);
      ptp4=libGAP_ADDR_PERM4(perm);
      ptf4=libGAP_ADDR_TRANS4(f);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      for(i=0;i<def;i++) ptp4[i]=i;
      for(i=0;i<deg;i++) ptp4[ptf4[i]]=ptg4[i];
      for(;i<def;i++)    ptp4[ptf4[i]]=i; 
    }
    return perm;
  }
  return libGAP_Fail;
}

/* transformation from image set and flat kernel, no checking*/
libGAP_Obj libGAP_FuncTRANS_IMG_KER_NC(libGAP_Obj self, libGAP_Obj img, libGAP_Obj ker){
  libGAP_UInt    deg=libGAP_LEN_LIST(ker);
  libGAP_Obj     f;
  libGAP_UInt2*  ptf2;
  libGAP_UInt4*  ptf4;
  libGAP_UInt    i;
  
  if(deg<=65536){
    f=libGAP_NEW_TRANS2(deg);
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=0;i<deg;i++){
      ptf2[i]=libGAP_INT_INTOBJ(libGAP_ELM_LIST(img, libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))))-1;
    }
  }else{
    f=libGAP_NEW_TRANS4(deg);
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=0;i<deg;i++){
      ptf4[i]=libGAP_INT_INTOBJ(libGAP_ELM_LIST(img, libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))))-1;
    }   
  }
  libGAP_IMG_TRANS(f)=img;
  libGAP_KER_TRANS(f)=ker;
  libGAP_CHANGED_BAG(f);
  return f;
}

/* idempotent from image set and flat kernel, no checking.
*  Note that this is not the same as the previous function */

libGAP_Obj libGAP_FuncIDEM_IMG_KER_NC(libGAP_Obj self, libGAP_Obj img, libGAP_Obj ker){
  libGAP_UInt    deg=libGAP_LEN_LIST(ker);
  libGAP_UInt    rank=libGAP_LEN_LIST(img);
  libGAP_Obj     f;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4, *pttmp;
  libGAP_UInt    i, j;
  
  if(!libGAP_IS_PLIST(img)) libGAP_PLAIN_LIST(img);
  if(!libGAP_IS_PLIST(ker)) libGAP_PLAIN_LIST(ker);
    
  if(libGAP_IS_MUTABLE_OBJ(img)) libGAP_RetypeBag(img, libGAP_TNUM_OBJ(img)+libGAP_IMMUTABLE);
  if(libGAP_IS_MUTABLE_OBJ(ker)) libGAP_RetypeBag(ker, libGAP_TNUM_OBJ(ker)+libGAP_IMMUTABLE);

  libGAP_ResizeTmpTrans(deg);
  pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  
  // setup the lookup table
  for(i=0;i<rank;i++){
    j=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, i+1));
    pttmp[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ker, j))-1]=j-1;
  }
  if(deg<=65536){
    f=libGAP_NEW_TRANS2(deg);
    ptf2=libGAP_ADDR_TRANS2(f);
    pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans)); 

    for(i=0;i<deg;i++) ptf2[i]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ker, i+1))-1];
  }else{
    f=libGAP_NEW_TRANS4(deg);
    ptf4=libGAP_ADDR_TRANS4(f);
    pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans)); 
    
    for(i=0;i<deg;i++) ptf4[i]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ker, i+1))-1];
  }
  libGAP_IMG_TRANS(f)=img;
  libGAP_KER_TRANS(f)=ker;
  libGAP_CHANGED_BAG(f);
  return f;
}

/* an inverse of a transformation f*g*f=f and g*f*g=g */

libGAP_Obj libGAP_FuncINV_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *ptf4, *ptg4;
  libGAP_UInt    deg, i;
  libGAP_Obj     libGAP_g;

  if(libGAP_FuncIS_ID_TRANS(self, f)==libGAP_True) return f;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    libGAP_g=libGAP_NEW_TRANS2(deg);
    ptf2=libGAP_ADDR_TRANS2(f);
    ptg2=libGAP_ADDR_TRANS2(libGAP_g);
    for(i=0;i<deg;i++) ptg2[i]=0;
    for(i=deg-1;i>0;i--) ptg2[ptf2[i]]=i;
    /* to ensure that 1 is in the image and so rank of g equals that of f*/
    ptg2[ptf2[0]]=0;
  }else{
    deg=libGAP_DEG_TRANS4(f);
    libGAP_g=libGAP_NEW_TRANS4(deg);
    ptf4=libGAP_ADDR_TRANS4(f);
    ptg4=libGAP_ADDR_TRANS4(libGAP_g);
    for(i=0;i<deg;i++) ptg4[i]=0;
    for(i=deg-1;i>0;i--) ptg4[ptf4[i]]=i;
    /* to ensure that 1 is in the image and so rank of g equals that of f*/
    ptg4[ptf4[0]]=0;
  }
  return libGAP_g;
}

/* a transformation g such that g: i^f -> i for all i in list 
 * where it is supposed that f is injective on list */
// JDM double-check
libGAP_Obj libGAP_FuncINV_LIST_TRANS(libGAP_Obj self, libGAP_Obj list, libGAP_Obj f){
  libGAP_UInt2   *ptf2, *ptg2; 
  libGAP_UInt4   *ptf4, *ptg4; 
  libGAP_UInt    deg, i, j;
  libGAP_Obj     libGAP_g;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);      
    libGAP_g=libGAP_NEW_TRANS2(deg);
    ptf2=libGAP_ADDR_TRANS2(f);
    ptg2=libGAP_ADDR_TRANS2(libGAP_g);
    
    for(j=0;j<deg;j++) ptg2[j]=j;
    for(j=1;j<=libGAP_LEN_LIST(list);j++){
      i=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, j))-1;
      if(i<deg) ptg2[ptf2[i]]=i;
    }
    return libGAP_g;
  }else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    deg=libGAP_DEG_TRANS4(f);      
    libGAP_g=libGAP_NEW_TRANS4(deg);
    ptf4=libGAP_ADDR_TRANS4(f);
    ptg4=libGAP_ADDR_TRANS4(libGAP_g);
    
    i=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, 1))-1;
    for(j=0;j<deg;j++) ptg4[j]=j;
    for(j=1;j<=libGAP_LEN_LIST(list);j++){
      i=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, j))-1;
      if(i<deg) ptg4[ptf4[i]]=i;
    }
    return libGAP_g;
  }
  return libGAP_Fail;
}

/* returns the permutation p conjugating image set f to image set g 
 * when ker(f)=ker(g) so that gf^-1(i)=p(i). 
 * This is the same as MappingPermListList(IMAGE_TRANS(f), IMAGE_TRANS(g)); */
libGAP_Obj libGAP_FuncTRANS_IMG_CONJ(libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_Obj     perm;
  libGAP_UInt2   *ptp2, *ptf2, *ptg2;
  libGAP_UInt4   *ptsrc, *ptdst, *ptp4, *ptf4, *ptg4;
  libGAP_UInt    def, deg, i, j;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    def=libGAP_DEG_TRANS2(f);
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS2){
      deg=libGAP_DEG_TRANS2(libGAP_g);
      if(def<=deg){
        perm=libGAP_NEW_PERM2(deg);
        ptsrc=libGAP_ResizeInitTmpTrans(2*deg);
        ptdst=ptsrc+deg;

        ptp2=libGAP_ADDR_PERM2(perm);
        ptf2=libGAP_ADDR_TRANS2(f);
        ptg2=libGAP_ADDR_TRANS2(libGAP_g);
        
        for(i=0;i<def;i++){
          ptsrc[ptf2[i]]=1;
          ptdst[ptg2[i]]=1;
          ptp2[ptf2[i]]=ptg2[i];
        }
        for(;i<deg;i++){
          //ptsrc[i]=1;
          ptdst[ptg2[i]]=1;
          ptp2[i]=ptg2[i];
        }
        j=0;
        for(i=0;i<def;i++){
          if(ptsrc[i]==0){
            while(ptdst[j]!=0){ j++; } 
            ptp2[i]=j;
            j++;
          }
        }
        return perm;
      } else {// def>deg
        perm=libGAP_NEW_PERM2(def);
        ptsrc=libGAP_ResizeInitTmpTrans(2*def);
        ptdst=ptsrc+def;
        
        ptp2=libGAP_ADDR_PERM2(perm);
        ptf2=libGAP_ADDR_TRANS2(f);
        ptg2=libGAP_ADDR_TRANS2(libGAP_g);
        
        for(i=0;i<deg;i++){
          ptsrc[ptf2[i]]=1;
          ptdst[ptg2[i]]=1;
          ptp2[ptf2[i]]=ptg2[i];
        }
        for(;i<def;i++){
          ptsrc[ptf2[i]]=1;
          ptdst[i]=1;
          ptp2[ptf2[i]]=i;
        }
        j=0;
        for(i=0;i<def;i++){
          if(ptsrc[i]==0){
            while(ptdst[j]!=0){ j++; } 
            ptp2[i]=j;
            j++;
          }
        }
        return perm;
      }      
    } else if (libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS4){ //deg>def
      deg=libGAP_DEG_TRANS4(libGAP_g);
      perm=libGAP_NEW_PERM4(deg);
      ptsrc=libGAP_ResizeInitTmpTrans(2*deg);
      ptdst=ptsrc+deg;

      ptp4=libGAP_ADDR_PERM4(perm);
      ptf2=libGAP_ADDR_TRANS2(f);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      
      for(i=0;i<def;i++){
        ptsrc[ptf2[i]]=1;
        ptdst[ptg4[i]]=1;
        ptp4[ptf2[i]]=ptg4[i];
      }
      for(;i<deg;i++){
        //ptsrc[i]=1;
        ptdst[ptg4[i]]=1;
        ptp4[i]=ptg4[i];
      }
      j=0;
      for(i=0;i<def;i++){
        if(ptsrc[i]==0){
          while(ptdst[j]!=0){ j++; } 
          ptp4[i]=j;
          j++;
        }
      }
      return perm;
    }
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4) { 
    def=libGAP_DEG_TRANS4(f);

    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS2){ //def>deg
      deg=libGAP_DEG_TRANS2(libGAP_g);
      perm=libGAP_NEW_PERM4(def);
      
      ptsrc=libGAP_ResizeInitTmpTrans(2*def);
      ptdst=ptsrc+def;
      ptp4=libGAP_ADDR_PERM4(perm);
      ptf4=libGAP_ADDR_TRANS4(f);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      
      for(i=0;i<deg;i++){
        ptsrc[ptf4[i]]=1;
        ptdst[ptg2[i]]=1;
        ptp4[ptf4[i]]=ptg2[i];
      }
      for(;i<def;i++){
        ptsrc[ptf4[i]]=1;
        ptdst[i]=1;
        ptp4[ptf4[i]]=i;
      }
      j=0;
      for(i=0;i<def;i++){
        if(ptsrc[i]==0){
          while(ptdst[j]!=0){ j++; } 
          ptp4[i]=j;
          j++;
        }
      }
      return perm;
    } else if (libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS4){
      deg=libGAP_DEG_TRANS4(libGAP_g);
      if(def<=deg){
        perm=libGAP_NEW_PERM4(deg);
        ptsrc=libGAP_ResizeInitTmpTrans(2*deg);
        ptdst=ptsrc+deg;

        ptp4=libGAP_ADDR_PERM4(perm);
        ptf4=libGAP_ADDR_TRANS4(f);
        ptg4=libGAP_ADDR_TRANS4(libGAP_g);
        
        for(i=0;i<def;i++){
          ptsrc[ptf4[i]]=1;
          ptdst[ptg4[i]]=1;
          ptp4[ptf4[i]]=ptg4[i];
        }
        for(;i<deg;i++){
          //ptsrc[i]=1;
          ptdst[ptg4[i]]=1;
          ptp4[i]=ptg4[i];
        }
        j=0;
        for(i=0;i<def;i++){
          if(ptsrc[i]==0){
            while(ptdst[j]!=0){ j++; } 
            ptp4[i]=j;
            j++;
          }
        }
        return perm;
      } else {// def>deg
        perm=libGAP_NEW_PERM4(def);
        ptsrc=libGAP_ResizeInitTmpTrans(2*def);
        ptdst=ptsrc+def;
        
        ptp4=libGAP_ADDR_PERM4(perm);
        ptf4=libGAP_ADDR_TRANS4(f);
        ptg4=libGAP_ADDR_TRANS4(libGAP_g);
        
        for(i=0;i<deg;i++){
          ptsrc[ptf4[i]]=1;
          ptdst[ptg4[i]]=1;
          ptp4[ptf4[i]]=ptg4[i];
        }
        for(;i<def;i++){
          ptsrc[ptf4[i]]=1;
          ptdst[i]=1;
          ptp4[ptf4[i]]=i;
        }
        j=0;
        for(i=0;i<def;i++){
          if(ptsrc[i]==0){
            while(ptdst[j]!=0){ j++; } 
            ptp4[i]=j;
            j++;
          }
        }
        return perm;
      }      
    }
  }
  return libGAP_Fail;
}

/* the least m, r such that f^m+r=f^m */

libGAP_Obj libGAP_FuncINDEX_PERIOD_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4, *ptseen, *ptlast, *ptcurrent, *tmp; 
  libGAP_UInt    deg, i, current, last, pow, len, j;
  libGAP_Obj     ord, out;
  libGAP_Int     s, t, gcd;
 
  deg=libGAP_DEG_TRANS(f);
  
  libGAP_ResizeTmpTrans(3*deg);
  
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptlast=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
  ptcurrent=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    last=deg; current=0; 
    for(i=0;i<deg;i++){ptseen[i]=0; ptcurrent[i]=0; ptlast[i]=i; }
    
    for(i=0;i<last;i++){ /* loop over the last image */
      if(ptseen[ptf2[ptlast[i]]]==0){
        ptseen[ptf2[ptlast[i]]]=1;
        ptcurrent[current++]=ptf2[ptlast[i]];
      }
      /* ptcurrent holds the image set of f^pow (unsorted) */
      /* ptseen is a lookup for membership in ptcurrent */
    }

    /* find least power of f which is a permutation of its image */
    for(pow=1;pow<=deg;){ 
      last=current; current=0; tmp=ptlast;
      ptlast=ptcurrent; ptcurrent=tmp;

      for(i=0;i<deg;i++){ ptseen[i]=0; ptcurrent[i]=0;}
      
      for(i=0;i<last;i++){ /* loop over the last image */
        if(ptseen[ptf2[ptlast[i]]]==0){
          ptseen[ptf2[ptlast[i]]]=1;
          ptcurrent[current++]=ptf2[ptlast[i]];
        }
        /* ptcurrent holds the image set of f^pow (unsorted) */
        /* ptseen is a lookup for membership in ptcurrent */
      }
      if(last==current) break;
      pow++;
    }

    /* find the order of the perm induced by f on im_set(f^pow) */
    /* clear the buffer bag (ptlast) */
    for(i=0;i<deg;i++) ptlast[i]=0;
    ord=libGAP_INTOBJ_INT(1);
    /* loop over all cycles */
    for(i=0;i<deg;i++){
      /* if we haven't looked at this cycle so far */
      if(ptlast[i]==0&&ptseen[i]!=0&&ptf2[i]!=i){

        /* find the length of this cycle                           */
        len=1;
        for(j=ptf2[i];j!=i;j=ptf2[j]){ len++; ptlast[j]=1; }

        /* compute the gcd with the previously order ord           */
        gcd=len;  s=libGAP_INT_INTOBJ(libGAP_ModInt(ord,libGAP_INTOBJ_INT(len)));
        while (s!= 0){ t=s;  s=gcd%s;  gcd=t; }
        ord=libGAP_ProdInt(ord,libGAP_INTOBJ_INT(len/gcd));
      }
    }
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 2);
    libGAP_SET_LEN_PLIST(out, 2);
    libGAP_SET_ELM_PLIST(out, 1, libGAP_INTOBJ_INT(pow));
    libGAP_SET_ELM_PLIST(out, 2, ord);
    return out;
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    ptf4=libGAP_ADDR_TRANS4(f);
    last=deg; current=0; 
    for(i=0;i<deg;i++){ ptseen[i]=0; ptcurrent[i]=0; ptlast[i]=i; }
    
    for(i=0;i<last;i++){ /* loop over the last image */
      if(ptseen[ptf4[ptlast[i]]]==0){
        ptseen[ptf4[ptlast[i]]]=1;
        ptcurrent[current++]=ptf4[ptlast[i]];
      }
      /* ptcurrent holds the image set of f^pow (unsorted) */
      /* ptseen is a lookup for membership in ptcurrent */
    }

    /* find least power of f which is a permutation of its image */
    for(pow=1;pow<=deg;){ 
      last=current; current=0; tmp=ptlast;
      ptlast=ptcurrent; ptcurrent=tmp;

      for(i=0;i<deg;i++){ptseen[i]=0; ptcurrent[i]=0;}
      
      for(i=0;i<last;i++){ /* loop over the last image */
        if(ptseen[ptf4[ptlast[i]]]==0){
          ptseen[ptf4[ptlast[i]]]=1;
          ptcurrent[current++]=ptf4[ptlast[i]];
        }
        /* ptcurrent holds the image set of f^pow (unsorted) */
        /* ptseen is a lookup for membership in ptcurrent */
      }
      if(last==current) break;
      pow++;
    }
    
    /* find the order of the perm induced by f on im_set(f^pow) */

    /* clear the buffer bag (ptlast) */
    for(i=0;i<deg;i++) ptlast[i]=0;
    ord=libGAP_INTOBJ_INT(1);
    
    /* loop over all cycles */
    for(i=0;i<deg;i++){
      /* if we haven't looked at this cycle so far */
      if(ptlast[i]==0&&ptseen[i]!=0&&ptf4[i]!=i){
        /* find the length of this cycle                           */
        len=1;
        for(j=ptf4[i];j!=i;j=ptf4[j]){ len++; ptlast[j]=1; }

        /* compute the gcd with the previously order ord           */
        /* Note that since len is single precision, ord % len is to*/
        gcd=len;  s=libGAP_INT_INTOBJ(libGAP_ModInt(ord,libGAP_INTOBJ_INT(len)));
        while (s!= 0){ t=s;  s=gcd%s;  gcd=t; }
        ord=libGAP_ProdInt(ord,libGAP_INTOBJ_INT(len/gcd));
      }
    }
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 2);
    libGAP_SET_LEN_PLIST(out, 2);
    libGAP_SET_ELM_PLIST(out, 1, libGAP_INTOBJ_INT(pow));
    libGAP_SET_ELM_PLIST(out, 2, ord);
    return out;
  }
  return libGAP_Fail;
}

/* the least power of <f> which is an idempotent */

libGAP_Obj libGAP_FuncSMALLEST_IDEM_POW_TRANS( libGAP_Obj self, libGAP_Obj f ){
  libGAP_Obj x, ind, per, pow;

  x=libGAP_FuncINDEX_PERIOD_TRANS(self, f);
  ind=libGAP_ELM_PLIST(x, 1);
  per=libGAP_ELM_PLIST(x, 2);
  pow=per;
  while(libGAP_LtInt(pow, ind)) pow=libGAP_SumInt(pow, per);
  return pow;
}

// the kernel of <f^p> where ker(f)=<ker> (where the length of the output equals
// the length of <ker>), assumes that <p> is a permutation of <[1..Length(ker)]>
// regardless of its degree
libGAP_Obj libGAP_FuncPOW_KER_PERM(libGAP_Obj self, libGAP_Obj ker, libGAP_Obj p){
  libGAP_UInt    len, rank, i, dep;
  libGAP_Obj     out;
  libGAP_UInt4   *ptcnj, *ptlkp, *ptp4;
  libGAP_UInt2   *ptp2;

  len=libGAP_LEN_LIST(ker);
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, len);
  libGAP_SET_LEN_PLIST(out, len);
  
  libGAP_ResizeTmpTrans(2*len);
  ptcnj = (libGAP_UInt4*) libGAP_ADDR_OBJ(libGAP_TmpTrans);
  
  rank  = 1;
  ptlkp = ptcnj+len;
  
  if(libGAP_TNUM_OBJ(p)==libGAP_T_PERM2){
    dep  = libGAP_DEG_PERM2(p);
    ptp2 = libGAP_ADDR_PERM2(p);
    
    if(dep<=len){
      // form the conjugate in ptcnj and init the lookup
      for(i=0;i<dep;i++){ //<p^-1*g*p> then <g> with ker(<g>)=<ker>
        ptcnj[ptp2[i]]=ptp2[libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))-1]; 
        ptlkp[i]=0;
      }
      for(;i<len;i++){
        ptcnj[i]=libGAP_IMAGE(libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))-1, ptp2, dep);
        ptlkp[i]=0;
      }

    }else{ //dep>len but p fixes [1..len] setwise
     
      // form the conjugate in ptcnj and init the lookup
      for(i=0;i<len;i++){ //<p^-1*g*p> then <g> with ker(<g>)=<ker>
        ptcnj[ptp2[i]]=ptp2[libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))-1]; 
        ptlkp[i]=0;
      }
    }
    
    // form the flat kernel
    for(i=0;i<len;i++){
      if(ptlkp[ptcnj[i]]==0) ptlkp[ptcnj[i]]=rank++;
      libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(ptlkp[ptcnj[i]]));
    }
    return out;
  } else if(libGAP_TNUM_OBJ(p)==libGAP_T_PERM4){
    dep  = libGAP_DEG_PERM4(p);
    ptp4 = libGAP_ADDR_PERM4(p);
    
    if(dep<=len){
      // form the conjugate in ptcnj and init the lookup
      for(i=0;i<dep;i++){ //<p^-1*g*p> then <g> with ker(<g>)=<ker>
        ptcnj[ptp4[i]]=ptp4[libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))-1]; 
        ptlkp[i]=0;
      }
      for(;i<len;i++){
        ptcnj[i]=libGAP_IMAGE(libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))-1, ptp4, dep);
        ptlkp[i]=0;
      }

    }else{ //dep>len but p fixes [1..len] setwise
     
      // form the conjugate in ptcnj and init the lookup
      for(i=0;i<len;i++){ //<p^-1*g*p> then <g> with ker(<g>)=<ker>
        ptcnj[ptp4[i]]=ptp4[libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, i+1))-1]; 
        ptlkp[i]=0;
      }
    }
    
    // form the flat kernel
    for(i=0;i<len;i++){
      if(ptlkp[ptcnj[i]]==0) ptlkp[ptcnj[i]]=rank++;
      libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(ptlkp[ptcnj[i]]));
    }
    return out;
  }
  libGAP_ErrorQuit("usage: the second argument must be a transformation,", 0L, 0L);
  return libGAP_Fail;
}

// the kernel obtained by multiplying f by any g with ker(g)=ker
libGAP_Obj libGAP_FuncON_KERNEL_ANTI_ACTION(libGAP_Obj self, libGAP_Obj ker, libGAP_Obj f, libGAP_Obj n){
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptf4, *pttmp;
  libGAP_UInt    deg, i, j, rank, len;
  libGAP_Obj     out;

  if(libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, libGAP_LEN_LIST(ker)))==0){ 
    return libGAP_FuncFLAT_KERNEL_TRANS_INT(self, f, n);
  }

  len=libGAP_LEN_LIST(ker);
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC+libGAP_IMMUTABLE, len);
  libGAP_SET_LEN_PLIST(out, len);
  
  rank=1;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    if(len>=deg){
      pttmp=libGAP_ResizeInitTmpTrans(len);
      ptf2=libGAP_ADDR_TRANS2(f);
    
      for(i=0;i<deg;i++){ //<f> then <g> with ker(<g>)=<ker>
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, ptf2[i]+1))-1; // f first!
        if(pttmp[j]==0) pttmp[j]=rank++;
        libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(pttmp[j]));
      }
      i++;
      for(;i<=len;i++){   //just <ker>
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker,i))-1;
        if(pttmp[j]==0) pttmp[j]=rank++;
        libGAP_SET_ELM_PLIST(out, i, libGAP_INTOBJ_INT(pttmp[j]));
      }
    } else {//len<deg
      pttmp=libGAP_ResizeInitTmpTrans(deg);
      ptf2=libGAP_ADDR_TRANS2(f);
      for(i=0;i<len;i++){  //<f> then <g> with ker(<g>)=<ker>
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, ptf2[i]+1))-1; // f first!
        if(pttmp[j]==0) pttmp[j]=rank++;
        libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(pttmp[j]));
      }
      for(;i<deg;i++){     //just <f>
        if(pttmp[ptf2[i]]==0) pttmp[ptf2[i]]=rank++;
        libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(pttmp[ptf2[i]]));
      }
    }
  } else { 
    deg=libGAP_DEG_TRANS4(f);
    if(len>=deg){
      pttmp=libGAP_ResizeInitTmpTrans(len);
      ptf4=libGAP_ADDR_TRANS4(f);
    
      for(i=0;i<deg;i++){ //<f> then <g> with ker(<g>)=<ker>
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, ptf4[i]+1))-1; // f first!
        if(pttmp[j]==0) pttmp[j]=rank++;
        libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(pttmp[j]));
      }
      i++;
      for(;i<=len;i++){   //just <ker>
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker,i))-1;
        if(pttmp[j]==0) pttmp[j]=rank++;
        libGAP_SET_ELM_PLIST(out, i, libGAP_INTOBJ_INT(pttmp[j]));
      }
    } else {//len<deg
      pttmp=libGAP_ResizeInitTmpTrans(deg);
      ptf4=libGAP_ADDR_TRANS4(f);
      for(i=0;i<len;i++){  //<f> then <g> with ker(<g>)=<ker>
        j=libGAP_INT_INTOBJ(libGAP_ELM_LIST(ker, ptf4[i]+1))-1; // f first!
        if(pttmp[j]==0) pttmp[j]=rank++;
        libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(pttmp[j]));
      }
      for(;i<deg;i++){     //just <f>
        if(pttmp[ptf4[i]]==0) pttmp[ptf4[i]]=rank++;
        libGAP_SET_ELM_PLIST(out, i+1, libGAP_INTOBJ_INT(pttmp[ptf4[i]]));
      }
    }
  }
  return out; 
}  

/* Let <x> be a transformation with <ker(x)=X> and <ker(fx)=f^ker(x)> has the
 * same number of classes as <ker(x)>. Then INV_KER_TRANS(X, f) returns a
 * transformation <g> such that <gf^ker(x)=ker(x)=ker(gfx)> and the action of
 * <gf> on <ker(x)> is the identity. 
 */

libGAP_Obj libGAP_FuncINV_KER_TRANS(libGAP_Obj self, libGAP_Obj X, libGAP_Obj f){
  libGAP_Obj     libGAP_g;
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *pttmp, *ptf4, *ptg4;
  libGAP_UInt    deg, i, len;
  
  len=libGAP_LEN_LIST(X);
  libGAP_ResizeTmpTrans(len);
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    if(len<=65536){   // deg(g)<=65536 and g is T_TRANS2
      libGAP_g=libGAP_NEW_TRANS2(len);
      pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
      ptf2=libGAP_ADDR_TRANS2(f);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      if(deg>=len){
        // calculate a transversal of f^ker(x)=ker(fx)
        for(i=0;i<len;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf2[i]+1))-1]=i;
        // install values in g
        for(i=len;i>=1;i--) ptg2[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }else{
        for(i=0;i<deg;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf2[i]+1))-1]=i;
        for(;i<len;i++)     pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i+1))-1]=i;
        for(i=len;i>=1;i--) ptg2[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }
      return libGAP_g;
    } else {        // deg(g)>65536 and g is T_TRANS4
      libGAP_g=libGAP_NEW_TRANS4(len);
      pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
      ptf2=libGAP_ADDR_TRANS2(f);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      if(deg>=len){
        // calculate a transversal of f^ker(x)=ker(fx)
        for(i=0;i<len;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf2[i]+1))-1]=i;
        // install values in g
        for(i=len;i>=1;i--) ptg4[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }else{
        for(i=0;i<deg;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf2[i]+1))-1]=i;
        for(;i<len;i++)     pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i+1))-1]=i;
        for(i=len;i>=1;i--) ptg4[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }
      return libGAP_g;
    }
  } else if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    deg=libGAP_DEG_TRANS4(f);
    if(len<=65536){   // deg(g)<=65536 and g is T_TRANS2
      libGAP_g=libGAP_NEW_TRANS2(len);
      pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
      ptf4=libGAP_ADDR_TRANS4(f);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      if(deg>=len){
        // calculate a transversal of f^ker(x)=ker(fx)
        for(i=0;i<len;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf4[i]+1))-1]=i;
        // install values in g
        for(i=len;i>=1;i--) ptg2[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }else{
        for(i=0;i<deg;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf4[i]+1))-1]=i;
        for(;i<len;i++)     pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i+1))-1]=i;
        for(i=len;i>=1;i--) ptg2[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }
      return libGAP_g;
    } else {        // deg(g)>65536 and g is T_TRANS4
      libGAP_g=libGAP_NEW_TRANS4(len);
      pttmp=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
      ptf4=libGAP_ADDR_TRANS4(f);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      if(deg>=len){
        // calculate a transversal of f^ker(x)=ker(fx)
        for(i=0;i<len;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf4[i]+1))-1]=i;
        // install values in g
        for(i=len;i>=1;i--) ptg4[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }else{
        for(i=0;i<deg;i++)  pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, ptf4[i]+1))-1]=i;
        for(;i<len;i++)     pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i+1))-1]=i;
        for(i=len;i>=1;i--) ptg4[i-1]=pttmp[libGAP_INT_INTOBJ(libGAP_ELM_LIST(X, i))-1];
      }
      return libGAP_g;
    }
  } else {
    libGAP_ErrorQuit("usage: the second argument must be a transformation,", 0L, 0L);
  }
  return libGAP_Fail;
}

/* test if a transformation is an idempotent. */

libGAP_Obj libGAP_FuncIS_IDEM_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt2*  ptf2;
  libGAP_UInt4*  ptf4;
  libGAP_UInt    deg, i;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=0;i<deg;i++){
      if(ptf2[ptf2[i]]!=ptf2[i]){
        return libGAP_False;
      }
    }
  } else {
    deg=libGAP_DEG_TRANS4(f);
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=0;i<deg;i++){
      if(ptf4[ptf4[i]]!=ptf4[i]){
        return libGAP_False;
      }
    }
  }
  return libGAP_True;
}

/* returns the least list <out> such that for all <i> in [1..degree(f)]
 * there exists <j> in <out> and a pos int <k> such that <j^(f^k)=i>. */

libGAP_Obj libGAP_FuncCOMPONENT_REPS_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_Obj     out;
  libGAP_UInt2   *ptf2; 
  libGAP_UInt4   *ptf4, *ptseen, *ptlookup, *ptlens, *ptimg;
  libGAP_UInt    deg, i, nr, count, m, j, k;

  deg=libGAP_INT_INTOBJ(libGAP_FuncDegreeOfTransformation(self, f));
  
  libGAP_ResizeTmpTrans(4*deg);
  out=libGAP_NEW_PLIST(libGAP_T_PLIST, deg);
  
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptlookup=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
  ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
  ptimg=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+3*deg;
    
  for(i=0;i<deg;i++){ ptseen[i]=0; ptlookup[i]=0; ptlens[i]=0; ptimg[i]=0; }

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    
    /* points in the image of f */
    for(i=0;i<deg;i++) ptimg[ptf2[i]]=1; 

    nr=0; m=0; count=0;

    /* components corresponding to points not in image */
    for(i=0;i<deg;i++){
      if(ptimg[i]==0&&ptseen[i]==0){
        m++;
        for(j=i;ptseen[j]==0;j=ptf2[j]){ ptseen[j]=m; count++;} 
        if(ptseen[j]==m){/* new component */
          k=nr;
          ptlookup[m-1]=nr;
          libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg-count));
          libGAP_CHANGED_BAG(out);
          ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
        }else{ /* old component */
          k=ptlookup[ptseen[j]-1];
          ptlookup[m-1]=k;
        }
        libGAP_AssPlist(libGAP_ELM_PLIST(out, k+1), ++ptlens[k], libGAP_INTOBJ_INT(i+1));
      }
      ptf2=libGAP_ADDR_TRANS2(f);
      ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
      ptlookup=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
      ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
      ptimg=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+3*deg;
    }

    for(i=0;i<nr;i++){
      libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(out, i+1), (libGAP_Int) ptlens[i]);
      libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, i+1), (libGAP_Int) ptlens[i]);
    }

    ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
    ptf2=libGAP_ADDR_TRANS2(f);

    /* components corresponding to cycles */
    for(i=0;i<deg;i++){
      if(ptseen[i]==0){
        for(j=ptf2[i];j!=i;j=ptf2[j]) ptseen[j]=1;
        
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, 1));
        libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, nr), 1);
        libGAP_SET_ELM_PLIST(libGAP_ELM_PLIST(out, nr), 1, libGAP_INTOBJ_INT(i+1));
        libGAP_CHANGED_BAG(out);
        
        ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
        ptf2=libGAP_ADDR_TRANS2(f);
      }
    }
  } else {

    ptf4=libGAP_ADDR_TRANS4(f);
    /* points in the image of f */
    for(i=0;i<deg;i++){ ptimg[ptf4[i]]=1; }

    nr=0; m=0; count=0;

    /* components corresponding to points not in image */
    for(i=0;i<deg;i++){
      if(ptimg[i]==0&&ptseen[i]==0){
        m++;
        for(j=i;ptseen[j]==0;j=ptf4[j]){ ptseen[j]=m; count++;} 
        if(ptseen[j]==m){/* new component */
          k=nr;
          ptlookup[m-1]=nr;
          libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg-count));
          libGAP_CHANGED_BAG(out);
          ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
        }else{ /* old component */
          k=ptlookup[ptseen[j]-1];
          ptlookup[m-1]=k;
        }
        libGAP_AssPlist(libGAP_ELM_PLIST(out, k+1), ++ptlens[k], libGAP_INTOBJ_INT(i+1));
      }
      ptf4=libGAP_ADDR_TRANS4(f); 
      ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
      ptlookup=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
      ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
      ptimg=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+3*deg;
    }

    for(i=0;i<nr-1;i++){
      libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(out, i+1), (libGAP_Int) ptlens[i]);
      libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, i+1), (libGAP_Int) ptlens[i]);
    }
    
    ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
    ptf4=libGAP_ADDR_TRANS4(f);

    /* components corresponding to cycles */
    for(i=0;i<deg;i++){
      if(ptseen[i]==0){
        for(j=ptf4[i];j!=i;j=ptf4[j]) ptseen[j]=1;
        
        libGAP_SET_ELM_PLIST(out, ++nr, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, 1));
        libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, nr), 1);
        libGAP_SET_ELM_PLIST(libGAP_ELM_PLIST(out, nr), 1, libGAP_INTOBJ_INT(i+1));
        libGAP_CHANGED_BAG(out);

        ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
        ptf4=libGAP_ADDR_TRANS4(f);
      }
    }
  }
 
  libGAP_SHRINK_PLIST(out, (libGAP_Int) nr);
  libGAP_SET_LEN_PLIST(out,  (libGAP_Int) nr);
  return out;
}

/* the number of components of a transformation (as a functional digraph) */

libGAP_Obj libGAP_FuncNR_COMPONENTS_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    nr, m, i, j, deg;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptf4;
  
  deg=libGAP_INT_INTOBJ(libGAP_FuncDegreeOfTransformation(self, f));
  ptseen=libGAP_ResizeInitTmpTrans(deg);
  nr=0; m=0;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=0;i<deg;i++){
      if(ptseen[i]==0){
        m++;
        for(j=i;ptseen[j]==0;j=ptf2[j]) ptseen[j]=m; 
        if(ptseen[j]==m) nr++;
      }
    }
  }else{
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=0;i<deg;i++){
      if(ptseen[i]==0){
        m++;
        for(j=i;ptseen[j]==0;j=ptf4[j]) ptseen[j]=m; 
        if(ptseen[j]==m) nr++;
      }
    }
  }
  return libGAP_INTOBJ_INT(nr);
}

/* the components of a transformation (as a functional digraph) */

libGAP_Obj libGAP_FuncCOMPONENTS_TRANS(libGAP_Obj self, libGAP_Obj f){
  libGAP_UInt    deg, i, nr, m, j;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptlookup, *ptlens, *ptf4;
  libGAP_Obj     out;
  
  deg=libGAP_INT_INTOBJ(libGAP_FuncDegreeOfTransformation(self, f));
  libGAP_ResizeTmpTrans(3*deg);
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptlookup=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
  ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
  
  for(i=0;i<deg;i++){ ptseen[i]=0; ptlookup[i]=0; ptlens[i]=0; }

  nr=0; m=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    //find components
    ptf2=libGAP_ADDR_TRANS2(f);
    for(i=0;i<deg;i++){
      if(ptseen[i]==0){
        m++;
        for(j=i;ptseen[j]==0;j=ptf2[j]){ ptseen[j]=m; } 
        if(ptseen[j]==m){
          ptlookup[m-1]=nr++;
        }else{
          ptlookup[m-1]=ptlookup[ptseen[j]-1];
        }
      }
    }
  } else {
    //find components 
    ptf4=libGAP_ADDR_TRANS4(f);
    for(i=0;i<deg;i++){
      if(ptseen[i]==0){
        m++;
        for(j=i;ptseen[j]==0;j=ptf4[j]){ ptseen[j]=m; } 
        if(ptseen[j]==m){
          ptlookup[m-1]=nr++;
        }else{
          ptlookup[m-1]=ptlookup[ptseen[j]-1];
        }
      }
    }
  }
  
  out=libGAP_NEW_PLIST(libGAP_T_PLIST, nr);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) nr);

  // install the points in out
  for(i=0;i<deg;i++){
    ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
    ptlookup=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
    ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
    
    m=ptlookup[ptseen[i]-1];
    if(ptlens[m]==0){
      libGAP_SET_ELM_PLIST(out, m+1, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, deg));
      libGAP_CHANGED_BAG(out);
      ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
    }
    libGAP_AssPlist(libGAP_ELM_PLIST(out, m+1), (libGAP_Int) ++ptlens[m], libGAP_INTOBJ_INT(i+1));
  }
  
  ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+2*deg;
  for(i=0;i<nr;i++){
    libGAP_SHRINK_PLIST(libGAP_ELM_PLIST(out, i+1), (libGAP_Int) ptlens[i]);
    libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, i+1), (libGAP_Int) ptlens[i]);
  }
  return out;
}


libGAP_Obj libGAP_FuncCOMPONENT_TRANS_INT(libGAP_Obj self, libGAP_Obj f, libGAP_Obj pt){
  libGAP_UInt    deg, cpt, len;
  libGAP_Obj     out;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptf4;
    
  deg=libGAP_INT_INTOBJ(libGAP_FuncDegreeOfTransformation(self, f));
  cpt=libGAP_INT_INTOBJ(pt)-1;
  
  if(cpt>=deg){
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, 1);
    libGAP_SET_LEN_PLIST(out, 1);
    libGAP_SET_ELM_PLIST(out, 1, pt);
    return out;
  }
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, deg);
  ptseen=libGAP_ResizeInitTmpTrans(deg);
  
  len=0;
  
  //install the points
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    do{ libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(cpt+1));
        ptseen[cpt]=1;
        cpt=ptf2[cpt];
    }while(ptseen[cpt]==0);
  } else {
    ptf4=libGAP_ADDR_TRANS4(f);
    do{ libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(cpt+1));
        ptseen[cpt]=1;
        cpt=ptf4[cpt];
    }while(ptseen[cpt]==0);
  }
  libGAP_SHRINK_PLIST(out, (libGAP_Int) len);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) len);
  return out;
}

libGAP_Obj libGAP_FuncCYCLE_TRANS_INT(libGAP_Obj self, libGAP_Obj f, libGAP_Obj pt){
  libGAP_UInt    deg, cpt, len, i;
  libGAP_Obj     out;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptf4;
    
  deg=libGAP_INT_INTOBJ(libGAP_FuncDegreeOfTransformation(self, f));
  cpt=libGAP_INT_INTOBJ(pt)-1;
  
  if(cpt>=deg){
    out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC_SSORT, 1);
    libGAP_SET_LEN_PLIST(out, 1);
    libGAP_SET_ELM_PLIST(out, 1, pt);
    return out;
  }
 
  out=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, deg);
  ptseen=libGAP_ResizeInitTmpTrans(deg);
  len=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2=libGAP_ADDR_TRANS2(f);
    //find component 
    do{ ptseen[cpt]=1;
        cpt=ptf2[cpt];
    }while(ptseen[cpt]==0);
    //find cycle
    i=cpt;
    do{ libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
        i=ptf2[i];
    }while(i!=cpt);
  } else {
    ptf4=libGAP_ADDR_TRANS4(f);
    //find component 
    do{ ptseen[cpt]=1;
        cpt=ptf4[cpt];
    }while(ptseen[cpt]==0);
    //find cycle
    i=cpt;
    do{ libGAP_SET_ELM_PLIST(out, ++len, libGAP_INTOBJ_INT(i+1));
        i=ptf4[i];
    }while(i!=cpt);
  }
  libGAP_SHRINK_PLIST (out, (libGAP_Int) len);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) len);
  return out;
}


libGAP_Obj libGAP_FuncCYCLES_TRANS_LIST(libGAP_Obj self, libGAP_Obj f, libGAP_Obj list){
  libGAP_UInt    deg, pt, len_list, len_out, i, j, m;
  libGAP_Obj     out;
  libGAP_UInt2   *ptf2;
  libGAP_UInt4   *ptseen, *ptlens, *ptf4;
   
  deg=libGAP_INT_INTOBJ(libGAP_FuncDegreeOfTransformation(self, f));
  len_list=libGAP_LEN_LIST(list);
 
  libGAP_ResizeTmpTrans(deg+len_list);
  out=libGAP_NEW_PLIST(libGAP_T_PLIST, len_list);
  
  ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;

  for(i=0;i<deg;i++){ ptseen[i]=0; ptlens[i]=0; }
  for(;i<len_list;i++){ ptlens[i]=0; }

  len_out=0; m=0;
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    for(i=1;i<=len_list;i++){
      pt=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i))-1;
      if(pt>=deg){
        libGAP_SET_ELM_PLIST(out, ++len_out, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 1));
        libGAP_SET_ELM_PLIST(libGAP_ELM_PLIST(out, len_out), 1, libGAP_INTOBJ_INT(pt+1));
        libGAP_CHANGED_BAG(out);
        (((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg)[len_out])++; //ptlens[len_out]++
      } else {
        m++;
        ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
        ptf2=libGAP_ADDR_TRANS2(f);
        while(ptseen[pt]==0){ //look for pts already seen
          ptseen[pt]=m;
          pt=ptf2[pt];
        }
        if(ptseen[pt]==m){//new cycle
          j=pt;
          libGAP_SET_ELM_PLIST(out, ++len_out, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 32));
          libGAP_CHANGED_BAG(out);
          ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
          do{ libGAP_AssPlist(libGAP_ELM_PLIST(out, len_out), ++ptlens[len_out], 
               libGAP_INTOBJ_INT(j+1));
              j=(libGAP_ADDR_TRANS2(f))[j];
              ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
          }while(j!=pt);
        }
      }
    }
  } else {
    for(i=1;i<=len_list;i++){
      pt=libGAP_INT_INTOBJ(libGAP_ELM_LIST(list, i))-1;
      if(pt>=deg){
        libGAP_SET_ELM_PLIST(out, ++len_out, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 1));
        libGAP_SET_ELM_PLIST(libGAP_ELM_PLIST(out, len_out), 1, libGAP_INTOBJ_INT(pt+1));
        libGAP_CHANGED_BAG(out);
        (((libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg)[len_out])++; //ptlens[len_out]++
      } else {
        m++;
        ptseen=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
        ptf4=libGAP_ADDR_TRANS4(f);
        while(ptseen[pt]==0){ //look for pts already seen
          ptseen[pt]=m;
          pt=ptf4[pt];
        }
        if(ptseen[pt]==m){//new cycle
          j=pt;
          libGAP_SET_ELM_PLIST(out, ++len_out, libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, 32));
          libGAP_CHANGED_BAG(out);
          ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
          do{ libGAP_AssPlist(libGAP_ELM_PLIST(out, len_out), ++ptlens[len_out], 
               libGAP_INTOBJ_INT(j+1));
              j=(libGAP_ADDR_TRANS4(f))[j];
              ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
          }while(j!=pt);
        }
      }
    }
  }
  ptlens=(libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans))+deg;
  for(i=1;i<=len_out;i++){
    libGAP_SHRINK_PLIST (libGAP_ELM_PLIST(out, i), (libGAP_Int) ptlens[i]);
    libGAP_SET_LEN_PLIST(libGAP_ELM_PLIST(out, i), (libGAP_Int) ptlens[i]);
  }
  libGAP_SHRINK_PLIST (out, (libGAP_Int) len_out);
  libGAP_SET_LEN_PLIST(out, (libGAP_Int) len_out);
  return out;
}

/* an idempotent transformation <e> with ker(e)=ker(f) */
libGAP_Obj libGAP_FuncLEFT_ONE_TRANS( libGAP_Obj self, libGAP_Obj f){
  libGAP_Obj   ker, img;
  libGAP_UInt  rank, n, i;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    rank=libGAP_RANK_TRANS2(f);
    ker=libGAP_KER_TRANS(f);
  } else {
    rank=libGAP_RANK_TRANS4(f);
    ker=libGAP_KER_TRANS(f);
  }
  img=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, rank);
  n=1;

  for(i=1;n<=rank;i++){
    if(libGAP_INT_INTOBJ(libGAP_ELM_PLIST(ker, i))==n){
      libGAP_SET_ELM_PLIST(img, n++, libGAP_INTOBJ_INT(i));
    }
  }
  
  libGAP_SET_LEN_PLIST(img, (libGAP_Int) n-1);
  return libGAP_FuncIDEM_IMG_KER_NC(self, img, ker);
}

/* an idempotent transformation <e> with im(e)=im(f) */
libGAP_Obj libGAP_FuncRIGHT_ONE_TRANS( libGAP_Obj self, libGAP_Obj f){
  libGAP_Obj   ker, img;
  libGAP_UInt  deg, len, i, j, n;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    deg=libGAP_DEG_TRANS2(f);
  } else {
    deg=libGAP_DEG_TRANS4(f);
  }

  img=libGAP_FuncIMAGE_SET_TRANS(self, f);
  ker=libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, deg);
  libGAP_SET_LEN_PLIST(ker, deg);
  len=libGAP_LEN_PLIST(img);
  j=1; n=0;

  for(i=0;i<deg;i++){
    if(j<len&&i+1==libGAP_INT_INTOBJ(libGAP_ELM_PLIST(img, j+1))) j++;
    libGAP_SET_ELM_PLIST(ker, ++n, libGAP_INTOBJ_INT(j));
  }
  return libGAP_FuncIDEM_IMG_KER_NC(self, img, ker);
}

//
libGAP_Obj libGAP_FuncIsCommutingTransformation( libGAP_Obj self, libGAP_Obj f, libGAP_Obj libGAP_g ){
  libGAP_UInt    def, deg, i;
  libGAP_UInt2   *ptf2, *ptg2;
  libGAP_UInt4   *ptf4, *ptg4;

  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    def=libGAP_DEG_TRANS2(f);
    ptf2=libGAP_ADDR_TRANS2(f);
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS2){
      deg=libGAP_DEG_TRANS2(libGAP_g);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      if(def<deg){
        for(i=0;i<def;i++){
          if(ptf2[ptg2[i]]!=ptg2[ptf2[i]]){
            return libGAP_False;
          }
        }
        for(;i<deg;i++){
          if(libGAP_IMAGE(ptg2[i], ptf2, def)!=ptg2[i]){
            return libGAP_False;
          }
        }
        return libGAP_True;
      }else{
        for(i=0;i<deg;i++){
          if(ptf2[ptg2[i]]!=ptg2[ptf2[i]]){
            return libGAP_False;
          }
        }
        for(;i<def;i++){
          if(libGAP_IMAGE(ptf2[i], ptg2, deg)!=ptf2[i]){
            return libGAP_False;
          }
        }
      }
    } else if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS4){
      deg=libGAP_DEG_TRANS4(libGAP_g);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      if(def<deg){
        for(i=0;i<def;i++){
          if(ptf2[ptg4[i]]!=ptg4[ptf2[i]]){
            return libGAP_False;
          }
        }
        for(;i<deg;i++){
          if(libGAP_IMAGE(ptg4[i], ptf2, def)!=ptg4[i]){
            return libGAP_False;
          }
        }
        return libGAP_True;
      }else{
        for(i=0;i<deg;i++){
          if(ptf2[ptg4[i]]!=ptg4[ptf2[i]]){
            return libGAP_False;
          }
        }
        for(;i<def;i++){
          if(libGAP_IMAGE(ptf2[i], ptg4, deg)!=ptf2[i]){
            return libGAP_False;
          }
        }
      }
    } else {
      libGAP_ErrorQuit("usage: the arguments must be transformations,", 0L, 0L);
    }
  } else if (libGAP_TNUM_OBJ(f)==libGAP_T_TRANS4){
    def=libGAP_DEG_TRANS4(f);
    ptf4=libGAP_ADDR_TRANS4(f);
    if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS2){
      deg=libGAP_DEG_TRANS2(libGAP_g);
      ptg2=libGAP_ADDR_TRANS2(libGAP_g);
      if(def<deg){
        for(i=0;i<def;i++){
          if(ptf4[ptg2[i]]!=ptg2[ptf4[i]]){
            return libGAP_False;
          }
        }
        for(;i<deg;i++){
          if(libGAP_IMAGE(ptg2[i], ptf4, def)!=ptg2[i]){
            return libGAP_False;
          }
        }
        return libGAP_True;
      }else{
        for(i=0;i<deg;i++){
          if(ptf4[ptg2[i]]!=ptg2[ptf4[i]]){
            return libGAP_False;
          }
        }
        for(;i<def;i++){
          if(libGAP_IMAGE(ptf4[i], ptg2, deg)!=ptf4[i]){
            return libGAP_False;
          }
        }
      }
    } else if(libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_TRANS4){
      deg=libGAP_DEG_TRANS4(libGAP_g);
      ptg4=libGAP_ADDR_TRANS4(libGAP_g);
      if(def<deg){
        for(i=0;i<def;i++){
          if(ptf4[ptg4[i]]!=ptg4[ptf4[i]]){
            return libGAP_False;
          }
        }
        for(;i<deg;i++){
          if(libGAP_IMAGE(ptg4[i], ptf4, def)!=ptg4[i]){
            return libGAP_False;
          }
        }
        return libGAP_True;
      }else{
        for(i=0;i<deg;i++){
          if(ptf4[ptg4[i]]!=ptg4[ptf4[i]]){
            return libGAP_False;
          }
        }
        for(;i<def;i++){
          if(libGAP_IMAGE(ptf4[i], ptg4, deg)!=ptf4[i]){
            return libGAP_False;
          }
        }
      }
    }
  } else {
    libGAP_ErrorQuit("usage: the arguments must be transformations,", 0L, 0L);
  }
  return libGAP_True;
}
/****************************************************************************/

/* GAP kernel functions */

/* one for a transformation */
libGAP_Obj libGAP_OneTrans( libGAP_Obj f ){
  return libGAP_IdentityTrans;
}

/* equality for transformations */

libGAP_Int libGAP_EqTrans22 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt    i, def, deg;
  libGAP_UInt2   *ptf, *ptg;

  ptf=libGAP_ADDR_TRANS2(f);   ptg=libGAP_ADDR_TRANS2(libGAP_g);
  def=libGAP_DEG_TRANS2(f);    deg=libGAP_DEG_TRANS2(libGAP_g);

  if(def<=deg){
    for(i=0;i<def;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<deg;i++)    if(*(ptg++)!=i) return 0L;
  } else {
    for(i=0;i<deg;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<def;i++)    if(*(ptf++)!=i) return 0L;
  }
  
  /* otherwise they must be equal */
  return 1L;
}


libGAP_Int libGAP_EqTrans24 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt   i, def, deg;
  libGAP_UInt2  *ptf;
  libGAP_UInt4  *ptg;

  ptf=libGAP_ADDR_TRANS2(f);   ptg=libGAP_ADDR_TRANS4(libGAP_g);
  def=libGAP_DEG_TRANS2(f);    deg=libGAP_DEG_TRANS4(libGAP_g);

  if(def<=deg){
    for(i=0;i<def;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<deg;i++)    if(*(ptg++)!=i) return 0L;
  } else {
    for(i=0;i<deg;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<def;i++)    if(*(ptf++)!=i) return 0L;
  }
  
  /* otherwise they must be equal */
  return 1L;
}


libGAP_Int libGAP_EqTrans42 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt   i, def, deg;
  libGAP_UInt4  *ptf;
  libGAP_UInt2  *ptg;

  ptf=libGAP_ADDR_TRANS4(f);   ptg=libGAP_ADDR_TRANS2(libGAP_g);
  def=libGAP_DEG_TRANS4(f);    deg=libGAP_DEG_TRANS2(libGAP_g);

  if(def<=deg){
    for(i=0;i<def;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<deg;i++)    if(*(ptg++)!=i)        return 0L;
  } else {
    for(i=0;i<deg;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<def;i++)    if(*(ptf++)!=i)        return 0L;
  }
  
  /* otherwise they must be equal */
  return 1L;
}


libGAP_Int libGAP_EqTrans44 (libGAP_Obj f, libGAP_Obj libGAP_g){
  libGAP_UInt   i, def, deg;
  libGAP_UInt4  *ptf, *ptg;

  ptf=libGAP_ADDR_TRANS4(f);   ptg=libGAP_ADDR_TRANS4(libGAP_g);
  def=libGAP_DEG_TRANS4(f);    deg=libGAP_DEG_TRANS4(libGAP_g);

  if(def<=deg){
    for(i=0;i<def;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<deg;i++)    if(*(ptg++)!=i) return 0L;
  } else {
    for(i=0;i<deg;i++) if(*(ptf++)!=*(ptg++)) return 0L;
    for(;i<def;i++)    if(*(ptf++)!=i) return 0L;
  }
  
  /* otherwise they must be equal */
  return 1L;
}

/* less than for transformations */

libGAP_Int libGAP_LtTrans22(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt   i, def, deg;
  libGAP_UInt2  *ptf, *ptg;

  ptf=libGAP_ADDR_TRANS2(f);   ptg=libGAP_ADDR_TRANS2(libGAP_g);
  def= libGAP_DEG_TRANS2(f);   deg= libGAP_DEG_TRANS2(libGAP_g);
  
  if(def<=deg){
    for(i=0;i<def;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<deg;i++){
      if(ptg[i]!=i){ 
        if(i<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
  } else { //def>deg
    for(i=0;i<deg;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<def;i++){
      if(ptf[i]!=i){ 
        if(i>ptf[i]){ return 1L; } else { return 0L; }
      }
    }
  }
  return 0L;
}


libGAP_Int libGAP_LtTrans24(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt   i, def, deg;
  libGAP_UInt2  *ptf;
  libGAP_UInt4  *ptg;

  ptf=libGAP_ADDR_TRANS2(f);   ptg=libGAP_ADDR_TRANS4(libGAP_g);
  def= libGAP_DEG_TRANS2(f);   deg= libGAP_DEG_TRANS4(libGAP_g);
  
  if(def<=deg){
    for(i=0;i<def;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<deg;i++){
      if(ptg[i]!=i){ 
        if(i<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
  } else {
    for(i=0;i<deg;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<def;i++){
      if(ptf[i]!=i){ 
        if(i>ptf[i]){ return 1L; } else { return 0L; }
      }
    }
  }
  return 0L;
}


libGAP_Int libGAP_LtTrans42(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt   i, def, deg;
  libGAP_UInt4  *ptf; 
  libGAP_UInt2  *ptg;

  ptf=libGAP_ADDR_TRANS4(f);   ptg=libGAP_ADDR_TRANS2(libGAP_g);
  def= libGAP_DEG_TRANS4(f);   deg= libGAP_DEG_TRANS2(libGAP_g);
  
  if(def<=deg){
    for(i=0;i<def;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<deg;i++){
      if(ptg[i]!=i){ 
        if(i<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
  } else {
    for(i=0;i<deg;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<def;i++){
      if(ptf[i]!=i){ 
        if(i>ptf[i]){ return 1L; } else { return 0L; }
      }
    }
  }
  return 0L;
}


libGAP_Int libGAP_LtTrans44(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt   i, def, deg;
  libGAP_UInt4  *ptf, *ptg;

  ptf=libGAP_ADDR_TRANS4(f);   ptg=libGAP_ADDR_TRANS4(libGAP_g);
  def= libGAP_DEG_TRANS4(f);   deg= libGAP_DEG_TRANS4(libGAP_g);
  
  if(def<=deg){
    for(i=0;i<def;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<deg;i++){
      if(ptg[i]!=i){ 
        if(i<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
  } else {
    for(i=0;i<deg;i++){ 
      if(ptf[i]!=ptg[i]){
        if(ptf[i]<ptg[i]){ return 1L; } else { return 0L; }
      }
    }
    for(;i<def;i++){
      if(ptf[i]!=i){ 
        if(i>ptf[i]){ return 1L; } else { return 0L; }
      }
    }
  }
  return 0L;
}

/* product of transformations */
libGAP_Obj libGAP_ProdTrans22(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt2   *ptf, *ptg, *ptfg;
  libGAP_UInt    i, def, deg, defg;
  libGAP_Obj     fg;

  def =libGAP_DEG_TRANS2(f);
  deg =libGAP_DEG_TRANS2(libGAP_g);
  defg=libGAP_MAX(def,deg);
  fg=libGAP_NEW_TRANS2(defg);
  
  ptfg=libGAP_ADDR_TRANS2(fg);
  ptf =libGAP_ADDR_TRANS2(f);
  ptg =libGAP_ADDR_TRANS2(libGAP_g);
  if(def<=deg){
    for(i=0;i<def;i++) *(ptfg++)=ptg[*(ptf++)];
    for(;i<deg;i++) *(ptfg++)=ptg[i];
  } else {
    for(i=0;i<def;i++) *(ptfg++)=libGAP_IMAGE(ptf[i], ptg, deg);
  }
  return fg;
}

libGAP_Obj libGAP_ProdTrans24(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt2   *ptf; 
  libGAP_UInt4   *ptg, *ptfg;
  libGAP_UInt    i, def, deg, defg;
  libGAP_Obj     fg;

  def =libGAP_DEG_TRANS2(f);
  deg =libGAP_DEG_TRANS4(libGAP_g);
  defg=libGAP_MAX(def,deg);
  fg=libGAP_NEW_TRANS4(defg);
  
  ptfg=libGAP_ADDR_TRANS4(fg);
  ptf =libGAP_ADDR_TRANS2(f);
  ptg =libGAP_ADDR_TRANS4(libGAP_g);
  if(def<=deg){
    for(i=0;i<def;i++) *ptfg++=ptg[*ptf++]; 
    for(;i<deg;i++) *ptfg++=ptg[i]; 
  } else {
    for(i=0;i<def;i++) *(ptfg++)=libGAP_IMAGE(ptf[i], ptg, deg);
  }
  return fg;
}

libGAP_Obj libGAP_ProdTrans42(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt4   *ptf, *ptfg;
  libGAP_UInt2   *ptg;
  libGAP_UInt    i, def, deg, defg;
  libGAP_Obj     fg;

  def=libGAP_DEG_TRANS4(f);
  deg=libGAP_DEG_TRANS2(libGAP_g);
  defg=libGAP_MAX(def,deg);
  fg=libGAP_NEW_TRANS4(defg);
  
  ptfg=libGAP_ADDR_TRANS4(fg);
  ptf =libGAP_ADDR_TRANS4(f);
  ptg =libGAP_ADDR_TRANS2(libGAP_g);
  if(def<=deg){
    for(i=0;i<def;i++) *(ptfg++)=ptg[*(ptf++)];
    for(;i<deg;i++) *(ptfg++)=ptg[i];
  } else {
    for(i=0;i<def;i++) *(ptfg++)=libGAP_IMAGE(ptf[i], ptg, deg);
  }
  return fg;
}

libGAP_Obj libGAP_ProdTrans44(libGAP_Obj f, libGAP_Obj libGAP_g){ 
  libGAP_UInt4   *ptf, *ptg, *ptfg;
  libGAP_UInt    i, def, deg, defg;
  libGAP_Obj     fg;

  def=libGAP_DEG_TRANS4(f);
  deg=libGAP_DEG_TRANS4(libGAP_g);
  defg=libGAP_MAX(def,deg);
  fg=libGAP_NEW_TRANS4(defg);
  
  ptfg=libGAP_ADDR_TRANS4(fg);
  ptf =libGAP_ADDR_TRANS4(f);
  ptg =libGAP_ADDR_TRANS4(libGAP_g);
  if(def<=deg){
    for(i=0;i<def;i++) *(ptfg++)=ptg[*(ptf++)];
    for(;i<deg;i++) *(ptfg++)=ptg[i];
  } else {
    for(i=0;i<def;i++) *(ptfg++)=libGAP_IMAGE(ptf[i], ptg, deg);
  }
  return fg;
}

/* product of transformation and permutation */
libGAP_Obj libGAP_ProdTrans2Perm2(libGAP_Obj f, libGAP_Obj p){ /* p(f(x)) */
  libGAP_UInt2   *ptf, *ptp, *ptfp;
  libGAP_UInt    i, def, dep, defp;
  libGAP_Obj     fp;

  dep =  libGAP_DEG_PERM2(p);
  def = libGAP_DEG_TRANS2(f);
  defp=libGAP_MAX(def,dep);
  fp  = libGAP_NEW_TRANS2(defp);

  ptfp=libGAP_ADDR_TRANS2(fp);
  ptf =libGAP_ADDR_TRANS2(f);
  ptp = libGAP_ADDR_PERM2(p);
  
  if(def<=dep){
    for(i=0;i<def;i++) *(ptfp++)=ptp[*(ptf++)];
    for(;i<dep;i++) *(ptfp++)=ptp[i];
  } else {
    for(i=0;i<def;i++) *(ptfp++)=libGAP_IMAGE(ptf[i], ptp, dep);
  }
  return fp;
}

libGAP_Obj libGAP_ProdTrans2Perm4(libGAP_Obj f, libGAP_Obj p){ /* p(f(x)) */
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *ptp, *ptfp;
  libGAP_UInt    i, def, dep, defp;
  libGAP_Obj     fp;

  dep =  libGAP_DEG_PERM4(p);
  def = libGAP_DEG_TRANS2(f);
  defp= libGAP_MAX(def,dep);
  fp  = libGAP_NEW_TRANS4(defp);

  ptfp=libGAP_ADDR_TRANS4(fp);
  ptf =libGAP_ADDR_TRANS2(f);
  ptp = libGAP_ADDR_PERM4(p);
  
  if(def<=dep){
    for(i=0;i<def;i++) *(ptfp++)=ptp[*(ptf++)];
    for(;i<dep;i++) *(ptfp++)=ptp[i];
  } else {
    for(i=0;i<def;i++) *(ptfp++)=libGAP_IMAGE(ptf[i], ptp, dep);
  }
  return fp;
}

libGAP_Obj libGAP_ProdTrans4Perm2(libGAP_Obj f, libGAP_Obj p){ /* p(f(x)) */
  libGAP_UInt4   *ptf, *ptfp;
  libGAP_UInt2   *ptp;
  libGAP_UInt    i, def, dep, defp;
  libGAP_Obj     fp;

  dep =  libGAP_DEG_PERM2(p);
  def = libGAP_DEG_TRANS4(f);
  defp= libGAP_MAX(def,dep);
  fp  = libGAP_NEW_TRANS4(defp);

  ptfp=libGAP_ADDR_TRANS4(fp);
  ptf =libGAP_ADDR_TRANS4(f);
  ptp = libGAP_ADDR_PERM2(p);
  
  if(def<=dep){
    for(i=0;i<def;i++) *(ptfp++)=ptp[*(ptf++)];
    for(;i<dep;i++) *(ptfp++)=ptp[i];
  } else {
    for(i=0;i<def;i++) *(ptfp++)=libGAP_IMAGE(ptf[i], ptp, dep);
  }
  return fp;
}

libGAP_Obj libGAP_ProdTrans4Perm4(libGAP_Obj f, libGAP_Obj p){ /* p(f(x)) */
  libGAP_UInt4   *ptf, *ptp, *ptfp;
  libGAP_UInt    i, def, dep, defp;
  libGAP_Obj     fp;

  dep =  libGAP_DEG_PERM4(p);
  def = libGAP_DEG_TRANS4(f);
  defp= libGAP_MAX(def,dep);
  fp  = libGAP_NEW_TRANS4(defp);
  
  ptfp=libGAP_ADDR_TRANS4(fp);
  ptf =libGAP_ADDR_TRANS4(f);
  ptp = libGAP_ADDR_PERM4(p);
  
  if(def<=dep){
    for(i=0;i<def;i++) *(ptfp++)=ptp[*(ptf++)];
    for(;i<dep;i++) *(ptfp++)=ptp[i];
  } else {
    for(i=0;i<def;i++) *(ptfp++)=libGAP_IMAGE(ptf[i], ptp, dep);
  }
  return fp;
}

/* product of permutation and transformation */
libGAP_Obj libGAP_ProdPerm2Trans2(libGAP_Obj p, libGAP_Obj f){ /* f(p(x)) */
  libGAP_UInt2   *ptf, *ptp, *ptpf;
  libGAP_UInt    i, def, dep, depf;
  libGAP_Obj     pf;

  dep =  libGAP_DEG_PERM2(p);
  def = libGAP_DEG_TRANS2(f);
  depf= libGAP_MAX(def,dep);
  pf  = libGAP_NEW_TRANS2(depf);

  ptpf=libGAP_ADDR_TRANS2(pf);
  ptf =libGAP_ADDR_TRANS2(f);
  ptp = libGAP_ADDR_PERM2(p);
  
  if(dep<=def){
    for(i=0;i<dep;i++) *(ptpf++)=ptf[*(ptp++)];
    for(;i<def;i++) *(ptpf++)=ptf[i];
  } else {
    for(i=0;i<dep;i++) *(ptpf++)=libGAP_IMAGE(ptp[i], ptf, def);
  }
  return pf;
}

libGAP_Obj libGAP_ProdPerm2Trans4(libGAP_Obj p, libGAP_Obj f){ /* f(p(x)) */
  libGAP_UInt4   *ptf, *ptpf; 
  libGAP_UInt2   *ptp;
  libGAP_UInt    i, def, dep, depf;
  libGAP_Obj     pf;

  dep =  libGAP_DEG_PERM2(p);
  def = libGAP_DEG_TRANS4(f);
  depf= libGAP_MAX(def,dep);
  pf  = libGAP_NEW_TRANS4(depf);

  ptpf=libGAP_ADDR_TRANS4(pf);
  ptf =libGAP_ADDR_TRANS4(f);
  ptp = libGAP_ADDR_PERM2(p);
  
  if(dep<=def){
    for(i=0;i<dep;i++) *(ptpf++)=ptf[*(ptp++)];
    for(;i<def;i++) *(ptpf++)=ptf[i];
  } else {
    for(i=0;i<dep;i++) *(ptpf++)=libGAP_IMAGE(ptp[i], ptf, def);
  }
  return pf;
}

libGAP_Obj libGAP_ProdPerm4Trans2(libGAP_Obj p, libGAP_Obj f){ /* f(p(x)) */
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *ptp, *ptpf;
  libGAP_UInt    i, def, dep, depf;
  libGAP_Obj     pf;

  dep =  libGAP_DEG_PERM4(p);
  def = libGAP_DEG_TRANS2(f);
  depf= libGAP_MAX(def,dep);
  pf  = libGAP_NEW_TRANS4(depf);

  ptpf=libGAP_ADDR_TRANS4(pf);
  ptf =libGAP_ADDR_TRANS2(f);
  ptp = libGAP_ADDR_PERM4(p);
  
  if(dep<=def){
    for(i=0;i<dep;i++) *(ptpf++)=ptf[*(ptp++)];
    for(;i<def;i++) *(ptpf++)=ptf[i];
  } else {
    for(i=0;i<dep;i++) *(ptpf++)=libGAP_IMAGE(ptp[i], ptf, def);
  }
  return pf;
}

libGAP_Obj libGAP_ProdPerm4Trans4(libGAP_Obj p, libGAP_Obj f){ /* f(p(x)) */
  libGAP_UInt4   *ptf, *ptp, *ptpf;
  libGAP_UInt    i, def, dep, depf;
  libGAP_Obj     pf;

  dep =  libGAP_DEG_PERM4(p);
  def = libGAP_DEG_TRANS4(f);
  depf= libGAP_MAX(def,dep);
  pf  = libGAP_NEW_TRANS4(depf);

  ptpf=libGAP_ADDR_TRANS4(pf);
  ptf =libGAP_ADDR_TRANS4(f);
  ptp = libGAP_ADDR_PERM4(p);
  
  if(dep<=def){
    for(i=0;i<dep;i++) *(ptpf++)=ptf[*(ptp++)];
    for(;i<def;i++) *(ptpf++)=ptf[i];
  } else {
    for(i=0;i<dep;i++) *(ptpf++)=libGAP_IMAGE(ptp[i], ptf, def);
  }
  return pf;
}

/* Conjugation: p^-1*f*p */
libGAP_Obj libGAP_PowTrans2Perm2(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt2   *ptf, *ptp, *ptcnj;
  libGAP_UInt    i, def, dep, decnj;
  libGAP_Obj     cnj;

  dep  =  libGAP_DEG_PERM2(p);
  def  = libGAP_DEG_TRANS2(f);
  decnj=libGAP_MAX(dep,def);
  cnj  = libGAP_NEW_TRANS2(decnj);

  ptcnj=libGAP_ADDR_TRANS2(cnj);
  ptf  =libGAP_ADDR_TRANS2(f);
  ptp  = libGAP_ADDR_PERM2(p);
  
  if(def==dep){
    for(i=0;i<decnj;i++) ptcnj[ptp[i]]=ptp[ptf[i]];
  } else {
    for(i=0;i<decnj;i++){
      ptcnj[libGAP_IMAGE(i, ptp, dep)]=libGAP_IMAGE(libGAP_IMAGE(i, ptf, def), ptp, dep);
    }
  }
  return cnj;
}

libGAP_Obj libGAP_PowTrans2Perm4(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *ptp, *ptcnj;
  libGAP_UInt    i, def, dep, decnj;
  libGAP_Obj     cnj;

  dep  =  libGAP_DEG_PERM4(p);
  def  = libGAP_DEG_TRANS2(f);
  decnj=libGAP_MAX(dep,def);
  cnj  = libGAP_NEW_TRANS4(decnj);

  ptcnj=libGAP_ADDR_TRANS4(cnj);
  ptf  =libGAP_ADDR_TRANS2(f);
  ptp  = libGAP_ADDR_PERM4(p);
  
  if(def==dep){
    for(i=0;i<decnj;i++) ptcnj[ptp[i]]=ptp[ptf[i]];
  } else {
    for(i=0;i<decnj;i++){
      ptcnj[libGAP_IMAGE(i, ptp, dep)]=libGAP_IMAGE(libGAP_IMAGE(i, ptf, def), ptp, dep);
    }
  }
  return cnj;
}

libGAP_Obj libGAP_PowTrans4Perm2(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt2   *ptp;
  libGAP_UInt4   *ptf, *ptcnj;
  libGAP_UInt    i, def, dep, decnj;
  libGAP_Obj     cnj;

  dep  =  libGAP_DEG_PERM2(p);
  def  = libGAP_DEG_TRANS4(f);
  decnj=libGAP_MAX(dep,def);
  cnj  = libGAP_NEW_TRANS4(decnj);

  ptcnj=libGAP_ADDR_TRANS4(cnj);
  ptf  =libGAP_ADDR_TRANS4(f);
  ptp  = libGAP_ADDR_PERM2(p);
  
  if(def==dep){
    for(i=0;i<decnj;i++) ptcnj[ptp[i]]=ptp[ptf[i]];
  } else {
    for(i=0;i<decnj;i++){
      ptcnj[libGAP_IMAGE(i, ptp, dep)]=libGAP_IMAGE(libGAP_IMAGE(i, ptf, def), ptp, dep);
    }
  }
  return cnj;
}

libGAP_Obj libGAP_PowTrans4Perm4(libGAP_Obj f, libGAP_Obj p){
  libGAP_UInt4   *ptf, *ptp, *ptcnj;
  libGAP_UInt    i, def, dep, decnj;
  libGAP_Obj     cnj;

  dep  =  libGAP_DEG_PERM4(p);
  def  = libGAP_DEG_TRANS4(f);
  decnj=libGAP_MAX(dep,def);
  cnj  = libGAP_NEW_TRANS4(decnj);

  ptcnj=libGAP_ADDR_TRANS4(cnj);
  ptf  =libGAP_ADDR_TRANS4(f);
  ptp  = libGAP_ADDR_PERM4(p);
  
  if(def==dep){
    for(i=0;i<decnj;i++) ptcnj[ptp[i]]=ptp[ptf[i]];
  } else {
    for(i=0;i<decnj;i++){
      ptcnj[libGAP_IMAGE(i, ptp, dep)]=libGAP_IMAGE(libGAP_IMAGE(i, ptf, def), ptp, dep);
    }
  }
  return cnj;
}

/* f*p^-1 */
libGAP_Obj libGAP_QuoTrans2Perm2(libGAP_Obj f, libGAP_Obj p){ 
  libGAP_UInt    def, dep, deq, i;
  libGAP_UInt2   *ptf, *ptquo, *ptp;
  libGAP_UInt4   *pttmp;
  libGAP_Obj     quo;

  def=libGAP_DEG_TRANS2(f);
  dep= libGAP_DEG_PERM2(p);
  deq=libGAP_MAX(def, dep);
  quo=libGAP_NEW_TRANS2( deq );
  libGAP_ResizeTmpTrans(libGAP_SIZE_OBJ(p));

  /* invert the permutation into the buffer bag */
  pttmp = (libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptp =   libGAP_ADDR_PERM2(p);
  for(i=0;i<dep;i++) pttmp[*ptp++]=i;

  ptf   = libGAP_ADDR_TRANS2(f);
  ptquo = libGAP_ADDR_TRANS2(quo);

  if(def<=dep){
    for(i=0;i<def;i++) *(ptquo++)=pttmp[*(ptf++)];
    for(i=def;i<dep;i++) *(ptquo++)=pttmp[i];
  }
  else {
    for(i=0;i<def;i++) *(ptquo++)=libGAP_IMAGE(ptf[i], pttmp, dep );
  }
  return quo;
}

libGAP_Obj libGAP_QuoTrans2Perm4(libGAP_Obj f, libGAP_Obj p){ 
  libGAP_UInt    def, dep, deq, i;
  libGAP_UInt2   *ptf;
  libGAP_UInt4   *ptquo, *ptp, *pttmp;
  libGAP_Obj     quo;

  def=libGAP_DEG_TRANS2(f);
  dep= libGAP_DEG_PERM4(p);
  deq=libGAP_MAX(def, dep);
  quo=libGAP_NEW_TRANS4( deq );
  libGAP_ResizeTmpTrans(libGAP_SIZE_OBJ(p));

  /* invert the permutation into the buffer bag */
  pttmp = (libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptp =   libGAP_ADDR_PERM4(p);
  for(i=0;i<dep;i++) pttmp[*ptp++]=i;

  ptf   = libGAP_ADDR_TRANS2(f);
  ptquo = libGAP_ADDR_TRANS4(quo);

  if(def<=dep){
    for(i=0;i<def;i++) *(ptquo++)=pttmp[*(ptf++)];
    for(i=def;i<dep;i++) *(ptquo++)=pttmp[i];
  }
  else {
    for(i=0;i<def;i++) *(ptquo++)=libGAP_IMAGE(ptf[i], pttmp, dep );
  }
  return quo;
}

libGAP_Obj libGAP_QuoTrans4Perm2(libGAP_Obj f, libGAP_Obj p){ 
  libGAP_UInt    def, dep, deq, i;
  libGAP_UInt4   *ptf, *ptquo, *pttmp;
  libGAP_UInt2   *ptp;
  libGAP_Obj     quo;

  def=libGAP_DEG_TRANS4(f);
  dep= libGAP_DEG_PERM2(p);
  deq=libGAP_MAX(def, dep);
  quo=libGAP_NEW_TRANS4( deq );
  
  libGAP_ResizeTmpTrans(libGAP_SIZE_OBJ(p));

  /* invert the permutation into the buffer bag */
  pttmp = (libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptp =   libGAP_ADDR_PERM2(p);
  for(i=0;i<dep;i++) pttmp[*ptp++]=i;

  ptf   = libGAP_ADDR_TRANS4(f);
  ptquo = libGAP_ADDR_TRANS4(quo);

  if(def<=dep){
    for(i=0;i<def;i++) *(ptquo++)=pttmp[*(ptf++)];
    for(i=def;i<dep;i++) *(ptquo++)=pttmp[i];
  }
  else {
    for(i=0;i<def;i++) *(ptquo++)=libGAP_IMAGE(ptf[i], pttmp, dep );
  }
  return quo;
}

libGAP_Obj libGAP_QuoTrans4Perm4(libGAP_Obj f, libGAP_Obj p){ 
  libGAP_UInt    def, dep, deq, i;
  libGAP_UInt4   *ptf, *pttmp, *ptquo, *ptp;
  libGAP_Obj     quo;

  def=libGAP_DEG_TRANS4(f);
  dep= libGAP_DEG_PERM4(p);
  deq=libGAP_MAX(def, dep);
  quo=libGAP_NEW_TRANS4( deq );

  libGAP_ResizeTmpTrans(libGAP_SIZE_OBJ(p));

  /* invert the permutation into the buffer bag */
  pttmp = (libGAP_UInt4*)(libGAP_ADDR_OBJ(libGAP_TmpTrans));
  ptp =   libGAP_ADDR_PERM4(p);
  for(i=0;i<dep;i++) pttmp[*ptp++]=i;

  ptf   = libGAP_ADDR_TRANS4(f);
  ptquo = libGAP_ADDR_TRANS4(quo);

  if(def<=dep){
    for(i=0;i<def;i++) *(ptquo++)=pttmp[*(ptf++)];
    for(i=def;i<dep;i++) *(ptquo++)=pttmp[i];
  }
  else {
    for(i=0;i<def;i++) *(ptquo++)=libGAP_IMAGE(ptf[i], pttmp, dep );
  }
  return quo;
}

/* p^-1*f */
libGAP_Obj libGAP_LQuoPerm2Trans2(libGAP_Obj opL, libGAP_Obj opR){ 
  libGAP_UInt   degL, degR, degM, p;
  libGAP_Obj    mod;
  libGAP_UInt2  *ptL, *ptR, *ptM; 

  degL = libGAP_DEG_PERM2(opL);
  degR = libGAP_DEG_TRANS2(opR);
  degM = degL < degR ? degR : degL;
  mod = libGAP_NEW_TRANS2( degM );

  /* set up the pointers                                                 */
  ptL = libGAP_ADDR_PERM2(opL);
  ptR = libGAP_ADDR_TRANS2(opR);
  ptM = libGAP_ADDR_TRANS2(mod);

  /* its one thing if the left (inner) permutation is smaller            */
  if ( degL <= degR ) {
      for ( p = 0; p < degL; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degL; p < degR; p++ )
          ptM[ p ] = *(ptR++);
  }

  /* and another if the right (outer) permutation is smaller             */
  else {
      for ( p = 0; p < degR; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degR; p < degL; p++ )
          ptM[ *(ptL++) ] = p;
  }

  /* return the result                                                   */
  return mod;
}

libGAP_Obj libGAP_LQuoPerm2Trans4(libGAP_Obj opL, libGAP_Obj opR){ 
  libGAP_UInt   degL, degR, degM, p;
  libGAP_Obj    mod;
  libGAP_UInt2  *ptL;
  libGAP_UInt4  *ptR, *ptM; 

  degL = libGAP_DEG_PERM2(opL);
  degR = libGAP_DEG_TRANS4(opR);
  degM = degL < degR ? degR : degL;
  mod = libGAP_NEW_TRANS4( degM );

  /* set up the pointers                                                 */
  ptL = libGAP_ADDR_PERM2(opL);
  ptR = libGAP_ADDR_TRANS4(opR);
  ptM = libGAP_ADDR_TRANS4(mod);

  /* its one thing if the left (inner) permutation is smaller            */
  if ( degL <= degR ) {
      for ( p = 0; p < degL; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degL; p < degR; p++ )
          ptM[ p ] = *(ptR++);
  }

  /* and another if the right (outer) permutation is smaller             */
  else {
      for ( p = 0; p < degR; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degR; p < degL; p++ )
          ptM[ *(ptL++) ] = p;
  }

  /* return the result                                                   */
  return mod;
}

libGAP_Obj libGAP_LQuoPerm4Trans2(libGAP_Obj opL, libGAP_Obj opR){ 
  libGAP_UInt   degL, degR, degM, p;
  libGAP_Obj    mod;
  libGAP_UInt4  *ptL, *ptM;
  libGAP_UInt2  *ptR; 

  degL = libGAP_DEG_PERM4(opL);
  degR = libGAP_DEG_TRANS2(opR);
  degM = degL < degR ? degR : degL;
  mod = libGAP_NEW_TRANS4( degM );

  /* set up the pointers                                                 */
  ptL = libGAP_ADDR_PERM4(opL);
  ptR = libGAP_ADDR_TRANS2(opR);
  ptM = libGAP_ADDR_TRANS4(mod);

  /* its one thing if the left (inner) permutation is smaller            */
  if ( degL <= degR ) {
      for ( p = 0; p < degL; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degL; p < degR; p++ )
          ptM[ p ] = *(ptR++);
  }

  /* and another if the right (outer) permutation is smaller             */
  else {
      for ( p = 0; p < degR; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degR; p < degL; p++ )
          ptM[ *(ptL++) ] = p;
  }

  /* return the result                                                   */
  return mod;
}

libGAP_Obj libGAP_LQuoPerm4Trans4(libGAP_Obj opL, libGAP_Obj opR){ 
  libGAP_UInt   degL, degR, degM, p;
  libGAP_Obj    mod;
  libGAP_UInt4  *ptL, *ptR, *ptM; 

  degL = libGAP_DEG_PERM4(opL);
  degR = libGAP_DEG_TRANS4(opR);
  degM = degL < degR ? degR : degL;
  mod = libGAP_NEW_TRANS4( degM );

  /* set up the pointers                                                 */
  ptL = libGAP_ADDR_PERM4(opL);
  ptR = libGAP_ADDR_TRANS4(opR);
  ptM = libGAP_ADDR_TRANS4(mod);

  /* its one thing if the left (inner) permutation is smaller            */
  if ( degL <= degR ) {
      for ( p = 0; p < degL; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degL; p < degR; p++ )
          ptM[ p ] = *(ptR++);
  }

  /* and another if the right (outer) permutation is smaller             */
  else {
      for ( p = 0; p < degR; p++ )
          ptM[ *(ptL++) ] = *(ptR++);
      for ( p = degR; p < degL; p++ )
          ptM[ *(ptL++) ] = p;
  }

  /* return the result                                                   */
  return mod;
}

/* i^f */
libGAP_Obj libGAP_PowIntTrans2(libGAP_Obj i, libGAP_Obj f){
  libGAP_Int    img;
 
  if(libGAP_TNUM_OBJ(i)==libGAP_T_INTPOS) return i; 

  if(libGAP_TNUM_OBJ(i)!=libGAP_T_INT){
    libGAP_ErrorQuit("usage: the first argument should be a positive integer", 0L, 0L);
  }
  
  img=libGAP_INT_INTOBJ(i);
  
  if(img<=0){
    libGAP_ErrorQuit("usage: the first argument should be a positive integer", 0L, 0L);
  }
  
  if(img<=libGAP_DEG_TRANS2(f)){
    img=(libGAP_ADDR_TRANS2(f))[img-1]+1;
  }
  
  return libGAP_INTOBJ_INT(img);
}

libGAP_Obj libGAP_PowIntTrans4(libGAP_Obj i, libGAP_Obj f){
  libGAP_Int    img;
 
  if(libGAP_TNUM_OBJ(i)==libGAP_T_INTPOS) return i; 

  if(libGAP_TNUM_OBJ(i)!=libGAP_T_INT){
    libGAP_ErrorQuit("usage: the first argument should be a positive integer", 0L, 0L);
  }
  
  img=libGAP_INT_INTOBJ(i);
  
  if(img<=0){
    libGAP_ErrorQuit("usage: the first argument should be a positive integer", 0L, 0L);
  }
  
  if(img<=libGAP_DEG_TRANS4(f)){
    img=(libGAP_ADDR_TRANS4(f))[img-1]+1;
  }
  
  return libGAP_INTOBJ_INT(img);
}

/* OnSetsTrans for use in FuncOnSets */
libGAP_Obj libGAP_OnSetsTrans (libGAP_Obj set, libGAP_Obj f){
  libGAP_UInt2  *ptf2;
  libGAP_UInt4  *ptf4;
  libGAP_UInt   deg;
  libGAP_Obj    *ptset, *ptres, tmp, res;
  libGAP_UInt   i, isint, k, h, len;

  res=libGAP_NEW_PLIST(libGAP_IS_MUTABLE_PLIST(set)?libGAP_T_PLIST:libGAP_T_PLIST+libGAP_IMMUTABLE,libGAP_LEN_LIST(set));
  libGAP_ADDR_OBJ(res)[0]=libGAP_ADDR_OBJ(set)[0]; 

  /* get the pointer                                                 */
  ptset = libGAP_ADDR_OBJ(set) + libGAP_LEN_LIST(set);
  ptres = libGAP_ADDR_OBJ(res) + libGAP_LEN_LIST(set);
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){   
    ptf2 = libGAP_ADDR_TRANS2(f);
    deg = libGAP_DEG_TRANS2(f);
    /* loop over the entries of the tuple                              */
    isint = 1;
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset--, ptres-- ) {
      if ( libGAP_TNUM_OBJ( *ptset ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *ptset ) ) {
        k = libGAP_INT_INTOBJ( *ptset );
        if ( k <= deg ) k = ptf2[k-1] + 1 ; 
        *ptres = libGAP_INTOBJ_INT(k);
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L); 
      }
    }
  } else {
    ptf4 = libGAP_ADDR_TRANS4(f);
    deg = libGAP_DEG_TRANS4(f);

    /* loop over the entries of the tuple                              */
    isint = 1;
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset--, ptres-- ) {
      if ( libGAP_TNUM_OBJ( *ptset ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *ptset ) ) {
        k = libGAP_INT_INTOBJ( *ptset );
        if ( k <= deg ) k = ptf4[k-1] + 1 ; 
        *ptres = libGAP_INTOBJ_INT(k);
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L); 
      }
    }
  }
  /* sort the result */
  len=libGAP_LEN_LIST(res);
  h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1;
  while ( 0 < h ) {
    for ( i = h+1; i <= len; i++ ) {
      tmp = libGAP_ADDR_OBJ(res)[i];  k = i;
      while ( h < k && ((libGAP_Int)tmp < (libGAP_Int)(libGAP_ADDR_OBJ(res)[k-h])) ) {
        libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h];
        k -= h;
      }
      libGAP_ADDR_OBJ(res)[k] = tmp;
    }
    h = h / 3;
  }

  /* remove duplicates */
  if ( 0 < len ) {
    tmp = libGAP_ADDR_OBJ(res)[1];  k = 1;
    for ( i = 2; i <= len; i++ ) {
      if ( ! libGAP_EQ( tmp, libGAP_ADDR_OBJ(res)[i] ) ) {
        k++;
        tmp = libGAP_ADDR_OBJ(res)[i];
        libGAP_ADDR_OBJ(res)[k] = tmp;
      }
    }
    if ( k < len ) {
      libGAP_ResizeBag( res, (k+1)*sizeof(libGAP_Obj) );
      libGAP_SET_LEN_PLIST(res, k);
    }
  }

  /* retype if we only have integers */
  if(isint){
    libGAP_RetypeBag( res, libGAP_IS_MUTABLE_PLIST(set) ? libGAP_T_PLIST_CYC_SSORT :
     libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE ); 
  }

  return res;
}

/* OnTuplesTrans for use in FuncOnTuples */
libGAP_Obj libGAP_OnTuplesTrans (libGAP_Obj tup, libGAP_Obj f){
  libGAP_UInt2  *ptf2;
  libGAP_UInt4  *ptf4;
  libGAP_UInt   deg, isint, i, k;
  libGAP_Obj    *pttup, *ptres, res;

  res=libGAP_NEW_PLIST(libGAP_IS_MUTABLE_PLIST(tup)?libGAP_T_PLIST:libGAP_T_PLIST+libGAP_IMMUTABLE,libGAP_LEN_LIST(tup));
  libGAP_ADDR_OBJ(res)[0]=libGAP_ADDR_OBJ(tup)[0]; 

  /* get the pointer                                                 */
  pttup = libGAP_ADDR_OBJ(tup) + libGAP_LEN_LIST(tup);
  ptres = libGAP_ADDR_OBJ(res) + libGAP_LEN_LIST(tup);
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){
    ptf2 = libGAP_ADDR_TRANS2(f);
    deg = libGAP_DEG_TRANS2(f);

    /* loop over the entries of the tuple                              */
    isint=1;
    for ( i =libGAP_LEN_LIST(tup) ; 1 <= i; i--, pttup--, ptres-- ) {
      if ( libGAP_TNUM_OBJ( *pttup ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *pttup ) ) {
        k = libGAP_INT_INTOBJ( *pttup );
        if ( k <= deg ) k = ptf2[k-1] + 1 ; 
        *ptres = libGAP_INTOBJ_INT(k);
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L);
      }
    }
  } else {
    ptf4 = libGAP_ADDR_TRANS4(f);
    deg = libGAP_DEG_TRANS4(f);

    /* loop over the entries of the tuple                              */
    isint=1;
    for ( i =libGAP_LEN_LIST(tup) ; 1 <= i; i--, pttup--, ptres-- ) {
      if ( libGAP_TNUM_OBJ( *pttup ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *pttup ) ) {
        k = libGAP_INT_INTOBJ( *pttup );
        if ( k <= deg ) k = ptf4[k-1] + 1 ; 
        *ptres = libGAP_INTOBJ_INT(k);
      } else {/* this case cannot occur since I think POW is not defined */
        libGAP_ErrorQuit("not yet implemented!", 0L, 0L);
      }
    }
  }
  if(isint){
    libGAP_RetypeBag( res, libGAP_IS_MUTABLE_PLIST(tup) ? libGAP_T_PLIST_CYC_SSORT :
     libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE );
  }
  return res;
}

libGAP_Obj libGAP_FuncOnPosIntSetsTrans (libGAP_Obj self, libGAP_Obj set, libGAP_Obj f, libGAP_Obj n){
  libGAP_UInt2  *ptf2;
  libGAP_UInt4  *ptf4;
  libGAP_UInt   deg;
  libGAP_Obj    *ptset, *ptres, tmp, res;
  libGAP_UInt   i, k, h, len;

  if(libGAP_LEN_LIST(set)==0) return set;

  if(libGAP_LEN_LIST(set)==1&&libGAP_INT_INTOBJ(libGAP_ELM_LIST(set, 1))==0){
    return libGAP_FuncIMAGE_SET_TRANS_INT(self, f, n);
  }  

  libGAP_PLAIN_LIST(set);
  res=libGAP_NEW_PLIST(libGAP_IS_MUTABLE_PLIST(set)?libGAP_T_PLIST:libGAP_T_PLIST+libGAP_IMMUTABLE,libGAP_LEN_LIST(set));
  libGAP_ADDR_OBJ(res)[0]=libGAP_ADDR_OBJ(set)[0]; 

  /* get the pointer                                                 */
  ptset = libGAP_ADDR_OBJ(set) + libGAP_LEN_LIST(set);
  ptres = libGAP_ADDR_OBJ(res) + libGAP_LEN_LIST(set);
  
  if(libGAP_TNUM_OBJ(f)==libGAP_T_TRANS2){   
    ptf2 = libGAP_ADDR_TRANS2(f);
    deg = libGAP_DEG_TRANS2(f);
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset--, ptres-- ) {
      k = libGAP_INT_INTOBJ( *ptset );
      if ( k <= deg ) k = ptf2[k-1] + 1 ; 
      *ptres = libGAP_INTOBJ_INT(k);
    }
  } else {
    ptf4 = libGAP_ADDR_TRANS4(f);
    deg = libGAP_DEG_TRANS4(f);
    for ( i =libGAP_LEN_LIST(set) ; 1 <= i; i--, ptset--, ptres-- ) {
      k = libGAP_INT_INTOBJ( *ptset );
      if ( k <= deg ) k = ptf4[k-1] + 1 ; 
      *ptres = libGAP_INTOBJ_INT(k);
    }
  }
  /* sort the result */
  len=libGAP_LEN_LIST(res);
  h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1;
  while ( 0 < h ) {
    for ( i = h+1; i <= len; i++ ) {
      tmp = libGAP_ADDR_OBJ(res)[i];  k = i;
      while ( h < k && ((libGAP_Int)tmp < (libGAP_Int)(libGAP_ADDR_OBJ(res)[k-h])) ) {
        libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h];
        k -= h;
      }
      libGAP_ADDR_OBJ(res)[k] = tmp;
    }
    h = h / 3;
  }

  /* remove duplicates */
  if ( 0 < len ) {
    tmp = libGAP_ADDR_OBJ(res)[1];  k = 1;
    for ( i = 2; i <= len; i++ ) {
      if ( ! libGAP_EQ( tmp, libGAP_ADDR_OBJ(res)[i] ) ) {
        k++;
        tmp = libGAP_ADDR_OBJ(res)[i];
        libGAP_ADDR_OBJ(res)[k] = tmp;
      }
    }
    if ( k < len ) {
      libGAP_ResizeBag( res, (k+1)*sizeof(libGAP_Obj) );
      libGAP_SET_LEN_PLIST(res, k);
    }
  }

  /* retype if we only have integers */
  libGAP_RetypeBag( res, libGAP_IS_MUTABLE_PLIST(set) ? libGAP_T_PLIST_CYC_SSORT :
   libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE ); 

  return res;
}

/******************************************************************************/
/******************************************************************************/

/* other internal things */

/* so that kernel and image set are preserved during garbage collection */

void libGAP_MarkTransSubBags( libGAP_Obj f ){
  if(libGAP_IMG_TRANS(f)!=NULL){
    libGAP_MARK_BAG(libGAP_IMG_TRANS(f));
    libGAP_MARK_BAG(libGAP_KER_TRANS(f));
  }
  if(libGAP_EXT_TRANS(f)!=NULL){
    libGAP_MARK_BAG(libGAP_EXT_TRANS(f));
  }
}

/* Save and load */
void libGAP_SaveTrans2( libGAP_Obj f){
  libGAP_UInt2   *ptr;
  libGAP_UInt    len, i;
  ptr=libGAP_ADDR_TRANS2(f); /* save the image list */
  len=libGAP_DEG_TRANS2(f);
  for (i = 0; i < len; i++) libGAP_SaveUInt2(*ptr++);
}

void libGAP_LoadTrans2( libGAP_Obj f){
  libGAP_UInt2   *ptr;
  libGAP_UInt    len, i;
  len=libGAP_DEG_TRANS2(f);
  ptr=libGAP_ADDR_TRANS2(f);
  for (i = 0; i < len; i++) *ptr++=libGAP_LoadUInt2();
}

void libGAP_SaveTrans4( libGAP_Obj f){
  libGAP_UInt4   *ptr;
  libGAP_UInt    len, i;
  ptr=libGAP_ADDR_TRANS4(f); /* save the image list */
  len=libGAP_DEG_TRANS4(f);
  for (i = 0; i < len; i++) libGAP_SaveUInt4(*ptr++);
}

void libGAP_LoadTrans4( libGAP_Obj f){
  libGAP_UInt4   *ptr;
  libGAP_UInt    len, i;
  len=libGAP_DEG_TRANS4(f);
  ptr=libGAP_ADDR_TRANS4(f);
  for (i = 0; i < len; i++) *ptr++=libGAP_LoadUInt4();
}

libGAP_Obj libGAP_TYPE_TRANS2;

libGAP_Obj libGAP_TypeTrans2(libGAP_Obj f){
  return libGAP_TYPE_TRANS2;
}

libGAP_Obj libGAP_TYPE_TRANS4;

libGAP_Obj libGAP_TypeTrans4(libGAP_Obj f){
  return libGAP_TYPE_TRANS4;
}

libGAP_Obj libGAP_IsTransFilt;

libGAP_Obj libGAP_IsTransHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 val )
{
    /* return 'true' if <val> is a transformation and 'false' otherwise       */
    switch ( libGAP_TNUM_OBJ(val) ) {

        case libGAP_T_TRANS2:
        case libGAP_T_TRANS4:
            return libGAP_True;

        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            return libGAP_DoFilter( self, val );

        default:
            return libGAP_False;
    }
}

/*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * */

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_TRANS", "obj", &libGAP_IsTransFilt,
      libGAP_IsTransHandler, "src/trans.c:IS_TRANS" },

    { 0 }

};

/******************************************************************************
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

  { "HAS_KER_TRANS", 1, "f",
     libGAP_FuncHAS_KER_TRANS,
    "src/TRANS.c:FuncHAS_KER_TRANS" },

  { "HAS_IMG_TRANS", 1, "f",
     libGAP_FuncHAS_IMG_TRANS,
    "src/TRANS.c:FuncHAS_IMG_TRANS" },

  { "INT_DEG_TRANS", 1, "f",
     libGAP_FuncINT_DEG_TRANS,
    "src/TRANS.c:FuncINT_DEG_TRANS" },
  
  { "NUMB_TRANS_INT", 2, "f, m",
     libGAP_FuncNUMB_TRANS_INT,
    "src/trans.c:FuncNUMB_TRANS_INT" },

  { "TRANS_NUMB_INT", 2, "f, m",
     libGAP_FuncTRANS_NUMB_INT,
    "src/trans.c:FuncTRANS_NUMB_INT" },

  { "TransformationNC", 1, "list",
     libGAP_FuncTransformationNC,
    "src/trans.c:FuncTransformationNC" },

  { "TransformationListListNC", 2, "src, ran",
     libGAP_FuncTransformationListListNC,
    "src/trans.c:FuncTransformationListListNC" },

  { "DegreeOfTransformation", 1, "f",
     libGAP_FuncDegreeOfTransformation,
    "src/trans.c:FuncDegreeOfTransformation" },

  { "HASH_FUNC_FOR_TRANS", 2, "f, data",
     libGAP_FuncHASH_FUNC_FOR_TRANS,
    "src/trans.c:FuncHASH_FUNC_FOR_TRANS" },
  
  { "RANK_TRANS", 1, "f",
     libGAP_FuncRANK_TRANS,
    "src/trans.c:FuncRANK_TRANS" },

  { "RANK_TRANS_INT", 2, "f, n",
     libGAP_FuncRANK_TRANS_INT,
    "src/trans.c:FuncRANK_TRANS_INT" },

  { "RANK_TRANS_LIST", 2, "f, list",
     libGAP_FuncRANK_TRANS_LIST,
    "src/trans.c:FuncRANK_TRANS_LIST" },

  { "LARGEST_MOVED_PT_TRANS", 1, "f",
     libGAP_FuncLARGEST_MOVED_PT_TRANS,
    "src/trans.c:FuncLARGEST_MOVED_PT_TRANS" },

  { "LARGEST_IMAGE_PT", 1, "f",
     libGAP_FuncLARGEST_IMAGE_PT,
    "src/trans.c:FuncLARGEST_IMAGE_PT" },

  { "SMALLEST_MOVED_PT_TRANS", 1, "f",
     libGAP_FuncSMALLEST_MOVED_PT_TRANS,
    "src/trans.c:FuncSMALLEST_MOVED_PT_TRANS" },

  { "SMALLEST_IMAGE_PT", 1, "f",
     libGAP_FuncSMALLEST_IMAGE_PT,
    "src/trans.c:FuncSMALLEST_IMAGE_PT" },

  { "NR_MOVED_PTS_TRANS", 1, "f",
     libGAP_FuncNR_MOVED_PTS_TRANS,
    "src/trans.c:FuncNR_MOVED_PTS_TRANS" },

  { "MOVED_PTS_TRANS", 1, "f",
     libGAP_FuncMOVED_PTS_TRANS,
    "src/trans.c:FuncMOVED_PTS_TRANS" },

  { "IMAGE_TRANS", 2, "f, n",
     libGAP_FuncIMAGE_TRANS,
    "src/trans.c:FuncIMAGE_TRANS" },

  { "FLAT_KERNEL_TRANS", 1, "f",
     libGAP_FuncFLAT_KERNEL_TRANS,
    "src/trans.c:FuncFLAT_KERNEL_TRANS" },

  { "FLAT_KERNEL_TRANS_INT", 2, "f, n",
     libGAP_FuncFLAT_KERNEL_TRANS_INT,
    "src/trans.c:FuncFLAT_KERNEL_TRANS_INT" },

  { "IMAGE_SET_TRANS", 1, "f",
     libGAP_FuncIMAGE_SET_TRANS,
    "src/trans.c:FuncIMAGE_SET_TRANS" },

  { "IMAGE_SET_TRANS_INT", 2, "f, n",
     libGAP_FuncIMAGE_SET_TRANS_INT,
    "src/trans.c:FuncIMAGE_SET_TRANS_INT" },

  { "KERNEL_TRANS", 2, "f, n",
     libGAP_FuncKERNEL_TRANS,
    "src/trans.c:FuncKERNEL_TRANS" },

  { "PREIMAGES_TRANS_INT", 2, "f, pt",
     libGAP_FuncPREIMAGES_TRANS_INT,
    "src/trans.c:FuncPREIMAGES_TRANS_INT" },

  { "AS_TRANS_PERM", 1, "f",
     libGAP_FuncAS_TRANS_PERM,
    "src/trans.c:FuncAS_TRANS_PERM" },

  { "AS_TRANS_PERM_INT", 2, "f, n",
     libGAP_FuncAS_TRANS_PERM_INT,
    "src/trans.c:FuncAS_TRANS_PERM_INT" },

  { "AS_PERM_TRANS", 1, "f",
     libGAP_FuncAS_PERM_TRANS,
    "src/trans.c:FuncAS_PERM_TRANS" },

  { "PERM_IMG_TRANS", 1, "f",
     libGAP_FuncPERM_IMG_TRANS,
    "src/trans.c:FuncPERM_IMG_TRANS" },

  { "RESTRICTED_TRANS", 2, "f, list",
     libGAP_FuncRESTRICTED_TRANS,
    "src/trans.c:FuncRESTRICTED_TRANS" },

  { "AS_TRANS_TRANS", 2, "f, m",
     libGAP_FuncAS_TRANS_TRANS,
    "src/trans.c:FuncAS_TRANS_TRANS" },

  { "TRIM_TRANS", 2, "f, m",
     libGAP_FuncTRIM_TRANS,
    "src/trans.c:FuncTRIM_TRANS" },

  { "IS_INJECTIVE_LIST_TRANS", 2, "t, l",
     libGAP_FuncIS_INJECTIVE_LIST_TRANS,
    "src/trans.c:FuncIS_INJECTIVE_LIST_TRANS" },

  { "PERM_LEFT_QUO_TRANS_NC", 2, "f, g",
     libGAP_FuncPERM_LEFT_QUO_TRANS_NC,
    "src/trans.c:FuncPERM_LEFT_QUO_TRANS_NC" },

  { "TRANS_IMG_KER_NC", 2, "img, ker",
     libGAP_FuncTRANS_IMG_KER_NC,
    "src/trans.c:FuncTRANS_IMG_KER_NC" },

  { "IDEM_IMG_KER_NC", 2, "img, ker",
     libGAP_FuncIDEM_IMG_KER_NC,
    "src/trans.c:FuncIDEM_IMG_KER_NC" },

  { "INV_TRANS", 1, "f",
     libGAP_FuncINV_TRANS,
    "src/trans.c:FuncINV_TRANS" },

  { "INV_LIST_TRANS", 2, "list, f",
     libGAP_FuncINV_LIST_TRANS,
    "src/trans.c:FuncINV_LIST_TRANS" },

  { "TRANS_IMG_CONJ", 2, "f,g",
     libGAP_FuncTRANS_IMG_CONJ,
    "src/trans.c:FuncTRANS_IMG_CONJ" },

  { "INDEX_PERIOD_TRANS", 1, "f",
     libGAP_FuncINDEX_PERIOD_TRANS,
    "src/trans.c:FuncINDEX_PERIOD_TRANS" },

  { "SMALLEST_IDEM_POW_TRANS", 1, "f",
     libGAP_FuncSMALLEST_IDEM_POW_TRANS,
    "src/trans.c:FuncSMALLEST_IDEM_POW_TRANS" },

  { "POW_KER_PERM", 2, "ker, f",
     libGAP_FuncPOW_KER_PERM,
    "src/trans.c:FuncPOW_KER_PERM" },

  { "ON_KERNEL_ANTI_ACTION", 3, "ker, f, n",
     libGAP_FuncON_KERNEL_ANTI_ACTION,
    "src/trans.c:FuncON_KERNEL_ANTI_ACTION" },

  { "INV_KER_TRANS", 2, "ker, f",
     libGAP_FuncINV_KER_TRANS,
    "src/trans.c:FuncINV_KER_TRANS" },

  { "IS_IDEM_TRANS", 1, "f",
    libGAP_FuncIS_IDEM_TRANS,
    "src/trans.c:FuncIS_IDEM_TRANS" },

  { "IS_ID_TRANS", 1, "f",
    libGAP_FuncIS_ID_TRANS,
    "src/trans.c:FuncIS_ID_TRANS" },
  
  { "COMPONENT_REPS_TRANS", 1, "f",
    libGAP_FuncCOMPONENT_REPS_TRANS,
    "src/trans.c:FuncCOMPONENT_REPS_TRANS" },
  
  { "NR_COMPONENTS_TRANS", 1, "f",
    libGAP_FuncNR_COMPONENTS_TRANS,
    "src/trans.c:FuncNR_COMPONENTS_TRANS" },

  { "COMPONENTS_TRANS", 1, "f",
    libGAP_FuncCOMPONENTS_TRANS,
    "src/trans.c:FuncCOMPONENTS_TRANS" },

  { "COMPONENT_TRANS_INT", 2, "f, pt",
    libGAP_FuncCOMPONENT_TRANS_INT,
    "src/trans.c:FuncCOMPONENT_TRANS_INT" },

  { "CYCLE_TRANS_INT", 2, "f, pt",
    libGAP_FuncCYCLE_TRANS_INT,
    "src/trans.c:FuncCYCLE_TRANS_INT" },

  { "CYCLES_TRANS_LIST", 2, "f, pt",
    libGAP_FuncCYCLES_TRANS_LIST,
    "src/trans.c:FuncCYCLES_TRANS_LIST" },

  { "LEFT_ONE_TRANS", 1, "f",
    libGAP_FuncLEFT_ONE_TRANS,
    "src/trans.c:FuncLEFT_ONE_TRANS" },

  { "RIGHT_ONE_TRANS", 1, "f",
    libGAP_FuncRIGHT_ONE_TRANS,
    "src/trans.c:FuncRIGHT_ONE_TRANS" },
  
  { "OnPosIntSetsTrans", 3, "set, f, n", 
    libGAP_FuncOnPosIntSetsTrans, 
    "src/trans.c:FuncOnPosIntSetsTrans" },

  { "IsCommutingTransformation", 2, "f, g", 
    libGAP_FuncIsCommutingTransformation, 
    "src/trans.c:FuncIsCommutingTransformation" },
  
  { 0 }

};
/******************************************************************************
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel ( libGAP_StructInitInfo *libGAP_module )
{

    /* install the marking function                                        */
    libGAP_InfoBags[ libGAP_T_TRANS2 ].name = "transformation (small)";
    libGAP_InfoBags[ libGAP_T_TRANS4 ].name = "transformation (large)";
    libGAP_InitMarkFuncBags( libGAP_T_TRANS2, libGAP_MarkTransSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_TRANS4, libGAP_MarkTransSubBags );
    
    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_TRANS2", &libGAP_TYPE_TRANS2 );
    libGAP_ImportGVarFromLibrary( "TYPE_TRANS4", &libGAP_TYPE_TRANS4 );

    libGAP_TypeObjFuncs[ libGAP_T_TRANS2 ] = libGAP_TypeTrans2;
    libGAP_TypeObjFuncs[ libGAP_T_TRANS4 ] = libGAP_TypeTrans4;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* make the buffer bag                                                 */
    libGAP_InitGlobalBag( &libGAP_TmpTrans, "src/trans.c:TmpTrans" );
    
    // make the identity trans
    libGAP_InitGlobalBag( &libGAP_IdentityTrans, "src/trans.c:IdentityTrans" );
    
    /* install the saving functions */
    libGAP_SaveObjFuncs[ libGAP_T_TRANS2 ] = libGAP_SaveTrans2;
    libGAP_LoadObjFuncs[ libGAP_T_TRANS2 ] = libGAP_LoadTrans2; 
    libGAP_SaveObjFuncs[ libGAP_T_TRANS4 ] = libGAP_SaveTrans4;
    libGAP_LoadObjFuncs[ libGAP_T_TRANS4 ] = libGAP_LoadTrans4; 

    /* install the comparison methods                                      */
    libGAP_EqFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_TRANS2  ] = libGAP_EqTrans22;    
    libGAP_EqFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_TRANS4  ] = libGAP_EqTrans24;    
    libGAP_EqFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_TRANS2  ] = libGAP_EqTrans42;    
    libGAP_EqFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_TRANS4  ] = libGAP_EqTrans44;    
    libGAP_LtFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_TRANS2  ] = libGAP_LtTrans22;
    libGAP_LtFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_TRANS4  ] = libGAP_LtTrans24;
    libGAP_LtFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_TRANS2  ] = libGAP_LtTrans42;
    libGAP_LtFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_TRANS4  ] = libGAP_LtTrans44;
    
    /* install the binary operations */
    libGAP_ProdFuncs [ libGAP_T_TRANS2  ][ libGAP_T_TRANS2 ] = libGAP_ProdTrans22;
    libGAP_ProdFuncs [ libGAP_T_TRANS4  ][ libGAP_T_TRANS4 ] = libGAP_ProdTrans44;
    libGAP_ProdFuncs [ libGAP_T_TRANS2  ][ libGAP_T_TRANS4 ] = libGAP_ProdTrans24;
    libGAP_ProdFuncs [ libGAP_T_TRANS4  ][ libGAP_T_TRANS2 ] = libGAP_ProdTrans42;
    libGAP_ProdFuncs [ libGAP_T_TRANS2  ][ libGAP_T_PERM2 ] = libGAP_ProdTrans2Perm2;
    libGAP_ProdFuncs [ libGAP_T_TRANS2  ][ libGAP_T_PERM4 ] = libGAP_ProdTrans2Perm4;
    libGAP_ProdFuncs [ libGAP_T_TRANS4  ][ libGAP_T_PERM2 ] = libGAP_ProdTrans4Perm2;
    libGAP_ProdFuncs [ libGAP_T_TRANS4  ][ libGAP_T_PERM4 ] = libGAP_ProdTrans4Perm4;
    libGAP_ProdFuncs [ libGAP_T_PERM2  ][ libGAP_T_TRANS2 ] = libGAP_ProdPerm2Trans2;
    libGAP_ProdFuncs [ libGAP_T_PERM4  ][ libGAP_T_TRANS2 ] = libGAP_ProdPerm4Trans2;
    libGAP_ProdFuncs [ libGAP_T_PERM2  ][ libGAP_T_TRANS4 ] = libGAP_ProdPerm2Trans4;
    libGAP_ProdFuncs [ libGAP_T_PERM4  ][ libGAP_T_TRANS4 ] = libGAP_ProdPerm4Trans4;
    libGAP_PowFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_PERM2 ] = libGAP_PowTrans2Perm2;    
    libGAP_PowFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_PERM4 ] = libGAP_PowTrans2Perm4;    
    libGAP_PowFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_PERM2 ] = libGAP_PowTrans4Perm2;    
    libGAP_PowFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_PERM4 ] = libGAP_PowTrans4Perm4;    
    libGAP_QuoFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_PERM2 ] = libGAP_QuoTrans2Perm2;
    libGAP_QuoFuncs  [ libGAP_T_TRANS2  ][ libGAP_T_PERM4 ] = libGAP_QuoTrans2Perm4;
    libGAP_QuoFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_PERM2 ] = libGAP_QuoTrans4Perm2;
    libGAP_QuoFuncs  [ libGAP_T_TRANS4  ][ libGAP_T_PERM4 ] = libGAP_QuoTrans4Perm4;
    libGAP_LQuoFuncs [ libGAP_T_PERM2  ][ libGAP_T_TRANS2 ] = libGAP_LQuoPerm2Trans2;
    libGAP_LQuoFuncs [ libGAP_T_PERM4  ][ libGAP_T_TRANS2 ] = libGAP_LQuoPerm4Trans2;
    libGAP_LQuoFuncs [ libGAP_T_PERM2  ][ libGAP_T_TRANS4 ] = libGAP_LQuoPerm2Trans4;
    libGAP_LQuoFuncs [ libGAP_T_PERM4  ][ libGAP_T_TRANS4 ] = libGAP_LQuoPerm4Trans4;
    libGAP_PowFuncs  [ libGAP_T_INT    ][ libGAP_T_TRANS2 ] = libGAP_PowIntTrans2;
    libGAP_PowFuncs  [ libGAP_T_INT    ][ libGAP_T_TRANS4 ] = libGAP_PowIntTrans4;
  
    /* install the 'ONE' function for transformations */
    libGAP_OneFuncs    [ libGAP_T_TRANS2 ] = libGAP_OneTrans;
    libGAP_OneMutFuncs [ libGAP_T_TRANS2 ] = libGAP_OneTrans;
    libGAP_OneFuncs    [ libGAP_T_TRANS4 ] = libGAP_OneTrans;
    libGAP_OneMutFuncs [ libGAP_T_TRANS4 ] = libGAP_OneTrans;

    /* return success                                                      */
    return 0;
}

/******************************************************************************
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary ( libGAP_StructInitInfo *libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_TmpTrans = libGAP_NEW_TRANS4(1000);
    libGAP_IdentityTrans = libGAP_NEW_TRANS2(0);

    /* return success                                                      */
    return 0;
}

/****************************************************************************
**
*F  InitInfoTrans()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "trans",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoTrans ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}

