/****************************************************************************
**
*W  gasman.c                    GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains  the functions of  Gasman,  the  GAP  storage manager.
**
**  {\Gasman} is a storage manager for applications written in C.  That means
**  that  an application  using {\Gasman}  requests  blocks  of  storage from
**  {\Gasman}.   Those   blocks of  storage are    called  *bags*.  Then  the
**  application writes data into and reads data from the bags.  Finally a bag
**  is no longer needed  and the application  simply forgets it.  We say that
**  such a bag that is no longer needed is *dead*.  {\Gasman} cares about the
**  allocation of bags and deallocation of  dead bags.  Thus these operations
**  are  transparent    to  the application,  enabling   the    programmer to
**  concentrate on algorithms instead of caring  about storage allocation and
**  deallocation.
**
**  {\Gasman} implements an automatic, cooperating, compacting, generational,
**  conservative storage management
**
**  *Automatic* means that the application only allocates  bags.  It need not
**  explicitly  deallocate them.    {\Gasman} automatically determines  which
**  bags are  dead and deallocates them.  This   is done by a  process called
**  *garbage  collection*.   Garbage refers to   the bags that  are dead, and
**  collection refers to the process of deallocating them.
**
**  *Cooperating* means  that the application  must cooperate with {\Gasman},
**  that is it must follow two rules.  One rule is that  it must not remember
**  the addresses of the data area of a bag for  too long.  The other rule is
**  that it must inform {\Gasman} when it has changed a bag.
**
**  *Compacting* means that after a garbage collection {\Gasman} compacts the
**  bags  that are   still  live,  so  that the  storage  made  available  by
**  deallocating the dead  bags  becomes one  large  contiguous  block.  This
**  helps to avoid *fragmentation* of the free storage.  The downside is that
**  the address of  the data  area  of a  bag  may  change during  a  garbage
**  collection,  which is the  reason why  the  application must not remember
**  this address for too long,  i.e., must not keep  pointers to or into  the
**  data area of a bag over a garbage collection.
**
**  *Generational*  means that {\Gasman}  distinguishes between old and young
**  bags.  Old bags have been allocated some time ago, i.e., they survived at
**  least one garbage collection.  During a garbage collection {\Gasman} will
**  first find and  deallocate the dead young  bags.   Only if that  does not
**  produce enough free storage, {\Gasman} will find  and deallocate the dead
**  old  bags.  The idea behind this  is that usually most  bags  have a very
**  short life, so  that  they will die young.    The downside is that   this
**  requires {\Gasman} to quickly  find  the young  bags that are  referenced
**  from   old bags, which  is  the  reason  why  an  application must inform
**  {\Gasman} when it has changed a bag.
**
**  *Conservative* means that there are  situations in which {\Gasman} cannot
**  decide with  absolute certainty whether  a bag is   still live or already
**  dead.  In these situations {\Gasman} has to  assume that the bag is still
**  live, and may thus keep a bag longer than it is necessary.
**
**  What follows describes the reasons for this  design, and at the same time
**  the assumptions that were  made about the application.   This is given so
**  you can make an educated guess as to whether  {\Gasman} is an appropriate
**  storage manager for your application.
**
**  {\Gasman} is automatic, because  this makes it  easier to use in large or
**  complex applications.  Namely in with a non-automatic storage manager the
**  application must  decide when  to  deallocate  a  bag.  This  requires in
**  general global knowledge, i.e., it is not sufficient  to know whether the
**  current function may still need  the bag, one also  needs to know whether
**  any  other  function may  still   need the   bag.  With growing   size or
**  complexity of the application it gets harder to obtain this knowledge.
**
**  {\Gasman} is cooperating,  because this is  a requirement  for compaction
**  and generations (at least without cooperation, compaction and generations
**  are very  difficult).  As described  below, the  former is  important for
**  storage efficiency,   the  latter for  time efficiency.     Note that the
**  cooperation requires only local knowledge, i.e., whether or not a certain
**  function of the application follows the two  rules can be decided by just
**  looking   at the function  without any  knowledge about the   rest of the
**  application.
**
**  {\Gasman} is  compacting, because this allows the  efficient usage of the
**  available storage by applications where the ratio between the size of the
**  smallest   and the largest bag is   large.   Namely with a non-compacting
**  storage manager,  a  part  of  the free  storage   may become unavailable
**  because it is fragmented into  many small pieces,  each  of which is  too
**  small to serve an allocation.
**
**  {\Gasman} is generational,  because this  makes it  very much faster,  at
**  least  for those  applications where   most bags  will indeed  die young.
**  Namely a non-generational storage manager must test  for each bag whether
**  or  not it  is still live  during  each garbage collection.  However with
**  many applications  the probability   that  an  old bag,  i.e.,  one  that
**  survived at least one  garbage  collection, will  also survive  the  next
**  garbage  collection  is  high.   A  generational  storage manager  simply
**  assumes that each old bag is still  live during most garbage collections.
**  Thereby it avoids  the expensive tests for most  bags during most garbage
**  collections.
**
**  {\Gasman}  is conservative, because  for most applications  only few bags
**  are incorrectly  assumed to be still live  and the additional cooperation
**  required   to make  {\Gasman} (more)     precise   would slow down    the
**  application.  Note that the problem appears since the C compiler provides
**  not enough information to distinguish between true references to bags and
**  other values  that just happen to look  like  references.  Thus {\Gasman}
**  has to assume that everything that could be interpreted as a reference to
**  a bag  is  indeed such  a reference, and  that  this bag is  still  live.
**  Therefore  some bags  may  be  kept by  {\Gasman}, even   though they are
**  already dead.
*/
#include        <string.h>
#include        "system.h"              /* Ints, UInts                     */



#include        "gasman.h"              /* garbage collector               */

#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "libgap_internal.h"     /* gasman callback                 */

/****************************************************************************
**

*F  WORDS_BAG( <size> ) . . . . . . . . . . words used by a bag of given size
**
**  The structure of a bag is a follows{\:}
**
**    <identifier>
**      __/
**     /
**    V
**    +---------+
**    |<masterp>|
**    +---------+
**          \____________
**                       \
**                        V
**    +---------+---------+--------------------------------------------+----+
**    |<sz>.<tp>|  <link> |         .         .         .         .    | pad|
**    +---------+---------+--------------------------------------------+----+
**
**  A bag consists of a masterpointer, and a body.
**
**  The *masterpointer* is a pointer  to the data  area of the bag.  During a
**  garbage collection  the masterpointer is the  only active  pointer to the
**  data area of the bag, because of the rule that no pointers to or into the
**  data  area of a  bag may be remembered  over calls  to functions that may
**  cause a garbage  collection.  It is the job  of the garbage collection to
**  update the masterpointer of a bag when it moves the bag.
**
**  The *identifier*  of  the  bag is a   pointer  to (the  address   of) the
**  masterpointer  of  the bag.   Thus   'PTR_BAG(<bag>)' is simply '\*<bag>'
**  plus a cast.
**
**  The *body* of a  bag consists of  the size-type word,  the link word, the
**  data area, and the padding.
**
**  The *size-type word* contains the size of the bag in the upper  (at least
**  24) bits, and the type (abbreviated as <tp> in the  above picture) in the
**  lower 8  bits.  Thus 'SIZE_BAG'   simply extracts the size-type  word and
**  shifts it 8 bits to the right, and 'TNUM_BAG' extracts the size-type word
**  and masks out everything except the lower 8 bits.
**
**  The  *link word* usually   contains the identifier of  the  bag,  i.e., a
**  pointer to the masterpointer of the bag.  Thus the garbage collection can
**  find  the masterpointer of a  bag through the link  word if  it knows the
**  address of the  data area of the bag.    The link word   is also used  by
**  {\Gasman} to  keep   bags  on  two linked  lists  (see  "ChangedBags" and
**  "MarkedBags").
**
**  The *data area* of a  bag is the area  that  contains the data stored  by
**  the application in this bag.
**
**  The *padding* consists  of up to 'sizeof(Bag)-1' bytes  and pads the body
**  so that the size of a  body is always  a multiple of 'sizeof(Bag)'.  This
**  is to ensure that bags are always aligned.  The macro 'WORDS_BAG(<size>)'
**  returns the number  of words occupied  by the data  area and padding of a
**  bag of size <size>.
**
**  A body in the workspace  whose  size-type word contains  the value 255 in
**  the lower 8 bits is the remainder of a 'ResizeBag'.  That  is it consists
**  either of the unused words after a bag has been shrunk or of the old body
**  of the bag after the contents of the body have  been copied elsewhere for
**  an extension.  The upper (at least 24) bits of the first word contain the
**  number of bytes in this area excluding the first  word itself.  Note that
**  such a body   has no link   word,  because  such  a  remainder  does  not
**  correspond to a bag (see "Implementation of ResizeBag").
**
**  A masterpointer with a value  congruent to 1  mod 4 is   the relic of  an
**  object  that was  weakly but not   strongly  marked in  a recent  garbage
**  collection.   These persist until  after the next full garbage collection
**  by which time all references to them should have been removed.
** 
*/

#ifndef C_PLUS_PLUS_BAGS
#define libGAP_SIZE_MPTR_BAGS  1
#endif
#ifdef  C_PLUS_PLUS_BAGS
#define libGAP_SIZE_MPTR_BAGS  2
#endif
#define libGAP_WORDS_BAG(size) (((size) + (sizeof(libGAP_Bag)-1)) / sizeof(libGAP_Bag))

#ifdef libGAP_USE_NEWSHAPE
#define libGAP_HEADER_SIZE 2
#else
#define libGAP_HEADER_SIZE 3
#endif

/* This could be 65536, but would waste memory in various tables */

#define libGAP_NTYPES 256

/****************************************************************************
**
*V  MptrBags  . . . . . . . . . . . . . . beginning of the masterpointer area
*V  OldBags . . . . . . . . . . . . . . . . .  beginning of the old bags area
*V  YoungBags . . . . . . . . . . . . . . .  beginning of the young bags area
*V  AllocBags . . . . . . . . . . . . . . .  beginning of the allocation area
*V  AllocSizeBags . . . . . . . . . . . . . . . . size of the allocation area
*V  StopBags  . . . . . . . . . . . . . . . beginning of the unavailable area
*V  EndBags . . . . . . . . . . . . . . . . . . . . . .  end of the workspace
**
**  {\Gasman} manages one large block of storage called the *workspace*.  The
**  layout of the workspace is as follows{\:}
**
**  +-------------+-----------------+------------+------------+-------------+
**  |masterpointer|    old bags     | young bags | allocation | unavailable |
**  |    area     |      area       |    area    |    area    |    area     |
**  +-------------+-----------------+------------+------------+-------------+
**  ^             ^                 ^            ^            ^             ^
**  MptrBags    OldBags         YoungBags    AllocBags     StopBags   EndBags
**
**  The *masterpointer area*  contains  all the masterpointers  of  the bags.
**  'MptrBags' points to the beginning of this area and 'OldBags' to the end.
**
**  The *old bags area* contains the bodies of all the  bags that survived at
**  least one  garbage collection.  This area is  only  scanned for dead bags
**  during a full garbage collection.  'OldBags'  points to the  beginning of
**  this area and 'YoungBags' to the end.
**
**  The *young bags area* contains the bodies of all  the bags that have been
**  allocated since the  last garbage collection.  This  area is scanned  for
**  dead  bags during  each garbage  collection.  'YoungBags'  points  to the
**  beginning of this area and 'AllocBags' to the end.
**
**  The *allocation area* is the storage  that is available for allocation of
**  new bags.  When a new bag is allocated the storage for  the body is taken
**  from  the beginning of   this area,  and  this  area  is  correspondingly
**  reduced.   If  the body does not   fit in the  allocation  area a garbage
**  collection is  performed.  'AllocBags' points   to the beginning of  this
**  area and 'StopBags' to the end.
**
**  The *unavailable  area* is  the free  storage that  is not  available for
**  allocation.   'StopBags'  points  to  the  beginning  of  this  area  and
**  'EndBags' to the end.
**
**  If <cache-size>  (see "InitBags") was 0,  'CollectBags' makes all of  the
**  free storage available for allocations by setting 'StopBags' to 'EndBags'
**  after garbage collections.   In   this case garbage  collections are only
**  performed when no  free storage   is left.  If <cache-size> was  nonzero,
**  'CollectBags' makes 'AllocSizeBags' bytes available by setting 'StopBags'
**  to  'AllocBags   + 2+WORDS_BAG(<size>) +  WORDS_BAG(AllocSizeBags)' after
**  garbage  collections, where <size>   is the  size  of the bag 'NewBag' is
**  currently allocating.  'AllocSizeBags'  is  usually <cache-size>,  but is
**  increased if only very few large bags have been  allocated since the last
**  garbage collection and decreased  again  if sufficiently many  bags  have
**  been allocated since the  last  garbage collection.  The  idea is to keep
**  the allocation area small enough so that it fits in the processor cache.
**
**  Note that  the  borders between the areas are not static.  In  particular
**  each allocation increases the size of the young bags area and reduces the
**  size of the  allocation area.  On the other hand each garbage  collection
**  empties the young bags area.
*/
libGAP_Bag *                   libGAP_MptrBags;
libGAP_Bag *                   libGAP_OldBags;
libGAP_Bag *                   libGAP_YoungBags;
libGAP_Bag *                   libGAP_AllocBags;
libGAP_UInt                    libGAP_AllocSizeBags;
libGAP_Bag *                   libGAP_StopBags;
libGAP_Bag *                   libGAP_EndBags;


/* These macros, are (a) for more readable code, but more importantly
   (b) to ensure that unsigned subtracts and divides are used (since
   we know the ordering of the pointers. This is needed on > 2GB
   workspaces on 32 but systems. The Size****Area functions return an
   answer in units of a word (ie sizeof(UInt) bytes), which should
   therefore be small enough not to cause problems. */

#define libGAP_SpaceBetweenPointers(a,b) (((libGAP_UInt)((libGAP_UInt)(a) - (libGAP_UInt)(b)))/sizeof(libGAP_Bag))

#define libGAP_SizeMptrsArea libGAP_SpaceBetweenPointers(libGAP_OldBags, libGAP_MptrBags)
#define libGAP_SizeOldBagsArea libGAP_SpaceBetweenPointers(libGAP_YoungBags,libGAP_OldBags)
#define libGAP_SizeYoungBagsArea libGAP_SpaceBetweenPointers(libGAP_AllocBags, libGAP_YoungBags)
#define libGAP_SizeAllocationArea libGAP_SpaceBetweenPointers(libGAP_StopBags, libGAP_AllocBags)
#define libGAP_SizeUnavailableArea libGAP_SpaceBetweenPointers(libGAP_EndBags, libGAP_StopBags)

#define libGAP_SizeAllBagsArea libGAP_SpaceBetweenPointers(libGAP_AllocBags, libGAP_OldBags)
#define libGAP_SizeWorkspace libGAP_SpaceBetweenPointers(libGAP_EndBags, libGAP_MptrBags)

/****************************************************************************
**
*V  FreeMptrBags  . . . . . . . . . . . . . . .  list of free bag identifiers
**
**  'FreeMptrBags' is the  first free bag identifier, i.e., it points  to the
**  first  available  masterpointer.   If 'FreeMptrBags'  is 0  there are  no
**  available masterpointers.  The available masterpointers are  managed in a
**  forward linked list,  i.e., each available  masterpointer  points  to the
**  next available masterpointer, except for the last, which contains 0.
**
**  When a new  bag is allocated  it gets the identifier  'FreeMptrBags', and
**  'FreeMptrBags' is set to the value stored in this masterpointer, which is
**  the next available masterpointer.  When a bag is deallocated by a garbage
**  collection  its  masterpointer  is  added   to  the  list  of   available
**  masterpointers again.
*/
libGAP_Bag libGAP_FreeMptrBags;


/****************************************************************************
**
*V  ChangedBags . . . . . . . . . . . . . . . . . .  list of changed old bags
**
**  'ChangedBags' holds a  list of old bags that  have been changed since the
**  last garbage collection, i.e., for  which either 'CHANGED_BAG' was called
**  or which have been resized.
**
**  This list starts with the bag  whose identifier is 'ChangedBags', and the
**  link word of each bag on the list contains the identifier of the next bag
**  on the list.  The link word of the  last bag on the list  contains 0.  If
**  'ChangedBags' has the value 0, the list is empty.
**
**  The garbage collection needs to know which young  bags are subbags of old
**  bags, since  it  must  not  throw   those away    in a partial    garbage
**  collection.  Only  those old bags that  have been changed  since the last
**  garbage collection can contain references to  young bags, which have been
**  allocated since the last garbage  collection.  The application cooperates
**  by informing {\Gasman} with 'CHANGED_BAG' which bags it has changed.  The
**  list of changed old  bags is scanned by a  partial garbage collection and
**  the young subbags of the old bags on this list are marked with 'MARK_BAG'
**  (see "MarkedBags").  Without this  list 'CollectBags' would have to  scan
**  all old bags for references to young bags, which would take too much time
**  (see "Implementation of CollectBags").
**
**  'CHANGED_BAG' puts a bag on the list  of changed old bags.  'CHANGED_BAG'
**  first checks that <bag> is an old bag by checking that 'PTR_BAG( <bag> )'
**  is smaller than 'YoungBags'.  Then it checks that  the bag is not already
**  on the list of changed bags by checking that the link word still contains
**  the identifier of <bag>.  If <bag> is an  old bag that  is not already on
**  the list of changed bags, 'CHANGED_BAG' puts <bag> on the list of changed
**  bags,  by  setting  the  link word of   <bag>   to the   current value of
**  'ChangedBags' and then setting 'ChangedBags' to <bag>.
*/
libGAP_Bag                     libGAP_ChangedBags;


/****************************************************************************
**
*V  MarkedBags  . . . . . . . . . . . . . . . . . . . . . list of marked bags
**
**  'MarkedBags' holds a list of bags that have already  been marked during a
**  garbage collection by 'MARK_BAG'.  This list is only used  during garbage
**  collections, so it is  always empty outside  of  garbage collections (see
**  "Implementation of CollectBags").
**
**  This list starts with the  bag whose identifier  is 'MarkedBags', and the
**  link word of each bag on the list contains the identifier of the next bag
**  on the list.  The link word of the  last bag on the list  contains 0.  If
**  'MarkedBags' has the value 0, the list is empty.
**
**  Note that some other  storage managers do not use  such a list during the
**  mark phase.   Instead  they simply let the  marking   functions call each
**  other.  While this is  somewhat simpler it  may use an unbound  amount of
**  space on the stack.  This is particularly  bad on systems where the stack
**  is not in a separate segment of the address space, and thus may grow into
**  the workspace, causing disaster.
**
**  'MARK_BAG'   puts a  bag <bag>  onto  this list.    'MARK_BAG'  has to be
**  careful, because it can be called  with an argument that  is not really a
**  bag identifier, and may  point  outside the programs  address space.   So
**  'MARK_BAG' first checks that <bag> points  to a properly aligned location
**  between 'MptrBags' and 'OldBags'.   Then 'MARK_BAG' checks that <bag>  is
**  the identifier  of a young bag by  checking that the masterpointer points
**  to  a  location between  'YoungBags'  and  'AllocBags'  (if <bag>  is the
**  identifier of an   old bag, the  masterpointer will  point to a  location
**  between  'OldBags' and 'YoungBags',  and if <bag>   only appears to be an
**  identifier, the masterpointer could be on the free list of masterpointers
**  and   point   to a  location  between  'MptrBags'  and  'OldBags').  Then
**  'MARK_BAG' checks  that <bag> is not  already marked by checking that the
**  link  word of <bag>  contains the identifier of the   bag.  If any of the
**  checks fails, 'MARK_BAG' does nothing.  If all checks succeed, 'MARK_BAG'
**  puts <bag> onto the  list of marked bags by  putting the current value of
**  'ChangedBags' into the link word  of <bag>  and setting 'ChangedBags'  to
**  <bag>.  Note that since bags are always placed  at the front of the list,
**  'CollectBags' will   mark the bags   in a  depth-first  order.   This  is
**  probably good to improve the locality of reference.
*/
libGAP_Bag                     libGAP_MarkedBags;


/****************************************************************************
**
*V  NrAllBags . . . . . . . . . . . . . . . . .  number of all bags allocated
*V  SizeAllBags . . . . . . . . . . . . . .  total size of all bags allocated
*V  NrLiveBags  . . . . . . . . . .  number of bags that survived the last gc
*V  SizeLiveBags  . . . . . . .  total size of bags that survived the last gc
*V  NrDeadBags  . . . . . . . number of bags that died since the last full gc
*V  SizeDeadBags  . . . . total size of bags that died since the last full gc
*V  NrHalfDeadBags  . . . . . number of bags that died since the last full gc
**                            but may still be weakly pointed to
*/
libGAP_UInt                    libGAP_NrAllBags;
libGAP_UInt                    libGAP_SizeAllBags;
libGAP_UInt                    libGAP_NrLiveBags;
libGAP_UInt                    libGAP_SizeLiveBags;
libGAP_UInt                    libGAP_NrDeadBags;
libGAP_UInt                    libGAP_SizeDeadBags;
libGAP_UInt                    libGAP_NrHalfDeadBags;


/****************************************************************************
**
*V  InfoBags[<type>]  . . . . . . . . . . . . . . . . .  information for bags
*/
libGAP_TNumInfoBags            libGAP_InfoBags [ libGAP_NTYPES ];

/****************************************************************************
**
*F  IS_BAG -- check if a value looks like a masterpointer reference.
*/
static inline libGAP_UInt libGAP_IS_BAG (
    libGAP_UInt                bid )
{
    return (((libGAP_UInt)libGAP_MptrBags <= bid)
         && (bid < (libGAP_UInt)libGAP_OldBags)
         && (bid & (sizeof(libGAP_Bag)-1)) == 0);
}

/****************************************************************************
**
*F  InitMsgsFuncBags(<msgs-func>) . . . . . . . . .  install message function
**
**  'InitMsgsFuncBags'  simply  stores  the  printing  function  in a  global
**  variable.
*/
libGAP_TNumMsgsFuncBags        libGAP_MsgsFuncBags;

void            libGAP_InitMsgsFuncBags (
    libGAP_TNumMsgsFuncBags    msgs_func )
{
    libGAP_MsgsFuncBags = msgs_func;
}


/****************************************************************************
**
*F  InitSweepFuncBags(<type>,<mark-func>)  . . . .  install sweeping function
*/

libGAP_TNumSweepFuncBags libGAP_TabSweepFuncBags [ libGAP_NTYPES ];


void libGAP_InitSweepFuncBags (
    libGAP_UInt                type,
    libGAP_TNumSweepFuncBags    sweep_func )
{
#ifdef CHECK_FOR_CLASH_IN_INIT_SWEEP_FUNC
    char                str[256];

    if ( libGAP_TabSweepFuncBags[type] != 0 ) {
        str[0] = 0;
        libGAP_SyStrncat( str, "warning: sweep function for type ", 33 );
        str[33] = '0' + ((type/100) % 10);
        str[34] = '0' + ((type/ 10) % 10);
        str[35] = '0' + ((type/  1) % 10);
        str[36] = 0;
        libGAP_SyStrncat( str, " already installed\n", 19 );
        libGAP_SyFputs( str, 0 );
    }
#endif
    libGAP_TabSweepFuncBags[type] = sweep_func;
}

#if ITANIUM
extern void * libGAP_ItaniumRegisterStackTop();

static libGAP_Bag* libGAP_ItaniumRegisterStackBottom = (libGAP_Bag *)0;

static void libGAP_ItaniumSpecialMarkingInit() {
    libGAP_ItaniumRegisterStackBottom = (libGAP_Bag *)libGAP_ItaniumRegisterStackTop();
}

#endif

/****************************************************************************
**
*F  InitMarkFuncBags(<type>,<mark-func>)  . . . . .  install marking function
*F  MarkNoSubBags(<bag>)  . . . . . . . . marking function that marks nothing
*F  MarkOneSubBags(<bag>) . . . . . .  marking function that marks one subbag
*F  MarkTwoSubBags(<bag>) . . . . . . marking function that marks two subbags
*F  MarkThreeSubBags(<bag>) . . . . marking function that marks three subbags
*F  MarkFourSubBags(<bag>)  . . . .  marking function that marks four subbags
*F  MarkAllSubBags(<bag>) . . . . . .  marking function that marks everything
**
**  'InitMarkFuncBags', 'MarkNoSubBags', 'MarkOneSubBags',  'MarkTwoSubBags',
**  and 'MarkAllSubBags' are really too simple for an explanation.
**
**  'MarkAllSubBagsDefault' is the same  as 'MarkAllSubBags' but is only used
**  by GASMAN as default.  This will allow to catch type clashes.
*/
libGAP_TNumMarkFuncBags libGAP_TabMarkFuncBags [ libGAP_NTYPES ];


void libGAP_InitMarkFuncBags (
    libGAP_UInt                type,
    libGAP_TNumMarkFuncBags    mark_func )
{
#ifdef CHECK_FOR_CLASH_IN_INIT_MARK_FUNC
    char                str[256];

    if ( libGAP_TabMarkFuncBags[type] != libGAP_MarkAllSubBagsDefault ) {
        str[0] = 0;
        libGAP_SyStrncat( str, "warning: mark function for type ", 32 );
        str[32] = '0' + ((type/100) % 10);
        str[33] = '0' + ((type/ 10) % 10);
        str[34] = '0' + ((type/  1) % 10);
        str[35] = 0;
        libGAP_SyStrncat( str, " already installed\n", 19 );
        libGAP_SyFputs( str, 0 );
    }
#endif
    libGAP_TabMarkFuncBags[type] = mark_func;
}

void libGAP_MarkNoSubBags (
    libGAP_Bag                 bag )
{
}

void libGAP_MarkOneSubBags (
    libGAP_Bag                 bag )
{
    libGAP_Bag                 sub;            /* one subbag identifier           */
    sub = libGAP_PTR_BAG(bag)[0];
    libGAP_MARK_BAG( sub );
}

void libGAP_MarkTwoSubBags (
    libGAP_Bag                 bag )
{
    libGAP_Bag                 sub;            /* one subbag identifier           */
    sub = libGAP_PTR_BAG(bag)[0];
    libGAP_MARK_BAG( sub );
    sub = libGAP_PTR_BAG(bag)[1];
    libGAP_MARK_BAG( sub );
}

void libGAP_MarkThreeSubBags (
    libGAP_Bag                 bag )
{
    libGAP_Bag                 sub;            /* one subbag identifier           */
    sub = libGAP_PTR_BAG(bag)[0];
    libGAP_MARK_BAG( sub );
    sub = libGAP_PTR_BAG(bag)[1];
    libGAP_MARK_BAG( sub );
    sub = libGAP_PTR_BAG(bag)[2];
    libGAP_MARK_BAG( sub );
}

void libGAP_MarkFourSubBags (
    libGAP_Bag                 bag )
{
    libGAP_Bag                 sub;            /* one subbag identifier           */
    sub = libGAP_PTR_BAG(bag)[0];
    libGAP_MARK_BAG( sub );
    sub = libGAP_PTR_BAG(bag)[1];
    libGAP_MARK_BAG( sub );
    sub = libGAP_PTR_BAG(bag)[2];
    libGAP_MARK_BAG( sub );
    sub = libGAP_PTR_BAG(bag)[3];
    libGAP_MARK_BAG( sub );
}

void libGAP_MarkAllSubBags (
    libGAP_Bag                 bag )
{
    libGAP_Bag *               ptr;            /* pointer into the bag            */
    libGAP_Bag                 sub;            /* one subbag identifier           */
    libGAP_UInt                i;              /* loop variable                   */

    /* mark everything                                                     */
    ptr = libGAP_PTR_BAG( bag );
    for ( i = libGAP_SIZE_BAG(bag)/sizeof(libGAP_Bag); 0 < i; i-- ) {
        sub = ptr[i-1];
        libGAP_MARK_BAG( sub );
    }

}

void libGAP_MarkAllSubBagsDefault (
    libGAP_Bag                 bag )
{
    libGAP_Bag *               ptr;            /* pointer into the bag            */
    libGAP_Bag                 sub;            /* one subbag identifier           */
    libGAP_UInt                i;              /* loop variable                   */

    /* mark everything                                                     */
    ptr = libGAP_PTR_BAG( bag );
    for ( i = libGAP_SIZE_BAG(bag)/sizeof(libGAP_Bag); 0 < i; i-- ) {
        sub = ptr[i-1];
        libGAP_MARK_BAG( sub );
    }

}


void libGAP_MarkBagWeakly(
    libGAP_Bag             bag )
{
  if ( (((libGAP_UInt)bag) & (sizeof(libGAP_Bag)-1)) == 0 /* really looks like a pointer */
       && (libGAP_Bag)libGAP_MptrBags <= bag              /* in plausible range */
       && bag < (libGAP_Bag)libGAP_OldBags                /*  "    "       "    */
       && libGAP_YoungBags < libGAP_PTR_BAG(bag)          /*  points to a young bag */
       && libGAP_PTR_BAG(bag) <= libGAP_AllocBags         /*    "     " "  "     "  */
       && libGAP_IS_MARKED_DEAD(bag) )             /*  and not marked already */
    {                          
      libGAP_LINK_BAG(bag) = (libGAP_Bag)libGAP_MARKED_HALFDEAD(bag);   /* mark it now as we
                                               don't have to recurse */
    }
}



/****************************************************************************
**
*F  CallbackForAllBags( <func> ) call a C function on all non-zero mptrs
**
**  This calls a C function on every bag, including garbage ones, by simply
**  walking the masterpointer area. Not terribly safe.
**
*/

void libGAP_CallbackForAllBags(
     void (*func)(libGAP_Bag) )
{
  libGAP_Bag ptr;
  for (ptr = (libGAP_Bag)libGAP_MptrBags; ptr < (libGAP_Bag)libGAP_OldBags; ptr ++)
    if (*ptr != 0 && !libGAP_IS_WEAK_DEAD_BAG(ptr) && (libGAP_Bag)(*ptr) >= (libGAP_Bag)libGAP_OldBags)
      {
        (*func)(ptr);
      }
}


/****************************************************************************
**  
*V  GlobalBags  . . . . . . . . . . . . . . . . . . . . . list of global bags
*/  
libGAP_TNumGlobalBags libGAP_GlobalBags;


/****************************************************************************
**
*F  InitGlobalBag(<addr>, <cookie>) inform Gasman about global bag identifier
**
**  'InitGlobalBag' simply leaves the address <addr> in a global array, where
**  it is used by 'CollectBags'. <cookie> is also recorded to allow things to
**  be matched up after loading a saved workspace.
*/
static libGAP_UInt libGAP_GlobalSortingStatus;
libGAP_Int libGAP_WarnInitGlobalBag;

extern libGAP_TNumAbortFuncBags   libGAP_AbortFuncBags;

void libGAP_ClearGlobalBags ( void )
{
  libGAP_UInt i;
  for (i = 0; i < libGAP_GlobalBags.nr; i++)
    {
      libGAP_GlobalBags.addr[i] = 0L;
      libGAP_GlobalBags.cookie[i] = 0L;
    }
  libGAP_GlobalBags.nr = 0;
  libGAP_GlobalSortingStatus = 0;
  libGAP_WarnInitGlobalBag = 0;
  return;
}    

void libGAP_InitGlobalBag (
    libGAP_Bag *               addr,
    const libGAP_Char *        cookie )
{

    if ( libGAP_GlobalBags.nr == libGAP_NR_GLOBAL_BAGS ) {
        (*libGAP_AbortFuncBags)(
            "Panic: Gasman cannot handle so many global variables" );
    }
#ifdef DEBUG_GLOBAL_BAGS
    {
      libGAP_UInt i;
      if (cookie != (libGAP_Char *)0)
        for (i = 0; i < libGAP_GlobalBags.nr; i++)
          if ( 0 == strcmp(libGAP_GlobalBags.cookie[i], cookie) )
            if (libGAP_GlobalBags.addr[i] == addr)
              libGAP_Pr("Duplicate global bag entry %s\n", (libGAP_Int)cookie, 0L);
            else
              libGAP_Pr("Duplicate global bag cookie %s\n", (libGAP_Int)cookie, 0L);
    }
#endif
    if ( libGAP_WarnInitGlobalBag ) {
        libGAP_Pr( "#W  global bag '%s' initialized\n", (libGAP_Int)cookie, 0L );
    } 
    libGAP_GlobalBags.addr[libGAP_GlobalBags.nr] = addr;
    libGAP_GlobalBags.cookie[libGAP_GlobalBags.nr] = cookie;
    libGAP_GlobalBags.nr++;
    libGAP_GlobalSortingStatus = 0;
}



static libGAP_Int libGAP_IsLessGlobal (
    const libGAP_Char *        cookie1, 
    const libGAP_Char *        cookie2,
    libGAP_UInt                byWhat )
{
  if (byWhat != 2)
    {
      (*libGAP_AbortFuncBags)("can only sort globals by cookie");
    }
  if (cookie1 == 0L && cookie2 == 0L)
    return 0;
  if (cookie1 == 0L)
    return -1;
  if (cookie2 == 0L)
    return 1;
  return strcmp(cookie1, cookie2) < 0;
}



void libGAP_SortGlobals( libGAP_UInt byWhat )
{
  const libGAP_Char *tmpcookie;
  libGAP_Bag * tmpaddr;
  libGAP_UInt len, h, i, k;
  if (byWhat != 2)
    {
      (*libGAP_AbortFuncBags)("can only sort globals by cookie");
    }
  if (libGAP_GlobalSortingStatus == byWhat)
    return;
  len = libGAP_GlobalBags.nr;
  h = 1;
  while ( 9*h + 4 < len ) 
    { h = 3*h + 1; }
  while ( 0 < h ) {
    for ( i = h; i < len; i++ ) {
      tmpcookie = libGAP_GlobalBags.cookie[i];
      tmpaddr = libGAP_GlobalBags.addr[i];
      k = i;
      while ( h <= k && libGAP_IsLessGlobal(tmpcookie,
                                     libGAP_GlobalBags.cookie[k-h],
                                     byWhat))
        {
          libGAP_GlobalBags.cookie[k] = libGAP_GlobalBags.cookie[k-h];
          libGAP_GlobalBags.addr[k] = libGAP_GlobalBags.addr[k-h];
          k -= h;
        }
      libGAP_GlobalBags.cookie[k] = tmpcookie;
      libGAP_GlobalBags.addr[k] = tmpaddr;
    }
    h = h / 3;
  }
  libGAP_GlobalSortingStatus = byWhat;
  return;
}


                       
libGAP_Bag * libGAP_GlobalByCookie(
       const libGAP_Char * cookie )
{
  libGAP_UInt i,top,bottom,middle;
  libGAP_Int res;
  if (cookie == 0L)
    {
      libGAP_Pr("Panic -- 0L cookie passed to GlobalByCookie\n",0L,0L);
      libGAP_SyExit(2);
    }
  if (libGAP_GlobalSortingStatus != 2) 
    {
      for (i = 0; i < libGAP_GlobalBags.nr; i++)
        {
          if (strcmp(cookie, libGAP_GlobalBags.cookie[i]) == 0)
            return libGAP_GlobalBags.addr[i];
        }
      return (libGAP_Bag *)0L;
    }
  else
    {
      top = libGAP_GlobalBags.nr;
      bottom = 0;
      while (top >= bottom) {
        middle = (top + bottom)/2;
        res = strcmp(cookie,libGAP_GlobalBags.cookie[middle]);
        if (res < 0)
          top = middle-1;
        else if (res > 0)
          bottom = middle+1;
        else
          return libGAP_GlobalBags.addr[middle];
      }
      return (libGAP_Bag *)0L;
    }
}
                                      

static libGAP_Bag libGAP_NextMptrRestoring;
extern libGAP_TNumAllocFuncBags       libGAP_AllocFuncBags;

void libGAP_StartRestoringBags( libGAP_UInt nBags, libGAP_UInt maxSize)
{
  libGAP_UInt target;
  libGAP_Bag *newmem;
/*Bag *ptr; */
  target = (8*nBags)/7 + (8*maxSize)/7; /* ideal workspace size */
  target = (target * sizeof (libGAP_Bag) + (512L*1024L) - 1)/(512L*1024L)*(512L*1024L)/sizeof (libGAP_Bag);
              /* make sure that the allocated amount of memory is divisible by 512 * 1024 */
  if (libGAP_SizeWorkspace < target)
    {
      newmem  = (*libGAP_AllocFuncBags)(sizeof(libGAP_Bag)*(target- libGAP_SizeWorkspace)/1024, 0);
      if (newmem == 0)
        {
          target = nBags + maxSize; /* absolute requirement */
          target = (target * sizeof (libGAP_Bag) + (512L*1024L) - 1)/(512L*1024L)*(512L*1024L)/sizeof (libGAP_Bag);
               /* make sure that the allocated amount of memory is divisible by 512 * 1024 */
          if (libGAP_SizeWorkspace < target)
            (*libGAP_AllocFuncBags)(sizeof(libGAP_Bag)*(target- libGAP_SizeWorkspace)/1024, 1);
        }
      libGAP_EndBags = libGAP_MptrBags + target;
    }
  libGAP_OldBags = libGAP_MptrBags + nBags + (libGAP_SizeWorkspace - nBags - maxSize)/8;
  libGAP_AllocBags = libGAP_OldBags;
  libGAP_NextMptrRestoring = (libGAP_Bag)libGAP_MptrBags;
  libGAP_SizeAllBags = 0;
  libGAP_NrAllBags = 0;
  return;
}

libGAP_Bag libGAP_NextBagRestoring( libGAP_UInt size, libGAP_UInt type)
{
  libGAP_Bag bag;
  libGAP_UInt i;
  *(libGAP_Bag **)libGAP_NextMptrRestoring = (libGAP_AllocBags+libGAP_HEADER_SIZE);
  bag = libGAP_NextMptrRestoring;
#ifdef libGAP_USE_NEWSHAPE
  ((libGAP_UInt *)libGAP_AllocBags)[0] = (size << 16 | type);
#else
  ((libGAP_UInt *)libGAP_AllocBags)[0] = type;
  ((libGAP_UInt *)libGAP_AllocBags)[1] = size;
#endif
  
  ((libGAP_Bag *)libGAP_AllocBags)[libGAP_HEADER_SIZE-1] = libGAP_NextMptrRestoring;
  libGAP_NextMptrRestoring++;
#ifdef DEBUG_LOADING
  if ((libGAP_Bag *)libGAP_NextMptrRestoring >= libGAP_OldBags)
    (*libGAP_AbortFuncBags)("Overran Masterpointer area");
#endif
  libGAP_AllocBags += libGAP_HEADER_SIZE;
  
  for (i = 0; i < libGAP_WORDS_BAG(size); i++)
    *libGAP_AllocBags++ = (libGAP_Bag)0;
  
#ifdef DEBUG_LOADING
  if (libGAP_AllocBags > libGAP_EndBags)
    (*libGAP_AbortFuncBags)("Overran data area");
#endif
#ifdef COUNT_BAGS
  libGAP_InfoBags[type].nrLive   += 1;
  libGAP_InfoBags[type].nrAll    += 1;
  libGAP_InfoBags[type].sizeLive += size;
  libGAP_InfoBags[type].sizeAll  += size;
#endif
  libGAP_SizeAllBags += size;
  libGAP_NrAllBags ++;
  return bag;
}

void libGAP_FinishedRestoringBags( void )
{
  libGAP_Bag p;
/*  Bag *ptr; */
  libGAP_YoungBags = libGAP_AllocBags;
  libGAP_StopBags = libGAP_AllocBags + libGAP_WORDS_BAG(libGAP_AllocSizeBags);
  if (libGAP_StopBags > libGAP_EndBags)
    libGAP_StopBags = libGAP_EndBags;
  libGAP_FreeMptrBags = libGAP_NextMptrRestoring;
  for (p = libGAP_NextMptrRestoring; p +1 < (libGAP_Bag)libGAP_OldBags; p++)
    *(libGAP_Bag *)p = p+1;
  *p = 0;
  libGAP_NrLiveBags = libGAP_NrAllBags;
  libGAP_SizeLiveBags = libGAP_SizeAllBags;
  libGAP_NrDeadBags = 0;
  libGAP_SizeDeadBags = 0;
  libGAP_NrHalfDeadBags = 0;
  libGAP_ChangedBags = 0;
  return;
}


/****************************************************************************
**
*F  InitFreeFuncBag(<type>,<free-func>) . . . . . .  install freeing function
**
**  'InitFreeFuncBag' is really too simple for an explanation.
*/
libGAP_TNumFreeFuncBags        libGAP_TabFreeFuncBags [ 256 ];

libGAP_UInt                    libGAP_NrTabFreeFuncBags;

void            libGAP_InitFreeFuncBag (
    libGAP_UInt                type,
    libGAP_TNumFreeFuncBags    free_func )
{
    if ( free_func != 0 ) {
        libGAP_NrTabFreeFuncBags = libGAP_NrTabFreeFuncBags + 1;
    }
    else {
        libGAP_NrTabFreeFuncBags = libGAP_NrTabFreeFuncBags - 1;
    }
    libGAP_TabFreeFuncBags[type] = free_func;
}


/****************************************************************************
**
*F  InitCollectFuncBags(<bfr-func>,<aft-func>) . install collection functions
**
**  'InitCollectFuncBags' is really too simple for an explanation.
*/
libGAP_TNumCollectFuncBags     libGAP_BeforeCollectFuncBags;

libGAP_TNumCollectFuncBags     libGAP_AfterCollectFuncBags;

void            libGAP_InitCollectFuncBags (
    libGAP_TNumCollectFuncBags before_func,
    libGAP_TNumCollectFuncBags after_func )
{
    libGAP_BeforeCollectFuncBags = before_func;
    libGAP_AfterCollectFuncBags  = after_func;
}


/****************************************************************************
**
*F  FinishBags() . . . . . . . . . . . . . . . . . . . . . . .finalize GASMAN
**
** `FinishBags()' ends GASMAN and returns all memory to the OS pool
**
*/

void libGAP_FinishBags( void )
{
  (*libGAP_AllocFuncBags)(-(sizeof(libGAP_Bag)*libGAP_SizeWorkspace/1024),2);
  return;
}

/****************************************************************************
**
*F  InitBags(...) . . . . . . . . . . . . . . . . . . . . . initialize Gasman
**
**  'InitBags'   remembers   <alloc-func>,  <stack-func>,     <stack-bottom>,
**  <stack-align>, <cache-size>,  <dirty>,    and   <abort-func>  in   global
**  variables.   It also  allocates  the initial workspace,   and sets up the
**  linked list of available masterpointer.
*/
libGAP_TNumAllocFuncBags       libGAP_AllocFuncBags;

libGAP_TNumStackFuncBags       libGAP_StackFuncBags;

libGAP_Bag *                   libGAP_StackBottomBags;

libGAP_UInt                    libGAP_StackAlignBags;

libGAP_UInt                    libGAP_CacheSizeBags;

libGAP_UInt                    libGAP_DirtyBags;

libGAP_TNumAbortFuncBags       libGAP_AbortFuncBags;

void            libGAP_InitBags (
    libGAP_TNumAllocFuncBags   alloc_func,
    libGAP_UInt                initial_size,
    libGAP_TNumStackFuncBags   stack_func,
    libGAP_Bag *               stack_bottom,
    libGAP_UInt                stack_align,
    libGAP_UInt                cache_size,
    libGAP_UInt                dirty,
    libGAP_TNumAbortFuncBags   abort_func )
{
    libGAP_Bag *               p;              /* loop variable                   */
    libGAP_UInt                i;              /* loop variable                   */

    libGAP_ClearGlobalBags();
    libGAP_WarnInitGlobalBag = 0;
    
    /* install the allocator and the abort function                        */
    libGAP_AllocFuncBags   = alloc_func;
    libGAP_AbortFuncBags   = abort_func;

    /* install the stack marking function and values                       */
    libGAP_StackFuncBags   = stack_func;
    libGAP_StackBottomBags = stack_bottom;
    libGAP_StackAlignBags  = stack_align;
#if ITANIUM
    libGAP_ItaniumSpecialMarkingInit();
#endif

    /* first get some storage from the operating system                    */
    initial_size    = (initial_size + 511) & ~(511);
    libGAP_MptrBags = (*libGAP_AllocFuncBags)( initial_size, 1 );
    if ( libGAP_MptrBags == 0 )
        (*libGAP_AbortFuncBags)("cannot get storage for the initial workspace.");
    libGAP_EndBags = libGAP_MptrBags + 1024*(initial_size / sizeof(libGAP_Bag*));

    /* 1/8th of the storage goes into the masterpointer area               */
    libGAP_FreeMptrBags = (libGAP_Bag)libGAP_MptrBags;
    for ( p = libGAP_MptrBags;
          p + 2*(libGAP_SIZE_MPTR_BAGS) <= libGAP_MptrBags+1024*initial_size/8/sizeof(libGAP_Bag*);
          p += libGAP_SIZE_MPTR_BAGS )
    {
        *p = (libGAP_Bag)(p + libGAP_SIZE_MPTR_BAGS);
    }

    /* the rest is for bags                                                */
    libGAP_OldBags   = libGAP_MptrBags + 1024*initial_size/8/sizeof(libGAP_Bag*);
    libGAP_YoungBags = libGAP_OldBags;
    libGAP_AllocBags = libGAP_OldBags;

    /* remember the cache size                                             */
    libGAP_CacheSizeBags = cache_size;
    if ( ! libGAP_CacheSizeBags ) {
        libGAP_AllocSizeBags = 256;
        libGAP_StopBags = libGAP_EndBags;
    }
    else {
        libGAP_AllocSizeBags = (libGAP_CacheSizeBags+1023)/1024;
        libGAP_StopBags  = libGAP_AllocBags + libGAP_WORDS_BAG(1024*libGAP_AllocSizeBags) <= libGAP_EndBags ?
                    libGAP_AllocBags + libGAP_WORDS_BAG(1024*libGAP_AllocSizeBags) :  libGAP_EndBags;
    }

    /* remember whether bags should be clean                               */
    libGAP_DirtyBags = dirty;

    /* install the marking functions                                       */
    for ( i = 0; i < 255; i++ )
        libGAP_TabMarkFuncBags[i] = libGAP_MarkAllSubBagsDefault;

    /* Set ChangedBags to a proper initial value */
    libGAP_ChangedBags = 0;

}


/****************************************************************************
**
*F  NewBag( <type>, <size> )  . . . . . . . . . . . . . .  allocate a new bag
**
**  'NewBag' is actually quite simple.
**
**  It first tests whether enough storage is available in the allocation area
**  and  whether a free   masterpointer is available.   If  not, it  starts a
**  garbage collection by calling 'CollectBags' passing <size> as the size of
**  the bag it is currently allocating and 0 to indicate  that only a partial
**  garbage collection is called for.   If 'CollectBags' fails and returns 0,
**  'NewBag' also fails and also returns 0.
**
**  Then it takes the first free  masterpointer from the  linked list of free
**  masterpointers (see "FreeMptrBags").
**
**  Then it  writes  the  size and the   type  into the word   pointed to  by
**  'AllocBags'.  Then  it writes the identifier,  i.e.,  the location of the
**  masterpointer, into the next word.
**
**  Then it advances 'AllocBags' by '2 + WORDS_BAG(<size>)'.
**
**  Finally it returns the identifier of the new bag.
**
**  Note that 'NewBag' never  initializes the new bag  to contain only 0.  If
**  this is desired because  the initialization flag <dirty> (see "InitBags")
**  was  0, it is the job  of 'CollectBags'  to initialize the new free space
**  after a garbage collection.
**
**  If {\Gasman} was compiled with the option 'COUNT_BAGS' then 'NewBag' also
**  updates the information in 'InfoBags' (see "InfoBags").
**
**  'NewBag'  is implemented as  a  function  instead of a  macro  for  three
**  reasons.  It  reduces the size of  the program, improving the instruction
**  cache  hit ratio.   The compiler  can do  anti-aliasing analysis  for the
**  local  variables  of the function.  To  enable  statistics only {\Gasman}
**  needs to be recompiled.
*/
libGAP_Bag libGAP_NewBag (
    libGAP_UInt                type,
    libGAP_UInt                size )
{
    libGAP_Bag                 bag;            /* identifier of the new bag       */
    libGAP_Bag *               dst;            /* destination of the new bag      */

#ifdef TREMBLE_HEAP
    libGAP_CollectBags(0,0);
#endif
    
    /* check that a masterpointer and enough storage are available         */
    if ( (libGAP_FreeMptrBags == 0 || libGAP_SizeAllocationArea < libGAP_HEADER_SIZE+libGAP_WORDS_BAG(size))
      && libGAP_CollectBags( size, 0 ) == 0 )
    {
        return 0;
    }

#ifdef  COUNT_BAGS
    /* update the statistics                                               */
    libGAP_NrAllBags               += 1;
    libGAP_InfoBags[type].nrLive   += 1;
    libGAP_InfoBags[type].nrAll    += 1;
    libGAP_InfoBags[type].sizeLive += size;
    libGAP_InfoBags[type].sizeAll  += size;
#endif
    libGAP_SizeAllBags             += size;

    /* get the identifier of the bag and set 'FreeMptrBags' to the next    */
    bag          = libGAP_FreeMptrBags;
    libGAP_FreeMptrBags = *(libGAP_Bag*)bag;

    /* allocate the storage for the bag                                    */
    dst       = libGAP_AllocBags;
    libGAP_AllocBags = dst + libGAP_HEADER_SIZE + libGAP_WORDS_BAG(size);


    /* enter size-type words                                               */
#ifdef libGAP_USE_NEWSHAPE
    *dst++ = (libGAP_Bag)(size << 16 | type);
#else
    *dst++ = (libGAP_Bag)(type);
    *dst++ = (libGAP_Bag)(size);
#endif


    /* enter link word                                                     */
    *dst++ = bag;

    /* set the masterpointer                                               */
    libGAP_PTR_BAG(bag) = dst;
#if 0
    {
      extern void * stderr;
      libGAP_UInt i;
      for (i = 0; i < libGAP_WORDS_BAG(size); i++)
        if (*dst++)
          fprintf(stderr, "dirty bag being returned\n");
    }
#endif
    /* return the identifier of the new bag                                */
    return bag;
}


/****************************************************************************
**
*F  RetypeBag(<bag>,<new>)  . . . . . . . . . . . .  change the type of a bag
**
**  'RetypeBag' is very simple.
**
**  All it has to do is to change the size-type word of the bag.
**
**  If  {\Gasman} was compiled with the  option 'COUNT_BAGS' then 'RetypeBag'
**  also updates the information in 'InfoBags' (see "InfoBags").
*/
void            libGAP_RetypeBag (
    libGAP_Bag                 bag,
    libGAP_UInt                new_type )
{

#ifdef  COUNT_BAGS
    /* update the statistics      */
    {
          libGAP_UInt                old_type;       /* old type of the bag */
	  libGAP_UInt                size;

          old_type = libGAP_TNUM_BAG(bag);
	  size = libGAP_SIZE_BAG(bag);
          libGAP_InfoBags[old_type].nrLive   -= 1;
          libGAP_InfoBags[new_type].nrLive   += 1;
          libGAP_InfoBags[old_type].nrAll    -= 1;
          libGAP_InfoBags[new_type].nrAll    += 1;
          libGAP_InfoBags[old_type].sizeLive -= size;
          libGAP_InfoBags[new_type].sizeLive += size;
          libGAP_InfoBags[old_type].sizeAll  -= size;
          libGAP_InfoBags[new_type].sizeAll  += size;
    }
#endif

    /* change the size-type word                                           */
#ifdef libGAP_USE_NEWSHAPE
    *(*bag-libGAP_HEADER_SIZE) &= 0xFFFFFFFFFFFF0000L;
    *(*bag-libGAP_HEADER_SIZE) |= new_type;
#else
    *(*bag-libGAP_HEADER_SIZE) = new_type;
#endif
}


/****************************************************************************
**
*F  ResizeBag(<bag>,<new>)  . . . . . . . . . . . .  change the size of a bag
**
**  Basically 'ResizeBag' is rather  simple, but there  are a few  traps that
**  must be avoided.
**
**  If the size of the bag changes only a little bit, so that  the  number of
**  words needed for the data area does not  change, 'ResizeBag' only changes
**  the size-type word of the bag.
**
**  If  the bag  is  to be  shrunk  and at  least   one  word becomes   free,
**  'ResizeBag'  changes  the  size-type word of  the bag, and stores a magic
**  size-type word in  the first free word.  This  magic size-type  word  has
**  type 255 and the size  is the number  of  following  free bytes, which is
**  always divisible by 'sizeof(Bag)'.  The  type 255 allows 'CollectBags' to
**  detect that  this body  is the remainder of a   resize operation, and the
**  size allows  it  to know how  many  bytes  there  are  in this  body (see
**  "Implementation of CollectBags").
**
**  So for example if 'ResizeBag' shrinks a bag of type 7 from 18 bytes to 10
**  bytes the situation before 'ResizeBag' is as follows{\:}
**
**    +---------+
**    |<masterp>|
**    +---------+
**         \_____________
**                       \
**                        V
**    +---------+---------+--------------------------------------------+----+
**    | 18  . 7 |  <link> |         .         .         .         .    | pad|
**    +---------+---------+--------------------------------------------+----+
**
**  And after 'ResizeBag' the situation is as follows{\:}
**
**    +---------+
**    |<masterp>|
**    +---------+
**         \_____________
**                       \
**                        V
**    +---------+---------+------------------------+----+---------+---------+
**    | 10  . 7 |  <link> |         .         .    | pad|  4  .255|         |
**    +---------+---------+------------------------+----+---------+---------+
**
**  If the bag is to be extended and it  is that last  allocated bag, so that
**  it  is  immediately adjacent  to the allocation  area, 'ResizeBag' simply
**  increments  'AllocBags' after making  sure that enough space is available
**  in the allocation area (see "Layout of the Workspace").
**
**  If the bag  is to be   extended and it  is not  the  last allocated  bag,
**  'ResizeBag'  first allocates a  new bag  similar to 'NewBag', but without
**  using  a new masterpointer.   Then it copies the old  contents to the new
**  bag.  Finally it resets the masterpointer of the bag to  point to the new
**  address.  Then it changes the type of the old body  to  255,  so that the
**  garbage collection can detect that this body is the remainder of a resize
**  (see "Implementation of NewBag" and "Implementation of CollectBags").
**
**  When an old bag is extended, it  will now reside  in the young bags area,
**  and thus appear  to be young.   Since old bags are   supposed to  survive
**  partial garbage  collections 'ResizeBag'  must   somehow protect this bag
**  from partial garbage collections.  This is  done by putting this bag onto
**  the linked  list  of  changed bags (see   "ChangedBags").  When a partial
**  garbage collection sees a young bag on the list of changed bags, it knows
**  that it is the result of 'ResizeBag' of an old bag, and does not throw it
**  away (see "Implementation of CollectBags").  Note  that  when 'ResizeBag'
**  tries this, the bag may already be on the linked  list, either because it
**  has been resized earlier, or because  it has been  changed.  In this case
**  'ResizeBag' simply keeps the bag on this linked list.
**
**  If {\Gasman}  was compiled with the  option 'COUNT_BAGS' then 'ResizeBag'
**  also updates the information in 'InfoBags' (see "InfoBags").
*/

   libGAP_UInt libGAP_ResizeBag (
    libGAP_Bag                 bag,
    libGAP_UInt                new_size )
{
    libGAP_UInt                type;           /* type of the bag                 */
    libGAP_UInt                old_size;       /* old size of the bag             */
    libGAP_Bag *               dst;            /* destination in copying          */
    libGAP_Bag *               src;            /* source in copying               */
    libGAP_Bag *               end;            /* end in copying                  */

    /* check the size                                                      */
    
#ifdef TREMBLE_HEAP
    libGAP_CollectBags(0,0);
#endif

    /* get type and old size of the bag                                    */
    type     = libGAP_TNUM_BAG(bag);
    old_size = libGAP_SIZE_BAG(bag);

#ifdef  COUNT_BAGS
    /* update the statistics                                               */
    libGAP_InfoBags[type].sizeLive += new_size - old_size;
    libGAP_InfoBags[type].sizeAll  += new_size - old_size;
#endif
    libGAP_SizeAllBags             += new_size - old_size;

    /* if the real size of the bag doesn't change                          */
    if ( libGAP_WORDS_BAG(new_size) == libGAP_WORDS_BAG(old_size) ) {

        /* change the size word                                            */
#ifdef libGAP_USE_NEWSHAPE
      *(*bag-2) = (new_size << 16 | type);
#else
      *(*bag-2) = new_size;
#endif
    }

    /* if the bag is shrunk                                                */
    /* we must not shrink the last bag by moving 'AllocBags',              */
    /* since the remainder may not be zero filled                          */
    else if ( libGAP_WORDS_BAG(new_size) < libGAP_WORDS_BAG(old_size) ) {

      /* leave magic size-type word for the sweeper, type must be 255    */
        if ((libGAP_WORDS_BAG(old_size)-libGAP_WORDS_BAG(new_size) == 1))
          *(libGAP_UInt*)(libGAP_PTR_BAG(bag) + libGAP_WORDS_BAG(new_size)) = 1 << 8 | 255;
        else
          {
#ifdef libGAP_USE_NEWSHAPE
            *(libGAP_UInt*)(libGAP_PTR_BAG(bag) + libGAP_WORDS_BAG(new_size)) = 
              (libGAP_WORDS_BAG(old_size)-libGAP_WORDS_BAG(new_size)-1)*sizeof(libGAP_Bag) << 16 | 255;
#else
            *(libGAP_UInt*)(libGAP_PTR_BAG(bag) + libGAP_WORDS_BAG(new_size)) = 255;
            *(libGAP_UInt*)(libGAP_PTR_BAG(bag) + libGAP_WORDS_BAG(new_size) + 1) =
              (libGAP_WORDS_BAG(old_size)-libGAP_WORDS_BAG(new_size)-1)*sizeof(libGAP_Bag);
#endif
          }

        /* change the size- word                                       */
#ifdef libGAP_USE_NEWSHAPE
      *(*bag-2) = (new_size << 16 | type);
#else
      *(*bag-2) = new_size;
#endif


    }

    /* if the last bag is to be enlarged                                   */
    else if ( libGAP_PTR_BAG(bag) + libGAP_WORDS_BAG(old_size) == libGAP_AllocBags ) {

        /* check that enough storage for the new bag is available          */
        if ( libGAP_StopBags < libGAP_PTR_BAG(bag)+libGAP_WORDS_BAG(new_size)
          && libGAP_CollectBags( new_size-old_size, 0 ) == 0 ) {
            return 0;
        }

        /* simply increase the free pointer                                */
        if ( libGAP_YoungBags == libGAP_AllocBags )
            libGAP_YoungBags += libGAP_WORDS_BAG(new_size) - libGAP_WORDS_BAG(old_size);
        libGAP_AllocBags += libGAP_WORDS_BAG(new_size) - libGAP_WORDS_BAG(old_size);

        /* change the size-type word                                       */
#ifdef libGAP_USE_NEWSHAPE
      *(*bag-2) = (new_size << 16 | type);
#else
      *(*bag-2) = new_size;
#endif
    }

    /* if the bag is enlarged                                              */
    else {

        /* check that enough storage for the new bag is available          */
        if ( libGAP_SizeAllocationArea <  libGAP_HEADER_SIZE+libGAP_WORDS_BAG(new_size)
          && libGAP_CollectBags( new_size, 0 ) == 0 ) {
            return 0;
        }

        /* allocate the storage for the bag                                */
        dst       = libGAP_AllocBags;
        libGAP_AllocBags = dst + libGAP_HEADER_SIZE + libGAP_WORDS_BAG(new_size);
        
        /* leave magic size-type word  for the sweeper, type must be 255   */
#ifdef libGAP_USE_NEWSHAPE
        *(*bag-2) = (((libGAP_WORDS_BAG(old_size)+1) * sizeof(libGAP_Bag))) << 16 | 255;
        *dst++ = (libGAP_Bag)(new_size << 16 | type);
#else
        *(*bag-3) = 255; 
        *(*bag-2) = (((libGAP_WORDS_BAG(old_size)+2) * sizeof(libGAP_Bag)));
        
        /* enter the new size-type word                                    */

        *dst++ = (libGAP_Bag)type;
        *dst++ = (libGAP_Bag)new_size;
#endif
        

        /* if the bag is already on the changed bags list, keep it there   */
        if ( libGAP_PTR_BAG(bag)[-1] != bag ) {
            *dst++ = libGAP_PTR_BAG(bag)[-1];
        }

        /* if the bag is old, put it onto the changed bags list            */
        else if ( libGAP_PTR_BAG(bag) <= libGAP_YoungBags ) {
            *dst++ = libGAP_ChangedBags;  libGAP_ChangedBags = bag;
        }

        /* if the bag is young, enter the normal link word                 */
        else {
            *dst++ = bag;
        }

        /* set the masterpointer                                           */
        src = libGAP_PTR_BAG(bag);
        end = src + libGAP_WORDS_BAG(old_size);
        libGAP_PTR_BAG(bag) = dst;

        /* copy the contents of the bag                                    */
        while ( src < end )
            *dst++ = *src++;

    }

    /* return success                                                      */
    return 1;
}


/****************************************************************************
**
*F  CollectBags( <size>, <full> ) . . . . . . . . . . . . . collect dead bags
**
**  'CollectBags' is the function that does most of the work of {\Gasman}.
**
**  A partial garbage collection where  every bag is  young is clearly a full
**  garbage    collection.  So  to     perform  a  full  garbage  collection,
**  'CollectBags' first sets 'YoungBags'  to   'OldBags', making every    bag
**  young, and empties the list  of changed old bags, since  there are no old
**  bags anymore, there  can be no changed old  bags anymore.  So from now on
**  we    can   assume that  'CollectBags'     is doing   a  partial  garbage
**  collection.   In  addition,    the   values 'NewWeakDeadBagMarker'    and
**  'OldWeakDeadBagMarker'  are exchanged, so  that bag idnetifiers that have
**  been  halfdead  since    before  this full    garbage  collection cab  be
**  distinguished from those which have died on this pass.
**
**  Garbage collection  is  performed in  three phases.  The  mark phase, the
**  sweep phase, and the check phase.
**
**  In the  *mark phase*, 'CollectBags' finds  all young bags that  are still
**  live and builds a linked list of those bags (see "MarkedBags").  A bag is
**  put on  this  list  of  marked bags   by   applying  'MARK_BAG' to    its
**  identifier.  Note that 'MARK_BAG' checks that a bag is not already on the
**  list of marked bags, before it puts it on the list, so  no bag can be put
**  twice on this list.
**
**  First, 'CollectBags' marks  all  young bags that are  directly accessible
**  through global   variables,  i.e.,  it   marks those young     bags whose
**  identifiers  appear  in  global variables.   It    does this  by applying
**  'MARK_BAG'  to the values at the  addresses  of global variables that may
**  hold bag identifiers provided by 'InitGlobalBag' (see "InitGlobalBag").
**
**  Next,  'CollectBags' marks  all  young bags  that are directly accessible
**  through   local  variables, i.e.,    it  marks those  young   bags  whose
**  identifiers appear in the  stack.   It  does  this by calling  the  stack
**  marking  function  <stack-func>  (see  "InitBags").   The   generic stack
**  marking function, which is called if <stack-func> (see "InitBags") was 0,
**  is described below.  The problem is  that there is usually not sufficient
**  information  available to decide  if a value on   the stack is really the
**  identifier of a bag, or is a  value of another  type that only appears to
**  be the  identifier  of a bag.  The  position  usually taken by  the stack
**  marking function is that everything  on the stack  that could possibly be
**  interpreted  as the identifier of  a bag is an  identifier of  a bag, and
**  that this bag is therefore live.  This position is what makes {\Gasman} a
**  conservative storage manager.
**
**  The generic stack marking function 'GenStackFuncBags', which is called if
**  <stack-func> (see "InitBags") was 0, works by  applying 'MARK_BAG' to all
**  the values on the stack,  which is supposed to extend  from <stack-start>
**  (see  "InitBags") to the address of  a local variable of   the  function.
**  Note that some local variables may  not  be stored on the  stack, because
**  they are  still in the processors registers.    'GenStackFuncBags' uses a
**  jump buffer 'RegsBags', filled by the C library function 'setjmp', marking
**  all bags  whose  identifiers appear in 'RegsBags'.  This  is a dirty hack,
**  that need not work, but actually works on a  surprisingly large number of
**  machines.  But it will not work on Sun  Sparc machines, which have larger
**  register  files, of which  only the part  visible to the current function
**  will be saved  by  'setjmp'.  For those machines 'GenStackFuncBags' first
**  calls the operating system to flush the whole register file.  Note that a
**  compiler may save  a register  somewhere else  if   it wants to  use this
**  register for something else.  Usually  this register is saved  further up
**  the  stack,  i.e.,   beyond the   address  of  the  local variable,   and
**  'GenStackFuncBags' would not see this value any more.   To deal with this
**  problem, 'setjmp' must be called *before* 'GenStackFuncBags'  is entered,
**  i.e.,  before the  registers may have been saved  elsewhere.   Thus it is
**  called from 'CollectBags'.
**
**  Next 'CollectBags' marks all young bags that are directly accessible from
**  old bags, i.e.,  it marks all young bags  whose identifiers appear in the
**  data areas  of  old bags.  It  does  this by applying 'MARK_BAG'  to each
**  identifier appearing in changed old bags, i.e., in those bags that appear
**  on the list of changed old bags (see "ChangedBags").   To be more precise
**  it calls the  marking function for the appropriate  type to  each changed
**  old bag (see "InitMarkFuncBags").  It need not apply the marking function
**  to each old  bag, because old bags that  have not been changed  since the
**  last garbage  collection cannot contain identifiers  of young bags, which
**  have been allocated since the last garbage collection.  Of course marking
**  the subbags of only  the changed old  bags is more efficient than marking
**  the subbags of  all old bags only  if the number of  changed old  bags is
**  smaller than the total number of old bags, but this  is a very reasonable
**  assumption.
**
**  Note that there may also be bags that  appear to be  young on the list of
**  changed old bags.  Those bags  are old bags that  were extended since the
**  last garbage  collection and therefore have their  body in the young bags
**  area (see "Implementation of  ResizeBag").  When 'CollectBags' finds such
**  a bag  on  the list of  changed  old bags  it  applies 'MARK_BAG'  to its
**  identifier and thereby  ensures that this bag will  not be thrown away by
**  this garbage collection.
**
**  Next,  'CollectBags'    marks all  young    bags  that  are  *indirectly*
**  accessible, i.e., it marks the subbags of  the already marked bags, their
**  subbags  and  so on.  It  does  so by walking   along the list of already
**  marked bags and applies  the marking function  of the appropriate type to
**  each bag on this list (see  "InitMarkFuncBags").  Those marking functions
**  then apply 'MARK_BAG' or 'MarkBagWeakly'  to each identifier appearing in
**  the bag.
**
**  After  the marking function has  been  applied to a   bag on the list  of
**  marked bag, this bag is removed from the list.  Thus the marking phase is
**  over when the list  of marked bags   has become empty.  Removing the  bag
**  from the list of marked  bags must be done  at  this time, because  newly
**  marked bags are *prepended* to the list of  marked bags.  This is done to
**  ensure that bags are marked in a  depth first order, which should usually
**  improve locality of   reference.  When a bag is   taken from the list  of
**  marked bags it is *tagged*.  This tag serves two purposes.  A bag that is
**  tagged is not put on the list  of marked bags  when 'MARK_BAG' is applied
**  to its identifier.  This ensures that  no bag is put  more than once onto
**  the list of marked bags, otherwise endless marking loops could happen for
**  structures that contain circular  references.  Also the sweep phase later
**  uses the presence of  the tag to decide the  status of the bag. There are
**  three possible statuses: LIVE, DEAD and  HALFDEAD. The default state of a
**  bag with its identifier in the link word, is  the tag for DEAD. Live bags
**  are tagged    with  MARKED_ALIVE(<identifier>)  in the   link   word, and
**  half-dead bags (ie bags pointed to weakly but not strongly) with the tage
**  MARKED_HALFDEAD(<identifier>).
** 
**  Note that 'CollectBags' cannot put a random or magic  value into the link
**  word, because the sweep phase must be able to find the masterpointer of a
**  bag by only looking at the link word of a bag. This is done using the macros
**  UNMARKED_XXX(<link word contents>).
**
**  In the   *sweep  phase*, 'CollectBags'   deallocates all dead   bags  and
**  compacts the live bags at the beginning of the workspace.
**
**  In this  phase 'CollectBags'   uses  a destination pointer   'dst', which
**  points to  the address a  body  will be copied to,   and a source pointer
**  'src',  which points to the address  a body currently has.  Both pointers
**  initially   point to  the   beginning  of  the   young bags area.    Then
**  'CollectBags' looks at the body pointed to by the source pointer.
**
**  If this body has type 255, it is the remainder of a resize operation.  In
**  this case 'CollectBags' simply moves the source pointer to the  next body
**  (see "Implementation of ResizeBag").
**
**
**  Otherwise, if the  link word contains the  identifier of the bag  itself,

**  marked dead,  'CollectBags' first adds the masterpointer   to the list of
**  available masterpointers (see  "FreeMptrBags") and then simply  moves the
**  source pointer to the next bag.
**
**  Otherwise, if the link  word contains  the  identifier of the  bag marked
**  alive, this   bag is still  live.  In  this case  'CollectBags' calls the
**  sweeping function for this bag, if one  is installed, or otherwise copies
**  the body from  the source address to the  destination address, stores the
**  address of the masterpointer   without the tag  in   the link word,   and
**  updates the masterpointer to point to the new address of the data area of
**  the bag.   After the copying  the source pointer points  to the next bag,
**  and the destination pointer points just past the copy.
**
**  Finally, if the link word contains the identifier of  the bag marked half
**  dead, then  'CollectBags' puts  the special value  'NewWeakDeadBagMarker'
**  into the masterpointer corresponding to the bag, to signify that this bag
**  has been collected as garbage.
**
**  This is repeated until  the source pointer  reaches the end of  the young
**  bags area, i.e., reaches 'AllocBags'.
**
**  The new free storage now is the area between  the destination pointer and
**  the source pointer.  If the initialization flag  <dirty> (see "InitBags")
**  was 0, this area is now cleared.
**
**  Next, 'CollectBags' sets   'YoungBags'  and 'AllocBags'  to   the address
**  pointed to by the destination  pointer.  So all the  young bags that have
**  survived this garbage  collection are now  promoted  to be old  bags, and
**  allocation of new bags will start at the beginning of the free storage.
**
**  Finally, the *check phase* checks  whether  the garbage collection  freed
**  enough storage and masterpointers.
**
**  After a partial garbage collection,  'CollectBags' wants at least '<size>
**  + AllocSizeBags' bytes  of free  storage  available, where <size> is  the
**  size of the bag that 'NewBag' is  currently trying to allocate.  Also the
**  number  of free masterpointers should be  larger than the  number of bags
**  allocated since   the  previous garbage collection  plus 4096  more to be
**  safe.   If less free   storage  or  fewer masterpointers  are  available,
**  'CollectBags' calls itself for a full garbage collection.
**
**  After a  full  garbage collection,  'CollectBags' wants at   least <size>
**  bytes of free storage available, where <size> is the size of the bag that
**  'NewBag' is  currently trying  to allocate.  Also it  wants  at least one
**  free  masterpointer.    If less free    storage or   no masterpointer are
**  available, 'CollectBags'  tries   to   extend  the  workspace   using the
**  allocation   function <alloc-func> (see    "InitBags").   If <alloc-func>
**  refuses  to extend  the   workspace,  'CollectBags' returns 0 to indicate
**  failure to 'NewBag'.  In  any case 'CollectBags' will  try to  extend the
**  workspace so that at least one eigth of the storage is free, that is, one
**  eight of the storage between 'OldBags' and 'EndBags' shall be  free.   If
**  <alloc-func> refuses this extension of the workspace, 'CollectBags' tries
**  to get along with  what it  got.   Also 'CollectBags' wants at least  one
**  masterpointer per 8 words of free storage available.  If  this is not the
**  case, 'CollectBags'  extends the masterpointer area  by moving the bodies
**  of all bags and readjusting the masterpointers.  
**
**  Also,  after   a  full  garbage   collection,  'CollectBags'   scans  the
**  masterpointer area for  identifiers containing 'OldWeakDeadBagMarker'. If
**  the sweep functions have done their work then no  references to these bag
**  identifiers can exist, and so 'CollectBags' frees these masterpointers.  
*/

libGAP_syJmp_buf libGAP_RegsBags;

#if defined(SPARC) && SPARC
void libGAP_SparcStackFuncBags( void )
{
  asm (" ta 0x3 ");
  asm (" mov %sp,%o0" );
  return;
}
#endif


void libGAP_GenStackFuncBags ( void )
{
    libGAP_Bag *               top;            /* top of stack                    */
    libGAP_Bag *               p;              /* loop variable                   */
    libGAP_UInt                i;              /* loop variable                   */

    top = (libGAP_Bag*)((void*)&top);
    if ( libGAP_StackBottomBags < top ) {
        for ( i = 0; i < sizeof(libGAP_Bag*); i += libGAP_StackAlignBags ) {
            for ( p = (libGAP_Bag*)((char*)libGAP_StackBottomBags + i); p < top; p++ )
                libGAP_MARK_BAG( *p );
        }
    }
    else {
        for ( i = 0; i < sizeof(libGAP_Bag*); i += libGAP_StackAlignBags ) {
            for ( p = (libGAP_Bag*)((char*)libGAP_StackBottomBags - i); top < p; p-- )
                libGAP_MARK_BAG( *p );
        }
    }
#if ITANIUM 
    /* Itanium has two stacks */
    top = libGAP_ItaniumRegisterStackTop();
    for ( i = 0; i < sizeof(libGAP_Bag*); i += libGAP_StackAlignBags ) {
        for ( p = (libGAP_Bag*)((char*)libGAP_ItaniumRegisterStackBottom + i); p < top; p++ )
            libGAP_MARK_BAG( *p );
    }
#endif

    /* mark from registers, dirty dirty hack                               */
    for ( p = (libGAP_Bag*)((void*)libGAP_RegsBags);
          p < (libGAP_Bag*)((void*)libGAP_RegsBags)+sizeof(libGAP_RegsBags)/sizeof(libGAP_Bag);
          p++ )
        libGAP_MARK_BAG( *p );

}

libGAP_UInt libGAP_FullBags;

/*  These are used to overwrite masterpointers which may still be
linked from weak pointer objects but whose bag bodies have been
collected.  Two values are used so that old masterpointers of this
kind can be reclaimed after a full garbage collection. The values must
not look like valid pointers, and should be congruent to 1 mod sizeof(Bag) */

libGAP_Bag * libGAP_NewWeakDeadBagMarker = (libGAP_Bag *)(1000*sizeof(libGAP_Bag) + 1L);
libGAP_Bag * libGAP_OldWeakDeadBagMarker = (libGAP_Bag *)(1001*sizeof(libGAP_Bag) + 1L); 



libGAP_UInt libGAP_CollectBags (
    libGAP_UInt                size,
    libGAP_UInt                full )
{
    libGAP_Bag                 first;          /* first bag on a linked list      */
    libGAP_Bag *               p;              /* loop variable                   */
    libGAP_Bag *               dst;            /* destination in sweeping         */
    libGAP_Bag *               src;            /* source in sweeping              */
    libGAP_Bag *               end;            /* end of a bag in sweeping        */
    libGAP_UInt                nrLiveBags;     /* number of live new bags         */
    libGAP_UInt                sizeLiveBags;   /* total size of live new bags     */
    libGAP_UInt                nrDeadBags;     /* number of dead new bags         */
    libGAP_UInt                nrHalfDeadBags; /* number of dead new bags         */
    libGAP_UInt                sizeDeadBags;   /* total size of dead new bags     */
    libGAP_UInt                done;           /* do we have to make a full gc    */
    libGAP_UInt                i;              /* loop variable                   */

    /*     Bag *               last;
           Char                type; */

#ifdef DEBUG_MASTERPOINTERS
    libGAP_CheckMasterPointers();
#endif

    
    /* call the before function (if any)                                   */
    if ( libGAP_BeforeCollectFuncBags != 0 )
        (*libGAP_BeforeCollectFuncBags)();

    /* copy 'full' into a global variable, to avoid warning from GNU C     */
    libGAP_FullBags = full;

    /* do we want to make a full garbage collection?                       */
again:
    if ( libGAP_FullBags ) {

        /* then every bag is considered to be a young bag                  */
        libGAP_YoungBags = libGAP_OldBags;
        libGAP_NrLiveBags = 0;
        libGAP_SizeLiveBags = 0;

        /* empty the list of changed old bags                              */
        while ( libGAP_ChangedBags != 0 ) {
            first = libGAP_ChangedBags;
            libGAP_ChangedBags = libGAP_PTR_BAG(first)[-1];
            libGAP_PTR_BAG(first)[-1] = first;
        }

        /* Also time to change the tag for dead children of weak
           pointer objects. After this collection, there can be no more 
           weak pointer objects pointing to anything with OldWeakDeadBagMarker
           in it */
        {
          libGAP_Bag * t;
          t = libGAP_OldWeakDeadBagMarker;
          libGAP_OldWeakDeadBagMarker = libGAP_NewWeakDeadBagMarker;
          libGAP_NewWeakDeadBagMarker = t;
        }
    }

    /* information at the beginning of garbage collections                 */
    if ( libGAP_MsgsFuncBags )
        (*libGAP_MsgsFuncBags)( libGAP_FullBags, 0, 0 );

    /* * * * * * * * * * * * * * *  mark phase * * * * * * * * * * * * * * */

    /* prepare the list of marked bags for the future                      */
    libGAP_MarkedBags = 0;

    /* call the libgap callback so library users can mark their own bags   */
    libgap_call_gasman_callback();

    /* mark from the static area                                           */
    for ( i = 0; i < libGAP_GlobalBags.nr; i++ )
        libGAP_MARK_BAG( *libGAP_GlobalBags.addr[i] );

    
    /* mark from the stack                                                 */
    if ( libGAP_StackFuncBags ) {
        (*libGAP_StackFuncBags)();
    }
    else {
      libGAP_sySetjmp( libGAP_RegsBags );
#ifdef  SPARC
#if SPARC
        libGAP_SparcStackFuncBags();
#endif
#endif
        libGAP_GenStackFuncBags();
    }

    /* mark the subbags of the changed old bags                            */
    while ( libGAP_ChangedBags != 0 ) {
        first = libGAP_ChangedBags;
        libGAP_ChangedBags = libGAP_PTR_BAG(first)[-1];
        libGAP_PTR_BAG(first)[-1] = first;
        if ( libGAP_PTR_BAG(first) <= libGAP_YoungBags )
            (*libGAP_TabMarkFuncBags[libGAP_TNUM_BAG(first)])( first );
        else
            libGAP_MARK_BAG(first);
    }


    /* tag all marked bags and mark their subbags                          */
    nrLiveBags = 0;
    sizeLiveBags = 0;
    while ( libGAP_MarkedBags != 0 ) {
        first = libGAP_MarkedBags;
        libGAP_MarkedBags = libGAP_PTR_BAG(first)[-1];
        libGAP_PTR_BAG(first)[-1] = libGAP_MARKED_ALIVE(first);
        (*libGAP_TabMarkFuncBags[libGAP_TNUM_BAG(first)])( first );
        nrLiveBags++;
        sizeLiveBags += libGAP_SIZE_BAG(first);
    }

    /* information after the mark phase                                    */
    libGAP_NrLiveBags += nrLiveBags;
    if ( libGAP_MsgsFuncBags )
        (*libGAP_MsgsFuncBags)( libGAP_FullBags, 1, nrLiveBags );
    libGAP_SizeLiveBags += sizeLiveBags;
    if ( libGAP_MsgsFuncBags )
        (*libGAP_MsgsFuncBags)( libGAP_FullBags, 2, sizeLiveBags/1024 );

    /* * * * * * * * * * * * * * * sweep phase * * * * * * * * * * * * * * */

#if 0
    /* call freeing function for all dead bags                             */
    if ( libGAP_NrTabFreeFuncBags ) {

        /* run through the young generation                                */
        src = libGAP_YoungBags;
        while ( src < libGAP_AllocBags ) {

            /* leftover of a resize of <n> bytes                           */
            if ( (*(libGAP_UInt*)src & 0xFFL) == 255 ) {

                if ((*(libGAP_UInt *)src >> 16) == 1) 
                  src++;
                else
                  src += libGAP_WORDS_BAG(((libGAP_UInt *)src)[1]);
                

            }

            /* dead or half-dead (only weakly pointed to bag               */
            /* here the usual check using UNMARKED_DEAD etc. is not
               safe, because we are looking at the bag body rather
               than its identifier, and a wrong guess for the bag
               status can involve following a misaligned pointer. It
               may cause bus errors or actual mistakes.

               Instead we look directly at the value in the link word
               and check its least significant bits */

            else if ( ((libGAP_UInt)(src[libGAP_HEADER_SIZE-1])) % sizeof(libGAP_Bag) == 0 ||
                      ((libGAP_UInt)(src[libGAP_HEADER_SIZE-1])) % sizeof(libGAP_Bag) == 2 )
              {
#ifdef DEBUG_MASTERPOINTERS
                if  ( (((libGAP_UInt)(src[1])) % sizeof(libGAP_Bag) == 0 && 
                       libGAP_PTR_BAG( libGAP_UNMARKED_DEAD(src[1]) ) != src+libGAP_HEADER_SIZE)  ||
                      (((libGAP_UInt)(src[1])) % sizeof(libGAP_Bag) == 2 &&
                       libGAP_PTR_BAG( libGAP_UNMARKED_HALFDEAD(src[1])) != src+libGAP_HEADER_SIZE))
                  {
                    (*libGAP_AbortFuncBags)("incorrectly marked bag");
                  }
#endif

                /* call freeing function                                   */
                if ( libGAP_TabFreeFuncBags[ *(libGAP_UInt*)src & 0xFFL ] != 0 )
                  (*libGAP_TabFreeFuncBags[ *(libGAP_UInt*)src & 0xFFL ])( src[libGAP_HEADER_SIZE-1] );
                
                /* advance src                                             */
                src += libGAP_HEADER_SIZE + libGAP_WORDS_BAG( ((libGAP_UInt*)src)[1] ) ;
                
              }
            

            /* live bag                                                    */
            else if ( ((libGAP_UInt)(src[libGAP_HEADER_SIZE-1])) % sizeof(libGAP_Bag) == 1 )
              {
#ifdef DEBUG_MASTERPOINTERS
                if  ( libGAP_PTR_BAG( libGAP_UNMARKED_ALIVE(src[libGAP_HEADER_SIZE-1]) ) != src+libGAP_HEADER_SIZE )
                  {
                    (*libGAP_AbortFuncBags)("incorrectly marked bag");
                  }
#endif

                /* advance src                                             */
#ifdef libGAP_USE_NEWSHAPE
                src += libGAP_HEADER_SIZE + libGAP_WORDS_BAG( ((libGAP_UInt*)src)[0] >>16  );
#else
                src += libGAP_HEADER_SIZE + libGAP_WORDS_BAG( ((libGAP_UInt*)src)[1]  );
#endif


            }

            /* oops                                                        */
            else {
                (*libGAP_AbortFuncBags)(
                  "Panic: Gasman found a bogus header (looking for dead bags)");
            }

        }

    }
#endif
    /* sweep through the young generation                                  */
    nrDeadBags = 0;
    nrHalfDeadBags = 0;
    sizeDeadBags = 0;
    dst = libGAP_YoungBags;
    src = libGAP_YoungBags;
    while ( src < libGAP_AllocBags ) {

        /* leftover of a resize of <n> bytes                               */
        if ( (*(libGAP_UInt*)src & 0xFFL) == 255 ) {     

            /* advance src                                                 */
            if ((*(libGAP_UInt *) src) >> 8 == 1)
              src++;
            else
#ifdef libGAP_USE_NEWSHAPE
              src += 1 + libGAP_WORDS_BAG(((libGAP_UInt *)src)[0] >> 16);
#else
              src += 1 + libGAP_WORDS_BAG(((libGAP_UInt *)src)[1]);
#endif

        }

        /* dead bag                                                        */

        else if ( ((libGAP_UInt)(src[libGAP_HEADER_SIZE-1])) % sizeof(libGAP_Bag) == 0 )
          {
#ifdef DEBUG_MASTERPOINTERS
            if  ( libGAP_PTR_BAG( libGAP_UNMARKED_DEAD(src[libGAP_HEADER_SIZE-1]) ) != src+libGAP_HEADER_SIZE )
              {
                (*libGAP_AbortFuncBags)("incorrectly marked bag");
              }
#endif


            /* update count                                                */
            if (libGAP_TabFreeFuncBags[ *(libGAP_UInt *)src & 0xFFL] != 0)
              (*libGAP_TabFreeFuncBags[ *(libGAP_UInt*)src & 0xFFL ])( src[libGAP_HEADER_SIZE-1] );
            nrDeadBags += 1;
#ifdef      libGAP_USE_NEWSHAPE
            sizeDeadBags +=  ((libGAP_UInt *)src)[0] >> 16;
#else
            sizeDeadBags += ((libGAP_UInt *)src)[1];
#endif    

#ifdef  COUNT_BAGS
            /* update the statistics                                       */
            libGAP_InfoBags[*(libGAP_UInt*)src & 0xFFL].nrLive -= 1;
#ifdef libGAP_USE_NEWSHAPE
            libGAP_InfoBags[*(libGAP_UInt*)src & 0xFFL].sizeLive -=
            ((libGAP_UInt *)src)[0] >>16;
#else
            libGAP_InfoBags[*(libGAP_UInt*)src & 0xFFL].sizeLive -=
            ((libGAP_UInt *)src)[1];
#endif
#endif

            /* free the identifier                                         */
            *(libGAP_Bag*)(src[libGAP_HEADER_SIZE-1]) = libGAP_FreeMptrBags;
            libGAP_FreeMptrBags = src[libGAP_HEADER_SIZE-1];

            /* advance src                                                 */
#ifdef libGAP_USE_NEWSHAPE
            src += libGAP_HEADER_SIZE +
              libGAP_WORDS_BAG( ((libGAP_UInt*)src)[0] >> 16 ) ;
#else
            src += libGAP_HEADER_SIZE +
              libGAP_WORDS_BAG( ((libGAP_UInt*)src)[1] ) ;
#endif

        }

        /* half-dead bag                                                   */
        else if ( ((libGAP_UInt)(src[libGAP_HEADER_SIZE-1])) % sizeof(libGAP_Bag) == 2 )
          {
#ifdef DEBUG_MASTERPOINTERS
            if  ( libGAP_PTR_BAG( libGAP_UNMARKED_HALFDEAD(src[libGAP_HEADER_SIZE-1]) ) != src+libGAP_HEADER_SIZE )
              {
                (*libGAP_AbortFuncBags)("incorrectly marked bag");
              }
#endif


            /* update count                                                */
            nrDeadBags += 1;
#ifdef libGAP_USE_NEWSHAPE
            sizeDeadBags += ((libGAP_UInt *)src)[0] >> 16;
#else
            sizeDeadBags += ((libGAP_UInt *)src)[1];
#endif

#ifdef  COUNT_BAGS
            /* update the statistics                                       */
            libGAP_InfoBags[*(libGAP_UInt*)src & 0xFFL].nrLive -= 1;
#ifdef libGAP_USE_NEWSHAPE
            libGAP_InfoBags[*(libGAP_UInt*)src & 0xFFL].sizeLive -=
            ((libGAP_UInt *)src)[0] >>16;
#else
            libGAP_InfoBags[*(libGAP_UInt*)src & 0xFFL].sizeLive -=
            ((libGAP_UInt *)src)[1];
#endif
#endif

            /* don't free the identifier                                   */
            if (((libGAP_UInt)libGAP_UNMARKED_HALFDEAD(src[libGAP_HEADER_SIZE-1])) % 4 != 0)
              (*libGAP_AbortFuncBags)("align error in halfdead bag");
                                              
           *(libGAP_Bag**)(libGAP_UNMARKED_HALFDEAD(src[libGAP_HEADER_SIZE-1])) = libGAP_NewWeakDeadBagMarker;
           nrHalfDeadBags ++;

            /* advance src                                                 */
#ifdef libGAP_USE_NEWSHAPE
            src += libGAP_HEADER_SIZE +
              libGAP_WORDS_BAG( ((libGAP_UInt*)src)[0] >> 16 ) ;
#else
            src += libGAP_HEADER_SIZE +
              libGAP_WORDS_BAG( ((libGAP_UInt*)src)[1] ) ;
#endif

        }

        /* live bag                                                        */
        else if ( ((libGAP_UInt)(src[libGAP_HEADER_SIZE-1])) % sizeof(libGAP_Bag) == 1 )
          {
#ifdef DEBUG_MASTERPOINTERS
            if  ( libGAP_PTR_BAG( libGAP_UNMARKED_ALIVE(src[libGAP_HEADER_SIZE-1]) ) != src+libGAP_HEADER_SIZE )
              {
                (*libGAP_AbortFuncBags)("incorrectly marked bag");
              }
#endif


            /* update identifier, copy size-type and link field            */
            libGAP_PTR_BAG( libGAP_UNMARKED_ALIVE(src[libGAP_HEADER_SIZE-1])) = dst+libGAP_HEADER_SIZE;
#ifdef libGAP_USE_NEWSHAPE
            end = src + libGAP_HEADER_SIZE +
              libGAP_WORDS_BAG( ((libGAP_UInt*)src)[0] >>16 ) ;
#else
            end = src + libGAP_HEADER_SIZE +
              libGAP_WORDS_BAG( ((libGAP_UInt*)src)[1] ) ;
#endif
            *dst++ = *src++;
#ifndef libGAP_USE_NEWSHAPE
            *dst++ = *src++;
#endif

            *dst++ = (libGAP_Bag)libGAP_UNMARKED_ALIVE(*src++);

            /* copy data area                                */
              if (libGAP_TabSweepFuncBags[(libGAP_UInt)(src[-libGAP_HEADER_SIZE]) & 0xFFL] != 0)
                {
                  /* Call the installed sweeping function */
                  (*(libGAP_TabSweepFuncBags[(libGAP_UInt)(src[-libGAP_HEADER_SIZE]) & 0xFFL]))(src,dst,end-src);
                  dst += end-src;
                  src = end;
                  
                }
              
              /* Otherwise do the default thing */
              else if ( dst != src ) {
                memmove((void *)dst, (void *)src, (end - src)*sizeof(*src));
                dst += (end-src);
                src = end;
                
                /*
                while ( src < end )
                  *dst++ = *src++;
                */
              }
              else {
                dst = end;
                src = end;
              }
        }

        /* oops                                                            */
        else {

            (*libGAP_AbortFuncBags)("Panic: Gasman found a bogus header");

        }

    }

    /* reset the pointer to the free storage                               */
    libGAP_AllocBags = libGAP_YoungBags = dst;

    /* clear the new free area                                             */
    if (!libGAP_DirtyBags)
      memset((void *)dst, 0, ((libGAP_Char  *)src)-((libGAP_Char *)dst));

    /*    if ( ! DirtyBags ) {
        while ( dst < src )
            *dst++ = 0;
            } */

    /* information after the sweep phase                                   */
    libGAP_NrDeadBags += nrDeadBags;
    libGAP_NrHalfDeadBags += nrHalfDeadBags;
    if ( libGAP_MsgsFuncBags )
        (*libGAP_MsgsFuncBags)( libGAP_FullBags, 3,
                         (libGAP_FullBags ? libGAP_NrDeadBags:nrDeadBags) );
    if ( libGAP_FullBags )
        libGAP_NrDeadBags = 0;
    libGAP_SizeDeadBags += sizeDeadBags;
    if ( libGAP_MsgsFuncBags )
        (*libGAP_MsgsFuncBags)( libGAP_FullBags, 4,
                         (libGAP_FullBags ? libGAP_SizeDeadBags:sizeDeadBags)/1024 );
    if ( libGAP_FullBags )
        libGAP_SizeDeadBags = 0;

    /* * * * * * * * * * * * * * * check phase * * * * * * * * * * * * * * */

    /* temporarily store in 'StopBags' where this allocation takes us      */
    libGAP_StopBags = libGAP_AllocBags + libGAP_HEADER_SIZE + libGAP_WORDS_BAG(size);



    /* if we only performed a partial garbage collection                   */
    if ( ! libGAP_FullBags ) {

        /* maybe adjust the size of the allocation area                    */
        if ( ! libGAP_CacheSizeBags ) {
            if ( nrLiveBags+nrDeadBags +nrHalfDeadBags < 512
                 
                 /* The test below should stop AllocSizeBags 
                    growing uncontrollably when all bags are big */
                 && libGAP_StopBags > libGAP_OldBags + 4*1024*libGAP_WORDS_BAG(libGAP_AllocSizeBags))
                libGAP_AllocSizeBags += 256L;
            else if ( 4096 < nrLiveBags+nrDeadBags+nrHalfDeadBags
                   && 256 < libGAP_AllocSizeBags )
                libGAP_AllocSizeBags -= 256;
        }
        else {
            if ( nrLiveBags+nrDeadBags < 512 )
                libGAP_AllocSizeBags += libGAP_CacheSizeBags/1024;
            else if ( 4096 < nrLiveBags+nrDeadBags+nrHalfDeadBags
                   && libGAP_CacheSizeBags < libGAP_AllocSizeBags )
                libGAP_AllocSizeBags -= libGAP_CacheSizeBags/1024;
        }

        /* if we dont get enough free storage or masterpointers do full gc */
        if ( libGAP_EndBags < libGAP_StopBags + libGAP_WORDS_BAG(1024*libGAP_AllocSizeBags)
          || libGAP_SizeMptrsArea <
             
             /*      nrLiveBags+nrDeadBags+nrHalfDeadBags+ 4096 */
             /*      If this test triggered, but the one below didn't
                     then a full collection would ensue which wouldn't
                     do anything useful. Possibly a version of the
                     above test should be moved into the full collection also
                     but I wasn't sure it always made sense         SL */

             /* change the test to avoid subtracting unsigned integers */
             
             libGAP_WORDS_BAG(libGAP_AllocSizeBags*1024)/7 +(libGAP_NrLiveBags + libGAP_NrHalfDeadBags) 
             ) {
            done = 0;
        }
        else {
            done = 1;
        }

    }

    /* if we already performed a full garbage collection                   */
    else {

      /* Clean up old half-dead bags                                      
         also reorder the free masterpointer linked list
         to get more locality */
      libGAP_FreeMptrBags = (libGAP_Bag)0L;
      for (p = libGAP_MptrBags; p < libGAP_OldBags; p+= libGAP_SIZE_MPTR_BAGS)
        {
          libGAP_Bag *mptr = (libGAP_Bag *)*p;
          if ( mptr == libGAP_OldWeakDeadBagMarker)
            libGAP_NrHalfDeadBags--;
          if ( mptr == libGAP_OldWeakDeadBagMarker || libGAP_IS_BAG((libGAP_UInt)mptr) || mptr == 0)
            {
              *p = libGAP_FreeMptrBags;
              libGAP_FreeMptrBags = (libGAP_Bag)p;
            }
        }


        /* get the storage we absolutly need                               */
        while ( libGAP_EndBags < libGAP_StopBags
             && (*libGAP_AllocFuncBags)(512,1) )
            libGAP_EndBags += libGAP_WORDS_BAG(512*1024L);

        /* if not enough storage is free, fail                             */
        if ( libGAP_EndBags < libGAP_StopBags )
            return 0;

        /* if less than 1/8th is free, get more storage (in 1/2 MBytes)    */
        while ( ( libGAP_SpaceBetweenPointers(libGAP_EndBags, libGAP_StopBags) <  libGAP_SpaceBetweenPointers(libGAP_StopBags, libGAP_OldBags)/7 ||
                  libGAP_SpaceBetweenPointers(libGAP_EndBags, libGAP_StopBags) < libGAP_WORDS_BAG(libGAP_AllocSizeBags) )
             && (*libGAP_AllocFuncBags)(512,0) )
            libGAP_EndBags += libGAP_WORDS_BAG(512*1024L);

        /* If we are having trouble, then cut our cap to fit our cloth *.
        if ( EndBags - StopBags < AllocSizeBags )
        AllocSizeBags = 7*(Endbags - StopBags)/8; */

        /* if less than 1/16th is free, prepare for an interrupt           */
        if (libGAP_SpaceBetweenPointers(libGAP_StopBags,libGAP_OldBags)/15 < libGAP_SpaceBetweenPointers(libGAP_EndBags,libGAP_StopBags) ) {
            /*N 1993/05/16 martin must change 'gap.c'                      */
            ;
        }

        /* if more than 1/8th is free, give back storage (in 1/2 MBytes)   */
        while (libGAP_SpaceBetweenPointers(libGAP_StopBags,libGAP_OldBags)/7 <= libGAP_SpaceBetweenPointers(libGAP_EndBags,libGAP_StopBags)-libGAP_WORDS_BAG(512*1024L)
                && libGAP_SpaceBetweenPointers(libGAP_EndBags,libGAP_StopBags) > libGAP_WORDS_BAG(libGAP_AllocSizeBags) + libGAP_WORDS_BAG(512*1024L)
             && (*libGAP_AllocFuncBags)(-512,0) )
            libGAP_EndBags -= libGAP_WORDS_BAG(512*1024L);

        /* if we want to increase the masterpointer area                   */
        if ( libGAP_SpaceBetweenPointers(libGAP_OldBags,libGAP_MptrBags)-libGAP_NrLiveBags < libGAP_SpaceBetweenPointers(libGAP_EndBags,libGAP_StopBags)/7 ) {

            /* this is how many new masterpointers we want                 */
            i = libGAP_SpaceBetweenPointers(libGAP_EndBags,libGAP_StopBags)/7 - (libGAP_SpaceBetweenPointers(libGAP_OldBags,libGAP_MptrBags)-libGAP_NrLiveBags);

            /* move the bags area                                          */
            memmove((void *)(libGAP_OldBags+i), (void *)libGAP_OldBags, libGAP_SpaceBetweenPointers(libGAP_AllocBags,libGAP_OldBags)*sizeof(*libGAP_OldBags));

            /* update the masterpointers                                   */
            for ( p = libGAP_MptrBags; p < libGAP_OldBags; p++ ) {
              if ( (libGAP_Bag)libGAP_OldBags <= *p)
                    *p += i;
            }

            /* link the new part of the masterpointer area                 */
            for ( p = libGAP_OldBags;
                  p + 2*libGAP_SIZE_MPTR_BAGS <= libGAP_OldBags+i;
                  p += libGAP_SIZE_MPTR_BAGS ) {
                *p = (libGAP_Bag)(p + libGAP_SIZE_MPTR_BAGS);
            }
            *p = (libGAP_Bag)libGAP_FreeMptrBags;
            libGAP_FreeMptrBags = (libGAP_Bag)libGAP_OldBags;

            /* update 'OldBags', 'YoungBags', 'AllocBags', and 'StopBags'  */
            libGAP_OldBags   += i;
            libGAP_YoungBags += i;
            libGAP_AllocBags += i;
            libGAP_StopBags  += i;

        }

        /* now we are done                                                 */
        done = 1;

    }

    /* information after the check phase                                   */
    if ( libGAP_MsgsFuncBags )
      (*libGAP_MsgsFuncBags)( libGAP_FullBags, 5,
                       libGAP_SpaceBetweenPointers(libGAP_EndBags, libGAP_StopBags)/(1024/sizeof(libGAP_Bag)));
    if ( libGAP_MsgsFuncBags )
        (*libGAP_MsgsFuncBags)( libGAP_FullBags, 6,
                         libGAP_SpaceBetweenPointers(libGAP_EndBags, libGAP_MptrBags)/(1024/sizeof(libGAP_Bag)));

    /* reset the stop pointer                                              */
    if ( ! libGAP_CacheSizeBags || libGAP_EndBags < libGAP_StopBags+libGAP_WORDS_BAG(1024*libGAP_AllocSizeBags) )
        libGAP_StopBags = libGAP_EndBags;
    else
        libGAP_StopBags = libGAP_StopBags + libGAP_WORDS_BAG(1024*libGAP_AllocSizeBags);

    /* if we are not done, then true again                                 */
    if ( ! done ) {
        libGAP_FullBags = 1;
        goto again;
    }

    /* call the after function (if any)                                    */
    if ( libGAP_AfterCollectFuncBags != 0 )
        (*libGAP_AfterCollectFuncBags)();


#ifdef DEBUG_MASTERPOINTERS
    libGAP_CheckMasterPointers();
#endif
    
    /* Possibly advise the operating system about unused pages:            */
    libGAP_SyMAdviseFree();

    /* return success                                                      */
    return 1;
}


/****************************************************************************
**
*F  CheckMasterPointers() . . . . do consistency checks on the masterpointers
**
*/

void libGAP_CheckMasterPointers( void )
{
  libGAP_Bag *ptr;
  for (ptr = libGAP_MptrBags; ptr < libGAP_OldBags; ptr++)
    {
      if (*ptr != (libGAP_Bag)0 &&             /* bottom of free chain */
          *ptr != (libGAP_Bag)libGAP_NewWeakDeadBagMarker &&
          *ptr != (libGAP_Bag)libGAP_OldWeakDeadBagMarker &&
          (((libGAP_Bag *)*ptr < libGAP_MptrBags &&
            (libGAP_Bag *)*ptr > libGAP_AllocBags) ||
           (libGAP_UInt)(*ptr) % sizeof(libGAP_Bag) != 0))
        (*libGAP_AbortFuncBags)("Bad master pointer detected in check");
    }
}


/****************************************************************************
**
*F  SwapMasterPoint( <bag1>, <bag2> ) . . . swap pointer of <bag1> and <bag2>
*/
void libGAP_SwapMasterPoint (
    libGAP_Bag                 bag1,
    libGAP_Bag                 bag2 )
{
    libGAP_Bag *               ptr1;
    libGAP_Bag *               ptr2;

    if ( bag1 == bag2 )
        return;

    /* get the pointers                                                    */
    ptr1 = libGAP_PTR_BAG(bag1);
    ptr2 = libGAP_PTR_BAG(bag2);

    /* check and update the link field and changed bags                    */
    if ( libGAP_PTR_BAG(bag1)[-1] == bag1 && libGAP_PTR_BAG(bag2)[-1] == bag2 ) {
        libGAP_PTR_BAG(bag1)[-1] = bag2;
        libGAP_PTR_BAG(bag2)[-1] = bag1;
    }
    else if ( libGAP_PTR_BAG(bag1)[-1] == bag1 ) {
        libGAP_PTR_BAG(bag1)[-1] = libGAP_ChangedBags;
        libGAP_ChangedBags = bag1; 
    }
    else if ( libGAP_PTR_BAG(bag2)[-1] == bag2 ) {
        libGAP_PTR_BAG(bag2)[-1] = libGAP_ChangedBags;
        libGAP_ChangedBags = bag2; 
    }

    /* swap them                                                           */
    libGAP_PTR_BAG(bag1) = ptr2;
    libGAP_PTR_BAG(bag2) = ptr1;
}



/****************************************************************************
**

*F  BID(<bag>)  . . . . . . . . . . . .  bag identifier (as unsigned integer)
*F  IS_BAG(<bid>) . . . . . .  test whether a bag identifier identifies a bag
*F  BAG(<bid>)  . . . . . . . . . . . . . . . . . . bag (from bag identifier)
*F  TNUM_BAG(<bag>) . . . . . . . . . . . . . . . . . . . . . . type of a bag
*F  SIZE_BAG(<bag>) . . . . . . . . . . . . . . . . . . . . . . size of a bag
*F  PTR_BAG(<bag>)  . . . . . . . . . . . . . . . . . . . .  pointer to a bag
*F  ELM_BAG(<bag>,<i>)  . . . . . . . . . . . . . . . <i>-th element of a bag
*F  SET_ELM_BAG(<bag>,<i>,<elm>)  . . . . . . . . set <i>-th element of a bag
**
**  'BID', 'IS_BAG', 'BAG', 'TNUM_BAG', 'TNAM_BAG', 'PTR_BAG', 'ELM_BAG', and
**  'SET_ELM_BAG' are functions to support  debugging.  They are not intended
**  to be used  in an application  using {\Gasman}.  Note  that the functions
**  'TNUM_BAG', 'SIZE_BAG', and 'PTR_BAG' shadow the macros of the same name,
**  which are usually not available in a debugger.
*/

#ifdef  DEBUG_FUNCTIONS_BAGS

#undef  libGAP_TNUM_BAG
#undef  libGAP_SIZE_BAG
#undef  libGAP_PTR_BAG

libGAP_UInt libGAP_BID (
    libGAP_Bag                 bag )
{
    return (libGAP_UInt) bag;
}


libGAP_Bag libGAP_BAG (
    libGAP_UInt                bid )
{
    if ( libGAP_IS_BAG(bid) )
        return (libGAP_Bag) bid;
    else
        return (libGAP_Bag) 0;
}

libGAP_UInt libGAP_TNUM_BAG (
    libGAP_Bag                 bag )
{
    return (*(*(bag)-3) & 0xFFL);
}

const libGAP_Char * libGAP_TNAM_BAG (
    libGAP_Bag                 bag )
{
    return libGAP_InfoBags[ (*(*(bag)-3) & 0xFFL) ].name;
}

libGAP_UInt libGAP_SIZE_BAG (
    libGAP_Bag                 bag )
{
    return (*(*(bag)-2));
}

libGAP_Bag * libGAP_PTR_BAG (
    libGAP_Bag                 bag )
{
    return (*(libGAP_Bag**)(bag));
}

libGAP_UInt libGAP_ELM_BAG (
    libGAP_Bag                 bag,
    libGAP_UInt                i )
{
    return (libGAP_UInt) ((*(libGAP_Bag**)(bag))[i]);
}

libGAP_UInt libGAP_SET_ELM_BAG (
    libGAP_Bag                 bag,
    libGAP_UInt                i,
    libGAP_UInt                elm )
{
    (*(libGAP_Bag**)(bag))[i] = (libGAP_Bag) elm;
    return elm;
}

#endif


/****************************************************************************
**
*E  gasman.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
