/****************************************************************************
**
*W  dteval.c                    GAP source                  Wolfgang Merkwitz
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the part of the deep thought package which uses the
**  deep thought polynomials to multiply in nilpotent groups.
**
**  The deep thought polynomials are stored in the list <dtpols> where
**  <dtpols>[i] contains the polynomials f_{i1},...,f_{in}.
**  <dtpols>[i] is a record consisting of the components <evlist> and
**  <evlistvec>. <evlist> is a list of all deep thought monomials occuring
**  in the polynomials f_{i1},...,f_{in}. <evlistvec>is a list of vectors
**  describing the coefficients of the corresponding deep thought monomials
**  in the polynomials f_{i1},..,f_{in}. For example when a pair [j,k]
**  occurs in <dtpols>[i].<evlistvec>[l]  then the deep thought monomial
**  <dtpols>[i].<evlist>[l] occurs in f_{ij} with the coefficient k.
**  If the polynomials f_{i1},..,f_{in} are trivial i.e. f_{ii} = x_i + y_i
**  and f_{ij} = x_j (j<>i),  then <dtpols>[i] is either 1 or 0. <dtpols>[i]
**  is 0 if also the polynomials f_{m1},...,f_{mn} for (m > i) are trivial .
*/
#include       "system.h"


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */
#include        "bool.h"                /* booleans                        */
#include        "calls.h"               /* generic call mechanism          */
#include        "gap.h"                 /* error handling, initialisation  */
#include        "gvars.h"               /* global variables                */
#include        "precord.h"             /* plain records                   */
#include        "records.h"             /* generic records                 */
#include        "integer.h"             /* integers                        */
#include        "dt.h"                  /* deep thought                    */
#include        "objcftl.h"             /* from the left collect           */

#include        "dteval.h"              /* deep though evaluation          */

#define   libGAP_CELM(list, pos)      (  libGAP_INT_INTOBJ( libGAP_ELM_PLIST(list, pos) ) )

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listfunc.h"            /* functions for generic lists     */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */


static int             libGAP_evlist, libGAP_evlistvec;

extern libGAP_Obj             libGAP_ShallowCopyPlist( libGAP_Obj  list );


/****************************************************************************
**

*F  MultGen( <xk>, <gen>, <power>, <dtpols> )
**
**  MultGen multiplies the word given by the exponent vector <xk> with
**  g_<gen>^<power> by evaluating the deep thought polynomials. The result
**  is an ordered word and stored in <xk>.
*/

/* See below: */
libGAP_Obj     libGAP_Evaluation( libGAP_Obj vec, libGAP_Obj xk, libGAP_Obj power );

void       libGAP_MultGen(
                    libGAP_Obj     xk,
                    libGAP_UInt    gen,
                    libGAP_Obj     power,
                    libGAP_Obj     dtpols    )
{
    libGAP_UInt  i, j, len, len2;
    libGAP_Obj   copy, sum, sum1, sum2, prod, ord, help;

    if ( libGAP_IS_INTOBJ(power)  &&  libGAP_INT_INTOBJ(power) == 0 )
        return;
    sum = libGAP_SumInt(libGAP_ELM_PLIST(xk, gen),  power);
    if ( libGAP_IS_INTOBJ( libGAP_ELM_PLIST(dtpols, gen) ) )
    {
        /* if f_{<gen>1},...,f_{<gen>n} are trivial we only have to add
        ** <power> to <xk>[ <gen> ].                                     */
        libGAP_SET_ELM_PLIST(xk, gen, sum);
        libGAP_CHANGED_BAG(xk);
        return;
    }
    copy = libGAP_ShallowCopyPlist(xk);
    /* first add <power> to <xk>[ gen> ].                                */
    libGAP_SET_ELM_PLIST(xk, gen, sum);
    libGAP_CHANGED_BAG(xk);     
    sum = libGAP_ElmPRec( libGAP_ELM_PLIST(dtpols, gen), libGAP_evlist );
    sum1 = libGAP_ElmPRec( libGAP_ELM_PLIST(dtpols, gen), libGAP_evlistvec);
    len = libGAP_LEN_PLIST(sum);
    for ( i=1;
          i <= len;
          i++ )
    {
        /* evaluate the deep thought monomial <sum>[<i>],        */
        ord = libGAP_Evaluation( libGAP_ELM_PLIST( sum, i), copy, power  );
        if ( !libGAP_IS_INTOBJ(ord)  ||  libGAP_INT_INTOBJ(ord) != 0 )
        {
            help = libGAP_ELM_PLIST(sum1, i);
            len2 = libGAP_LEN_PLIST(help);
            for ( j=1; 
                  j < len2;
                  j+=2    )
            {
                /* and add the result multiplicated with the right coefficient
                ** to <xk>[ <help>[j] ].                                    */
                prod = libGAP_ProdInt( ord, libGAP_ELM_PLIST(  help, j+1 ) );
                sum2 = libGAP_SumInt(libGAP_ELM_PLIST( xk, libGAP_CELM( help,j ) ),
                              prod);
                libGAP_SET_ELM_PLIST(xk, libGAP_CELM( help, j ),  
                              sum2 );
                libGAP_CHANGED_BAG(xk);
            }
        }
    }
}



/****************************************************************************
**
*F  Evaluation( <vec>, <xk>, <power>)
**
**  Evaluation evaluates the deep thought monomial <vec> at the entries in
**  <xk> and at <power>.
*/

libGAP_Obj     libGAP_Evaluation(
                    libGAP_Obj     vec,
                    libGAP_Obj     xk,
                    libGAP_Obj     power      )
{
    libGAP_UInt i, len;
    libGAP_Obj  prod, help;

    if ( libGAP_IS_INTOBJ(power)  &&  libGAP_INT_INTOBJ(power) > 0  &&  
         power < libGAP_ELM_PLIST(vec, 6)     )
        return libGAP_INTOBJ_INT(0);
    prod = libGAP_binomial(power, libGAP_ELM_PLIST(vec, 6) );
    len = libGAP_LEN_PLIST(vec);
    for (i=7; i < len; i+=2)
    {
        help = libGAP_ELM_PLIST(xk, libGAP_CELM(vec, i) );
        if ( libGAP_IS_INTOBJ( help )                       &&
             ( libGAP_INT_INTOBJ(help) == 0                 ||
               ( libGAP_INT_INTOBJ(help) > 0  &&  help < libGAP_ELM_PLIST(vec, i+1) )  ) )
            return libGAP_INTOBJ_INT(0);
        prod = libGAP_ProdInt( prod, libGAP_binomial( help, libGAP_ELM_PLIST(vec, i+1) ) );
    }
    return prod;
}



/****************************************************************************
**
*F  Multbound( <xk>, <y>, <anf>, <end>, <dtpols> )
**
**  Multbound multiplies the word given by the exponent vector <xk> with
**  <y>{ [<anf>..<end>] } by evaluating the deep thought polynomials <dtpols>
**  The result is an ordered word and is stored in <xk>.
*/

void        libGAP_Multbound(
                  libGAP_Obj    xk,
                  libGAP_Obj    y,
                  libGAP_Int    anf,
                  libGAP_Int    end,
                  libGAP_Obj    dtpols  )
{
    int     i;

    for (i=anf; i < end; i+=2)
        libGAP_MultGen(xk, libGAP_CELM( y, i), libGAP_ELM_PLIST( y, i+1) , dtpols);
}



/****************************************************************************
**
*F  Multiplybound( <x>, <y>, <anf>, <end>, <dtpols> )
**
**  Multiplybound returns the product of the word <x> with the word
**  <y>{ [<anf>..<end>] } by evaluating the deep thought polynomials <dtpols>.
**  The result is an ordered word.
*/

libGAP_Obj       libGAP_Multiplybound(
                     libGAP_Obj      x,
                     libGAP_Obj      y,       
                     libGAP_Int      anf,
                     libGAP_Int      end,
                     libGAP_Obj      dtpols  )
{
    libGAP_UInt   i, j, k, len, help;
    libGAP_Obj    xk, res, sum;

    if ( libGAP_LEN_PLIST( x ) == 0 )
        return y;
    if ( anf > end )
        return x;
    /* first deal with the case that <y>{ [<anf>..<end>] } lies in the center
    ** of the group defined by <dtpols>                                    */
    if ( libGAP_IS_INTOBJ( libGAP_ELM_PLIST(dtpols, libGAP_CELM(y, anf) ) )   &&
         libGAP_CELM(dtpols, libGAP_CELM(y, anf) ) == 0                          )
    {
        res = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*libGAP_LEN_PLIST( dtpols ) );
        len = libGAP_LEN_PLIST(x);
        j = 1;
        k = anf;
        i = 1;
        while ( j<len && k<end )
        {
            if ( libGAP_ELM_PLIST(x, j) == libGAP_ELM_PLIST(y, k) )
            {
                sum = libGAP_SumInt( libGAP_ELM_PLIST(x, j+1), libGAP_ELM_PLIST(y, k+1) );
                libGAP_SET_ELM_PLIST(res, i, libGAP_ELM_PLIST(x, j) );
                libGAP_SET_ELM_PLIST(res, i+1, sum );
                j+=2;
                k+=2;
            }
            else if ( libGAP_ELM_PLIST(x, j) < libGAP_ELM_PLIST(y, k) )
            {
                libGAP_SET_ELM_PLIST(res, i, libGAP_ELM_PLIST(x, j) );
                libGAP_SET_ELM_PLIST(res, i+1, libGAP_ELM_PLIST(x, j+1) );
                j+=2;
            }
            else
            {
                libGAP_SET_ELM_PLIST(res, i, libGAP_ELM_PLIST(y, k) );
                libGAP_SET_ELM_PLIST(res, i+1, libGAP_ELM_PLIST(y, k+1) );
                k+=2;
            }
            libGAP_CHANGED_BAG(res);
            i+=2;
        }
        if ( j>=len )
            while ( k<end )
            {
                libGAP_SET_ELM_PLIST(res, i, libGAP_ELM_PLIST(y, k) );
                libGAP_SET_ELM_PLIST(res, i+1, libGAP_ELM_PLIST(y, k+1 ) );
                libGAP_CHANGED_BAG(res);
                k+=2;
                i+=2;
            }
        else
            while ( j<len )
            {
                libGAP_SET_ELM_PLIST(res, i, libGAP_ELM_PLIST(x, j) );
                libGAP_SET_ELM_PLIST(res, i+1, libGAP_ELM_PLIST(x, j+1) );
                libGAP_CHANGED_BAG(res);
                j+=2;
                i+=2;
            }
        libGAP_SET_LEN_PLIST(res, i-1);
        libGAP_SHRINK_PLIST(res, i-1);
        return res;
    }
    len = libGAP_LEN_PLIST(dtpols);
    help = libGAP_LEN_PLIST(x);
    /* convert <x> into a exponent vector                             */
    xk = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
    libGAP_SET_LEN_PLIST(xk, len );
    j = 1;
    for (i=1; i <= len; i++)
    {
        if ( j >= help  ||  i < libGAP_CELM(x, j) )
            libGAP_SET_ELM_PLIST(xk, i, libGAP_INTOBJ_INT(0) );
        else
        {
            libGAP_SET_ELM_PLIST(xk, i, libGAP_ELM_PLIST(x, j+1) );
            j+=2;
        }
    }
    /* let Multbound do the work                                       */
    libGAP_Multbound(xk, y, anf, end, dtpols);
    /* finally convert the result back into a word                     */
    res = libGAP_NEW_PLIST(libGAP_T_PLIST, 2*len);
    j = 0;
    for (i=1; i <= len; i++)
    {
        if ( !( libGAP_IS_INTOBJ( libGAP_ELM_PLIST(xk, i) )  &&  libGAP_CELM(xk, i) == 0 ) )
        {
            j+=2;
            libGAP_SET_ELM_PLIST(res, j-1, libGAP_INTOBJ_INT(i) );
            libGAP_SET_ELM_PLIST(res, j, libGAP_ELM_PLIST(xk, i) );
        }
    }
    libGAP_SET_LEN_PLIST(res, j);
    libGAP_SHRINK_PLIST(res, j);
    return res;
}



/****************************************************************************
**
*F  FuncMultiply( <self>, <x>, <y>, <dtpols> )
**
**  FuncMultiply implements the internal function
**
*F  Multiply( <x>, <y>, <dtpols> ).
**
**  Multiply returns the product of the words <x> and <y> as ordered word
**  by evaluating the deep thought polynomials <dtpols>.
*/

libGAP_Obj      libGAP_FuncMultiply(
                       libGAP_Obj      self,
                       libGAP_Obj      x,
                       libGAP_Obj      y,
                       libGAP_Obj      dtpols      )
{
    return libGAP_Multiplybound(x, y, 1, libGAP_LEN_PLIST(y), dtpols);
}



/****************************************************************************
**
*F  Power( <x>, <n>, <dtpols> )
**
**  Power returns the <n>-th power of the word <x> as ordered word by
**  evaluating the deep thought polynomials <dtpols>.
*/

/* See below: */
libGAP_Obj libGAP_Solution( libGAP_Obj x, libGAP_Obj y, libGAP_Obj dtpols );

libGAP_Obj      libGAP_Power(
                libGAP_Obj         x,
                libGAP_Obj         n,
                libGAP_Obj         dtpols     )
{
    libGAP_Obj     res, m, y;
    libGAP_UInt    i,len;

    if ( libGAP_LEN_PLIST(x) == 0 )
        return x;
    /* first deal with the case that <x> lies in the centre of the group
    ** defined by <dtpols>                                              */
    if ( libGAP_IS_INTOBJ( libGAP_ELM_PLIST( dtpols, libGAP_CELM(x, 1) ) )   &&
         libGAP_CELM( dtpols, libGAP_CELM(x, 1) ) == 0                     )
    {
        len = libGAP_LEN_PLIST(x);
        res = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
        libGAP_SET_LEN_PLIST(res, len );
        for (i=2;i<=len;i+=2)
        {
            m = libGAP_ProdInt( libGAP_ELM_PLIST(x, i), n );
            libGAP_SET_ELM_PLIST(res, i, m );
            libGAP_SET_ELM_PLIST(res, i-1, libGAP_ELM_PLIST(x, i-1) );
            libGAP_CHANGED_BAG( res );
        }
        return res;
    }
    /* if <n> is a negative integer compute ( <x>^-1 )^(-<n>)           */
    if (  libGAP_TNUM_OBJ(n) == libGAP_T_INTNEG  ||  libGAP_INT_INTOBJ(n) < 0  ) 
    {
        y = libGAP_NEW_PLIST( libGAP_T_PLIST, 0);
        libGAP_SET_LEN_PLIST(y, 0);
        return  libGAP_Power( libGAP_Solution(x, y, dtpols), 
                       libGAP_ProdInt(libGAP_INTOBJ_INT(-1), n),   dtpols  );    
    }
    res = libGAP_NEW_PLIST(libGAP_T_PLIST, 2);
    libGAP_SET_LEN_PLIST(res, 0);
    if ( libGAP_IS_INTOBJ(n)  &&  libGAP_INT_INTOBJ(n) == 0  )
        return res;
    /* now use the russian peasant rule to get the result               */
    while( libGAP_LtInt(libGAP_INTOBJ_INT(0), n) )
    {
        len = libGAP_LEN_PLIST(x);
        if ( libGAP_ModInt(n, libGAP_INTOBJ_INT(2) ) == libGAP_INTOBJ_INT(1)  )
            res = libGAP_Multiplybound(res, x, 1, len, dtpols);
        if ( libGAP_LtInt(libGAP_INTOBJ_INT(1), n) )
            x = libGAP_Multiplybound(x, x, 1, len, dtpols);
        n = libGAP_QuoInt(n, libGAP_INTOBJ_INT(2) );
    }
    return res;
}



/****************************************************************************
**
*F  FuncPower( <self>, <x>, <n>, <dtpols> )
**
**  FuncPower implements the internal function
**
*F  Pover( <x>, <n>, <dtpols> )
**
**  Pover returns the <n>-th power of the word <x> by evaluating the deep
**  thought pols <dtpols>. The result is an oredered word.
*/

libGAP_Obj        libGAP_FuncPower(
                      libGAP_Obj     self,
                      libGAP_Obj     x,
                      libGAP_Obj     n,
                      libGAP_Obj     dtpols     )
{
    return libGAP_Power(x, n, dtpols);
}



/****************************************************************************
**
*F  Solution( <x>, <y>, <dtpols> )
**
**  Solution returns a solution for the equation <x>*a = <y> by evaluating
**  the deep thought polynomials <dtpols>. The result is an ordered word.
*/ 

libGAP_Obj      libGAP_Solution( libGAP_Obj       x,
                   libGAP_Obj       y,
                   libGAP_Obj       dtpols  )

{
    libGAP_Obj    xk, res, m;
    libGAP_UInt   i,j,k, len1, len2;

    if ( libGAP_LEN_PLIST(x) == 0)
        return y;
    /* first deal with the case that <x> and <y> ly in the centre of the
    ** group defined by <dtpols>.                                       */
    if ( libGAP_IS_INTOBJ( libGAP_ELM_PLIST( dtpols, libGAP_CELM(x, 1) )  )  &&
         libGAP_CELM( dtpols, libGAP_CELM(x, 1) ) == 0                &&
         (  libGAP_LEN_PLIST(y) == 0                              ||
            (  libGAP_IS_INTOBJ( libGAP_ELM_PLIST( dtpols, libGAP_CELM(y, 1) )  )  &&
               libGAP_CELM( dtpols, libGAP_CELM(y, 1) ) == 0                    )  )   )
    {
        res = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*libGAP_LEN_PLIST( dtpols ) );
        i = 1;
        j = 1;
        k = 1;
        len1 = libGAP_LEN_PLIST(x);
        len2 = libGAP_LEN_PLIST(y);
        while ( j < len1 && k < len2 )
        {
            if ( libGAP_ELM_PLIST(x, j) == libGAP_ELM_PLIST(y, k) )
            {
                m = libGAP_DiffInt( libGAP_ELM_PLIST(y, k+1), libGAP_ELM_PLIST(x, j+1) );
                libGAP_SET_ELM_PLIST( res, i, libGAP_ELM_PLIST(x, j) );
                libGAP_SET_ELM_PLIST( res, i+1, m );
                libGAP_CHANGED_BAG( res );
                i+=2; j+=2; k+=2;
            }
            else if ( libGAP_CELM(x, j) < libGAP_CELM(y, k) )
            {
                m = libGAP_ProdInt( libGAP_INTOBJ_INT(-1), libGAP_ELM_PLIST(x, j+1) );
                libGAP_SET_ELM_PLIST( res, i, libGAP_ELM_PLIST(x, j) );
                libGAP_SET_ELM_PLIST( res, i+1, m );
                libGAP_CHANGED_BAG( res );
                i+=2; j+=2;
            }
            else
            {
                libGAP_SET_ELM_PLIST( res, i, libGAP_ELM_PLIST(y, k) );
                libGAP_SET_ELM_PLIST( res, i+1, libGAP_ELM_PLIST(y, k+1) );
                libGAP_CHANGED_BAG( res );
                i+=2; k+=2;
            }
        }
        if ( j < len1 )
            while( j < len1 )
            {
                m = libGAP_ProdInt( libGAP_INTOBJ_INT(-1), libGAP_ELM_PLIST( x, j+1 ) );
                libGAP_SET_ELM_PLIST( res, i, libGAP_ELM_PLIST(x, j) );
                libGAP_SET_ELM_PLIST( res, i+1, m );
                libGAP_CHANGED_BAG( res );
                i+=2; j+=2;
            }
        else
            while( k < len2 )
            {
                libGAP_SET_ELM_PLIST( res, i ,libGAP_ELM_PLIST(y, k) );
                libGAP_SET_ELM_PLIST( res, i+1, libGAP_ELM_PLIST(y, k+1) );
                libGAP_CHANGED_BAG( res );
                i+=2; k+=2;
            }
        libGAP_SET_LEN_PLIST( res, i-1 );
        libGAP_SHRINK_PLIST( res, i-1);
        return res;
    }
    /* convert <x> into an exponent vector                           */
    xk = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_LEN_PLIST(dtpols) );
    libGAP_SET_LEN_PLIST(xk, libGAP_LEN_PLIST(dtpols) );
    j = 1;
    for (i=1; i <= libGAP_LEN_PLIST(dtpols); i++)
    {
        if ( j >= libGAP_LEN_PLIST(x)  ||  i < libGAP_CELM(x, j) )
            libGAP_SET_ELM_PLIST(xk, i, libGAP_INTOBJ_INT(0) );
        else
        {
            libGAP_SET_ELM_PLIST(xk, i, libGAP_ELM_PLIST(x, j+1) );
            j+=2;
        }
    }
    res = libGAP_NEW_PLIST( libGAP_T_PLIST, 2*libGAP_LEN_PLIST( xk ) );
    j = 1;
    k = 1;
    len1 = libGAP_LEN_PLIST(xk);
    len2 = libGAP_LEN_PLIST(y);
    for (i=1; i <= len1; i++)
    {
        if ( k < len2   &&   i == libGAP_CELM(y, k)  )
        {
            if  ( !libGAP_EqInt( libGAP_ELM_PLIST(xk, i), libGAP_ELM_PLIST(y, k+1) )  )
            {
                m = libGAP_DiffInt( libGAP_ELM_PLIST(y, k+1), libGAP_ELM_PLIST(xk, i) );
                libGAP_SET_ELM_PLIST(res, j, libGAP_INTOBJ_INT(i) );
                libGAP_SET_ELM_PLIST(res, j+1, m);
                libGAP_CHANGED_BAG(res);
                libGAP_MultGen(xk, i, m, dtpols);
                j+=2;
            }
            k+=2;
        }
        else if ( !libGAP_IS_INTOBJ( libGAP_ELM_PLIST(xk, i) )  ||  libGAP_CELM( xk, i ) != 0 )
        {
            m = libGAP_ProdInt( libGAP_INTOBJ_INT(-1), libGAP_ELM_PLIST(xk, i) );
            libGAP_SET_ELM_PLIST( res, j, libGAP_INTOBJ_INT(i) );
            libGAP_SET_ELM_PLIST( res, j+1, m );
            libGAP_CHANGED_BAG(res);
            libGAP_MultGen(xk, i, m, dtpols);
            j+=2;
        }
    }
    libGAP_SET_LEN_PLIST(res, j-1);
    libGAP_SHRINK_PLIST(res, j-1);
    return res;
}



/****************************************************************************
**
*F  Commutator( <x>, <y>, <dtpols> )
**
**  Commutator returns the commutator of the word <x> and <y> by evaluating
**  the deep thought polynomials <dtpols>.
*/

libGAP_Obj       libGAP_Commutator( libGAP_Obj     x,
                      libGAP_Obj     y,
                      libGAP_Obj     dtpols  )
{
    libGAP_Obj    res, help;

    res = libGAP_Multiplybound(x, y, 1, libGAP_LEN_PLIST(y), dtpols);
    help = libGAP_Multiplybound(y, x, 1, libGAP_LEN_PLIST(x), dtpols);
    res = libGAP_Solution(help, res, dtpols);
    return res;
}



/****************************************************************************
**
*F  Conjugate( <x>, <y>, <dtpols> )
**
**  Conjugate returns <x>^<y> for the words <x> and <y> by evaluating the 
**  deep thought polynomials <dtpols>. The result is an ordered word.
*/

libGAP_Obj       libGAP_Conjugate( libGAP_Obj     x,
                     libGAP_Obj     y,
                     libGAP_Obj     dtpols  )
{
    libGAP_Obj    res;

    res = libGAP_Multiplybound(x, y, 1, libGAP_LEN_PLIST(y), dtpols);
    res = libGAP_Solution(y, res, dtpols);
    return res;
}



/****************************************************************************
**
*F  Multiplyboundred( <x>, <y>, <anf>, <end>, <pcp> )
**
**  Multiplyboundred returns the product of the words <x> and <y>. The result
**  is an ordered word with the additional property that all word exponents
**  are reduced modulo the the corresponding generator orders given by the
**  deep thought rewriting system <pcp>..
*/

libGAP_Obj       libGAP_Multiplyboundred( libGAP_Obj     x,
                            libGAP_Obj     y,
                            libGAP_UInt    anf,
                            libGAP_UInt    end,
                            libGAP_Obj     pcp )
{
    libGAP_Obj   orders, res, mod, c;
    libGAP_UInt  i, len, len2, help;

    orders = libGAP_ELM_PLIST(pcp, libGAP_PC_ORDERS);
    res = libGAP_Multiplybound(x,y,anf, end, libGAP_ELM_PLIST( pcp, libGAP_PC_DEEP_THOUGHT_POLS) );
    len = libGAP_LEN_PLIST(res);
    len2 = libGAP_LEN_PLIST(orders);
    for (i=2; i<=len; i+=2)
        if ( (help=libGAP_CELM(res, i-1)) <= len2        &&
             ( c=libGAP_ELM_PLIST( orders, help )) != 0 )
        {
            mod = libGAP_ModInt( libGAP_ELM_PLIST(res, i), c );
            libGAP_SET_ELM_PLIST( res, i, mod);
            libGAP_CHANGED_BAG(res);
        }
    return res;
}



/****************************************************************************
**
*F  Powerred( <x>, <n>, <pcp>
**
**  Powerred returns the <n>-th power of the word <x>. The result is an
**  ordered word with the additional property that all word exponents are
**  reduced modulo the generator orders given by the deep thought rewriting
**  system <pcp>.
*/

libGAP_Obj       libGAP_Powerred( libGAP_Obj       x,
                    libGAP_Obj       n,
                    libGAP_Obj       pcp  )
{
    libGAP_Obj   orders, res, mod, c;
    libGAP_UInt  i, len, len2,help;

    orders = libGAP_ELM_PLIST(pcp, libGAP_PC_ORDERS);
    res = libGAP_Power(x, n, libGAP_ELM_PLIST( pcp, libGAP_PC_DEEP_THOUGHT_POLS) );
    len = libGAP_LEN_PLIST(res);
    len2 = libGAP_LEN_PLIST(orders);
    for (i=2; i<=len; i+=2)
        if ( (help=libGAP_CELM(res, i-1)) <= len2         &&
             ( c=libGAP_ELM_PLIST( orders, help )) != 0 )
        {
            mod = libGAP_ModInt( libGAP_ELM_PLIST(res, i), c );
            libGAP_SET_ELM_PLIST( res, i, mod);
            libGAP_CHANGED_BAG(res);
        }
    return res;
}



/****************************************************************************
**
*F  Solutionred( <x>, <y>, <pcp> )
**
**  Solutionred returns the solution af the equation <x>*a = <y>.  The result
**  is an ordered word with the additional property that all word exponents
**  are reduced modulo the generator orders given by the deep thought
**  rewriting system <pcp>.
*/

libGAP_Obj       libGAP_Solutionred( libGAP_Obj       x,
                       libGAP_Obj       y,
                       libGAP_Obj       pcp  )
{
    libGAP_Obj   orders, res, mod, c;
    libGAP_UInt  i, len, len2, help;

    orders = libGAP_ELM_PLIST(pcp, libGAP_PC_ORDERS);
    res = libGAP_Solution(x, y, libGAP_ELM_PLIST( pcp, libGAP_PC_DEEP_THOUGHT_POLS) );
    len = libGAP_LEN_PLIST(res);
    len2 = libGAP_LEN_PLIST(orders);
    for (i=2; i<=len; i+=2)
        if ( (help=libGAP_CELM(res, i-1)) <= len2       &&
             ( c=libGAP_ELM_PLIST( orders, help )) != 0 )
        {
            mod = libGAP_ModInt( libGAP_ELM_PLIST(res, i), c );
            libGAP_SET_ELM_PLIST( res, i, mod);
            libGAP_CHANGED_BAG(res);
        }
    return res;
}



/****************************************************************************
**
**  Commutatorred( <x>, <y>, <pcp> )
**
**  Commutatorred returns the commutator of the words <x> and <y>. The result
**  is an ordered word with the additional property that all word exponents
**  are reduced modulo the corresponding generator orders given by the deep
**  thought rewriting system <pcp>.
*/

libGAP_Obj       libGAP_Commutatorred( libGAP_Obj    x,
                         libGAP_Obj    y,
                         libGAP_Obj    pcp  )
{
    libGAP_Obj    orders, mod, c, res;
    libGAP_UInt   i, len, len2, help;

    orders = libGAP_ELM_PLIST(pcp, libGAP_PC_ORDERS);
    res = libGAP_Commutator(x, y, libGAP_ELM_PLIST( pcp, libGAP_PC_DEEP_THOUGHT_POLS) );
    len = libGAP_LEN_PLIST(res);
    len2 = libGAP_LEN_PLIST(orders);
    for (i=2; i<=len; i+=2)
        if ( (help=libGAP_CELM(res, i-1)) <= len2         &&
             ( c=libGAP_ELM_PLIST( orders, help )) != 0 )
        {
            mod = libGAP_ModInt( libGAP_ELM_PLIST(res, i), c );
            libGAP_SET_ELM_PLIST( res, i, mod);
            libGAP_CHANGED_BAG(res);
        }
    return res;
}



/****************************************************************************
**
*F  Conjugate( <x>, <y>, <pcp> )
**
**  Conjugate returns <x>^<y> for the words <x> and <y>. The result is an
**  ordered word with the additional property that all word exponents are
**  reduced modulo the corresponding generator orders given by the deep
**  thought rewriting system <pcp>.
*/

libGAP_Obj       libGAP_Conjugatered( libGAP_Obj    x,
                         libGAP_Obj    y,
                         libGAP_Obj    pcp  )
{
    libGAP_Obj    orders, mod, c, res;
    libGAP_UInt   i, len, len2, help;

    orders = libGAP_ELM_PLIST(pcp, libGAP_PC_ORDERS);
    res = libGAP_Conjugate(x, y, libGAP_ELM_PLIST( pcp, libGAP_PC_DEEP_THOUGHT_POLS) );
    len = libGAP_LEN_PLIST(res);
    len2 = libGAP_LEN_PLIST(orders);
    for (i=2; i<=len; i+=2)
        if ( (help=libGAP_CELM(res, i-1)) <= len2         &&
             ( c=libGAP_ELM_PLIST( orders, help )) != 0 )
        {
            mod = libGAP_ModInt( libGAP_ELM_PLIST(res, i), c );
            libGAP_SET_ELM_PLIST( res, i, mod);
            libGAP_CHANGED_BAG(res);
        }
    return res;
}



/****************************************************************************
**
**  compress( <list> )
**
**  compress removes pairs (n,0) from the list of GAP integers <list>.
*/

void     libGAP_compress( libGAP_Obj        list )
{    
    libGAP_UInt    i, skip, len;
    
    skip = 0;
    i = 2;
    len = libGAP_LEN_PLIST( list );
    while  ( i <= len )
    {
        while ( i<=len  &&  libGAP_CELM(list, i) == 0)
        {
            skip+=2;
            i+=2;
        }
        if ( i <= len )
        {
            libGAP_SET_ELM_PLIST(list, i-skip, libGAP_ELM_PLIST(list, i) );
            libGAP_SET_ELM_PLIST(list, i-1-skip, libGAP_ELM_PLIST( list, i-1 ) );
        }
        i+=2;
    }
    libGAP_SET_LEN_PLIST( list, len-skip );
    libGAP_CHANGED_BAG( list );
    libGAP_SHRINK_PLIST( list, len-skip );
}



/****************************************************************************
**
*F  Funccompress( <self>, <list> )
**
**  Funccompress implements the internal function Compress.
*/

libGAP_Obj      libGAP_Funccompress( libGAP_Obj         self, 
                       libGAP_Obj         list  )
{
    libGAP_compress(list);
    return  (libGAP_Obj)0;
}



/****************************************************************************
**
*F  ReduceWord( <x>, <pcp> )
**
**  ReduceWord reduces the ordered word <x> with respect to the deep thought
**  rewriting system <pcp> i.e after applying ReduceWord <x> is an ordered
**  word with exponents less than the corresponding relative orders given
**  by <pcp>.
*/

void     libGAP_ReduceWord( libGAP_Obj      x,
                      libGAP_Obj      pcp )   
{
    libGAP_Obj       powers, exponent;
    libGAP_Obj       deepthoughtpols, help, potenz, quo, mod, prel;
    libGAP_UInt      i,j,flag, len, gen, lenexp, lenpow;

    powers = libGAP_ELM_PLIST(pcp, libGAP_PC_POWERS);
    exponent = libGAP_ELM_PLIST(pcp, libGAP_PC_EXPONENTS);
    deepthoughtpols = libGAP_ELM_PLIST(pcp, libGAP_PC_DEEP_THOUGHT_POLS);
    len = **deepthoughtpols;
    lenexp = libGAP_LEN_PLIST(exponent);
    lenpow = libGAP_LEN_PLIST(powers);
    libGAP_GROW_PLIST(x, 2*len );
    flag = libGAP_LEN_PLIST(x);
    for (i=1; i<flag; i+=2)
    {
        if ( (gen = libGAP_CELM(x, i) ) <= lenexp              &&
             (potenz = libGAP_ELM_PLIST(exponent, gen) ) != 0                    )
        {
            quo = libGAP_ELM_PLIST(x, i+1);
            if  ( !libGAP_IS_INTOBJ(quo) || libGAP_INT_INTOBJ(quo) >= libGAP_INT_INTOBJ(potenz) || 
                  libGAP_INT_INTOBJ(quo)<0 )
            {
                /* reduce the exponent of the generator <gen>            */
                mod = libGAP_ModInt( quo, potenz );
                libGAP_SET_ELM_PLIST(x, i+1, mod);
                libGAP_CHANGED_BAG(x);
                if ( gen <= lenpow            &&
                     (prel = libGAP_ELM_PLIST( powers, gen) )  != 0  )
                {
                    if ( ( libGAP_IS_INTOBJ(quo) && libGAP_INT_INTOBJ(quo) >= libGAP_INT_INTOBJ(potenz) )   ||
                         libGAP_TNUM_OBJ(quo) == libGAP_T_INTPOS    )
                    {
                        help = libGAP_Powerred(  prel,
                                          libGAP_QuoInt(quo, potenz),
                                          pcp    );
                        help = libGAP_Multiplyboundred( help, x, i+2, flag, pcp);
                    }
                    else
                    {
                        quo = libGAP_INT_INTOBJ(mod) == 0? libGAP_QuoInt(quo,potenz):libGAP_SumInt(libGAP_QuoInt(quo, potenz),libGAP_INTOBJ_INT(-1));
                        help = libGAP_Powerred(  prel, 
                                          quo, 
                                          pcp );
                        help = libGAP_Multiplyboundred( help, x, i+2, flag, pcp);
                    }
                    len = libGAP_LEN_PLIST(help);
                    for (j=1; j<=len; j++)
                        libGAP_SET_ELM_PLIST(x, j+i+1, libGAP_ELM_PLIST(help, j) );
                    libGAP_CHANGED_BAG(x);
                    flag = i+len+1;
                    /*SET_LEN_PLIST(x, flag);*/
                }
            }
        }
    }
    libGAP_SET_LEN_PLIST(x, flag);
    libGAP_SHRINK_PLIST(x, flag);
    /* remove all syllables with exponent 0 from <x>.                  */
    libGAP_compress(x);
}



/****************************************************************************
**
*F  FuncDTMultiply( <self>, <x>, <y>, <pcp> )
**
**  FuncDTMultiply implements the internal function
**
*F  DTMultiply( <x>, <y>, <pcp> ).
**
**  DTMultiply returns the product of <x> and <y>. The result is reduced
**  with respect to the deep thought rewriting system <pcp>.
*/

libGAP_Obj       libGAP_FuncDTMultiply( libGAP_Obj      self,
                          libGAP_Obj      x,
                          libGAP_Obj      y,
                          libGAP_Obj      pcp    )
{
    libGAP_Obj res;

    if  ( libGAP_LEN_PLIST(x) == 0 )
        return y;
    if  ( libGAP_LEN_PLIST(y) == 0 )
        return x;
    res = libGAP_Multiplyboundred(x, y, 1, libGAP_LEN_PLIST(y), pcp);
    libGAP_ReduceWord(res, pcp);
    return res;
}



/****************************************************************************
**
*F  FuncDTPower( <self>, <x>, <n>, <pcp> )
**
**  FuncDTPower implements the internal function
**
*F  DTPower( <x>, <n>, <pcp> ).
**
**  DTPower returns the <n>-th power of the word <x>. The result is reduced
**  with respect to the deep thought rewriting system <pcp>.
*/

libGAP_Obj       libGAP_FuncDTPower( libGAP_Obj       self,
                       libGAP_Obj       x,
                       libGAP_Obj       n,
                       libGAP_Obj       pcp  )
{
    libGAP_Obj    res;

    res = libGAP_Powerred(x, n, pcp);
    libGAP_ReduceWord(res, pcp);
    return res;
}



/****************************************************************************
**
*F  FuncDTSolution( <self>, <x>, <y>, <pcp> )
**
**  FuncDTSolution implements the internal function
**
*F  DTSolution( <x>, <y>, <pcp> ).
**
**  DTSolution returns the solution of the equation <x>*a = <y>. The result
**  is reduced with respect to the deep thought rewriting system <pcp>.
*/

libGAP_Obj      libGAP_FuncDTSolution( libGAP_Obj     self,
                         libGAP_Obj     x,
                         libGAP_Obj     y,
                         libGAP_Obj     pcp )
{
    libGAP_Obj     res;

    if  ( libGAP_LEN_PLIST(x) == 0 )
        return y;
    res = libGAP_Solutionred(x, y, pcp);
    libGAP_ReduceWord(res, pcp);
    return res;
}



/****************************************************************************
**
*F  FuncDTCommutator( <self>, <x>, <y>. <pcp> )
**
**  FuncDTCommutator implements the internal function
**
*F  DTCommutator( <x>, <y>, <pcp> )
**
**  DTCommutator returns the commutator of the words <x> and <y>.  The result
**  is reduced with respect to the deep thought rewriting sytem <pcp>.
*/

libGAP_Obj        libGAP_FuncDTCommutator( libGAP_Obj      self,
                             libGAP_Obj      x,
                             libGAP_Obj      y,
                             libGAP_Obj      pcp  )
{
    libGAP_Obj   res;

    res = libGAP_Commutatorred(x, y, pcp);
    libGAP_ReduceWord(res, pcp);
    return res;
}



/****************************************************************************
**
*F  FuncConjugate( <self>, <x>, <y>, <pcp> )
**
**  FuncConjugate implements the internal function
**
*F  Conjugate( <x>, <y>, <pcp> ).
**
**  Conjugate returns <x>^<y> for the words <x> and <y>.  The result is
**  ewduced with respect to the deep thought rewriting system <pcp>.
*/

libGAP_Obj        libGAP_FuncDTConjugate( libGAP_Obj      self,
                            libGAP_Obj      x,
                            libGAP_Obj      y,
                            libGAP_Obj      pcp  )
{
    libGAP_Obj   res;

    if  ( libGAP_LEN_PLIST(y) == 0 )
        return x;
    res = libGAP_Conjugatered(x, y, pcp);
    libGAP_ReduceWord(res, pcp);
    return res;
}



/****************************************************************************
**
*F  FuncDTQuotient( <self>, <x>, <y>, <pcp> )
**
**  FuncDTQuotient implements the internal function
**
*F  DTQuotient( <x>, <y>, <pcp> ).
**
*F  DTQuotient returns the <x>/<y> for the words <x> and <y>. The result is
**  reduced with respect to the deep thought rewriting system <pcp>.
*/

libGAP_Obj       libGAP_FuncDTQuotient( libGAP_Obj      self,
                           libGAP_Obj      x,
                           libGAP_Obj      y,
                           libGAP_Obj      pcp )
{
    libGAP_Obj     help, res;

    if  ( libGAP_LEN_PLIST(y) == 0 )
        return x;
    help = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
    libGAP_SET_LEN_PLIST(help, 0);
    res = libGAP_Solutionred(y, help, pcp);
    res = libGAP_Multiplyboundred(x, res, 1, libGAP_LEN_PLIST(res), pcp);
    libGAP_ReduceWord(res, pcp);
    return(res);
}



/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "Compress", 1, "list",
      libGAP_Funccompress, "src/dteval.c:Compress" },

    { "Multiply", 3, "lword, rword, representatives",
      libGAP_FuncMultiply, "src/dteval.c:Multiply" },

    { "Pover", 3, "word, exponent, representatives",
      libGAP_FuncPower, "src/dteval.c:Pover" },

    { "DTMultiply", 3, "lword, rword, rws",
      libGAP_FuncDTMultiply, "src/dteval.c:DTMultiply" },

    { "DTPower", 3, "word, exponent, rws",
      libGAP_FuncDTPower, "src/dteval.c:DTPower" },

    { "DTSolution", 3, "lword, rword, rws",
      libGAP_FuncDTSolution, "src/dteval.c:DTSolution" },

    { "DTCommutator", 3, "lword, rword, rws",
      libGAP_FuncDTCommutator, "src/dteval.c:DTCommutator" },

    { "DTQuotient", 3, "lword, rword, rws",
      libGAP_FuncDTQuotient, "src/dteval.c:DTQuotient" },

    { "DTConjugate", 3, "lword, rword, rws",
      libGAP_FuncDTConjugate, "src/dteval.c:DTConjugate" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_evlist    = libGAP_RNamName("evlist");
    libGAP_evlistvec = libGAP_RNamName("evlistvec");

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return libGAP_PostRestore( libGAP_module );
}


/****************************************************************************
**
*F  InitInfoDTEvaluation()  . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "dteval",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoDTEvaluation ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  dteval.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
**
*/
