#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */
#include        "finfield.h"            /* finite fields and ff elements   */

#include        "bool.h"                /* booleans                        */
#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "listoper.h"            /* lists operations                */
#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "blister.h"             /* boolean lists                   */
#include        "string.h"              /* strings                         */

#include        "vector.h"              /* vectors */
#include        "listoper.h"              /* default list ops */

#include        "vec8bit.h"              /* GFQ vectors                     */

#include        "saveload.h"            /* saving and loading              */
#include        "opers.h"
#include        "integer.h"             /* integer functions needed for NUMBER_ */

#include        "vecgf2.h"              /* needed for the conversion to
             GF(2^n) n>1) */

#include        "code.h"               /* for TakeInterrupt */
#include        "stats.h" 

#ifndef DEBUG
#ifndef NDEBUG
#define NDEBUG 1
#endif
#endif
#include        <assert.h>

/****************************************************************************
**
**
*H  There is a representations of GFQ vectors with entries packed into
**  bytes, called IsVec8BitRep, which inherits from IsDataObjectRep
**  The 1st 4 bytes  stores the actual vector length (in field elements)
**  as a C integer. The 2nd component stores the field size as a C integer
**  The data bytes begin at the 3rd component
**  
**  In addition, this file defines format and access for the fieldinfo
**  objects which contain the meat-axe tables for the arithmetics
**
**  There is a special representation for matrices, all of whose rows
**  are immutable packed GFQ vectors over the same q, which is a positional
**  representation Is8BitMatrixRep. Some special methods for such matrices are
**  included here
** 
*/



/****************************************************************************
**
*F  IS_VEC8BIT_REP( <obj> )  . . . . . . check that <obj> is in 8bit GFQ vector rep
**
** #define IS_VEC8BIT_REP(obj) \
**  (TNUM_OBJ(obj)==T_DATOBJ && True == DoFilter(IsVec8bitRep,obj))
*/
libGAP_Obj libGAP_IsVec8bitRep;




/****************************************************************************
**
*F  LEN_VEC8BIT( <vec> ) . . . . .. . . . . . . .length of an 8 bit GF vector
**
**  'LEN_VEC8BIT' returns the logical length of the 8bit GFQ vector <list>,
**  as a C integer.
**
**  Note that 'LEN_VEC8BIT' is a macro, so do not call it with  arguments that
**  have sideeffects.
*/
#define libGAP_LEN_VEC8BIT(list)         ((libGAP_Int)(libGAP_ADDR_OBJ(list)[1]))

/****************************************************************************
**
*F  SET_LEN_VEC8BIT( <vec>, <len> )  . . . . set length of an 8 bit GF vector
**
**  'SET_LEN_VEC8BIT' sets the logical length of the 8bit GFQ vector <vec>,
**  to the C integer <len>.
**
*/
#define libGAP_SET_LEN_VEC8BIT(list,len)         ((libGAP_ADDR_OBJ(list)[1] = (libGAP_Obj)(len)))

/****************************************************************************
**
*F  FIELD_VEC8BIT( <vec> ) . . . .. . . . . .field size of an 8 bit GF vector
**
**  'FIELD_VEC8BIT' returns the field size Q of the 8bit GFQ vector <list>,
**  as a C integer.
**
**  Note that 'FIELD_VEC8BIT' is a macro, so do not call it with  arguments
**  that have sideeffects.
*/

#define libGAP_FIELD_VEC8BIT(list)         ((libGAP_Int)(libGAP_ADDR_OBJ(list)[2]))

/****************************************************************************
**
*F  SET_FIELD_VEC8BIT( <vec>, <q> )  . . set field size of an 8 bit GF vector
**
**  'SET_FIELD_VEC8BIT' sets the field size of the 8bit GFQ vector <vec>,
**  to the C integer <q>.
**
*/
#define libGAP_SET_FIELD_VEC8BIT(list,q)         ((libGAP_ADDR_OBJ(list)[2] = (libGAP_Obj)(q)))


/****************************************************************************
**
*F  BYTES_VEC8BIT( <list> ) . . . . . . . . . first byte of a 8bit GFQ vector
**
**  returns a pointer to the start of the data of the 8bit GFQ vector
*/
#define libGAP_BYTES_VEC8BIT(list)             ((libGAP_UInt1*)(libGAP_ADDR_OBJ(list)+3))


/****************************************************************************
**
*V  FieldInfo8Bit . .  . . . . . . . . .plain list (length 256) of field info
**
**  This list caches the field info used for the fast arithmetic
*/

static libGAP_Obj libGAP_FieldInfo8Bit;


/****************************************************************************
**
*F  Q_FIELDINFO_8BIT( <obj> )       . . . access to fields in structure
*F  P_FIELDINFO_8BIT( <obj> )
*F  ELS_BYTE_FIELDINFO_8BIT( <obj> )
*F  SETELT_FIELDINFO_8BIT( <obj> )
*F  GETELT_FIELDINFO_8BIT( <obj> )
*F  SCALMUL_FIELDINFO_8BIT( <obj> )
*F  ADD_FIELDINFO_8BIT( <obj> )
*F  SET_XXX_FIELDINFO_8BIOT( <obj>, <xxx> ) . . .setters needed by ANSI
**                                         needed for scalar but not pointers
**
**  For machines with alignment restrictions. it's important to put all
**  the word-sized data BEFORE all the byte-sized data (especially FFE_FELT...
**  which may have odd length
**
**  Note ADD has to be last, because it is not there in characteristic 2
*/

#define libGAP_Q_FIELDINFO_8BIT( info ) ((libGAP_UInt)(libGAP_ADDR_OBJ(info)[1]))
#define libGAP_SET_Q_FIELDINFO_8BIT( info, q ) (libGAP_ADDR_OBJ(info)[1] = (libGAP_Obj)(q))
#define libGAP_P_FIELDINFO_8BIT( info ) ((libGAP_UInt)(libGAP_ADDR_OBJ(info)[2]))
#define libGAP_SET_P_FIELDINFO_8BIT( info, p ) (libGAP_ADDR_OBJ(info)[2] = (libGAP_Obj)(p))
#define libGAP_D_FIELDINFO_8BIT( info ) ((libGAP_UInt)(libGAP_ADDR_OBJ(info)[3]))
#define libGAP_SET_D_FIELDINFO_8BIT( info, d ) (libGAP_ADDR_OBJ(info)[3] = (libGAP_Obj)(d))
#define libGAP_ELS_BYTE_FIELDINFO_8BIT( info ) ((libGAP_UInt)(libGAP_ADDR_OBJ(info)[4]))
#define libGAP_SET_ELS_BYTE_FIELDINFO_8BIT( info, e ) (libGAP_ADDR_OBJ(info)[4] = (libGAP_Obj)(e))
#define libGAP_FFE_FELT_FIELDINFO_8BIT( info ) (libGAP_ADDR_OBJ(info)+5)
#define libGAP_GAPSEQ_FELT_FIELDINFO_8BIT( info ) (libGAP_ADDR_OBJ(info)+5+libGAP_Q_FIELDINFO_8BIT(info))
#define libGAP_FELT_FFE_FIELDINFO_8BIT( info ) ((libGAP_UInt1*)(libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info)+libGAP_Q_FIELDINFO_8BIT(info)))
#define libGAP_SETELT_FIELDINFO_8BIT( info ) (libGAP_FELT_FFE_FIELDINFO_8BIT( info ) + libGAP_Q_FIELDINFO_8BIT(info))
#define libGAP_GETELT_FIELDINFO_8BIT( info ) \
     (libGAP_SETELT_FIELDINFO_8BIT(info) + \
      256*libGAP_Q_FIELDINFO_8BIT(info)*libGAP_ELS_BYTE_FIELDINFO_8BIT(info))
#define libGAP_SCALAR_FIELDINFO_8BIT( info ) \
     (libGAP_GETELT_FIELDINFO_8BIT(info)+256*libGAP_ELS_BYTE_FIELDINFO_8BIT(info))
#define libGAP_INNER_FIELDINFO_8BIT( info ) \
     (libGAP_SCALAR_FIELDINFO_8BIT( info ) + 256*libGAP_Q_FIELDINFO_8BIT(info))
#define libGAP_PMULL_FIELDINFO_8BIT( info ) \
     (libGAP_INNER_FIELDINFO_8BIT( info ) + 256*256)
#define libGAP_PMULU_FIELDINFO_8BIT( info ) \
     (libGAP_PMULL_FIELDINFO_8BIT( info ) + 256*256)
#define libGAP_ADD_FIELDINFO_8BIT( info ) \
     (libGAP_PMULU_FIELDINFO_8BIT( info ) + ((libGAP_ELS_BYTE_FIELDINFO_8BIT(info) == 1) ? 0 : 256*256))





/****************************************************************************
**

*F * * * * * * * * * * * imported library variables * * * * * * * * * * * * *
*/


/****************************************************************************
**
*F  TypeVec8Bit( <q>, <mut> ) . . .  . . .type of a  vector object
**
*/
libGAP_Obj libGAP_TYPES_VEC8BIT;
libGAP_Obj libGAP_TYPE_VEC8BIT;
libGAP_Obj libGAP_TYPE_VEC8BIT_LOCKED;

libGAP_Obj libGAP_TypeVec8Bit( libGAP_UInt q, libGAP_UInt mut)
{
  libGAP_UInt col = mut ? 1 : 2;
  libGAP_Obj type;
  type = libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_TYPES_VEC8BIT, col),q);
  if (type == 0)
    return libGAP_CALL_2ARGS(libGAP_TYPE_VEC8BIT, libGAP_INTOBJ_INT(q), mut ? libGAP_True: libGAP_False);
  else
    return type;
}

libGAP_Obj libGAP_TypeVec8BitLocked( libGAP_UInt q, libGAP_UInt mut)
{
  libGAP_UInt col = mut ? 3 : 4;
  libGAP_Obj type;
  type = libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_TYPES_VEC8BIT, col),q);
  if (type == 0)
    return libGAP_CALL_2ARGS(libGAP_TYPE_VEC8BIT_LOCKED, libGAP_INTOBJ_INT(q), mut ? libGAP_True : libGAP_False);
  else
    return type;
}

/****************************************************************************
**
*F  TypeMat8Bit( <q>, <mut> ) . . .  . . .type of a  matrix object
**
*/
libGAP_Obj libGAP_TYPES_MAT8BIT;
libGAP_Obj libGAP_TYPE_MAT8BIT;

libGAP_Obj libGAP_TypeMat8Bit( libGAP_UInt q, libGAP_UInt mut)
{
  libGAP_UInt col = mut ? 1 : 2;
  libGAP_Obj type;
  type = libGAP_ELM_PLIST(libGAP_ELM_PLIST(libGAP_TYPES_MAT8BIT, col),q);
  if (type == 0)
    return libGAP_CALL_2ARGS(libGAP_TYPE_MAT8BIT, libGAP_INTOBJ_INT(q), mut ? libGAP_True: libGAP_False);
  else
    return type;
}


/****************************************************************************
**
*V  TYPE_FIELDINFO_8BIT
**
**  A type of data object with essentially no GAP visible semantics at all
**  
*/

libGAP_Obj libGAP_TYPE_FIELDINFO_8BIT;


#define libGAP_SIZE_VEC8BIT(len,elts) (3*sizeof(libGAP_UInt)+((len)+(elts)-1)/(elts))

/****************************************************************************
**                    
*V  GetFieldInfo( <q> ) . .make or recover the meataxe table for a field
**                         always call this, as the tables are lost by
**                         save/restore. It's very cheap if the table already
**                         exists
**
*/


static const libGAP_UInt1 libGAP_GF4Lookup[] =  {0,2,1,3};
static const libGAP_UInt1 libGAP_GF8Lookup[] =  {0, 4, 2, 1, 6, 3, 7, 5};

static const libGAP_UInt1 libGAP_GF16Lookup[] = {0, 8, 4, 2, 1, 12, 6, 3, 13, 10, 5,
14, 7, 15, 11, 9};

static const libGAP_UInt1 libGAP_GF32Lookup[] = {0, 16, 8, 4, 2, 1, 20, 10, 5, 22,
11, 17, 28, 14, 7, 23, 31, 27, 25, 24, 12, 6, 3, 21, 30, 15, 19, 29,
26, 13, 18, 9};

static const libGAP_UInt1 libGAP_GF64Lookup[] = { 0, 32, 16, 8, 4, 2, 1, 54, 27, 59,
43, 35, 39, 37, 36, 18, 9, 50, 25, 58, 29, 56, 28, 14, 7, 53, 44, 22,
11, 51, 47, 33, 38, 19, 63, 41, 34, 17, 62, 31, 57, 42, 21, 60, 30,
15, 49, 46, 23, 61, 40, 20, 10, 5, 52, 26, 13, 48, 24, 12, 6, 3, 55,
45 };

static const libGAP_UInt1 libGAP_GF128Lookup[] = { 0, 64, 32, 16, 8, 4, 2, 1, 96,
48, 24, 12, 6, 3, 97, 80, 40, 20, 10, 5, 98, 49, 120, 60, 30, 15, 103,
83, 73, 68, 34, 17, 104, 52, 26, 13, 102, 51, 121, 92, 46, 23, 107,
85, 74, 37, 114, 57, 124, 62, 31, 111, 87, 75, 69, 66, 33, 112, 56,
28, 14, 7, 99, 81, 72, 36, 18, 9, 100, 50, 25, 108, 54, 27, 109, 86,
43, 117, 90, 45, 118, 59, 125, 94, 47, 119, 91, 77, 70, 35, 113, 88,
44, 22, 11, 101, 82, 41, 116, 58, 29, 110, 55, 123, 93, 78, 39, 115,
89, 76, 38, 19, 105, 84, 42, 21, 106, 53, 122, 61, 126, 63, 127, 95,
79, 71, 67, 65 };

static const libGAP_UInt1 libGAP_GF256Lookup[] = { 0, 128, 64, 32, 16, 8, 4, 2, 1,
184, 92, 46, 23, 179, 225, 200, 100, 50, 25, 180, 90, 45, 174, 87,
147, 241, 192, 96, 48, 24, 12, 6, 3, 185, 228, 114, 57, 164, 82, 41,
172, 86, 43, 173, 238, 119, 131, 249, 196, 98, 49, 160, 80, 40, 20,
10, 5, 186, 93, 150, 75, 157, 246, 123, 133, 250, 125, 134, 67, 153,
244, 122, 61, 166, 83, 145, 240, 120, 60, 30, 15, 191, 231, 203, 221,
214, 107, 141, 254, 127, 135, 251, 197, 218, 109, 142, 71, 155, 245,
194, 97, 136, 68, 34, 17, 176, 88, 44, 22, 11, 189, 230, 115, 129,
248, 124, 62, 31, 183, 227, 201, 220, 110, 55, 163, 233, 204, 102, 51,
161, 232, 116, 58, 29, 182, 91, 149, 242, 121, 132, 66, 33, 168, 84,
42, 21, 178, 89, 148, 74, 37, 170, 85, 146, 73, 156, 78, 39, 171, 237,
206, 103, 139, 253, 198, 99, 137, 252, 126, 63, 167, 235, 205, 222,
111, 143, 255, 199, 219, 213, 210, 105, 140, 70, 35, 169, 236, 118,
59, 165, 234, 117, 130, 65, 152, 76, 38, 19, 177, 224, 112, 56, 28,
14, 7, 187, 229, 202, 101, 138, 69, 154, 77, 158, 79, 159, 247, 195,
217, 212, 106, 53, 162, 81, 144, 72, 36, 18, 9, 188, 94, 47, 175, 239,
207, 223, 215, 211, 209, 208, 104, 52, 26, 13, 190, 95, 151, 243, 193,
216, 108, 54, 27, 181, 226, 113};

static const libGAP_UInt1 libGAP_PbyQ[] = { 0, 1, 2, 3, 2, 5, 0, 7, 2, 3, 0, 11, 0,
13, 0, 0, 2, 17, 0, 19, 0, 0, 0, 23, 0, 5, 0, 3, 0, 29, 0, 31, 2, 0,
0, 0, 0, 37, 0, 0, 0, 41, 0, 43, 0, 0, 0, 47, 0, 7, 0, 0, 0, 53, 0, 0,
0, 0, 0, 59, 0, 61, 0, 0, 2, 0, 0, 67, 0, 0, 0, 71, 0, 73, 0, 0, 0, 0,
0, 79, 0, 3, 0, 83, 0, 0, 0, 0, 0, 89, 0, 0, 0, 0, 0, 0, 0, 97, 0, 0,
0, 101, 0, 103, 0, 0, 0, 107, 0, 109, 0, 0, 0, 113, 0, 0, 0, 0, 0, 0,
0, 11, 0, 0, 0, 5, 0, 127, 2, 0, 0, 131, 0, 0, 0, 0, 0, 137, 0, 139,
0, 0, 0, 0, 0, 0, 0, 0, 0, 149, 0, 151, 0, 0, 0, 0, 0, 157, 0, 0, 0,
0, 0, 163, 0, 0, 0, 167, 0, 13, 0, 0, 0, 173, 0, 0, 0, 0, 0, 179, 0,
181, 0, 0, 0, 0, 0, 0, 0, 0, 0, 191, 0, 193, 0, 0, 0, 197, 0, 199, 0,
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 211, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
223, 0, 0, 0, 227, 0, 229, 0, 0, 0, 233, 0, 0, 0, 0, 0, 239, 0, 241,
0, 3, 0, 0, 0, 0, 0, 0, 0, 251, 0, 0, 0, 0, 2 };

static const libGAP_UInt1 libGAP_DbyQ[] = { 0, 1, 1, 1, 2, 1, 0, 1, 3, 2, 0, 1, 0,
1, 0, 0, 4, 1, 0, 1, 0, 0, 0, 1, 0, 2, 0, 3, 0, 1, 0, 1, 5, 0, 0, 0,
0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 2, 0, 0, 0, 1, 0, 0, 0, 0, 0,
1, 0, 1, 0, 0, 6, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 4,
0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0,
0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 3, 0, 1,
7, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 2, 0, 0, 0, 1,
0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0,
1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0,
5, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 8};



static const libGAP_UInt1 * libGAP_Char2Lookup[9] = {
  0L, 0L,
  libGAP_GF4Lookup,
  libGAP_GF8Lookup,
  libGAP_GF16Lookup,
  libGAP_GF32Lookup,
  libGAP_GF64Lookup,
  libGAP_GF128Lookup,
  libGAP_GF256Lookup};


void libGAP_MakeFieldInfo8Bit( libGAP_UInt q)
{
  libGAP_FF   gfq;     /* the field */
  libGAP_UInt p;     /* characteristic */
  libGAP_UInt d;     /* degree */
  libGAP_UInt i,j,k,l;     /* loop variables */
  libGAP_UInt e;     /* number of elements per byte */
  libGAP_UInt size;      /* data structure size */
  libGAP_UInt pows[7];     /* table of powers of q for packing
           and unpacking bytes */
  libGAP_Obj info;     /* The table being constructed */
  libGAP_FFV mult;     /* multiplier for scalar product */
  libGAP_FFV prod;     /* used in scalar product */
  libGAP_UInt val;                     /* used to build up some answers */
  libGAP_UInt val0;
  libGAP_UInt elt,el1,el2;             /* used to build up some answers */
  libGAP_FFV *succ;
  
  
  p = (libGAP_UInt)libGAP_PbyQ[q];
  d = (libGAP_UInt)libGAP_DbyQ[q];
  gfq = libGAP_FiniteField(p,d);
  e = 0;
  for (i = 1; i <= 256; i *= q)
    pows[e++] = i;
  pows[e] = i;    /* simplifies things to have one more */
  e--;
      
  size = sizeof(libGAP_Obj) +    /* type */
    sizeof(libGAP_Obj) +   /* q */
    sizeof(libGAP_Obj) +   /* p */
    sizeof(libGAP_Obj) +   /* d */
    sizeof(libGAP_Obj) +   /* els per byte */
    q*sizeof(libGAP_Obj) +             /* position in GAP < order  by number */
    q*sizeof(libGAP_Obj) +             /* numbering from FFV */
    q*sizeof(libGAP_Obj) +   /* immediate FFE by number */
    256*q*e +     /* set element lookup */
    256*e +     /* get element lookup */
    256*q +     /* scalar multiply */
    2* 256*256 +    /* inner product, 1 lot of polynomial multiply data */
    ((e == 1) ? 0 : (256*256))+ /* the other lot of polynomial data */
    ((p == 2) ? 0 : (256*256)); /* add byte */
  
  info = libGAP_NewBag(libGAP_T_DATOBJ, size);
  libGAP_TYPE_DATOBJ(info) = libGAP_TYPE_FIELDINFO_8BIT;
  
  succ = libGAP_SUCC_FF(gfq);

  /* from here to the end, no garbage collections should happen */
  libGAP_SET_Q_FIELDINFO_8BIT(info,q);
  libGAP_SET_P_FIELDINFO_8BIT(info, p);
  libGAP_SET_D_FIELDINFO_8BIT(info, d);
  libGAP_SET_ELS_BYTE_FIELDINFO_8BIT(info, e);
  
  /* conversion tables FFV to/from our numbering
     we assume that 0 and 1 are always the zero and one
     of the field. In char 2, we assume that xor corresponds
     to addition, otherwise, the order doesn't matter */
  
  if (p != 2)   
    for (i = 0; i < q; i++)
      libGAP_FELT_FFE_FIELDINFO_8BIT(info)[i] = (libGAP_UInt1)i;
  else      
    for (i = 0; i < q; i++)
      libGAP_FELT_FFE_FIELDINFO_8BIT(info)[i] = libGAP_Char2Lookup[d][i];
  
  /* simply invert the permutation to get the other one */
  for (i = 0; i < q; i++)
    libGAP_FFE_FELT_FIELDINFO_8BIT(info)[libGAP_FELT_FFE_FIELDINFO_8BIT(info)[i]] =
      libGAP_NEW_FFE(gfq,i);


  /* Now we need to store the position in Elements(GF(q)) of each field element
     for the sake of NumberFFVector

     The rules for < between finite field elements make this a bit
     complex for non-prime fields */

  /* deal with zero and one */
  libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info)[0] = libGAP_INTOBJ_INT(0);
  libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info)[libGAP_FELT_FFE_FIELDINFO_8BIT(info)[1]] = libGAP_INTOBJ_INT(1);

  if (q != 2)
    {
      if (d == 1)
  for (i = 2; i < q; i++)
    libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info)[i] = libGAP_INTOBJ_INT(i);
      else
  {
    /* run through subfields, filling in entry for all the new elements
       of each field in turn */
    libGAP_UInt q1 = 1;
    libGAP_UInt pos = 2;
    for (i = 1; i <= d; i++)
      {
        q1 *= p;
        if (d % i == 0)
    {
      for (j = 2; j < q1; j++)
        {
          libGAP_UInt place = libGAP_FELT_FFE_FIELDINFO_8BIT(info)[1+(j-1)*(q-1)/(q1-1)];
          if (libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info)[place] == 0)
      {
        libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info)[place] = libGAP_INTOBJ_INT(pos);
        pos++;
      }
        }
    }
      }
  }
    }
    
  
  /* entry setting table SETELT...[(i*e+j)*256 +k] is the result
     of overwriting the jth element with i in the byte k */
  for (i = 0; i < q; i++)
    for (j = 0; j < e; j++)
      for (k = 0; k < 256; k++)
  libGAP_SETELT_FIELDINFO_8BIT(info)[(i*e + j)*256 + k] = (libGAP_UInt1)
    ((k/pows[j+1])*pows[j+1] + i*pows[j] + (k % pows[j]));
  
  /* entry access GETELT...[i*256+j] recovers the ith entry from the
     byte j */
  for (i = 0; i < e; i++)
    for(j = 0; j < 256; j++)
      libGAP_GETELT_FIELDINFO_8BIT(info)[i*256 + j] = (libGAP_UInt1)(j/pows[i])%q;
  
  /* scalar * vector multiply SCALAR...[i*256+j] is the scalar
     product of the byte j with the felt i */
  for (i = 0; i < q; i++)
    {
      mult = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)[i]);
      for(j = 0; j < 256; j++)
  {
    val = 0;
    for (k  = 0; k < e; k++)
      {
        elt = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
          [libGAP_GETELT_FIELDINFO_8BIT(info)[k*256+j]]);
        prod = libGAP_PROD_FFV(elt,mult,succ);
        val += pows[k]*libGAP_FELT_FFE_FIELDINFO_8BIT(info)[prod];
      }
    libGAP_SCALAR_FIELDINFO_8BIT(info)[i*256+j] = val;
  }
    }
  
  /* inner product INNER...[i+256*j] is a byte whose LS entry is the contribution
     to the inner product of bytes i and j */
  
  for ( i = 0; i < 256; i++)
    for (j = i; j < 256; j++)
      {
  val = 0;
  for (k = 0; k < e; k++)
    {
      el1 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
        [libGAP_GETELT_FIELDINFO_8BIT(info)[k*256+i]]);
      el2 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
        [libGAP_GETELT_FIELDINFO_8BIT(info)[k*256+j]]);
      elt = libGAP_PROD_FFV( el1, el2, succ);
      val = libGAP_SUM_FFV(val, elt, succ);
    }
  val = libGAP_SETELT_FIELDINFO_8BIT(info)[256*e*libGAP_FELT_FFE_FIELDINFO_8BIT(info)[val]];
  libGAP_INNER_FIELDINFO_8BIT(info)[i+256*j] = val;
  libGAP_INNER_FIELDINFO_8BIT(info)[j+256*i] = val;
      }

  /* PMULL and PMULU are the lower and upper bytes of the product
     of single-byte polynomials */
  for ( i = 0; i < 256; i++)
    for (j = i; j < 256; j++)
      {
  val0 = 0;
  for (k = 0; k < e; k++)
    {
      val = 0;
      for (l = 0; l <= k; l++)
        {
    el1 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
            [libGAP_GETELT_FIELDINFO_8BIT(info)[l*256+i]]);
    el2 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
            [libGAP_GETELT_FIELDINFO_8BIT(info)[(k-l)*256+j]]);
    elt = libGAP_PROD_FFV( el1, el2, succ);
    val = libGAP_SUM_FFV(val, elt, succ);
        }
      val0 += pows[k]*libGAP_FELT_FFE_FIELDINFO_8BIT(info)[val];
    }
  libGAP_PMULL_FIELDINFO_8BIT(info)[i+256*j] = val0;
  libGAP_PMULL_FIELDINFO_8BIT(info)[j+256*i] = val0;

  /* if there is just one entry per byte then we don't need the upper half */
  if (libGAP_ELS_BYTE_FIELDINFO_8BIT(info) > 1)
    {
      val0 = 0;
      for (k = e; k < 2*e-1; k++)
        {
    val = 0;
    for (l = k-e+1; l < e; l++)
      {
        el1 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
          [libGAP_GETELT_FIELDINFO_8BIT(info)[l*256+i]]);
        el2 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
          [libGAP_GETELT_FIELDINFO_8BIT(info)[(k-l)*256+j]]);
        elt = libGAP_PROD_FFV( el1, el2, succ);
        val = libGAP_SUM_FFV(val, elt, succ);
      }
    val0 += pows[k-e]*libGAP_FELT_FFE_FIELDINFO_8BIT(info)[val];
        }
      libGAP_PMULU_FIELDINFO_8BIT(info)[i+256*j] = val0;
      libGAP_PMULU_FIELDINFO_8BIT(info)[j+256*i] = val0;
    }
      }
  
  
  /* In odd characteristic, we need the addition table
     ADD...[i*256+j] is the vector sum of bytes i and j */
  if (p != 2)
    {
      for (i = 0; i < 256; i++)
  for (j = i; j < 256; j++)
    {
      val = 0;
      for (k = 0; k < e; k++)
        {
    el1 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
            [libGAP_GETELT_FIELDINFO_8BIT(info)[k*256+i]]);
    el2 = libGAP_VAL_FFE(libGAP_FFE_FELT_FIELDINFO_8BIT(info)
            [libGAP_GETELT_FIELDINFO_8BIT(info)[k*256+j]]);
    val += pows[k]*
      libGAP_FELT_FFE_FIELDINFO_8BIT(info)[libGAP_SUM_FFV(el1,el2, succ)];
        }
      libGAP_ADD_FIELDINFO_8BIT(info)[i+256*j] = val;
      libGAP_ADD_FIELDINFO_8BIT(info)[j+256*i] = val;
    }
    }
  
  
  /* remember the result */
  libGAP_SET_ELM_PLIST(libGAP_FieldInfo8Bit,q,info);
  libGAP_CHANGED_BAG(libGAP_FieldInfo8Bit);
}
     
libGAP_Obj libGAP_GetFieldInfo8Bit( libGAP_UInt q)
{
  libGAP_Obj info;
  assert(2< q && q <= 256);
  info = libGAP_ELM_PLIST(libGAP_FieldInfo8Bit, q );
  if ( info == 0)
    {
      libGAP_MakeFieldInfo8Bit( q );
      info = libGAP_ELM_PLIST(libGAP_FieldInfo8Bit, q );
    }
  return info;
}
  

/****************************************************************************
**
*F  RewriteVec8Bit( <vec>, <q> ) . . . . . . . . . . rewrite <vec> over GF(q)
**
** <vec> should be an 8 bit vector over a smaller field of the same
** characteristic 
*/

static libGAP_Obj libGAP_IsLockedRepresentationVector;


void libGAP_RewriteVec8Bit( libGAP_Obj vec, libGAP_UInt q)
{
  libGAP_UInt q1 = libGAP_FIELD_VEC8BIT(vec);
  libGAP_Obj info, info1;
  libGAP_UInt len;
  libGAP_UInt els, els1;
/*UInt mut = IS_MUTABLE_OBJ(vec); */
  libGAP_UInt mult;

  libGAP_UInt1 *gettab1, *ptr1, byte1;
  libGAP_UInt1 *settab, *ptr, byte;
  libGAP_UInt1 * convtab;
  libGAP_Obj *convtab1;
  libGAP_FFV val;

  libGAP_Int i;
  
  if (q1 == q)
    return;
  assert(q > q1);

  if (libGAP_DoFilter(libGAP_IsLockedRepresentationVector,vec) == libGAP_True)
    {
      libGAP_ErrorMayQuit("You cannot convert a locked vector compressed over GF(%i) to GF(%i)",
          q1, q);
      return;
    }
  
  /* extract the required info */
  len = libGAP_LEN_VEC8BIT(vec);
  info = libGAP_GetFieldInfo8Bit(q);
  info1 = libGAP_GetFieldInfo8Bit(q1);
  assert(libGAP_P_FIELDINFO_8BIT(info) == libGAP_P_FIELDINFO_8BIT(info1));
  assert(!(libGAP_D_FIELDINFO_8BIT(info) % libGAP_D_FIELDINFO_8BIT(info1)));
  els = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  els1 = libGAP_ELS_BYTE_FIELDINFO_8BIT(info1);

  if (len == 0)
    {
      libGAP_SET_FIELD_VEC8BIT(vec, q);
      return;
    }
  
  /* enlarge the bag */
  libGAP_ResizeBag(vec, libGAP_SIZE_VEC8BIT( len, els));

  gettab1 = libGAP_GETELT_FIELDINFO_8BIT(info1);
  convtab1 = libGAP_FFE_FELT_FIELDINFO_8BIT(info1);
  settab = libGAP_SETELT_FIELDINFO_8BIT(info);
  convtab = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
  ptr1 = libGAP_BYTES_VEC8BIT(vec) + (len - 1)/els1;
  byte1 = *ptr1;
  ptr = libGAP_BYTES_VEC8BIT(vec) + (len - 1)/els;
  byte = 0;
  i = len-1;
  
  assert( ((q-1) % (q1 -1)) == 0);
  mult = (q-1)/(q1-1);
  while (i >= 0)
    {
      val= libGAP_VAL_FFE(convtab1[ gettab1[byte1 + 256 * (i % els1)]]);
      if (val != 0)
  val = 1+ (val-1)*mult;
      byte = settab[ byte + 256 * (i % els + els* convtab[ val ])];
      if (0 == i % els)
  {
    *ptr-- = byte;
    byte = 0;
  }
      if (0 == i % els1)
  byte1 = *--ptr1;
      i--;
    }
  libGAP_SET_FIELD_VEC8BIT(vec,q);
}

/****************************************************************************
**
*F  RewriteGF2Vec( <vec>, <q> ) . . . . . . . . . . rewrite <vec> over GF(q)
**
** <vec> should be a GF2 vector and q a larger power of 2
**
** This function uses the interface in vecgf2.h
*/


void libGAP_RewriteGF2Vec( libGAP_Obj vec, libGAP_UInt q )
{
  libGAP_Obj info;
  libGAP_UInt len;
  libGAP_UInt els;
  libGAP_UInt mut = libGAP_IS_MUTABLE_OBJ(vec);
  libGAP_UInt *ptr1;
  libGAP_UInt block;
  libGAP_UInt1 *settab, *ptr, byte;
  libGAP_UInt1 *convtab;
  libGAP_UInt1 zero, one;
  libGAP_Int i;
  libGAP_Obj type;
  
  assert(q % 2 == 0);
  
  if (libGAP_DoFilter(libGAP_IsLockedRepresentationVector,vec) == libGAP_True)
    {
      libGAP_ErrorReturnVoid("You cannot convert a locked vector compressed over GF(2) to GF(%i)",
          q, 0, "You can `return;' to ignore the conversion");
      return;
    }
  
  /* extract the required info */
  len = libGAP_LEN_GF2VEC(vec);
  info = libGAP_GetFieldInfo8Bit(q);
  els = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);

  /* enlarge the bag */
  libGAP_ResizeBag(vec, libGAP_SIZE_VEC8BIT( len, els));

  settab = libGAP_SETELT_FIELDINFO_8BIT(info);
  convtab = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
  zero = convtab[0];
  one = convtab[1];
  ptr1 = libGAP_BLOCKS_GF2VEC(vec) + libGAP_NUMBER_BLOCKS_GF2VEC(vec) -1;
  block = *ptr1;
  ptr = libGAP_BYTES_VEC8BIT(vec) + (len - 1)/els;
  byte = 0;
  i = len-1;
  
  while (i >= 0)
    {
      byte = settab[ byte + 256 * (i % els + els* ((block & libGAP_MASK_POS_GF2VEC( i+1)) ? one: zero))];
      if (0 == i % els)
  {
    *ptr-- = byte;
    byte = 0;
  }
      if (0 == i % libGAP_BIPEB)
  block = *--ptr1;
      i--;
    }
  libGAP_SET_FIELD_VEC8BIT(vec, q);
  libGAP_SET_LEN_VEC8BIT(vec,len);
  type = libGAP_TypeVec8Bit( q, mut);
  libGAP_SET_TYPE_POSOBJ(vec, type);
}


/****************************************************************************
**
*F  ConvVec8Bit( <list>, <q> )  . . .  convert a list into 8bit vector object
*/

void libGAP_ConvVec8Bit (
    libGAP_Obj                 list,
    libGAP_UInt                q)
{
    libGAP_Int                 len;            /* logical length of the vector    */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_UInt                p;  /* char */
    libGAP_UInt                d;  /* degree */
    libGAP_FF                  f;  /* field */
 /* Obj                 x;  / an element */
    libGAP_Obj                 info; /* field info object */
    libGAP_UInt                elts; /* elements per byte */
    libGAP_UInt1 *             settab; /* element setting table */
    libGAP_UInt1 *             convtab; /* FFE -> FELT conversion table */
    libGAP_Obj                 firstthree[3]; /* the first three entries
          may get clobbered my the early bytes */
    libGAP_UInt                e;  /* loop varibale */
    libGAP_UInt1               byte; /* byte under construction */
    libGAP_UInt1*              ptr;  /* place to put byte */
    libGAP_Obj                elt;
    libGAP_UInt               val;
    libGAP_UInt               nsize;
    libGAP_Obj                type;
        
    if (q > 256)
      libGAP_ErrorQuit("Field size %d is too much for 8 bits\n",
    q, 0L);
    if (q == 2)
      libGAP_ErrorQuit("GF2 has its own representation\n",0L,0L);

    /* already in the correct representation                               */
    if ( libGAP_IS_VEC8BIT_REP(list))
      {
  if(libGAP_FIELD_VEC8BIT(list) == q ) 
    return;
  else if (libGAP_FIELD_VEC8BIT(list) < q)
    {
      libGAP_RewriteVec8Bit(list,q);
      return;
    }
  /* remaining case is list is written over too large a field
     pass through to the generic code */
  
    }
    else if ( libGAP_IS_GF2VEC_REP(list))
      {
  libGAP_RewriteGF2Vec(list, q);
  return;
      }
    
    len   = libGAP_LEN_LIST(list);
    
    /* OK, so now we know which field we want, set up data */
    info = libGAP_GetFieldInfo8Bit(q);
    p = libGAP_P_FIELDINFO_8BIT(info);
    d = libGAP_D_FIELDINFO_8BIT(info);
    f = libGAP_FiniteField(p,d);
    
    elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);

    /* We may need to resize first, as small lists get BIGGER
       in this process */
    nsize = libGAP_SIZE_VEC8BIT(len,elts);
    if (nsize > libGAP_SIZE_OBJ(list))
      libGAP_ResizeBag(list,nsize);

    
    /* writing the first byte may clobber the third list entry
       before we have read it, so we take a copy */
    firstthree[0] = libGAP_ELM0_LIST(list,1);
    firstthree[1] = libGAP_ELM0_LIST(list,2);
    firstthree[2] = libGAP_ELM0_LIST(list,3);
    
    /* main loop -- e is the element within byte */
    e = 0;
    byte = 0;
    ptr = libGAP_BYTES_VEC8BIT(list);
    for ( i = 1;  i <= len;  i++ ) {
      elt = (i <= 3) ? firstthree[i-1] : libGAP_ELM_LIST(list,i);
      assert(libGAP_CHAR_FF(libGAP_FLD_FFE(elt)) == p);
      assert( d % libGAP_DegreeFFE(elt) == 0);
      val = libGAP_VAL_FFE(elt);
      if (val != 0 && libGAP_FLD_FFE(elt) != f)
  {
    val = 1+(val-1)*(q-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(elt))-1);
  }
      /* Must get these afresh after every list access, just in case this is
       a virtual list whose accesses might cause a garbage collection */
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      convtab = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
      byte = settab[(e + elts*convtab[val])*256 + byte];
      if (++e == elts || i == len)
  {
    *ptr++ = byte;
    byte = 0;
    e = 0;
  }
    }

    /* it can happen that the few bytes after the end of the data are
       not zero, because they had data in them in the old version of the list
       In most cases this doesn't matter, but in characteristic 2, we must
       clear up to the end of the word, so that AddCoeffs behaves correctly.
    SL -- lets do this in all characteristics, it can never hurt */
    
    while ((ptr - libGAP_BYTES_VEC8BIT(list)) % sizeof(libGAP_UInt))
      *ptr++ = 0;

    /* retype and resize bag */
    if (nsize != libGAP_SIZE_OBJ(list))
      libGAP_ResizeBag( list, nsize );
    libGAP_SET_LEN_VEC8BIT( list, len );
    libGAP_SET_FIELD_VEC8BIT(list, q );
    type = libGAP_TypeVec8Bit( q, libGAP_HAS_FILT_LIST( list, libGAP_FN_IS_MUTABLE));
    libGAP_TYPE_DATOBJ( list ) = type;
    libGAP_RetypeBag( list, libGAP_T_DATOBJ );
}



/****************************************************************************
**
*F  LcmDegree( <d>, <d1> )
**
*/

libGAP_UInt libGAP_LcmDegree( libGAP_UInt d, libGAP_UInt d1)
{
  libGAP_UInt x,y,libGAP_g;
  x = d;
  y = d1;
  while (x != 0 && y != 0)
    {
      if (x <= y)
  y = y % x;
      else 
  x = x % y;
    }
  if (x == 0)
    libGAP_g = y;
  else
    libGAP_g = x;
  return (d*d1)/libGAP_g;
}

/****************************************************************************
**
*F  FuncCONV_VEC8BIT( <self>, <list> ) . . . . . convert into 8bit vector rep
*/
libGAP_Obj libGAP_FuncCONV_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 q)
{
  if (!libGAP_IS_INTOBJ(q))
    {
      libGAP_ErrorMayQuit("CONV_VEC8BIT: q must be a small integer (3--256) not a %s",
       (libGAP_Int)libGAP_TNAM_OBJ(q), 0);
    }
    
  libGAP_ConvVec8Bit(list, libGAP_INT_INTOBJ(q));
  
  /* return nothing                                                      */
  return 0;
}


/****************************************************************************
**
*F  PlainVec8Bit( <list> ) . . . convert an 8bit vector into an ordinary list
**
**  'PlainVec8Bit' converts the  vector <list> to a plain list.
*/
    
void libGAP_PlainVec8Bit (
    libGAP_Obj                 list )
{
    libGAP_Int                 len;            /* length of <list>                */
    libGAP_UInt                i;              /* loop variable                   */
    libGAP_Obj                 first;          /* first entry                     */
    libGAP_Obj                 second = 0;
    libGAP_UInt                q;
    libGAP_UInt                elts;
    libGAP_Obj                 info;
    libGAP_UInt1              *gettab;
    libGAP_UInt                tnum;



    /* resize the list and retype it, in this order                        */
    if (libGAP_True == libGAP_DoFilter(libGAP_IsLockedRepresentationVector, list))
      {
  libGAP_ErrorMayQuit("Attempt to convert locked compressed vector to plain list",0,0);
  return;
      }

    len = libGAP_LEN_VEC8BIT(list);
    q = libGAP_FIELD_VEC8BIT(list);
    info = libGAP_GetFieldInfo8Bit(q);
    elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);


    if (len == 0)
      tnum = libGAP_T_PLIST_EMPTY;
    else
      tnum = libGAP_T_PLIST_FFE;
    if (!libGAP_IS_MUTABLE_OBJ(list))
      tnum += libGAP_IMMUTABLE;
    libGAP_RetypeBag( list, tnum);
    
    libGAP_GROW_PLIST( list, (libGAP_UInt)len );
    libGAP_SET_LEN_PLIST( list, len );

    if (len != 0)
      {
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  /* keep the first two entries
     because setting the third destroys them  */
  
  first = libGAP_FFE_FELT_FIELDINFO_8BIT(info)[gettab[libGAP_BYTES_VEC8BIT(list)[0]]];
  if (len > 1)
    second =
      libGAP_FFE_FELT_FIELDINFO_8BIT(info)
      [gettab[256*(1 % elts)+libGAP_BYTES_VEC8BIT(list)[1/elts]]];
  
  /* replace the bits by FF elts as the case may be        */
  /* this must of course be done from the end of the list backwards      */
  for ( i = len;  2 < i;  i-- )
    libGAP_SET_ELM_PLIST( list, i,
       libGAP_FFE_FELT_FIELDINFO_8BIT(info)
           [gettab[256*((i-1)%elts)+
             libGAP_BYTES_VEC8BIT( list )[ (i-1) /elts]]] );
  if (len > 1)
    libGAP_SET_ELM_PLIST( list, 2, second );
  libGAP_SET_ELM_PLIST( list, 1, first );
      }
    /* Null out any entries after the end of valid data */
    for (i = len+1; i < (libGAP_SIZE_BAG(list)+sizeof(libGAP_Obj) -1 ) /sizeof(libGAP_Obj); i++)
      libGAP_SET_ELM_PLIST(list, i, (libGAP_Obj) 0);
    
    libGAP_CHANGED_BAG(list);
}

/****************************************************************************
**
*F  FuncPLAIN_VEC8BIT( <self>, <list> ) . . .  convert back into ordinary list
*/
libGAP_Obj libGAP_FuncPLAIN_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    /* check whether <list> is an 8bit vector                                */
    while ( ! libGAP_IS_VEC8BIT_REP(list) ) {
        list = libGAP_ErrorReturnObj(
            "CONV_BLIST: <list> must be an 8bit vector (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    if (libGAP_DoFilter(libGAP_IsLockedRepresentationVector,list) == libGAP_True)
    {
      libGAP_ErrorMayQuit("You cannot convert a locked vector compressed over GF(%i) to a plain list",
          libGAP_FIELD_VEC8BIT(list) , 0);
      return 0;
    }
    libGAP_PlainVec8Bit(list);

    /* return nothing                                                      */
    return 0;
}


/****************************************************************************
**
*F * * * * * * * * * * * * arithmetic operations  * * * * * * * * * * * * * *
*/

#define libGAP_NUMBLOCKS_VEC8BIT(len,elts) \
        (((len) + sizeof(libGAP_UInt)*(elts)-1)/(sizeof(libGAP_UInt)*(elts)))

#define libGAP_BLOCKS_VEC8BIT(vec) ((libGAP_UInt *)libGAP_BYTES_VEC8BIT(vec))     


/****************************************************************************
**
*F  CopyVec8Bit( <list>, <mut> ) .copying function
**
*/


libGAP_Obj libGAP_CopyVec8Bit( libGAP_Obj list, libGAP_UInt mut )
{
  libGAP_Obj copy;
  libGAP_UInt size;
  libGAP_UInt q;
  libGAP_Obj type;

  size = libGAP_SIZE_BAG(list);
  copy = libGAP_NewBag( libGAP_T_DATOBJ, size);
  q = libGAP_FIELD_VEC8BIT(list);
  type = libGAP_TypeVec8Bit(q,mut);
  libGAP_TYPE_DATOBJ(copy) = type;
  libGAP_CHANGED_BAG(copy);
  libGAP_SET_LEN_VEC8BIT(copy, libGAP_LEN_VEC8BIT(list));
  libGAP_SET_FIELD_VEC8BIT(copy,q);
  memcpy( libGAP_BYTES_VEC8BIT(copy), libGAP_BYTES_VEC8BIT(list), size - 3*sizeof(libGAP_UInt));
  return copy;
}



/****************************************************************************
**
*F  AddVec8BitVec8BitInner( <sum>, <vl>, <vr>, <start>, <stop> )
**
**  This is the real vector add routine. Others are all calls to this
**  one.
**  Addition is done from THE BLOCK containing <start> to the one
**  containing <stop> INCLUSIVE. The remainder of <sum> is unchanged.
**  <sum> may be the same vector as <vl> or
**  <vr>. <vl> and <vr> must be over the same field and <sum> must be
**  initialized as a vector over this field of length at least <stop>.
**
*/

void  libGAP_AddVec8BitVec8BitInner( libGAP_Obj sum,
  libGAP_Obj vl,
  libGAP_Obj vr,
  libGAP_UInt start,
  libGAP_UInt stop )
{
  libGAP_Obj info;
  libGAP_UInt p;
  libGAP_UInt elts;

  /* Maybe there's nothing to do */
  if (!stop)
    return;
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(sum));
  assert(libGAP_Q_FIELDINFO_8BIT(info) == libGAP_FIELD_VEC8BIT(vl));
  assert(libGAP_Q_FIELDINFO_8BIT(info) == libGAP_FIELD_VEC8BIT(vr));
  assert(libGAP_LEN_VEC8BIT(sum) >= stop);
  assert(libGAP_LEN_VEC8BIT(vl) >= stop);
  assert(libGAP_LEN_VEC8BIT(vr) >= stop);
  p = libGAP_P_FIELDINFO_8BIT(info);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  /* Convert from 1 based to zero based addressing */
  start --;
  stop --;
  if (p == 2)
  {
    libGAP_UInt *ptrL2;
    libGAP_UInt *ptrR2;
    libGAP_UInt *ptrS2;
    libGAP_UInt *endS2;
    ptrL2 = libGAP_BLOCKS_VEC8BIT(vl) + start/(sizeof(libGAP_UInt)*elts);
    ptrR2 = libGAP_BLOCKS_VEC8BIT(vr) + start/(sizeof(libGAP_UInt)*elts);
    ptrS2 = libGAP_BLOCKS_VEC8BIT(sum) + start/(sizeof(libGAP_UInt)*elts);
    endS2 = libGAP_BLOCKS_VEC8BIT(sum) + stop/(sizeof(libGAP_UInt)*elts)+1;
    if (sum == vl)
    {
      while (ptrL2 < endS2)
      {
        *ptrL2 ^= *ptrR2;
        ptrL2++; ptrR2++; 
      }
    }
    else if (sum == vr)
    {
      while (ptrR2 < endS2)
      {
        *ptrR2 ^=  *ptrL2;
        ptrL2++; ptrR2++;
      }

    }
    else
      while (ptrS2 < endS2)
        *ptrS2++ = *ptrL2++ ^ *ptrR2++;
    }
    else
    {
      libGAP_UInt1 *ptrL;
      libGAP_UInt1 *ptrR;
      libGAP_UInt1 *ptrS;
      libGAP_UInt1 *endS;
      libGAP_UInt x;
      const libGAP_UInt1 *addtab = libGAP_ADD_FIELDINFO_8BIT(info);
      ptrL = libGAP_BYTES_VEC8BIT(vl) + start/elts;
      ptrR = libGAP_BYTES_VEC8BIT(vr) + start/elts;
      ptrS = libGAP_BYTES_VEC8BIT(sum) + start/elts;
      endS = libGAP_BYTES_VEC8BIT(sum) + stop/elts + 1;
      if (vl == sum)
      {
        while (ptrL < endS)
        {
          if ((x = *ptrR) != 0)
            *ptrL = addtab[256* (*ptrL) + x];
          ptrR++; ptrL++;
        }
      }
      else if (vr == sum)
      {
        while (ptrR < endS)
        {
          if ((x = *ptrL) != 0)
            *ptrR = addtab[256* (x) + *ptrR];
          ptrR++; ptrL++;
        }
      }
      else
        while (ptrS < endS)
          *ptrS++ = addtab[256* (*ptrL++) + *ptrR++];
      }
      return;
    }

/****************************************************************************
**
*F  SumVec8BitVec8Bit( <vl>, <vr> )
**
**  This is perhaps the simplest wrapper for the Add..Inner function
**  it allocates a new vector for the result, and adds the whole vectors into
**  it. No checking is done. The result follows the new mutability convention
**  (mutable if either argument is).
*/

libGAP_Obj libGAP_SumVec8BitVec8Bit( libGAP_Obj vl, libGAP_Obj vr )
{
  libGAP_Obj sum;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt q;
  libGAP_UInt len;
  libGAP_Obj type;

  q = libGAP_FIELD_VEC8BIT(vl);
  len = libGAP_LEN_VEC8BIT(vl);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  sum = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT(len,elts));
  libGAP_SET_LEN_VEC8BIT(sum, len);
  type = libGAP_TypeVec8Bit(q, libGAP_IS_MUTABLE_OBJ(vl) || libGAP_IS_MUTABLE_OBJ(vr));
  libGAP_TYPE_DATOBJ(sum) = type;
  libGAP_SET_FIELD_VEC8BIT(sum, q);
  libGAP_CHANGED_BAG(sum);
  libGAP_AddVec8BitVec8BitInner( sum, vl, vr, 1, len);
  return sum;
}

/****************************************************************************
**
*F  FuncSUM_VEC8BIT_VEC8BIT( <self>, <vl>, <vr> )
**
** This is the GAP callable method for +. The method installation should
** ensure that we have matching characteristics, but we may not have a common
** field or the same lengths
**
*/

static libGAP_Obj libGAP_ConvertToVectorRep;  /* BH: changed to static */


libGAP_Obj libGAP_FuncSUM_VEC8BIT_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr)
{
  libGAP_Obj sum;
  if (libGAP_FIELD_VEC8BIT(vl) != libGAP_FIELD_VEC8BIT(vr))
    {
      libGAP_UInt ql = libGAP_FIELD_VEC8BIT(vl), qr = libGAP_FIELD_VEC8BIT(vr);
      libGAP_Obj infol = libGAP_GetFieldInfo8Bit(ql), infor = libGAP_GetFieldInfo8Bit(qr);
      libGAP_UInt newd = libGAP_LcmDegree( libGAP_D_FIELDINFO_8BIT(infol), libGAP_D_FIELDINFO_8BIT(infor));
      libGAP_UInt p,newq;
      libGAP_UInt i;
      p = libGAP_P_FIELDINFO_8BIT(infol);
      assert(p == libGAP_P_FIELDINFO_8BIT(infor));
      newq = 1;
      for (i = 0; i < newd; i++)
  newq *= p;
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (newd>8 || newq > 256 ||
    (ql != newq && libGAP_True == libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector,vl)) ||
    (qr != newq && libGAP_True == libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector,vr)))
  {
    sum = libGAP_SumListList(vl,vr);
    return sum;
  }
      else
  {
    libGAP_RewriteVec8Bit( vl, newq);
    libGAP_RewriteVec8Bit( vr, newq);
  }
    }

  
  /* just add if they're the same length,
     otherwise copy the longer and add in the shorter */

  if (libGAP_LEN_VEC8BIT(vl) == libGAP_LEN_VEC8BIT(vr))
    return libGAP_SumVec8BitVec8Bit(vl, vr);
  else if (libGAP_LEN_VEC8BIT(vl) > libGAP_LEN_VEC8BIT(vr))
    {
      sum = libGAP_CopyVec8Bit( vl, libGAP_IS_MUTABLE_OBJ(vl) || libGAP_IS_MUTABLE_OBJ(vr));
      libGAP_AddVec8BitVec8BitInner(sum, sum, vr, 1, libGAP_LEN_VEC8BIT(vr));
    }
  else
    {
      sum = libGAP_CopyVec8Bit( vr, libGAP_IS_MUTABLE_OBJ(vl) || libGAP_IS_MUTABLE_OBJ(vr));
      libGAP_AddVec8BitVec8BitInner(sum, sum, vl, 1, libGAP_LEN_VEC8BIT(vl));
    }
  
  return sum;
}


/****************************************************************************
**
*F  MultVec8BitFFEInner( <prod>, <vec>, <scal>, <start>, <stop> )
**
**  This is the real vector * scalar routine. Others are all calls to this
**  one.
**  Multiplication is done from THE BLOCK containing <start> to the one
**  containing <stop> INCLUSIVE. The remainder of <prod> is unchanged.
**  <prod> may be the same vector as <vec> 
**  <scal> must be written over the field of <vec> and
**  <prod> must be
**  initialized as a vector over this field of length at least <stop>.
**
*/

void libGAP_MultVec8BitFFEInner( libGAP_Obj prod,
        libGAP_Obj vec,
        libGAP_Obj scal,
        libGAP_UInt start,
        libGAP_UInt stop )
{
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt1 *ptrV;
  libGAP_UInt1 *ptrS;
  libGAP_UInt1 *endS;
  libGAP_UInt1 *tab;

  if (!stop)
    return;
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(prod));
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  
  assert(libGAP_Q_FIELDINFO_8BIT(info) == libGAP_FIELD_VEC8BIT(vec));
  assert(libGAP_LEN_VEC8BIT(prod) >= stop);
  assert(libGAP_LEN_VEC8BIT(vec) >= stop);
  assert(libGAP_Q_FIELDINFO_8BIT(info) == libGAP_SIZE_FF(libGAP_FLD_FFE(scal)));


  /* convert to 0 based addressing */
  start--;
  stop--;
  tab = libGAP_SCALAR_FIELDINFO_8BIT(info) +
    256*libGAP_FELT_FFE_FIELDINFO_8BIT(info)[libGAP_VAL_FFE(scal)];
  ptrV = libGAP_BYTES_VEC8BIT(vec) + start/elts;
  ptrS = libGAP_BYTES_VEC8BIT(prod) + start/elts;
  endS = libGAP_BYTES_VEC8BIT(prod) + stop/elts + 1;
  while (ptrS < endS)
    *ptrS++ = tab[*ptrV++];
  return;
}

/****************************************************************************
**
*F  MultVec8BitFFE( <vec>, <scal> ) . . . simple scalar multiply
**
**  This is a basic wrapper for Mult...Inner. It allocates space for
**  the result, promotes the scalar to the proper field if necessary and
**  runs over the whole vector
**
*/

libGAP_Obj libGAP_MultVec8BitFFE( libGAP_Obj vec, libGAP_Obj scal )
{
  libGAP_Obj prod;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt q;
  libGAP_UInt len;
  libGAP_UInt v;
  libGAP_Obj type;

  q = libGAP_FIELD_VEC8BIT(vec);
  len = libGAP_LEN_VEC8BIT(vec);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  prod = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT(len,elts));
  libGAP_SET_LEN_VEC8BIT(prod, len);
  type = libGAP_TypeVec8Bit(q, libGAP_IS_MUTABLE_OBJ(vec));
  libGAP_TYPE_DATOBJ(prod) = type;
  libGAP_SET_FIELD_VEC8BIT(prod, q);
  libGAP_CHANGED_BAG(prod);
  if (libGAP_SIZE_FF(libGAP_FLD_FFE(scal)) != q)
    {
      v = libGAP_VAL_FFE(scal);
      if (v != 0)
  v = 1+ (v-1)*(q-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(scal))-1);
      scal = libGAP_NEW_FFE( libGAP_FiniteField(libGAP_P_FIELDINFO_8BIT(info),
          libGAP_D_FIELDINFO_8BIT(info)),v);
    }
  libGAP_MultVec8BitFFEInner( prod, vec, scal, 1, len);
  return prod;
}


/****************************************************************************
**
*F  ZeroVec8Bit( <q>, <len>, <mut> ). . . . . . . . . . .return a zero vector
**
*/

libGAP_Obj libGAP_ZeroVec8Bit ( libGAP_UInt q, libGAP_UInt len, libGAP_UInt mut )
{
  libGAP_Obj zerov;
/*UInt1 *ptr; */
  libGAP_UInt size;
  libGAP_Obj info;
  libGAP_Obj type;
  info = libGAP_GetFieldInfo8Bit(q);
  size = libGAP_SIZE_VEC8BIT( len, libGAP_ELS_BYTE_FIELDINFO_8BIT(info));
  zerov = libGAP_NewBag( libGAP_T_DATOBJ, size);
  type = libGAP_TypeVec8Bit(q,mut);
  libGAP_TYPE_DATOBJ(zerov)  = type;
  libGAP_CHANGED_BAG(zerov);
  libGAP_SET_LEN_VEC8BIT(zerov, len);
  libGAP_SET_FIELD_VEC8BIT(zerov, q);
  return zerov;
}


/****************************************************************************
**
*F  FuncPROD_VEC8BIT_FFE( <self>, <vec>, <ffe> )
**
** This is the GAP callable method for *. The method installation should
** ensure that we have matching characteristics, but we may not have a common
** field
**
*/


libGAP_Obj libGAP_FuncPROD_VEC8BIT_FFE( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj ffe)
{
  libGAP_Obj prod;
  libGAP_Obj info;
  libGAP_UInt d;

  if (libGAP_VAL_FFE(ffe) == 1) /* ffe is the one */
    {
      prod = libGAP_CopyVec8Bit( vec, libGAP_IS_MUTABLE_OBJ(vec));
    }
  else if (libGAP_VAL_FFE(ffe) == 0)
    return libGAP_ZeroVec8Bit(libGAP_FIELD_VEC8BIT(vec),
           libGAP_LEN_VEC8BIT(vec),
           libGAP_IS_MUTABLE_OBJ(vec));
  
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(vec));
  d = libGAP_D_FIELDINFO_8BIT(info);

  /* family predicate should have handled this */
  assert(libGAP_CHAR_FF(libGAP_FLD_FFE(ffe)) == libGAP_P_FIELDINFO_8BIT(info));

  /* check for field compatibility */
  if (d % libGAP_DEGR_FF(libGAP_FLD_FFE(ffe)))
    {
      prod = libGAP_ProdListScl(vec,ffe);
      libGAP_CALL_1ARGS(libGAP_ConvertToVectorRep, prod);
      return prod;
    }
  
  /* Finally the main line */
  return libGAP_MultVec8BitFFE(vec, ffe);
}

/****************************************************************************
**
*F  FuncZERO_VEC8BIT( <self>, <vec> )
**
*/

libGAP_Obj libGAP_FuncZERO_VEC8BIT( libGAP_Obj self, libGAP_Obj vec )
{
  return libGAP_ZeroVec8Bit( libGAP_FIELD_VEC8BIT(vec),
          libGAP_LEN_VEC8BIT(vec),
          1);
}

/****************************************************************************
**
*F  FuncZERO_VEC8BIT_2( <self>, <q>, <len> )
**
*/

libGAP_Obj libGAP_FuncZERO_VEC8BIT_2( libGAP_Obj self, libGAP_Obj q, libGAP_Obj len )
{
  if (!libGAP_ARE_INTOBJS(q,len))
    libGAP_ErrorQuit("ZERO_VEC8BIT2: arguments must be small integers, not a %s and a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(q), (libGAP_Int)libGAP_TNAM_OBJ(len));
  return libGAP_ZeroVec8Bit( libGAP_INT_INTOBJ(q),
          libGAP_INT_INTOBJ(len),
          1L);
}

/****************************************************************************
**
*F  FuncPROD_FFE_VEC8BIT( <self>, <ffe>, <vec> )
**
** This is the GAP callable method for *. The method installation should
** ensure that we have matching characteristics, but we may not have a common
** field
**
** Here we can fall back on the method above.
*/

libGAP_Obj libGAP_FuncPROD_FFE_VEC8BIT( libGAP_Obj self, libGAP_Obj ffe, libGAP_Obj vec)
{
  return libGAP_FuncPROD_VEC8BIT_FFE(self, vec, ffe);
}

/****************************************************************************
**
*F  FuncAINV_VEC8BIT_*( <self>, <vec>)
**
** GAP Callable methods for unary -
*/

libGAP_Obj libGAP_AinvVec8Bit( libGAP_Obj vec, libGAP_UInt mut)
{
  libGAP_Obj info;
  libGAP_UInt p;
/*UInt d; */
  libGAP_UInt minusOne;
  libGAP_Obj neg;
  libGAP_FF f;
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(vec));
  p = libGAP_P_FIELDINFO_8BIT(info);

  neg = libGAP_CopyVec8Bit( vec, mut);
  /* characteristic 2 case */
  if (2 == p)
    {
      return neg;
    }

  /* Otherwise */
  f = libGAP_FiniteField( p, libGAP_D_FIELDINFO_8BIT(info));
  minusOne = libGAP_NEG_FFV( 1,libGAP_SUCC_FF(f) );
  libGAP_MultVec8BitFFEInner( neg, neg, libGAP_NEW_FFE(f,minusOne), 1, libGAP_LEN_VEC8BIT(neg));
  return neg;
}

libGAP_Obj libGAP_FuncAINV_VEC8BIT_MUTABLE( libGAP_Obj self, libGAP_Obj vec )
{
  return libGAP_AinvVec8Bit(vec, 1);
}

libGAP_Obj libGAP_FuncAINV_VEC8BIT_SAME_MUTABILITY( libGAP_Obj self, libGAP_Obj vec )
{
  return libGAP_AinvVec8Bit(vec, libGAP_IS_MUTABLE_OBJ(vec));
}

libGAP_Obj libGAP_FuncAINV_VEC8BIT_IMMUTABLE( libGAP_Obj self, libGAP_Obj vec )
{
  return libGAP_AinvVec8Bit(vec, 0);
}



/****************************************************************************
**
*F  AddVec8BitVec8BitMultInner( <sum>, <vl>, <vr>, <mult> <start>, <stop> )
**
**  This is the real vector add multiple routine. Others are all calls to this
**  one. It adds <mult>*<vr> to <vl> leaving the result in <sum>
** 
**  Addition is done from THE BLOCK containing <start> to the one
**  containing <stop> INCLUSIVE. The remainder of <sum> is unchanged.
**  <sum> may be the same vector as <vl> or
**  <vr>. <vl> and <vr> must be over the same field and <sum> must be
**  initialized as a vector over this field of length at least <stop>.
**
**  <mult> is assumed to be over the correct field  and may not be zero
**
*/

void  libGAP_AddVec8BitVec8BitMultInner( libGAP_Obj sum,
          libGAP_Obj vl,
          libGAP_Obj vr,
          libGAP_Obj mult,
          libGAP_UInt start,
          libGAP_UInt stop )
{
  libGAP_Obj info;
  libGAP_UInt p;
  libGAP_UInt elts;
  libGAP_UInt1 *ptrL;
  libGAP_UInt1 *ptrR;
  libGAP_UInt1 *ptrS;
  libGAP_UInt1 *endS;
  libGAP_UInt1 *addtab = 0;
  libGAP_UInt1 *multab;
  libGAP_UInt x;

  if (!stop)
    return;
  
  /* Handle special cases of <mult> */
  if (libGAP_VAL_FFE(mult) == 0 && sum == vl)
    return;
  
  if (libGAP_VAL_FFE(mult) == 1)
    {
      libGAP_AddVec8BitVec8BitInner( sum, vl, vr, start, stop );
      return;
    }
  
  /*  so we have some work. get the tables */
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(sum));

  /* check everything */
#if 0
  assert(libGAP_Q_FIELDINFO_8BIT(info) == libGAP_FIELD_VEC8BIT(vl));
  assert(libGAP_Q_FIELDINFO_8BIT(info) == libGAP_FIELD_VEC8BIT(vr));
  assert(libGAP_LEN_VEC8BIT(sum) >= stop);
  assert(libGAP_LEN_VEC8BIT(vl) >= stop);
  assert(libGAP_LEN_VEC8BIT(vr) >= stop);
  assert(libGAP_SIZE_FF(libGAP_FLD_FFE(mult)) == libGAP_FIELD_VEC8BIT(vl));
#endif
  
  p = libGAP_P_FIELDINFO_8BIT(info);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  
  /* Convert from 1 based to zero based addressing */
  start --;
  stop --;
  if (p != 2)
    addtab = libGAP_ADD_FIELDINFO_8BIT(info);

  multab = libGAP_SCALAR_FIELDINFO_8BIT(info) +
    256*libGAP_FELT_FFE_FIELDINFO_8BIT(info)[libGAP_VAL_FFE(mult)];
  
  ptrL = libGAP_BYTES_VEC8BIT(vl) + start/elts;
  ptrR = libGAP_BYTES_VEC8BIT(vr) + start/elts;
  ptrS = libGAP_BYTES_VEC8BIT(sum) + start/elts;
  endS = libGAP_BYTES_VEC8BIT(sum) + stop/elts + 1;
  if (p != 2)
    if (sum == vl)
      {
  const libGAP_UInt1* endS1 = endS;
  const libGAP_UInt1* addtab1 = addtab;
  const libGAP_UInt1* multab1 = multab;
  while (ptrL < endS1)
    {
      if ((x = *ptrR) != 0)
        *ptrL = addtab1[256* (*ptrL) + multab1[x]];
      ptrL++; ptrR++; 
    }
      }
    else
      while (ptrS < endS)
  *ptrS++ = addtab[256* (*ptrL++) + multab[*ptrR++]];
  else
    if (sum == vl)
      {
  while (ptrL < endS)
    {
      if ((x = *ptrR) != 0)
        *ptrL = *ptrL ^ multab[x];
      ptrR++; ptrL++;
    }
      }
    else
      while (ptrS < endS)
  *ptrS++ = *ptrL++ ^ multab[*ptrR++];
  return;
}

/****************************************************************************
**
*F  FuncMULT_ROW_VECTOR( <self>, <vec>, <mul> )
**
**  In-place scalar multiply
*/


libGAP_Obj libGAP_FuncMULT_ROWVECTOR_VEC8BITS( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj mul)
{
  libGAP_UInt q;
  q = libGAP_FIELD_VEC8BIT(vec);

  if (libGAP_VAL_FFE(mul) == 1)
    return (libGAP_Obj)0;
  
  /* Now check the field of <mul> */
  if (q != libGAP_SIZE_FF(libGAP_FLD_FFE(mul)))
    {
      libGAP_Obj info;
      libGAP_UInt d,d1;
      libGAP_FFV val;
      info = libGAP_GetFieldInfo8Bit(q);
      d = libGAP_D_FIELDINFO_8BIT(info);
      d1 = libGAP_DegreeFFE(mul);
      if (d % d1)
  return libGAP_TRY_NEXT_METHOD;
      val = libGAP_VAL_FFE(mul);
      if (val != 0)
  val = 1 + (val-1)*(q-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(mul))-1);
      mul = libGAP_NEW_FFE(libGAP_FiniteField(libGAP_P_FIELDINFO_8BIT(info),d),val);
    }
  libGAP_MultVec8BitFFEInner( vec, vec, mul, 1, libGAP_LEN_VEC8BIT(vec));
  return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  FuncADD_ROWVECTOR_VEC8BITS_5( <self>, <vl>, <vr>, <mul>, <from>, <to> )
**
**  The three argument method for AddRowVector
**
*/

libGAP_Obj libGAP_AddRowVector;

libGAP_Obj libGAP_FuncADD_ROWVECTOR_VEC8BITS_5( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr, libGAP_Obj mul, libGAP_Obj from, libGAP_Obj to)
{
  libGAP_UInt q;
  libGAP_UInt len;
  len = libGAP_LEN_VEC8BIT(vl);
  /* There may be nothing to do */
  if (libGAP_LT(to,from))
    return (libGAP_Obj) 0;
  
  if (len != libGAP_LEN_VEC8BIT(vr))
    {
      vr = libGAP_ErrorReturnObj( "AddRowVector: <left> and <right> must be vectors of the same length",
         0L,0L,
                           "you can replace <right> via 'return <right>;'");
      
      /* Now redispatch, because vr could be anything */
      return libGAP_CALL_3ARGS(libGAP_AddRowVector, vl, vr, mul);
    }
  while (libGAP_LT(libGAP_INTOBJ_INT(len),to))
    {
      to = libGAP_ErrorReturnObj( "AddRowVector: <to> (%d) is greater than the length of the vectors (%d)",
         libGAP_INT_INTOBJ(to), len,
         "you can replace <to> via 'return <to>;'");
    }
  if (libGAP_LT(to,from))
    return (libGAP_Obj) 0;
  
  /* Now we know that the characteristics must match, but not the fields */
  q = libGAP_FIELD_VEC8BIT(vl);

  /* fix up fields if necessary */
  if (q != libGAP_FIELD_VEC8BIT(vr) || q != libGAP_SIZE_FF(libGAP_FLD_FFE(mul)))
    {
      libGAP_Obj info, info1;
      libGAP_UInt d, d1, q1,d2, d0, q0, p, i;
      libGAP_FFV val;
      /* find a common field */
      info = libGAP_GetFieldInfo8Bit(q);
      d = libGAP_D_FIELDINFO_8BIT(info);
      q1 = libGAP_FIELD_VEC8BIT(vr);
      info1 = libGAP_GetFieldInfo8Bit(q1);
      d1 = libGAP_D_FIELDINFO_8BIT(info1);
      d2 = libGAP_DegreeFFE(mul);
      d0 = libGAP_LcmDegree(d,d1);
      d0 = libGAP_LcmDegree(d0,d2);
      p = libGAP_P_FIELDINFO_8BIT(info);
      assert(p == libGAP_P_FIELDINFO_8BIT(info1));
      assert(p == libGAP_CHAR_FF(libGAP_FLD_FFE(mul)));
      q0 = 1;
      for (i = 0; i < d0; i++)
  q0 *= p;
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (d0>8||q0 > 256)
  return libGAP_TRY_NEXT_METHOD;
      if ( (q0 > q && libGAP_DoFilter(libGAP_IsLockedRepresentationVector, vl) == libGAP_True) ||
     (q0 > q1 && libGAP_DoFilter(libGAP_IsLockedRepresentationVector, vr) == libGAP_True))
  return libGAP_TRY_NEXT_METHOD;
      libGAP_RewriteVec8Bit(vl,q0);
      libGAP_RewriteVec8Bit(vr,q0);
      val = libGAP_VAL_FFE(mul);
      if (val != 0)
  val = 1 + (val-1)*(q0-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(mul))-1);
      mul = libGAP_NEW_FFE(libGAP_FiniteField(p,d0),val);
      q = q0;
    }

  libGAP_AddVec8BitVec8BitMultInner( vl, vl, vr, mul, libGAP_INT_INTOBJ(from), libGAP_INT_INTOBJ(to));
  return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  FuncADD_ROWVECTOR_VEC8BITS_3( <self>, <vl>, <vr>, <mul> )
**
**  The three argument method for AddRowVector
**
*/


libGAP_Obj libGAP_FuncADD_ROWVECTOR_VEC8BITS_3( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr, libGAP_Obj mul)
{
  libGAP_UInt q;
  if (libGAP_LEN_VEC8BIT(vl) != libGAP_LEN_VEC8BIT(vr))
    {
      vr = libGAP_ErrorReturnObj( "SUM: <left> and <right> must be vectors of the same length",
         0L,0L,
                           "you can replace <right> via 'return <right>;'");
      
      /* Now redispatch, because vr could be anything */
      return libGAP_CALL_3ARGS(libGAP_AddRowVector, vl, vr, mul);
    }
  /* Now we know that the characteristics must match, but not the fields */
  q = libGAP_FIELD_VEC8BIT(vl);
  /* fix up fields if necessary */
  if (q != libGAP_FIELD_VEC8BIT(vr) || q != libGAP_SIZE_FF(libGAP_FLD_FFE(mul)))
    {
      libGAP_Obj info, info1;
      libGAP_UInt d,d1, q1,d2, d0, q0, p, i;
      libGAP_FFV val;
      /* find a common field */
      info = libGAP_GetFieldInfo8Bit(q);
      d = libGAP_D_FIELDINFO_8BIT(info);
      q1 = libGAP_FIELD_VEC8BIT(vr);
      info1 = libGAP_GetFieldInfo8Bit(q1);
      d1 = libGAP_D_FIELDINFO_8BIT(info1);
      d2 = libGAP_DegreeFFE(mul);
      d0 = libGAP_LcmDegree(d,d1);
      d0 = libGAP_LcmDegree(d0,d2);
      p = libGAP_P_FIELDINFO_8BIT(info);
      assert(p == libGAP_P_FIELDINFO_8BIT(info1));
      assert(p == libGAP_CHAR_FF(libGAP_FLD_FFE(mul)));
      q0 = 1;
      for (i = 0; i < d0; i++)
  q0 *= p;
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (d0>8||q0 > 256)
  return libGAP_TRY_NEXT_METHOD;
      if ( (q0 > q && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vl) == libGAP_True) ||
     (q0 > q1 && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vr) == libGAP_True))
  return libGAP_TRY_NEXT_METHOD;
      libGAP_RewriteVec8Bit(vl,q0);
      libGAP_RewriteVec8Bit(vr,q0);
      val = libGAP_VAL_FFE(mul);
      if (val != 0)
  val = 1 + (val-1)*(q0-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(mul))-1);
      mul = libGAP_NEW_FFE(libGAP_FiniteField(p,d0),val);
      q = q0;
    }
  libGAP_AddVec8BitVec8BitMultInner( vl, vl, vr, mul, 1, libGAP_LEN_VEC8BIT(vl));
  return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  FuncADD_ROWVECTOR_VEC8BITS_2( <self>, <vl>, <vr>)
**
**  The two argument method for AddRowVector
**
*/


libGAP_Obj libGAP_FuncADD_ROWVECTOR_VEC8BITS_2( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr)
{
  libGAP_UInt q;
  if (libGAP_LEN_VEC8BIT(vl) != libGAP_LEN_VEC8BIT(vr))
    {
      vr = libGAP_ErrorReturnObj( "SUM: <left> and <right> must be vectors of the same length",
         0L,0L,
                           "you can replace <right> via 'return <right>;'");
      
      /* Now redispatch, because vr could be anything */
      return libGAP_CALL_2ARGS(libGAP_AddRowVector, vl, vr);
    }
  /* Now we know that the characteristics must match, but not the fields */
  q = libGAP_FIELD_VEC8BIT(vl);
  /* fix up fields if necessary */
  if (q != libGAP_FIELD_VEC8BIT(vr))
    {
      libGAP_Obj info1;
      libGAP_Obj info;
      libGAP_UInt d, d1, q1, d0, q0, p, i;
      /* find a common field */
      info = libGAP_GetFieldInfo8Bit(q);
      d = libGAP_D_FIELDINFO_8BIT(info);
      q1 = libGAP_FIELD_VEC8BIT(vr);
      info1 = libGAP_GetFieldInfo8Bit(q1);
      d1 = libGAP_D_FIELDINFO_8BIT(info1);
      d0 = libGAP_LcmDegree(d,d1);
      p = libGAP_P_FIELDINFO_8BIT(info);
      assert(p == libGAP_P_FIELDINFO_8BIT(info1));
      q0 = 1;
      for (i = 0; i < d0; i++)
  q0 *= p;
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (d0>8||q0 > 256)
  return libGAP_TRY_NEXT_METHOD;
      if ( (q0 > q && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vl) == libGAP_True) ||
     (q0 > q1 && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vr) == libGAP_True))
  return libGAP_TRY_NEXT_METHOD;
      libGAP_RewriteVec8Bit(vl,q0);
      libGAP_RewriteVec8Bit(vr,q0);
      q = q0;
    }
  libGAP_AddVec8BitVec8BitInner( vl, vl, vr, 1, libGAP_LEN_VEC8BIT(vl));
  return (libGAP_Obj)0;
}


/****************************************************************************
**
*F  SumVec8BitVec8BitMult( <vl>, <vr>, <mult> )
**
**  This is perhaps the simplest wrapper for the Add..MultInner function
**  it allocates a new vector for the result, and adds the whole vectors into
**  it. Mult is promoted to the proper field if necessary.
**  The result follows the new mutability convention
**  (mutable if either argument is).
*/

libGAP_Obj libGAP_SumVec8BitVec8BitMult( libGAP_Obj vl, libGAP_Obj vr, libGAP_Obj mult )
{
  libGAP_Obj sum;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt q;
  libGAP_UInt len;
  libGAP_FFV v;
  libGAP_Obj type;

  q = libGAP_FIELD_VEC8BIT(vl);
  len = libGAP_LEN_VEC8BIT(vl);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  sum = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT(len,elts));
  libGAP_SET_LEN_VEC8BIT(sum, len);
  type = libGAP_TypeVec8Bit(q, libGAP_IS_MUTABLE_OBJ(vl) || libGAP_IS_MUTABLE_OBJ(vr));
  libGAP_TYPE_DATOBJ(sum) = type;
  libGAP_SET_FIELD_VEC8BIT(sum,q );
  libGAP_CHANGED_BAG(sum);
  if (libGAP_SIZE_FF(libGAP_FLD_FFE(mult)) != q)
    {
      v = libGAP_VAL_FFE(mult);
      if (v != 0)
  v = 1+ (v-1)*(q-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(mult))-1);
      mult = libGAP_NEW_FFE( libGAP_FiniteField(libGAP_P_FIELDINFO_8BIT(info),
          libGAP_D_FIELDINFO_8BIT(info)),v);
    }
  libGAP_AddVec8BitVec8BitMultInner( sum, vl, vr, mult, 1, len);
  return sum;
}

/****************************************************************************
**
*F  DiffVec8BitVec8Bit( <vl>, <vr> )
**
*/

libGAP_Obj libGAP_DiffVec8BitVec8Bit( libGAP_Obj vl, libGAP_Obj vr)
{
  libGAP_Obj info;
  libGAP_FF f;
  libGAP_FFV minusOne;
  libGAP_Obj MinusOne;
  libGAP_Obj dif;
  libGAP_Obj type;
  
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(vl));
  f = libGAP_FiniteField( libGAP_P_FIELDINFO_8BIT(info), libGAP_D_FIELDINFO_8BIT(info));
  minusOne = libGAP_NEG_FFV( 1,libGAP_SUCC_FF(f) );
  MinusOne = libGAP_NEW_FFE(f,minusOne);

  if (libGAP_LEN_VEC8BIT(vl) == libGAP_LEN_VEC8BIT(vr))
    return libGAP_SumVec8BitVec8BitMult(vl, vr, MinusOne );
  else if (libGAP_LEN_VEC8BIT(vl) < libGAP_LEN_VEC8BIT(vr))
    {
      dif = libGAP_MultVec8BitFFE( vr, MinusOne);
      libGAP_AddVec8BitVec8BitInner( dif, dif, vl, 1, libGAP_LEN_VEC8BIT(vl));
      if (libGAP_IS_MUTABLE_OBJ(vl) && !libGAP_IS_MUTABLE_OBJ(vr))
  {
    type = libGAP_TypeVec8Bit(libGAP_Q_FIELDINFO_8BIT(info), 1);
    libGAP_TYPE_DATOBJ(dif) = type;
  }
      return dif;
    }
  else
    {
      dif = libGAP_CopyVec8Bit(vl, libGAP_IS_MUTABLE_OBJ(vl) || libGAP_IS_MUTABLE_OBJ(vr));
      libGAP_AddVec8BitVec8BitMultInner( dif, dif, vr, MinusOne, 1, libGAP_LEN_VEC8BIT(vr));
      return dif;
    }
}


/****************************************************************************
**
*F  FuncDIFF_VEC8BIT_VEC8BIT ( <self>, <vl>, <vr> )
**
**  GAP callable method for binary -
*/
libGAP_Obj libGAP_FuncDIFF_VEC8BIT_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr)
{
  libGAP_Obj diff;
/*UInt p; */
  

  if (libGAP_FIELD_VEC8BIT(vl) != libGAP_FIELD_VEC8BIT(vr))
    {
      libGAP_UInt ql = libGAP_FIELD_VEC8BIT(vl), qr = libGAP_FIELD_VEC8BIT(vr);
      libGAP_Obj infol = libGAP_GetFieldInfo8Bit(ql), infor = libGAP_GetFieldInfo8Bit(qr);
      libGAP_UInt newd = libGAP_LcmDegree( libGAP_D_FIELDINFO_8BIT(infol), libGAP_D_FIELDINFO_8BIT(infor));
      libGAP_UInt p,newq;
      libGAP_UInt i;
      p = libGAP_P_FIELDINFO_8BIT(infol);
      assert(p == libGAP_P_FIELDINFO_8BIT(infor));
      newq = 1;
      for (i = 0; i < newd; i++)
  newq *= p;
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (newd>8||newq > 256 ||
    (ql != newq && libGAP_True == libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector,vl)) ||
    (qr != newq && libGAP_True == libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector,vr)))
  {
    diff = libGAP_DiffListList(vl,vr);
    libGAP_CALL_1ARGS(libGAP_ConvertToVectorRep, diff);
    return diff;
  }
      else
  {
    libGAP_RewriteVec8Bit( vl, newq);
    libGAP_RewriteVec8Bit( vr, newq);
  }
    }
  
  /* Finally the main line */
  return libGAP_DiffVec8BitVec8Bit(vl, vr);
}

/****************************************************************************
**
*F  CmpVec8BitVec8Bit( <vl>, <vr> ) .. comparison, returns -1, 0 or 1
**
**  characteristic and field should have been checked outside, but we must
**  deal with length variations
*/

libGAP_Int libGAP_CmpVec8BitVec8Bit( libGAP_Obj vl, libGAP_Obj vr )
{
  libGAP_Obj info;
  libGAP_UInt q;
  libGAP_UInt lenl;
  libGAP_UInt lenr;
  libGAP_UInt1 *ptrL;
  libGAP_UInt1 *ptrR;
  libGAP_UInt1 *endL;
  libGAP_UInt1 *endR;
  libGAP_UInt elts;
  libGAP_UInt vall,valr;
  libGAP_UInt e;
  libGAP_UInt1 *gettab;
  libGAP_Obj *ffe_elt;
  libGAP_UInt len;
  assert(libGAP_FIELD_VEC8BIT(vl) == libGAP_FIELD_VEC8BIT(vr));
  q = libGAP_FIELD_VEC8BIT(vl);
  info = libGAP_GetFieldInfo8Bit(q);
  lenl = libGAP_LEN_VEC8BIT(vl);
  lenr = libGAP_LEN_VEC8BIT(vr);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  ptrL = libGAP_BYTES_VEC8BIT(vl);
  ptrR = libGAP_BYTES_VEC8BIT(vr);

  /* we stop a little short, so as to handle the final byte
     separately */
  endL = ptrL + lenl/elts;
  endR = ptrR + lenr/elts;
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  ffe_elt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);
  while (ptrL < endL && ptrR < endR)
    {
      if (*ptrL == *ptrR)
  {
    ptrL++;
    ptrR++;
  }
      else
  {
    for (e = 0; e < elts; e++)
      {
        vall = gettab[*ptrL + 256*e];
        valr = gettab[*ptrR + 256*e];
        if (vall != valr)
    {
      if (libGAP_LT( ffe_elt[vall], ffe_elt[valr]))
        return -1;
      else
        return 1;
    }
      }
    libGAP_ErrorQuit("panic: bytes differed but all entries the same",
        0L, 0L);
  }
    }
  /* now the final byte */

  /* a quick and easy case */
  if (lenl == lenr && *ptrL == *ptrR) 
    return 0;

  /* the more general case */
  if (lenl < lenr)
    len = lenl;
  else
    len = lenr;

  /* look first at the shared part */
  for (e = 0; e < (len % elts); e++) 
    {
      vall = gettab[*ptrL + 256*e];
      valr = gettab[*ptrR + 256*e];
      if (vall != valr)  
  {
    if (libGAP_LT( ffe_elt[vall], ffe_elt[valr]))  
      return -1;
    else
      return 1;
  }
    }
    /* if that didn't decide then the longer list is bigger */
  if (lenr > lenl)
    return -1;
  else if (lenr == lenl)
    return 0;
  else
    return 1;
}

/****************************************************************************
**
*F  ScalarProductVec8Bits( <vl>, <vr> ) scalar product of vectors
**
**  Assumes that length and field match
**
*/

libGAP_Obj libGAP_ScalarProductVec8Bits( libGAP_Obj vl, libGAP_Obj vr )
{
  libGAP_Obj info;
  libGAP_UInt1 acc;
  libGAP_UInt1 *ptrL;
  libGAP_UInt1 *ptrR;
  libGAP_UInt1 *endL;
  libGAP_UInt len;
  libGAP_UInt q;
  libGAP_UInt elts;
  libGAP_UInt1 contrib;
  libGAP_UInt1 *inntab;
  libGAP_UInt1 *addtab;
  len = libGAP_LEN_VEC8BIT(vl);
  if (len > libGAP_LEN_VEC8BIT(vr))
    len = libGAP_LEN_VEC8BIT(vr);
  q = libGAP_FIELD_VEC8BIT(vl);
  assert(q == libGAP_FIELD_VEC8BIT(vr));
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  
  ptrL = libGAP_BYTES_VEC8BIT(vl);
  ptrR = libGAP_BYTES_VEC8BIT(vr);
  endL = ptrL + (len+elts-1)/elts;
  acc = 0;
  inntab = libGAP_INNER_FIELDINFO_8BIT(info);
  if (libGAP_P_FIELDINFO_8BIT(info) == 2)
    {
      while (ptrL < endL)
  {
    contrib = inntab [*ptrL++ + 256* *ptrR++];
    acc ^= contrib;
  }
    }
  else
    {
      addtab = libGAP_ADD_FIELDINFO_8BIT(info);
      while (ptrL < endL)
  {
    contrib = inntab [*ptrL++ + 256* *ptrR++];
    acc = addtab[256*acc+contrib];
  }
      
    }
  return libGAP_FFE_FELT_FIELDINFO_8BIT(info)[libGAP_GETELT_FIELDINFO_8BIT(info)[acc]];
  
}

/****************************************************************************
**
*F  FuncPROD_VEC8BIT_VEC8BIT( <self>, <vl>, <vr> )
*/

libGAP_Obj libGAP_FuncPROD_VEC8BIT_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr )
{
  if (libGAP_FIELD_VEC8BIT(vl) != libGAP_FIELD_VEC8BIT(vr))
    return libGAP_ProdListList(vl,vr);

  return libGAP_ScalarProductVec8Bits(vl, vr);
}

/****************************************************************************
**
*F  UInt DistanceVec8Bits( <vl>, <vr> ) Hamming distance
**
**  Assumes that length and field match
**
*/

libGAP_UInt libGAP_DistanceVec8Bits( libGAP_Obj vl, libGAP_Obj vr )
{
  libGAP_Obj info;
  libGAP_UInt1 *ptrL;
  libGAP_UInt1 *ptrR;
  libGAP_UInt1 *endL;
  libGAP_UInt len;
  libGAP_UInt q;
  libGAP_UInt elts;
  libGAP_UInt acc;
  libGAP_UInt i;
  libGAP_UInt1 *gettab;

  len = libGAP_LEN_VEC8BIT(vl);
  q = libGAP_FIELD_VEC8BIT(vl);
  assert(q == libGAP_FIELD_VEC8BIT(vr));
  assert(len == libGAP_LEN_VEC8BIT(vr));
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  
  ptrL = libGAP_BYTES_VEC8BIT(vl);
  ptrR = libGAP_BYTES_VEC8BIT(vr);
  endL = ptrL + (len+elts-1)/elts;

  acc = 0;
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  
  while (ptrL < endL)
    {
      if (*ptrL != *ptrR)
  {
    for (i = 0; i < elts; i++)
      if (gettab[*ptrL + 256*i] != gettab[*ptrR + 256*i])
        acc++;
  }
      ptrL++;
      ptrR++;
    }
  return acc;
}

/****************************************************************************
**
*F  FuncDISTANCE_VEC8BIT_VEC8BIT( <self>, <vl>, <vr> )
*/

libGAP_Obj libGAP_FuncDISTANCE_VEC8BIT_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr )
{
  if (libGAP_FIELD_VEC8BIT(vl) != libGAP_FIELD_VEC8BIT(vr) ||
      libGAP_LEN_VEC8BIT(vl) != libGAP_LEN_VEC8BIT(vr))
    return libGAP_TRY_NEXT_METHOD;

  return libGAP_INTOBJ_INT(libGAP_DistanceVec8Bits(vl, vr));
}



/****************************************************************************
**
*F  DistDistrib8Bits( <veclis>, <ovec>, <d>, <osum>, <pos>, <l>, <m>)
**
*/
void libGAP_DistDistrib8Bits(
  libGAP_Obj   veclis, /* pointers to matrix vectors and their multiples */
  libGAP_Obj           vec,    /* vector we compute distance to */
  libGAP_Obj   d,  /* distances list */
  libGAP_Obj           sum,  /* position of the sum vector */
  libGAP_UInt    pos,  /* recursion depth */
  libGAP_UInt    l /* length of basis */
  ) 
{
  libGAP_UInt    i;
  libGAP_UInt    di;
  libGAP_Obj   cnt;
  libGAP_Obj   vp;
  libGAP_Obj           one;
  libGAP_Obj           tmp;
  libGAP_UInt          len;
  libGAP_UInt          q;

  vp = libGAP_ELM_PLIST(veclis,pos);
  one = libGAP_INTOBJ_INT(1);

  len = libGAP_LEN_VEC8BIT(sum);
  q = libGAP_FIELD_VEC8BIT(sum);
  for (i=0; i<q; i++)
    {
      if (pos < l)
  {
    libGAP_DistDistrib8Bits( veclis, vec, d, sum, pos+1, l);
  }
      else
  {
    di=libGAP_DistanceVec8Bits(sum,vec);
    cnt=libGAP_ELM_PLIST(d,di+1);
    if (libGAP_IS_INTOBJ(cnt) && libGAP_SUM_INTOBJS(tmp, cnt, one))
      {
        cnt = tmp;
        libGAP_SET_ELM_PLIST(d,di+1,cnt);
      }
    else
      {
        cnt=libGAP_SumInt(cnt,one);
        libGAP_SET_ELM_PLIST(d,di+1,cnt);
        libGAP_CHANGED_BAG(d);
      }
      }
    libGAP_AddVec8BitVec8BitInner(sum,sum,libGAP_ELM_PLIST(vp,i+1),1,len);
  }
  libGAP_TakeInterrupt();
}

libGAP_Obj libGAP_FuncDISTANCE_DISTRIB_VEC8BITS(
  libGAP_Obj   self,
  libGAP_Obj   veclis, /* pointers to matrix vectors and their multiples */
  libGAP_Obj   vec,    /* vector we compute distance to */
  libGAP_Obj   d ) /* distances list */

{
  libGAP_Obj   sum; /* sum vector */
  libGAP_UInt    len;
  libGAP_UInt           q;

  len = libGAP_LEN_VEC8BIT(vec);
  q = libGAP_FIELD_VEC8BIT(vec);

  /* get space for sum vector and zero out */
  sum = libGAP_ZeroVec8Bit(q, len, 0);
  /* do the recursive work */
  libGAP_DistDistrib8Bits(veclis,vec,d,sum,1, libGAP_LEN_PLIST(veclis));

  return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F
*/

void libGAP_OverwriteVec8Bit( libGAP_Obj dst, libGAP_Obj src)
{
  libGAP_UInt1 *ptrS;
  libGAP_UInt1 *ptrD;
  libGAP_UInt size;
  libGAP_UInt n;
  size = libGAP_SIZE_BAG(src);
  ptrS = libGAP_BYTES_VEC8BIT(src);
  ptrD = libGAP_BYTES_VEC8BIT(dst);
  for (n = 3*sizeof(libGAP_UInt); n < size; n++)
    *ptrD++ = *ptrS++;

}

libGAP_UInt libGAP_AClosVec8Bit( 
      libGAP_Obj   veclis, /* pointers to matrix vectors and their multiples */
      libGAP_Obj           vec,    /* vector we compute distance to */
      libGAP_Obj           sum,  /* position of the sum vector */
      libGAP_UInt    pos,  /* recursion depth */
      libGAP_UInt    l,  /* length of basis */
      libGAP_UInt    cnt,  /* numbr of vectors used already */
      libGAP_UInt    stop, /* stop value */
      libGAP_UInt    bd, /* best distance so far */
      libGAP_Obj   bv, /* best vector so far */
      libGAP_Obj           coords,
      libGAP_Obj           bcoords
      )
{
  libGAP_UInt    i,j;
  libGAP_UInt    di;
  libGAP_Obj   vp;
/*Obj           one; */
/*Obj           tmp; */
  libGAP_UInt q;
  libGAP_UInt len;
  
  /* This is the case where we do not add any multiple of
     the current basis vector */
  if ( pos+cnt<l ) {
    bd = libGAP_AClosVec8Bit(veclis,vec,sum,pos+1,l,cnt,stop,bd,bv,coords,bcoords);
    if (bd<=stop) {
      return bd;
    }
  }
  q = libGAP_FIELD_VEC8BIT(vec);
  len = libGAP_LEN_VEC8BIT(vec);
  vp = libGAP_ELM_PLIST(veclis,pos);

  /* we need to add each scalar multiple and recurse */
  for (i=1; i <  q ; i++) {
    libGAP_AddVec8BitVec8BitInner(sum,sum,libGAP_ELM_PLIST(vp,i),1,len);
    if (coords)
      libGAP_SET_ELM_PLIST(coords,pos,libGAP_INTOBJ_INT(i));
    if (cnt == 0)
      {
  /* do we have a new best case */
  di=libGAP_DistanceVec8Bits(sum,vec);
  if (di < bd) {
    bd=di;
    libGAP_OverwriteVec8Bit(bv, sum);
    if (coords)
      for (j = 1; j <= l; j++)
        {
    libGAP_Obj x;
    x = libGAP_ELM_PLIST(coords,j);
    libGAP_SET_ELM_PLIST(bcoords,j,x);
        }
    if (bd <= stop)
      return bd;
  }
      }
    else if (pos < l)
      {
  bd = libGAP_AClosVec8Bit(veclis,vec,sum,pos+1,l,cnt-1,stop,bd,bv,coords,bcoords);
  if (bd<=stop) {
    return bd;
  }
      }
  }
  /* reset component */
  libGAP_AddVec8BitVec8BitInner(sum,sum,libGAP_ELM_PLIST(vp,q),1,len);
  if (coords)
    libGAP_SET_ELM_PLIST(coords,pos,libGAP_INTOBJ_INT(0));
  
  libGAP_TakeInterrupt();
  return bd;
}

/****************************************************************************
**
*F  
*/

libGAP_Obj libGAP_FuncAClosVec8Bits( 
          libGAP_Obj   self,
          libGAP_Obj   veclis, /* pointers to matrix vectors and their multiples */
          libGAP_Obj   vec,    /* vector we compute distance to */
          libGAP_Obj   cnt,  /* distances list */
          libGAP_Obj   stop) /* distances list */
     
{
  libGAP_Obj   sum; /* sum vector */
  libGAP_Obj   best; /* best vector */
/*UInt *  ptr; */
/*UInt *        end; */
  libGAP_UInt    len;
  libGAP_UInt q;

  if (!libGAP_ARE_INTOBJS(cnt,stop))
    libGAP_ErrorQuit("A_CLOSEST_VEC8BIT: cnt and stop must be smal integers, not a %s and a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(cnt), (libGAP_Int)libGAP_TNAM_OBJ(stop));
  

  q = libGAP_FIELD_VEC8BIT(vec);
  len = libGAP_LEN_VEC8BIT(vec);

  /* get space for sum vector and zero out */

  sum = libGAP_ZeroVec8Bit(q, len, 1);
  best = libGAP_ZeroVec8Bit(q, len, 1);

  /* do the recursive work */
  libGAP_AClosVec8Bit(veclis,vec,sum,1, libGAP_LEN_PLIST(veclis),
         libGAP_INT_INTOBJ(cnt),libGAP_INT_INTOBJ(stop),len+1, /* maximal value +1 */
         best, (libGAP_Obj)0, (libGAP_Obj)0);
  
  return best;
}

/****************************************************************************
**
*F  
*/

libGAP_Obj libGAP_FuncAClosVec8BitsCoords( 
          libGAP_Obj   self,
          libGAP_Obj   veclis, /* pointers to matrix vectors and their multiples */
          libGAP_Obj   vec,    /* vector we compute distance to */
          libGAP_Obj   cnt,  /* distances list */
          libGAP_Obj   stop) /* distances list */
     
{
  libGAP_Obj   sum; /* sum vector */
  libGAP_Obj   best; /* best vector */
/*UInt *  ptr; */
/*UInt *        end; */
  libGAP_UInt    len, len2,i;
  libGAP_UInt q;
  libGAP_Obj coords;
  libGAP_Obj bcoords;
  libGAP_Obj res;

  if (!libGAP_ARE_INTOBJS(cnt,stop))
    libGAP_ErrorQuit("A_CLOSEST_VEC8BIT: cnt and stop must be smal integers, not a %s and a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(cnt), (libGAP_Int)libGAP_TNAM_OBJ(stop));
  

  q = libGAP_FIELD_VEC8BIT(vec);
  len = libGAP_LEN_VEC8BIT(vec);

  /* get space for sum vector and zero out */

  sum = libGAP_ZeroVec8Bit(q, len, 1);
  best = libGAP_ZeroVec8Bit(q, len, 1);
  len2 = libGAP_LEN_PLIST(veclis);
  coords = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, len2);
  bcoords = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, len2);
  libGAP_SET_LEN_PLIST(coords,len2);
  libGAP_SET_LEN_PLIST(bcoords,len2);
  for (i = 1; i <= len2; i++)
    {
      libGAP_SET_ELM_PLIST(coords,i,libGAP_INTOBJ_INT(0));
      libGAP_SET_ELM_PLIST(bcoords,i,libGAP_INTOBJ_INT(0));
    }

  /* do the recursive work */
  libGAP_AClosVec8Bit(veclis,vec,sum,1, libGAP_LEN_PLIST(veclis),
         libGAP_INT_INTOBJ(cnt),libGAP_INT_INTOBJ(stop),len+1, /* maximal value +1 */
         best, coords, bcoords);

  res = libGAP_NEW_PLIST(libGAP_T_PLIST_DENSE_NHOM,2);
  libGAP_SET_LEN_PLIST(res,2);
  libGAP_SET_ELM_PLIST(res,1,best);
  libGAP_SET_ELM_PLIST(res,2,bcoords);
  libGAP_CHANGED_BAG(res);
  return res;
}


/****************************************************************************
**
*F  FuncNUMBER_VEC8BIT( <self>, <vec> )
**
*/

libGAP_Obj libGAP_FuncNUMBER_VEC8BIT (libGAP_Obj self, libGAP_Obj vec)

{
    libGAP_Obj     info;
    libGAP_UInt                elts;
    libGAP_UInt                len;
    libGAP_UInt                i;
    libGAP_Obj                 elt;
    libGAP_UInt1               *gettab;
    libGAP_UInt1               *ptrS;
    libGAP_Obj                 *convtab;

    libGAP_Obj                 res;
    libGAP_Obj     f;
    
    info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(vec));
    elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
    gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
    convtab = libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info);
    ptrS = libGAP_BYTES_VEC8BIT(vec);
    len = libGAP_LEN_VEC8BIT(vec);
    res = libGAP_INTOBJ_INT(0);
    f = libGAP_INTOBJ_INT(libGAP_FIELD_VEC8BIT(vec)); /* Field size as GAP integer */

    for (i = 0; i < len; i++)
      {
  elt = convtab[gettab[ptrS[i/elts] + 256*(i%elts)]];
  res=libGAP_ProdInt(res,f); /* ``shift'' */
  res=libGAP_SumInt(res,elt);
  if (!libGAP_IS_INTOBJ(res)) {
    /* a garbage collection might have moved the pointers */
    gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
    convtab = libGAP_GAPSEQ_FELT_FIELDINFO_8BIT(info);
    ptrS = libGAP_BYTES_VEC8BIT(vec);
  }
      }
    
    return res;
}

/****************************************************************************
**
*F FuncCOSET_LEADERS_INNER_8BITS( <self>, <veclis>, <weight>, <tofind>, <leaders> )
**
** Search for new coset leaders of weight <weight>
*/


#include <stdio.h>

libGAP_UInt libGAP_CosetLeadersInner8Bits( libGAP_Obj veclis,
         libGAP_Obj v,
         libGAP_Obj w,
         libGAP_UInt weight,
         libGAP_UInt pos,
         libGAP_Obj leaders,
         libGAP_UInt tofind,
         libGAP_Obj felts)
{
  libGAP_UInt found = 0;
  libGAP_UInt len = libGAP_LEN_VEC8BIT(v);
  libGAP_UInt lenw = libGAP_LEN_VEC8BIT(w);
  libGAP_UInt sy;
  libGAP_Obj u;
  libGAP_Obj vc;
  libGAP_UInt i,j;
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_UInt1 *settab;
  libGAP_UInt elts;
  libGAP_UInt1 *ptr, *ptrw;
  libGAP_UInt1 *gettab;
  libGAP_UInt1 *feltffe;
  libGAP_Obj x;
  libGAP_Obj vp;
  
  q = libGAP_FIELD_VEC8BIT(v);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  settab = libGAP_SETELT_FIELDINFO_8BIT(info);
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  ptrw = libGAP_BYTES_VEC8BIT(w);
  if (weight == 1)
    {
      for (i = pos; i <= len; i++)
  {
                          vp = libGAP_ELM_PLIST(veclis, i);
    u = libGAP_ELM_PLIST(vp,1);
    libGAP_AddVec8BitVec8BitInner(w,w,u,1,lenw);
    ptr = libGAP_BYTES_VEC8BIT(v)+ (i-1)/elts;
    *ptr = settab[*ptr + 256 * (elts + ((i-1) % elts))];
    sy = 0;
    for (j = 0; j < lenw; j++)
      {
          libGAP_UInt xxxx;
          sy *= q;
          xxxx = gettab[ptrw[j / elts] + 256* (j % elts)];
          sy += xxxx;
      }
    if ((libGAP_Obj) 0 == libGAP_ELM_PLIST(leaders,sy+1))
      {
        libGAP_UInt k;
        libGAP_Obj qk;
        libGAP_Obj wc;
        vc = libGAP_CopyVec8Bit(v, 0);
        libGAP_SET_ELM_PLIST(leaders,sy+1,vc);
        libGAP_CHANGED_BAG(leaders);
        /* Also record all the multiples here */
        wc = libGAP_ZeroVec8Bit(q,lenw,1);
        settab = libGAP_SETELT_FIELDINFO_8BIT(info);
        gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
                     ptr = libGAP_BYTES_VEC8BIT(v)+ (i-1)/elts;
        ptrw = libGAP_BYTES_VEC8BIT(w);
        for (k = 2; k < q; k++)
    {     
      qk = libGAP_FFE_FELT_FIELDINFO_8BIT(info)[k];
      libGAP_MultVec8BitFFEInner(wc,w,qk,1,lenw);
                          ptrw = libGAP_BYTES_VEC8BIT(wc);
      sy = 0;
      for (j = 0; j < lenw; j++)
        {
          libGAP_UInt xxxx;
          sy *= q;
          xxxx = gettab[ptrw[j / elts] + 256* (j % elts)];
          sy += xxxx;
        }
      vc = libGAP_ZeroVec8Bit(q, len, 0);
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
      ptr = libGAP_BYTES_VEC8BIT(v)+ (i-1)/elts;
      ptrw = libGAP_BYTES_VEC8BIT(w);
      libGAP_MultVec8BitFFEInner(vc,v,qk,1,len);
      libGAP_SET_ELM_PLIST(leaders, sy+1, vc);
      libGAP_CHANGED_BAG(leaders);
    }
        found += (q-1);
        if (found == tofind)
    return found;
      }
    u = libGAP_ELM_PLIST(vp,q+1);
    libGAP_AddVec8BitVec8BitInner(w,w,u,1,lenw);
    *ptr = settab[*ptr + 256 * ((i-1) % elts)];
  }
    }
  else
    {
      if (pos + weight <= len)
  {
      found += libGAP_CosetLeadersInner8Bits(veclis, v, w, weight, pos+1, leaders, tofind, felts);
    if (found == tofind)
      return found;
  }
      
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      feltffe = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
      vp = libGAP_ELM_PLIST(veclis, pos);
      for (i = 1; i < q; i++)
  {
    u = libGAP_ELM_PLIST(vp,i);
    libGAP_AddVec8BitVec8BitInner(w,w,u,1,lenw);
    ptr = libGAP_BYTES_VEC8BIT(v)+ (pos-1)/elts;
    x = libGAP_ELM_PLIST(felts,i+1);
    *ptr = settab[*ptr + 256 * (elts*feltffe[libGAP_VAL_FFE(x)] + ((pos-1) % elts))];
    found += libGAP_CosetLeadersInner8Bits(veclis,v,w,weight-1,pos+1,leaders,tofind-found,felts);
    if (found == tofind)
      return found;
  }
       settab = libGAP_SETELT_FIELDINFO_8BIT(info);
    feltffe = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
    u = libGAP_ELM_PLIST(vp,q);
    libGAP_AddVec8BitVec8BitInner(w,w,u,1,lenw);
  
      ptr = libGAP_BYTES_VEC8BIT(v) + (pos-1)/elts;
      *ptr = settab[*ptr + 256 * ((pos-1) % elts)];
      
    }
  libGAP_TakeInterrupt();
  return found;
}




libGAP_Obj libGAP_FuncCOSET_LEADERS_INNER_8BITS( libGAP_Obj self, libGAP_Obj veclis, libGAP_Obj weight, libGAP_Obj tofind, libGAP_Obj leaders, libGAP_Obj felts)
{
  libGAP_Obj v,w;
  libGAP_UInt lenv, lenw, q;
  if (!libGAP_ARE_INTOBJS(weight, tofind))
    libGAP_ErrorQuit("COSET_LEADERS_INNER_8BITS: weight and tofind must be small integers, not a %s and a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(weight), (libGAP_Int)libGAP_TNAM_OBJ(tofind));
  lenv = libGAP_LEN_PLIST(veclis);
  q = libGAP_LEN_PLIST(felts);
  v = libGAP_ZeroVec8Bit( q, lenv, 1);
  lenw = libGAP_LEN_VEC8BIT(libGAP_ELM_PLIST(libGAP_ELM_PLIST(veclis,1),1));
  w = libGAP_ZeroVec8Bit(q, lenw, 1);
  return libGAP_INTOBJ_INT(libGAP_CosetLeadersInner8Bits( veclis, v, w, libGAP_INT_INTOBJ(weight), 1, leaders, libGAP_INT_INTOBJ(tofind), felts));
}


/****************************************************************************
**
*F  FuncEQ_VEC8BIT_VEC8BIT( <self>, <vl>, <vr> )
**
*/

libGAP_Obj libGAP_FuncEQ_VEC8BIT_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr )
{
  if (libGAP_FIELD_VEC8BIT(vl) != libGAP_FIELD_VEC8BIT(vr))
    return libGAP_EqListList(vl,vr) ? libGAP_True : libGAP_False;

  if (libGAP_LEN_VEC8BIT(vl) != libGAP_LEN_VEC8BIT(vr))
    return libGAP_False;
  
  return (libGAP_CmpVec8BitVec8Bit(vl,vr) == 0) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  FuncLT_VEC8BIT_VEC8BIT( <self>, <vl>, <vr> )
**
*/

libGAP_Obj libGAP_FuncLT_VEC8BIT_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj vr )
{
  if (libGAP_FIELD_VEC8BIT(vl) != libGAP_FIELD_VEC8BIT(vr))
    return libGAP_LtListList(vl,vr) ? libGAP_True : libGAP_False;
  
  return (libGAP_CmpVec8BitVec8Bit(vl,vr) == -1) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F * * * * * * * * * * * * list access functions  * * * * * * * * * * * * * *
*/

/****************************************************************************
**
*F  FuncSHALLOWCOPY_VEC8BIT( <self>, <list> ) . shallowcopy method
**
*/


libGAP_Obj libGAP_FuncSHALLOWCOPY_VEC8BIT( libGAP_Obj self, libGAP_Obj list )
{
  return libGAP_CopyVec8Bit(list, 1);
}


/****************************************************************************
**
*F  FuncLEN_VEC8BIT( <self>, <list> )  . . . . . . . .  length of a vector
*/
libGAP_Obj libGAP_FuncLEN_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_INTOBJ_INT(libGAP_LEN_VEC8BIT(list));
}

/****************************************************************************
**
*F  FuncQ_VEC8BIT( <self>, <list> )  . . . . . . . .  length of a vector
*/
libGAP_Obj libGAP_FuncQ_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    return libGAP_INTOBJ_INT(libGAP_FIELD_VEC8BIT(list));
}


/****************************************************************************
**
*F  FuncELM0_VEC8BIT( <self>, <list>, <pos> )  . select an elm of an 8bit vector
**
**  'ELM0_VEC8BIT'  returns the element at the  position  <pos> of the boolean
**  list <list>, or `Fail' if <list> has no assigned  object at <pos>.  It is
**  the  responsibility of  the caller to   ensure  that <pos> is  a positive
**  integer.
*/

libGAP_Obj libGAP_FuncELM0_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UInt                p;
    libGAP_Obj     info;
    libGAP_UInt elts;

    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorQuit("ELM0_VEC8BIT: position must be a small integer, not a %s",
    (libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);
    if ( libGAP_LEN_VEC8BIT(list) < p ) {
        return libGAP_Fail;
    }
    else {
      info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
      elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
      return libGAP_FFE_FELT_FIELDINFO_8BIT(info)[
    libGAP_GETELT_FIELDINFO_8BIT(info)[libGAP_BYTES_VEC8BIT(list)[(p-1)/elts] +
             256*((p-1)%elts)]];
    }
}


/****************************************************************************
**
*F  FuncELM_VEC8BIT( <self>, <list>, <pos> ) . . select an elm of a GF2 vector
**
**  'ELM_VEC8BIT' returns the element at the position <pos>  of the GF2 vector
**  <list>.   An  error  is signalled  if  <pos>  is  not bound.    It is the
**  responsibility of the caller to ensure that <pos> is a positive integer.
*/
libGAP_Obj libGAP_FuncELM_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UInt                p;
    libGAP_Obj     info;
    libGAP_UInt elts;

    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorQuit("ELM0_VEC8BIT: position must be a small integer, not a %s",
    (libGAP_Int)libGAP_TNAM_OBJ(pos),0L);
    p = libGAP_INT_INTOBJ(pos);
    if ( libGAP_LEN_VEC8BIT(list) < p ) {
        libGAP_ErrorReturnVoid(
            "List Element: <list>[%d] must have an assigned value",
            p, 0L, "you can 'return;' after assigning a value" );
        return libGAP_ELM_LIST( list, p );
    }
    else {
      info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
      elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
      return libGAP_FFE_FELT_FIELDINFO_8BIT(info)[
    libGAP_GETELT_FIELDINFO_8BIT(info)[libGAP_BYTES_VEC8BIT(list)[(p-1)/elts] +
             256*((p-1)%elts)]];

    }
}


/****************************************************************************
**
*F  FuncELMS_VEC8BIT( <self>, <list>, <poss> ) . select elms of an 8 bit vector
**
**  The results are returned in the compressed format
*/
libGAP_Obj libGAP_FuncELMS_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_UInt                p;
    libGAP_Obj                 pos;
    libGAP_Obj     info;
    libGAP_UInt                elts;
    libGAP_UInt                len;
    libGAP_Obj                 res;
    libGAP_UInt                i;
    libGAP_UInt                elt;
    libGAP_UInt1               *gettab;
    libGAP_UInt1               *settab;
    libGAP_UInt1               *ptrS;
    libGAP_UInt1               *ptrD;
    libGAP_UInt                 e;
    libGAP_UInt1                byte;
    libGAP_UInt                 len2;

    len = libGAP_LEN_PLIST(poss);
    info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
    len2 = libGAP_LEN_VEC8BIT(list);
    elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
    res = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT( len, elts));
    libGAP_TYPE_DATOBJ(res) = libGAP_TYPE_DATOBJ(list);
    libGAP_SET_FIELD_VEC8BIT(res, libGAP_FIELD_VEC8BIT(list));
    libGAP_SET_LEN_VEC8BIT(res, len);
    gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
    settab = libGAP_SETELT_FIELDINFO_8BIT(info);
    ptrS = libGAP_BYTES_VEC8BIT(list);
    ptrD = libGAP_BYTES_VEC8BIT(res);
    e = 0;
    byte = 0;
    for (i = 1; i <= len; i++)
      {
  pos = libGAP_ELM_PLIST(poss,i);
  if (!libGAP_IS_INTOBJ(pos))
    libGAP_ErrorQuit("ELMS_VEC8BIT: positions list includes a %s, should all be small integers",
        (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L);
  if (pos <= libGAP_INTOBJ_INT(0))
    libGAP_ErrorQuit("ELMS_VEC8BIT: positions list includes a non-positive number", 0L, 0L);
  p = libGAP_INT_INTOBJ(pos);
  if (p > len2)
    libGAP_ErrorQuit("ELMS_VEC8BIT: positions list includes index %d in a list of length %d",
        (libGAP_Int)p,(libGAP_Int)len2);
  elt = gettab[ptrS[(p-1)/elts] + 256*((p-1)%elts)];
  byte = settab[ byte + 256*(e + elts*elt)];
  e++;
  if (e == elts)
    {
      *ptrD++ = byte;
      e = 0;
      byte = 0;
    }
      }
    if (e)
      *ptrD = byte;
    
    return res;
}



/****************************************************************************
**
*F  FuncELMS_VEC8BIT_RANGE( <self>, <list>, <range> ) .
**                                         select elms of an 8 bit vector
**
**  The results are returned in the compressed format
*/
libGAP_Obj libGAP_FuncELMS_VEC8BIT_RANGE (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 range  )
{
    libGAP_UInt                p;
    libGAP_Obj     info;
    libGAP_UInt                elts;
    libGAP_UInt                len;
    libGAP_UInt                lenl;
    libGAP_UInt                low;
    libGAP_Int                inc;
    libGAP_Obj                 res;
    libGAP_UInt                i;
    libGAP_UInt                elt;
    libGAP_UInt1               *gettab;
    libGAP_UInt1               *settab;
    libGAP_UInt1               *ptrS;
    libGAP_UInt1               *ptrD;
    libGAP_UInt                 e;
    libGAP_UInt1                byte;
    
    info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
    elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
    len = libGAP_GET_LEN_RANGE(range);
    low = libGAP_GET_LOW_RANGE(range);
    inc = libGAP_GET_INC_RANGE(range);
    lenl = libGAP_LEN_VEC8BIT(list);
    if (inc < 0)
      {
  if (low > lenl || low + inc*(len-1) < 1)
    libGAP_ErrorQuit("ELMS_VEC8BIT_RANGE: Range includes indices which are too high or too low",
        0L, 0L);
      }
    else
      if (low < 1 || low + inc*(len-1) > lenl)
  libGAP_ErrorQuit("ELMS_VEC8BIT_RANGE: Range includes indices which are too high or too low",
      0L, 0L);
    res = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT( len, elts));
    libGAP_TYPE_DATOBJ(res) = libGAP_TYPE_DATOBJ(list);
    libGAP_SET_FIELD_VEC8BIT(res, libGAP_FIELD_VEC8BIT(list));
    libGAP_SET_LEN_VEC8BIT(res, len);
    gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
    settab = libGAP_SETELT_FIELDINFO_8BIT(info);
    ptrS = libGAP_BYTES_VEC8BIT(list);
    ptrD = libGAP_BYTES_VEC8BIT(res);
    e = 0;
    byte = 0;
    p = low-1;      /* the -1 converts to 0 base */
    if (p % elts == 0 && inc == 1 && len >= elts)
      {
  while (p < low + len - elts)
    {
      *ptrD++ =ptrS[p/elts];
      p += elts;
    }
  byte = 0;
  e = 0;
  if ( p < low + len - 1)
    {
      while (p < low + len - 1)
        {
    elt = gettab[ptrS[p/elts] + 256*(p%elts)];
    byte = settab[ byte + 256 *(e + elts *elt)];
    e++;
    p++;
        }
      *ptrD = byte;
    }
      }
    else
      {
  for (i = 1; i <= len; i++)
    {
      elt = gettab[ptrS[p/elts] + 256*(p%elts)];
      byte = settab[ byte + 256*(e + elts*elt)];
      e++;
      if (e == elts)
        {
    *ptrD++ = byte;
    e = 0;
    byte = 0;
        }
      p += inc;
    }
  if (e)
    *ptrD = byte;
      }
    
    return res;
}




/****************************************************************************
**
*F  FuncASS_VEC8BIT( <self>, <list>, <pos>, <elm> ) set an elm of a GF2 vector
**
**  'ASS_VEC8BIT' assigns the element  <elm> at the position  <pos> to the GF2
**  vector <list>.
**
**  It is the responsibility of the caller  to ensure that <pos> is positive,
**  and that <elm> is not 0.
*/

static libGAP_Obj libGAP_AsInternalFFE;

libGAP_Obj libGAP_FuncASS_VEC8BIT (
		     libGAP_Obj                 self,
		     libGAP_Obj                 list,
		     libGAP_Obj                 pos,
		     libGAP_Obj                 elm )
{
  libGAP_UInt                p;
  libGAP_Obj                 info;
  libGAP_UInt                elts;
  libGAP_UInt                chr;
  libGAP_UInt                d;
  libGAP_UInt                q;
  libGAP_FF                  f;
  libGAP_UInt v;
  libGAP_Obj newelm;

  /* check that <list> is mutable                                        */
  if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
    libGAP_ErrorReturnVoid(
		    "Lists Assignment: <list> must be a mutable list",
		    0L, 0L,
		    "you can 'return;' and ignore the assignment" );
    return 0;
  }

  /* get the position                                                    */
  if (!libGAP_IS_INTOBJ(pos))
    libGAP_ErrorQuit("ASS_VEC8BIT: position should be a small integer, not a %s",
	      (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L);
  p = libGAP_INT_INTOBJ(pos);
  if (p <= 0)
    libGAP_ErrorQuit("ASS_VEC8BIT: position must be positive", 0L, 0L);
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  chr = libGAP_P_FIELDINFO_8BIT(info);
  d = libGAP_D_FIELDINFO_8BIT(info);
  q = libGAP_Q_FIELDINFO_8BIT(info);


  if ( p <= libGAP_LEN_VEC8BIT(list)+1 ) {
    if ( libGAP_LEN_VEC8BIT(list)+1 == p ) {
      if (libGAP_True == libGAP_DoFilter(libGAP_IsLockedRepresentationVector, list))
	{
	  libGAP_ErrorReturnVoid("List assignment would increase length of locked compressed vector",0,0,
			  "You can `return;' to ignore the assignment");
	  return 0;
	}
      libGAP_ResizeBag( list, libGAP_SIZE_VEC8BIT(p,elts));
      libGAP_SET_LEN_VEC8BIT( list, p);
      /*  Pr("Extending 8 bit vector by 1",0,0); */
    }
    if (!libGAP_IS_FFE(elm)) {
      newelm = libGAP_DoAttribute(libGAP_AsInternalFFE, elm);
      if (newelm != libGAP_Fail)
	elm = newelm;
    }
    if (libGAP_IS_FFE(elm) && chr == libGAP_CharFFE(elm))
      {

	/* We may need to rewrite the vector over a larger field */
	if (d % libGAP_DegreeFFE(elm) !=  0)
	  {
	    /*        Pr("Rewriting over larger field",0,0);*/
	    f = libGAP_CommonFF(libGAP_FiniteField(chr,d),d,
			 libGAP_FLD_FFE(elm),libGAP_DegreeFFE(elm));
	    if (f && libGAP_SIZE_FF(f) <= 256)
	      {
		libGAP_RewriteVec8Bit(list, libGAP_SIZE_FF(f));
		info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
		elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
		chr = libGAP_P_FIELDINFO_8BIT(info);
		d = libGAP_D_FIELDINFO_8BIT(info);
		q = libGAP_Q_FIELDINFO_8BIT(info);
	      }
	    else
	      {
		libGAP_PlainVec8Bit(list);
		libGAP_AssPlistFfe( list, p, elm );
		return 0;
	      }
	  }

   
	v = libGAP_VAL_FFE(elm);

	/* may need to promote the element to a bigger field
	   or restrict it to a smaller one */
	if (v != 0 && q != libGAP_SIZE_FF(libGAP_FLD_FFE(elm)))
	  {
	    assert (((v-1)*(q-1)) % (libGAP_SIZE_FF(libGAP_FLD_FFE(elm))-1) == 0);
	    v = 1+(v-1)*(q-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(elm))-1);
	  }

	/* finally do the assignment */
	libGAP_BYTES_VEC8BIT(list)[(p-1) / elts] =
	  libGAP_SETELT_FIELDINFO_8BIT(info)
	  [256*(elts*libGAP_FELT_FFE_FIELDINFO_8BIT(info)[v]+(p-1)%elts) +
	   libGAP_BYTES_VEC8BIT(list)[(p-1)/elts]];
	return 0;
      }
  }

  /* We fall through here if the assignment position is so large
     as to leave a hole, or if the object to be assigned is
     not of the right characteristic, or would create too large a field */

  /*     Pr("Random assignment (8 bit)",0,0);*/

  libGAP_PlainVec8Bit(list);
  libGAP_AssPlistFfe( list, p, elm );
  return 0;
}



/****************************************************************************
**
*F  FuncUNB_VEC8BIT( <self>, <list>, <pos> ) . unbind position of a GFQ vector
**
**  'UNB_VEC8BIT' unbind  the element at  the position  <pos> in  a GFQ vector
**  <list>.
**
**  It is the responsibility of the caller  to ensure that <pos> is positive.
*/
libGAP_Obj libGAP_FuncUNB_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UInt                p;
    libGAP_Obj                 info;
    libGAP_UInt elts;

    /* check that <list> is mutable                                        */
    if ( ! libGAP_IS_MUTABLE_OBJ(list) ) {
        libGAP_ErrorReturnVoid(
            "Lists Assignment: <list> must be a mutable list",
            0L, 0L,
            "you can 'return;' and ignore the assignment" );
        return 0;
    }
    if (libGAP_True == libGAP_DoFilter(libGAP_IsLockedRepresentationVector, list))
      {
  libGAP_ErrorReturnVoid("Unbind of entry of locked compressed vector is forbidden",0,0,
      "You can `return;' to ignore the assignment");
  return 0;
      }

    /* get the position                                                    */
    if (!libGAP_IS_INTOBJ(pos))
      libGAP_ErrorQuit("UNB_VEC8BIT: position should be a small integer, not a %s",
    (libGAP_Int)libGAP_TNAM_OBJ(pos), 0L);
    p = libGAP_INT_INTOBJ(pos);
    if (p <= 0)
      libGAP_ErrorQuit("UNB_VEC9BIT: position must be positive", 0L, 0L);

    /* if we unbind the last position keep the representation              */
    if ( libGAP_LEN_VEC8BIT(list) < p ) {
        ;
    }
    else if ( /* p > 1 && */ libGAP_LEN_VEC8BIT(list) == p ) {
      /* zero out the last entry first, for safety */
      info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
      elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
      libGAP_BYTES_VEC8BIT(list)[(p-1)/elts] =
  libGAP_SETELT_FIELDINFO_8BIT(info)[((p-1) % elts)*256 +
           libGAP_BYTES_VEC8BIT(list)[(p-1)/elts]];
        libGAP_ResizeBag( list, 3*sizeof(libGAP_UInt)+(p+elts-2)/elts );
        libGAP_SET_LEN_VEC8BIT( list, p-1);
    }
    else {
        libGAP_PlainVec8Bit(list);
        libGAP_UNB_LIST( list, p );
    }
    return 0;
}

/****************************************************************************
**
*F  FuncPOSITION_NONZERO_VEC8BIT( <self>, <list>, <zero> ) .
**                               
**  The pointless zero argument is because this is a method for PositionNot
**  It is *not* used in the code and can be replaced by a dummy argument.
**
*/

libGAP_UInt libGAP_PositionNonZeroVec8Bit ( libGAP_Obj list, libGAP_UInt from )
{
    libGAP_Obj                 info;
    libGAP_UInt len;
    libGAP_UInt nb;
    libGAP_UInt i,j;
    libGAP_UInt elts;
    libGAP_UInt1 *ptr;
    libGAP_UInt1 byte;
    libGAP_UInt1 *gettab;

    len = libGAP_LEN_VEC8BIT(list);
    info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(list));
    elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
    gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
    nb = (len + elts -1)/elts;
    ptr = libGAP_BYTES_VEC8BIT(list);
    i = from/elts;
    j = from %elts;
    /* might be an initial part byte */
    if (j) {
      if (i < nb && ptr[i])
  for (j = from % elts; j < elts && (i*elts+j < len); j++)
    if (gettab[256*j + ptr[i]] != 0)
      return elts*i+j+1;
      i++;
    }

    /* skip empty bytes */
    while (i < nb && !ptr[i])
      i++;
    
    if (i >= nb)
      return len+1;

    /* Found a non-empty byte, locate the entry */
    byte = ptr[i];
    j = 0;
    while (gettab[byte + 256*j] == 0)
      j++;
    return elts*i+j+1;
}

          

libGAP_Obj libGAP_FuncPOSITION_NONZERO_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 zero )
{
  return libGAP_INTOBJ_INT(libGAP_PositionNonZeroVec8Bit(list, 0));
}

libGAP_Obj libGAP_FuncPOSITION_NONZERO_VEC8BIT3 (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 zero,
    libGAP_Obj                 from)
{
  return libGAP_INTOBJ_INT(libGAP_PositionNonZeroVec8Bit(list, libGAP_INT_INTOBJ(from)));
}

/****************************************************************************
**
*F  FuncAPPEND_VEC8BIT( <self>, <vecl>, <vecr> ) .
**                               
**
*/
libGAP_Obj libGAP_FuncAPPEND_VEC8BIT (
    libGAP_Obj                 self,
    libGAP_Obj                 vecl,
    libGAP_Obj                 vecr )
{
    libGAP_Obj                 info;
    libGAP_UInt lenl,lenr;
    libGAP_UInt nb;
    libGAP_UInt i;
    libGAP_UInt elts;
    libGAP_UInt1 *ptrl,*ptrr;
    libGAP_UInt1 bytel, byter, elt;
    libGAP_UInt1 *gettab, *settab;
    libGAP_UInt posl, posr;

    if (libGAP_FIELD_VEC8BIT(vecl) != libGAP_FIELD_VEC8BIT(vecr))
      return libGAP_TRY_NEXT_METHOD;
    
    lenl = libGAP_LEN_VEC8BIT(vecl);
    lenr = libGAP_LEN_VEC8BIT(vecr);
    if (libGAP_True == libGAP_DoFilter(libGAP_IsLockedRepresentationVector, vecl) && lenr > 0)
      {
  libGAP_ErrorReturnVoid("Append to locked compressed vector is forbidden", 0, 0,
      "You can `return;' to ignore the operation");
  return 0;
      }
    info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(vecl));
    elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
    libGAP_ResizeBag( vecl, libGAP_SIZE_VEC8BIT(lenl+lenr,elts));

    if (lenl % elts == 0)
      {
  ptrl = libGAP_BYTES_VEC8BIT(vecl) + lenl/elts;
  ptrr = libGAP_BYTES_VEC8BIT(vecr);
  nb = (lenr + elts - 1)/elts;
  for (i = 0; i < nb; i++)
    *ptrl++ = *ptrr++;
      }
    else
      {
  ptrl = libGAP_BYTES_VEC8BIT(vecl) + (lenl -1)/elts;
  bytel = *ptrl;
  posl = lenl;
  posr = 0;
  ptrr = libGAP_BYTES_VEC8BIT(vecr);
  byter = *ptrr;
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  settab = libGAP_SETELT_FIELDINFO_8BIT(info);
  while (posr < lenr)
    {
      elt = gettab[ byter + 256 * (posr % elts) ];
      bytel = settab[ bytel + 256 * ( posl % elts + elts * elt)];
      if (++posl % elts == 0)
        {
    *ptrl++ = bytel; 
    bytel = 0;
        }
      if ( ++posr % elts == 0)
        {
    byter = *++ptrr;
        }
    }
  /* Write last byte only if not already written: */
  if (posl % elts != 0) *ptrl = bytel;
      }
    libGAP_SET_LEN_VEC8BIT(vecl, lenl + lenr);
    return (libGAP_Obj) 0;
}


/****************************************************************************
**
*F  FuncPROD_VEC8BIT_MATRIX( <self>, <vec>, <mat> )
**
**  Method selection should ensure that <mat> is a matrix of ffes in the
**  right characteristic. We aim to be fast in the case where it is
**  actually a plain list of vec8bits over the same field as <vec>. We also
**  know that <vec> and <mat> are non-empty
** */

libGAP_Obj libGAP_FuncPROD_VEC8BIT_MATRIX( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj mat)
{
  libGAP_Obj res;
  libGAP_Obj info;
  libGAP_UInt q;
  libGAP_UInt len,l2;
  libGAP_UInt len1;
  libGAP_Obj row1;
  libGAP_UInt i;
  libGAP_UInt elts;
  libGAP_UInt1* gettab;
  libGAP_Obj *ffefelt;
  libGAP_Obj x;
  
  len = libGAP_LEN_VEC8BIT(vec);
  l2 = libGAP_LEN_PLIST(mat);
  q = libGAP_FIELD_VEC8BIT(vec);

  /* Get the first row, to establish the size of the result */
  row1 = libGAP_ELM_PLIST(mat,1);  
  if (! libGAP_IS_VEC8BIT_REP(row1) || libGAP_FIELD_VEC8BIT(row1) != q)
        return libGAP_TRY_NEXT_METHOD;
  len1 = libGAP_LEN_VEC8BIT(row1);

  /* create the result space */
  res = libGAP_ZeroVec8Bit(q, len1, libGAP_IS_MUTABLE_OBJ(vec) || libGAP_IS_MUTABLE_OBJ(row1));

  /* Finally, we start work */
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  ffefelt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);
  
  for (i = 0; i < len; i++)
    if (i < l2)
      {
  x = ffefelt[gettab[libGAP_BYTES_VEC8BIT(vec)[i/elts] + 256*(i % elts)]];
  if (libGAP_VAL_FFE(x) != 0)
    {
      row1 = libGAP_ELM_PLIST(mat,i+1);  
      /* This may be unduly draconian. Later we may want to be able to promote the rows
         to a bigger field */
      if ((! libGAP_IS_VEC8BIT_REP(row1)) || (libGAP_FIELD_VEC8BIT(row1) != q))
        return libGAP_TRY_NEXT_METHOD;
      libGAP_AddVec8BitVec8BitMultInner( res, res, row1, x, 1, len1);
    }
      }
  return res;
  
}


/****************************************************************************
**
*F  * * * * * * *  special rep for matrices over thses fields * * * * * * *
*/

#define libGAP_LEN_MAT8BIT(mat)                   libGAP_INT_INTOBJ(libGAP_ADDR_OBJ(mat)[1])
#define libGAP_SET_LEN_MAT8BIT(mat, l)            (libGAP_ADDR_OBJ(mat)[1] = libGAP_INTOBJ_INT(l))
#define libGAP_ELM_MAT8BIT(mat, i)                libGAP_ADDR_OBJ(mat)[i+1]
#define libGAP_SET_ELM_MAT8BIT(mat, i, row)       (libGAP_ADDR_OBJ(mat)[i+1] = row)

/****************************************************************************
**
*F  PlainMat8Bit( <mat> )
**
*/
void libGAP_PlainMat8Bit(  libGAP_Obj mat)
{
  libGAP_UInt i, l;
  libGAP_Obj row;
  l  = libGAP_LEN_MAT8BIT(mat);
  libGAP_RetypeBag(mat, libGAP_IS_MUTABLE_OBJ(mat) ? libGAP_T_PLIST_TAB : libGAP_T_PLIST_TAB + libGAP_IMMUTABLE);
  libGAP_SET_LEN_PLIST(mat, l);
  for (i = 1; i <= l; i++)
    {
      row = libGAP_ELM_MAT8BIT(mat, i);
      libGAP_SET_ELM_PLIST(mat, i, row);
    }
  libGAP_SET_ELM_PLIST(mat, l+1, 0);
}

/****************************************************************************
**
*F  FuncPLAIN_MAT8BIT( <self>, <mat> )
**
*/

libGAP_Obj libGAP_FuncPLAIN_MAT8BIT( libGAP_Obj self, libGAP_Obj mat)
{
  libGAP_PlainMat8Bit(mat);
  return 0;
}


/****************************************************************************
**
*F  FuncCONV_MAT8BT( <self>, <list> , <q> )
**
**  The library should have taken care of <list> containing only locked
** 8 bit vectors, written over the correct field
*/

libGAP_Obj libGAP_FuncCONV_MAT8BIT( libGAP_Obj self, libGAP_Obj list, libGAP_Obj q )
{
  libGAP_UInt len,i, mut;
  libGAP_Obj tmp;
  libGAP_Obj type;

  if (!libGAP_IS_INTOBJ(q))
    libGAP_ErrorQuit("CONV_MAT8BIT: q must be a small integer, not a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(q), (libGAP_Int)0L);
  libGAP_PLAIN_LIST(list);
  len = libGAP_LEN_PLIST(list);
  mut = libGAP_IS_MUTABLE_OBJ(list);
  libGAP_GROW_PLIST(list, len+1);
  for (i = len; i >= 1; i--)
    {
      tmp = libGAP_ELM_PLIST(list, i);
    type = libGAP_TypeVec8BitLocked( libGAP_INT_INTOBJ(q), libGAP_IS_MUTABLE_OBJ(tmp));
      libGAP_TYPE_DATOBJ(tmp) = type;
      libGAP_SET_ELM_MAT8BIT( list, i, tmp);
      libGAP_CHANGED_BAG(list);
    }
 libGAP_SET_LEN_MAT8BIT(list, len);
 libGAP_RetypeBag(list, libGAP_T_POSOBJ);
 type = libGAP_TypeMat8Bit(libGAP_INT_INTOBJ(q), mut); 
 libGAP_TYPE_POSOBJ(list) = type;
 return 0;
}



/****************************************************************************
**
*F ProdVec8BitMat8Bit( <vec>, <mat> )
**
** The caller must ensure that <vec> and <mat> are compatible
*/

libGAP_Obj libGAP_ProdVec8BitMat8Bit( libGAP_Obj vec, libGAP_Obj mat )
{
  libGAP_UInt q, len, len1, lenm, elts;
  libGAP_UInt i,j;
  libGAP_UInt1 byte;
  libGAP_UInt1 *bptr;
  libGAP_UInt1 y;
  libGAP_Obj row1;
  libGAP_Obj res;
  libGAP_Obj info;
  libGAP_UInt1 * gettab;
  libGAP_Obj *ffefelt;
  libGAP_Obj x;
  
  q = libGAP_FIELD_VEC8BIT(vec);
  len = libGAP_LEN_VEC8BIT(vec);
  lenm = libGAP_LEN_MAT8BIT(mat);
  row1 = libGAP_ELM_MAT8BIT(mat,1);
  assert( q == libGAP_FIELD_VEC8BIT(row1));
  len1 = libGAP_LEN_VEC8BIT(row1);
  res = libGAP_ZeroVec8Bit(q, len1, libGAP_IS_MUTABLE_OBJ(vec) || libGAP_IS_MUTABLE_OBJ(row1));

  /* Finally, we start work */
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  ffefelt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);

  bptr = libGAP_BYTES_VEC8BIT(vec);
  for (i = 0; i +elts < len; i += elts, bptr++)
    {
      if ((byte = *bptr)) {
  for (j = 0; j < elts ; j++)
    {
      if (i + j < lenm)
        {
    y = gettab[byte + 256*j];
    if (y)
      {
        x = ffefelt[y];
        row1 = libGAP_ELM_MAT8BIT(mat,i+j+1);  
        libGAP_AddVec8BitVec8BitMultInner( res, res, row1, x, 1, len1);
      }
        }
    }
      }
    }
  if ((byte = *bptr)) {
    for (j = 0; i+j < len  ; j++)
      {
  if (i + j < lenm)
    {
      y = gettab[byte + 256*j];
      if (y)
        {
    x = ffefelt[y];
    row1 = libGAP_ELM_MAT8BIT(mat,i+j+1);  
    libGAP_AddVec8BitVec8BitMultInner( res, res, row1, x, 1, len1);
        }
    }
      }
  }
  return res;
 }

/****************************************************************************
**
*F  FuncPROD_VEC8BIT_MAT8BIT( <self>, <vec>, <mat> )
**
**  The caller should ensure that characteristics are right and that the
**  arguments ARE an 8 bit vector and matrix. We still have to check actual
**  fields and lengths.
*/

libGAP_Obj libGAP_FuncPROD_VEC8BIT_MAT8BIT( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj mat)
{
  libGAP_UInt q, q1, q2;

  /* Sort out length mismatches */
    
  /* Now field mismatches -- consider promoting the vector */
  q = libGAP_FIELD_VEC8BIT(vec);
  q1 = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(mat,1));
  if (q != q1)
    {
      if (q > q1 || libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vec) == libGAP_True)
  return libGAP_TRY_NEXT_METHOD;
      q2 = q;
      while (q2 < q1)
  {
    q2 *= q;
  }
      if (q2 == q1)
  libGAP_RewriteVec8Bit(vec, q1);
      else
  return libGAP_TRY_NEXT_METHOD;
    }

  /* OK, now we can do the work */
  return libGAP_ProdVec8BitMat8Bit(vec,mat);
}

/****************************************************************************
**
*F ProdMat8BitVec8Bit( <mat>, <vec> )
**
** The caller must ensure compatibility
*/

libGAP_Obj libGAP_ProdMat8BitVec8Bit( libGAP_Obj mat, libGAP_Obj vec)
{
  libGAP_UInt len, i, q;
  libGAP_Obj info;
  libGAP_UInt1 *settab;
  libGAP_Obj res;
  libGAP_Obj row1;
  libGAP_UInt1 byte;
  libGAP_UInt elts;
  libGAP_UInt1 *feltffe;
  libGAP_UInt1* ptr;
  libGAP_Obj entry;
  len = libGAP_LEN_MAT8BIT(mat);
  q = libGAP_FIELD_VEC8BIT(vec);
  row1 = libGAP_ELM_MAT8BIT(mat,1);
  assert(q = libGAP_FIELD_VEC8BIT(row1));
  res = libGAP_ZeroVec8Bit( q, len, libGAP_IS_MUTABLE_OBJ(row1) || libGAP_IS_MUTABLE_OBJ(vec));
  info = libGAP_GetFieldInfo8Bit(q);
  settab = libGAP_SETELT_FIELDINFO_8BIT(info);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  feltffe = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
  byte = 0;
  ptr = libGAP_BYTES_VEC8BIT(res);
  for (i = 0; i < len; i++)
    {
      entry = libGAP_ScalarProductVec8Bits( vec, libGAP_ELM_MAT8BIT(mat, i+1));
      byte = settab[ byte + 256 * ( elts * feltffe[libGAP_VAL_FFE(entry)] +  i % elts )];
      if (i % elts == elts-1)
  {
    *ptr++ = byte;
    byte = 0;
  }
    }
    if (len % elts != 0)
      *ptr++ = byte;
    return res;
}

/****************************************************************************
**
*F  FuncPROD_MAT8BIT_VEC8BIT( <self>, <mat>, <vec> )
**
**  The caller should ensure that characteristics are right and that the
**  arguments ARE an 8 bit vector and matrix. We still have to check actual
**  fields and lengths.
*/

libGAP_Obj libGAP_FuncPROD_MAT8BIT_VEC8BIT( libGAP_Obj self, libGAP_Obj mat, libGAP_Obj vec)
{
  libGAP_UInt q, q1, q2;
  libGAP_Obj row;

  /* Sort out length mismatches */

  row = libGAP_ELM_MAT8BIT(mat, 1);
    

  /* Now field mismatches -- consider promoting the vector */
  q = libGAP_FIELD_VEC8BIT(vec);
  q1 = libGAP_FIELD_VEC8BIT(row);
  if (q != q1)
    {
      if (q > q1 || libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vec) == libGAP_True)
  return libGAP_TRY_NEXT_METHOD;
      q2 = q;
      while (q2 < q1)
  {
    q2 *= q;
  }
      if (q2 == q1)
  libGAP_RewriteVec8Bit(vec, q1);
      else
  return libGAP_TRY_NEXT_METHOD;
    }

  /* OK, now we can do the work */
  return libGAP_ProdMat8BitVec8Bit(mat,vec);
}


/****************************************************************************
**
*F  ProdMat8BitMat8Bit( <matl>, <matr> )
**
**  Caller must check matriox sizes and field
*/

libGAP_Obj libGAP_ProdMat8BitMat8Bit( libGAP_Obj matl, libGAP_Obj matr)
{
  libGAP_Obj prod;
  libGAP_UInt i;
  libGAP_UInt len,q;
  libGAP_Obj row;
  libGAP_Obj locked_type;
  libGAP_Obj type;

  len = libGAP_LEN_MAT8BIT(matl);
  q = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(matl,1));

  assert(q == libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(matr,1)));
  assert(libGAP_LEN_MAT8BIT(matr) == libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(matl,1)));
  
  prod = libGAP_NewBag(libGAP_T_POSOBJ, sizeof(libGAP_Obj)*(len+2));
  libGAP_SET_LEN_MAT8BIT(prod,len);
  type = libGAP_TypeMat8Bit(q, libGAP_IS_MUTABLE_OBJ(matl) || libGAP_IS_MUTABLE_OBJ(matr));
  libGAP_TYPE_POSOBJ(prod) = type;
  locked_type  = libGAP_TypeVec8BitLocked(q, libGAP_IS_MUTABLE_OBJ(libGAP_ELM_MAT8BIT(matl,1)) || libGAP_IS_MUTABLE_OBJ(libGAP_ELM_MAT8BIT(matr,1)));
  for (i = 1; i <= len; i++)
    {
      row = libGAP_ProdVec8BitMat8Bit(libGAP_ELM_MAT8BIT(matl,i),matr);

      /* Since I'm going to put this vector into a matrix, I must lock its
   representation, so that it doesn't get rewritten over GF(q^k) */
      libGAP_TYPE_DATOBJ(row) = locked_type;
      libGAP_SET_ELM_MAT8BIT(prod,i,row);
      libGAP_CHANGED_BAG(prod);
      libGAP_TakeInterrupt();
    }
  return prod;

}

/****************************************************************************
**
*F  FuncPROD_MAT8BIT_MAT8BIT( <self>, <matl>, <matr> )
**
*/

libGAP_Obj libGAP_FuncPROD_MAT8BIT_MAT8BIT( libGAP_Obj self, libGAP_Obj matl, libGAP_Obj matr)
{
  libGAP_UInt ql, qr;
  libGAP_Obj rowl;

  rowl = libGAP_ELM_MAT8BIT(matl,1);
  ql = libGAP_FIELD_VEC8BIT(rowl);
  qr = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(matr,1));

  if (ql != qr)
    return libGAP_TRY_NEXT_METHOD;

  return libGAP_ProdMat8BitMat8Bit(matl, matr);
}


/****************************************************************************
**
*F  InverseMat8Bit( <mat> )
**
*/

libGAP_Obj libGAP_InverseMat8Bit( libGAP_Obj mat, libGAP_UInt mut)
{
  libGAP_Obj cmat, inv;
  libGAP_UInt len, off;
  libGAP_UInt i,j, k;
  libGAP_Obj zero;
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_UInt1 *ptr;
  libGAP_UInt elts;
  libGAP_UInt1 *settab, *gettab;
  libGAP_UInt1 byte;
  libGAP_Obj row, row1, row2;
  libGAP_Obj *ffefelt;
  libGAP_UInt1 *feltffe;
  libGAP_UInt pos;
  libGAP_UInt1 x = 0;
  libGAP_UInt o;
  libGAP_Obj xi;
  libGAP_Obj xn;
  libGAP_Obj type;

  row = libGAP_ELM_MAT8BIT(mat, 1);
  q = libGAP_FIELD_VEC8BIT(row);
  len = libGAP_LEN_MAT8BIT(mat);
  assert(len == libGAP_LEN_VEC8BIT(row));
  inv = libGAP_NEW_PLIST(libGAP_T_PLIST, len+1);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);

  if (len == 1)
    {
      gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
      ffefelt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);
      x = gettab[libGAP_BYTES_VEC8BIT(row)[0]];
      if ( x == 0 )
  return libGAP_Fail;
      xi = libGAP_INV(ffefelt[x]);
      row1 = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT(1, elts));
    type = libGAP_TypeVec8BitLocked(q, mut == 2 || (mut == 1 && libGAP_IS_MUTABLE_OBJ(row)));
      libGAP_TYPE_DATOBJ(row1) = type;
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      feltffe = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
      libGAP_BYTES_VEC8BIT(row1)[0] = settab[256*elts*feltffe[libGAP_VAL_FFE(xi)]];
      libGAP_SET_LEN_VEC8BIT(row1,1);
      libGAP_SET_FIELD_VEC8BIT(row1,q);
      libGAP_SET_ELM_PLIST(inv, 1, libGAP_INTOBJ_INT(1));
      libGAP_SET_ELM_PLIST(inv, 2, row1);
      libGAP_CHANGED_BAG(inv);
      libGAP_RetypeBag(inv, libGAP_T_POSOBJ);
    type = libGAP_TypeMat8Bit(q, mut == 2 || (mut == 1 && libGAP_IS_MUTABLE_OBJ(mat)));
      libGAP_TYPE_POSOBJ(inv) = type;
      libGAP_SET_LEN_MAT8BIT(inv,1);
      return inv;
    }
  
  /* set up cmat and inv. Note that the row numbering is offset */
  cmat = libGAP_NEW_PLIST(libGAP_T_PLIST, len); 
  zero = libGAP_ZeroVec8Bit(q, len, 1);
  o = libGAP_FELT_FFE_FIELDINFO_8BIT(info)[1];
  for (i = 1; i <= len; i++)
    {
      row = libGAP_ELM_MAT8BIT(mat, i);
      row = libGAP_SHALLOW_COPY_OBJ(row);
      libGAP_SET_ELM_PLIST( cmat, i, row);
      libGAP_CHANGED_BAG(cmat);
      row = libGAP_SHALLOW_COPY_OBJ(zero);
      ptr = libGAP_BYTES_VEC8BIT(row) + (i-1)/elts;

      /* we can't retain this pointer, because of garbage collections */
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      /* we know we are replacing a zero  */
      *ptr = settab[256*((i-1)%elts + o*elts)];
      libGAP_SET_ELM_PLIST( inv, i+1, row);
      libGAP_CHANGED_BAG(inv);
    }
  
  /* Now do Gaussian elimination in cmat and mirror it on inv
     from here, no garbage collections are allowed until the end */
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  ffefelt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);

  for (i = 1; i <= len; i++)
    {
      off = (i-1)/elts;
      pos = (i-1) % elts;
      /* find a non-zero entry in column i */
      for (j = i; j <= len; j++)
  {
    row = libGAP_ELM_PLIST(cmat, j);
    byte = libGAP_BYTES_VEC8BIT(row)[off];
    if (byte != 0 && (x = gettab[byte + 256 * pos]) != 0)
      break;
  }

      /* if we didn't find one */
      if ( j > len)
  return libGAP_Fail;
      
      /* swap and normalize */
      row1 = libGAP_ELM_PLIST(inv, j+1);
      if (i != j)
  {
    libGAP_SET_ELM_PLIST(cmat, j, libGAP_ELM_PLIST(cmat, i));
    libGAP_SET_ELM_PLIST(cmat, i, row );
    libGAP_SET_ELM_PLIST(inv, j+1, libGAP_ELM_PLIST(inv, i+1));
    libGAP_SET_ELM_PLIST(inv,i+1, row1);
  }
      if (x != o)
  {
    xi = libGAP_INV(ffefelt[x]);
    libGAP_MultVec8BitFFEInner( row, row, xi, i, len);
    libGAP_MultVec8BitFFEInner( row1, row1, xi, 1, len);
  }

      /* Now clean out column */
      for (k = 1; k <= len; k++)
  {
    if (k < i || k > j)
      {
        row2 = libGAP_ELM_PLIST(cmat, k);
        byte = libGAP_BYTES_VEC8BIT(row2)[off];
        if (byte != 0 && (x = gettab[byte + 256 * pos]) != 0)
    {
      xn = libGAP_AINV(ffefelt[x]);
      libGAP_AddVec8BitVec8BitMultInner( row2, row2, row, xn, i, len);
      row2 = libGAP_ELM_PLIST(inv, k+1);
      libGAP_AddVec8BitVec8BitMultInner( row2, row2, row1, xn, 1, len);
    }
      }
  }
      if (libGAP_TakeInterrupt()) {
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  ffefelt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);
      }
    }

  /* Now clean up inv and return it */
  libGAP_SET_ELM_PLIST(inv,1,libGAP_INTOBJ_INT(len));
  type = libGAP_TypeVec8BitLocked(q, mut == 2 || (mut == 1 && libGAP_IS_MUTABLE_OBJ(libGAP_ELM_MAT8BIT(mat,1))));
  for (i =2 ; i <= len+1; i++)
    {
      row = libGAP_ELM_PLIST(inv, i);
      libGAP_TYPE_DATOBJ(row) = type;
    }
  libGAP_RetypeBag(inv, libGAP_T_POSOBJ);
  type = libGAP_TypeMat8Bit(q, mut == 2 || (mut == 1 && libGAP_IS_MUTABLE_OBJ(mat)));
  libGAP_TYPE_POSOBJ(inv) = type;
  libGAP_CHANGED_BAG(inv);
  return inv;
}

/****************************************************************************
**
*F FuncINV_MAT8BIT_MUTABLE( <self>, <mat> )
**
*/

libGAP_Obj libGAP_FuncINV_MAT8BIT_MUTABLE( libGAP_Obj self, libGAP_Obj mat)
{
  if (libGAP_LEN_MAT8BIT(mat) != libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(mat, 1)))
    {
      mat = libGAP_ErrorReturnObj("InverseOp: matrix must be square, not %d by %d",
         libGAP_LEN_MAT8BIT(mat),
         libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(mat, 1)),
         "you can replace matrix <inv> via 'return <inv>;'");
      return libGAP_INV(mat);
    }
  
  return libGAP_InverseMat8Bit(mat, 2);
}

/****************************************************************************
**
*F FuncINV_MAT8BIT_SAME_MUTABILITY( <self>, <mat> )
**
*/

libGAP_Obj libGAP_FuncINV_MAT8BIT_SAME_MUTABILITY( libGAP_Obj self, libGAP_Obj mat)
{
  if (libGAP_LEN_MAT8BIT(mat) != libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(mat, 1)))
    {
      mat = libGAP_ErrorReturnObj("INVOp: matrix must be square, not %d by %d",
         libGAP_LEN_MAT8BIT(mat),
         libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(mat, 1)),
         "you can replace matrix <inv> via 'return <inv>;'");
      return libGAP_INV_MUT(mat);
    }
  
  return libGAP_InverseMat8Bit(mat, 1);
}

/****************************************************************************
**
*F FuncINV_MAT8BIT_IMMUTABLE( <self>, <mat> )
**
*/

libGAP_Obj libGAP_FuncINV_MAT8BIT_IMMUTABLE( libGAP_Obj self, libGAP_Obj mat)
{
  if (libGAP_LEN_MAT8BIT(mat) != libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(mat, 1)))
    {
      libGAP_Obj inv;
      mat = libGAP_ErrorReturnObj("Inverse: matrix must be square, not %d by %d",
         libGAP_LEN_MAT8BIT(mat),
         libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(mat, 1)),
         "you can replace matrix <inv> via 'return <inv>;'");
      inv = libGAP_INV_MUT(mat);
      libGAP_MakeImmutable(inv);
      return inv;
    }
  
  return libGAP_InverseMat8Bit(mat, 0);
}

/****************************************************************************
**
*F  FuncASS_MAT8BIT( <self>, <mat>, <pos>, <obj> )
**
*/

libGAP_Obj libGAP_FuncASS_MAT8BIT(libGAP_Obj self, libGAP_Obj mat, libGAP_Obj p, libGAP_Obj obj)
{
  libGAP_UInt len;
  libGAP_UInt len1;
  libGAP_UInt len2;
  libGAP_UInt q;
  libGAP_UInt q1, q2;
  libGAP_Obj row;
  libGAP_UInt pos;
  libGAP_Obj type;

  if (!libGAP_IS_INTOBJ(p))
    libGAP_ErrorQuit("ASS_MAT8BIT: position should be a small integer, not a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(p), 0L);
  pos = libGAP_INT_INTOBJ(p);
    if (pos <= 0)
      libGAP_ErrorQuit("ASS_MAT8BIT: position must be positive", 0L, 0L);

  len = libGAP_LEN_MAT8BIT(mat);
  if (!libGAP_IS_VEC8BIT_REP(obj) && !libGAP_IS_GF2VEC_REP(obj))
    goto cantdo;

  if (pos > len + 1)
    goto cantdo;
  
  if (len == 1 && pos == 1)
    {
      if (libGAP_IS_VEC8BIT_REP(obj))
  {
    q = libGAP_FIELD_VEC8BIT(obj);
    goto cando;
  }
      else
  {
    libGAP_TYPE_POSOBJ(mat) = libGAP_IS_MUTABLE_OBJ(mat) ? libGAP_TYPE_LIST_GF2MAT : libGAP_TYPE_LIST_GF2MAT_IMM;
    libGAP_TYPE_DATOBJ(obj) = libGAP_IS_MUTABLE_OBJ(obj) ? libGAP_TYPE_LIST_GF2VEC_LOCKED : libGAP_TYPE_LIST_GF2VEC_IMM_LOCKED;
    libGAP_SET_ELM_GF2MAT(mat, 1, obj);
    return (libGAP_Obj) 0;
  }
    }
  
  row = libGAP_ELM_MAT8BIT(mat,1);
  len1 = libGAP_LEN_VEC8BIT(row);

  if (libGAP_IS_VEC8BIT_REP(obj))
    len2 = libGAP_LEN_VEC8BIT(obj);
  else
    len2 = libGAP_LEN_GF2VEC(obj);
  
  if (len2 != len1)
    goto cantdo;

  q = libGAP_FIELD_VEC8BIT(row);
  if ( libGAP_IS_GF2VEC_REP(obj)) {
    if (q % 2 != 0 || libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, obj) == libGAP_True) 
      goto cantdo;
    else
      {
  libGAP_RewriteGF2Vec(obj, q);
  goto cando;
      }
  }
  
  q1 = libGAP_FIELD_VEC8BIT(obj);

  if (q1 == q)
    goto cando;

  if (q1 > q || libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, obj) == libGAP_True )
    goto cantdo;
  
  q2 = q1*q1;
  while (q2 <= 256)
    {
      if (q2 == q)
  {
    libGAP_RewriteVec8Bit(obj, q);
    goto cando;
  }
      q2 *= q1;
    }
  goto cantdo;
  
 cando:
  if (pos > len)
    {
      libGAP_ResizeBag(mat, sizeof(libGAP_Obj)*(pos+2));
      libGAP_SET_LEN_MAT8BIT(mat, pos);
    }
  type = libGAP_TypeVec8BitLocked(q, libGAP_IS_MUTABLE_OBJ(obj));
  libGAP_TYPE_DATOBJ(obj) = type;
  libGAP_SET_ELM_MAT8BIT(mat, pos, obj);
  libGAP_CHANGED_BAG(mat);
  return (libGAP_Obj) 0;
  
 cantdo:
  libGAP_PlainMat8Bit(mat);
  libGAP_ASS_LIST(mat, pos, obj);
  libGAP_CHANGED_BAG(mat);
  
  return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  SumMat8BitMat8Bit( <ml> ,<mr>)
**
**  Caller's job to do all checks
*/

libGAP_Obj libGAP_SumMat8BitMat8Bit( libGAP_Obj ml, libGAP_Obj mr)
{
  libGAP_Obj sum;
  libGAP_UInt ll,lr,wl,wr,ls;
  libGAP_UInt q;
  libGAP_UInt i;
  libGAP_Obj  row;
  libGAP_Obj type;
  ll = libGAP_LEN_MAT8BIT(ml);
  lr = libGAP_LEN_MAT8BIT(mr);
  wl = libGAP_LEN_MAT8BIT(libGAP_ELM_MAT8BIT(ml,1));
  wr = libGAP_LEN_MAT8BIT(libGAP_ELM_MAT8BIT(mr,1));

  /* We have to track the cases where the result is not rectangular */
  if (((ll > lr) && (wr > wl)) ||
      ((lr > ll) && (wl > wr)))
    return libGAP_TRY_NEXT_METHOD;

  /* Now sort out the size of the result */
  if (ll > lr)
    {
      ls = ll;
      assert(wl > wr);
    }
  else
    {
      ls = lr;
      assert(wr >= wl);
    }
      
  q = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(ml,1));
  sum = libGAP_NewBag(libGAP_T_POSOBJ, sizeof(libGAP_Obj)*(ls+2));
  type = libGAP_TypeMat8Bit( q, libGAP_IS_MUTABLE_OBJ(ml) || libGAP_IS_MUTABLE_OBJ(mr));
  libGAP_TYPE_POSOBJ(sum) = type;
  libGAP_SET_LEN_MAT8BIT(sum, ls);
  
  type = libGAP_TypeVec8BitLocked(q, libGAP_IS_MUTABLE_OBJ(libGAP_ELM_MAT8BIT(ml,1)) || libGAP_IS_MUTABLE_OBJ(libGAP_ELM_MAT8BIT(mr,1)));
  
  for (i = 1; i <= ls; i++)
    {
      if (i > ll)
  row = libGAP_CopyVec8Bit(libGAP_ELM_MAT8BIT(mr,i),1);
      else if (i > lr)
  row = libGAP_CopyVec8Bit(libGAP_ELM_MAT8BIT(ml,i),1);
      else
  row = libGAP_SumVec8BitVec8Bit(libGAP_ELM_MAT8BIT(ml,i), libGAP_ELM_MAT8BIT(mr,i));
      
      libGAP_TYPE_DATOBJ(row) = type;
      libGAP_SET_ELM_MAT8BIT(sum, i, row);
      libGAP_CHANGED_BAG(sum);
    }
  return sum;
}

/****************************************************************************
**
*F  FuncSUM_MAT8BIT_MAT8BIT( <self>, <ml>, <mr> )
**
**  Caller should check that both args are mat8bit over the same field
*/

libGAP_Obj libGAP_FuncSUM_MAT8BIT_MAT8BIT( libGAP_Obj self, libGAP_Obj ml, libGAP_Obj mr)
{
  libGAP_UInt q;
  q = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(ml,1));
  if (q != libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(mr,1)))
    return libGAP_TRY_NEXT_METHOD;
  else
    return libGAP_SumMat8BitMat8Bit( ml, mr);
}

/****************************************************************************
**
*F  DiffMat8BitMat8Bit( <ml> ,<mr>)
**
**  Caller's job to do all checks
*/

libGAP_Obj libGAP_DiffMat8BitMat8Bit( libGAP_Obj ml, libGAP_Obj mr)
{
  libGAP_Obj diff;
  libGAP_UInt q;
  libGAP_UInt i;
  libGAP_Obj  row;
  libGAP_Obj type;
  libGAP_Obj info;
  libGAP_FF f;
  libGAP_FFV minusOne;
  libGAP_Obj mone;
  libGAP_UInt ll,lr,wl,wr,ld;
  
  ll = libGAP_LEN_MAT8BIT(ml);
  lr = libGAP_LEN_MAT8BIT(mr);
  wl = libGAP_LEN_MAT8BIT(libGAP_ELM_MAT8BIT(ml,1));
  wr = libGAP_LEN_MAT8BIT(libGAP_ELM_MAT8BIT(mr,1));

  /* We have to track the cases where the result is not rectangular */
  if (((ll > lr) && (wr > wl)) ||
      ((lr > ll) && (wl > wr)))
    return libGAP_TRY_NEXT_METHOD;

  /* Now sort out the size of the result */
  if (ll > lr)
    {
      ld = ll;
      assert(wl > wr);
    }
  else
    {
      ld = lr;
      assert(wr >= wl);
    }
  q = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(ml,1));

  if (q % 2 == 0)
    return libGAP_SumMat8BitMat8Bit(ml,mr);
  
  diff = libGAP_NewBag(libGAP_T_POSOBJ, sizeof(libGAP_Obj)*(ld+2));
  type = libGAP_TypeMat8Bit( q, libGAP_IS_MUTABLE_OBJ(ml) || libGAP_IS_MUTABLE_OBJ(mr));
  libGAP_TYPE_POSOBJ(diff) = type;
  libGAP_SET_LEN_MAT8BIT(diff, ld);
  type = libGAP_TypeVec8BitLocked(q,libGAP_IS_MUTABLE_OBJ(libGAP_ELM_MAT8BIT(ml,1)) || libGAP_IS_MUTABLE_OBJ(libGAP_ELM_MAT8BIT(mr,1)) );
  info = libGAP_GetFieldInfo8Bit(q);
  f = libGAP_FiniteField( libGAP_P_FIELDINFO_8BIT(info), libGAP_D_FIELDINFO_8BIT(info));
  minusOne = libGAP_NEG_FFV( 1,libGAP_SUCC_FF(f) );
  mone = libGAP_NEW_FFE( f, minusOne);

  for (i = 1; i <= ld; i++)
    {
      if (i > ll)
  row = libGAP_MultVec8BitFFE( libGAP_ELM_MAT8BIT(mr,i), mone);
      else if (i > lr)
  row = libGAP_CopyVec8Bit(libGAP_ELM_MAT8BIT(ml,i),1);
      else
  row = libGAP_SumVec8BitVec8BitMult(libGAP_ELM_MAT8BIT(ml,i), libGAP_ELM_MAT8BIT(mr,i), mone);
      
      libGAP_TYPE_DATOBJ(row) = type;
      libGAP_SET_ELM_MAT8BIT(diff, i, row);
      libGAP_CHANGED_BAG(diff);
    }
  return diff;
}

/****************************************************************************
**
*F  FuncDIFF_MAT8BIT_MAT8BIT( <self>, <ml>, <mr> )
**
**  Caller should check that both args are mat8bit over the same field
*/

libGAP_Obj libGAP_FuncDIFF_MAT8BIT_MAT8BIT( libGAP_Obj self, libGAP_Obj ml, libGAP_Obj mr)
{
  libGAP_UInt q;

  q = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(ml,1));
  if (q != libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(mr,1)))
    return libGAP_TRY_NEXT_METHOD;
  else
    return libGAP_DiffMat8BitMat8Bit( ml, mr);
}


/****************************************************************************
**
*f * * * * * * polynomial support functions * * * * * * * * * * * * * * * * *
**
** The first batch are utilities for the others
*/

libGAP_UInt libGAP_RightMostNonZeroVec8Bit( libGAP_Obj vec)
{
  libGAP_UInt q;
  libGAP_UInt len;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt1 *ptr, *ptrS;
  libGAP_Int i;
  libGAP_UInt1 *gettab;
/*UInt1 byte; */
  len = libGAP_LEN_VEC8BIT(vec);
  if (len == 0)
    return 0;
  q = libGAP_FIELD_VEC8BIT(vec);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  ptrS = libGAP_BYTES_VEC8BIT(vec);
  ptr = ptrS + (len -1)/elts;

  /* handle last byte specially, unless it happens to be full */
  if (len % elts != 0)
    {
      gettab = libGAP_GETELT_FIELDINFO_8BIT(info) + *ptr;
      for (i = len % elts -1; i >= 0; i--)
  {
    if (gettab[256*i] != 0)
      return (elts*(len/elts) + i + 1);
  }
      ptr--;
    }
  
  /* now skip over empty bytes */
  while (ptr >= ptrS && *ptr == 0)
    ptr --;
  if (ptr < ptrS)
    return 0;


  /* Now look in the rightmost non-empty byte for the position */
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info) + *ptr;
  for (i = elts-1; i >= 0; i--)
    {
      if (gettab[256 * i]  != 0)
  return (elts*(ptr - ptrS) + i + 1);
    }
  libGAP_Pr("panic: this should never happen\n", 0, 0);
  libGAP_SyExit(1);
  /* please picky compiler */
  return 0;
}

void libGAP_ResizeVec8Bit( libGAP_Obj vec, libGAP_UInt newlen, libGAP_UInt knownclean )
{
  libGAP_UInt q;
  libGAP_UInt len;
  libGAP_UInt elts;
  libGAP_Obj info;
  libGAP_UInt1 *settab;
  libGAP_UInt i;
  libGAP_UInt1 *ptr, *ptr2, byte;
  len = libGAP_LEN_VEC8BIT(vec);
  if (len == newlen)
    return;

  if (libGAP_True == libGAP_DoFilter(libGAP_IsLockedRepresentationVector, vec ))
    {
      libGAP_ErrorReturnVoid("Resize of locked compressed vector is forbidden", 0, 0,
          "You can `return;' to ignore the operation");
      return;
    }

  /*  
  if (newlen == 0)
    {
      RetypeBag(vec, T_PLIST_EMPTY);
      SET_LEN_PLIST(vec,0);
      SHRINK_PLIST(vec,0);
      return;
    }
  */
  q = libGAP_FIELD_VEC8BIT(vec);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  libGAP_SET_LEN_VEC8BIT(vec,newlen);
  libGAP_ResizeBag( vec, libGAP_SIZE_VEC8BIT( newlen, elts));
  /* vector has got shorter. */
  if (len > newlen) {
    if (newlen % elts) {
      /* clean spare entries in last byte */
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      byte = libGAP_BYTES_VEC8BIT(vec)[(newlen-1)/elts];
      for (i = newlen % elts; i < elts; i++)
        byte = settab[ byte + 256 * i];
      libGAP_BYTES_VEC8BIT(vec)[(newlen-1)/elts] = byte;
    }
    /* Clean spare bytes in last word for characteristic 2 */
    if ((q % 2) == 0) 
      for (i = (newlen+elts -1)/elts; i % sizeof(libGAP_UInt); i++ )
        libGAP_BYTES_VEC8BIT(vec)[i] = 0;
  }

  /* vector has got longer and might be dirty */
  if (!knownclean && newlen > len)
    {
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      ptr = libGAP_BYTES_VEC8BIT(vec);
      if (len) {
        ptr += (len -1) / elts;
        byte = *ptr;
        for (i = (len-1) % elts + 1; i < elts; i++)
          byte = settab[ byte + 256 * i];
        *ptr++ = byte;
      }
      ptr2 = libGAP_BYTES_VEC8BIT(vec) + (newlen + elts -1) / elts;
      while (ptr < ptr2)
      *ptr++ = (libGAP_UInt1)0;
    }
  return;
}


void libGAP_ShiftLeftVec8Bit( libGAP_Obj vec, libGAP_UInt amount)
{
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt len;
  libGAP_UInt1 *ptr1, *ptr2, *end;
  libGAP_UInt1 fbyte, tbyte;
  libGAP_UInt from, to;
  libGAP_UInt1 *gettab, *settab;
  libGAP_UInt1 x;

  /* A couple of trivial cases */
  if (amount == 0)
    return;
  len = libGAP_LEN_VEC8BIT(vec);
  if (amount >= len)
    {
      libGAP_ResizeVec8Bit(vec,0, 0);
      return;
    }

  
  q = libGAP_FIELD_VEC8BIT(vec);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  ptr1 = libGAP_BYTES_VEC8BIT(vec);
  ptr2 = libGAP_BYTES_VEC8BIT(vec) + amount /elts;
  
  /* The easy case is just a shift by bytes */
  if (amount % elts == 0)
    {
      end = libGAP_BYTES_VEC8BIT(vec) + (len +elts -1)/elts;
      while (ptr2 < end)
  *ptr1++ = *ptr2++;
    }
  else
    /* The general case */
    {
      from = amount;
      to = 0;
      fbyte = *ptr2;
      tbyte = 0;
      gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      
      while (from < len)
  {
    x = gettab[fbyte + 256*(from % elts)];
    tbyte = settab[tbyte + 256*(to % elts + elts *x)];
    if (++from % elts == 0)
      fbyte = *++ptr2;
    if (++to % elts == 0)
      {
        *ptr1++ = tbyte;
        tbyte = 0;
      }
  }
      if (to % elts != 0)
  *ptr1 = tbyte;
    }
  libGAP_ResizeVec8Bit(vec, len - amount, 0);
  return;
}

void libGAP_ShiftRightVec8Bit( libGAP_Obj vec, libGAP_UInt amount) /* pads with zeros */
{
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt len;
  libGAP_UInt1 *ptr1, *ptr2, *end;
  libGAP_UInt1 fbyte, tbyte;
  libGAP_Int from, to;
  libGAP_UInt1 *gettab, *settab;
  libGAP_UInt1 x;

  /* A trivial cases */
  if (amount == 0)
    return;

  /* make room */
  len = libGAP_LEN_VEC8BIT(vec);
  libGAP_ResizeVec8Bit(vec, len + amount, 0);

  q = libGAP_FIELD_VEC8BIT(vec);
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  ptr1 = libGAP_BYTES_VEC8BIT(vec) + (len - 1 + amount)/elts;
  ptr2 = libGAP_BYTES_VEC8BIT(vec) + (len -1)/elts;

  /* The easy case is just a shift by bytes */
  if (amount % elts == 0)
    {
      end = libGAP_BYTES_VEC8BIT(vec);
      while (ptr2 >= end)
  *ptr1-- = *ptr2--;
      while ( ptr1 >= end)
  *ptr1-- = (libGAP_UInt1)0;
    }
  else
    /* The general case */
    {
      from = len-1;
      to = len + amount -1;
      fbyte = *ptr2;
      tbyte = 0;
      gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      
      while (from >= 0)
  {
    x = gettab[fbyte + 256*(from % elts)];
    tbyte = settab[tbyte + 256*(to % elts + elts *x)];
    if (from-- % elts == 0)
      fbyte = *--ptr2;
    if (to-- % elts == 0)
      {
        *ptr1-- = tbyte;
        tbyte = 0;
      }
  }
      if (to % elts != elts-1)
  *ptr1-- = tbyte;
      end = libGAP_BYTES_VEC8BIT(vec);
      while (ptr1 >= end)
  *ptr1-- = (libGAP_UInt1)0;
    }
  return;
}




/****************************************************************************
**
*F FuncADD_COEFFS_VEC8BIT_3( <self>, <vec1>, <vec2>, <mult> )
**
** This is very like AddRowVector, except that it will enlarge <vec1> if
** necessary and returns the position of the rightmost non-zero entry in the
** result.
*/

libGAP_Obj libGAP_FuncADD_COEFFS_VEC8BIT_3( libGAP_Obj self, libGAP_Obj vec1, libGAP_Obj vec2, libGAP_Obj mult )
{
  libGAP_UInt q;
  libGAP_UInt len;
  len = libGAP_LEN_VEC8BIT(vec2);
  if (libGAP_VAL_FFE(mult) == 0)
    return libGAP_INTOBJ_INT(libGAP_RightMostNonZeroVec8Bit(vec1));
  if (libGAP_LEN_VEC8BIT(vec1) < len)
    {
      libGAP_ResizeVec8Bit(vec1, len, 0);
    }
  /* Now we know that the characteristics must match, but not the fields */
  q = libGAP_FIELD_VEC8BIT(vec1);
  /* fix up fields if necessary */
  if (q != libGAP_FIELD_VEC8BIT(vec2) || q != libGAP_SIZE_FF(libGAP_FLD_FFE(mult)))
    {
      libGAP_Obj info, info1;
      libGAP_UInt d,d1, q1,d2, d0, q0, p, i;
      libGAP_FFV val;
      /* find a common field */
      info = libGAP_GetFieldInfo8Bit(q);
      d = libGAP_D_FIELDINFO_8BIT(info);
      q1 = libGAP_FIELD_VEC8BIT(vec2);
      info1 = libGAP_GetFieldInfo8Bit(q1);
      d1 = libGAP_D_FIELDINFO_8BIT(info1);
      d2 = libGAP_DegreeFFE(mult);
      d0 = libGAP_LcmDegree(d,d1);
      d0 = libGAP_LcmDegree(d0,d2);
      p = libGAP_P_FIELDINFO_8BIT(info);
      assert(p == libGAP_P_FIELDINFO_8BIT(info1));
      assert(p == libGAP_CHAR_FF(libGAP_FLD_FFE(mult)));
      q0 = 1;
      for (i = 0; i < d0; i++)
  q0 *= p;
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (d0>8||q0 > 256)
  return libGAP_TRY_NEXT_METHOD;
      if ( (q0 > q && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vec1) == libGAP_True) ||
     (q0 > q1 && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vec2) == libGAP_True))
  return libGAP_TRY_NEXT_METHOD;
      libGAP_RewriteVec8Bit(vec1,q0);
      libGAP_RewriteVec8Bit(vec2,q0);
      val = libGAP_VAL_FFE(mult);
      if (val != 0)
  val = 1 + (val-1)*(q0-1)/(libGAP_SIZE_FF(libGAP_FLD_FFE(mult))-1);
      mult = libGAP_NEW_FFE(libGAP_FiniteField(p,d0),val);
      q = q0;
    }
  libGAP_AddVec8BitVec8BitMultInner( vec1, vec1, vec2, mult, 1, len);
  return libGAP_INTOBJ_INT(libGAP_RightMostNonZeroVec8Bit(vec1));
}

/****************************************************************************
**
*F FuncADD_COEFFS_VEC8BIT_2( <self>, <vec1>, <vec2> )
**
** This is very like AddRowVector, except that it will enlarge <vec1> if
** necessary and returns the position of the rightmost non-zero entry in the
** result.
*/

libGAP_Obj libGAP_FuncADD_COEFFS_VEC8BIT_2( libGAP_Obj self, libGAP_Obj vec1, libGAP_Obj vec2 )
{
  libGAP_UInt q;
  libGAP_UInt len;
  len = libGAP_LEN_VEC8BIT(vec2);
  if (libGAP_LEN_VEC8BIT(vec1) < len)
    {
      libGAP_ResizeVec8Bit(vec1, len, 0);
    }
  /* Now we know that the characteristics must match, but not the fields */
  q = libGAP_FIELD_VEC8BIT(vec1);
  /* fix up fields if necessary */
  if (q != libGAP_FIELD_VEC8BIT(vec2))
    {
      libGAP_Obj info, info1;
      libGAP_UInt d,d1, q1,d0, q0, p, i;
      /* find a common field */
      info = libGAP_GetFieldInfo8Bit(q);
      d = libGAP_D_FIELDINFO_8BIT(info);
      q1 = libGAP_FIELD_VEC8BIT(vec2);
/*Pr("q= %d q1= %d ",q,q1);*/
      info1 = libGAP_GetFieldInfo8Bit(q1);
      d1 = libGAP_D_FIELDINFO_8BIT(info1);
      d0 = libGAP_LcmDegree(d,d1);
      p = libGAP_P_FIELDINFO_8BIT(info);
      assert(p == libGAP_P_FIELDINFO_8BIT(info1));
      q0 = 1;
      for (i = 0; i < d0; i++)
  q0 *= p;
/*Pr("q0= %d d0= %d\n",q0,d0); */
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (d0>8||q0 > 256)
  return libGAP_TRY_NEXT_METHOD;
      if ( (q0 > q && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vec1) == libGAP_True) ||
     (q0 > q1 && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vec2) == libGAP_True))
  return libGAP_TRY_NEXT_METHOD;
      libGAP_RewriteVec8Bit(vec1,q0);
      libGAP_RewriteVec8Bit(vec2,q0);
      q = q0;
    }
  libGAP_AddVec8BitVec8BitInner( vec1, vec1, vec2, 1, len);
  return libGAP_INTOBJ_INT(libGAP_RightMostNonZeroVec8Bit(vec1));
}

/****************************************************************************
**
*F  FuncSHIFT_VEC8BIT_LEFT( <self>, <vec>, <amount> )
**
*/

libGAP_Obj libGAP_FuncSHIFT_VEC8BIT_LEFT( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj amount)
{
  /* should be checked in method selection */
  assert(libGAP_IS_MUTABLE_OBJ(vec));
  while (!libGAP_IS_INTOBJ(amount) || libGAP_INT_INTOBJ(amount) < 0)
    {
      amount = libGAP_ErrorReturnObj("SHIFT_VEC8BIT_LEFT: <amount> must be a non-negative small integer",
            0,0,
            "you can replace <amount> via 'return <amount>;'");
    }
  libGAP_ShiftLeftVec8Bit( vec, libGAP_INT_INTOBJ(amount));
  return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  FuncSHIFT_VEC8BIT_RIGHT( <self>, <vec>, <amount>, <zero> )
**
*/

libGAP_Obj libGAP_FuncSHIFT_VEC8BIT_RIGHT( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj amount, libGAP_Obj zero)
{
  assert(libGAP_IS_MUTABLE_OBJ(vec));
  while (!libGAP_IS_INTOBJ(amount) || libGAP_INT_INTOBJ(amount) < 0)
    {
      amount = libGAP_ErrorReturnObj("SHIFT_VEC8BIT_RIGHT: <amount> must be a non-negative small integer",
            0,0,
            "you can replace <amount> via 'return <amount>;'");
    }
  libGAP_ShiftRightVec8Bit( vec, libGAP_INT_INTOBJ(amount));
  return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  FuncRESIZE_VEC8BIT( <self>, <vec>, <newsize> )
**
*/

libGAP_Obj libGAP_FuncRESIZE_VEC8BIT( libGAP_Obj self, libGAP_Obj vec, libGAP_Obj newsize )
{
  if (!libGAP_IS_MUTABLE_OBJ(vec))
    libGAP_ErrorReturnVoid("RESIZE_VEC8BIT: vector must be mutable", 0, 0, 
                    "you can 'return;'");
  while (libGAP_IS_INTOBJ(newsize) && libGAP_INT_INTOBJ(newsize) < 0)
    {
      newsize = libGAP_ErrorReturnObj("RESIZE_VEC8BIT: <amount> must be a non-negative integer, not %d",libGAP_INT_INTOBJ(newsize),0,
             "you can replace <amount> via 'return <amount>;'");
    }
  libGAP_ResizeVec8Bit( vec, libGAP_INT_INTOBJ(newsize), 0);
  return (libGAP_Obj) 0;
}
  
/****************************************************************************
**
*F  FuncRIGHTMOST_NONZERO_VEC8BIT( <self>, <vec> )
**
*/

libGAP_Obj libGAP_FuncRIGHTMOST_NONZERO_VEC8BIT( libGAP_Obj self, libGAP_Obj vec)
{
  return libGAP_INTOBJ_INT(libGAP_RightMostNonZeroVec8Bit(vec));
}

/****************************************************************************
**
*F  ProdCoeffsVec8Bit( <res>, <vl>, <ll>, <vr>, <lr>)
**
*/

void libGAP_ProdCoeffsVec8Bit ( libGAP_Obj res, libGAP_Obj vl, libGAP_UInt ll, libGAP_Obj vr, libGAP_UInt lr )
{
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt1 * addtab = 0;
  libGAP_UInt1 * pmulltab;
  libGAP_UInt1 * pmulutab = 0;
  libGAP_UInt p;
  libGAP_UInt i,j;
  libGAP_UInt1 *ptrl, *ptrr, *ptrp, bytel, byter;
  libGAP_UInt1 byte1, byte2;
  libGAP_UInt1 * gettab, *settab;
  libGAP_UInt1 partl = 0, partr = 0;
  q = libGAP_FIELD_VEC8BIT(vl);
  assert(q == libGAP_FIELD_VEC8BIT(vr));
  assert(q == libGAP_FIELD_VEC8BIT(res));
  assert(ll <= libGAP_LEN_VEC8BIT(vl));
  assert(lr <= libGAP_LEN_VEC8BIT(vr));
  assert(ll+lr-1 <= libGAP_LEN_VEC8BIT(res));
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  p = libGAP_P_FIELDINFO_8BIT(info);
  pmulltab = libGAP_PMULL_FIELDINFO_8BIT(info);
  if ( q <= 16)
    pmulutab = libGAP_PMULU_FIELDINFO_8BIT(info);
  if (p != 2)
    addtab = libGAP_ADD_FIELDINFO_8BIT(info);
  ptrl = libGAP_BYTES_VEC8BIT(vl);
  ptrr = libGAP_BYTES_VEC8BIT(vr);
  ptrp = libGAP_BYTES_VEC8BIT(res);

  /* This calculation is done in four parts. The first deals with the whole
     bytes from both polynomials */
  for (i = 0; i < ll/elts; i++)
    {
      bytel = ptrl[i];
      if (bytel != 0)
  for (j = 0; j < lr/elts; j++)
    {
      byter = ptrr[j];
      if (byter != 0)
        {
    byte1 = pmulltab[ 256*bytel + byter];
    if (byte1 != 0) {
      if (p != 2)
        ptrp[i+j] = addtab[ ptrp[i+j] + 256*byte1];
      else
        ptrp[i+j] ^= byte1;
    }
    if (elts > 1)
      {
        byte2 = pmulutab[ 256*bytel + byter];
        if (byte2 != 0) {
          if (p != 2)
      ptrp[i+j+1] = addtab[ ptrp[i+j+1] + 256*byte2];
          else
      ptrp[i+j+1] ^= byte2;
        }
      }
        }
    }
    }

  /* The next two deal with the end byte from each polynomial, in combination with the whole
     bytes from the other polynomial */
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  settab = libGAP_SETELT_FIELDINFO_8BIT(info);
  if (ll % elts != 0)
    {
      bytel = ptrl[ll/elts];
      if (bytel != 0)
  {
    partl = 0;
    for (i = (ll/elts)*elts; i < ll; i++)
      {
        byte1 = gettab[bytel+256*(i%elts)];
        partl = settab[partl + 256*(i  %elts + elts*byte1)];
      }
    if (partl != 0)
      for (j = 0; j < lr/elts; j++)
        {
    byter = ptrr[j];
    if (byter != 0)
      {
        byte2 = pmulltab[ 256*partl + byter];
        if (byte2 != 0) {
          if (p != 2)
      ptrp[ll/elts + j] = addtab[ ptrp[ll/elts+j] + 256*byte2];
          else
      ptrp[ll/elts+j] ^= byte2;
        }
        if (elts > 1)
          {
      byte2 = pmulutab[ 256*partl + byter];
      if (byte2 != 0) {
        if (p !=2)
          ptrp[ll/elts +j+1] = addtab[ ptrp[ll/elts+j+1] + 256*byte2];
        else
          ptrp[ll/elts+j+1] ^= byte2;
      }
          }
      }
        }
  }
    }
  if (lr % elts != 0)
    {
      byter = ptrr[lr/elts];
      if (byter != 0)
  {
    partr = 0;
    for (i = (lr/elts)*elts; i < lr; i++)
      partr = settab[partr + 256*(i  %elts + elts*gettab[byter+256*(i%elts)])];
    if (partr != 0)
      for (i = 0; i < ll/elts; i++)
        {
    bytel = ptrl[i];
    if (bytel != 0)
      {
        byte1 = pmulltab[ 256*partr + bytel];
        if (byte1 != 0) {
          if (p != 2)
      ptrp[lr/elts  + i] = addtab[ ptrp[lr/elts+i] + 256*byte1];
          else
      ptrp[lr/elts + i] ^= byte1;
        }
        if (elts > 1)
          {
      byte1 = pmulutab[ 256*partr + bytel];
      if (byte1 != 0) {
        if (p != 2)
          ptrp[lr/elts +i+1] = addtab[ ptrp[lr/elts+i+1] + 256*byte1];
        else
          ptrp[lr/elts+i+1] ^= byte1;
      }
          }
      }
        }
  }
    }

  /* Finally, we have to multiply the two end bytes */
  if (ll % elts != 0 && lr % elts != 0 && partl != 0 && partr != 0)
    {
      byte1 = pmulltab[ partl + 256*partr];
      if (byte1 != 0) {
  if (p != 2)
    ptrp[ll/elts + lr/elts] = addtab[ptrp[ll/elts + lr/elts] + 256 * byte1];
  else
    ptrp[ll/elts + lr/elts] ^= byte1;
      }
      if (elts > 1)
  {
    byte2 = pmulutab[ partl + 256*partr];
    if (byte2 != 0) {
      if (p != 2)
        ptrp[ll/elts + lr/elts + 1] = addtab[ptrp[ll/elts + lr/elts+ 1] + 256 * byte2];
      else
        ptrp[ll/elts + lr/elts + 1] ^= byte2;
    }
  }
    }
  return;
}

/****************************************************************************
**
*F  FuncPROD_COEFFS_VEC8BIT( <self>, <vl>, <ll>, <vr>, <lr> )
**
*/

libGAP_Obj libGAP_FuncPROD_COEFFS_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj ll, libGAP_Obj vr, libGAP_Obj lr  )
{
  libGAP_Int ll1,lr1;
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_Obj res;
  libGAP_UInt lenp;
  libGAP_UInt last;
  q = libGAP_FIELD_VEC8BIT(vl);
  if (q != libGAP_FIELD_VEC8BIT(vr))
    {
      libGAP_Obj  info1;
      libGAP_UInt d,d1, q1,d0, q0, p, i;
      /* find a common field */
      info = libGAP_GetFieldInfo8Bit(q);
      d = libGAP_D_FIELDINFO_8BIT(info);
      q1 = libGAP_FIELD_VEC8BIT(vr);
      info1 = libGAP_GetFieldInfo8Bit(q1);
      d1 = libGAP_D_FIELDINFO_8BIT(info1);
      d0 = libGAP_LcmDegree(d,d1);
      p = libGAP_P_FIELDINFO_8BIT(info);
      assert(p == libGAP_P_FIELDINFO_8BIT(info1));
      q0 = 1;
      for (i = 0; i < d0; i++)
  q0 *= p;
      /* if the exponent is bigger than 31, overflow changes the value to 0 */
      if (d0>8||q0 > 256)
  return libGAP_TRY_NEXT_METHOD;
      if ( (q0 > q && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vl) == libGAP_True) ||
     (q0 > q1 && libGAP_CALL_1ARGS(libGAP_IsLockedRepresentationVector, vr) == libGAP_True))
  return libGAP_TRY_NEXT_METHOD;
      libGAP_RewriteVec8Bit(vl,q0);
      libGAP_RewriteVec8Bit(vr,q0);
      q = q0;
    }
  if (!libGAP_ARE_INTOBJS(ll,lr))
    libGAP_ErrorQuit("PROD_COEFFS_VEC8BIT: both lengths must be small integers, not a %s and a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(ll), (libGAP_Int)libGAP_TNAM_OBJ(lr));
  ll1 = libGAP_INT_INTOBJ(ll);
  lr1 = libGAP_INT_INTOBJ(lr);
  if (0 > ll1 || ll1 > libGAP_LEN_VEC8BIT(vl))
    libGAP_ErrorQuit("ProdCoeffs: given length <ll> of left argt (%d)\n is negative or longer than the argt (%d)",
        libGAP_INT_INTOBJ(ll), libGAP_LEN_VEC8BIT(vl));
  if (0 > lr1 || lr1 > libGAP_LEN_VEC8BIT(vr))
    libGAP_ErrorQuit("ProdCoeffs: given length <lr> of right argt (%d)\n is negative or longer than the argt (%d)",
        libGAP_INT_INTOBJ(lr), libGAP_LEN_VEC8BIT(vr));
  info = libGAP_GetFieldInfo8Bit(q);
  if (ll1 == 0 && lr1 == 0)
    lenp = 0;
  else
    lenp = ll1 + lr1 - 1;
  res = libGAP_ZeroVec8Bit(q, lenp , 1);
  libGAP_ProdCoeffsVec8Bit( res, vl, ll1, vr, lr1);
  last = libGAP_RightMostNonZeroVec8Bit(res);
  if (last != lenp)
    libGAP_ResizeVec8Bit(res, last, 1);
  return res;
}

/****************************************************************************
**
*F  ReduceCoeffsVec8Bit( <vl>, <ll>, <vr>, <lr> )
**
*/

libGAP_Obj libGAP_MakeShiftedVecs( libGAP_Obj v, libGAP_UInt len)
{
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_Obj shifts;
  libGAP_Obj ashift;
  libGAP_Obj vn, xi;
  libGAP_UInt i,j;
  libGAP_Obj *ffefelt;
  libGAP_UInt1 *gettab;
  libGAP_UInt1 *settab;
  libGAP_UInt len1;
  libGAP_UInt1 x;
  libGAP_UInt1 *ptr;
  libGAP_UInt1 *ptrs[5]; /* 5 is the largest value of elts we ever meet */
  libGAP_Obj type;
  
  q = libGAP_FIELD_VEC8BIT(v);
  assert( len <= libGAP_LEN_VEC8BIT(v));
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);

  /* normalize a copy of v in vn -- normalize means monic, and actual length
     equal to length parameter */
  vn = libGAP_CopyVec8Bit(v,1);
  libGAP_ResizeVec8Bit(vn, len, 0);
  len1 = (len == 0) ? 0 : libGAP_RightMostNonZeroVec8Bit(vn);
  if (len1 == 0)
    libGAP_ErrorReturnVoid("Zero coefficient vector for reduction",0,0,
                    "you can 'return;'");
  if (len1 != len)
    {
      libGAP_ResizeVec8Bit(vn, len1, 1);
      len = len1;
    }

  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  ffefelt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);

  x = gettab[libGAP_BYTES_VEC8BIT(vn)[(len-1)/elts] + 256*((len-1) % elts)];
  assert(x != 0);
  xi = libGAP_INV(ffefelt[x]);
  libGAP_MultVec8BitFFEInner( vn, vn,  xi ,1, len);
  type = libGAP_TypeVec8Bit(q, 0);
  libGAP_TYPE_DATOBJ(vn) = type;

  /* Now we start to build up the result */
  shifts = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB + libGAP_IMMUTABLE, elts + 2);
  libGAP_SET_ELM_PLIST( shifts, elts + 1, libGAP_INTOBJ_INT(len));
  libGAP_SET_ELM_PLIST( shifts, elts + 2, xi);
  libGAP_SET_LEN_PLIST(shifts, elts+2);

  /* vn can simply be stored in one place */
  libGAP_SET_ELM_PLIST( shifts, (len-1) % elts + 1, vn);
  libGAP_CHANGED_BAG(shifts);

  if (elts > 1)
    {
      /* fill the rest up with zero vectors of suitable lengths */
      for (i = 1; i < elts; i++)
  {
    ashift = libGAP_ZeroVec8Bit(q, len+i, 0);
    libGAP_SET_ELM_PLIST( shifts, (len + i-1) % elts + 1, ashift);
    libGAP_CHANGED_BAG(shifts);
  }

      /* reload the tables, in case there was a garbage collection */
      gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      /* Now run through the entries of vn inserting them into the shifted versions */
      ptr = libGAP_BYTES_VEC8BIT(vn);
      for (j = 1; j < elts; j++)
  ptrs[j] = libGAP_BYTES_VEC8BIT(libGAP_ELM_PLIST(shifts, (len + j -1) % elts + 1));
      for (i = 0; i < len; i++)
  {
    x = gettab[*ptr + 256 *(i % elts)];
    if (x != 0)
      {
        for (j = 1; j < elts; j++)
    {
      *(ptrs[j]) = settab[*(ptrs[j]) + 256*((i+j)%elts + elts*x)];
    }
      }
    if (i % elts == elts-1)
      ptr++;
    else
      ptrs[elts - 1 - (i % elts)] ++;
  }
    }
  return shifts;
}

void libGAP_ReduceCoeffsVec8Bit ( libGAP_Obj vl, libGAP_Obj vrshifted, libGAP_Obj quot )
{
  libGAP_UInt q;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_Int i,j,jj;
  libGAP_UInt1 *gettab;
  libGAP_UInt1 *ptrl1, *ptrl, *ptrr, *qptr = 0;
  libGAP_UInt1 x;
  libGAP_UInt1 xn;
  libGAP_UInt p;
  libGAP_UInt lr;
  libGAP_UInt lrs;
  libGAP_UInt1 *multab, *settab=0;
  libGAP_UInt1 *addtab = 0;
  libGAP_UInt1 * feltffe;
  libGAP_UInt1 y;
  libGAP_UInt ll = libGAP_LEN_VEC8BIT(vl);
  libGAP_Obj vrs;
  libGAP_Obj *ffefelt;
  q = libGAP_FIELD_VEC8BIT(vl);
  info = libGAP_GetFieldInfo8Bit(q);
  p = libGAP_P_FIELDINFO_8BIT(info);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  feltffe = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
  ffefelt = libGAP_FFE_FELT_FIELDINFO_8BIT(info);
  if (quot)
    {
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      qptr = libGAP_BYTES_VEC8BIT(quot);
    }
  if (p != 2)
    addtab = libGAP_ADD_FIELDINFO_8BIT(info);
  ptrl = libGAP_BYTES_VEC8BIT(vl);
  lr = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(vrshifted, elts+1));
  for (i = ll-1, jj = ll-lr; i+1 >= lr; i--,jj--)
    {
      ptrl1 = ptrl + i/elts;
      x = gettab[*ptrl1 + 256*(i%elts)];
      if (qptr)
  qptr[jj/elts] = settab[qptr[jj/elts]+ 256 * ( jj % elts + elts* x )];
      if (x != 0)
  {
    
    if (p == 2)
      xn = x;
    else
      xn = feltffe[libGAP_VAL_FFE(libGAP_AINV(ffefelt[x]))];
    multab = libGAP_SCALAR_FIELDINFO_8BIT(info) + 256*xn;
    vrs = libGAP_ELM_PLIST(vrshifted, 1+i%elts);
    lrs = libGAP_LEN_VEC8BIT(vrs);
    ptrr = libGAP_BYTES_VEC8BIT(vrs) + (lrs-1)/elts;
    for (j = (lrs -1)/elts; j >= 0; j--)
      {
        y = multab[*ptrr];
        if (p == 2)
    *ptrl1 ^= y;
        else
    *ptrl1 = addtab[*ptrl1 + 256 * y];
        ptrl1--;
        ptrr--;
      }
    assert( ! gettab[ptrl[i/elts] + 256*(i%elts)]);
  }
    }
  if (quot)
    {
      libGAP_MultVec8BitFFEInner(quot,quot,libGAP_ELM_PLIST(vrshifted,elts+2), 1, ll-lr+1);
    }
  return;
}

/****************************************************************************
**
*F  FuncREDUCE_COEFFS_VEC8BIT( <self>, <vl>, <ll>, <vr>, <lr> )
**
**  NB note that these are not methods and MAY NOT return TRY_NEXT_METHOD
*/

libGAP_Obj libGAP_FuncMAKE_SHIFTED_COEFFS_VEC8BIT( libGAP_Obj self, libGAP_Obj vr, libGAP_Obj lr)
{
  if (!libGAP_IS_INTOBJ(lr))
    libGAP_ErrorQuit("ReduceCoeffs: Length of right argument should be a small integer, not a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(lr), 0L);
  if (libGAP_INT_INTOBJ(lr) < 0 || libGAP_INT_INTOBJ(lr) > libGAP_LEN_VEC8BIT(vr))
    {
      libGAP_ErrorQuit("ReduceCoeffs: given length <lr> of right argt (%d)\n is negative or longer than the argt (%d)",
    libGAP_INT_INTOBJ(lr), libGAP_LEN_VEC8BIT(vr)); 
    }
  return libGAP_MakeShiftedVecs(vr, libGAP_INT_INTOBJ(lr));
}


libGAP_Obj libGAP_FuncREDUCE_COEFFS_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj ll, libGAP_Obj vrshifted)
{
  libGAP_UInt q;
  libGAP_UInt last;
  q = libGAP_FIELD_VEC8BIT(vl);
  if (q != libGAP_FIELD_VEC8BIT(libGAP_ELM_PLIST(vrshifted,1)))
    return libGAP_Fail;
  if (!libGAP_IS_INTOBJ(ll))
    libGAP_ErrorQuit("ReduceCoeffs: Length of left argument should be a small integer, not a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(ll), 0L);
  if (0 > libGAP_INT_INTOBJ(ll) || libGAP_INT_INTOBJ(ll) > libGAP_LEN_VEC8BIT(vl))
    {
       libGAP_ErrorQuit( "ReduceCoeffs: given length <ll> of left argt (%d)\n is negative or longer than the argt (%d)",
          libGAP_INT_INTOBJ(ll), libGAP_LEN_VEC8BIT(vl)); 
    }
  libGAP_ResizeVec8Bit(vl, libGAP_INT_INTOBJ(ll), 0);
  libGAP_ReduceCoeffsVec8Bit( vl,  vrshifted, (libGAP_Obj)0);
  last = libGAP_RightMostNonZeroVec8Bit(vl);
  libGAP_ResizeVec8Bit(vl, last, 1);
  return libGAP_INTOBJ_INT(last);
}

libGAP_Obj libGAP_FuncQUOTREM_COEFFS_VEC8BIT( libGAP_Obj self, libGAP_Obj vl, libGAP_Obj ll, libGAP_Obj vrshifted)
{
  libGAP_UInt q;
  libGAP_Obj rem, quot, ret, info;
  libGAP_UInt elts;
  libGAP_Int ill,lr;
  libGAP_Obj type;

  q = libGAP_FIELD_VEC8BIT(vl);
  if (q != libGAP_FIELD_VEC8BIT(libGAP_ELM_PLIST(vrshifted,1)))
    return libGAP_Fail;
  if (!libGAP_IS_INTOBJ(ll))
    libGAP_ErrorQuit("QuotRemCoeffs: Length of left argument should be a small integer, not a %s",
        (libGAP_Int)libGAP_TNAM_OBJ(ll), 0L);
  if (0 > libGAP_INT_INTOBJ(ll) || libGAP_INT_INTOBJ(ll) > libGAP_LEN_VEC8BIT(vl))
    {
       libGAP_ErrorQuit( "QuotRemCoeffs: given length <ll> of left argt (%d)\n is negative or longer than the argt (%d)",
          libGAP_INT_INTOBJ(ll), libGAP_LEN_VEC8BIT(vl)); 
    }
  ill = libGAP_INT_INTOBJ(ll);
  rem = libGAP_CopyVec8Bit( vl, 1);
  info = libGAP_GetFieldInfo8Bit(q);
  libGAP_ResizeVec8Bit(rem, ill, 0);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  lr = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(vrshifted, elts+1));
  quot = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT(ill-lr+1, elts));
  type = libGAP_TypeVec8Bit(q,1);
  libGAP_SET_TYPE_DATOBJ(quot, type);
  libGAP_SET_FIELD_VEC8BIT(quot, q);
  libGAP_SET_LEN_VEC8BIT(quot, ill-lr+1);
  libGAP_ReduceCoeffsVec8Bit( rem,  vrshifted, quot);
  ret = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB, 2);
  libGAP_SET_LEN_PLIST(ret,2);
  libGAP_SET_ELM_PLIST(ret,1,quot);
  libGAP_SET_ELM_PLIST(ret,2,rem);
  libGAP_CHANGED_BAG(ret);
  return ret;
}


/****************************************************************************
**
*F  Obj SemiechelonListVec8Bits( <mat>, <transformationneeded> )
**
**
** This is essentially a method for SemiEchelonMat or SemiEchelonMatTransformation
**
** <mat> is assumed by this point to be a list of mutable 8 bit vectors over
** the same field, which can be overwritten if necessary
*/

static libGAP_UInt libGAP_RNheads, libGAP_RNvectors, libGAP_RNcoeffs, libGAP_RNrelns;


libGAP_Obj libGAP_SemiEchelonListVec8Bits( libGAP_Obj mat, libGAP_UInt TransformationsNeeded )
{
  libGAP_UInt nrows, ncols;
  libGAP_UInt i,j,h;
/*UInt block; */
  libGAP_Obj heads,vectors, coeffs = 0, relns = 0;
  libGAP_UInt nvecs, nrels = 0;
  libGAP_Obj coeffrow = 0;
  libGAP_Obj row;
  libGAP_Obj res;
  libGAP_UInt q, elts;
  libGAP_Obj info;
  libGAP_UInt1 *settab, *convtab, *gettab;
  libGAP_Obj *convtab1;
  libGAP_UInt1 zero, one;
  libGAP_UInt1 x = 0;
  libGAP_UInt1 *rowp;
  libGAP_UInt1 byte;
  libGAP_Obj y;
  libGAP_Obj type;
 
  nrows = libGAP_LEN_PLIST(mat);
  ncols = libGAP_LEN_VEC8BIT(libGAP_ELM_PLIST(mat,1));

  /* Find the field info */
  q = libGAP_FIELD_VEC8BIT(libGAP_ELM_PLIST(mat, 1));
  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);

  /* Get the Felt numbers for zero and one */
  convtab = libGAP_FELT_FFE_FIELDINFO_8BIT(info);
  zero = convtab[0];
  one = convtab[1];

  /* Set up the lists for the results */
  heads = libGAP_NEW_PLIST(libGAP_T_PLIST_CYC, ncols);
  libGAP_SET_LEN_PLIST(heads, ncols);
  vectors = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT, nrows);
  libGAP_SET_LEN_PLIST(vectors, 0);
  nvecs = 0;
  if (TransformationsNeeded)
    {
      coeffs = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT, nrows);
      libGAP_SET_LEN_PLIST(coeffs, 0);
      relns  = libGAP_NEW_PLIST(libGAP_T_PLIST_TAB_RECT, nrows);
      libGAP_SET_LEN_PLIST(relns, 0);
      nrels = 0;
    }
  for (i = 1; i <= ncols; i++)
    libGAP_SET_ELM_PLIST(heads, i, libGAP_INTOBJ_INT(0));

  /* Main loop starts here */
  for (i = 1; i <= nrows; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (TransformationsNeeded)
  {
      coeffrow = libGAP_NewBag(libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT(nrows,elts));
      libGAP_SET_LEN_VEC8BIT(coeffrow, nrows);
      type = libGAP_TypeVec8Bit(q, 1);
      libGAP_TYPE_DATOBJ(coeffrow) = type;
      libGAP_SET_FIELD_VEC8BIT(coeffrow, q);
      libGAP_CHANGED_BAG(coeffrow);
      
      /* No garbage collection risk from here */
      settab = libGAP_SETELT_FIELDINFO_8BIT(info);
      libGAP_BYTES_VEC8BIT(coeffrow)[(i-1)/elts] = settab[256*((i-1) % elts + elts*one)];
  }
      /* No garbage collection risk from here */
      gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
      convtab1 = libGAP_FFE_FELT_FIELDINFO_8BIT(info);
      
      /* Clear out the current row */
      for (j = 1; j <= ncols; j++)
  {
    h = libGAP_INT_INTOBJ(libGAP_ELM_PLIST(heads, j));
    if (h != 0)
      {
        byte = libGAP_BYTES_VEC8BIT(row)[(j-1)/elts];
        if (byte && zero != (x = gettab[ byte+256*((j-1) % elts)]))
    {
      y = libGAP_AINV(convtab1[x]);
      libGAP_AddVec8BitVec8BitMultInner(row, row, libGAP_ELM_PLIST(vectors,h),y,1,ncols);
      if (TransformationsNeeded)
        libGAP_AddVec8BitVec8BitMultInner(coeffrow, coeffrow, libGAP_ELM_PLIST(coeffs,h),y,1,nrows);
    }
      }
  }
      j = 1;
      rowp = libGAP_BYTES_VEC8BIT(row);
      while (j <= ncols && !*rowp )
  {
    j += elts;
    rowp++;
  }
      while ( j <= ncols && (zero == (x = gettab[ *rowp + 256*((j-1) % elts)])))
  j++;

      if (j <= ncols)
  {
    y = libGAP_INV(convtab1[x]);
    libGAP_MultVec8BitFFEInner(row,row,y,1,ncols);
    libGAP_SET_ELM_PLIST(vectors, ++nvecs, row);
    libGAP_CHANGED_BAG(vectors);
    libGAP_SET_LEN_PLIST(vectors, nvecs);
    libGAP_SET_ELM_PLIST( heads, j, libGAP_INTOBJ_INT(nvecs));
    if (TransformationsNeeded)
      {
        libGAP_MultVec8BitFFEInner(coeffrow,coeffrow,y,1,nrows);
        libGAP_SET_ELM_PLIST(coeffs, nvecs, coeffrow);
        libGAP_CHANGED_BAG(coeffs);
        libGAP_SET_LEN_PLIST(coeffs, nvecs);
      }
    /* garbage collection OK again after here */
  }
      else if (TransformationsNeeded)
  {
    libGAP_SET_ELM_PLIST(relns, ++nrels, coeffrow);
    libGAP_CHANGED_BAG(relns);
    libGAP_SET_LEN_PLIST(relns, nrels);
  }
      libGAP_TakeInterrupt();
    }
  if (libGAP_RNheads == 0)
    {
      libGAP_RNheads = libGAP_RNamName("heads");
      libGAP_RNvectors = libGAP_RNamName("vectors");
    }
  res = libGAP_NEW_PREC( TransformationsNeeded ? 4 : 2);
  libGAP_AssPRec(res,libGAP_RNheads,heads);
  libGAP_AssPRec(res,libGAP_RNvectors,vectors);
  if (libGAP_LEN_PLIST(vectors) == 0)
    libGAP_RetypeBag(vectors, libGAP_T_PLIST_EMPTY);
  if (TransformationsNeeded)
    {
      if (libGAP_RNcoeffs == 0)
  {
    libGAP_RNcoeffs = libGAP_RNamName("coeffs");
    libGAP_RNrelns = libGAP_RNamName("relations");
  }
      libGAP_AssPRec(res,libGAP_RNcoeffs,coeffs);
      if (libGAP_LEN_PLIST(coeffs) == 0)
  libGAP_RetypeBag(coeffs, libGAP_T_PLIST_EMPTY);
      libGAP_AssPRec(res,libGAP_RNrelns,relns);
      if (libGAP_LEN_PLIST(relns) == 0)
  libGAP_RetypeBag(relns, libGAP_T_PLIST_EMPTY);
    }
  libGAP_SortPRecRNam(res,0);
  return res;
}


/****************************************************************************
**
*F  UInt TriangulizeListVec8Bits( <mat>, <clearup>, <deterp> ) -- returns the rank
**
*/

libGAP_UInt libGAP_TriangulizeListVec8Bits( libGAP_Obj mat, libGAP_UInt clearup, libGAP_Obj *deterp)
{
  libGAP_UInt nrows;
  libGAP_UInt ncols;
  libGAP_UInt workcol;
  libGAP_UInt workrow;
  libGAP_UInt rank;
  libGAP_Obj row, row2;
  libGAP_UInt byte;
  libGAP_UInt j;
  libGAP_Obj info;
  libGAP_UInt elts;
  libGAP_UInt1 x = 0;
  libGAP_UInt1 *gettab, *getcol;
  libGAP_Obj deter = 0;
  libGAP_UInt sign = 0;
  libGAP_Obj *convtab;
  libGAP_Obj y;
  libGAP_UInt1 x2;

  nrows = libGAP_LEN_PLIST( mat );
  row = libGAP_ELM_PLIST(mat, 1);
  ncols = libGAP_LEN_VEC8BIT( row);
  rank = 0;
  info = libGAP_GetFieldInfo8Bit(libGAP_FIELD_VEC8BIT(row));
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);

  /* Nothing here can cause a garbage collection */

  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  convtab = libGAP_FFE_FELT_FIELDINFO_8BIT(info);

  if (deterp != (libGAP_Obj *) 0)
    {
      deter = libGAP_ONE(convtab[1]);
      sign = 1;
    }
  for (workcol = 1; workcol <= ncols; workcol++)
    {
      byte = (workcol-1)/elts;
      getcol = gettab + 256*((workcol - 1) % elts);
      
      for (workrow = rank+1; workrow <= nrows; workrow ++)
  {
    row = libGAP_ELM_PLIST( mat, workrow);
    x = getcol[libGAP_BYTES_VEC8BIT(row)[byte]];
    if (x)
      break;
  }
      if (workrow <= nrows)
  {
    rank++;
    y = convtab[x];
    libGAP_MultVec8BitFFEInner(row, row, libGAP_INV(y), workcol, ncols);
    if (deterp)
      deter = libGAP_PROD(deter, y);
    
    if (workrow != rank)
      {
        if (deterp)
    sign = -sign;
        libGAP_SET_ELM_PLIST(mat, workrow, libGAP_ELM_PLIST(mat, rank));
        libGAP_SET_ELM_PLIST(mat, rank, row);
      }
    if (clearup)
      for (j = 1; j < rank; j ++)
        {
    row2 = libGAP_ELM_PLIST(mat, j);
    if ((x2 =  getcol[libGAP_BYTES_VEC8BIT(row2)[byte]]))
      libGAP_AddVec8BitVec8BitMultInner(row2,row2,row,libGAP_AINV(convtab[x2]),workcol,ncols);
        }
    for (j = workrow+1; j <= nrows; j++)
      {
        row2 = libGAP_ELM_PLIST(mat, j);
        if ((x2 =  getcol[libGAP_BYTES_VEC8BIT(row2)[byte]]))
    libGAP_AddVec8BitVec8BitMultInner(row2,row2,row,libGAP_AINV(convtab[x2]),workcol,ncols);
      }
  }
      if (libGAP_TakeInterrupt()) {
  gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
  convtab = libGAP_FFE_FELT_FIELDINFO_8BIT(info);
      }
    }
  if (deterp)
    {
      if (rank < nrows)
  deter = libGAP_ZERO(deter);
      else if (sign == -1)
  deter = libGAP_AINV(deter);
      *deterp = deter;
    }
  return rank;
}


/****************************************************************************
**
*F  FuncSEMIECHELON_LIST_VEC8BITS( <self>, <mat> )
**
**  Method for SemiEchelonMat for plain lists of 8 bit vectors
**
** Method selection can guarantee us a plain list of vectors in same char
*/

libGAP_Obj libGAP_FuncSEMIECHELON_LIST_VEC8BITS( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_UInt i,len,width;
  libGAP_Obj row;
  libGAP_UInt q;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_VEC8BIT_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  q = libGAP_FIELD_VEC8BIT(row);
  width = libGAP_LEN_VEC8BIT(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_VEC8BIT_REP(row) || libGAP_FIELD_VEC8BIT(row) != q || libGAP_LEN_VEC8BIT(row) != width)
      {
  return libGAP_TRY_NEXT_METHOD;
      }
    }
  return libGAP_SemiEchelonListVec8Bits( mat, 0);
}

/****************************************************************************
**
*F  FuncSEMIECHELON_LIST_VEC8BITS_TRANSFORMATIONS( <self>, <mat> )
**
**  Method for SemiEchelonMatTransformations for plain lists of 8 bit vectors
**
** Method selection can guarantee us a plain list of vectors in same characteristic
*/

libGAP_Obj libGAP_FuncSEMIECHELON_LIST_VEC8BITS_TRANSFORMATIONS( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_UInt i,len;
  libGAP_Obj row;
  libGAP_UInt q;
  libGAP_UInt width;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_VEC8BIT_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  q = libGAP_FIELD_VEC8BIT(row);
  width = libGAP_LEN_VEC8BIT(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_VEC8BIT_REP(row) ||
    libGAP_FIELD_VEC8BIT(row) != q ||
    libGAP_LEN_VEC8BIT(row) != width)
      {
  return libGAP_TRY_NEXT_METHOD;
      }
    }
  return libGAP_SemiEchelonListVec8Bits( mat, 1);
}


/****************************************************************************
**
*F  FuncTRIANGULIZE_LIST_VEC8BITS( <self>, <mat> )
**
**  Method for TriangulizeMat for plain lists of 8 bit vectors
**
** Method selection can guarantee us a plain list of vectors in same characteristic
*/

libGAP_Obj libGAP_FuncTRIANGULIZE_LIST_VEC8BITS( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_UInt i,len, width;
  libGAP_Obj row;
  libGAP_UInt q;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_VEC8BIT_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  q = libGAP_FIELD_VEC8BIT(row);
  width = libGAP_LEN_VEC8BIT(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_MUTABLE_OBJ(row) || !libGAP_IS_VEC8BIT_REP(row)
    || libGAP_FIELD_VEC8BIT(row) != q || libGAP_LEN_VEC8BIT(row) != width)
      {
  return libGAP_TRY_NEXT_METHOD;
      }
    }
  libGAP_TriangulizeListVec8Bits( mat, 1, (libGAP_Obj *) 0);
  return (libGAP_Obj) 0;
}

/****************************************************************************
**
*F  FuncRANK_LIST_VEC8BITS( <self>, <mat> )
**
**  Method for RankMatDestructive for plain lists of 8 bit vectors
**
** Method selection can guarantee us a plain list of vectors in same characteristic
*/

libGAP_Obj libGAP_FuncRANK_LIST_VEC8BITS( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_UInt i,len, width;
  libGAP_Obj row;
  libGAP_UInt q;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_VEC8BIT_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  q = libGAP_FIELD_VEC8BIT(row);
  width = libGAP_LEN_VEC8BIT(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_VEC8BIT_REP(row) || libGAP_FIELD_VEC8BIT(row) != q
    || libGAP_LEN_VEC8BIT(row) != width)
      {
  return libGAP_TRY_NEXT_METHOD;
      }
    }
  return libGAP_INTOBJ_INT(libGAP_TriangulizeListVec8Bits( mat, 0, (libGAP_Obj *) 0));
}

/****************************************************************************
**
*F  FuncDETERMINANT_LIST_VEC8BITS( <self>, <mat> )
**
**  Method for DeterminantMatDestructive for plain lists of 8 bit vectors
**
** Method selection can guarantee us a plain list of vectors in same characteristic
*/

libGAP_Obj libGAP_FuncDETERMINANT_LIST_VEC8BITS( libGAP_Obj self, libGAP_Obj mat )
{
  libGAP_UInt i,len,width;
  libGAP_Obj row;
  libGAP_UInt q;
  libGAP_Obj det;
  /* check argts */
  len = libGAP_LEN_PLIST(mat);
  if (!len)
    return libGAP_TRY_NEXT_METHOD;
  row = libGAP_ELM_PLIST(mat,1);
  if (!libGAP_IS_VEC8BIT_REP(row))
    return libGAP_TRY_NEXT_METHOD;
  q = libGAP_FIELD_VEC8BIT(row);
  width = libGAP_FIELD_VEC8BIT(row);
  if (width == 0)
    return libGAP_TRY_NEXT_METHOD;
  for (i = 2; i <= len; i++)
    {
      row = libGAP_ELM_PLIST(mat, i);
      if (!libGAP_IS_VEC8BIT_REP(row) || libGAP_FIELD_VEC8BIT(row) != q ||
    libGAP_LEN_VEC8BIT(row) != width)
      {
  return libGAP_TRY_NEXT_METHOD;
      }
    }
  libGAP_TriangulizeListVec8Bits( mat, 0, &det);
  return det;
}




/****************************************************************************
**
*F  Cmp_MAT8BIT_MAT8BIT( <ml>, <mr> )   compare matrices
**
**  Assumes the matrices are over compatible fields
*/

libGAP_Int libGAP_Cmp_MAT8BIT_MAT8BIT( libGAP_Obj ml, libGAP_Obj mr)
{
  libGAP_UInt l1, l2,l, i;
  libGAP_Int c;
  l1 = libGAP_LEN_MAT8BIT(ml);
  l2 = libGAP_LEN_MAT8BIT(mr);
  l = (l1 < l2) ? l1 : l2;
  for (i = 1; i <= l; i++)
    {
      c = libGAP_CmpVec8BitVec8Bit(libGAP_ELM_MAT8BIT(ml,i), libGAP_ELM_MAT8BIT(mr,i));
      if (c != 0)
  return c;
    }
  if (l1 < l2)
    return -1;
  if (l1 > l2)
    return 1;
  return 0;
  
}

/****************************************************************************
**
*F  FuncEQ_MAT8BIT_MAT8BIT( <ml>, <mr> )   compare matrices
*/

libGAP_Obj libGAP_FuncEQ_MAT8BIT_MAT8BIT( libGAP_Obj self, libGAP_Obj ml, libGAP_Obj mr)
{
  if (libGAP_LEN_MAT8BIT(ml) != libGAP_LEN_MAT8BIT(mr))
    return libGAP_False;
  if (libGAP_LEN_MAT8BIT(ml) == 0)
    return libGAP_True;
  if (libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(ml,1)) != libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(mr,1)))
    return libGAP_EqListList(ml,mr) ? libGAP_True : libGAP_False;
  return (0 == libGAP_Cmp_MAT8BIT_MAT8BIT(ml,mr)) ? libGAP_True : libGAP_False;
}

/****************************************************************************
**
*F  FuncLT_MAT8BIT_MAT8BIT( <ml>, <mr> )   compare matrices
*/

libGAP_Obj libGAP_FuncLT_MAT8BIT_MAT8BIT( libGAP_Obj self, libGAP_Obj ml, libGAP_Obj mr)
{
  if (libGAP_LEN_MAT8BIT(ml) == 0)
    return (libGAP_LEN_MAT8BIT(mr) != 0) ? libGAP_True : libGAP_False;
  if (libGAP_LEN_MAT8BIT(mr) == 0)
    return libGAP_False;
  if (libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(ml,1)) != libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(mr,1)))
    return libGAP_LtListList(ml,mr) ? libGAP_True : libGAP_False;
  return (libGAP_Cmp_MAT8BIT_MAT8BIT(ml,mr) < 0) ? libGAP_True : libGAP_False;
}


/****************************************************************************
**
*F  FuncTRANSPOSED_MAT8BIT( <self>, <mat>) Fully mutable results
**
*/
libGAP_Obj libGAP_FuncTRANSPOSED_MAT8BIT( libGAP_Obj self, libGAP_Obj mat)
{
  libGAP_UInt l,w;
  libGAP_Obj tra,row;
  libGAP_Obj r1;
  libGAP_UInt1 vals[libGAP_BIPEB];
  libGAP_UInt val;
  libGAP_UInt imod,nrb,nstart;
  libGAP_UInt i,j,k,n, q, elts;
  libGAP_UInt1 * ptr;
  libGAP_Obj info;
  libGAP_UInt1 *gettab = 0,*settab = 0;
  libGAP_Obj type;

  /* check argument */
  if (libGAP_TNUM_OBJ(mat) != libGAP_T_POSOBJ) {
     mat = libGAP_ErrorReturnObj(
            "TRANSPOSED_MAT8BIT: Need compressed matrix\n",
            0, 0, 
            "You can return such matrix with 'return mat;'\n");
  }
  /* we will give result same type as mat */
  
  /* we assume here that there is a first row  -- a zero row mat8bit is a bad thing*/
  r1 = libGAP_ELM_MAT8BIT(mat,1);
  
  l = libGAP_LEN_MAT8BIT(mat);
  w = libGAP_LEN_VEC8BIT(r1);


  tra = libGAP_NewBag(libGAP_T_POSOBJ, sizeof(libGAP_Obj)*(w+2));
  q = libGAP_FIELD_VEC8BIT(r1);
  type = libGAP_TypeMat8Bit(q,1);
  libGAP_TYPE_POSOBJ(tra) = type;
  
  libGAP_SET_LEN_MAT8BIT(tra, w);

  info = libGAP_GetFieldInfo8Bit(q);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  nrb =  (w+elts-1)/elts;
  
  /* create new matrix */
  for (i = 1; i <= w; i++) {
    row = libGAP_NewBag( libGAP_T_DATOBJ, libGAP_SIZE_VEC8BIT(l, elts));
    libGAP_SET_LEN_VEC8BIT(row,l);
    libGAP_SET_FIELD_VEC8BIT(row,q);
  type = libGAP_TypeVec8BitLocked(q,1);
    libGAP_TYPE_DATOBJ(row) = type;
    libGAP_SET_ELM_MAT8BIT( tra, i, row);
    libGAP_CHANGED_BAG(tra);
  }

  if (elts > 1) {
    gettab = libGAP_GETELT_FIELDINFO_8BIT(info);
    settab = libGAP_SETELT_FIELDINFO_8BIT(info);
  }
  
  /* set entries */
  /* run over elts row chunks of the original matrix */
  for (i = 1; i <= l; i += elts) {
    imod=(i-1)/elts;
    
    /* run through these rows in chunks, extract the bytes corresponding to an
     elts x elts submatrix into vals */
    for (n=0; n<nrb; n++) {
      for (j=0;j<elts;j++) {
  if ((i+j)>l) {
    
    vals[j]=0; /* outside matrix */
    
  }
  else {
    vals[j]=libGAP_BYTES_VEC8BIT(libGAP_ELM_MAT8BIT(mat,i+j))[n];   
  }
      }
      
      /* write transposed values in new matrix */
      nstart=n*elts+1;
      for (j=0;j<elts;j++) { /* bit number = Row in transpose */
  if ((nstart+j)<=w) {
    
    /* still within matrix */
    if (elts > 1)
      {
        val=0;
        for (k=0;k<elts;k++) {
    val = settab[val + 256* ( k + elts*gettab[vals[k] + 256*j])];
        }
      }
    else
      val = vals[0];
    
    /* set entry */
    ptr=libGAP_BYTES_VEC8BIT(libGAP_ELM_GF2MAT(tra,nstart+j))+imod;
    *ptr=val;
  }
      }
    }
  }
  return tra;
}


/****************************************************************************
**
*F  FuncKRONECKERPRODUCT_MAT8BIT_MAT8BIT( <self>, <matl>, <matr>)
**
*/
libGAP_Obj libGAP_FuncKRONECKERPRODUCT_MAT8BIT_MAT8BIT( libGAP_Obj self, libGAP_Obj matl, libGAP_Obj matr)
{
  libGAP_UInt nrowl, nrowr, ncoll, ncolr, ncol, p, q, i, j, k, l, s, zero,
    mutable, elts;
  libGAP_Obj mat, type, row, info, shift[5];
  libGAP_UInt1 *getelt, *setelt, *scalar, *add, *datar, *data;

  nrowl = libGAP_LEN_MAT8BIT(matl);
  nrowr = libGAP_LEN_MAT8BIT(matr);
  ncoll = libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(matl,1));
  ncolr = libGAP_LEN_VEC8BIT(libGAP_ELM_MAT8BIT(matr,1));
  q = libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(matl,1));
  if(q != libGAP_FIELD_VEC8BIT(libGAP_ELM_MAT8BIT(matr,1)))
    return libGAP_TRY_NEXT_METHOD;

  mutable = libGAP_IS_MUTABLE_OBJ(matl) || libGAP_IS_MUTABLE_OBJ(matr);

  info = libGAP_GetFieldInfo8Bit(q);
  p = libGAP_P_FIELDINFO_8BIT(info);
  elts = libGAP_ELS_BYTE_FIELDINFO_8BIT(info);
  zero = libGAP_FELT_FFE_FIELDINFO_8BIT(info)[0];

  /* create a matrix */
  mat = libGAP_NewBag(libGAP_T_POSOBJ, sizeof(libGAP_Obj)*(nrowl*nrowr+2));
  libGAP_SET_LEN_MAT8BIT(mat,nrowl*nrowr);
  libGAP_TYPE_POSOBJ(mat) = libGAP_TypeMat8Bit(q, mutable);
  type = libGAP_TypeVec8BitLocked(q, mutable);

  /* allocate 0 matrix */
  for (i = 1; i <= nrowl*nrowr; i++) {
    row = libGAP_ZeroVec8Bit(q, ncoll*ncolr, mutable);
    libGAP_TYPE_DATOBJ(row) = type; /* locked type */
    libGAP_SET_ELM_MAT8BIT(mat,i,row);
    libGAP_CHANGED_BAG(mat);
  }

  /* allocate data for shifts of rows of matr */
  for (i = 0; i < elts; i++) {
    shift[i] = libGAP_NewBag(libGAP_T_DATOBJ, ncolr/elts+200+sizeof(libGAP_Obj));
  }

  /* allocation is done. speed up operations by getting lookup tables */
  getelt = libGAP_GETELT_FIELDINFO_8BIT(info);
  setelt = libGAP_SETELT_FIELDINFO_8BIT(info);
  scalar = libGAP_SCALAR_FIELDINFO_8BIT(info);
  add = libGAP_ADD_FIELDINFO_8BIT(info);

  /* fill in matrix */
  for (j = 1; j <= nrowr; j++) {
    /* create shifts of rows of matr */
    for (i = 0; i < elts; i++) {
      data = (libGAP_UInt1 *) libGAP_ADDR_OBJ(shift[i]);
      datar = libGAP_BYTES_VEC8BIT(libGAP_ELM_MAT8BIT(matr,j));
      for (k = 0; k < ncolr; k++)
  data[(k+i)/elts] = setelt[data[(k+i)/elts] + 256*((k+i)%elts + getelt[datar[k/elts]+256*(k%elts)]*elts)];
    }
    for (i = 1; i <= nrowl; i++) {
      data = libGAP_BYTES_VEC8BIT(libGAP_ELM_MAT8BIT(mat,(i-1)*nrowr+j));
      ncol = 0;
      for (k = 0; k < ncoll; k++) {
  s = getelt[libGAP_BYTES_VEC8BIT(libGAP_ELM_MAT8BIT(matl,i))[k/elts] + 256*(k%elts)];
  l = 0;
  if (s != zero) {
    /* append s*shift[ncol%elts] to data */
    datar = (libGAP_UInt1 *) libGAP_ADDR_OBJ(shift[ncol%elts]);
    if (ncol % elts) {
      if (p == 2)
        data[-1] ^= scalar[*datar++ + 256*s];
      else
        data[-1] = add[data[-1] + 256*scalar[*datar++ + 256*s]];
      l = elts - ncol%elts;
    }
    for (; l < ncolr; l += elts)
      *data++ = scalar[*datar++ + 256*s];
  } else {
    if (ncol % elts)
      l = elts - ncol%elts;
    data += (ncolr+elts-1-l)/elts;
  }
  ncol += ncolr;
      }
    }
  }

  return mat;
}


/****************************************************************************
**
*f * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * * */




/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "CONV_VEC8BIT", 2, "list,q",
      libGAP_FuncCONV_VEC8BIT, "src/vec8bit.c:CONV_VEC8BIT" },

    { "PLAIN_VEC8BIT", 1, "gfqvec",
      libGAP_FuncPLAIN_VEC8BIT, "src/vec8bit.c:PLAIN_VEC8BIT" },

    { "LEN_VEC8BIT", 1, "gfqvec",
      libGAP_FuncLEN_VEC8BIT, "src/vec8bit.c:LEN_VEC8BIT" },

    { "ELM0_VEC8BIT", 2, "gfqvec, pos",
      libGAP_FuncELM0_VEC8BIT, "src/vec8bit.c:ELM0_VEC8BIT" },

    { "ELM_VEC8BIT", 2, "gfqvec, pos",
      libGAP_FuncELM_VEC8BIT, "src/vec8bit.c:ELM_VEC8BIT" },

    { "ELMS_VEC8BIT", 2, "gfqvec, poss",
      libGAP_FuncELMS_VEC8BIT, "src/vec8bit.c:ELMS_VEC8BIT" },

    { "ELMS_VEC8BIT_RANGE", 2, "gfqvec, range",
      libGAP_FuncELMS_VEC8BIT_RANGE, "src/vec8bit.c:ELMS_VEC8BIT_RANGE" },

    { "ASS_VEC8BIT", 3, "gfqvec, pos, elm",
      libGAP_FuncASS_VEC8BIT, "src/vec8bit.c:ASS_VEC8BIT" },

    { "UNB_VEC8BIT", 2, "gfqvec, pos",
      libGAP_FuncUNB_VEC8BIT, "src/vec8bit.c:UNB_VEC8BIT" },

    { "Q_VEC8BIT", 1, "gfqvec",
      libGAP_FuncQ_VEC8BIT, "src/vec8bit.c:Q_VEC8BIT" },

    { "SHALLOWCOPY_VEC8BIT", 1, "gfqvec",
      libGAP_FuncSHALLOWCOPY_VEC8BIT, "src/vec8bit.c:SHALLOWCOPY_VEC8BIT" },

    { "SUM_VEC8BIT_VEC8BIT", 2, "gfqvecl, gfqvecr",
      libGAP_FuncSUM_VEC8BIT_VEC8BIT, "src/vec8bit.c:SUM_VEC8BIT_VEC8BIT" },

    { "DIFF_VEC8BIT_VEC8BIT", 2, "gfqvecl, gfqvecr",
      libGAP_FuncDIFF_VEC8BIT_VEC8BIT, "src/vec8bit.c:DIFF_VEC8BIT_VEC8BIT" },

    { "PROD_VEC8BIT_FFE", 2, "gfqvec, gfqelt",
      libGAP_FuncPROD_VEC8BIT_FFE, "src/vec8bit.c:PROD_VEC8BIT_FFE" },

    { "PROD_FFE_VEC8BIT", 2, "gfqelt, gfqvec",
      libGAP_FuncPROD_FFE_VEC8BIT, "src/vec8bit.c:PROD_FFE_VEC8BIT" },
    
    { "AINV_VEC8BIT_MUTABLE", 1, "gfqvec",
      libGAP_FuncAINV_VEC8BIT_MUTABLE, "src/vec8bit.c:AINV_VEC8BIT_MUTABLE" },

    { "AINV_VEC8BIT_IMMUTABLE", 1, "gfqvec",
      libGAP_FuncAINV_VEC8BIT_IMMUTABLE, "src/vec8bit.c:AINV_VEC8BIT_IMMUTABLE" },

    { "AINV_VEC8BIT_SAME_MUTABILITY", 1, "gfqvec",
      libGAP_FuncAINV_VEC8BIT_SAME_MUTABILITY, "src/vec8bit.c:AINV_VEC8BIT_SAME_MUTABILITY" },

    { "ZERO_VEC8BIT", 1, "gfqvec",
      libGAP_FuncZERO_VEC8BIT, "src/vec8bit.c:ZERO_VEC8BIT" },
    
    { "ZERO_VEC8BIT_2", 2, "q,len",
      libGAP_FuncZERO_VEC8BIT_2, "src/vec8bit.c:ZERO_VEC8BIT_2" },

    { "EQ_VEC8BIT_VEC8BIT", 2, "gfqvecl, gfqvecr",
      libGAP_FuncEQ_VEC8BIT_VEC8BIT, "src/vec8bit.c:EQ_VEC8BIT_VEC8BIT" },

    { "LT_VEC8BIT_VEC8BIT", 2, "gfqvecl, gfqvecr",
      libGAP_FuncLT_VEC8BIT_VEC8BIT, "src/vec8bit.c:LT_VEC8BIT_VEC8BIT" },

    { "PROD_VEC8BIT_VEC8BIT", 2, "gfqvecl, gfqvecr",
      libGAP_FuncPROD_VEC8BIT_VEC8BIT, "src/vec8bit.c:PROD_VEC8BIT_VEC8BIT" },

    { "DISTANCE_VEC8BIT_VEC8BIT", 2, "gfqvecl, gfqvecr",
      libGAP_FuncDISTANCE_VEC8BIT_VEC8BIT, "src/vec8bit.c:DISTANCE_VEC8BIT_VEC8BIT" },

    {"ADD_ROWVECTOR_VEC8BITS_5", 5, "gfqvecl, gfqvecr, mul, from, to",
      libGAP_FuncADD_ROWVECTOR_VEC8BITS_5, "src/vec8bit.c:ADD_ROWVECTOR_VEC8BITS_5" },

    {"ADD_ROWVECTOR_VEC8BITS_3", 3, "gfqvecl, gfqvecr, mul",
      libGAP_FuncADD_ROWVECTOR_VEC8BITS_3, "src/vec8bit.c:ADD_ROWVECTOR_VEC8BITS_3" },

    {"ADD_ROWVECTOR_VEC8BITS_2", 2, "gfqvecl, gfqvecr",
      libGAP_FuncADD_ROWVECTOR_VEC8BITS_2, "src/vec8bit.c:ADD_ROWVECTOR_VEC8BITS_2" },

    {"MULT_ROWVECTOR_VEC8BITS", 2, "gfqvec, ffe",
      libGAP_FuncMULT_ROWVECTOR_VEC8BITS, "src/vec8bit.c:MULT_ROWVECTOR_VEC8BITS" },

    {"POSITION_NONZERO_VEC8BIT", 2, "vec8bit, zero",
       libGAP_FuncPOSITION_NONZERO_VEC8BIT, "src/vec8bit.c:POSITION_NONZERO_VEC8BIT" },

    {"POSITION_NONZERO_VEC8BIT3", 3, "vec8bit, zero, from",
       libGAP_FuncPOSITION_NONZERO_VEC8BIT3, "src/vec8bit.c:POSITION_NONZERO_VEC8BIT3" },

    {"APPEND_VEC8BIT", 2, "vec8bitl, vec8bitr",
       libGAP_FuncAPPEND_VEC8BIT, "src/vec8bit.c:APPEND_VEC8BIT" },

    {"NUMBER_VEC8BIT", 1, "gfqvec",
       libGAP_FuncNUMBER_VEC8BIT, "src/vec8bit.c:NUMBER_VEC8BIT" },

    {"PROD_VEC8BIT_MATRIX", 2, "gfqvec, mat",
       libGAP_FuncPROD_VEC8BIT_MATRIX, "src/vec8bit.c:PROD_VEC8BIT_MATRIX" },

    {"CONV_MAT8BIT", 2, "list, q",
       libGAP_FuncCONV_MAT8BIT, "src/vec8bit.c:CONV_MAT8BIT" },

    {"PLAIN_MAT8BIT", 1, "mat",
       libGAP_FuncPLAIN_MAT8BIT, "src/vec8bit.c:PLAIN_MAT8BIT" },

    {"PROD_VEC8BIT_MAT8BIT", 2, "vec, mat",
       libGAP_FuncPROD_VEC8BIT_MAT8BIT, "src/vec8bit.c:PROD_VEC8BIT_MAT8BIT" },

    {"PROD_MAT8BIT_VEC8BIT", 2, "mat, vec",
       libGAP_FuncPROD_MAT8BIT_VEC8BIT, "src/vec8bit.c:PROD_MAT8BIT_VEC8BIT" },

    {"PROD_MAT8BIT_MAT8BIT", 2, "matl, matr",
       libGAP_FuncPROD_MAT8BIT_MAT8BIT, "src/vec8bit.c:PROD_MAT8BIT_MAT8BIT" },

    {"INV_MAT8BIT_MUTABLE", 1, "mat",
       libGAP_FuncINV_MAT8BIT_MUTABLE, "src/vec8bit.c:INV_MAT8BIT_MUTABLE" },

    {"INV_MAT8BIT_SAME_MUTABILITY", 1, "mat",
       libGAP_FuncINV_MAT8BIT_SAME_MUTABILITY, "src/vec8bit.c:INV_MAT8BIT_SAME_MUTABILITY" },

    {"INV_MAT8BIT_IMMUTABLE", 1, "mat",
       libGAP_FuncINV_MAT8BIT_IMMUTABLE, "src/vec8bit.c:INV_MAT8BIT_IMMUTABLE" },
    
    {"ASS_MAT8BIT", 3, "mat, pos, obj",
       libGAP_FuncASS_MAT8BIT, "src/vec8bit.c:ASS_MAT8BIT" },

    {"SUM_MAT8BIT_MAT8BIT", 2, "ml, mr",
       libGAP_FuncSUM_MAT8BIT_MAT8BIT, "src/vec8bit.c:SUM_MAT8BIT_MAT8BIT" },
    
    {"DIFF_MAT8BIT_MAT8BIT", 2, "ml, mr",
       libGAP_FuncDIFF_MAT8BIT_MAT8BIT, "src/vec8bit.c:DIFF_MAT8BIT_MAT8BIT" },

    {"ADD_COEFFS_VEC8BIT_3", 3, "vec1, vec2, mult",
       libGAP_FuncADD_COEFFS_VEC8BIT_3, "src/vec8bit.c:ADD_COEFFS_VEC8BIT_3" },

    {"ADD_COEFFS_VEC8BIT_2", 2, "vec1, vec2",
       libGAP_FuncADD_COEFFS_VEC8BIT_2, "src/vec8bit.c:ADD_COEFFS_VEC8BIT_2" },

    {"SHIFT_VEC8BIT_LEFT", 2, "vec, amount",
       libGAP_FuncSHIFT_VEC8BIT_LEFT, "src/vec8bit.c:SHIFT_VEC8BIT_LEFT" },

    {"SHIFT_VEC8BIT_RIGHT", 3, "vec, amount, zero",
       libGAP_FuncSHIFT_VEC8BIT_RIGHT, "src/vec8bit.c:SHIFT_VEC8BIT_RIGHT" },

    {"RESIZE_VEC8BIT", 2, "vec, newsize",
       libGAP_FuncRESIZE_VEC8BIT, "src/vec8bit.c:RESIZE_VEC8BIT" },

    {"RIGHTMOST_NONZERO_VEC8BIT", 1, "vec",
       libGAP_FuncRIGHTMOST_NONZERO_VEC8BIT, "src/vec8bit.c:RIGHTMOST_NONZERO_VEC8BIT" },

    {"PROD_COEFFS_VEC8BIT", 4, "vl, ll, vr, lr",
       libGAP_FuncPROD_COEFFS_VEC8BIT, "src/vec8bit.c:PROD_COEFFS_VEC8BIT" },
    
    {"REDUCE_COEFFS_VEC8BIT", 3, "vl, ll, vrshifted",
       libGAP_FuncREDUCE_COEFFS_VEC8BIT, "src/vec8bit.c:REDUCE_COEFFS_VEC8BIT" },

    {"QUOTREM_COEFFS_VEC8BIT", 3, "vl, ll, vrshifted",
       libGAP_FuncQUOTREM_COEFFS_VEC8BIT, "src/vec8bit.c:QUOTREM_COEFFS_VEC8BIT" },

    {"MAKE_SHIFTED_COEFFS_VEC8BIT", 2, " vr, lr",
       libGAP_FuncMAKE_SHIFTED_COEFFS_VEC8BIT, "src/vec8bit.c:MAKE_SHIFTED_COEFFS_VEC8BIT" },
    
    {"DISTANCE_DISTRIB_VEC8BITS", 3, " veclis, vec, d",
       libGAP_FuncDISTANCE_DISTRIB_VEC8BITS, "src/vec8bit.c:DISTANCE_DISTRIB_VEC8BITS" },
    
    {"A_CLOSEST_VEC8BIT", 4, " veclis, vec, k, stop",
       libGAP_FuncAClosVec8Bits, "src/vec8bit.c:A_CLOSEST_VEC8BIT" },

    {"A_CLOSEST_VEC8BIT_COORDS", 4, " veclis, vec, k, stop",
       libGAP_FuncAClosVec8BitsCoords, "src/vec8bit.c:A_CLOSEST_VEC8BIT_COORDS" },
    
    {"COSET_LEADERS_INNER_8BITS", 5, " veclis, weight, tofind, leaders, felts",
       libGAP_FuncCOSET_LEADERS_INNER_8BITS, "src/vec8bit.c:COSET_LEADERS_INNER_8BITS" },
    
    { "SEMIECHELON_LIST_VEC8BITS", 1, "mat",
      libGAP_FuncSEMIECHELON_LIST_VEC8BITS, "sec/vec8bit.c:SEMIECHELON_LIST_VEC8BITS" },

    { "SEMIECHELON_LIST_VEC8BITS_TRANSFORMATIONS", 1, "mat",
      libGAP_FuncSEMIECHELON_LIST_VEC8BITS_TRANSFORMATIONS, "sec/vec8bit.c:SEMIECHELON_LIST_VEC8BITS_TRANSFORMATIONS" },

    { "TRIANGULIZE_LIST_VEC8BITS", 1, "mat",
      libGAP_FuncTRIANGULIZE_LIST_VEC8BITS, "sec/vec8bit.c:TRIANGULIZE_LIST_VEC8BITS" },
    
    { "RANK_LIST_VEC8BITS", 1, "mat",
      libGAP_FuncRANK_LIST_VEC8BITS, "sec/vec8bit.c:RANK_LIST_VEC8BITS" },
    
    { "DETERMINANT_LIST_VEC8BITS", 1, "mat",
      libGAP_FuncDETERMINANT_LIST_VEC8BITS, "sec/vec8bit.c:DETERMINANT_LIST_VEC8BITS" },

    { "EQ_MAT8BIT_MAT8BIT", 2, "mat8bit, mat8bit",
      libGAP_FuncEQ_MAT8BIT_MAT8BIT, "src/vec8bit.c:EQ_MAT8BIT_MAT8BIT" },

    { "LT_MAT8BIT_MAT8BIT", 2, "mat8bit, mat8bit",
      libGAP_FuncLT_MAT8BIT_MAT8BIT, "src/vec8bit.c:LT_MAT8BIT_MAT8BIT" },

    { "TRANSPOSED_MAT8BIT", 1, "mat8bit",
      libGAP_FuncTRANSPOSED_MAT8BIT, "src/vec8bit.c:TRANSPOSED_MAT8BIT" },

    { "KRONECKERPRODUCT_MAT8BIT_MAT8BIT", 2, "mat8bit, mat8bit",
      libGAP_FuncKRONECKERPRODUCT_MAT8BIT_MAT8BIT, "src/vec8bit.c:KRONECKERPRODUCT_MAT8BIT_MAT8BIT" },

    { 0 }

};


/****************************************************************************
**
*F  PreSave( <module ) . . . . . . discard big recoverable data before saving
**
**  It will get rebuilt automatically, both in the saving workspace and in
** the loaded one and is not endian-safe anyway
*/

static libGAP_Int libGAP_PreSave( libGAP_StructInitInfo * libGAP_module )
{
  libGAP_UInt q;
  for (q = 3; q <= 256; q++)
    libGAP_SET_ELM_PLIST(libGAP_FieldInfo8Bit, q, (libGAP_Obj) 0);

  /* return success */
  return 0;
}

/****************************************************************************
**
*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
  libGAP_RNheads = 0;
  libGAP_RNvectors = 0;
  libGAP_RNcoeffs = 0;
  libGAP_RNrelns = 0;
  
  
  /* import kind functions                                               */
  libGAP_ImportFuncFromLibrary( "TYPE_VEC8BIT",       &libGAP_TYPE_VEC8BIT     );
  libGAP_ImportFuncFromLibrary( "TYPE_VEC8BIT_LOCKED",&libGAP_TYPE_VEC8BIT_LOCKED );
  libGAP_ImportGVarFromLibrary( "TYPES_VEC8BIT",      &libGAP_TYPES_VEC8BIT     );
  libGAP_ImportFuncFromLibrary( "TYPE_MAT8BIT",       &libGAP_TYPE_MAT8BIT     );
  libGAP_ImportGVarFromLibrary( "TYPES_MAT8BIT",      &libGAP_TYPES_MAT8BIT     );
  libGAP_ImportFuncFromLibrary( "Is8BitVectorRep",    &libGAP_IsVec8bitRep       );
  libGAP_ImportGVarFromLibrary( "TYPE_FIELDINFO_8BIT",&libGAP_TYPE_FIELDINFO_8BIT     );
  
  /* init filters and functions                                          */
  libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );
  
  libGAP_InitGlobalBag( &libGAP_FieldInfo8Bit, "src/vec8bit.c:FieldInfo8Bit" );

  libGAP_InitFopyGVar("ConvertToVectorRep", &libGAP_ConvertToVectorRep);
  libGAP_InitFopyGVar("AddRowVector", &libGAP_AddRowVector);
  libGAP_InitFopyGVar("IsLockedRepresentationVector", &libGAP_IsLockedRepresentationVector);
  libGAP_InitFopyGVar("AsInternalFFE", &libGAP_AsInternalFFE);

  
  /* return success                                                      */
  return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{

  libGAP_FieldInfo8Bit = libGAP_NEW_PLIST(libGAP_T_PLIST_NDENSE,257);
  libGAP_SET_ELM_PLIST(libGAP_FieldInfo8Bit,257,libGAP_INTOBJ_INT(1));
  libGAP_SET_LEN_PLIST(libGAP_FieldInfo8Bit,257);
    /* init filters and functions                                          */
  libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );
  
  
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoGF2Vec()  . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "vec8bit",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    libGAP_PreSave,                            /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoVec8bit ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  vec8bit.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
