/****************************************************************************
**
*W  compiler.c                  GAP source                       Frank Celler
*W                                                         & Ferenc Ràkòczi
*W                                                         & Martin Schönert
**
**
*Y  Copyright (C)  1997,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the GAP to C compiler.
*/
#include        <stdarg.h>              /* variable argument list macros   */
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */

#include        "ariths.h"              /* basic arithmetic                */
#include        "integer.h"

#include        "bool.h"                /* booleans                        */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "calls.h"               /* generic call mechanism          */
/*N 1996/06/16 mschoene func expressions should be different from funcs    */

#include        "lists.h"               /* generic lists                   */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */

#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */

#include        "exprs.h"               /* expressions                     */
#include        "stats.h"               /* statements                      */

#include        "vars.h"                /* variables                       */

#include        "compiler.h"            /* compiler                        */


/****************************************************************************
**

*F * * * * * * * * * * * * * compilation flags  * * * * * * * * * * * * * * *
*/


/****************************************************************************
**


*V  CompFastIntArith  . . option to emit code that handles small ints. faster
*/
libGAP_Int libGAP_CompFastIntArith;


/****************************************************************************
**
*V  CompFastPlainLists  . option to emit code that handles plain lists faster
*/
libGAP_Int libGAP_CompFastPlainLists ;


/****************************************************************************
**
*V  CompFastListFuncs . . option to emit code that inlines calls to functions
*/
libGAP_Int libGAP_CompFastListFuncs;


/****************************************************************************
**
*V  CompCheckTypes  . . . . option to emit code that assumes all types are ok.
*/
libGAP_Int libGAP_CompCheckTypes ;


/****************************************************************************
**
*V  CompCheckListElements .  option to emit code that assumes list elms exist
*/
libGAP_Int libGAP_CompCheckListElements;

/****************************************************************************
**
*V  CompOptNames . .  names for all the compiler options passed by gac
**
*/

struct libGAP_CompOptStruc { const libGAP_Char *extname;
  libGAP_Int *variable;
  libGAP_Int val;};

struct libGAP_CompOptStruc libGAP_CompOptNames[] = {
  { "FAST_INT_ARITH", &libGAP_CompFastIntArith, 1 },
  { "FAST_PLAIN_LISTS", &libGAP_CompFastPlainLists, 1 },
  { "FAST_LIST_FUNCS", &libGAP_CompFastListFuncs, 1 },
  { "NO_CHECK_TYPES", &libGAP_CompCheckTypes, 0 },
  { "NO_CHECK_LIST_ELMS", &libGAP_CompCheckListElements, 0 }};

#define libGAP_N_CompOpts  (sizeof(libGAP_CompOptNames)/sizeof(struct libGAP_CompOptStruc))


/****************************************************************************
**
*F  SetCompileOpts( <string> ) . . parse the compiler options from <string>
**                                 and set the appropriate variables
**                                 unrecognised options are ignored for now
*/
#include <ctype.h>

void libGAP_SetCompileOpts( libGAP_Char *opts )
{
  libGAP_Char *s = opts;
  libGAP_Int i;
  while (*s)
    {
      while (libGAP_IsSpace(*s))
        s++;
      for (i = 0; i < libGAP_N_CompOpts; i++)
        {
          if (0 == strncmp(libGAP_CompOptNames[i].extname,
                             s,
                             strlen(libGAP_CompOptNames[i].extname)))
            {
              *(libGAP_CompOptNames[i].variable) = libGAP_CompOptNames[i].val;
              break;
            }
        }
      while (*s && *s != ',')
        s++;
      if (*s == ',')
        s++;
    }
  return;
}

/****************************************************************************
**
*V  CompCheckPosObjElements .  option to emit code that assumes pos elm exist
*/
libGAP_Int libGAP_CompCheckPosObjElements;


/****************************************************************************
**
*V  CompPass  . . . . . . . . . . . . . . . . . . . . . . . . . compiler pass
**
**  'CompPass' holds the number of the current pass.
**
**  The compiler does two passes over the source.
**
**  In the first pass it only collects information but emits no code.
**
**  It finds  out which global  variables and record names  are used, so that
**  the  compiler can output  code to define  and initialize global variables
**  'G_<name>' resp. 'R_<name>' to hold their identifiers.
**
**  It finds out   which arguments and local  variables  are used  as  higher
**  variables  from inside local functions,  so that  the compiler can output
**  code to allocate and manage a stack frame for them.
**
**  It finds out how many temporary variables are used, so that the compiler
**  can output code to define corresponding local variables.
**
**  In the second pass it emits code.
**
**  The only difference between the  first pass and  the second pass is  that
**  'Emit'  emits  no code  during the first  pass.   While  this causes many
**  unneccessary  computations during the first pass,  the  advantage is that
**  the two passes are guaranteed to do exactely the same computations.
*/
libGAP_Int libGAP_CompPass;


/****************************************************************************
**

*F * * * * * * * * * * * * temp, C, local functions * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  compilerMagic1  . . . . . . . . . . . . . . . . . . . . .  current magic1
*/
static libGAP_Int libGAP_compilerMagic1;


/****************************************************************************
**
*V  compilerMagic2  . . . . . . . . . . . . . . . . . . . . .  current magic2
*/
static libGAP_Char * libGAP_compilerMagic2;


/****************************************************************************
**
*T  CVar  . . . . . . . . . . . . . . . . . . . . . . .  type for C variables
**
**  A C variable represents the result of compiling an expression.  There are
**  three cases (distinguished by the least significant two bits).
**
**  If the  expression is an  immediate integer  expression, the  C  variable
**  contains the value of the immediate integer expression.
**
**  If the  expression is an immediate reference  to a  local variable, the C
**  variable contains the index of the local variable.
**
**  Otherwise the expression  compiler emits code  that puts the value of the
**  expression into a  temporary variable,  and  the C variable contains  the
**  index of that temporary variable.
*/
typedef libGAP_UInt           libGAP_CVar;

#define libGAP_IS_INTG_CVAR(c) ((((libGAP_UInt)(c)) & 0x03) == 0x01)
#define libGAP_INTG_CVAR(c)    (((libGAP_Int)(c)) >> 2)
#define libGAP_CVAR_INTG(i)    ((((libGAP_UInt)(i)) << 2) + 0x01)

#define libGAP_IS_TEMP_CVAR(c) ((((libGAP_UInt)(c)) & 0x03) == 0x02)
#define libGAP_TEMP_CVAR(c)    (((libGAP_UInt)(c)) >> 2)
#define libGAP_CVAR_TEMP(l)    ((((libGAP_UInt)(l)) << 2) + 0x02)

#define libGAP_IS_LVAR_CVAR(c) ((((libGAP_UInt)(c)) & 0x03) == 0x03)
#define libGAP_LVAR_CVAR(c)    (((libGAP_UInt)(c)) >> 2)
#define libGAP_CVAR_LVAR(l)    ((((libGAP_UInt)(l)) << 2) + 0x03)


/****************************************************************************
**
*F  SetInfoCVar( <cvar>, <type> ) . . . . . . .  set the type of a C variable
*F  GetInfoCVar( <cvar> ) . . . . . . . . . . .  get the type of a C variable
*F  HasInfoCVar( <cvar>, <type> ) . . . . . . . test the type of a C variable
**
*F  NewInfoCVars()  . . . . . . . . . allocate a new info bag for C variables
*F  CopyInfoCVars( <dst>, <src> ) . .  copy between info bags for C variables
*F  MergeInfoCVars( <dst>, <src> )  . . . merge two info bags for C variables
*F  IsEqInfoCVars( <dst>, <src> ) . . . compare two info bags for C variables
**
**  With each function we  associate a C  variables information bag.  In this
**  bag we store  the number of the  function, the number of local variables,
**  the  number of local  variables that  are used  as higher variables,  the
**  number  of temporaries  used,  the number of  loop  variables needed, the
**  current  number  of used temporaries.
**
**  Furthermore for  each local variable and  temporary we store what we know
**  about this local variable or temporary, i.e., whether the variable has an
**  assigned value, whether that value is an integer, a boolean, etc.
**
**  'SetInfoCVar' sets the    information   for  the  C variable      <cvar>.
**  'GetInfoCVar' gets   the   information  for   the  C    variable  <cvar>.
**  'HasInfoCVar' returns true if the C variable <cvar> has the type <type>.
**
**  'NewInfoCVars'  creates    a    new    C  variables     information  bag.
**  'CopyInfoCVars' copies the C  variables information from <src> to  <dst>.
**  'MergeInfoCVars' merges the C variables information  from <src> to <dst>,
**  i.e., if there are two paths to a  certain place in  the source and <dst>
**  is the information gathered  along one path  and <src> is the information
**  gathered along the other path, then  'MergeInfoCVars' stores in <dst> the
**  information for   that   point  (independent   of  the  path  travelled).
**  'IsEqInfoCVars' returns   true  if <src>    and <dst> contain   the  same
**  information.
**
**  Note that  the numeric  values for the  types  are defined such  that  if
**  <type1> implies <type2>, then <type1> is a bitwise superset of <type2>.
*/
typedef libGAP_UInt4           libGAP_LVar;

#define libGAP_INFO_FEXP(fexp)         libGAP_PROF_FUNC(fexp)
#define libGAP_NEXT_INFO(info)         libGAP_PTR_BAG(info)[0]
#define libGAP_NR_INFO(info)           (*((libGAP_Int*)(libGAP_PTR_BAG(info)+1)))
#define libGAP_NLVAR_INFO(info)        (*((libGAP_Int*)(libGAP_PTR_BAG(info)+2)))
#define libGAP_NHVAR_INFO(info)        (*((libGAP_Int*)(libGAP_PTR_BAG(info)+3)))
#define libGAP_NTEMP_INFO(info)        (*((libGAP_Int*)(libGAP_PTR_BAG(info)+4)))
#define libGAP_NLOOP_INFO(info)        (*((libGAP_Int*)(libGAP_PTR_BAG(info)+5)))
#define libGAP_CTEMP_INFO(info)        (*((libGAP_Int*)(libGAP_PTR_BAG(info)+6)))
#define libGAP_TNUM_LVAR_INFO(info,i)  (*((libGAP_Int*)(libGAP_PTR_BAG(info)+7+(i))))

#define libGAP_TNUM_TEMP_INFO(info,i)  \
    (*((libGAP_Int*)(libGAP_PTR_BAG(info)+7+libGAP_NLVAR_INFO(info)+(i))))

#define libGAP_SIZE_INFO(nlvar,ntemp)  (sizeof(libGAP_Int) * (8 + (nlvar) + (ntemp)))

#define libGAP_W_UNUSED                0       /* TEMP is currently unused        */
#define libGAP_W_HIGHER                (1L<<0) /* LVAR is used as higher variable */
#define libGAP_W_UNKNOWN               ((1L<<1) | libGAP_W_HIGHER)
#define libGAP_W_UNBOUND               ((1L<<2) | libGAP_W_UNKNOWN)
#define libGAP_W_BOUND                 ((1L<<3) | libGAP_W_UNKNOWN)
#define libGAP_W_INT                   ((1L<<4) | libGAP_W_BOUND)
#define libGAP_W_INT_SMALL             ((1L<<5) | libGAP_W_INT)
#define libGAP_W_INT_POS               ((1L<<6) | libGAP_W_INT)
#define libGAP_W_BOOL                  ((1L<<7) | libGAP_W_BOUND)
#define libGAP_W_FUNC                  ((1L<<8) | libGAP_W_BOUND)
#define libGAP_W_LIST                  ((1L<<9) | libGAP_W_BOUND)

#define libGAP_W_INT_SMALL_POS         (libGAP_W_INT_SMALL | libGAP_W_INT_POS)

void            libGAP_SetInfoCVar (
    libGAP_CVar                cvar,
    libGAP_UInt                type )
{
    libGAP_Bag                 info;           /* its info bag                    */

    /* get the information bag                                             */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );

    /* set the type of a temporary                                         */
    if ( libGAP_IS_TEMP_CVAR(cvar) ) {
        libGAP_TNUM_TEMP_INFO( info, libGAP_TEMP_CVAR(cvar) ) = type;
    }

    /* set the type of a lvar (but do not change if its a higher variable) */
    else if ( libGAP_IS_LVAR_CVAR(cvar)
           && libGAP_TNUM_LVAR_INFO( info, libGAP_LVAR_CVAR(cvar) ) != libGAP_W_HIGHER ) {
        libGAP_TNUM_LVAR_INFO( info, libGAP_LVAR_CVAR(cvar) ) = type;
    }
}

libGAP_Int             libGAP_GetInfoCVar (
    libGAP_CVar                cvar )
{
    libGAP_Bag                 info;           /* its info bag                    */

    /* get the information bag                                             */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );

    /* get the type of an integer                                          */
    if ( libGAP_IS_INTG_CVAR(cvar) ) {
        return ((0 < libGAP_INTG_CVAR(cvar)) ? libGAP_W_INT_SMALL_POS : libGAP_W_INT_SMALL);
    }

    /* get the type of a temporary                                         */
    else if ( libGAP_IS_TEMP_CVAR(cvar) ) {
        return libGAP_TNUM_TEMP_INFO( info, libGAP_TEMP_CVAR(cvar) );
    }

    /* get the type of a lvar                                              */
    else if ( libGAP_IS_LVAR_CVAR(cvar) ) {
        return libGAP_TNUM_LVAR_INFO( info, libGAP_LVAR_CVAR(cvar) );
    }

    /* hmm, avoid warning by compiler                                      */
    else {
        return 0;
    }
}

libGAP_Int             libGAP_HasInfoCVar (
    libGAP_CVar                cvar,
    libGAP_Int                 type )
{
    return ((libGAP_GetInfoCVar( cvar ) & type) == type);
}


libGAP_Bag             libGAP_NewInfoCVars ( void )
{
    libGAP_Bag                 old;
    libGAP_Bag                 new;
    old = libGAP_INFO_FEXP( libGAP_CURR_FUNC );
    new = libGAP_NewBag( libGAP_TNUM_BAG(old), libGAP_SIZE_BAG(old) );
    return new;
}

void            libGAP_CopyInfoCVars (
    libGAP_Bag                 dst,
    libGAP_Bag                 src )
{
    libGAP_Int                 i;
    if ( libGAP_SIZE_BAG(dst) < libGAP_SIZE_BAG(src) )  libGAP_ResizeBag( dst, libGAP_SIZE_BAG(src) );
    if ( libGAP_SIZE_BAG(src) < libGAP_SIZE_BAG(dst) )  libGAP_ResizeBag( src, libGAP_SIZE_BAG(dst) );
    libGAP_NR_INFO(dst)    = libGAP_NR_INFO(src);
    libGAP_NLVAR_INFO(dst) = libGAP_NLVAR_INFO(src);
    libGAP_NHVAR_INFO(dst) = libGAP_NHVAR_INFO(src);
    libGAP_NTEMP_INFO(dst) = libGAP_NTEMP_INFO(src);
    libGAP_NLOOP_INFO(dst) = libGAP_NLOOP_INFO(src);
    libGAP_CTEMP_INFO(dst) = libGAP_CTEMP_INFO(src);
    for ( i = 1; i <= libGAP_NLVAR_INFO(src); i++ ) {
        libGAP_TNUM_LVAR_INFO(dst,i) = libGAP_TNUM_LVAR_INFO(src,i);
    }
    for ( i = 1; i <= libGAP_NTEMP_INFO(dst) && i <= libGAP_NTEMP_INFO(src); i++ ) {
        libGAP_TNUM_TEMP_INFO(dst,i) = libGAP_TNUM_TEMP_INFO(src,i);
    }
}

void            libGAP_MergeInfoCVars (
    libGAP_Bag                 dst,
    libGAP_Bag                 src )
{
    libGAP_Int                 i;
    if ( libGAP_SIZE_BAG(dst) < libGAP_SIZE_BAG(src) )  libGAP_ResizeBag( dst, libGAP_SIZE_BAG(src) );
    if ( libGAP_SIZE_BAG(src) < libGAP_SIZE_BAG(dst) )  libGAP_ResizeBag( src, libGAP_SIZE_BAG(dst) );
    if ( libGAP_NTEMP_INFO(dst)<libGAP_NTEMP_INFO(src) )  libGAP_NTEMP_INFO(dst)=libGAP_NTEMP_INFO(src);
    for ( i = 1; i <= libGAP_NLVAR_INFO(src); i++ ) {
        libGAP_TNUM_LVAR_INFO(dst,i) &= libGAP_TNUM_LVAR_INFO(src,i);
    }
    for ( i = 1; i <= libGAP_NTEMP_INFO(dst) && i <= libGAP_NTEMP_INFO(src); i++ ) {
        libGAP_TNUM_TEMP_INFO(dst,i) &= libGAP_TNUM_TEMP_INFO(src,i);
    }
}

libGAP_Int             libGAP_IsEqInfoCVars (
    libGAP_Bag                 dst,
    libGAP_Bag                 src )
{
    libGAP_Int                 i;
    if ( libGAP_SIZE_BAG(dst) < libGAP_SIZE_BAG(src) )  libGAP_ResizeBag( dst, libGAP_SIZE_BAG(src) );
    if ( libGAP_SIZE_BAG(src) < libGAP_SIZE_BAG(dst) )  libGAP_ResizeBag( src, libGAP_SIZE_BAG(dst) );
    for ( i = 1; i <= libGAP_NLVAR_INFO(src); i++ ) {
        if ( libGAP_TNUM_LVAR_INFO(dst,i) != libGAP_TNUM_LVAR_INFO(src,i) ) {
            return 0;
        }
    }
    for ( i = 1; i <= libGAP_NTEMP_INFO(dst) && i <= libGAP_NTEMP_INFO(src); i++ ) {
        if ( libGAP_TNUM_TEMP_INFO(dst,i) != libGAP_TNUM_TEMP_INFO(src,i) ) {
            return 0;
        }
    }
    return 1;
}


/****************************************************************************
**
*F  NewTemp( <name> ) . . . . . . . . . . . . . . .  allocate a new temporary
*F  FreeTemp( <temp> )  . . . . . . . . . . . . . . . . . .  free a temporary
**
**  'NewTemp' allocates  a  new  temporary   variable (<name>  is   currently
**  ignored).
**
**  'FreeTemp' frees the temporary <temp>.
**
**  Currently  allocations and deallocations   of  temporaries are done  in a
**  strict nested (laff -- last allocated, first freed) order.  This means we
**  do not have to search for unused temporaries.
*/
typedef libGAP_UInt4           libGAP_Temp;

libGAP_Temp            libGAP_NewTemp (
    const libGAP_Char *        name )
{
    libGAP_Temp                temp;           /* new temporary, result           */
    libGAP_Bag                 info;           /* information bag                 */

    /* get the information bag                                             */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );

    /* take the next available temporary                                   */
    libGAP_CTEMP_INFO( info )++;
    temp = libGAP_CTEMP_INFO( info );

    /* maybe make room for more temporaries                                */
    if ( libGAP_NTEMP_INFO( info ) < temp ) {
        if ( libGAP_SIZE_BAG(info) < libGAP_SIZE_INFO( libGAP_NLVAR_INFO(info), temp ) ) {
            libGAP_ResizeBag( info, libGAP_SIZE_INFO( libGAP_NLVAR_INFO(info), temp+7 ) );
        }
        libGAP_NTEMP_INFO( info ) = temp;
    }
    libGAP_TNUM_TEMP_INFO( info, temp ) = libGAP_W_UNKNOWN;

    /* return the temporary                                                */
    return temp;
}

void            libGAP_FreeTemp (
    libGAP_Temp                temp )
{
    libGAP_Bag                 info;           /* information bag                 */

    /* get the information bag                                             */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );

    /* check that deallocations happens in the correct order               */
    if ( temp != libGAP_CTEMP_INFO( info ) && libGAP_CompPass == 2 ) {
        libGAP_Pr("PROBLEM: freeing t_%d, should be t_%d\n",(libGAP_Int)temp,libGAP_CTEMP_INFO(info));
    }

    /* free the temporary                                                  */
    libGAP_TNUM_TEMP_INFO( info, temp ) = libGAP_W_UNUSED;
    libGAP_CTEMP_INFO( info )--;
}


/****************************************************************************
**
*F  CompSetUseHVar( <hvar> )  . . . . . . . . register use of higher variable
*F  CompGetUseHVar( <hvar> )  . . . . . . . . get use mode of higher variable
*F  GetLevlHVar( <hvar> ) . . . . . . . . . . .  get level of higher variable
*F  GetIndxHVar( <hvar> ) . . . . . . . . . . .  get index of higher variable
**
**  'CompSetUseHVar'  register (during pass 1)   that the variable <hvar>  is
**  used  as   higher  variable, i.e.,  is  referenced   from inside  a local
**  function.  Such variables  must be allocated  in  a stack frame  bag (and
**  cannot be mapped to C variables).
**
**  'CompGetUseHVar' returns nonzero if the variable <hvar> is used as higher
**  variable.
**
**  'GetLevlHVar' returns the level of the  higher variable <hvar>, i.e., the
**  number of  frames  that must be  walked upwards   for the  one containing
**  <hvar>.  This may be properly  smaller than 'LEVEL_HVAR(<hvar>)', because
**  only those compiled functions that have local variables  that are used as
**  higher variables allocate a stack frame.
**
**  'GetIndxHVar' returns the index of the higher  variable <hvar>, i.e., the
**  position of <hvar> in the stack frame.  This may be properly smaller than
**  'INDEX_HVAR(<hvar>)', because only those  local variable that are used as
**  higher variables are allocated in a stack frame.
*/
typedef libGAP_UInt4           libGAP_HVar;

void            libGAP_CompSetUseHVar (
    libGAP_HVar                hvar )
{
    libGAP_Bag                 info;           /* its info bag                    */
    libGAP_Int                 i;              /* loop variable                   */

    /* only mark in pass 1                                                 */
    if ( libGAP_CompPass != 1 )  return;

    /* walk up                                                             */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );
    for ( i = 1; i <= (hvar >> 16); i++ ) {
        info = libGAP_NEXT_INFO( info );
    }

    /* set mark                                                            */
    if ( libGAP_TNUM_LVAR_INFO( info, (hvar & 0xFFFF) ) != libGAP_W_HIGHER ) {
        libGAP_TNUM_LVAR_INFO( info, (hvar & 0xFFFF) ) = libGAP_W_HIGHER;
        libGAP_NHVAR_INFO(info) = libGAP_NHVAR_INFO(info) + 1;
    }

}

libGAP_Int             libGAP_CompGetUseHVar (
    libGAP_HVar                hvar )
{
    libGAP_Bag                 info;           /* its info bag                    */
    libGAP_Int                 i;              /* loop variable                   */

    /* walk up                                                             */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );
    for ( i = 1; i <= (hvar >> 16); i++ ) {
        info = libGAP_NEXT_INFO( info );
    }

    /* get mark                                                            */
    return (libGAP_TNUM_LVAR_INFO( info, (hvar & 0xFFFF) ) == libGAP_W_HIGHER);
}

libGAP_UInt            libGAP_GetLevlHVar (
    libGAP_HVar                hvar )
{
    libGAP_UInt                levl;           /* level of higher variable        */
    libGAP_Bag                 info;           /* its info bag                    */
    libGAP_Int                 i;              /* loop variable                   */

    /* walk up                                                             */
    levl = 0;
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );
#if 0
    if ( libGAP_NHVAR_INFO(info) != 0 ) 
#endif
      levl++;
    for ( i = 1; i <= (hvar >> 16); i++ ) {
        info = libGAP_NEXT_INFO( info );
#if 0
        if ( libGAP_NHVAR_INFO(info) != 0 ) 
#endif
          levl++;
    }

    /* return level (the number steps to go up)                            */
    return levl - 1;
}

libGAP_UInt            libGAP_GetIndxHVar (
    libGAP_HVar                hvar )
{
    libGAP_UInt                indx;           /* index of higher variable        */
    libGAP_Bag                 info;           /* its info bag                    */
    libGAP_Int                 i;              /* loop variable                   */

    /* walk up                                                             */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );
    for ( i = 1; i <= (hvar >> 16); i++ ) {
        info = libGAP_NEXT_INFO( info );
    }

    /* walk right                                                          */
    indx = 0;
    for ( i = 1; i <= (hvar & 0xFFFF); i++ ) {
        if ( libGAP_TNUM_LVAR_INFO( info, i ) == libGAP_W_HIGHER )  indx++;
    }

    /* return the index                                                    */
    return indx;
}


/****************************************************************************
**
*F  CompSetUseGVar( <gvar>, <mode> )  . . . . register use of global variable
*F  CompGetUseGVar( <gvar> )  . . . . . . . . get use mode of global variable
**
**  'CompSetUseGVar' registers (during pass 1) the use of the global variable
**  with identifier <gvar>.
**
**  'CompGetUseGVar'  returns the bitwise OR  of all the <mode> arguments for
**  the global variable with identifier <gvar>.
**
**  Currently the interpretation of the <mode> argument is as follows
**
**  If '<mode> &  COMP_USE_GVAR_ID' is nonzero, then  the produced code shall
**  define  and initialize 'G_<name>'    with  the identifier of  the  global
**  variable (which may  be different from  <gvar>  by the time the  compiled
**  code is actually run).
**
**  If '<mode> & COMP_USE_GVAR_COPY' is nonzero, then the produced code shall
**  define  and initialize 'GC_<name>' as a  copy of  the the global variable
**  (see 'InitCopyGVar' in 'gvars.h').
**
**  If '<mode> & COMP_USE_GVAR_FOPY' is nonzero, then the produced code shall
**  define and  initialize  'GF_<name>' as   a  function copy  of the  global
**  variable (see 'InitFopyGVar' in 'gvars.h').
*/
typedef libGAP_UInt    libGAP_GVar;

#define libGAP_COMP_USE_GVAR_ID        (1L << 0)
#define libGAP_COMP_USE_GVAR_COPY      (1L << 1)
#define libGAP_COMP_USE_GVAR_FOPY      (1L << 2)

libGAP_Bag             libGAP_CompInfoGVar;

void            libGAP_CompSetUseGVar (
    libGAP_GVar                gvar,
    libGAP_UInt                mode )
{
    /* only mark in pass 1                                                 */
    if ( libGAP_CompPass != 1 )  return;

    /* resize if neccessary                                                */
    if ( libGAP_SIZE_OBJ(libGAP_CompInfoGVar)/sizeof(libGAP_UInt) <= gvar ) {
        libGAP_ResizeBag( libGAP_CompInfoGVar, sizeof(libGAP_UInt)*(gvar+1) );
    }

    /* or with <mode>                                                      */
    ((libGAP_UInt*)libGAP_PTR_BAG(libGAP_CompInfoGVar))[gvar] |= mode;
}

libGAP_UInt            libGAP_CompGetUseGVar (
    libGAP_GVar                gvar )
{
    return ((libGAP_UInt*)libGAP_PTR_BAG(libGAP_CompInfoGVar))[gvar];
}


/****************************************************************************
**
*F  CompSetUseRNam( <rnam>, <mode> )  . . . . . . register use of record name
*F  CompGetUseRNam( <rnam> )  . . . . . . . . . . get use mode of record name
**
**  'CompSetUseRNam' registers  (during pass  1) the use   of the record name
**  with identifier <rnam>.  'CompGetUseRNam'  returns the bitwise OR  of all
**  the <mode> arguments for the global variable with identifier <rnam>.
**
**  Currently the interpretation of the <mode> argument is as follows
**
**  If '<mode> & COMP_USE_RNAM_ID'  is nonzero, then  the produced code shall
**  define and initialize  'R_<name>' with the  identifier of the record name
**  (which may be  different from <rnam> when the  time the  compiled code is
**  actually run).
*/
typedef libGAP_UInt    libGAP_RNam;

#define libGAP_COMP_USE_RNAM_ID        (1L << 0)

libGAP_Bag             libGAP_CompInfoRNam;

void            libGAP_CompSetUseRNam (
    libGAP_RNam                rnam,
    libGAP_UInt                mode )
{
    /* only mark in pass 1                                                 */
    if ( libGAP_CompPass != 1 )  return;

    /* resize if neccessary                                                */
    if ( libGAP_SIZE_OBJ(libGAP_CompInfoRNam)/sizeof(libGAP_UInt) <= rnam ) {
        libGAP_ResizeBag( libGAP_CompInfoRNam, sizeof(libGAP_UInt)*(rnam+1) );
    }

    /* or with <mode>                                                      */
    ((libGAP_UInt*)libGAP_PTR_BAG(libGAP_CompInfoRNam))[rnam] |= mode;
}

libGAP_UInt            libGAP_CompGetUseRNam (
    libGAP_RNam                rnam )
{
    return ((libGAP_UInt*)libGAP_PTR_BAG(libGAP_CompInfoRNam))[rnam];
}


/****************************************************************************
**
*F  Emit( <fmt>, ... )  . . . . . . . . . . . . . . . . . . . . . . emit code
**
**  'Emit' outputs the   string  <fmt> and the  other  arguments,  which must
**  correspond  to the '%'  format elements  in  <fmt>.  Nothing  is actually
**  outputted if 'CompPass' is not 2.
**
**  'Emit'   supports the following   '%'  format elements:  '%d' formats  an
**  integer,   '%s' formats a  string,  '%S' formats a    string with all the
**  necessary escapes, %C does the same  but uses only  valid C escapes, '%n'
**  formats a  name   ('_' is  converted   to '__',  special  characters  are
**  converted to     '_<hex1><hex2>'),    '%c'  formats     a  C     variable
**  ('INTOBJ_INT(<int>)'  for integers,  'a_<name>' for arguments, 'l_<name>'
**  for locals, 't_<nr>' for temporaries), and '%%' outputs a single '%'.
*/
libGAP_Int             libGAP_EmitIndent;

libGAP_Int             libGAP_EmitIndent2;

void            libGAP_Emit (
    const char *        fmt,
    ... )
{
    libGAP_Int                 narg;           /* number of arguments             */
    va_list             ap;             /* argument list pointer           */
    libGAP_Int                 dint;           /* integer argument                */
    libGAP_CVar                cvar;           /* C variable argument             */
    libGAP_Char *              string;         /* string argument                 */
    const libGAP_Char *        p;              /* loop variable                   */
    libGAP_Char *              q;              /* loop variable                   */
    const libGAP_Char *        hex = "0123456789ABCDEF";

    /* are we in pass 2?                                                   */
    if ( libGAP_CompPass != 2 )  return;

    /* get the information bag                                             */
    narg = (libGAP_NARG_FUNC( libGAP_CURR_FUNC ) != -1 ? libGAP_NARG_FUNC( libGAP_CURR_FUNC ) : 1);

    /* loop over the format string                                         */
    va_start( ap, fmt );
    for ( p = fmt; *p != '\0'; p++ ) {

        /* print an indent                                                 */
        if ( 0 < libGAP_EmitIndent2 && *p == '}' ) libGAP_EmitIndent2--;
        while ( 0 < libGAP_EmitIndent2-- )  libGAP_Pr( " ", 0L, 0L );

        /* format an argument                                              */
        if ( *p == '%' ) {
            p++;

            /* emit an integer                                             */
            if ( *p == 'd' ) {
                dint = va_arg( ap, libGAP_Int );
                libGAP_Pr( "%d", dint, 0L );
            }

            /* emit a string                                               */
            else if ( *p == 's' ) {
                string = va_arg( ap, libGAP_Char* );
                libGAP_Pr( "%s", (libGAP_Int)string, 0L );
            }

            /* emit a string                                               */
            else if ( *p == 'S' ) {
                string = va_arg( ap, libGAP_Char* );
                libGAP_Pr( "%S", (libGAP_Int)string, 0L );
            }

            /* emit a string                                               */
            else if ( *p == 'C' ) {
                string = va_arg( ap, libGAP_Char* );
                libGAP_Pr( "%C", (libGAP_Int)string, 0L );
            }

            /* emit a name                                                 */
            else if ( *p == 'n' ) {
                string = va_arg( ap, libGAP_Char* );
                for ( q = string; *q != '\0'; q++ ) {
                    if ( libGAP_IsAlpha(*q) || libGAP_IsDigit(*q) ) {
                        libGAP_Pr( "%c", (libGAP_Int)(*q), 0L );
                    }
                    else if ( *q == '_' ) {
                        libGAP_Pr( "__", 0L, 0L );
                    }
                    else {
                        libGAP_Pr("_%c%c",hex[((libGAP_UInt)*q)/16],hex[((libGAP_UInt)*q)%16]);
                    }
                }
            }

            /* emit a C variable                                           */
            else if ( *p == 'c' ) {
                cvar = va_arg( ap, libGAP_CVar );
                if ( libGAP_IS_INTG_CVAR(cvar) ) {
		  libGAP_Int x = libGAP_INTG_CVAR(cvar);
		  if (x >= -(1L <<28) && x < (1L << 28))
                    libGAP_Pr( "INTOBJ_INT(%d)", x, 0L );
		  else
		    libGAP_Pr( "C_MAKE_MED_INT(%d)", x, 0L );
                }
                else if ( libGAP_IS_TEMP_CVAR(cvar) ) {
                    libGAP_Pr( "t_%d", libGAP_TEMP_CVAR(cvar), 0L );
                }
                else if ( libGAP_LVAR_CVAR(cvar) <= narg ) {
                    libGAP_Emit( "a_%n", libGAP_NAME_LVAR( libGAP_LVAR_CVAR(cvar) ) );
                }
                else {
                    libGAP_Emit( "l_%n", libGAP_NAME_LVAR( libGAP_LVAR_CVAR(cvar) ) );
                }
            }

            /* emit a C variable                                           */
            else if ( *p == 'i' ) {
                cvar = va_arg( ap, libGAP_CVar );
                if ( libGAP_IS_INTG_CVAR(cvar) ) {
                    libGAP_Pr( "%d", libGAP_INTG_CVAR(cvar), 0L );
                }
                else if ( libGAP_IS_TEMP_CVAR(cvar) ) {
                    libGAP_Pr( "INT_INTOBJ(t_%d)", libGAP_TEMP_CVAR(cvar), 0L );
                }
                else if ( libGAP_LVAR_CVAR(cvar) <= narg ) {
                    libGAP_Emit( "INT_INTOBJ(a_%n)", libGAP_NAME_LVAR( libGAP_LVAR_CVAR(cvar) ) );
                }
                else {
                    libGAP_Emit( "INT_INTOBJ(l_%n)", libGAP_NAME_LVAR( libGAP_LVAR_CVAR(cvar) ) );
                }
            }

            /* emit a '%'                                                  */
            else if ( *p == '%' ) {
                libGAP_Pr( "%%", 0L, 0L );
            }

            /* what                                                        */
            else {
                libGAP_Pr( "%%illegal format statement", 0L, 0L );
            }

        }

        else if ( *p == '{' ) {
            libGAP_Pr( "{", 0L, 0L );
            libGAP_EmitIndent++;
        }
        else if ( *p == '}' ) {
            libGAP_Pr( "}", 0L, 0L );
            libGAP_EmitIndent--;
        }
        else if ( *p == '\n' ) {
            libGAP_Pr( "\n", 0L, 0L );
            libGAP_EmitIndent2 = libGAP_EmitIndent;
        }

        else {
            libGAP_Pr( "%c", (libGAP_Int)(*p), 0L );
        }

    }
    va_end( ap );

}


/****************************************************************************
**

*F * * * * * * * * * * * * * * compile checks * * * * * * * * * * * * * * * *
*/


/****************************************************************************
**


*F  CompCheckBound( <obj>, <name> ) emit code to check that <obj> has a value
*/
void libGAP_CompCheckBound (
    libGAP_CVar                obj,
    libGAP_Char *              name )
{
    if ( ! libGAP_HasInfoCVar( obj, libGAP_W_BOUND ) ) {
        if ( libGAP_CompCheckTypes ) {
            libGAP_Emit( "CHECK_BOUND( %c, \"%s\" )\n", obj, name );
        }
        libGAP_SetInfoCVar( obj, libGAP_W_BOUND );
    }
}


/****************************************************************************
**
*F  CompCheckFuncResult( <obj> )  . emit code to check that <obj> has a value
*/
void libGAP_CompCheckFuncResult (
    libGAP_CVar                obj )
{
    if ( ! libGAP_HasInfoCVar( obj, libGAP_W_BOUND ) ) {
        if ( libGAP_CompCheckTypes ) {
            libGAP_Emit( "CHECK_FUNC_RESULT( %c )\n", obj );
        }
        libGAP_SetInfoCVar( obj, libGAP_W_BOUND );
    }
}


/****************************************************************************
**
*F  CompCheckIntSmall( <obj> )   emit code to check that <obj> is a small int
*/
void libGAP_CompCheckIntSmall (
    libGAP_CVar                obj )
{
    if ( ! libGAP_HasInfoCVar( obj, libGAP_W_INT_SMALL ) ) {
        if ( libGAP_CompCheckTypes ) {
            libGAP_Emit( "CHECK_INT_SMALL( %c )\n", obj );
        }
        libGAP_SetInfoCVar( obj, libGAP_W_INT_SMALL );
    }
}



/****************************************************************************
**
*F  CompCheckIntSmallPos( <obj> ) emit code to check that <obj> is a position
*/
void libGAP_CompCheckIntSmallPos (
    libGAP_CVar                obj )
{
    if ( ! libGAP_HasInfoCVar( obj, libGAP_W_INT_SMALL_POS ) ) {
        if ( libGAP_CompCheckTypes ) {
            libGAP_Emit( "CHECK_INT_SMALL_POS( %c )\n", obj );
        }
        libGAP_SetInfoCVar( obj, libGAP_W_INT_SMALL_POS );
    }
}

/****************************************************************************
**
*F  CompCheckIntPos( <obj> ) emit code to check that <obj> is a position
*/
void libGAP_CompCheckIntPos (
    libGAP_CVar                obj )
{
    if ( ! libGAP_HasInfoCVar( obj, libGAP_W_INT_POS ) ) {
        if ( libGAP_CompCheckTypes ) {
            libGAP_Emit( "CHECK_INT_POS( %c )\n", obj );
        }
        libGAP_SetInfoCVar( obj, libGAP_W_INT_POS );
    }
}


/****************************************************************************
**
*F  CompCheckBool( <obj> )  . . .  emit code to check that <obj> is a boolean
*/
void libGAP_CompCheckBool (
    libGAP_CVar                obj )
{
    if ( ! libGAP_HasInfoCVar( obj, libGAP_W_BOOL ) ) {
        if ( libGAP_CompCheckTypes ) {
            libGAP_Emit( "CHECK_BOOL( %c )\n", obj );
        }
        libGAP_SetInfoCVar( obj, libGAP_W_BOOL );
    }
}



/****************************************************************************
**
*F  CompCheckFunc( <obj> )  . . . emit code to check that <obj> is a function
*/
void libGAP_CompCheckFunc (
    libGAP_CVar                obj )
{
    if ( ! libGAP_HasInfoCVar( obj, libGAP_W_FUNC ) ) {
        if ( libGAP_CompCheckTypes ) {
            libGAP_Emit( "CHECK_FUNC( %c )\n", obj );
        }
        libGAP_SetInfoCVar( obj, libGAP_W_FUNC );
    }
}


/****************************************************************************
**

*F * * * * * * * * * * * *  compile expressions * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  CompExpr( <expr> )  . . . . . . . . . . . . . . . . compile an expression
**
**  'CompExpr' compiles the expression <expr> and returns the C variable that
**  will contain the result.
*/
libGAP_CVar (* libGAP_CompExprFuncs[256]) ( libGAP_Expr expr );


libGAP_CVar libGAP_CompExpr (
    libGAP_Expr                expr )
{
    return (* libGAP_CompExprFuncs[ libGAP_TNUM_EXPR(expr) ])( expr );
}


/****************************************************************************
**
*F  CompUnknownExpr( <expr> ) . . . . . . . . . . . .  log unknown expression
*/
libGAP_CVar libGAP_CompUnknownExpr (
    libGAP_Expr                expr )
{
    libGAP_Emit( "CANNOT COMPILE EXPRESSION OF TNUM %d;\n", libGAP_TNUM_EXPR(expr) );
    return 0;
}



/****************************************************************************
**
*F  CompBoolExpr( <expr> )  . . . . . . . compile bool expr and return C bool
*/
libGAP_CVar (* libGAP_CompBoolExprFuncs[256]) ( libGAP_Expr expr );

libGAP_CVar libGAP_CompBoolExpr (
    libGAP_Expr                expr )
{
    return (* libGAP_CompBoolExprFuncs[ libGAP_TNUM_EXPR(expr) ])( expr );
}


/****************************************************************************
**
*F  CompUnknownBool( <expr> ) . . . . . . . . . .  use 'CompExpr' and convert
*/
libGAP_CVar libGAP_CompUnknownBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                res;            /* result                          */
    libGAP_CVar                val;            /* value of expression             */

    /* allocate a new temporary for the result                             */
    res = libGAP_CVAR_TEMP( libGAP_NewTemp( "res" ) );

    /* compile the expression and check that the value is boolean          */
    val = libGAP_CompExpr( expr );
    libGAP_CompCheckBool( val );

    /* emit code to store the C boolean value in the result                */
    libGAP_Emit( "%c = (Obj)(UInt)(%c != False);\n", res, val );

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( res, libGAP_W_BOOL );

    /* free the temporary                                                  */
    if ( libGAP_IS_TEMP_CVAR( val ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( val ) );

    /* return the result                                                   */
    return res;
}
    
/****************************************************************************
**
*V  G_Length  . . . . . . . . . . . . . . . . . . . . . . . function 'Length'
*/
libGAP_GVar libGAP_G_Length;



/****************************************************************************
**
*F  CompFunccall0to6Args( <expr> )  . . . T_FUNCCALL_0ARGS...T_FUNCCALL_6ARGS
*/
extern libGAP_CVar libGAP_CompRefGVarFopy (
            libGAP_Expr                expr );


libGAP_CVar libGAP_CompFunccall0to6Args (
    libGAP_Expr                expr )
{
    libGAP_CVar                result;         /* result, result                  */
    libGAP_CVar                func;           /* function                        */
    libGAP_CVar                args [8];       /* arguments                       */
    libGAP_Int                 narg;           /* number of arguments             */
    libGAP_Int                 i;              /* loop variable                   */

    /* special case to inline 'Length'                                     */
    if ( libGAP_CompFastListFuncs
      && libGAP_TNUM_EXPR( libGAP_FUNC_CALL(expr) ) == libGAP_T_REF_GVAR
      && libGAP_ADDR_EXPR( libGAP_FUNC_CALL(expr) )[0] == libGAP_G_Length
      && libGAP_NARG_SIZE_CALL(libGAP_SIZE_EXPR(expr)) == 1 ) {
        result = libGAP_CVAR_TEMP( libGAP_NewTemp( "result" ) );
        args[1] = libGAP_CompExpr( libGAP_ARGI_CALL(expr,1) );
        if ( libGAP_CompFastPlainLists ) {
            libGAP_Emit( "C_LEN_LIST_FPL( %c, %c )\n", result, args[1] );
        }
        else {
            libGAP_Emit( "C_LEN_LIST( %c, %c )\n", result, args[1] );
        }
        libGAP_SetInfoCVar( result, libGAP_W_INT_SMALL );
        if ( libGAP_IS_TEMP_CVAR( args[1] ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( args[1] ) );
        return result;
    }

    /* allocate a temporary for the result                                 */
    result = libGAP_CVAR_TEMP( libGAP_NewTemp( "result" ) );

    /* compile the reference to the function                               */
    if ( libGAP_TNUM_EXPR( libGAP_FUNC_CALL(expr) ) == libGAP_T_REF_GVAR ) {
        func = libGAP_CompRefGVarFopy( libGAP_FUNC_CALL(expr) );
    }
    else {
        func = libGAP_CompExpr( libGAP_FUNC_CALL(expr) );
        libGAP_CompCheckFunc( func );
    }

    /* compile the argument expressions                                    */
    narg = libGAP_NARG_SIZE_CALL(libGAP_SIZE_EXPR(expr));
    for ( i = 1; i <= narg; i++ ) {
        args[i] = libGAP_CompExpr( libGAP_ARGI_CALL(expr,i) );
    }

    /* emit the code for the procedure call                                */
    libGAP_Emit( "%c = CALL_%dARGS( %c", result, narg, func );
    for ( i = 1; i <= narg; i++ ) {
        libGAP_Emit( ", %c", args[i] );
    }
    libGAP_Emit( " );\n" );

    /* emit code for the check (sets the information for the result)       */
    libGAP_CompCheckFuncResult( result );

    /* free the temporaries                                                */
    for ( i = narg; 1 <= i; i-- ) {
        if ( libGAP_IS_TEMP_CVAR( args[i] ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( args[i] ) );
    }
    if ( libGAP_IS_TEMP_CVAR( func ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( func ) );

    /* return the result                                                   */
    return result;
}


/****************************************************************************
**
*F  CompFunccallXArgs( <expr> ) . . . . . . . . . . . . . .  T_FUNCCALL_XARGS
*/
libGAP_CVar libGAP_CompFunccallXArgs (
    libGAP_Expr                expr )
{
    libGAP_CVar                result;         /* result, result                  */
    libGAP_CVar                func;           /* function                        */
    libGAP_CVar                argl;           /* argument list                   */
    libGAP_CVar                argi;           /* <i>-th argument                 */
    libGAP_UInt                narg;           /* number of arguments             */
    libGAP_UInt                i;              /* loop variable                   */

    /* allocate a temporary for the result                                 */
    result = libGAP_CVAR_TEMP( libGAP_NewTemp( "result" ) );

    /* compile the reference to the function                               */
    if ( libGAP_TNUM_EXPR( libGAP_FUNC_CALL(expr) ) == libGAP_T_REF_GVAR ) {
        func = libGAP_CompRefGVarFopy( libGAP_FUNC_CALL(expr) );
    }
    else {
        func = libGAP_CompExpr( libGAP_FUNC_CALL(expr) );
        libGAP_CompCheckFunc( func );
    }

    /* compile the argument expressions                                    */
    narg = libGAP_NARG_SIZE_CALL(libGAP_SIZE_EXPR(expr));
    argl = libGAP_CVAR_TEMP( libGAP_NewTemp( "argl" ) );
    libGAP_Emit( "%c = NEW_PLIST( T_PLIST, %d );\n", argl, narg );
    libGAP_Emit( "SET_LEN_PLIST( %c, %d );\n", argl, narg );
    for ( i = 1; i <= narg; i++ ) {
        argi = libGAP_CompExpr( libGAP_ARGI_CALL( expr, i ) );
        libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", argl, i, argi );
        if ( ! libGAP_HasInfoCVar( argi, libGAP_W_INT_SMALL ) ) {
            libGAP_Emit( "CHANGED_BAG( %c );\n", argl );
        }
        if ( libGAP_IS_TEMP_CVAR( argi ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( argi ) );
    }

    /* emit the code for the procedure call                                */
    libGAP_Emit( "%c = CALL_XARGS( %c, %c );\n", result, func, argl );

    /* emit code for the check (sets the information for the result)       */
    libGAP_CompCheckFuncResult( result );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( argl ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( argl ) );
    if ( libGAP_IS_TEMP_CVAR( func ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( func ) );

    /* return the result                                                   */
    return result;
}

/****************************************************************************
**
*F  CompFunccallXArgs( <expr> ) . . . . . . . . . . . . . .  T_FUNCCALL_OPTS
*/
libGAP_CVar libGAP_CompFunccallOpts(
                      libGAP_Expr expr)
{
  libGAP_CVar opts = libGAP_CompExpr(libGAP_ADDR_STAT(expr)[0]);
  libGAP_GVar pushOptions;
  libGAP_GVar popOptions;
  libGAP_CVar result;
  pushOptions = libGAP_GVarName("PushOptions");
  popOptions = libGAP_GVarName("PopOptions");
  libGAP_CompSetUseGVar(pushOptions, libGAP_COMP_USE_GVAR_FOPY);
  libGAP_CompSetUseGVar(popOptions, libGAP_COMP_USE_GVAR_FOPY);
  libGAP_Emit("CALL_1ARGS( GF_PushOptions, %c );\n", opts);
  if (libGAP_IS_TEMP_CVAR( opts) ) libGAP_FreeTemp( libGAP_TEMP_CVAR( opts ));
  result = libGAP_CompExpr(libGAP_ADDR_STAT(expr)[1]);
  libGAP_Emit("CALL_0ARGS( GF_PopOptions );\n");
  return result;
}
     

/****************************************************************************
**
*F  CompFuncExpr( <expr> )  . . . . . . . . . . . . . . . . . . . T_FUNC_EXPR
*/
libGAP_CVar libGAP_CompFuncExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                func;           /* function, result                */
    libGAP_CVar                tmp;            /* dummy body                      */

    libGAP_Obj                 fexs;           /* function expressions list       */
    libGAP_Obj                 fexp;           /* function expression             */
    libGAP_Int                 nr;             /* number of the function          */

    /* get the number of the function                                      */
    fexs = libGAP_FEXS_FUNC( libGAP_CURR_FUNC );
    fexp = libGAP_ELM_PLIST( fexs, ((libGAP_Int*)libGAP_ADDR_EXPR(expr))[0] );
    nr   = libGAP_NR_INFO( libGAP_INFO_FEXP( fexp ) );

    /* allocate a new temporary for the function                           */
    func = libGAP_CVAR_TEMP( libGAP_NewTemp( "func" ) );

    /* make the function (all the pieces are in global variables)          */
    libGAP_Emit( "%c = NewFunction( NameFunc[%d], NargFunc[%d], NamsFunc[%d]",
          func, nr, nr, nr );
    libGAP_Emit( ", HdlrFunc%d );\n", nr );

    /* this should probably be done by 'NewFunction'                       */
    libGAP_Emit( "ENVI_FUNC( %c ) = CurrLVars;\n", func );
    tmp = libGAP_CVAR_TEMP( libGAP_NewTemp( "body" ) );
    libGAP_Emit( "%c = NewBag( T_BODY, NUMBER_HEADER_ITEMS_BODY*sizeof(Obj) );\n", tmp );
    libGAP_Emit( "STARTLINE_BODY(%c) = INTOBJ_INT(%d);\n", tmp, libGAP_INT_INTOBJ(libGAP_STARTLINE_BODY(libGAP_BODY_FUNC(fexp))));
    libGAP_Emit( "ENDLINE_BODY(%c) = INTOBJ_INT(%d);\n", tmp, libGAP_INT_INTOBJ(libGAP_ENDLINE_BODY(libGAP_BODY_FUNC(fexp))));
    libGAP_Emit( "FILENAME_BODY(%c) = FileName;\n",tmp);
    libGAP_Emit( "BODY_FUNC(%c) = %c;\n", func, tmp );
    libGAP_FreeTemp( libGAP_TEMP_CVAR( tmp ) );

    libGAP_Emit( "CHANGED_BAG( CurrLVars );\n" );

    /* we know that the result is a function                               */
    libGAP_SetInfoCVar( func, libGAP_W_FUNC );

    /* return the number of the C variable that will hold the function     */
    return func;
}


/****************************************************************************
**
*F  CompOr( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_OR
*/
libGAP_CVar libGAP_CompOr (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* or, result                      */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */
    libGAP_Bag                 only_left;      /* info after evaluating only left */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the left expression                                         */
    left = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Emit( "%c = (%c ? True : False);\n", val, left );
    libGAP_Emit( "if ( %c == False ) {\n", val );
    only_left = libGAP_NewInfoCVars();
    libGAP_CopyInfoCVars( only_left, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

    /* compile the right expression                                        */
    right = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Emit( "%c = (%c ? True : False);\n", val, right );
    libGAP_Emit( "}\n" );

    /* we know that the result is boolean                                  */
    libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), only_left );
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompOrBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_OR
*/
libGAP_CVar libGAP_CompOrBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* or, result                      */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */
    libGAP_Bag                 only_left;      /* info after evaluating only left */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the left expression                                         */
    left = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Emit( "%c = %c;\n", val, left );
    libGAP_Emit( "if ( ! %c ) {\n", val );
    only_left = libGAP_NewInfoCVars();
    libGAP_CopyInfoCVars( only_left, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

    /* compile the right expression                                        */
    right = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Emit( "%c = %c;\n", val, right );
    libGAP_Emit( "}\n" );

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), only_left );
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompAnd( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . . . T_AND
*/
libGAP_CVar libGAP_CompAnd (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right1;         /* right operand 1                 */
    libGAP_CVar                right2;         /* right operand 2                 */
    libGAP_Bag                 only_left;      /* info after evaluating only left */

    /* allocate a temporary for the result                                 */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the left expression                                         */
    left = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    only_left = libGAP_NewInfoCVars();
    libGAP_CopyInfoCVars( only_left, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

    /* emit the code for the case that the left value is 'false'           */
    libGAP_Emit( "if ( %c == False ) {\n", left );
    libGAP_Emit( "%c = %c;\n", val, left );
    libGAP_Emit( "}\n" );

    /* emit the code for the case that the left value is 'true'            */
    libGAP_Emit( "else if ( %c == True ) {\n", left );
    right1 = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_CompCheckBool( right1 );
    libGAP_Emit( "%c = %c;\n", val, right1 );
    libGAP_Emit( "}\n" );

    /* emit the code for the case that the left value is a filter          */
    libGAP_Emit( "else {\n" );
    libGAP_CompCheckFunc( left );
    right2 = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_CompCheckFunc( right2 );
    libGAP_Emit( "%c = NewAndFilter( %c, %c );\n", val, left, right2 );
    libGAP_Emit( "}\n" );

    /* we know precious little about the result                            */
    libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), only_left );
    libGAP_SetInfoCVar( val, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right2 ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right2 ) );
    if ( libGAP_IS_TEMP_CVAR( right1 ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right1 ) );
    if ( libGAP_IS_TEMP_CVAR( left   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left   ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompAndBool( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . T_AND
*/
libGAP_CVar libGAP_CompAndBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* or, result                      */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */
    libGAP_Bag                 only_left;      /* info after evaluating only left */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the left expression                                         */
    left = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[0] );
    libGAP_Emit( "%c = %c;\n", val, left );
    libGAP_Emit( "if ( %c ) {\n", val );
    only_left = libGAP_NewInfoCVars();
    libGAP_CopyInfoCVars( only_left, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

    /* compile the right expression                                        */
    right = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_Emit( "%c = %c;\n", val, right );
    libGAP_Emit( "}\n" );

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), only_left );
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompNot( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . . . T_NOT
*/
libGAP_CVar libGAP_CompNot (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* operand                         */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the operand                                                 */
    left = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* invert the operand                                                  */
    libGAP_Emit( "%c = (%c ? False : True);\n", val, left );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( left ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompNotBoot( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . T_NOT
*/
libGAP_CVar libGAP_CompNotBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* operand                         */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the operand                                                 */
    left = libGAP_CompBoolExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* invert the operand                                                  */
    libGAP_Emit( "%c = (Obj)(UInt)( ! ((Int)%c) );\n", val, left );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( left ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompEq( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_EQ
*/
libGAP_CVar libGAP_CompEq (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
libGAP_Emit("%c = ((((Int)%c) == ((Int)%c)) ? True : False);\n", val, left, right);
    }
    else {
        libGAP_Emit( "%c = (EQ( %c, %c ) ? True : False);\n", val, left, right );
    }

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompEqBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_EQ
*/
libGAP_CVar libGAP_CompEqBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "%c = (Obj)(UInt)(((Int)%c) == ((Int)%c));\n", val, left, right);
    }
    else {
        libGAP_Emit( "%c = (Obj)(UInt)(EQ( %c, %c ));\n", val, left, right );
    }

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompNe( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_NE
*/
libGAP_CVar libGAP_CompNe (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
libGAP_Emit("%c = ((((Int)%c) == ((Int)%c)) ? False : True);\n", val, left, right);
    }
    else {
        libGAP_Emit( "%c = (EQ( %c, %c ) ? False : True);\n", val, left, right );
    }

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompNeBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_NE
*/
libGAP_CVar libGAP_CompNeBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "%c = (Obj)(UInt)(((Int)%c) != ((Int)%c));\n", val, left, right );
    }
    else {
        libGAP_Emit( "%c = (Obj)(UInt)( ! EQ( %c, %c ));\n", val, left, right );
    }

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompLt( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_LT
*/
libGAP_CVar libGAP_CompLt (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
libGAP_Emit( "%c = ((((Int)%c) < ((Int)%c)) ? True : False);\n", val, left, right );
    }
    else {
        libGAP_Emit( "%c = (LT( %c, %c ) ? True : False);\n", val, left, right );
    }

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompLtBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_LT
*/
libGAP_CVar libGAP_CompLtBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "%c = (Obj)(UInt)(((Int)%c) < ((Int)%c));\n", val, left, right );
    }
    else {
        libGAP_Emit( "%c = (Obj)(UInt)(LT( %c, %c ));\n", val, left, right );
    }

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompGe( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_GE
*/
libGAP_CVar libGAP_CompGe (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
 libGAP_Emit("%c = ((((Int)%c) < ((Int)%c)) ? False : True);\n", val, left, right);
    }
    else {
        libGAP_Emit( "%c = (LT( %c, %c ) ? False : True);\n", val, left, right );
    }

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompGeBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_GE
*/
libGAP_CVar libGAP_CompGeBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "%c = (Obj)(UInt)(((Int)%c) >= ((Int)%c));\n", val, left, right );
    }
    else {
        libGAP_Emit( "%c = (Obj)(UInt)(! LT( %c, %c ));\n", val, left, right );
    }

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompGt( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_GT
*/
libGAP_CVar libGAP_CompGt (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
 libGAP_Emit("%c = ((((Int)%c) < ((Int)%c)) ? True : False);\n", val, right, left);
    }
    else {
        libGAP_Emit( "%c = (LT( %c, %c ) ? True : False);\n", val, right, left );
    }

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompGtBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_GT
*/
libGAP_CVar libGAP_CompGtBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "%c = (Obj)(UInt)(((Int)%c) < ((Int)%c));\n", val, right, left );
    }
    else {
        libGAP_Emit( "%c = (Obj)(UInt)(LT( %c, %c ));\n", val, right, left );
    }

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompLe( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_LE
*/
libGAP_CVar libGAP_CompLe (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
libGAP_Emit("%c = ((((Int)%c) < ((Int)%c)) ?  False : True);\n", val, right, left);
    }
    else {
        libGAP_Emit( "%c = (LT( %c, %c ) ?  False : True);\n", val, right, left );
    }

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompLeBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_LE
*/
libGAP_CVar            libGAP_CompLeBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "%c = (Obj)(UInt)(((Int)%c) >= ((Int)%c));\n", val, right, left );
    }
    else {
        libGAP_Emit( "%c = (Obj)(UInt)(! LT( %c, %c ));\n", val, right, left );
    }

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompIn( <expr> )  . . . . . . . . . . . . . . . . . . . . . . . . .  T_IN
*/
libGAP_CVar libGAP_CompIn (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    libGAP_Emit( "%c = (IN( %c, %c ) ?  True : False);\n", val, left, right );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompInBool( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_IN
*/
libGAP_CVar libGAP_CompInBool (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    libGAP_Emit( "%c = (Obj)(UInt)(IN( %c, %c ));\n", val, left, right );

    /* we know that the result is boolean (should be 'W_CBOOL')            */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompSum( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . . . T_SUM
*/
libGAP_CVar libGAP_CompSum (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "C_SUM_INTOBJS( %c, %c, %c )\n", val, left, right );
    }
    else if ( libGAP_CompFastIntArith ) {
        libGAP_Emit( "C_SUM_FIA( %c, %c, %c )\n", val, left, right );
    }
    else {
        libGAP_Emit( "C_SUM( %c, %c, %c )\n", val, left, right );
    }

    /* set the information for the result                                  */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT) && libGAP_HasInfoCVar(right,libGAP_W_INT) ) {
        libGAP_SetInfoCVar( val, libGAP_W_INT );
    }
    else {
        libGAP_SetInfoCVar( val, libGAP_W_BOUND );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompAInv( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_AINV
*/
libGAP_CVar libGAP_CompAInv (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the operands                                                */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "C_AINV_INTOBJS( %c, %c )\n", val, left );
    }
    else if ( libGAP_CompFastIntArith ) {
        libGAP_Emit( "C_AINV_FIA( %c, %c )\n", val, left );
    }
    else {
        libGAP_Emit( "C_AINV( %c, %c )\n", val, left );
    }

    /* set the information for the result                                  */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT) ) {
        libGAP_SetInfoCVar( val, libGAP_W_INT );
    }
    else {
        libGAP_SetInfoCVar( val, libGAP_W_BOUND );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompDiff( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_DIFF
*/
libGAP_CVar libGAP_CompDiff (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "C_DIFF_INTOBJS( %c, %c, %c )\n", val, left, right );
    }
    else if ( libGAP_CompFastIntArith ) {
        libGAP_Emit( "C_DIFF_FIA( %c, %c, %c )\n", val, left, right );
    }
    else {
        libGAP_Emit( "C_DIFF( %c, %c, %c )\n", val, left, right );
    }

    /* set the information for the result                                  */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT) && libGAP_HasInfoCVar(right,libGAP_W_INT) ) {
        libGAP_SetInfoCVar( val, libGAP_W_INT );
    }
    else {
        libGAP_SetInfoCVar( val, libGAP_W_BOUND );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompProd( <expr> )  . . . . . . . . . . . . . . . . . . . . . . .  T_PROD
*/
libGAP_CVar libGAP_CompProd (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT_SMALL) && libGAP_HasInfoCVar(right,libGAP_W_INT_SMALL) ) {
        libGAP_Emit( "C_PROD_INTOBJS( %c, %c, %c )\n", val, left, right );
    }
    else if ( libGAP_CompFastIntArith ) {
        libGAP_Emit( "C_PROD_FIA( %c, %c, %c )\n", val, left, right );
    }
    else {
        libGAP_Emit( "C_PROD( %c, %c, %c )\n", val, left, right );
    }

    /* set the information for the result                                  */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT) && libGAP_HasInfoCVar(right,libGAP_W_INT) ) {
        libGAP_SetInfoCVar( val, libGAP_W_INT );
    }
    else {
        libGAP_SetInfoCVar( val, libGAP_W_BOUND );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompInv( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . . . T_INV
**
** C_INV is not defined, so I guess this never gets called SL
**
*/
libGAP_CVar libGAP_CompInv (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the operands                                                */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* emit the code                                                       */
    libGAP_Emit( "C_INV( %c, %c )\n", val, left );

    /* set the information for the result                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompQuo( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . . . T_QUO
*/
libGAP_CVar libGAP_CompQuo (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    libGAP_Emit( "%c = QUO( %c, %c );\n", val, left, right );

    /* set the information for the result                                  */
    libGAP_SetInfoCVar( val, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompMod( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . . . T_MOD
*/
libGAP_CVar libGAP_CompMod (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    libGAP_Emit( "%c = MOD( %c, %c );\n", val, left, right );

    /* set the information for the result                                  */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT) && libGAP_HasInfoCVar(right,libGAP_W_INT) ) {
        libGAP_SetInfoCVar( val, libGAP_W_INT );
    }
    else {
        libGAP_SetInfoCVar( val, libGAP_W_BOUND );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompPow( <expr> ) . . . . . . . . . . . . . . . . . . . . . . . . . T_POW
*/
libGAP_CVar libGAP_CompPow (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */
    libGAP_CVar                left;           /* left operand                    */
    libGAP_CVar                right;          /* right operand                   */

    /* allocate a new temporary for the result                             */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* compile the two operands                                            */
    left  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
    right = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code                                                       */
    libGAP_Emit( "%c = POW( %c, %c );\n", val, left, right );

    /* set the information for the result                                  */
    if ( libGAP_HasInfoCVar(left,libGAP_W_INT) && libGAP_HasInfoCVar(right,libGAP_W_INT) ) {
        libGAP_SetInfoCVar( val, libGAP_W_INT );
    }
    else {
        libGAP_SetInfoCVar( val, libGAP_W_BOUND );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( right ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( right ) );
    if ( libGAP_IS_TEMP_CVAR( left  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( left  ) );

    /* return the result                                                   */
    return val;
}


/****************************************************************************
**
*F  CompIntExpr( <expr> ) . . . . . . . . . . . . . . .  T_INTEXPR/T_INT_EXPR
*
* This is complicated by the need to produce code that will compile correctly
* in 32 or 64 bit and with or without GMP.
*
* The problem is that when we compile the code, we know the integer representation
* of the stored literal in the compiling process
* but NOT the representation which will apply to the compiled code or the endianness
*
* The solution to this is macros: C_MAKE_INTEGER_BAG( size, type) 
*                                 C_SET_LIMB2(bag, limbnumber, value)
*                                 C_SET_LIMB4(bag, limbnumber, value)
*                                 C_SET_LIMB8(bag, limbnumber, value)
*
* we compile using the one appropriate for the compiling system, but their
* definition depends on the limb size of the target system.
*
*/

libGAP_CVar libGAP_CompIntExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;
    libGAP_Int                 siz;
    libGAP_Int                 i;
    libGAP_UInt                typ;

    if ( libGAP_IS_INTEXPR(expr) ) {
        return libGAP_CVAR_INTG( libGAP_INT_INTEXPR(expr) );
    }
    else {
        val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );
        siz = libGAP_SIZE_EXPR(expr) - sizeof(libGAP_UInt);
	typ = *(libGAP_UInt *)libGAP_ADDR_EXPR(expr);
	libGAP_Emit( "%c = C_MAKE_INTEGER_BAG(%d, %d);\n",val, siz, typ);
        if ( typ == libGAP_T_INTPOS ) {
            libGAP_SetInfoCVar(val, libGAP_W_INT_POS);
        }
        else {
            libGAP_SetInfoCVar(val, libGAP_W_INT);
	}

        for ( i = 0; i < siz/libGAP_INTEGER_UNIT_SIZE; i++ ) {
#if libGAP_INTEGER_UNIT_SIZE == 2
	    libGAP_Emit( "C_SET_LIMB2( %c, %d, %d);\n",val, i, ((libGAP_UInt2 *)((libGAP_UInt *)libGAP_ADDR_EXPR(expr) + 1))[i]);
#else
#if libGAP_INTEGER_UNIT_SIZE == 4
	    libGAP_Emit( "C_SET_LIMB4( %c, %d, %dL);\n",val, i, ((libGAP_UInt4 *)((libGAP_UInt *)libGAP_ADDR_EXPR(expr) + 1))[i]);
#else
	    libGAP_Emit( "C_SET_LIMB8( %c, %d, %dLL);\n",val, i, ((libGAP_UInt8*)((libGAP_UInt *)libGAP_ADDR_EXPR(expr) + 1))[i]);
#endif
#endif
        }
	if (siz <= 8)
	  libGAP_Emit("%c = C_NORMALIZE_64BIT(%c);\n", val,val);
        return val;
    }
}


/****************************************************************************
**
*F  CompTrueExpr( <expr> )  . . . . . . . . . . . . . . . . . . . T_TRUE_EXPR
*/
libGAP_CVar libGAP_CompTrueExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* value, result                   */

    /* allocate a new temporary for the 'true' value                       */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* emit the code                                                       */
    libGAP_Emit( "%c = True;\n", val );

    /* we know that the result is boolean ;-)                              */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* return 'true'                                                       */
    return val;
}


/****************************************************************************
**
*F  CompFalseExpr( <expr> ) . . . . . . . . . . . . . . . . . .  T_FALSE_EXPR
*/
libGAP_CVar libGAP_CompFalseExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* value, result                   */

    /* allocate a new temporary for the 'false' value                      */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* emit the code                                                       */
    libGAP_Emit( "%c = False;\n", val );

    /* we know that the result is boolean ;-)                              */
    libGAP_SetInfoCVar( val, libGAP_W_BOOL );

    /* return 'false'                                                      */
    return val;
}


/****************************************************************************
**
*F  CompCharExpr( <expr> )  . . . . . . . . . . . . . . . . . . . T_CHAR_EXPR
*/
libGAP_CVar            libGAP_CompCharExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* result                          */

    /* allocate a new temporary for the char value                         */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* emit the code                                                       */
    libGAP_Emit( "%c = ObjsChar[%d];\n", val, (libGAP_Int)(((libGAP_UChar*)libGAP_ADDR_EXPR(expr))[0]));

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( val, libGAP_W_BOUND );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  CompPermExpr( <expr> )  . . . . . . . . . . . . . . . . . . . T_PERM_EXPR
*/
libGAP_CVar libGAP_CompPermExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                perm;           /* result                          */
    libGAP_CVar                lcyc;           /* one cycle as list               */
    libGAP_CVar                lprm;           /* perm as list of list cycles     */
    libGAP_CVar                val;            /* one point                       */
    libGAP_Int                 i;
    libGAP_Int                 j;
    libGAP_Int                 n;
    libGAP_Int                 csize;
    libGAP_Expr                cycle;

    /* check for the identity                                              */
    if ( libGAP_SIZE_EXPR(expr) == 0 ) {
        perm = libGAP_CVAR_TEMP( libGAP_NewTemp( "idperm" ) );
        libGAP_Emit( "%c = IdentityPerm;\n", perm );
        libGAP_SetInfoCVar( perm, libGAP_W_BOUND );
        return perm;
    }

    /* for each cycle create a list                                        */
    perm = libGAP_CVAR_TEMP( libGAP_NewTemp( "perm" ) );
    lcyc = libGAP_CVAR_TEMP( libGAP_NewTemp( "lcyc" ) );
    lprm = libGAP_CVAR_TEMP( libGAP_NewTemp( "lprm" ) );

    /* start with the identity permutation                                 */
    libGAP_Emit( "%c = IdentityPerm;\n", perm );

    /* loop over the cycles                                                */
    n = libGAP_SIZE_EXPR(expr)/sizeof(libGAP_Expr);
    libGAP_Emit( "%c = NEW_PLIST( T_PLIST, %d );\n", lprm, n );
    libGAP_Emit( "SET_LEN_PLIST( %c, %d );\n", lprm, n );

    for ( i = 1;  i <= n;  i++ ) {
        cycle = libGAP_ADDR_EXPR(expr)[i-1];
        csize = libGAP_SIZE_EXPR(cycle)/sizeof(libGAP_Expr);
        libGAP_Emit( "%c = NEW_PLIST( T_PLIST, %d );\n", lcyc, csize );
        libGAP_Emit( "SET_LEN_PLIST( %c, %d );\n", lcyc, csize );
        libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", lprm, i, lcyc );
        libGAP_Emit( "CHANGED_BAG( %c );\n", lprm );

        /* loop over the entries of the cycle                              */
        for ( j = 1;  j <= csize;  j++ ) {
            val = libGAP_CompExpr( libGAP_ADDR_EXPR(cycle)[j-1] );
            libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", lcyc, j, val );
            libGAP_Emit( "CHANGED_BAG( %c );\n", lcyc );
            if ( libGAP_IS_TEMP_CVAR(val) )  libGAP_FreeTemp( libGAP_TEMP_CVAR(val) );
        }
    }
    libGAP_Emit( "%c = Array2Perm( %c );\n", perm, lprm );

    /* free the termporaries                                               */
    libGAP_FreeTemp( libGAP_TEMP_CVAR(lprm) );
    libGAP_FreeTemp( libGAP_TEMP_CVAR(lcyc) );

    return perm;
}


/****************************************************************************
**
*F  CompListExpr( <expr> )  . . . . . . . . . . . . . . . . . . . T_LIST_EXPR
*/
extern libGAP_CVar libGAP_CompListExpr1 ( libGAP_Expr expr );
extern void libGAP_CompListExpr2 ( libGAP_CVar list, libGAP_Expr expr );
extern libGAP_CVar libGAP_CompRecExpr1 ( libGAP_Expr expr );
extern void libGAP_CompRecExpr2 ( libGAP_CVar rec, libGAP_Expr expr );

libGAP_CVar libGAP_CompListExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                list;           /* list, result                    */

    /* compile the list expression                                         */
    list = libGAP_CompListExpr1( expr );
    libGAP_CompListExpr2( list, expr );

    /* return the result                                                   */
    return list;
}


/****************************************************************************
**
*F  CompListTildeExpr( <expr> ) . . . . . . . . . . . . . .  T_LIST_TILD_EXPR
*/
libGAP_CVar libGAP_CompListTildeExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                list;           /* list value, result              */
    libGAP_CVar                tilde;          /* old value of tilde              */

    /* remember the old value of '~'                                       */
    tilde = libGAP_CVAR_TEMP( libGAP_NewTemp( "tilde" ) );
    libGAP_Emit( "%c = VAL_GVAR( Tilde );\n", tilde );

    /* create the list value                                               */
    list = libGAP_CompListExpr1( expr );

    /* assign the list to '~'                                              */
    libGAP_Emit( "AssGVar( Tilde, %c );\n", list );

    /* evaluate the subexpressions into the list value                     */
    libGAP_CompListExpr2( list, expr );

    /* restore old value of '~'                                            */
    libGAP_Emit( "AssGVar( Tilde, %c );\n", tilde );
    if ( libGAP_IS_TEMP_CVAR( tilde ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( tilde ) );

    /* return the list value                                               */
    return list;
}


/****************************************************************************
**
*F  CompListExpr1( <expr> ) . . . . . . . . . . . . . . . . . . . . . . local
*/
libGAP_CVar libGAP_CompListExpr1 (
    libGAP_Expr                expr )
{
    libGAP_CVar                list;           /* list, result                    */
    libGAP_Int                 len;            /* logical length of the list      */

    /* get the length of the list                                          */
    len = libGAP_SIZE_EXPR( expr ) / sizeof(libGAP_Expr);

    /* allocate a temporary for the list                                   */
    list = libGAP_CVAR_TEMP( libGAP_NewTemp( "list" ) );

    /* emit the code to make the list                                      */
    libGAP_Emit( "%c = NEW_PLIST( T_PLIST, %d );\n", list, len );
    libGAP_Emit( "SET_LEN_PLIST( %c, %d );\n", list, len );

    /* we know that <list> is a list                                       */
    libGAP_SetInfoCVar( list, libGAP_W_LIST );

    /* return the list                                                     */
    return list;
}


/****************************************************************************
**
*F  CompListExpr2( <list>, <expr> ) . . . . . . . . . . . . . . . . . . local
*/
void libGAP_CompListExpr2 (
    libGAP_CVar                list,
    libGAP_Expr                expr )
{
    libGAP_CVar                sub;            /* subexpression                   */
    libGAP_Int                 len;            /* logical length of the list      */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length of the list                                          */
    len = libGAP_SIZE_EXPR( expr ) / sizeof(libGAP_Expr);

    /* emit the code to fill the list                                      */
    for ( i = 1; i <= len; i++ ) {

        /* if the subexpression is empty                                   */
        if ( libGAP_ADDR_EXPR(expr)[i-1] == 0 ) {
            continue;
        }

        /* special case if subexpression is a list expression              */
        else if ( libGAP_TNUM_EXPR( libGAP_ADDR_EXPR(expr)[i-1] ) == libGAP_T_LIST_EXPR ) {
            sub = libGAP_CompListExpr1( libGAP_ADDR_EXPR(expr)[i-1] );
            libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", list, i, sub );
            libGAP_Emit( "CHANGED_BAG( %c );\n", list );
            libGAP_CompListExpr2( sub, libGAP_ADDR_EXPR(expr)[i-1] );
            if ( libGAP_IS_TEMP_CVAR( sub ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( sub ) );
        }

        /* special case if subexpression is a record expression            */
        else if ( libGAP_TNUM_EXPR( libGAP_ADDR_EXPR(expr)[i-1] ) == libGAP_T_REC_EXPR ) {
            sub = libGAP_CompRecExpr1( libGAP_ADDR_EXPR(expr)[i-1] );
            libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", list, i, sub );
            libGAP_Emit( "CHANGED_BAG( %c );\n", list );
            libGAP_CompRecExpr2( sub, libGAP_ADDR_EXPR(expr)[i-1] );
            if ( libGAP_IS_TEMP_CVAR( sub ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( sub ) );
        }

        /* general case                                                    */
        else {
            sub = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[i-1] );
            libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", list, i, sub );
            if ( ! libGAP_HasInfoCVar( sub, libGAP_W_INT_SMALL ) ) {
                libGAP_Emit( "CHANGED_BAG( %c );\n", list );
            }
            if ( libGAP_IS_TEMP_CVAR( sub ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( sub ) );
        }

    }

}


/****************************************************************************
**
*F  CompRangeExpr( <expr> ) . . . . . . . . . . . . . . . . . .  T_RANGE_EXPR
*/
libGAP_CVar libGAP_CompRangeExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                range;          /* range, result                   */
    libGAP_CVar                first;          /* first  element                  */
    libGAP_CVar                second;         /* second element                  */
    libGAP_CVar                last;           /* last   element                  */

    /* allocate a new temporary for the range                              */
    range = libGAP_CVAR_TEMP( libGAP_NewTemp( "range" ) );

    /* evaluate the expressions                                            */
    if ( libGAP_SIZE_EXPR(expr) == 2 * sizeof(libGAP_Expr) ) {
        first  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
        second = 0;
        last   = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    }
    else {
        first  = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );
        second = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
        last   = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[2] );
    }

    /* emit the code                                                       */
    if ( libGAP_SIZE_EXPR(expr) == 2 * sizeof(libGAP_Expr) ) {
        libGAP_Emit( "%c = Range2Check( %c, %c );\n",
              range, first, last );
    }
    else {
        libGAP_Emit( "%c = Range3Check( %c, %c, %c );\n",
              range, first, second, last );
    }

    /* we know that the result is a list                                   */
    libGAP_SetInfoCVar( range, libGAP_W_LIST );

    /* free the temporaries                                                */
    if ( libGAP_SIZE_EXPR(expr) == 2 * sizeof(libGAP_Expr) ) {
        if ( libGAP_IS_TEMP_CVAR( last   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( last   ) );
        if ( libGAP_IS_TEMP_CVAR( first  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( first  ) );
    }
    else {
        if ( libGAP_IS_TEMP_CVAR( last   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( last   ) );
        if ( libGAP_IS_TEMP_CVAR( second ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( second ) );
        if ( libGAP_IS_TEMP_CVAR( first  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( first  ) );
    }

    /* return the range                                                    */
    return range;
}


/****************************************************************************
**
*F  CompStringExpr( <expr> )  . . . . . . . . . . compile a string expression
*/
libGAP_CVar libGAP_CompStringExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                string;         /* string value, result            */

    /* allocate a new temporary for the string                             */
    string = libGAP_CVAR_TEMP( libGAP_NewTemp( "string" ) );

    /* create the string and copy the stuff                                */
    libGAP_Emit( "C_NEW_STRING( %c, %d, \"%C\" )\n",

          /* the sizeof(UInt) offset is to get past the length of the string
             which is now stored in the front of the literal */
          string, libGAP_SIZE_EXPR(expr)-1-sizeof(libGAP_UInt),
          sizeof(libGAP_UInt)+ (libGAP_Char*)libGAP_ADDR_EXPR(expr) );

    /* we know that the result is a list                                   */
    libGAP_SetInfoCVar( string, libGAP_W_LIST );

    /* return the string                                                   */
    return string;
}


/****************************************************************************
**
*F  CompRecExpr( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_REC_EXPR
*/
libGAP_CVar libGAP_CompRecExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                rec;            /* record value, result            */

    /* compile the record expression                                       */
    rec = libGAP_CompRecExpr1( expr );
    libGAP_CompRecExpr2( rec, expr );

    /* return the result                                                   */
    return rec;
}


/****************************************************************************
**
*F  CompRecTildeExpr( <expr> )  . . . . . . . . . . . . . . . T_REC_TILD_EXPR
*/
libGAP_CVar libGAP_CompRecTildeExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                rec;            /* record value, result            */
    libGAP_CVar                tilde;          /* old value of tilde              */

    /* remember the old value of '~'                                       */
    tilde = libGAP_CVAR_TEMP( libGAP_NewTemp( "tilde" ) );
    libGAP_Emit( "%c = VAL_GVAR( Tilde );\n", tilde );

    /* create the record value                                             */
    rec = libGAP_CompRecExpr1( expr );

    /* assign the record value to the variable '~'                         */
    libGAP_Emit( "AssGVar( Tilde, %c );\n", rec );

    /* evaluate the subexpressions into the record value                   */
    libGAP_CompRecExpr2( rec, expr );

    /* restore the old value of '~'                                        */
    libGAP_Emit( "AssGVar( Tilde, %c );\n", tilde );
    if ( libGAP_IS_TEMP_CVAR( tilde ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( tilde ) );

    /* return the record value                                             */
    return rec;
}


/****************************************************************************
**
*F  CompRecExpr1( <expr> )  . . . . . . . . . . . . . . . . . . . . . . local
*/
libGAP_CVar libGAP_CompRecExpr1 (
    libGAP_Expr                expr )
{
    libGAP_CVar                rec;            /* record value, result            */
    libGAP_Int                 len;            /* number of components            */

    /* get the number of components                                        */
    len = libGAP_SIZE_EXPR( expr ) / (2*sizeof(libGAP_Expr));

    /* allocate a new temporary for the record                             */
    rec = libGAP_CVAR_TEMP( libGAP_NewTemp( "rec" ) );

    /* emit the code to allocate the new record object                     */
    libGAP_Emit( "%c = NEW_PREC( %d );\n", rec, len );

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( rec, libGAP_W_BOUND );

    /* return the record                                                   */
    return rec;
}


/****************************************************************************
**
*F  CompRecExpr2( <rec>, <expr> ) . . . . . . . . . . . . . . . . . . . local
*/
void            libGAP_CompRecExpr2 (
    libGAP_CVar                rec,
    libGAP_Expr                expr )
{
    libGAP_CVar                rnam;           /* name of component               */
    libGAP_CVar                sub;            /* value of subexpression          */
    libGAP_Int                 len;            /* number of components            */
    libGAP_Expr                tmp;            /* temporary variable              */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the number of components                                        */
    len = libGAP_SIZE_EXPR( expr ) / (2*sizeof(libGAP_Expr));

    /* handle the subexpressions                                           */
    for ( i = 1; i <= len; i++ ) {

        /* handle the name                                                 */
        tmp = libGAP_ADDR_EXPR(expr)[2*i-2];
        rnam = libGAP_CVAR_TEMP( libGAP_NewTemp( "rnam" ) );
        if ( libGAP_IS_INTEXPR(tmp) ) {
            libGAP_CompSetUseRNam( (libGAP_UInt)libGAP_INT_INTEXPR(tmp), libGAP_COMP_USE_RNAM_ID );
            libGAP_Emit( "%c = (Obj)R_%n;\n",
                  rnam, libGAP_NAME_RNAM((libGAP_UInt)libGAP_INT_INTEXPR(tmp)) );
        }
        else {
            sub = libGAP_CompExpr( tmp );
            libGAP_Emit( "%c = (Obj)RNamObj( %c );\n", rnam, sub );
        }

        /* if the subexpression is empty (cannot happen for records)       */
        tmp = libGAP_ADDR_EXPR(expr)[2*i-1];
        if ( tmp == 0 ) {
            if ( libGAP_IS_TEMP_CVAR( rnam ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam ) );
            continue;
        }

        /* special case if subexpression is a list expression             */
        else if ( libGAP_TNUM_EXPR( tmp ) == libGAP_T_LIST_EXPR ) {
            sub = libGAP_CompListExpr1( tmp );
            libGAP_Emit( "AssPRec( %c, (UInt)%c, %c );\n", rec, rnam, sub );
            libGAP_CompListExpr2( sub, tmp );
            if ( libGAP_IS_TEMP_CVAR( sub ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( sub ) );
        }

        /* special case if subexpression is a record expression            */
        else if ( libGAP_TNUM_EXPR( tmp ) == libGAP_T_REC_EXPR ) {
            sub = libGAP_CompRecExpr1( tmp );
            libGAP_Emit( "AssPRec( %c, (UInt)%c, %c );\n", rec, rnam, sub );
            libGAP_CompRecExpr2( sub, tmp );
            if ( libGAP_IS_TEMP_CVAR( sub ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( sub ) );
        }

        /* general case                                                    */
        else {
            sub = libGAP_CompExpr( tmp );
            libGAP_Emit( "AssPRec( %c, (UInt)%c, %c );\n", rec, rnam, sub );
            if ( libGAP_IS_TEMP_CVAR( sub ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( sub ) );
        }

        if ( libGAP_IS_TEMP_CVAR( rnam ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam ) );
    }
    libGAP_Emit( "SortPRecRNam( %c, 0 );\n", rec );

}


/****************************************************************************
**
*F  CompRefLVar( <expr> ) . . . . . . .  T_REFLVAR/T_REF_LVAR...T_REF_LVAR_16
*/
libGAP_CVar libGAP_CompRefLVar (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* value, result                   */
    libGAP_LVar                lvar;           /* local variable                  */

    /* get the local variable                                              */
    if ( libGAP_IS_REFLVAR(expr) ) {
        lvar = libGAP_LVAR_REFLVAR(expr);
    }
    else {
        lvar = (libGAP_LVar)(libGAP_ADDR_EXPR(expr)[0]);
    }

    /* emit the code to get the value                                      */
    if ( libGAP_CompGetUseHVar( lvar ) ) {
        val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );
        libGAP_Emit( "%c = OBJ_LVAR( %d );\n", val, libGAP_GetIndxHVar(lvar) );
    }
    else {
        val = libGAP_CVAR_LVAR(lvar);
    }

    /* emit code to check that the variable has a value                    */
    libGAP_CompCheckBound( val, libGAP_NAME_LVAR(lvar) );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  CompIsbLVar( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_ISB_LVAR
*/
libGAP_CVar libGAP_CompIsbLVar (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                val;            /* value                           */
    libGAP_LVar                lvar;           /* local variable                  */

    /* get the local variable                                              */
    lvar = (libGAP_LVar)(libGAP_ADDR_EXPR(expr)[0]);

    /* allocate a new temporary for the result                             */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* emit the code to get the value                                      */
    if ( libGAP_CompGetUseHVar( lvar ) ) {
        val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );
        libGAP_Emit( "%c = OBJ_LVAR( %d );\n", val, libGAP_GetIndxHVar(lvar) );
    }
    else {
        val = libGAP_CVAR_LVAR(lvar);
    }

    /* emit the code to check that the variable has a value                */
    libGAP_Emit( "%c = ((%c != 0) ? True : False);\n", isb, val );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( val ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( val ) );

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  CompRefHVar( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_REF_HVAR
*/
libGAP_CVar libGAP_CompRefHVar (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* value, result                   */
    libGAP_HVar                hvar;           /* higher variable                 */

    /* get the higher variable                                             */
    hvar = (libGAP_HVar)(libGAP_ADDR_EXPR(expr)[0]);
    libGAP_CompSetUseHVar( hvar );

    /* allocate a new temporary for the value                              */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* emit the code to get the value                                      */
    libGAP_Emit( "%c = OBJ_LVAR_%dUP( %d );\n",
          val, libGAP_GetLevlHVar(hvar), libGAP_GetIndxHVar(hvar) );

    /* emit the code to check that the variable has a value                */
    libGAP_CompCheckBound( val, libGAP_NAME_HVAR(hvar) );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  CompIsbHVar( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_ISB_HVAR
*/
libGAP_CVar libGAP_CompIsbHVar (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                val;            /* value                           */
    libGAP_HVar                hvar;           /* higher variable                 */

    /* get the higher variable                                             */
    hvar = (libGAP_HVar)(libGAP_ADDR_EXPR(expr)[0]);
    libGAP_CompSetUseHVar( hvar );

    /* allocate new temporaries for the value and the result               */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* emit the code to get the value                                      */
    libGAP_Emit( "%c = OBJ_LVAR_%dUP( %d );\n",
          val, libGAP_GetLevlHVar(hvar), libGAP_GetIndxHVar(hvar) );

    /* emit the code to check that the variable has a value                */
    libGAP_Emit( "%c = ((%c != 0) ? True : False);\n", isb, val );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( val ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( val ) );

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  CompRefGVar( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_REF_GVAR
*/
libGAP_CVar libGAP_CompRefGVar (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* value, result                   */
    libGAP_GVar                gvar;           /* higher variable                 */

    /* get the global variable                                             */
    gvar = (libGAP_GVar)(libGAP_ADDR_EXPR(expr)[0]);
    libGAP_CompSetUseGVar( gvar, libGAP_COMP_USE_GVAR_COPY );

    /* allocate a new global variable for the value                        */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* emit the code to get the value                                      */
    libGAP_Emit( "%c = GC_%n;\n", val, libGAP_NameGVar(gvar) );

    /* emit the code to check that the variable has a value                */
    libGAP_CompCheckBound( val, libGAP_NameGVar(gvar) );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  CompRefGVarFopy( <expr> ) . . . . . . . . . . . . . . . . . . . . . local
*/
libGAP_CVar libGAP_CompRefGVarFopy (
    libGAP_Expr                expr )
{
    libGAP_CVar                val;            /* value, result                   */
    libGAP_GVar                gvar;           /* higher variable                 */

    /* get the global variable                                             */
    gvar = (libGAP_GVar)(libGAP_ADDR_EXPR(expr)[0]);
    libGAP_CompSetUseGVar( gvar, libGAP_COMP_USE_GVAR_FOPY );

    /* allocate a new temporary for the value                              */
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* emit the code to get the value                                      */
    libGAP_Emit( "%c = GF_%n;\n", val, libGAP_NameGVar(gvar) );

    /* we know that the object in a function copy is a function            */
    libGAP_SetInfoCVar( val, libGAP_W_FUNC );

    /* return the value                                                    */
    return val;
}


/****************************************************************************
**
*F  CompIsbGVar( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_ISB_GVAR
*/
libGAP_CVar libGAP_CompIsbGVar (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                val;            /* value, result                   */
    libGAP_GVar                gvar;           /* higher variable                 */

    /* get the global variable                                             */
    gvar = (libGAP_GVar)(libGAP_ADDR_EXPR(expr)[0]);
    libGAP_CompSetUseGVar( gvar, libGAP_COMP_USE_GVAR_COPY );

    /* allocate new temporaries for the value and the result               */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );
    val = libGAP_CVAR_TEMP( libGAP_NewTemp( "val" ) );

    /* emit the code to get the value                                      */
    libGAP_Emit( "%c = GC_%n;\n", val, libGAP_NameGVar(gvar) );

    /* emit the code to check that the variable has a value                */
    libGAP_Emit( "%c = ((%c != 0) ? True : False);\n", isb, val );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( val ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( val ) );

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  CompElmList( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_ELM_LIST
*/
libGAP_CVar libGAP_CompElmList (
    libGAP_Expr                expr )
{
    libGAP_CVar                elm;            /* element, result                 */
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                pos;            /* position                        */

    /* allocate a new temporary for the element                            */
    elm = libGAP_CVAR_TEMP( libGAP_NewTemp( "elm" ) );

    /* compile the list expression (checking is done by 'ELM_LIST')        */
    list = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_CompCheckIntPos( pos );

    /* emit the code to get the element                                    */
    if (        libGAP_CompCheckListElements &&   libGAP_CompFastPlainLists ) {
        libGAP_Emit( "C_ELM_LIST_FPL( %c, %c, %c )\n", elm, list, pos );
    }
    else if (   libGAP_CompCheckListElements && ! libGAP_CompFastPlainLists ) {
        libGAP_Emit( "C_ELM_LIST( %c, %c, %c );\n", elm, list, pos );
    }
    else if ( ! libGAP_CompCheckListElements &&   libGAP_CompFastPlainLists ) {
        libGAP_Emit( "C_ELM_LIST_NLE_FPL( %c, %c, %c );\n", elm, list, pos );
    }
    else {
        libGAP_Emit( "C_ELM_LIST_NLE( %c, %c, %c );\n", elm, list, pos );
    }

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( elm, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  CompElmsList( <expr> )  . . . . . . . . . . . . . . . . . . . T_ELMS_LIST
*/
libGAP_CVar libGAP_CompElmsList (
    libGAP_Expr                expr )
{
    libGAP_CVar                elms;           /* elements, result                */
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                poss;           /* positions                       */

    /* allocate a new temporary for the elements                           */
    elms = libGAP_CVAR_TEMP( libGAP_NewTemp( "elms" ) );

    /* compile the list expression (checking is done by 'ElmsListCheck')   */
    list = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile the position expression (checking done by 'ElmsListCheck')  */
    poss = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code to get the element                                    */
    libGAP_Emit( "%c = ElmsListCheck( %c, %c );\n", elms, list, poss );

    /* we know that the elements are a list                                */
    libGAP_SetInfoCVar( elms, libGAP_W_LIST );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( poss ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( poss ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );

    /* return the elements                                                 */
    return elms;
}


/****************************************************************************
**
*F  CompElmListLev( <expr> )  . . . . . . . . . . . . . . . .  T_ELM_LIST_LEV
*/
libGAP_CVar libGAP_CompElmListLev (
    libGAP_Expr                expr )
{
    libGAP_CVar                lists;          /* lists                           */
    libGAP_CVar                pos;            /* position                        */
    libGAP_Int                 level;          /* level                           */

    /* compile the lists expression                                        */
    lists = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_CompCheckIntSmallPos( pos );

    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_EXPR(expr)[2]);

    /* emit the code to select the elements from several lists (to <lists>)*/
    libGAP_Emit( "ElmListLevel( %c, %c, %d );\n", lists, pos, level );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( pos   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos   ) );

    /* return the lists                                                    */
    return lists;
}


/****************************************************************************
**
*F  CompElmsListLev( <expr> ) . . . . . . . . . . . . . . . . T_ELMS_LIST_LEV
*/
libGAP_CVar libGAP_CompElmsListLev (
    libGAP_Expr                expr )
{
    libGAP_CVar                lists;          /* lists                           */
    libGAP_CVar                poss;           /* positions                       */
    libGAP_Int                 level;          /* level                           */

    /* compile the lists expression                                        */
    lists = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile the position expression (checking done by 'ElmsListLevel')  */
    poss = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_EXPR(expr)[2]);

    /* emit the code to select the elements from several lists (to <lists>)*/
    libGAP_Emit( "ElmsListLevelCheck( %c, %c, %d );\n", lists, poss, level );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( poss  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( poss  ) );

    /* return the lists                                                    */
    return lists;
}


/****************************************************************************
**
*F  CompIsbList( <expr> ) . . . . . . . . . . . . . . . . . . . .  T_ISB_LIST
*/
libGAP_CVar libGAP_CompIsbList (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                pos;            /* position                        */

    /* allocate a new temporary for the result                             */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* compile the list expression (checking is done by 'ISB_LIST')        */
    list = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_CompCheckIntPos( pos );

    /* emit the code to test the element                                   */
    libGAP_Emit( "%c = C_ISB_LIST( %c, %c );\n", isb, list, pos );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );

    /* return the element                                                  */
    return isb;
}


/****************************************************************************
**
*F  CompElmRecName( <expr> )  . . . . . . . . . . . . . . . .  T_ELM_REC_NAME
*/
libGAP_CVar libGAP_CompElmRecName (
    libGAP_Expr                expr )
{
    libGAP_CVar                elm;            /* element, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the element                            */
    elm = libGAP_CVAR_TEMP( libGAP_NewTemp( "elm" ) );

    /* compile the record expression (checking is done by 'ELM_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* emit the code to select the element of the record                   */
    libGAP_Emit( "%c = ELM_REC( %c, R_%n );\n", elm, record, libGAP_NAME_RNAM(rnam) );

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( elm, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  CompElmRecExpr( <expr> )  . . . . . . . . . . . . . . . .  T_ELM_REC_EXPR
*/
libGAP_CVar libGAP_CompElmRecExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                elm;            /* element, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_CVar                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the element                            */
    elm = libGAP_CVAR_TEMP( libGAP_NewTemp( "elm" ) );

    /* compile the record expression (checking is done by 'ELM_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile the record name expression                                  */
    rnam = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code to select the element of the record                   */
    libGAP_Emit( "%c = ELM_REC( %c, RNamObj(%c) );\n", elm, record, rnam );

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( elm, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  CompIsbRecName( <expr> )  . . . . . . . . . . . . . . . .  T_ISB_REC_NAME
*/
libGAP_CVar libGAP_CompIsbRecName (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the result                             */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* compile the record expression (checking is done by 'ISB_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* emit the code to test the element                                   */
    libGAP_Emit( "%c = (ISB_REC( %c, R_%n ) ? True : False);\n",
          isb, record, libGAP_NAME_RNAM(rnam) );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  CompIsbRecExpr( <expr> )  . . . . . . . . . . . . . . . .  T_ISB_REC_EXPR
*/
libGAP_CVar libGAP_CompIsbRecExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_CVar                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the result                             */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* compile the record expression (checking is done by 'ISB_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile the record name expression                                  */
    rnam = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code to test the element                                   */
    libGAP_Emit( "%c = (ISB_REC( %c, RNamObj(%c) ) ? True : False);\n",
          isb, record, rnam );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  CompElmPosObj( <expr> ) . . . . . . . . . . . . . . . . . .  T_ELM_POSOBJ
*/
libGAP_CVar libGAP_CompElmPosObj (
    libGAP_Expr                expr )
{
    libGAP_CVar                elm;            /* element, result                 */
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                pos;            /* position                        */

    /* allocate a new temporary for the element                            */
    elm = libGAP_CVAR_TEMP( libGAP_NewTemp( "elm" ) );

    /* compile the list expression (checking is done by 'ELM_LIST')        */
    list = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_CompCheckIntSmallPos( pos );

    /* emit the code to get the element                                    */
    if (        libGAP_CompCheckPosObjElements ) {
        libGAP_Emit( "C_ELM_POSOBJ( %c, %c, %i )\n", elm, list, pos );
    }
    else if ( ! libGAP_CompCheckPosObjElements ) {
        libGAP_Emit( "C_ELM_POSOBJ_NLE( %c, %c, %i );\n", elm, list, pos );
    }

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( elm, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  CompElmsPosObj( <expr> )  . . . . . . . . . . . . . . . . . T_ELMS_POSOBJ
*/
libGAP_CVar libGAP_CompElmsPosObj (
    libGAP_Expr                expr )
{
    libGAP_Emit( "CANNOT COMPILE EXPRESSION OF TNUM %d;\n", libGAP_TNUM_EXPR(expr) );
    return 0;
}


/****************************************************************************
**
*F  CompElmPosObjLev( <expr> )  . . . . . . . . . . . . . .  T_ELM_POSOBJ_LEV
*/
libGAP_CVar libGAP_CompElmPosObjLev (
    libGAP_Expr                expr )
{
    libGAP_Emit( "CANNOT COMPILE EXPRESSION OF TNUM %d;\n", libGAP_TNUM_EXPR(expr) );
    return 0;
}


/****************************************************************************
**
*F  CompElmsPosObjLev( <expr> ) . . . . . . . . . . . . . . . . T_ELMS_POSOBJ
*/
libGAP_CVar libGAP_CompElmsPosObjLev (
    libGAP_Expr                expr )
{
    libGAP_Emit( "CANNOT COMPILE EXPRESSION OF TNUM %d;\n", libGAP_TNUM_EXPR(expr) );
    return 0;
}


/****************************************************************************
**
*F  CompIsbPosObj( <expr> ) . . . . . . . . . . . . . . . . . .  T_ISB_POSOBJ
*/
libGAP_CVar libGAP_CompIsbPosObj (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                pos;            /* position                        */

    /* allocate a new temporary for the result                             */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* compile the list expression (checking is done by 'ISB_LIST')        */
    list = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );
    libGAP_CompCheckIntSmallPos( pos );

    /* emit the code to test the element                                   */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_POSOBJ ) {\n", list );
    libGAP_Emit( "%c = (%i <= SIZE_OBJ(%c)/sizeof(Obj)-1\n", isb, pos, list );
    libGAP_Emit( "   && ELM_PLIST(%c,%i) != 0 ? True : False);\n", list, pos );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "%c = (ISB_LIST( %c, %i ) ? True : False);\n", isb, list, pos );
    libGAP_Emit( "}\n" );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );

    /* return the element                                                  */
    return isb;
}


/****************************************************************************
**
*F  CompElmObjName( <expr> )  . . . . . . . . . . . . . . . T_ELM_COMOBJ_NAME
*/
libGAP_CVar libGAP_CompElmComObjName (
    libGAP_Expr                expr )
{
    libGAP_CVar                elm;            /* element, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the element                            */
    elm = libGAP_CVAR_TEMP( libGAP_NewTemp( "elm" ) );

    /* compile the record expression (checking is done by 'ELM_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* emit the code to select the element of the record                   */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "%c = ElmPRec( %c, R_%n );\n", elm, record, libGAP_NAME_RNAM(rnam) );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "%c = ELM_REC( %c, R_%n );\n", elm, record, libGAP_NAME_RNAM(rnam) );
    libGAP_Emit( "}\n" );

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( elm, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the element                                                  */
    return elm;
}



/****************************************************************************
**
*F  CompElmComObjExpr( <expr> ) . . . . . . . . . . . . . . T_ELM_COMOBJ_EXPR
*/
libGAP_CVar libGAP_CompElmComObjExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                elm;            /* element, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_CVar                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the element                            */
    elm = libGAP_CVAR_TEMP( libGAP_NewTemp( "elm" ) );

    /* compile the record expression (checking is done by 'ELM_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code to select the element of the record                   */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "%c = ElmPRec( %c, RNamObj(%c) );\n", elm, record, rnam );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "%c = ELM_REC( %c, RNamObj(%c) );\n", elm, record, rnam );
    libGAP_Emit( "}\n" );

    /* we know that we have a value                                        */
    libGAP_SetInfoCVar( elm, libGAP_W_BOUND );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the element                                                  */
    return elm;
}


/****************************************************************************
**
*F  CompIsbComObjName( <expr> ) . . . . . . . . . . . . . . T_ISB_COMOBJ_NAME
*/
libGAP_CVar libGAP_CompIsbComObjName (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the result                             */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* compile the record expression (checking is done by 'ISB_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = (libGAP_UInt)(libGAP_ADDR_EXPR(expr)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* emit the code to test the element                                   */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "%c = (IsbPRec( %c, R_%n ) ? True : False);\n",
          isb, record, libGAP_NAME_RNAM(rnam) );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "%c = (ISB_REC( %c, R_%n ) ? True : False);\n",
          isb, record, libGAP_NAME_RNAM(rnam) );
    libGAP_Emit( "}\n" );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**
*F  CompIsbComObjExpr( <expr> ) . . . . . . . . . . . . . . T_ISB_COMOBJ_EXPR
*/
libGAP_CVar libGAP_CompIsbComObjExpr (
    libGAP_Expr                expr )
{
    libGAP_CVar                isb;            /* isbound, result                 */
    libGAP_CVar                record;         /* the record, left operand        */
    libGAP_UInt                rnam;           /* the name, right operand         */

    /* allocate a new temporary for the result                             */
    isb = libGAP_CVAR_TEMP( libGAP_NewTemp( "isb" ) );

    /* compile the record expression (checking is done by 'ISB_REC')       */
    record = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[0] );

    /* get the name (stored immediately in the expression)                 */
    rnam = libGAP_CompExpr( libGAP_ADDR_EXPR(expr)[1] );

    /* emit the code to test the element                                   */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "%c = (IsbPRec( %c, RNamObj(%c) ) ? True : False);\n",
          isb, record, rnam );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "%c = (ISB_REC( %c, RNamObj(%c) ) ? True : False);\n",
          isb, record, rnam );
    libGAP_Emit( "}\n" );

    /* we know that the result is boolean                                  */
    libGAP_SetInfoCVar( isb, libGAP_W_BOOL );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );

    /* return the result                                                   */
    return isb;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * compile statements * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  CompStat( <stat> )  . . . . . . . . . . . . . . . . . compile a statement
**
**  'CompStat' compiles the statement <stat>.
*/
void (* libGAP_CompStatFuncs[256]) ( libGAP_Stat stat );

void libGAP_CompStat (
    libGAP_Stat                stat )
{
    (* libGAP_CompStatFuncs[ libGAP_TNUM_STAT(stat) ])( stat );
}


/****************************************************************************
**
*F  CompUnknownStat( <stat> ) . . . . . . . . . . . . . . . . signal an error
*/
void libGAP_CompUnknownStat (
    libGAP_Stat                stat )
{
    libGAP_Emit( "CANNOT COMPILE STATEMENT OF TNUM %d;\n", libGAP_TNUM_STAT(stat) );
}


/****************************************************************************
**
*V  G_Add . . . . . . . . . . . . . . . . . . . . . . . . . .  function 'Add'
*/
libGAP_GVar libGAP_G_Add;


/****************************************************************************
**
*F  CompProccall0to6Args( <stat> )  . . . T_PROCCALL_0ARGS...T_PROCCALL_6ARGS
*/
void libGAP_CompProccall0to6Args (
    libGAP_Stat                stat )
{
    libGAP_CVar                func;           /* function                        */
    libGAP_CVar                args[8];        /* arguments                       */
    libGAP_UInt                narg;           /* number of arguments             */
    libGAP_UInt                i;              /* loop variable                   */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* special case to inline 'Add'                                        */
    if ( libGAP_CompFastListFuncs
      && libGAP_TNUM_EXPR( libGAP_FUNC_CALL(stat) ) == libGAP_T_REF_GVAR
      && libGAP_ADDR_EXPR( libGAP_FUNC_CALL(stat) )[0] == libGAP_G_Add
      && libGAP_NARG_SIZE_CALL(libGAP_SIZE_EXPR(stat)) == 2 ) {
        args[1] = libGAP_CompExpr( libGAP_ARGI_CALL(stat,1) );
        args[2] = libGAP_CompExpr( libGAP_ARGI_CALL(stat,2) );
        if ( libGAP_CompFastPlainLists ) {
            libGAP_Emit( "C_ADD_LIST_FPL( %c, %c )\n", args[1], args[2] );
        }
        else {
            libGAP_Emit( "C_ADD_LIST( %c, %c )\n", args[1], args[2] );
        }
        if ( libGAP_IS_TEMP_CVAR( args[2] ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( args[2] ) );
        if ( libGAP_IS_TEMP_CVAR( args[1] ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( args[1] ) );
        return;
    }

    /* compile the reference to the function                               */
    if ( libGAP_TNUM_EXPR( libGAP_FUNC_CALL(stat) ) == libGAP_T_REF_GVAR ) {
        func = libGAP_CompRefGVarFopy( libGAP_FUNC_CALL(stat) );
    }
    else {
        func = libGAP_CompExpr( libGAP_FUNC_CALL(stat) );
        libGAP_CompCheckFunc( func );
    }

    /* compile the argument expressions                                    */
    narg = libGAP_NARG_SIZE_CALL(libGAP_SIZE_STAT(stat));
    for ( i = 1; i <= narg; i++ ) {
        args[i] = libGAP_CompExpr( libGAP_ARGI_CALL(stat,i) );
    }

    /* emit the code for the procedure call                                */
    libGAP_Emit( "CALL_%dARGS( %c", narg, func );
    for ( i = 1; i <= narg; i++ ) {
        libGAP_Emit( ", %c", args[i] );
    }
    libGAP_Emit( " );\n" );

    /* free the temporaries                                                */
    for ( i = narg; 1 <= i; i-- ) {
        if ( libGAP_IS_TEMP_CVAR( args[i] ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( args[i] ) );
    }
    if ( libGAP_IS_TEMP_CVAR( func ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( func ) );
}


/****************************************************************************
**
*F  CompProccallXArgs . . . . . . . . . . . . . . . . . . .  T_PROCCALL_XARGS
*/
void libGAP_CompProccallXArgs (
    libGAP_Stat                stat )
{
    libGAP_CVar                func;           /* function                        */
    libGAP_CVar                argl;           /* argument list                   */
    libGAP_CVar                argi;           /* <i>-th argument                 */
    libGAP_UInt                narg;           /* number of arguments             */
    libGAP_UInt                i;              /* loop variable                   */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the reference to the function                               */
    if ( libGAP_TNUM_EXPR( libGAP_FUNC_CALL(stat) ) == libGAP_T_REF_GVAR ) {
        func = libGAP_CompRefGVarFopy( libGAP_FUNC_CALL(stat) );
    }
    else {
        func = libGAP_CompExpr( libGAP_FUNC_CALL(stat) );
        libGAP_CompCheckFunc( func );
    }

    /* compile the argument expressions                                    */
    narg = libGAP_NARG_SIZE_CALL(libGAP_SIZE_STAT(stat));
    argl = libGAP_CVAR_TEMP( libGAP_NewTemp( "argl" ) );
    libGAP_Emit( "%c = NEW_PLIST( T_PLIST, %d );\n", argl, narg );
    libGAP_Emit( "SET_LEN_PLIST( %c, %d );\n", argl, narg );
    for ( i = 1; i <= narg; i++ ) {
        argi = libGAP_CompExpr( libGAP_ARGI_CALL( stat, i ) );
        libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", argl, i, argi );
        if ( ! libGAP_HasInfoCVar( argi, libGAP_W_INT_SMALL ) ) {
            libGAP_Emit( "CHANGED_BAG( %c );\n", argl );
        }
        if ( libGAP_IS_TEMP_CVAR( argi ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( argi ) );
    }

    /* emit the code for the procedure call                                */
    libGAP_Emit( "CALL_XARGS( %c, %c );\n", func, argl );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( argl ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( argl ) );
    if ( libGAP_IS_TEMP_CVAR( func ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( func ) );
}

/****************************************************************************
**
*F  CompProccallXArgs( <expr> ) . . . . . . . . . . . . . .  T_PROCCALL_OPTS
*/
void libGAP_CompProccallOpts(
                      libGAP_Stat stat)
{
  libGAP_CVar opts = libGAP_CompExpr(libGAP_ADDR_STAT(stat)[0]);
  libGAP_GVar pushOptions;
  libGAP_GVar popOptions;
  pushOptions = libGAP_GVarName("PushOptions");
  popOptions = libGAP_GVarName("PopOptions");
  libGAP_CompSetUseGVar(pushOptions, libGAP_COMP_USE_GVAR_FOPY);
  libGAP_CompSetUseGVar(popOptions, libGAP_COMP_USE_GVAR_FOPY);
  libGAP_Emit("CALL_1ARGS( GF_PushOptions, %c );\n", opts);
  if (libGAP_IS_TEMP_CVAR( opts) ) libGAP_FreeTemp( libGAP_TEMP_CVAR( opts ));
  libGAP_CompStat(libGAP_ADDR_STAT(stat)[1]);
  libGAP_Emit("CALL_0ARGS( GF_PopOptions );\n");
}
     

/****************************************************************************
**
*F  CompSeqStat( <stat> ) . . . . . . . . . . . . .  T_SEQ_STAT...T_SEQ_STAT7
*/
void libGAP_CompSeqStat (
    libGAP_Stat                stat )
{
    libGAP_UInt                nr;             /* number of statements            */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the number of statements                                        */
    nr = libGAP_SIZE_STAT( stat ) / sizeof(libGAP_Stat);

    /* compile the statements                                              */
    for ( i = 1; i <= nr; i++ ) {
        libGAP_CompStat( libGAP_ADDR_STAT( stat )[i-1] );
    }
}


/****************************************************************************
**
*F  CompIf( <stat> )  . . . . . . . . T_IF/T_IF_ELSE/T_IF_ELIF/T_IF_ELIF_ELSE
*/
void libGAP_CompIf (
    libGAP_Stat                stat )
{
    libGAP_CVar                cond;           /* condition                       */
    libGAP_UInt                nr;             /* number of branches              */
    libGAP_Bag                 info_in;        /* information at branch begin     */
    libGAP_Bag                 info_out;       /* information at branch end       */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the number of branches                                          */
    nr = libGAP_SIZE_STAT( stat ) / (2*sizeof(libGAP_Stat));

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* if " );
        libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
        libGAP_Emit( " then */\n" );
    }

    /* compile the expression                                              */
    cond = libGAP_CompBoolExpr( libGAP_ADDR_STAT( stat )[0] );

    /* emit the code to test the condition                                 */
    libGAP_Emit( "if ( %c ) {\n", cond );
    if ( libGAP_IS_TEMP_CVAR( cond ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cond ) );

    /* remember what we know after evaluating the first condition          */
    info_in = libGAP_NewInfoCVars();
    libGAP_CopyInfoCVars( info_in, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

    /* compile the body                                                    */
    libGAP_CompStat( libGAP_ADDR_STAT( stat )[1] );

    /* remember what we know after executing the first body                */
    info_out = libGAP_NewInfoCVars();
    libGAP_CopyInfoCVars( info_out, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

    /* emit the rest code                                                  */
    libGAP_Emit( "\n}\n" );

    /* loop over the 'elif' branches                                       */
    for ( i = 2; i <= nr; i++ ) {

        /* do not handle 'else' branch here                                */
        if ( i == nr && libGAP_TNUM_EXPR(libGAP_ADDR_STAT(stat)[2*(i-1)]) == libGAP_T_TRUE_EXPR )
            break;

        /* print a comment                                                 */
        if ( libGAP_CompPass == 2 ) {
            libGAP_Emit( "\n/* elif " );
            libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[2*(i-1)] );
            libGAP_Emit( " then */\n" );
        }

        /* emit the 'else' to connect this branch to the 'if' branch       */
        libGAP_Emit( "else {\n" );

        /* this is what we know if we enter this branch                    */
        libGAP_CopyInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), info_in );

        /* compile the expression                                          */
        cond = libGAP_CompBoolExpr( libGAP_ADDR_STAT( stat )[2*(i-1)] );

        /* emit the code to test the condition                             */
        libGAP_Emit( "if ( %c ) {\n", cond );
        if ( libGAP_IS_TEMP_CVAR( cond ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cond ) );

        /* remember what we know after evaluating all previous conditions  */
        libGAP_CopyInfoCVars( info_in, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

        /* compile the body                                                */
        libGAP_CompStat( libGAP_ADDR_STAT( stat )[2*(i-1)+1] );

        /* remember what we know after executing one of the previous bodies*/
        libGAP_MergeInfoCVars( info_out, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

        /* emit the rest code                                              */
        libGAP_Emit( "\n}\n" );

    }

    /* handle 'else' branch                                                */
    if ( i == nr ) {

        /* print a comment                                                 */
        if ( libGAP_CompPass == 2 ) {
            libGAP_Emit( "\n/* else */\n" );
        }

        /* emit the 'else' to connect this branch to the 'if' branch       */
        libGAP_Emit( "else {\n" );

        /* this is what we know if we enter this branch                    */
        libGAP_CopyInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), info_in );

        /* compile the body                                                */
        libGAP_CompStat( libGAP_ADDR_STAT( stat )[2*(i-1)+1] );

        /* remember what we know after executing one of the previous bodies*/
        libGAP_MergeInfoCVars( info_out, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

        /* emit the rest code                                              */
        libGAP_Emit( "\n}\n" );

    }

    /* fake empty 'else' branch                                            */
    else {

        /* this is what we know if we enter this branch                    */
        libGAP_CopyInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), info_in );

        /* remember what we know after executing one of the previous bodies*/
        libGAP_MergeInfoCVars( info_out, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );

    }

    /* close all unbalanced parenthesis                                    */
    for ( i = 2; i <= nr; i++ ) {
        if ( i == nr && libGAP_TNUM_EXPR(libGAP_ADDR_STAT(stat)[2*(i-1)]) == libGAP_T_TRUE_EXPR )
            break;
        libGAP_Emit( "}\n" );
    }
    libGAP_Emit( "/* fi */\n" );

    /* put what we know into the current info                              */
    libGAP_CopyInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), info_out );

}


/****************************************************************************
**
*F  CompFor( <stat> ) . . . . . . . T_FOR...T_FOR3/T_FOR_RANGE...T_FOR_RANGE3
*/
void libGAP_CompFor (
    libGAP_Stat                stat )
{
    libGAP_UInt                var;            /* loop variable                   */
    libGAP_Char                vart;           /* variable type                   */
    libGAP_CVar                list;           /* list to loop over               */
    libGAP_CVar                islist;         /* is the list a proper list       */
    libGAP_CVar                first;          /* first loop index                */
    libGAP_CVar                last;           /* last  loop index                */
    libGAP_CVar                lidx;           /* loop index variable             */
    libGAP_CVar                elm;            /* element of list                 */
    libGAP_Int                 pass;           /* current pass                    */
    libGAP_Bag                 prev;           /* previous temp-info              */
    libGAP_Int                 i;              /* loop variable                   */

    /* handle 'for <lvar> in [<first>..<last>] do'                         */
    if ( libGAP_IS_REFLVAR( libGAP_ADDR_STAT(stat)[0] )
      && ! libGAP_CompGetUseHVar( libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] ) )
      && libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[1] ) == libGAP_T_RANGE_EXPR
      && libGAP_SIZE_EXPR( libGAP_ADDR_STAT(stat)[1] ) == 2*sizeof(libGAP_Expr) ) {

        /* print a comment                                                 */
        if ( libGAP_CompPass == 2 ) {
            libGAP_Emit( "\n/* for " );
            libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
            libGAP_Emit( " in " );
            libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
            libGAP_Emit( " do */\n" );
        }

        /* get the local variable                                          */
        var = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );

        /* allocate a new temporary for the loop variable                  */
        lidx = libGAP_CVAR_TEMP( libGAP_NewTemp( "lidx" ) );

        /* compile and check the first and last value                      */
        first = libGAP_CompExpr( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[0] );
        libGAP_CompCheckIntSmall( first );

        /* compile and check the last value                                */
        /* if the last value is in a local variable,                       */
        /* we must copy it into a temporary,                               */
        /* because the local variable may change its value in the body     */
        last  = libGAP_CompExpr( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[1] )[1] );
        libGAP_CompCheckIntSmall( last  );
        if ( libGAP_IS_LVAR_CVAR(last) ) {
            elm = libGAP_CVAR_TEMP( libGAP_NewTemp( "last" ) );
            libGAP_Emit( "%c = %c;\n", elm, last );
            last = elm;
        }

        /* find the invariant temp-info                                    */
        pass = libGAP_CompPass;
        libGAP_CompPass = 99;
        prev = libGAP_NewInfoCVars();
        do {
            libGAP_CopyInfoCVars( prev, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );
            if ( libGAP_HasInfoCVar( first, libGAP_W_INT_SMALL_POS ) ) {
                libGAP_SetInfoCVar( libGAP_CVAR_LVAR(var), libGAP_W_INT_SMALL_POS );
            }
            else {
                libGAP_SetInfoCVar( libGAP_CVAR_LVAR(var), libGAP_W_INT_SMALL );
            }
            for ( i = 2; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
                libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
            }
            libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev );
        } while ( ! libGAP_IsEqInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev ) );
        libGAP_CompPass = pass;

        /* emit the code for the loop                                      */
        libGAP_Emit( "for ( %c = %c;\n",                lidx, first );
        libGAP_Emit( "      ((Int)%c) <= ((Int)%c);\n", lidx, last  );
        libGAP_Emit( "      %c = (Obj)(((UInt)%c)+4) ", lidx, lidx  );
        libGAP_Emit( ") {\n" );

        /* emit the code to copy the loop index into the loop variable     */
        libGAP_Emit( "%c = %c;\n", libGAP_CVAR_LVAR(var), lidx );

        /* set what we know about the loop variable                        */
        if ( libGAP_HasInfoCVar( first, libGAP_W_INT_SMALL_POS ) ) {
            libGAP_SetInfoCVar( libGAP_CVAR_LVAR(var), libGAP_W_INT_SMALL_POS );
        }
        else {
            libGAP_SetInfoCVar( libGAP_CVAR_LVAR(var), libGAP_W_INT_SMALL );
        }

        /* compile the body                                                */
        for ( i = 2; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
            libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
        }

        /* emit the end code                                               */
        libGAP_Emit( "\n}\n" );
        libGAP_Emit( "/* od */\n" );

        /* free the temporaries                                            */
        if ( libGAP_IS_TEMP_CVAR( last  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( last  ) );
        if ( libGAP_IS_TEMP_CVAR( first ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( first ) );
        if ( libGAP_IS_TEMP_CVAR( lidx  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lidx  ) );

    }

    /* handle other loops                                                  */
    else {

        /* print a comment                                                 */
        if ( libGAP_CompPass == 2 ) {
            libGAP_Emit( "\n/* for " );
            libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
            libGAP_Emit( " in " );
            libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[1] );
            libGAP_Emit( " do */\n" );
        }

        /* get the variable (initialize them first to please 'lint')       */
        if ( libGAP_IS_REFLVAR( libGAP_ADDR_STAT(stat)[0] )
          && ! libGAP_CompGetUseHVar( libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] ) ) ) {
            var = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );
            vart = 'l';
        }
        else if ( libGAP_IS_REFLVAR( libGAP_ADDR_STAT(stat)[0] ) ) {
            var = libGAP_LVAR_REFLVAR( libGAP_ADDR_STAT(stat)[0] );
            vart = 'm';
        }
        else if ( libGAP_T_REF_LVAR <= libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] )
               && libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) <= libGAP_T_REF_LVAR_16
               && ! libGAP_CompGetUseHVar( libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0] ) ) {
            var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
            vart = 'l';
        }
        else if ( libGAP_T_REF_LVAR <= libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] )
               && libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) <= libGAP_T_REF_LVAR_16 ) {
            var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
            vart = 'm';
        }
        else if ( libGAP_TNUM_EXPR( libGAP_ADDR_STAT(stat)[0] ) == libGAP_T_REF_HVAR ) {
            var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
            vart = 'h';
        }
        else /* if ( TNUM_EXPR( ADDR_STAT(stat)[0] ) == T_REF_GVAR ) */ {
            var = (libGAP_UInt)(libGAP_ADDR_EXPR( libGAP_ADDR_STAT(stat)[0] )[0]);
            libGAP_CompSetUseGVar( var, libGAP_COMP_USE_GVAR_ID );
            vart = 'g';
        }

        /* allocate a new temporary for the loop variable                  */
        lidx   = libGAP_CVAR_TEMP( libGAP_NewTemp( "lidx"   ) );
        elm    = libGAP_CVAR_TEMP( libGAP_NewTemp( "elm"    ) );
        islist = libGAP_CVAR_TEMP( libGAP_NewTemp( "islist" ) );

        /* compile and check the first and last value                      */
        list = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

        /* SL Patch added to try and avoid a bug */
        if (libGAP_IS_LVAR_CVAR(list))
          {
            libGAP_CVar copylist;
            copylist = libGAP_CVAR_TEMP( libGAP_NewTemp( "copylist" ) );
            libGAP_Emit("%c = %c;\n",copylist, list);
            list = copylist;
          }
        /* end of SL patch */

        /* find the invariant temp-info                                    */
        pass = libGAP_CompPass;
        libGAP_CompPass = 99;
        prev = libGAP_NewInfoCVars();
        do {
            libGAP_CopyInfoCVars( prev, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );
            if ( vart == 'l' ) {
                libGAP_SetInfoCVar( libGAP_CVAR_LVAR(var), libGAP_W_BOUND );
            }
            for ( i = 2; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
                libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
            }
            libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev );
        } while ( ! libGAP_IsEqInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev ) );
        libGAP_CompPass = pass;

        /* emit the code for the loop                                      */
        /* (plenty ugly because of iterator handling)                      */
        libGAP_Emit( "if ( IS_SMALL_LIST(%c) ) {\n", list );
        libGAP_Emit( "%c = (Obj)(UInt)1;\n", islist );
        libGAP_Emit( "%c = INTOBJ_INT(1);\n", lidx );
        libGAP_Emit( "}\n" );
        libGAP_Emit( "else {\n" );
        libGAP_Emit( "%c = (Obj)(UInt)0;\n", islist );
        libGAP_Emit( "%c = CALL_1ARGS( GF_ITERATOR, %c );\n", lidx, list );
        libGAP_Emit( "}\n" );
        libGAP_Emit( "while ( 1 ) {\n" );
        libGAP_Emit( "if ( %c ) {\n", islist );
        libGAP_Emit( "if ( LEN_LIST(%c) < %i )  break;\n", list, lidx );
        libGAP_Emit( "%c = ELMV0_LIST( %c, %i );\n", elm, list, lidx );
        libGAP_Emit( "%c = (Obj)(((UInt)%c)+4);\n", lidx, lidx );
        libGAP_Emit( "if ( %c == 0 )  continue;\n", elm );
        libGAP_Emit( "}\n" );
        libGAP_Emit( "else {\n" );
        libGAP_Emit( "if ( CALL_1ARGS( GF_IS_DONE_ITER, %c ) != False )  break;\n",
              lidx );
        libGAP_Emit( "%c = CALL_1ARGS( GF_NEXT_ITER, %c );\n", elm, lidx );
        libGAP_Emit( "}\n" );

        /* emit the code to copy the loop index into the loop variable     */
        if ( vart == 'l' ) {
            libGAP_Emit( "%c = %c;\n",
                  libGAP_CVAR_LVAR(var), elm );
        }
        else if ( vart == 'm' ) {
            libGAP_Emit( "ASS_LVAR( %d, %c );\n",
                  libGAP_GetIndxHVar(var), elm );
        }
        else if ( vart == 'h' ) {
            libGAP_Emit( "ASS_LVAR_%dUP( %d, %c );\n",
                  libGAP_GetLevlHVar(var), libGAP_GetIndxHVar(var), elm );
        }
        else if ( vart == 'g' ) {
            libGAP_Emit( "AssGVar( G_%n, %c );\n",
                  libGAP_NameGVar(var), elm );
        }

        /* set what we know about the loop variable                        */
        if ( vart == 'l' ) {
            libGAP_SetInfoCVar( libGAP_CVAR_LVAR(var), libGAP_W_BOUND );
        }

        /* compile the body                                                */
        for ( i = 2; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
            libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
        }

        /* emit the end code                                               */
        libGAP_Emit( "\n}\n" );
        libGAP_Emit( "/* od */\n" );

        /* free the temporaries                                            */
        if ( libGAP_IS_TEMP_CVAR( list   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list   ) );
        if ( libGAP_IS_TEMP_CVAR( islist ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( islist ) );
        if ( libGAP_IS_TEMP_CVAR( elm    ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( elm    ) );
        if ( libGAP_IS_TEMP_CVAR( lidx   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lidx   ) );

    }

}


/****************************************************************************
**
*F  CompWhile( <stat> ) . . . . . . . . . . . . . . . . .  T_WHILE...T_WHILE3
*/
void libGAP_CompWhile (
    libGAP_Stat                stat )
{
    libGAP_CVar                cond;           /* condition                       */
    libGAP_Int                 pass;           /* current pass                    */
    libGAP_Bag                 prev;           /* previous temp-info              */
    libGAP_UInt                i;              /* loop variable                   */

    /* find an invariant temp-info                                         */
    /* the emits are probably not needed                                   */
    pass = libGAP_CompPass;
    libGAP_CompPass = 99;
    libGAP_Emit( "while ( 1 ) {\n" );
    prev = libGAP_NewInfoCVars();
    do {
        libGAP_CopyInfoCVars( prev, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );
        cond = libGAP_CompBoolExpr( libGAP_ADDR_STAT(stat)[0] );
        libGAP_Emit( "if ( ! %c ) break;\n", cond );
        if ( libGAP_IS_TEMP_CVAR( cond ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cond ) );
        for ( i = 1; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
            libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
        }
        libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev );
    } while ( ! libGAP_IsEqInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev ) );
    libGAP_Emit( "}\n" );
    libGAP_CompPass = pass;

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* while " );
        libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
        libGAP_Emit( " od */\n" );
    }

    /* emit the code for the loop                                          */
    libGAP_Emit( "while ( 1 ) {\n" );

    /* compile the condition                                               */
    cond = libGAP_CompBoolExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Emit( "if ( ! %c ) break;\n", cond );
    if ( libGAP_IS_TEMP_CVAR( cond ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cond ) );

    /* compile the body                                                    */
    for ( i = 1; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
        libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
    }

    /* thats it                                                            */
    libGAP_Emit( "\n}\n" );
    libGAP_Emit( "/* od */\n" );

}


/****************************************************************************
**
*F  CompRepeat( <stat> )  . . . . . . . . . . . . . . .  T_REPEAT...T_REPEAT3
*/
void libGAP_CompRepeat (
    libGAP_Stat                stat )
{
    libGAP_CVar                cond;           /* condition                       */
    libGAP_Int                 pass;           /* current pass                    */
    libGAP_Bag                 prev;           /* previous temp-info              */
    libGAP_UInt                i;              /* loop variable                   */

    /* find an invariant temp-info                                         */
    /* the emits are probably not needed                                   */
    pass = libGAP_CompPass;
    libGAP_CompPass = 99;
    libGAP_Emit( "do {\n" );
    prev = libGAP_NewInfoCVars();
    do {
        libGAP_CopyInfoCVars( prev, libGAP_INFO_FEXP(libGAP_CURR_FUNC) );
        for ( i = 1; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
            libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
        }
        cond = libGAP_CompBoolExpr( libGAP_ADDR_STAT(stat)[0] );
        libGAP_Emit( "if ( %c ) break;\n", cond );
        if ( libGAP_IS_TEMP_CVAR( cond ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cond ) );
        libGAP_MergeInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev );
    } while ( ! libGAP_IsEqInfoCVars( libGAP_INFO_FEXP(libGAP_CURR_FUNC), prev ) );
    libGAP_Emit( "} while ( 1 );\n" );
    libGAP_CompPass = pass;

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* repeat */\n" );
    }

    /* emit the code for the loop                                          */
    libGAP_Emit( "do {\n" );

    /* compile the body                                                    */
    for ( i = 1; i < libGAP_SIZE_STAT(stat)/sizeof(libGAP_Stat); i++ ) {
        libGAP_CompStat( libGAP_ADDR_STAT(stat)[i] );
    }

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* until " );
        libGAP_PrintExpr( libGAP_ADDR_STAT(stat)[0] );
        libGAP_Emit( " */\n" );
    }

    /* compile the condition                                               */
    cond = libGAP_CompBoolExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Emit( "if ( %c ) break;\n", cond );
    if ( libGAP_IS_TEMP_CVAR( cond ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cond ) );

    /* thats it                                                            */
    libGAP_Emit( "} while ( 1 );\n" );
}


/****************************************************************************
**
*F  CompBreak( <stat> ) . . . . . . . . . . . . . . . . . . . . . . . T_BREAK
*/
void libGAP_CompBreak (
    libGAP_Stat                stat )
{
    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    libGAP_Emit( "break;\n" );
}

/****************************************************************************
**
*F  CompContinue( <stat> ) . . . . . . . . . . . . . . . . . . . . T_CONTINUE
*/
void libGAP_CompContinue (
    libGAP_Stat                stat )
{
    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    libGAP_Emit( "continue;\n" );
}


/****************************************************************************
**
*F  CompReturnObj( <stat> ) . . . . . . . . . . . . . . . . . .  T_RETURN_OBJ
*/
void libGAP_CompReturnObj (
    libGAP_Stat                stat )
{
    libGAP_CVar                obj;            /* returned object                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the expression                                              */
    obj = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* emit code to remove stack frame                                     */
    libGAP_Emit( "RES_BRK_CURR_STAT();\n" );
    libGAP_Emit( "SWITCH_TO_OLD_FRAME(oldFrame);\n" );

    /* emit code to return from function                                   */
    libGAP_Emit( "return %c;\n", obj );

    /* free the temporary                                                  */
    if ( libGAP_IS_TEMP_CVAR( obj ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( obj ) );
}


/****************************************************************************
**
*F  CompReturnVoid( <stat> )  . . . . . . . . . . . . . . . . . T_RETURN_VOID
*/
void libGAP_CompReturnVoid (
    libGAP_Stat                stat )
{
    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* emit code to remove stack frame                                     */
    libGAP_Emit( "RES_BRK_CURR_STAT();\n");
    libGAP_Emit( "SWITCH_TO_OLD_FRAME(oldFrame);\n" );

    /* emit code to return from function                                   */
    libGAP_Emit( "return 0;\n" );
}


/****************************************************************************
**
*F  CompAssLVar( <stat> ) . . . . . . . . . . . .  T_ASS_LVAR...T_ASS_LVAR_16
*/
void            libGAP_CompAssLVar (
    libGAP_Stat                stat )
{
    libGAP_LVar                lvar;           /* local variable                  */
    libGAP_CVar                rhs;            /* right hand side                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the right hand side expression                              */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* emit the code for the assignment                                    */
    lvar = (libGAP_LVar)(libGAP_ADDR_STAT(stat)[0]);
    if ( libGAP_CompGetUseHVar( lvar ) ) {
        libGAP_Emit( "ASS_LVAR( %d, %c );\n", libGAP_GetIndxHVar(lvar), rhs );
    }
    else {
        libGAP_Emit( "%c = %c;\n", libGAP_CVAR_LVAR(lvar), rhs );
        libGAP_SetInfoCVar( libGAP_CVAR_LVAR(lvar), libGAP_GetInfoCVar( rhs ) );
    }

    /* free the temporary                                                  */
    if ( libGAP_IS_TEMP_CVAR( rhs ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs ) );
}


/****************************************************************************
**
*F  CompUnbLVar( <stat> ) . . . . . . . . . . . . . . . . . . . .  T_UNB_LVAR
*/
void libGAP_CompUnbLVar (
    libGAP_Stat                stat )
{
    libGAP_LVar                lvar;           /* local variable                  */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* emit the code for the assignment                                    */
    lvar = (libGAP_LVar)(libGAP_ADDR_STAT(stat)[0]);
    if ( libGAP_CompGetUseHVar( lvar ) ) {
        libGAP_Emit( "ASS_LVAR( %d, 0 );\n", libGAP_GetIndxHVar(lvar) );
    }
    else {
        libGAP_Emit( "%c = 0;\n", libGAP_CVAR_LVAR( lvar ) );
        libGAP_SetInfoCVar( lvar, libGAP_W_UNBOUND );
    }
}


/****************************************************************************
**
*F  CompAssHVar( <stat> ) . . . . . . . . . . . . . . . . . . . .  T_ASS_HVAR
*/
void libGAP_CompAssHVar (
    libGAP_Stat                stat )
{
    libGAP_HVar                hvar;           /* higher variable                 */
    libGAP_CVar                rhs;            /* right hand side                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the right hand side expression                              */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* emit the code for the assignment                                    */
    hvar = (libGAP_HVar)(libGAP_ADDR_STAT(stat)[0]);
    libGAP_CompSetUseHVar( hvar );
    libGAP_Emit( "ASS_LVAR_%dUP( %d, %c );\n",
          libGAP_GetLevlHVar(hvar), libGAP_GetIndxHVar(hvar), rhs );

    /* free the temporary                                                  */
    if ( libGAP_IS_TEMP_CVAR( rhs ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs ) );
}


/****************************************************************************
**
*F  CompUnbHVar( <stat> ) . . . . . . . . . . . . . . . . . . . .  T_UNB_HVAR
*/
void libGAP_CompUnbHVar (
    libGAP_Stat                stat )
{
    libGAP_HVar                hvar;           /* higher variable                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* emit the code for the assignment                                    */
    hvar = (libGAP_HVar)(libGAP_ADDR_STAT(stat)[0]);
    libGAP_CompSetUseHVar( hvar );
    libGAP_Emit( "ASS_LVAR_%dUP( %d, 0 );\n",
          libGAP_GetLevlHVar(hvar), libGAP_GetIndxHVar(hvar) );
}


/****************************************************************************
**
*F  CompAssGVar( <stat> ) . . . . . . . . . . . . . . . . . . . .  T_ASS_GVAR
*/
void libGAP_CompAssGVar (
    libGAP_Stat                stat )
{
    libGAP_GVar                gvar;           /* global variable                 */
    libGAP_CVar                rhs;            /* right hand side                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the right hand side expression                              */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* emit the code for the assignment                                    */
    gvar = (libGAP_GVar)(libGAP_ADDR_STAT(stat)[0]);
    libGAP_CompSetUseGVar( gvar, libGAP_COMP_USE_GVAR_ID );
    libGAP_Emit( "AssGVar( G_%n, %c );\n", libGAP_NameGVar(gvar), rhs );

    /* free the temporary                                                  */
    if ( libGAP_IS_TEMP_CVAR( rhs ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs ) );
}


/****************************************************************************
**
*F  CompUnbGVar( <stat> ) . . . . . . . . . . . . . . . . . . . .  T_UNB_GVAR
*/
void            libGAP_CompUnbGVar (
    libGAP_Stat                stat )
{
    libGAP_GVar                gvar;           /* global variable                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* emit the code for the assignment                                    */
    gvar = (libGAP_GVar)(libGAP_ADDR_STAT(stat)[0]);
    libGAP_CompSetUseGVar( gvar, libGAP_COMP_USE_GVAR_ID );
    libGAP_Emit( "AssGVar( G_%n, 0 );\n", libGAP_NameGVar(gvar) );
}


/****************************************************************************
**
*F  CompAssList( <stat> ) . . . . . . . . . . . . . . . . . . . .  T_ASS_LIST
*/
void libGAP_CompAssList (
    libGAP_Stat                stat )
{
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                pos;            /* position                        */
    libGAP_CVar                rhs;            /* right hand side                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expression                                         */
    list = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_CompCheckIntPos( pos );

    /* compile the right hand side                                         */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code                                                       */
    if ( libGAP_CompFastPlainLists ) {
        if ( libGAP_HasInfoCVar( rhs, libGAP_W_INT_SMALL ) ) {
            libGAP_Emit( "C_ASS_LIST_FPL_INTOBJ( %c, %c, %c )\n", list, pos, rhs );
        }
        else {
            libGAP_Emit( "C_ASS_LIST_FPL( %c, %c, %c )\n", list, pos, rhs );
        }
    }
    else {
        libGAP_Emit( "C_ASS_LIST( %c, %c, %c );\n", list, pos, rhs );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhs  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs  ) );
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );
}


/****************************************************************************
**
*F  CompAsssList( <stat> )  . . . . . . . . . . . . . . . . . . . T_ASSS_LIST
*/
void libGAP_CompAsssList (
    libGAP_Stat                stat )
{
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                poss;           /* positions                       */
    libGAP_CVar                rhss;           /* right hand sides                */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expression                                         */
    list = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    poss = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* compile the right hand side                                         */
    rhss = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code                                                       */
    libGAP_Emit( "AsssListCheck( %c, %c, %c );\n", list, poss, rhss );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhss ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhss ) );
    if ( libGAP_IS_TEMP_CVAR( poss ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( poss ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );
}


/****************************************************************************
**
*F  CompAssListLev( <stat> )  . . . . . . . . . . . . . . . .  T_ASS_LIST_LEV
*/
void libGAP_CompAssListLev (
    libGAP_Stat                stat )
{
    libGAP_CVar                lists;          /* lists                           */
    libGAP_CVar                pos;            /* position                        */
    libGAP_CVar                rhss;           /* right hand sides                */
    libGAP_Int                 level;          /* level                           */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expressions                                        */
    lists = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_CompCheckIntSmallPos( pos );

    /* compile the right hand sides                                        */
    rhss = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_STAT(stat)[3]);

    /* emit the code                                                       */
    libGAP_Emit( "AssListLevel( %c, %c, %c, %d );\n", lists, pos, rhss, level );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhss  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhss  ) );
    if ( libGAP_IS_TEMP_CVAR( pos   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos   ) );
    if ( libGAP_IS_TEMP_CVAR( lists ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lists ) );
}


/****************************************************************************
**
*F  CompAsssListLev( <stat> ) . . . . . . . . . . . . . . . . T_ASSS_LIST_LEV
*/
void libGAP_CompAsssListLev (
    libGAP_Stat                stat )
{
    libGAP_CVar                lists;          /* list                            */
    libGAP_CVar                poss;           /* positions                       */
    libGAP_CVar                rhss;           /* right hand sides                */
    libGAP_Int                 level;          /* level                           */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expressions                                        */
    lists = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    poss = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* compile the right hand side                                         */
    rhss = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* get the level                                                       */
    level = (libGAP_Int)(libGAP_ADDR_STAT(stat)[3]);

    /* emit the code                                                       */
    libGAP_Emit( "AsssListLevelCheck( %c, %c, %c, %d );\n",
          lists, poss, rhss, level );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhss  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhss ) );
    if ( libGAP_IS_TEMP_CVAR( poss  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( poss ) );
    if ( libGAP_IS_TEMP_CVAR( lists ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lists ) );
}


/****************************************************************************
**
*F  CompUnbList( <stat> ) . . . . . . . . . . . . . . . . . . . .  T_UNB_LIST
*/
void libGAP_CompUnbList (
    libGAP_Stat                stat )
{
    libGAP_CVar                list;           /* list, left operand              */
    libGAP_CVar                pos;            /* position, left operand          */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expression                                         */
    list = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_CompCheckIntPos( pos );

    /* emit the code                                                       */
    libGAP_Emit( "C_UNB_LIST( %c, %c );\n", list, pos );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );
}


/****************************************************************************
**
*F  CompAssRecName( <stat> )  . . . . . . . . . . . . . . . .  T_ASS_REC_NAME
*/
void libGAP_CompAssRecName (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_CVar                rhs;            /* rhs, right operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* compile the right hand side                                         */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "ASS_REC( %c, R_%n, %c );\n", record, libGAP_NAME_RNAM(rnam), rhs );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhs    ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs    ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}


/****************************************************************************
**
*F  CompAssRecExpr( <stat> )  . . . . . . . . . . . . . . . .  T_ASS_REC_EXPR
*/
void libGAP_CompAssRecExpr (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_CVar                rnam;           /* name, left operand              */
    libGAP_CVar                rhs;            /* rhs, right operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* compile the right hand side                                         */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "ASS_REC( %c, RNamObj(%c), %c );\n", record, rnam, rhs );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhs    ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs    ) );
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}


/****************************************************************************
**
*F  CompUnbRecName( <stat> )  . . . . . . . . . . . . . . . .  T_UNB_REC_NAME
*/
void libGAP_CompUnbRecName (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "UNB_REC( %c, R_%n );\n", record, libGAP_NAME_RNAM(rnam) );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}


/****************************************************************************
**
*F  CompUnbRecExpr( <stat> )  . . . . . . . . . . . . . . . .  T_UNB_REC_EXPR
*/
void            libGAP_CompUnbRecExpr (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_CVar                rnam;           /* name, left operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "UNB_REC( %c, RNamObj(%c) );\n", record, rnam );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}


/****************************************************************************
**
*F  CompAssPosObj( <stat> ) . . . . . . . . . . . . . . . . . .  T_ASS_POSOBJ
*/
void libGAP_CompAssPosObj (
    libGAP_Stat                stat )
{
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                pos;            /* position                        */
    libGAP_CVar                rhs;            /* right hand side                 */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expression                                         */
    list = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_CompCheckIntSmallPos( pos );

    /* compile the right hand side                                         */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code                                                       */
    if ( libGAP_HasInfoCVar( rhs, libGAP_W_INT_SMALL ) ) {
        libGAP_Emit( "C_ASS_POSOBJ_INTOBJ( %c, %i, %c )\n", list, pos, rhs );
    }
    else {
        libGAP_Emit( "C_ASS_POSOBJ( %c, %i, %c )\n", list, pos, rhs );
    }

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhs  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs  ) );
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );
}



/****************************************************************************
**
*F  CompAsssPosObj( <stat> )  . . . . . . . . . . . . . . . . . T_ASSS_POSOBJ
*/
void libGAP_CompAsssPosObj (
    libGAP_Stat                stat )
{
    libGAP_CVar                list;           /* list                            */
    libGAP_CVar                poss;           /* positions                       */
    libGAP_CVar                rhss;           /* right hand sides                */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expression                                         */
    list = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    poss = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* compile the right hand side                                         */
    rhss = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code                                                       */
    libGAP_Emit( "AsssPosObjCheck( %c, %c, %c );\n", list, poss, rhss );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhss ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhss ) );
    if ( libGAP_IS_TEMP_CVAR( poss ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( poss ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );
}


/****************************************************************************
**
*F  CompAssPosObjLev( <stat> )  . . . . . . . . . . . . . .  T_ASS_POSOBJ_LEV
*/
void libGAP_CompAssPosObjLev (
    libGAP_Stat                stat )
{
    libGAP_Emit( "CANNOT COMPILE STATEMENT OF TNUM %d;\n", libGAP_TNUM_STAT(stat) );
}


/****************************************************************************
**
*F  CompAsssPosObjLev( <stat> ) . . . . . . . . . . . . . . T_ASSS_POSOBJ_LEV
*/
void libGAP_CompAsssPosObjLev (
    libGAP_Stat                stat )
{
    libGAP_Emit( "CANNOT COMPILE STATEMENT OF TNUM %d;\n", libGAP_TNUM_STAT(stat) );
}


/****************************************************************************
**
*F  CompUnbPosObj( <stat> ) . . . . . . . . . . . . . . . . . .  T_UNB_POSOBJ
*/
void libGAP_CompUnbPosObj (
    libGAP_Stat                stat )
{
    libGAP_CVar                list;           /* list, left operand              */
    libGAP_CVar                pos;            /* position, left operand          */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the list expression                                         */
    list = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* compile and check the position expression                           */
    pos = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_CompCheckIntSmallPos( pos );

    /* emit the code                                                       */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_POSOBJ ) {\n", list );
    libGAP_Emit( "if ( %i <= SIZE_OBJ(%c)/sizeof(Obj)-1 ) {\n", pos, list );
    libGAP_Emit( "SET_ELM_PLIST( %c, %i, 0 );\n", list, pos );
    libGAP_Emit( "}\n}\n" );
    libGAP_Emit( "else {\n" );
    libGAP_Emit( "UNB_LIST( %c, %i );\n", list, pos );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( pos  ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( pos  ) );
    if ( libGAP_IS_TEMP_CVAR( list ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( list ) );
}


/****************************************************************************
**
*F  CompAssComObjName( <stat> ) . . . . . . . . . . . . . . T_ASS_COMOBJ_NAME
*/
void libGAP_CompAssComObjName (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */
    libGAP_CVar                rhs;            /* rhs, right operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* compile the right hand side                                         */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "AssPRec( %c, R_%n, %c );\n", record, libGAP_NAME_RNAM(rnam), rhs );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "ASS_REC( %c, R_%n, %c );\n", record, libGAP_NAME_RNAM(rnam), rhs );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhs    ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs    ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}


/****************************************************************************
**
*F  CompAssComObjExpr( <stat> ) . . . . . . . . . . . . . . T_ASS_COMOBJ_EXPR
*/
void libGAP_CompAssComObjExpr (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_CVar                rnam;           /* name, left operand              */
    libGAP_CVar                rhs;            /* rhs, right operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );

    /* compile the right hand side                                         */
    rhs = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "AssPRec( %c, RNamObj(%c), %c );\n", record, rnam, rhs );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "ASS_REC( %c, RNamObj(%c), %c );\n", record, rnam, rhs );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rhs    ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rhs    ) );
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}


/****************************************************************************
**
*F  CompUnbComObjName( <stat> ) . . . . . . . . . . . . . . T_UNB_COMOBJ_NAME
*/
void libGAP_CompUnbComObjName (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = (libGAP_UInt)(libGAP_ADDR_STAT(stat)[1]);
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "UnbPRec( %c, R_%n );\n", record, libGAP_NAME_RNAM(rnam) );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "UNB_REC( %c, R_%n );\n", record, libGAP_NAME_RNAM(rnam) );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}


/****************************************************************************
**
*F  CompUnbComObjExpr( <stat> ) . . . . . . . . . . . . . . T_UNB_COMOBJ_EXPR
*/
void libGAP_CompUnbComObjExpr (
    libGAP_Stat                stat )
{
    libGAP_CVar                record;         /* record, left operand            */
    libGAP_UInt                rnam;           /* name, left operand              */

    /* print a comment                                                     */
    if ( libGAP_CompPass == 2 ) {
        libGAP_Emit( "\n/* " ); libGAP_PrintStat( stat ); libGAP_Emit( " */\n" );
    }

    /* compile the record expression                                       */
    record = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );

    /* get the name (stored immediately in the statement)                  */
    rnam = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_CompSetUseRNam( rnam, libGAP_COMP_USE_RNAM_ID );

    /* emit the code for the assignment                                    */
    libGAP_Emit( "if ( TNUM_OBJ(%c) == T_COMOBJ ) {\n", record );
    libGAP_Emit( "UnbPRec( %c, RNamObj(%c) );\n", record, rnam );
    libGAP_Emit( "}\nelse {\n" );
    libGAP_Emit( "UNB_REC( %c, RNamObj(%c) );\n", record, rnam );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( rnam   ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( rnam   ) );
    if ( libGAP_IS_TEMP_CVAR( record ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( record ) );
}

/****************************************************************************
**
*F  CompEmpty( <stat> )  . . . . . . . . . . . . . . . . . . . . . . . T_EMPY
*/
void libGAP_CompEmpty (
    libGAP_Stat                stat )
{
  libGAP_Emit("\n/* ; */\n");
  libGAP_Emit(";");
}
  
/****************************************************************************
**
*F  CompInfo( <stat> )  . . . . . . . . . . . . . . . . . . . . . . .  T_INFO
*/
void libGAP_CompInfo (
    libGAP_Stat                stat )
{
    libGAP_CVar                tmp;
    libGAP_CVar                sel;
    libGAP_CVar                lev;
    libGAP_CVar                lst;
    libGAP_Int                 narg;
    libGAP_Int                 i;

    libGAP_Emit( "\n/* Info( ... ); */\n" );
    sel = libGAP_CompExpr( libGAP_ARGI_INFO( stat, 1 ) );
    lev = libGAP_CompExpr( libGAP_ARGI_INFO( stat, 2 ) );
    lst = libGAP_CVAR_TEMP( libGAP_NewTemp( "lst" ) );
    tmp = libGAP_CVAR_TEMP( libGAP_NewTemp( "tmp" ) );
    libGAP_Emit( "%c = CALL_2ARGS( InfoDecision, %c, %c );\n", tmp, sel, lev );
    libGAP_Emit( "if ( %c == True ) {\n", tmp );
    if ( libGAP_IS_TEMP_CVAR( tmp ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( tmp ) );
    narg = libGAP_NARG_SIZE_INFO(libGAP_SIZE_STAT(stat))-2;
    libGAP_Emit( "%c = NEW_PLIST( T_PLIST, %d );\n", lst, narg );
    libGAP_Emit( "SET_LEN_PLIST( %c, %d );\n", lst, narg );
    for ( i = 1;  i <= narg;  i++ ) {
        tmp = libGAP_CompExpr( libGAP_ARGI_INFO( stat, i+2 ) );
        libGAP_Emit( "SET_ELM_PLIST( %c, %d, %c );\n", lst, i, tmp );
        libGAP_Emit( "CHANGED_BAG(%c);\n", lst );
        if ( libGAP_IS_TEMP_CVAR( tmp ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( tmp ) );
    }
    libGAP_Emit( "CALL_1ARGS( InfoDoPrint, %c );\n", lst );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( lst ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lst ) );
    if ( libGAP_IS_TEMP_CVAR( lev ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lev ) );
    if ( libGAP_IS_TEMP_CVAR( sel ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( sel ) );
}


/****************************************************************************
**
*F  CompAssert2( <stat> ) . . . . . . . . . . . . . . . . . .  T_ASSERT_2ARGS
*/
void libGAP_CompAssert2 (
    libGAP_Stat                stat )
{
    libGAP_CVar                lev;            /* the level                       */
    libGAP_CVar                cnd;            /* the condition                   */

    libGAP_Emit( "\n/* Assert( ... ); */\n" );
    lev = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Emit( "if ( ! LT(CurrentAssertionLevel, %c) ) {\n", lev );
    cnd = libGAP_CompBoolExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Emit( "if ( ! %c ) {\n", cnd );
    libGAP_Emit( "ErrorReturnVoid(\"Assertion failure\",0L,0L,\"you may 'return;'\"" );
    libGAP_Emit( ");\n");
    libGAP_Emit( "}\n" );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( cnd ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cnd ) );
    if ( libGAP_IS_TEMP_CVAR( lev ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lev ) );
}


/****************************************************************************
**
*F  CompAssert3( <stat> ) . . . . . . . . . . . . . . . . . .  T_ASSERT_3ARGS
*/
void libGAP_CompAssert3 (
    libGAP_Stat                stat )
{
    libGAP_CVar                lev;            /* the level                       */
    libGAP_CVar                cnd;            /* the condition                   */
    libGAP_CVar                msg;            /* the message                     */

    libGAP_Emit( "\n/* Assert( ... ); */\n" );
    lev = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[0] );
    libGAP_Emit( "if ( ! LT(CurrentAssertionLevel, %c) ) {\n", lev );
    cnd = libGAP_CompBoolExpr( libGAP_ADDR_STAT(stat)[1] );
    libGAP_Emit( "if ( ! %c ) {\n", cnd );
    msg = libGAP_CompExpr( libGAP_ADDR_STAT(stat)[2] );
    libGAP_Emit( "if ( %c != (Obj)(UInt)0 )", msg );
    libGAP_Emit( "{\n if ( IS_STRING_REP ( %c ) )\n", msg);
    libGAP_Emit( "   PrintString1( %c);\n else\n   PrintObj(%c);\n}\n", msg, msg );
    libGAP_Emit( "}\n" );
    libGAP_Emit( "}\n" );

    /* free the temporaries                                                */
    if ( libGAP_IS_TEMP_CVAR( msg ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( msg ) );
    if ( libGAP_IS_TEMP_CVAR( cnd ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( cnd ) );
    if ( libGAP_IS_TEMP_CVAR( lev ) )  libGAP_FreeTemp( libGAP_TEMP_CVAR( lev ) );
}



/****************************************************************************
**

*F * * * * * * * * * * * * * * start compiling  * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  CompFunc( <func> )  . . . . . . . . . . . . . . . . .  compile a function
**
**  'CompFunc' compiles the function <func>, i.e., it emits  the code for the
**  handler of the function <func> and the handlers of all its subfunctions.
*/
libGAP_Obj libGAP_CompFunctions;
libGAP_Int libGAP_CompFunctionsNr;

void libGAP_CompFunc (
    libGAP_Obj                 func )
{
    libGAP_Bag                 info;           /* info bag for this function      */
    libGAP_Int                 narg;           /* number of arguments             */
    libGAP_Int                 nloc;           /* number of locals                */
    libGAP_Obj                 fexs;           /* function expression list        */
    libGAP_Bag                 oldFrame;       /* old frame                       */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the number of arguments and locals                              */
    narg = (libGAP_NARG_FUNC(func) != -1 ? libGAP_NARG_FUNC(func) : 1);
    nloc = libGAP_NLOC_FUNC(func);

    /* in the first pass allocate the info bag                             */
    if ( libGAP_CompPass == 1 ) {

        libGAP_CompFunctionsNr++;
        libGAP_GROW_PLIST(    libGAP_CompFunctions, libGAP_CompFunctionsNr );
        libGAP_SET_ELM_PLIST( libGAP_CompFunctions, libGAP_CompFunctionsNr, func );
        libGAP_SET_LEN_PLIST( libGAP_CompFunctions, libGAP_CompFunctionsNr );
        libGAP_CHANGED_BAG(   libGAP_CompFunctions );

        info = libGAP_NewBag( libGAP_T_STRING, libGAP_SIZE_INFO(narg+nloc,8) );
        libGAP_NEXT_INFO(info)  = libGAP_INFO_FEXP( libGAP_CURR_FUNC );
        libGAP_NR_INFO(info)    = libGAP_CompFunctionsNr;
        libGAP_NLVAR_INFO(info) = narg + nloc;
        libGAP_NHVAR_INFO(info) = 0;
        libGAP_NTEMP_INFO(info) = 0;
        libGAP_NLOOP_INFO(info) = 0;

        libGAP_INFO_FEXP(func) = info;
        libGAP_CHANGED_BAG(func);

    }

    /* switch to this function (so that 'ADDR_STAT' and 'ADDR_EXPR' work)  */
    libGAP_SWITCH_TO_NEW_LVARS( func, narg, nloc, oldFrame );

    /* get the info bag                                                    */
    info = libGAP_INFO_FEXP( libGAP_CURR_FUNC );

    /* compile the innner functions                                        */
    fexs = libGAP_FEXS_FUNC(func);
    for ( i = 1;  i <= libGAP_LEN_PLIST(fexs);  i++ ) {
        libGAP_CompFunc( libGAP_ELM_PLIST( fexs, i ) );
    }

    /* emit the code for the function header and the arguments             */
    libGAP_Emit( "\n/* handler for function %d */\n", libGAP_NR_INFO(info));
    if ( narg == 0 ) {
        libGAP_Emit( "static Obj  HdlrFunc%d (\n", libGAP_NR_INFO(info) );
        libGAP_Emit( " Obj  self )\n" );
        libGAP_Emit( "{\n" );
    }
    else if ( narg <= 6 ) {
        libGAP_Emit( "static Obj  HdlrFunc%d (\n", libGAP_NR_INFO(info) );
        libGAP_Emit( " Obj  self,\n" );
        for ( i = 1; i < narg; i++ ) {
            libGAP_Emit( " Obj  %c,\n", libGAP_CVAR_LVAR(i) );
        }
        libGAP_Emit( " Obj  %c )\n", libGAP_CVAR_LVAR(narg) );
        libGAP_Emit( "{\n" );
    }
    else {
        libGAP_Emit( "static Obj  HdlrFunc%d (\n", libGAP_NR_INFO(info) );
        libGAP_Emit( " Obj  self,\n" );
        libGAP_Emit( " Obj  args )\n" );
        libGAP_Emit( "{\n" );
        for ( i = 1; i <= narg; i++ ) {
            libGAP_Emit( "Obj  %c;\n", libGAP_CVAR_LVAR(i) );
        }
    }

    /* emit the code for the local variables                               */
    for ( i = 1; i <= nloc; i++ ) {
        if ( ! libGAP_CompGetUseHVar( i+narg ) ) {
            libGAP_Emit( "Obj %c = 0;\n", libGAP_CVAR_LVAR(i+narg) );
        }
    }

    /* emit the code for the temporaries                                   */
    for ( i = 1; i <= libGAP_NTEMP_INFO(info); i++ ) {
        libGAP_Emit( "Obj %c = 0;\n", libGAP_CVAR_TEMP(i) );
    }
    for ( i = 1; i <= libGAP_NLOOP_INFO(info); i++ ) {
        libGAP_Emit( "Int l_%d = 0;\n", i );
    }

    /* emit the code for the higher variables                              */
    libGAP_Emit( "Bag oldFrame;\n" );
    libGAP_Emit( "OLD_BRK_CURR_STAT\n");

    /* emit the code to get the arguments for xarg functions               */
    if ( 6 < narg ) {
        libGAP_Emit( "CHECK_NR_ARGS( %d, args )\n", narg );
        for ( i = 1; i <= narg; i++ ) {
            libGAP_Emit( "%c = ELM_PLIST( args, %d );\n", libGAP_CVAR_LVAR(i), i );
        }
    }

    /* emit the code to switch to a new frame for outer functions          */
#if 1
    /* Try and get better debugging by always doing this */
    if (1) {
#else
      /* this was the old code */
    if ( libGAP_NHVAR_INFO(info) != 0 ) {
#endif
        libGAP_Emit( "\n/* allocate new stack frame */\n" );
        libGAP_Emit( "SWITCH_TO_NEW_FRAME(self,%d,0,oldFrame);\n",libGAP_NHVAR_INFO(info));
        for ( i = 1; i <= narg; i++ ) {
            if ( libGAP_CompGetUseHVar( i ) ) {
                libGAP_Emit( "ASS_LVAR( %d, %c );\n",libGAP_GetIndxHVar(i),libGAP_CVAR_LVAR(i));
            }
        }
    }
    else {
        libGAP_Emit( "\n/* restoring old stack frame */\n" );
        libGAP_Emit( "oldFrame = CurrLVars;\n" );
        libGAP_Emit( "SWITCH_TO_OLD_FRAME(ENVI_FUNC(self));\n" );
    }

    /* emit the code to save and zero the "current statement" information
     so that the break loop behaves */
    libGAP_Emit( "REM_BRK_CURR_STAT();\n");
    libGAP_Emit( "SET_BRK_CURR_STAT(0);\n");
    
    /* we know all the arguments have values                               */
    for ( i = 1; i <= narg; i++ ) {
        libGAP_SetInfoCVar( libGAP_CVAR_LVAR(i), libGAP_W_BOUND );
    }
    for ( i = narg+1; i <= narg+nloc; i++ ) {
        libGAP_SetInfoCVar( libGAP_CVAR_LVAR(i), libGAP_W_UNBOUND );
    }

    /* compile the body                                                    */
    libGAP_CompStat( libGAP_FIRST_STAT_CURR_FUNC );

    /* emit the code to switch back to the old frame and return            */
    libGAP_Emit( "\n/* return; */\n" );
    libGAP_Emit( "RES_BRK_CURR_STAT();\n" );
    libGAP_Emit( "SWITCH_TO_OLD_FRAME(oldFrame);\n" );
    libGAP_Emit( "return 0;\n" );
    libGAP_Emit( "}\n" );

    /* switch back to old frame                                            */
    libGAP_SWITCH_TO_OLD_LVARS( oldFrame );
}


/****************************************************************************
**
*F  CompileFunc( <output>, <func>, <name>, <magic1>, <magic2> ) . . . compile
*/
libGAP_Int libGAP_CompileFunc (
    libGAP_Char *              output,
    libGAP_Obj                 func,
    libGAP_Char *              name,
    libGAP_Int                 magic1,
    libGAP_Char *              magic2 )
{
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Obj                 n;              /* temporary                       */
    libGAP_UInt                col;

    /* open the output file                                                */
    if ( ! libGAP_OpenOutput( output ) ) {
        return 0;
    }
    col = libGAP_SyNrCols;
    libGAP_SyNrCols = 255;

    /* store the magic values                                              */
    libGAP_compilerMagic1 = magic1;
    libGAP_compilerMagic2 = magic2;

    /* create 'CompInfoGVar' and 'CompInfoRNam'                            */
    libGAP_CompInfoGVar = libGAP_NewBag( libGAP_T_STRING, sizeof(libGAP_UInt) * 1024 );
    libGAP_CompInfoRNam = libGAP_NewBag( libGAP_T_STRING, sizeof(libGAP_UInt) * 1024 );

    /* create the list to collection the function expressions              */
    libGAP_CompFunctionsNr = 0;
    libGAP_CompFunctions = libGAP_NEW_PLIST( libGAP_T_PLIST, 8 );
    libGAP_SET_LEN_PLIST( libGAP_CompFunctions, 0 );

    /* first collect information about variables                           */
    libGAP_CompPass = 1;
    libGAP_CompFunc( func );

    /* ok, lets emit some code now                                         */
    libGAP_CompPass = 2;

    /* emit code to include the interface files                            */
    libGAP_Emit( "/* C file produced by GAC */\n" );
    libGAP_Emit( "#include \"src/compiled.h\"\n" );

    /* emit code for global variables                                      */
    libGAP_Emit( "\n/* global variables used in handlers */\n" );
    for ( i = 1; i < libGAP_SIZE_OBJ(libGAP_CompInfoGVar)/sizeof(libGAP_UInt); i++ ) {
        if ( libGAP_CompGetUseGVar( i ) ) {
            libGAP_Emit( "static GVar G_%n;\n", libGAP_NameGVar(i) );
        }
        if ( libGAP_CompGetUseGVar( i ) & libGAP_COMP_USE_GVAR_COPY ) {
            libGAP_Emit( "static Obj  GC_%n;\n", libGAP_NameGVar(i) );
        }
        if ( libGAP_CompGetUseGVar( i ) & libGAP_COMP_USE_GVAR_FOPY ) {
            libGAP_Emit( "static Obj  GF_%n;\n", libGAP_NameGVar(i) );
        }
    }

    /* emit code for record names                                          */
    libGAP_Emit( "\n/* record names used in handlers */\n" );
    for ( i = 1; i < libGAP_SIZE_OBJ(libGAP_CompInfoRNam)/sizeof(libGAP_UInt); i++ ) {
        if ( libGAP_CompGetUseRNam( i ) ) {
            libGAP_Emit( "static RNam R_%n;\n", libGAP_NAME_RNAM(i) );
        }
    }

    /* emit code for the functions                                         */
    libGAP_Emit( "\n/* information for the functions */\n" );
    libGAP_Emit( "static Obj  NameFunc[%d];\n", libGAP_CompFunctionsNr+1 );
    libGAP_Emit( "static Obj  NamsFunc[%d];\n", libGAP_CompFunctionsNr+1 );
    libGAP_Emit( "static Int  NargFunc[%d];\n", libGAP_CompFunctionsNr+1 );
    libGAP_Emit( "static Obj  DefaultName;\n" );
    libGAP_Emit( "static Obj FileName;\n" );


    /* now compile the handlers                                            */
    libGAP_CompFunc( func );

    /* emit the code for the function that links this module to GAP        */
    libGAP_Emit( "\n/* 'InitKernel' sets up data structures, fopies, copies, handlers */\n" );
    libGAP_Emit( "static Int InitKernel ( StructInitInfo * module )\n" );
    libGAP_Emit( "{\n" );
    libGAP_Emit( "\n/* global variables used in handlers */\n" );
    for ( i = 1; i < libGAP_SIZE_OBJ(libGAP_CompInfoGVar)/sizeof(libGAP_UInt); i++ ) {
        if ( libGAP_CompGetUseGVar( i ) & libGAP_COMP_USE_GVAR_COPY ) {
            libGAP_Emit( "InitCopyGVar( \"%s\", &GC_%n );\n",
                  libGAP_NameGVar(i), libGAP_NameGVar(i) );
        }
        if ( libGAP_CompGetUseGVar( i ) & libGAP_COMP_USE_GVAR_FOPY ) {
            libGAP_Emit( "InitFopyGVar( \"%s\", &GF_%n );\n",
                  libGAP_NameGVar(i), libGAP_NameGVar(i) );
        }
    }
    libGAP_Emit( "\n/* information for the functions */\n" );
    libGAP_Emit( "InitGlobalBag( &DefaultName, \"%s:DefaultName(%d)\" );\n",
          magic2, magic1 );
    libGAP_Emit( "InitGlobalBag( &FileName, \"%s:FileName(%d)\" );\n",
          magic2, magic1 );
    for ( i = 1; i <= libGAP_CompFunctionsNr; i++ ) {
        libGAP_Emit( "InitHandlerFunc( HdlrFunc%d, \"%s:HdlrFunc%d(%d)\" );\n",
              i, libGAP_compilerMagic2, i, libGAP_compilerMagic1 );
        libGAP_Emit( "InitGlobalBag( &(NameFunc[%d]), \"%s:NameFunc[%d](%d)\" );\n", 
               i, magic2, i, magic1 );
        n = libGAP_NAME_FUNC(libGAP_ELM_PLIST(libGAP_CompFunctions,i));
        if ( n != 0 && libGAP_IsStringConv(n) ) {
            libGAP_Emit( "InitGlobalBag( &(NamsFunc[%d]), \"%s:NamsFunc[%d](%d)\" );\n",
                  i, magic2, i, magic1 );
        }
    }
    libGAP_Emit( "\n/* return success */\n" );
    libGAP_Emit( "return 0;\n" );
    libGAP_Emit( "\n}\n" );

    libGAP_Emit( "\n/* 'InitLibrary' sets up gvars, rnams, functions */\n" );
    libGAP_Emit( "static Int InitLibrary ( StructInitInfo * module )\n" );
    libGAP_Emit( "{\n" );
    libGAP_Emit( "Obj func1;\n" );
    libGAP_Emit( "Obj body1;\n" );
    libGAP_Emit( "\n/* Complete Copy/Fopy registration */\n" );
    libGAP_Emit( "UpdateCopyFopyInfo();\n" );
    libGAP_Emit( "\n/* global variables used in handlers */\n" );
    for ( i = 1; i < libGAP_SIZE_OBJ(libGAP_CompInfoGVar)/sizeof(libGAP_UInt); i++ ) {
        if ( libGAP_CompGetUseGVar( i ) ) {
            libGAP_Emit( "G_%n = GVarName( \"%s\" );\n",
                   libGAP_NameGVar(i), libGAP_NameGVar(i) );
        }
    }
    libGAP_Emit( "\n/* record names used in handlers */\n" );
    for ( i = 1; i < libGAP_SIZE_OBJ(libGAP_CompInfoRNam)/sizeof(libGAP_UInt); i++ ) {
        if ( libGAP_CompGetUseRNam( i ) ) {
            libGAP_Emit( "R_%n = RNamName( \"%s\" );\n",
                  libGAP_NAME_RNAM(i), libGAP_NAME_RNAM(i) );
        }
    }
    libGAP_Emit( "\n/* information for the functions */\n" );
    libGAP_Emit( "C_NEW_STRING( DefaultName, 14, \"local function\" )\n" );
    libGAP_Emit( "C_NEW_STRING( FileName, %d, \"%s\" )\n", strlen(magic2), magic2 );
    for ( i = 1; i <= libGAP_CompFunctionsNr; i++ ) {
        n = libGAP_NAME_FUNC(libGAP_ELM_PLIST(libGAP_CompFunctions,i));
        if ( n != 0 && libGAP_IsStringConv(n) ) {
            libGAP_Emit( "C_NEW_STRING( NameFunc[%d], %d, \"%S\" )\n",
                  i, strlen(libGAP_CSTR_STRING(n)), libGAP_CSTR_STRING(n) );
        }
        else {
            libGAP_Emit( "NameFunc[%d] = DefaultName;\n", i );
        }
        libGAP_Emit( "NamsFunc[%d] = 0;\n", i );
        libGAP_Emit( "NargFunc[%d] = %d;\n", i, libGAP_NARG_FUNC(libGAP_ELM_PLIST(libGAP_CompFunctions,i)));
    }
    libGAP_Emit( "\n/* create all the functions defined in this module */\n" );
    libGAP_Emit( "func1 = NewFunction(NameFunc[1],NargFunc[1],NamsFunc[1],HdlrFunc1);\n" );
    libGAP_Emit( "ENVI_FUNC( func1 ) = CurrLVars;\n" );
    libGAP_Emit( "CHANGED_BAG( CurrLVars );\n" );
    libGAP_Emit( "body1 = NewBag( T_BODY, NUMBER_HEADER_ITEMS_BODY*sizeof(Obj));\n" );
    libGAP_Emit( "BODY_FUNC( func1 ) = body1;\n" );
    libGAP_Emit( "CHANGED_BAG( func1 );\n");
    libGAP_Emit( "CALL_0ARGS( func1 );\n" );
    libGAP_Emit( "\n/* return success */\n" );
    libGAP_Emit( "return 0;\n" );
    libGAP_Emit( "\n}\n" );

    libGAP_Emit( "\n/* 'PostRestore' restore gvars, rnams, functions */\n" );
    libGAP_Emit( "static Int PostRestore ( StructInitInfo * module )\n" );
    libGAP_Emit( "{\n" );
    libGAP_Emit( "\n/* global variables used in handlers */\n" );
    for ( i = 1; i < libGAP_SIZE_OBJ(libGAP_CompInfoGVar)/sizeof(libGAP_UInt); i++ ) {
        if ( libGAP_CompGetUseGVar( i ) ) {
            libGAP_Emit( "G_%n = GVarName( \"%s\" );\n",
                   libGAP_NameGVar(i), libGAP_NameGVar(i) );
        }
    }
    libGAP_Emit( "\n/* record names used in handlers */\n" );
    for ( i = 1; i < libGAP_SIZE_OBJ(libGAP_CompInfoRNam)/sizeof(libGAP_UInt); i++ ) {
        if ( libGAP_CompGetUseRNam( i ) ) {
            libGAP_Emit( "R_%n = RNamName( \"%s\" );\n",
                  libGAP_NAME_RNAM(i), libGAP_NAME_RNAM(i) );
        }
    }
    libGAP_Emit( "\n/* information for the functions */\n" );
    for ( i = 1; i <= libGAP_CompFunctionsNr; i++ ) {
        n = libGAP_NAME_FUNC(libGAP_ELM_PLIST(libGAP_CompFunctions,i));
        if ( n == 0 || ! libGAP_IsStringConv(n) ) {
            libGAP_Emit( "NameFunc[%d] = DefaultName;\n", i );
        }
        libGAP_Emit( "NamsFunc[%d] = 0;\n", i );
        libGAP_Emit( "NargFunc[%d] = %d;\n", i, libGAP_NARG_FUNC(libGAP_ELM_PLIST(libGAP_CompFunctions,i)));
    }
    libGAP_Emit( "\n/* return success */\n" );
    libGAP_Emit( "return 0;\n" );
    libGAP_Emit( "\n}\n" );
    libGAP_Emit( "\n" );

    /* emit the initialization code                                        */
    libGAP_Emit( "\n/* <name> returns the description of this module */\n" );
    libGAP_Emit( "static StructInitInfo module = {\n" );
    if ( ! strcmp( "Init_Dynamic", name ) ) {
        libGAP_Emit( "/* type        = */ %d,\n",     libGAP_MODULE_DYNAMIC ); 
    }
    else {
        libGAP_Emit( "/* type        = */ %d,\n",     libGAP_MODULE_STATIC ); 
    }
    libGAP_Emit( "/* name        = */ \"%C\",\n", magic2 );
    libGAP_Emit( "/* revision_c  = */ %d,\n",     0 );
    libGAP_Emit( "/* revision_h  = */ %d,\n",     0 );
    libGAP_Emit( "/* version     = */ %d,\n",     0 );
    libGAP_Emit( "/* crc         = */ %d,\n",     magic1 );
    libGAP_Emit( "/* initKernel  = */ InitKernel,\n" );
    libGAP_Emit( "/* initLibrary = */ InitLibrary,\n" );
    libGAP_Emit( "/* checkInit   = */ 0,\n" );
    libGAP_Emit( "/* preSave     = */ 0,\n" );
    libGAP_Emit( "/* postSave    = */ 0,\n" );
    libGAP_Emit( "/* postRestore = */ PostRestore\n" );
    libGAP_Emit( "};\n" );
    libGAP_Emit( "\n" );
    libGAP_Emit( "StructInitInfo * %n ( void )\n", name );
    libGAP_Emit( "{\n" );
    libGAP_Emit( "return &module;\n" );
    libGAP_Emit( "}\n" );
    libGAP_Emit( "\n/* compiled code ends here */\n" );

    /* close the output file                                               */
    libGAP_SyNrCols = col;
    libGAP_CloseOutput();

    /* return success                                                      */
    return libGAP_CompFunctionsNr;
}


/****************************************************************************
**
*F  FuncCOMPILE_FUNC( <self>, <output>, <func>, <name>, <magic1>, <magic2> )
*/
libGAP_Obj libGAP_FuncCOMPILE_FUNC (
    libGAP_Obj                 self,
    libGAP_Obj                 arg )
{
    libGAP_Obj                 output;
    libGAP_Obj                 func;
    libGAP_Obj                 name;
    libGAP_Obj                 magic1;
    libGAP_Obj                 magic2;
    libGAP_Int                 nr;
    libGAP_Int                 len;

    /* unravel the arguments                                               */
    len = libGAP_LEN_LIST(arg); 
    if ( len < 5 ) {
        libGAP_ErrorQuit( "usage: COMPILE_FUNC( <output>, <func>, <name>, %s",
                   (libGAP_Int)"<magic1>, <magic2>, ... )", 0 );
        return 0;
    }
    output = libGAP_ELM_LIST( arg, 1 );
    func   = libGAP_ELM_LIST( arg, 2 );
    name   = libGAP_ELM_LIST( arg, 3 );
    magic1 = libGAP_ELM_LIST( arg, 4 );
    magic2 = libGAP_ELM_LIST( arg, 5 );

    /* check the arguments                                                 */
    if ( ! libGAP_IsStringConv( output ) ) {
        libGAP_ErrorQuit("CompileFunc: <output> must be a string",0L,0L);
    }
    if ( libGAP_TNUM_OBJ(func) != libGAP_T_FUNCTION ) {
        libGAP_ErrorQuit("CompileFunc: <func> must be a function",0L,0L);
    }
    if ( ! libGAP_IsStringConv( name ) ) {
        libGAP_ErrorQuit("CompileFunc: <name> must be a string",0L,0L);
    }
    if ( ! libGAP_IS_INTOBJ(magic1) ) {
        libGAP_ErrorQuit("CompileFunc: <magic1> must be an integer",0L,0L);
    }
    if ( ! libGAP_IsStringConv(magic2) ) {
        libGAP_ErrorQuit("CompileFunc: <magic2> must be a string",0L,0L);
    }

    /* possible optimiser flags                                            */
    libGAP_CompFastIntArith        = 1;
    libGAP_CompFastPlainLists      = 1;
    libGAP_CompFastListFuncs       = 1;
    libGAP_CompCheckTypes          = 1;
    libGAP_CompCheckListElements   = 1;
    libGAP_CompCheckPosObjElements = 0;

    if ( 6 <= len ) {
        libGAP_CompFastIntArith        = libGAP_EQ( libGAP_ELM_LIST( arg,  6 ), libGAP_True );
    }
    if ( 7 <= len ) {
        libGAP_CompFastPlainLists      = libGAP_EQ( libGAP_ELM_LIST( arg,  7 ), libGAP_True );
    }
    if ( 8 <= len ) {
        libGAP_CompFastListFuncs       = libGAP_EQ( libGAP_ELM_LIST( arg,  8 ), libGAP_True );
    }
    if ( 9 <= len ) {
        libGAP_CompCheckTypes          = libGAP_EQ( libGAP_ELM_LIST( arg,  9 ), libGAP_True );
    }
    if ( 10 <= len ) {
        libGAP_CompCheckListElements   = libGAP_EQ( libGAP_ELM_LIST( arg, 10 ), libGAP_True );
    }
    if ( 11 <= len ) {
        libGAP_CompCheckPosObjElements = libGAP_EQ( libGAP_ELM_LIST( arg, 11 ), libGAP_True );
    }
    
    /* compile the function                                                */
    nr = libGAP_CompileFunc(
        libGAP_CSTR_STRING(output), func, libGAP_CSTR_STRING(name),
        libGAP_INT_INTOBJ(magic1), libGAP_CSTR_STRING(magic2) );


    /* return the result                                                   */
    return libGAP_INTOBJ_INT(nr);
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "COMPILE_FUNC", -1, "arg",
      libGAP_FuncCOMPILE_FUNC, "src/compiler.c:COMPILE_FUNC" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int                 i;              /* loop variable                   */

    libGAP_CompFastIntArith = 1;
    libGAP_CompFastListFuncs = 1;
    libGAP_CompFastPlainLists = 1;
    libGAP_CompCheckTypes = 1;
    libGAP_CompCheckListElements = 1;
    libGAP_CompCheckPosObjElements = 0;
    libGAP_CompPass = 0;
    
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* announce the global variables                                       */
    libGAP_InitGlobalBag( &libGAP_CompInfoGVar,  "src/compiler.c:CompInfoGVar"  );
    libGAP_InitGlobalBag( &libGAP_CompInfoRNam,  "src/compiler.c:CompInfoRNam"  );
    libGAP_InitGlobalBag( &libGAP_CompFunctions, "src/compiler.c:CompFunctions" );

    /* enter the expression compilers into the table                       */
    for ( i = 0; i < 256; i++ ) {
        libGAP_CompExprFuncs[ i ] = libGAP_CompUnknownExpr;
    }

    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_0ARGS  ] = libGAP_CompFunccall0to6Args;
    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_1ARGS  ] = libGAP_CompFunccall0to6Args;
    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_2ARGS  ] = libGAP_CompFunccall0to6Args;
    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_3ARGS  ] = libGAP_CompFunccall0to6Args;
    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_4ARGS  ] = libGAP_CompFunccall0to6Args;
    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_5ARGS  ] = libGAP_CompFunccall0to6Args;
    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_6ARGS  ] = libGAP_CompFunccall0to6Args;
    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_XARGS  ] = libGAP_CompFunccallXArgs;
    libGAP_CompExprFuncs[ libGAP_T_FUNC_EXPR       ] = libGAP_CompFuncExpr;

    libGAP_CompExprFuncs[ libGAP_T_OR              ] = libGAP_CompOr;
    libGAP_CompExprFuncs[ libGAP_T_AND             ] = libGAP_CompAnd;
    libGAP_CompExprFuncs[ libGAP_T_NOT             ] = libGAP_CompNot;
    libGAP_CompExprFuncs[ libGAP_T_EQ              ] = libGAP_CompEq;
    libGAP_CompExprFuncs[ libGAP_T_NE              ] = libGAP_CompNe;
    libGAP_CompExprFuncs[ libGAP_T_LT              ] = libGAP_CompLt;
    libGAP_CompExprFuncs[ libGAP_T_GE              ] = libGAP_CompGe;
    libGAP_CompExprFuncs[ libGAP_T_GT              ] = libGAP_CompGt;
    libGAP_CompExprFuncs[ libGAP_T_LE              ] = libGAP_CompLe;
    libGAP_CompExprFuncs[ libGAP_T_IN              ] = libGAP_CompIn;

    libGAP_CompExprFuncs[ libGAP_T_SUM             ] = libGAP_CompSum;
    libGAP_CompExprFuncs[ libGAP_T_AINV            ] = libGAP_CompAInv;
    libGAP_CompExprFuncs[ libGAP_T_DIFF            ] = libGAP_CompDiff;
    libGAP_CompExprFuncs[ libGAP_T_PROD            ] = libGAP_CompProd;
    libGAP_CompExprFuncs[ libGAP_T_INV             ] = libGAP_CompInv;
    libGAP_CompExprFuncs[ libGAP_T_QUO             ] = libGAP_CompQuo;
    libGAP_CompExprFuncs[ libGAP_T_MOD             ] = libGAP_CompMod;
    libGAP_CompExprFuncs[ libGAP_T_POW             ] = libGAP_CompPow;

    libGAP_CompExprFuncs[ libGAP_T_INTEXPR         ] = libGAP_CompIntExpr;
    libGAP_CompExprFuncs[ libGAP_T_INT_EXPR        ] = libGAP_CompIntExpr;
    libGAP_CompExprFuncs[ libGAP_T_TRUE_EXPR       ] = libGAP_CompTrueExpr;
    libGAP_CompExprFuncs[ libGAP_T_FALSE_EXPR      ] = libGAP_CompFalseExpr;
    libGAP_CompExprFuncs[ libGAP_T_CHAR_EXPR       ] = libGAP_CompCharExpr;
    libGAP_CompExprFuncs[ libGAP_T_PERM_EXPR       ] = libGAP_CompPermExpr;
    libGAP_CompExprFuncs[ libGAP_T_PERM_CYCLE      ] = libGAP_CompUnknownExpr;
    libGAP_CompExprFuncs[ libGAP_T_LIST_EXPR       ] = libGAP_CompListExpr;
    libGAP_CompExprFuncs[ libGAP_T_LIST_TILD_EXPR  ] = libGAP_CompListTildeExpr;
    libGAP_CompExprFuncs[ libGAP_T_RANGE_EXPR      ] = libGAP_CompRangeExpr;
    libGAP_CompExprFuncs[ libGAP_T_STRING_EXPR     ] = libGAP_CompStringExpr;
    libGAP_CompExprFuncs[ libGAP_T_REC_EXPR        ] = libGAP_CompRecExpr;
    libGAP_CompExprFuncs[ libGAP_T_REC_TILD_EXPR   ] = libGAP_CompRecTildeExpr;

    libGAP_CompExprFuncs[ libGAP_T_REFLVAR         ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR        ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_01     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_02     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_03     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_04     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_05     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_06     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_07     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_08     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_09     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_10     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_11     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_12     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_13     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_14     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_15     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_LVAR_16     ] = libGAP_CompRefLVar;
    libGAP_CompExprFuncs[ libGAP_T_ISB_LVAR        ] = libGAP_CompIsbLVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_HVAR        ] = libGAP_CompRefHVar;
    libGAP_CompExprFuncs[ libGAP_T_ISB_HVAR        ] = libGAP_CompIsbHVar;
    libGAP_CompExprFuncs[ libGAP_T_REF_GVAR        ] = libGAP_CompRefGVar;
    libGAP_CompExprFuncs[ libGAP_T_ISB_GVAR        ] = libGAP_CompIsbGVar;

    libGAP_CompExprFuncs[ libGAP_T_ELM_LIST        ] = libGAP_CompElmList;
    libGAP_CompExprFuncs[ libGAP_T_ELMS_LIST       ] = libGAP_CompElmsList;
    libGAP_CompExprFuncs[ libGAP_T_ELM_LIST_LEV    ] = libGAP_CompElmListLev;
    libGAP_CompExprFuncs[ libGAP_T_ELMS_LIST_LEV   ] = libGAP_CompElmsListLev;
    libGAP_CompExprFuncs[ libGAP_T_ISB_LIST        ] = libGAP_CompIsbList;
    libGAP_CompExprFuncs[ libGAP_T_ELM_REC_NAME    ] = libGAP_CompElmRecName;
    libGAP_CompExprFuncs[ libGAP_T_ELM_REC_EXPR    ] = libGAP_CompElmRecExpr;
    libGAP_CompExprFuncs[ libGAP_T_ISB_REC_NAME    ] = libGAP_CompIsbRecName;
    libGAP_CompExprFuncs[ libGAP_T_ISB_REC_EXPR    ] = libGAP_CompIsbRecExpr;

    libGAP_CompExprFuncs[ libGAP_T_ELM_POSOBJ      ] = libGAP_CompElmPosObj;
    libGAP_CompExprFuncs[ libGAP_T_ELMS_POSOBJ     ] = libGAP_CompElmsPosObj;
    libGAP_CompExprFuncs[ libGAP_T_ELM_POSOBJ_LEV  ] = libGAP_CompElmPosObjLev;
    libGAP_CompExprFuncs[ libGAP_T_ELMS_POSOBJ_LEV ] = libGAP_CompElmsPosObjLev;
    libGAP_CompExprFuncs[ libGAP_T_ISB_POSOBJ      ] = libGAP_CompIsbPosObj;
    libGAP_CompExprFuncs[ libGAP_T_ELM_COMOBJ_NAME ] = libGAP_CompElmComObjName;
    libGAP_CompExprFuncs[ libGAP_T_ELM_COMOBJ_EXPR ] = libGAP_CompElmComObjExpr;
    libGAP_CompExprFuncs[ libGAP_T_ISB_COMOBJ_NAME ] = libGAP_CompIsbComObjName;
    libGAP_CompExprFuncs[ libGAP_T_ISB_COMOBJ_EXPR ] = libGAP_CompIsbComObjExpr;

    libGAP_CompExprFuncs[ libGAP_T_FUNCCALL_OPTS   ] = libGAP_CompFunccallOpts;
    
    /* enter the boolean expression compilers into the table               */
    for ( i = 0; i < 256; i++ ) {
        libGAP_CompBoolExprFuncs[ i ] = libGAP_CompUnknownBool;
    }

    libGAP_CompBoolExprFuncs[ libGAP_T_OR              ] = libGAP_CompOrBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_AND             ] = libGAP_CompAndBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_NOT             ] = libGAP_CompNotBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_EQ              ] = libGAP_CompEqBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_NE              ] = libGAP_CompNeBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_LT              ] = libGAP_CompLtBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_GE              ] = libGAP_CompGeBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_GT              ] = libGAP_CompGtBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_LE              ] = libGAP_CompLeBool;
    libGAP_CompBoolExprFuncs[ libGAP_T_IN              ] = libGAP_CompInBool;

    /* enter the statement compilers into the table                        */
    for ( i = 0; i < 256; i++ ) {
        libGAP_CompStatFuncs[ i ] = libGAP_CompUnknownStat;
    }

    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_0ARGS  ] = libGAP_CompProccall0to6Args;
    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_1ARGS  ] = libGAP_CompProccall0to6Args;
    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_2ARGS  ] = libGAP_CompProccall0to6Args;
    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_3ARGS  ] = libGAP_CompProccall0to6Args;
    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_4ARGS  ] = libGAP_CompProccall0to6Args;
    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_5ARGS  ] = libGAP_CompProccall0to6Args;
    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_6ARGS  ] = libGAP_CompProccall0to6Args;
    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_XARGS  ] = libGAP_CompProccallXArgs;

    libGAP_CompStatFuncs[ libGAP_T_SEQ_STAT        ] = libGAP_CompSeqStat;
    libGAP_CompStatFuncs[ libGAP_T_SEQ_STAT2       ] = libGAP_CompSeqStat;
    libGAP_CompStatFuncs[ libGAP_T_SEQ_STAT3       ] = libGAP_CompSeqStat;
    libGAP_CompStatFuncs[ libGAP_T_SEQ_STAT4       ] = libGAP_CompSeqStat;
    libGAP_CompStatFuncs[ libGAP_T_SEQ_STAT5       ] = libGAP_CompSeqStat;
    libGAP_CompStatFuncs[ libGAP_T_SEQ_STAT6       ] = libGAP_CompSeqStat;
    libGAP_CompStatFuncs[ libGAP_T_SEQ_STAT7       ] = libGAP_CompSeqStat;
    libGAP_CompStatFuncs[ libGAP_T_IF              ] = libGAP_CompIf;
    libGAP_CompStatFuncs[ libGAP_T_IF_ELSE         ] = libGAP_CompIf;
    libGAP_CompStatFuncs[ libGAP_T_IF_ELIF         ] = libGAP_CompIf;
    libGAP_CompStatFuncs[ libGAP_T_IF_ELIF_ELSE    ] = libGAP_CompIf;
    libGAP_CompStatFuncs[ libGAP_T_FOR             ] = libGAP_CompFor;
    libGAP_CompStatFuncs[ libGAP_T_FOR2            ] = libGAP_CompFor;
    libGAP_CompStatFuncs[ libGAP_T_FOR3            ] = libGAP_CompFor;
    libGAP_CompStatFuncs[ libGAP_T_FOR_RANGE       ] = libGAP_CompFor;
    libGAP_CompStatFuncs[ libGAP_T_FOR_RANGE2      ] = libGAP_CompFor;
    libGAP_CompStatFuncs[ libGAP_T_FOR_RANGE3      ] = libGAP_CompFor;
    libGAP_CompStatFuncs[ libGAP_T_WHILE           ] = libGAP_CompWhile;
    libGAP_CompStatFuncs[ libGAP_T_WHILE2          ] = libGAP_CompWhile;
    libGAP_CompStatFuncs[ libGAP_T_WHILE3          ] = libGAP_CompWhile;
    libGAP_CompStatFuncs[ libGAP_T_REPEAT          ] = libGAP_CompRepeat;
    libGAP_CompStatFuncs[ libGAP_T_REPEAT2         ] = libGAP_CompRepeat;
    libGAP_CompStatFuncs[ libGAP_T_REPEAT3         ] = libGAP_CompRepeat;
    libGAP_CompStatFuncs[ libGAP_T_BREAK           ] = libGAP_CompBreak;
    libGAP_CompStatFuncs[ libGAP_T_CONTINUE        ] = libGAP_CompContinue;
    libGAP_CompStatFuncs[ libGAP_T_RETURN_OBJ      ] = libGAP_CompReturnObj;
    libGAP_CompStatFuncs[ libGAP_T_RETURN_VOID     ] = libGAP_CompReturnVoid;

    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR        ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_01     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_02     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_03     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_04     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_05     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_06     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_07     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_08     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_09     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_10     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_11     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_12     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_13     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_14     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_15     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LVAR_16     ] = libGAP_CompAssLVar;
    libGAP_CompStatFuncs[ libGAP_T_UNB_LVAR        ] = libGAP_CompUnbLVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_HVAR        ] = libGAP_CompAssHVar;
    libGAP_CompStatFuncs[ libGAP_T_UNB_HVAR        ] = libGAP_CompUnbHVar;
    libGAP_CompStatFuncs[ libGAP_T_ASS_GVAR        ] = libGAP_CompAssGVar;
    libGAP_CompStatFuncs[ libGAP_T_UNB_GVAR        ] = libGAP_CompUnbGVar;

    libGAP_CompStatFuncs[ libGAP_T_ASS_LIST        ] = libGAP_CompAssList;
    libGAP_CompStatFuncs[ libGAP_T_ASSS_LIST       ] = libGAP_CompAsssList;
    libGAP_CompStatFuncs[ libGAP_T_ASS_LIST_LEV    ] = libGAP_CompAssListLev;
    libGAP_CompStatFuncs[ libGAP_T_ASSS_LIST_LEV   ] = libGAP_CompAsssListLev;
    libGAP_CompStatFuncs[ libGAP_T_UNB_LIST        ] = libGAP_CompUnbList;
    libGAP_CompStatFuncs[ libGAP_T_ASS_REC_NAME    ] = libGAP_CompAssRecName;
    libGAP_CompStatFuncs[ libGAP_T_ASS_REC_EXPR    ] = libGAP_CompAssRecExpr;
    libGAP_CompStatFuncs[ libGAP_T_UNB_REC_NAME    ] = libGAP_CompUnbRecName;
    libGAP_CompStatFuncs[ libGAP_T_UNB_REC_EXPR    ] = libGAP_CompUnbRecExpr;

    libGAP_CompStatFuncs[ libGAP_T_ASS_POSOBJ      ] = libGAP_CompAssPosObj;
    libGAP_CompStatFuncs[ libGAP_T_ASSS_POSOBJ     ] = libGAP_CompAsssPosObj;
    libGAP_CompStatFuncs[ libGAP_T_ASS_POSOBJ_LEV  ] = libGAP_CompAssPosObjLev;
    libGAP_CompStatFuncs[ libGAP_T_ASSS_POSOBJ_LEV ] = libGAP_CompAsssPosObjLev;
    libGAP_CompStatFuncs[ libGAP_T_UNB_POSOBJ      ] = libGAP_CompUnbPosObj;
    libGAP_CompStatFuncs[ libGAP_T_ASS_COMOBJ_NAME ] = libGAP_CompAssComObjName;
    libGAP_CompStatFuncs[ libGAP_T_ASS_COMOBJ_EXPR ] = libGAP_CompAssComObjExpr;
    libGAP_CompStatFuncs[ libGAP_T_UNB_COMOBJ_NAME ] = libGAP_CompUnbComObjName;
    libGAP_CompStatFuncs[ libGAP_T_UNB_COMOBJ_EXPR ] = libGAP_CompUnbComObjExpr;

    libGAP_CompStatFuncs[ libGAP_T_INFO            ] = libGAP_CompInfo;
    libGAP_CompStatFuncs[ libGAP_T_ASSERT_2ARGS    ] = libGAP_CompAssert2;
    libGAP_CompStatFuncs[ libGAP_T_ASSERT_3ARGS    ] = libGAP_CompAssert3;
    libGAP_CompStatFuncs[ libGAP_T_EMPTY           ] = libGAP_CompEmpty;

    libGAP_CompStatFuncs[ libGAP_T_PROCCALL_OPTS   ] = libGAP_CompProccallOpts;
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* get the identifiers of 'Length' and 'Add' (for inlining)            */
    libGAP_G_Length = libGAP_GVarName( "Length" );
    libGAP_G_Add    = libGAP_GVarName( "Add"    );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return libGAP_PostRestore( libGAP_module );
}


/****************************************************************************
**
*F  InitInfoCompiler() . . . . . . . . . . . . . . .  table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "compiler",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoCompiler ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  compiler.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/



