/****************************************************************************
**
*W  sctable.c                   GAP source                     Marcel Roelofs
**
**
*Y  Copyright (C)  1996,        CWI,        Amsterdam,        The Netherlands
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains a fast access function  for structure constants tables
**  and the multiplication of two elements using a structure constants table.
**
**  Structure constants tables in GAP have the following layout
**  
**        [ [ 1 ],
**          ...
**          [ i ],  ---> [ [ 1 ], ..., [ j ], ..., [ n ] ]
**          ...                          |
**          [ n ],                       |
**          flag,                        |  
**          zero ]                       V  
**                                       [ [ k        , ... ],
**                                         [ c_{ij}^k , ... ]  ]
**
**  where the two outer lists for i and j are full lists  of the dimension of
**  the underlying vectorspace,   and the lists for k and c_{ij}^k are stored 
**  sparsely, that is, only for those k with non-zero c_{ij}^k.
**  
**  The last two elements of the outermost list have a special meaning.
**
**  The flag is an integer that indicates  whether the product defined by the
**  table is commutative (+1) or anti-commutative (-1) or neither (0).
**
**  zero is the zero element of the coefficient ring/field of the algebra.
**
**  NOTE: most of the code consists of dimension- and type checks,  as a user
**        can fool around with SCTables as s/he likes. 
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "ariths.h"              /* basic arithmetic                */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "sctable.h"             /* structure constant table        */


/****************************************************************************
**

*F  SCTableEntry( <table>, <i>, <j>, <k> )  . . . .  entry of structure table
**
**  'SCTableEntry' returns the coefficient $c_{i,j}^{k}$ from the structure
**  constants table <table>.
*/
libGAP_Obj libGAP_SCTableEntryFunc;

libGAP_Obj libGAP_SCTableEntryHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 table,
    libGAP_Obj                 i,
    libGAP_Obj                 j,
    libGAP_Obj                 k )
{
    libGAP_Obj                 tmp;            /* temporary                       */
    libGAP_Obj                 basis;          /* basis  list                     */
    libGAP_Obj                 coeffs;         /* coeffs list                     */
    libGAP_Int                 dim;            /* dimension                       */
    libGAP_Int                 len;            /* length of basis/coeffs lists    */
    libGAP_Int                 l;              /* loop variable                   */

    /* check the table                                                     */
    if ( ! libGAP_IS_SMALL_LIST(table) ) {
        table = libGAP_ErrorReturnObj(
            "SCTableEntry: <table> must be a small list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(table), 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }
    dim = libGAP_LEN_LIST(table) - 2;
    if ( dim <= 0 ) {
        table = libGAP_ErrorReturnObj(
            "SCTableEntry: <table> must be a list with at least 3 elements",
            0L, 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* check <i>                                                           */
    if ( ! libGAP_IS_INTOBJ(i) || libGAP_INT_INTOBJ(i) <= 0 || dim < libGAP_INT_INTOBJ(i) ) {
        i = libGAP_ErrorReturnObj(
            "SCTableEntry: <i> must be an integer between 0 and %d",
            dim, 0L,
            "you can replace <i> via 'return <i>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* get and check the relevant row                                      */
    tmp = libGAP_ELM_LIST( table, libGAP_INT_INTOBJ(i) );
    if ( ! libGAP_IS_SMALL_LIST(tmp) || libGAP_LEN_LIST(tmp) != dim ) {
        table = libGAP_ErrorReturnObj(
            "SCTableEntry: <table>[%d] must be a list with %d elements",
            libGAP_INT_INTOBJ(i), dim,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );

    }

    /* check <j>                                                           */
    if ( ! libGAP_IS_INTOBJ(j) || libGAP_INT_INTOBJ(j) <= 0 || dim < libGAP_INT_INTOBJ(j) ) {
        j = libGAP_ErrorReturnObj(
            "SCTableEntry: <j> must be an integer between 0 and %d",
            dim, 0L,
            "you can replace <j> via 'return <j>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* get and check the basis and coefficients list                       */
    tmp = libGAP_ELM_LIST( tmp, libGAP_INT_INTOBJ(j) );
    if ( ! libGAP_IS_SMALL_LIST(tmp) || libGAP_LEN_LIST(tmp) != 2 ) {
        table = libGAP_ErrorReturnObj(
            "SCTableEntry: <table>[%d][%d] must be a basis/coeffs list",
            0L, 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* get and check the basis list                                        */
    basis = libGAP_ELM_LIST( tmp, 1 );
    if ( ! libGAP_IS_SMALL_LIST(basis) ) {
        table = libGAP_ErrorReturnObj(
            "SCTableEntry: <table>[%d][%d][1] must be a basis list",
            0L, 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* get and check the coeffs list                                       */
    coeffs = libGAP_ELM_LIST( tmp, 2 );
    if ( ! libGAP_IS_SMALL_LIST(coeffs) ) {
        table = libGAP_ErrorReturnObj(
            "SCTableEntry: <table>[%d][%d][2] must be a coeffs list",
            0L, 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* check that they have the same length                                */
    len = libGAP_LEN_LIST(basis);
    if ( libGAP_LEN_LIST(coeffs) != len ) {
        table = libGAP_ErrorReturnObj(
            "SCTableEntry: <table>[%d][%d][1], ~[2] must have equal length",
            0L, 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* check <k>                                                           */
    if ( ! libGAP_IS_INTOBJ(k) || libGAP_INT_INTOBJ(k) <= 0 || dim < libGAP_INT_INTOBJ(k) ) {
        k = libGAP_ErrorReturnObj(
            "SCTableEntry: <k> must be an integer between 0 and %d",
            dim, 0L,
            "you can replace <k> via 'return <k>;'" );
        return libGAP_SCTableEntryHandler( self, table, i, j, k );
    }

    /* look for the (i,j,k) entry                                          */
    for ( l = 1; l <= len; l++ ) {
        if ( libGAP_EQ( libGAP_ELM_LIST( basis, l ), k ) )
            break;
    }

    /* return the coefficient of zero                                      */
    if ( l <= len ) {
        return libGAP_ELM_LIST( coeffs, l );
    }
    else {
        return libGAP_ELM_LIST( table, dim+2 );
    }
}


/****************************************************************************
**
*F  SCTableProduct( <table>, <list1>, <list2> ) . product wrt structure table
**
**  'SCTableProduct'  returns the product   of  the two elements <list1>  and
**  <list2> with respect to the structure constants table <table>.
*/
void libGAP_SCTableProdAdd (
    libGAP_Obj                 res,
    libGAP_Obj                 coeff,
    libGAP_Obj                 basis_coeffs,
    libGAP_Int                 dim )
{
    libGAP_Obj                 basis;
    libGAP_Obj                 coeffs;
    libGAP_Int                 len;
    libGAP_Obj                 k;
    libGAP_Obj                 c1, c2;
    libGAP_Int                 l;

    basis  = libGAP_ELM_LIST( basis_coeffs, 1 );
    coeffs = libGAP_ELM_LIST( basis_coeffs, 2 );
    len = libGAP_LEN_LIST( basis );
    if ( libGAP_LEN_LIST( coeffs ) != len ) {
        libGAP_ErrorQuit("SCTableProduct: corrupted <table>",0L,0L);
    }
    for ( l = 1; l <= len; l++ ) {
        k = libGAP_ELM_LIST( basis, l );
        if ( ! libGAP_IS_INTOBJ(k) || libGAP_INT_INTOBJ(k) <= 0 || dim < libGAP_INT_INTOBJ(k) ) {
            libGAP_ErrorQuit("SCTableProduct: corrupted <table>",0L,0L);
        }
        c1 = libGAP_ELM_LIST( coeffs, l );
        c1 = libGAP_PROD( coeff, c1 );
        c2 = libGAP_ELM_PLIST( res, libGAP_INT_INTOBJ(k) );
        c2 = libGAP_SUM( c2, c1 );
        libGAP_SET_ELM_PLIST( res, libGAP_INT_INTOBJ(k), c2 );
        libGAP_CHANGED_BAG( res );
    }
}

libGAP_Obj libGAP_SCTableProductFunc;

libGAP_Obj libGAP_SCTableProductHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 table,
    libGAP_Obj                 list1,
    libGAP_Obj                 list2 )
{
    libGAP_Obj                 res;            /* result list                     */
    libGAP_Obj                 row;            /* one row of sc table             */
    libGAP_Obj                 zero;           /* zero from sc table              */
    libGAP_Obj                 ai, aj;         /* elements from list1             */
    libGAP_Obj                 bi, bj;         /* elements from list2             */
    libGAP_Obj                 c, c1, c2;      /* products of above               */
    libGAP_Int                 dim;            /* dimension of vectorspace        */
    libGAP_Int                 i, j;           /* loop variables                  */

    /* check the arguments a bit                                           */
    if ( ! libGAP_IS_SMALL_LIST(table) ) {
        table = libGAP_ErrorReturnObj(
            "SCTableProduct: <table> must be a list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(table), 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableProductHandler( self, table, list1, list2 );
    }
    dim = libGAP_LEN_LIST(table) - 2;
    if ( dim <= 0 ) {
        table = libGAP_ErrorReturnObj(
            "SCTableProduct: <table> must be a list with at least 3 elements",
            0L, 0L,
            "you can replace <table> via 'return <table>;'" );
        return libGAP_SCTableProductHandler( self, table, list1, list2 );
    }
    zero = libGAP_ELM_LIST( table, dim+2 );
    if ( ! libGAP_IS_SMALL_LIST(list1) || libGAP_LEN_LIST(list1) != dim ) {
        list1 = libGAP_ErrorReturnObj(
            "SCTableProduct: <list1> must be a list with %d elements",
            dim, 0L,
            "you can replace <list1> via 'return <list1>;'" );
        return libGAP_SCTableProductHandler( self, table, list1, list2 );
    }
    if ( ! libGAP_IS_SMALL_LIST(list2) || libGAP_LEN_LIST(list2) != dim ) {
        list2 = libGAP_ErrorReturnObj(
            "SCTableProduct: <list2> must be a list with %d elements",
            dim, 0L,
            "you can replace <list2> via 'return <list2>;'" );
        return libGAP_SCTableProductHandler( self, table, list1, list2 );
    }

    /* make the result list                                                */
    res = libGAP_NEW_PLIST( libGAP_T_PLIST, dim );
    libGAP_SET_LEN_PLIST( res, dim );
    for ( i = 1; i <= dim; i++ ) {
        libGAP_SET_ELM_PLIST( res, i, zero );
    }
    libGAP_CHANGED_BAG( res );

    /* general case                                                        */
    if      ( libGAP_EQ( libGAP_ELM_LIST( table, dim+1 ), libGAP_INTOBJ_INT(0) ) ) {
        for ( i = 1; i <= dim; i++ ) {
            ai = libGAP_ELM_LIST( list1, i );
            if ( libGAP_EQ( ai, zero ) )  continue;
            row = libGAP_ELM_LIST( table, i );
            for ( j = 1; j <= dim; j++ ) {
                bj = libGAP_ELM_LIST( list2, j );
                if ( libGAP_EQ( bj, zero ) )  continue;
                c = libGAP_PROD( ai, bj );
                if ( ! libGAP_EQ( c, zero ) ) {
                    libGAP_SCTableProdAdd( res, c, libGAP_ELM_LIST( row, j ), dim );
                }
            }
        }
    }

    /* commutative case                                                    */
    else if ( libGAP_EQ( libGAP_ELM_LIST( table, dim+1 ), libGAP_INTOBJ_INT(1) ) ) {
        for ( i = 1; i <= dim; i++ ) {
            ai = libGAP_ELM_LIST( list1, i );
            bi = libGAP_ELM_LIST( list2, i );
            if ( libGAP_EQ( ai, zero ) && libGAP_EQ( bi, zero ) )  continue;
            row = libGAP_ELM_LIST( table, i );
            c = libGAP_PROD( ai, bi );
            if ( ! libGAP_EQ( c, zero ) ) {
                libGAP_SCTableProdAdd( res, c, libGAP_ELM_LIST( row, i ), dim );
            }
            for ( j = i+1; j <= dim; j++ ) {
                bj = libGAP_ELM_LIST( list2, j );
                aj = libGAP_ELM_LIST( list1, j );
                if ( libGAP_EQ( aj, zero ) && libGAP_EQ( bj, zero ) )  continue;
                c1 = libGAP_PROD( ai, bj );
                c2 = libGAP_PROD( aj, bi );
                c = libGAP_SUM( c1, c2 );
                if ( ! libGAP_EQ( c, zero ) ) {
                    libGAP_SCTableProdAdd( res, c, libGAP_ELM_LIST( row, j ), dim );
                }
            }
        }
    }

    /* anticommutative case                                                */
    else if ( libGAP_EQ( libGAP_ELM_LIST( table, dim+1 ), libGAP_INTOBJ_INT(-1) ) ) {
        for ( i = 1; i <= dim; i++ ) {
            ai = libGAP_ELM_LIST( list1, i );
            bi = libGAP_ELM_LIST( list2, i );
            if ( libGAP_EQ( ai, zero ) && libGAP_EQ( bi, zero ) )  continue;
            row = libGAP_ELM_LIST( table, i );
            for ( j = i+1; j <= dim; j++ ) {
                bj = libGAP_ELM_LIST( list2, j );
                aj = libGAP_ELM_LIST( list1, j ); 
                if ( libGAP_EQ( aj, zero ) && libGAP_EQ( bj, zero ) )  continue;
                c1 = libGAP_PROD( ai, bj );
                c2 = libGAP_PROD( aj, bi );
                c = libGAP_DIFF( c1, c2 );
                if ( ! libGAP_EQ( c, zero ) ) {
                    libGAP_SCTableProdAdd( res, c, libGAP_ELM_LIST( row, j ), dim );
                }
            }
        }
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "SC_TABLE_ENTRY", 4, "table, i, j, k",
      libGAP_SCTableEntryHandler, "src/sctable.c:SC_TABLE_ENTRY" },

    { "SC_TABLE_PRODUCT", 3, "table, list1, list2",
      libGAP_SCTableProductHandler, "src/sctable.c:SC_TABLE_PRODUCT" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoSCTable() . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "sctable",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoSCTable ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  sctable.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/



