/****************************************************************************
**
*W  objccoll.c                  GAP source                      Werner Nickel
**
**
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file  contains  the collection functions of  combinatorial collectors
**  for finite p-groups.  The code in this file  is an extension to the single
**  collector module.  All necessary initialisations  are done in that module.
**  The interface to a combinatorial collection function is identical with the
**  interface to the corresponding single collector function.
**
*/
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gvars.h"               /* global variables                */
#include        "gap.h"                 /* error handling, initialisation  */

#include        "calls.h"               /* generic call mechanism          */

#include        "records.h"             /* generic records                 */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "code.h"                /* coder                           */

#include        "objfgelm.h"            /* objects of free groups          */

#include        "objscoll.h"            /* single collector                */

#include        "objccoll.h"            /* combinatorial collector         */

/****************************************************************************
**
*F  C8Bits_CombiCollectWord( <sc>, <vv>, <w> )
**
**  If a stack overflow occurs, we simply stop and return false.
**
**  SC_PUSH_GEN( gen, exp )
**    push a generator <gen>  with exponent <exp> onto the stack.
**
**  SC_PUSH_WORD( word, exp )
**    push <word>  with global exponent <exp>  into the stack, the macro uses
**    <word> and <exp> only once.
**
**  SC_POP_WORD()
**    remove topmost word from stack
*/
#define libGAP_SC_PUSH_WORD( word, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(word); \
    *++lw = *nw + (libGAP_INT_INTOBJ((((libGAP_Obj*)(*nw))[-1])) - 1); \
    *++pw = *nw; \
    *++ew = (**pw) & expm; \
    *++ge = exp

#define libGAP_SC_PUSH_GEN( gen, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(gen); \
    *++lw = *nw; \
    *++pw = *nw; \
    *++ew = exp; \
    *++ge = 1

#define libGAP_SC_POP_WORD() \
    sp--;  nw--;  lw--;  pw--;  ew--;  ge--

/****************************************************************************
**
**  The following functions are  used to add  a word into the exponent vector
**  without collection.  There are three different cases that can occur:
**
**  Add   a word  into  the exponent  vector.   Here we   can  use the global
**  exponent.
**
**  Add  a commutator   into the exponent  vector.   In  this  case the first
**  generator in the conjugate has to be skipped.  Here we can use the global
**  exponent.
**
**  Add part  of a word  into the  exponent vector.   Here  we cannot use the
**  global exponent because the beginning of  the word might not commute with
**  the rest.
**/
static void libGAP_C8Bits_AddWordIntoExpVec( libGAP_Int *v, libGAP_UInt1 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt1 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C8Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt1*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

static void libGAP_C8Bits_AddCommIntoExpVec( libGAP_Int *v, libGAP_UInt1 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt1 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;

    /* Skip the first generator because we need the commutator here.  */
    w++;
    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C8Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt1*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

static void libGAP_C8Bits_AddPartIntoExpVec( libGAP_Int *v, libGAP_UInt1 *w, libGAP_UInt1 *wend,
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_Int        i;
    libGAP_Int        ex;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm);     /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C8Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt1*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

libGAP_Int libGAP_C8Bits_CombiCollectWord ( libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    libGAP_Int         ebits;      /* number of bits in the exponent              */
    libGAP_UInt        expm;       /* unsigned exponent mask                      */
    libGAP_UInt        exps;       /* sign exponent mask                          */

    libGAP_Obj         vnw;        /* word stack                                  */
    libGAP_UInt1 **    nw;         /* address of <vnw>                            */
    libGAP_Obj         vlw;        /* last syllable stack                         */
    libGAP_UInt1 **    lw;         /* address of <vlw>                            */
    libGAP_Obj         vpw;        /* current syllable stack                      */
    libGAP_UInt1 **    pw;         /* address of <vpw>                            */
    libGAP_Obj         vew;        /* unprocessed exponent stack                  */
    libGAP_UInt1 *     ew;         /* address of <vew>                            */
    libGAP_Obj         vge;        /* global exponent stack                       */
    libGAP_Int *       ge;         /* address of <vge>                            */

    libGAP_Obj         vpow;       /* rhs of power relations                      */
    libGAP_Int         lpow;       /* length of <vpow>                            */
    libGAP_Obj *       pow;        /* address of <vpow>                           */

    libGAP_Obj         vcnj;       /* rhs of conjugate relations                  */
    libGAP_Int         lcnj;       /* length of <vcnj>                            */
    libGAP_Obj *       cnj;        /* address of <vcnj>                           */

    libGAP_Obj *       avc;        /* address of the avector                      */
    libGAP_Obj *       avc2;       /* address of the avector 2                    */
    libGAP_Obj *       wt;         /* address of the weights array                */
    libGAP_Obj *       gns;        /* address of the list of generators           */
    libGAP_Obj *       ro;         /* address of the list of relative orders      */
    libGAP_Obj *       inv;        /* address of the list of inverses             */

    libGAP_Int *       v;          /* address of <vv>                             */

    libGAP_Int         max;        /* maximal stack size                          */
    libGAP_Int         sp;         /* stack pointer                               */
    libGAP_Int         i, j;       /* loop variable                               */
    libGAP_Int         gn;         /* current generator number                    */
    libGAP_Int         ex;         /* current exponent                            */
    libGAP_Int         cl;         /* p-class of the collector                    */
    libGAP_Int         p;          /* the prime                                   */

    libGAP_Obj         tmp;        /* temporary obj for power                     */

    libGAP_Int         resized = 0;/* indicates whether a Resize() happend        */

    /* if <w> is the identity return now                                   */
    if ( libGAP_NPAIRS_WORD(w) == 0 ) {
        return libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* get the exponent sign masks                                         */
    exps = 1UL << (ebits-1);

    /* <nw> contains the stack of words to insert                          */
    vnw = libGAP_SC_NW_STACK(sc);

    /* <lw> contains the word end of the word in <nw>                      */
    vlw = libGAP_SC_LW_STACK(sc);

    /* <pw> contains the position of the word in <nw> to look at           */
    vpw = libGAP_SC_PW_STACK(sc);

    /* <ew> contains the unprocessed exponents at position <pw>            */
    vew = libGAP_SC_EW_STACK(sc);

    /* <ge> contains the global exponent of the word                       */
    vge = libGAP_SC_GE_STACK(sc);

    /* get the maximal stack size                                          */
    max = libGAP_SC_MAX_STACK_SIZE(sc);

    /* ensure that the stacks are large enough                             */
    if ( libGAP_SIZE_OBJ(vnw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vnw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vnw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vlw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vlw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vlw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vpw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vpw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vpw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vew)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vew, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vew, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vge)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vge, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vge, libGAP_T_STRING );
        resized = 1;
    }
    if( resized ) return -1;

    /* from now on we use addresses instead of handles most of the time    */
    v  = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    nw = (libGAP_UInt1**)libGAP_ADDR_OBJ(vnw);
    lw = (libGAP_UInt1**)libGAP_ADDR_OBJ(vlw);
    pw = (libGAP_UInt1**)libGAP_ADDR_OBJ(vpw);
    ew = (libGAP_UInt1*)libGAP_ADDR_OBJ(vew);
    ge = (libGAP_Int*)libGAP_ADDR_OBJ(vge);

    /* conjugates, powers, order, generators, avector, inverses            */
    vpow = libGAP_SC_POWERS(sc);
    lpow = libGAP_LEN_PLIST(vpow);
    pow  = libGAP_ADDR_OBJ(vpow);

    vcnj = libGAP_SC_CONJUGATES(sc);
    lcnj = libGAP_LEN_PLIST(vcnj);
    (void) lcnj; /* please compiler -- lcnj not actually used */
    cnj  = libGAP_ADDR_OBJ(vcnj);

    avc = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR(sc) );
    gns = libGAP_ADDR_OBJ( libGAP_SC_RWS_GENERATORS(sc) );

    cl   = libGAP_INT_INTOBJ( libGAP_SC_CLASS(sc) );
    wt   = libGAP_ADDR_OBJ( libGAP_SC_WEIGHTS(sc) );
    avc2 = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR2(sc) );

    ro  = libGAP_ADDR_OBJ( libGAP_SC_RELATIVE_ORDERS(sc) );
    p   = libGAP_INT_INTOBJ(ro[1]);
    inv = libGAP_ADDR_OBJ( libGAP_SC_INVERSES(sc) );

    /* initalize the stack with <w>                                        */
    sp = 0;
    libGAP_SC_PUSH_WORD( w, 1 );

    /* run until the stack is empty                                        */
    while ( 0 < sp ) {

        /* if <ew> is negative use inverse                                 */
        if ( *ew & exps ) {
            gn = ((**pw) >> ebits) + 1;
            ex = ( *ew & (exps-1) ) - exps;
            *ew = 0;
            libGAP_SC_PUSH_WORD( inv[gn], -ex );
        }

        /* if <ew> is zero get next syllable                               */
        else if ( 0 == *ew ) {

            /* if <pw> has reached <lw> get next & reduce globale exponent */
            if ( *pw == *lw ) {

                /* if the globale exponent is greater one reduce it        */
                if ( 1 < *ge ) {
                    (*ge)--;
                    *pw = *nw;
                    *ew = (**pw) & expm;
                }

                /* otherwise get the next word from the stack              */
                else {
                    libGAP_SC_POP_WORD();

                }
            }

            /* otherwise set <ew> to exponent of next syllable             */
            else {
                (*pw)++;
                *ew = (**pw) & expm;
            }
        }

        /* now move the next generator/word to the correct position        */
        else {

            /* get generator number                                        */
            gn = ((**pw) >> ebits) + 1;

            /* collect a single generator on the stack                     */
            if( *lw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              v[gn] += *ew * *ge;
              *ew = *ge = 0;
              if ( p <= v[gn] ) {
                ex = v[gn] / p;
                v[gn] -= ex * p;
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_C8Bits_AddWordIntoExpVec( 
                      v, (libGAP_UInt1*)libGAP_DATA_WORD(pow[gn]), ex, 
                      ebits, expm, p, pow, lpow  );
                }
              }

              continue;
            }

            /* collect a whole word exponent pair                          */
            else if( sp > 1 && *pw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              libGAP_C8Bits_AddWordIntoExpVec( 
                   v, *pw, *ge, ebits, expm, p, pow, lpow  );
              *pw = *lw;
              *ew = *ge = 0;

              continue;
            }

            /* collect the rest of a word                                  */
            else if( sp > 1 && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              libGAP_C8Bits_AddPartIntoExpVec( 
                   v, *pw, *lw, ebits, expm, p, pow, lpow  );
              *pw = *lw;
              *ew = 0;

              continue;
            }

            else if( sp > 1 && 3*libGAP_INT_INTOBJ(wt[gn]) > cl ) {
                /* Collect <gn>^<ew> without stacking commutators.  
                   This is step 6 in (Vaughan-Lee 1990).                   */
                i = libGAP_INT_INTOBJ(avc[gn]);
                for ( ;  gn < i;  i-- ) {
                    if ( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) ) {
                            libGAP_C8Bits_AddCommIntoExpVec( 
                                v, (libGAP_UInt1*)libGAP_DATA_WORD(tmp), v[i] * (*ew),
                                ebits, expm, p, pow, lpow );
                        }
                    }
                }
                
                v[gn] += (*ew);
                (*ew) = 0;

                /* If the exponent is too big, we have to stack up the 
                   entries in the exponent vector.                         */
                if ( p <= v[gn] ) {
                    ex  = v[gn] / p;
                    v[gn] -= ex * p;
                    if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                        /* stack the exponent vector first. */
                        i = libGAP_INT_INTOBJ(avc[gn]);
                        for ( ; gn < i;  i-- ) {
                            if ( v[i] ) {
                                libGAP_SC_PUSH_GEN( gns[i], v[i] );
                                v[i] = 0;
                            }
                        }
                        libGAP_C8Bits_AddWordIntoExpVec( 
                             v, (libGAP_UInt1*)libGAP_DATA_WORD(pow[i]), ex,
                             ebits, expm, p, pow, lpow  );
                    }
                }
            }
            /* we have to move <gn> step by step                           */
            else {
              
              (*ew)--;
              v[gn]++;

              i = libGAP_INT_INTOBJ(avc[gn]);
              j = libGAP_INT_INTOBJ(avc2[gn]);
              if( sp > 1 ) {
                  /* Do combinatorial collection as far as possible.       */
                  for( ; j < i; i-- )
                      if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                          tmp = libGAP_ELM_PLIST( cnj[i], gn );
                          if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                              libGAP_C8Bits_AddCommIntoExpVec( 
                                  v, (libGAP_UInt1*)libGAP_DATA_WORD(tmp), v[i],
                                  ebits, expm, p, pow, lpow );
                      }
              }

              /* Find the first position in v from where on ordinary
                 collection  has to be applied.                            */
              for( ; gn < i; i-- )
                  if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                      tmp = libGAP_ELM_PLIST( cnj[i], gn );
                      if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                        break;
                  }

              /* Stack up this part of v if we run through the next 
                 for-loop or if a power relation will be applied           */ 
              if( gn < i || (p <= v[gn] &&
                  gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn])) ) {
                j = libGAP_INT_INTOBJ(avc[gn]);
                for( ; i < j; j-- )
                  if( v[j] ) {
                    libGAP_SC_PUSH_GEN( gns[j], v[j] );
                    v[j] = 0;
                  }
              }

                /* We finish with ordinary collection from the left        */
                for ( ;  gn < i;  i-- ) {
                    if ( v[i] ) {
                      if ( libGAP_LEN_PLIST(cnj[i]) < gn ) {
                        libGAP_SC_PUSH_GEN( gns[i], v[i] );
                      }
                      else {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp == 0 || libGAP_NPAIRS_WORD(tmp) == 0 ) {
                          libGAP_SC_PUSH_GEN( gns[i], v[i] );
                        }
                        else {
                          libGAP_SC_PUSH_WORD( tmp, v[i] );
                        }
                      }
                      v[i] = 0;
                    }
                }
            }

            /* check that the exponent is not too big                      */
            if ( p <= v[gn] ) {
                i = v[gn] / p;
                v[gn] -= i * p;
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_SC_PUSH_WORD( pow[gn], i );
                }
            }
        }
    }
    return libGAP_SC_NUMBER_RWS_GENERATORS(sc);
}
#undef libGAP_SC_PUSH_WORD
#undef libGAP_SC_PUSH_GEN
#undef libGAP_SC_POP_WORD

/****************************************************************************
**
*F  C16Bits_CombiCollectWord( <sc>, <vv>, <w> )
**
**  If a stack overflow occurs, we simply stop and return false.
**
**  SC_PUSH_GEN( gen, exp )
**    push a generator <gen>  with exponent <exp> onto the stack.
**
**  SC_PUSH_WORD( word, exp )
**    push <word>  with global exponent <exp>  into the stack, the macro uses
**    <word> and <exp> only once.
**
**  SC_POP_WORD()
**    remove topmost word from stack
*/
#define libGAP_SC_PUSH_WORD( word, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(word); \
    *++lw = *nw + (libGAP_INT_INTOBJ((((libGAP_Obj*)(*nw))[-1])) - 1); \
    *++pw = *nw; \
    *++ew = (**pw) & expm; \
    *++ge = exp

#define libGAP_SC_PUSH_GEN( gen, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(gen); \
    *++lw = *nw; \
    *++pw = *nw; \
    *++ew = exp; \
    *++ge = 1

#define libGAP_SC_POP_WORD() \
    sp--;  nw--;  lw--;  pw--;  ew--;  ge--

/****************************************************************************
**
**  The following functions are  used to add  a word into the exponent vector
**  without collection.  There are three different cases that can occur:
**
**  Add   a word  into  the exponent  vector.   Here we   can  use the global
**  exponent.
**
**  Add  a commutator   into the exponent  vector.   In  this  case the first
**  generator in the conjugate has to be skipped.  Here we can use the global
**  exponent.
**
**  Add part  of a word  into the  exponent vector.   Here  we cannot use the
**  global exponent because the beginning of  the word might not commute with
**  the rest.
**/
static void libGAP_C16Bits_AddWordIntoExpVec( libGAP_Int *v, libGAP_UInt2 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt2 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C16Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt2*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

static void libGAP_C16Bits_AddCommIntoExpVec( libGAP_Int *v, libGAP_UInt2 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt2 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;

    /* Skip the first generator because we need the commutator here.  */
    w++;
    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C16Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt2*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

static void libGAP_C16Bits_AddPartIntoExpVec( libGAP_Int *v, libGAP_UInt2 *w, libGAP_UInt2 *wend,
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_Int        i;
    libGAP_Int        ex;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm);     /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C16Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt2*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

libGAP_Int libGAP_C16Bits_CombiCollectWord ( libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    libGAP_Int         ebits;      /* number of bits in the exponent              */
    libGAP_UInt        expm;       /* unsigned exponent mask                      */
    libGAP_UInt        exps;       /* sign exponent mask                          */

    libGAP_Obj         vnw;        /* word stack                                  */
    libGAP_UInt2 **    nw;         /* address of <vnw>                            */
    libGAP_Obj         vlw;        /* last syllable stack                         */
    libGAP_UInt2 **    lw;         /* address of <vlw>                            */
    libGAP_Obj         vpw;        /* current syllable stack                      */
    libGAP_UInt2 **    pw;         /* address of <vpw>                            */
    libGAP_Obj         vew;        /* unprocessed exponent stack                  */
    libGAP_UInt2 *     ew;         /* address of <vew>                            */
    libGAP_Obj         vge;        /* global exponent stack                       */
    libGAP_Int *       ge;         /* address of <vge>                            */

    libGAP_Obj         vpow;       /* rhs of power relations                      */
    libGAP_Int         lpow;       /* length of <vpow>                            */
    libGAP_Obj *       pow;        /* address of <vpow>                           */

    libGAP_Obj         vcnj;       /* rhs of conjugate relations                  */
    libGAP_Int         lcnj;       /* length of <vcnj>                            */
    libGAP_Obj *       cnj;        /* address of <vcnj>                           */

    libGAP_Obj *       avc;        /* address of the avector                      */
    libGAP_Obj *       avc2;       /* address of the avector 2                    */
    libGAP_Obj *       wt;         /* address of the weights array                */
    libGAP_Obj *       gns;        /* address of the list of generators           */
    libGAP_Obj *       ro;         /* address of the list of relative orders      */
    libGAP_Obj *       inv;        /* address of the list of inverses             */

    libGAP_Int *       v;          /* address of <vv>                             */

    libGAP_Int         max;        /* maximal stack size                          */
    libGAP_Int         sp;         /* stack pointer                               */
    libGAP_Int         i, j;       /* loop variable                               */
    libGAP_Int         gn;         /* current generator number                    */
    libGAP_Int         ex;         /* current exponent                            */
    libGAP_Int         cl;         /* p-class of the collector                    */
    libGAP_Int         p;          /* the prime                                   */

    libGAP_Obj         tmp;        /* temporary obj for power                     */

    libGAP_Int         resized = 0;/* indicates whether a Resize() happend        */

    /* if <w> is the identity return now                                   */
    if ( libGAP_NPAIRS_WORD(w) == 0 ) {
        return libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* get the exponent sign masks                                         */
    exps = 1UL << (ebits-1);

    /* <nw> contains the stack of words to insert                          */
    vnw = libGAP_SC_NW_STACK(sc);

    /* <lw> contains the word end of the word in <nw>                      */
    vlw = libGAP_SC_LW_STACK(sc);

    /* <pw> contains the position of the word in <nw> to look at           */
    vpw = libGAP_SC_PW_STACK(sc);

    /* <ew> contains the unprocessed exponents at position <pw>            */
    vew = libGAP_SC_EW_STACK(sc);

    /* <ge> contains the global exponent of the word                       */
    vge = libGAP_SC_GE_STACK(sc);

    /* get the maximal stack size                                          */
    max = libGAP_SC_MAX_STACK_SIZE(sc);

    /* ensure that the stacks are large enough                             */
    if ( libGAP_SIZE_OBJ(vnw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vnw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vnw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vlw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vlw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vlw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vpw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vpw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vpw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vew)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vew, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vew, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vge)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vge, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vge, libGAP_T_STRING );
        resized = 1;
    }
    if( resized ) return -1;

    /* from now on we use addresses instead of handles most of the time    */
    v  = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    nw = (libGAP_UInt2**)libGAP_ADDR_OBJ(vnw);
    lw = (libGAP_UInt2**)libGAP_ADDR_OBJ(vlw);
    pw = (libGAP_UInt2**)libGAP_ADDR_OBJ(vpw);
    ew = (libGAP_UInt2*)libGAP_ADDR_OBJ(vew);
    ge = (libGAP_Int*)libGAP_ADDR_OBJ(vge);

    /* conjugates, powers, order, generators, avector, inverses            */
    vpow = libGAP_SC_POWERS(sc);
    lpow = libGAP_LEN_PLIST(vpow);
    pow  = libGAP_ADDR_OBJ(vpow);

    vcnj = libGAP_SC_CONJUGATES(sc);
    lcnj = libGAP_LEN_PLIST(vcnj);
    (void) lcnj; /* please compiler -- lcnj not actually used */
    cnj  = libGAP_ADDR_OBJ(vcnj);

    avc = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR(sc) );
    gns = libGAP_ADDR_OBJ( libGAP_SC_RWS_GENERATORS(sc) );

    cl   = libGAP_INT_INTOBJ( libGAP_SC_CLASS(sc) );
    wt   = libGAP_ADDR_OBJ( libGAP_SC_WEIGHTS(sc) );
    avc2 = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR2(sc) );

    ro  = libGAP_ADDR_OBJ( libGAP_SC_RELATIVE_ORDERS(sc) );
    p   = libGAP_INT_INTOBJ(ro[1]);
    inv = libGAP_ADDR_OBJ( libGAP_SC_INVERSES(sc) );

    /* initalize the stack with <w>                                        */
    sp = 0;
    libGAP_SC_PUSH_WORD( w, 1 );

    /* run until the stack is empty                                        */
    while ( 0 < sp ) {

        /* if <ew> is negative use inverse                                 */
        if ( *ew & exps ) {
            gn = ((**pw) >> ebits) + 1;
            ex = ( *ew & (exps-1) ) - exps;
            *ew = 0;
            libGAP_SC_PUSH_WORD( inv[gn], -ex );
        }

        /* if <ew> is zero get next syllable                               */
        else if ( 0 == *ew ) {

            /* if <pw> has reached <lw> get next & reduce globale exponent */
            if ( *pw == *lw ) {

                /* if the globale exponent is greater one reduce it        */
                if ( 1 < *ge ) {
                    (*ge)--;
                    *pw = *nw;
                    *ew = (**pw) & expm;
                }

                /* otherwise get the next word from the stack              */
                else {
                    libGAP_SC_POP_WORD();

                }
            }

            /* otherwise set <ew> to exponent of next syllable             */
            else {
                (*pw)++;
                *ew = (**pw) & expm;
            }
        }

        /* now move the next generator/word to the correct position        */
        else {

            /* get generator number                                        */
            gn = ((**pw) >> ebits) + 1;

            /* collect a single generator on the stack                     */
            if( *lw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              v[gn] += *ew * *ge;
              *ew = *ge = 0;
              if ( p <= v[gn] ) {
                ex = v[gn] / p;
                v[gn] -= ex * p;
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_C16Bits_AddWordIntoExpVec( 
                      v, (libGAP_UInt2*)libGAP_DATA_WORD(pow[gn]), ex, 
                      ebits, expm, p, pow, lpow  );
                }
              }

              continue;
            }

            /* collect a whole word exponent pair                          */
            else if( sp > 1 && *pw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              libGAP_C16Bits_AddWordIntoExpVec( 
                   v, *pw, *ge, ebits, expm, p, pow, lpow  );
              *pw = *lw;
              *ew = *ge = 0;

              continue;
            }

            /* collect the rest of a word                                  */
            else if( sp > 1 && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              libGAP_C16Bits_AddPartIntoExpVec( 
                   v, *pw, *lw, ebits, expm, p, pow, lpow  );
              *pw = *lw;
              *ew = 0;

              continue;
            }

            else if( sp > 1 && 3*libGAP_INT_INTOBJ(wt[gn]) > cl ) {
                /* Collect <gn>^<ew> without stacking commutators.  
                   This is step 6 in (Vaughan-Lee 1990).                   */
                i = libGAP_INT_INTOBJ(avc[gn]);
                for ( ;  gn < i;  i-- ) {
                    if ( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) ) {
                            libGAP_C16Bits_AddCommIntoExpVec( 
                                v, (libGAP_UInt2*)libGAP_DATA_WORD(tmp), v[i] * (*ew),
                                ebits, expm, p, pow, lpow );
                        }
                    }
                }
                
                v[gn] += (*ew);
                (*ew) = 0;

                /* If the exponent is too big, we have to stack up the 
                   entries in the exponent vector.                         */
                if ( p <= v[gn] ) {
                    ex  = v[gn] / p;
                    v[gn] -= ex * p;
                    if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                        /* stack the exponent vector first. */
                        i = libGAP_INT_INTOBJ(avc[gn]);
                        for ( ; gn < i;  i-- ) {
                            if ( v[i] ) {
                                libGAP_SC_PUSH_GEN( gns[i], v[i] );
                                v[i] = 0;
                            }
                        }
                        libGAP_C16Bits_AddWordIntoExpVec( 
                             v, (libGAP_UInt2*)libGAP_DATA_WORD(pow[i]), ex,
                             ebits, expm, p, pow, lpow  );
                    }
                }
            }
            /* we have to move <gn> step by step                           */
            else {
              
              (*ew)--;
              v[gn]++;

              i = libGAP_INT_INTOBJ(avc[gn]);
              if( sp > 1 ) {
                  /* Do combinatorial collection as far as possible.       */
                  j = libGAP_INT_INTOBJ(avc2[gn]);
                  for( ; j < i; i-- )
                      if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                          tmp = libGAP_ELM_PLIST( cnj[i], gn );
                          if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                              libGAP_C16Bits_AddCommIntoExpVec( 
                                  v, (libGAP_UInt2*)libGAP_DATA_WORD(tmp), v[i],
                                  ebits, expm, p, pow, lpow );
                      }
              }

              /* Find the first position in v from where on ordinary
                 collection  has to be applied.                            */
              for( ; gn < i; i-- )
                  if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                      tmp = libGAP_ELM_PLIST( cnj[i], gn );
                      if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                        break;
                  }

              /* Stack up this part of v if we run through the next 
                 for-loop or if a power relation will be applied           */ 
              if( gn < i || (p <= v[gn] &&
                  gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn])) ) {
                j = libGAP_INT_INTOBJ(avc[gn]);
                for( ; i < j; j-- )
                  if( v[j] ) {
                    libGAP_SC_PUSH_GEN( gns[j], v[j] );
                    v[j] = 0;
                  }
              }

                /* We finish with ordinary collection from the left        */
                for ( ;  gn < i;  i-- ) {
                    if ( v[i] ) {
                      if ( libGAP_LEN_PLIST(cnj[i]) < gn ) {
                        libGAP_SC_PUSH_GEN( gns[i], v[i] );
                      }
                      else {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp == 0 || libGAP_NPAIRS_WORD(tmp) == 0 ) {
                          libGAP_SC_PUSH_GEN( gns[i], v[i] );
                        }
                        else {
                          libGAP_SC_PUSH_WORD( tmp, v[i] );
                        }
                      }
                      v[i] = 0;
                    }
                }
            }

            /* check that the exponent is not too big                      */
            if ( p <= v[gn] ) {
                i = v[gn] / p;
                v[gn] -= i * p;
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_SC_PUSH_WORD( pow[gn], i );
                }
            }
        }
    }
    return libGAP_SC_NUMBER_RWS_GENERATORS(sc);
}
#undef libGAP_SC_PUSH_WORD
#undef libGAP_SC_PUSH_GEN
#undef libGAP_SC_POP_WORD

/****************************************************************************
**
*F  C32Bits_CombiCollectWord( <sc>, <vv>, <w> )
**
**  If a stack overflow occurs, we simply stop and return false.
**
**  SC_PUSH_GEN( gen, exp )
**    push a generator <gen>  with exponent <exp> onto the stack.
**
**  SC_PUSH_WORD( word, exp )
**    push <word>  with global exponent <exp>  into the stack, the macro uses
**    <word> and <exp> only once.
**
**  SC_POP_WORD()
**    remove topmost word from stack
*/
#define libGAP_SC_PUSH_WORD( word, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(word); \
    *++lw = *nw + (libGAP_INT_INTOBJ((((libGAP_Obj*)(*nw))[-1])) - 1); \
    *++pw = *nw; \
    *++ew = (**pw) & expm; \
    *++ge = exp

#define libGAP_SC_PUSH_GEN( gen, exp ) \
    if ( ++sp == max ) { \
        libGAP_SC_SET_MAX_STACK_SIZE( sc, 2 * libGAP_SC_MAX_STACK_SIZE(sc) ); \
        return -1; \
    } \
    *++nw = (void*)libGAP_DATA_WORD(gen); \
    *++lw = *nw; \
    *++pw = *nw; \
    *++ew = exp; \
    *++ge = 1

#define libGAP_SC_POP_WORD() \
    sp--;  nw--;  lw--;  pw--;  ew--;  ge--

/****************************************************************************
**
**  The following functions are  used to add  a word into the exponent vector
**  without collection.  There are three different cases that can occur:
**
**  Add   a word  into  the exponent  vector.   Here we   can  use the global
**  exponent.
**
**  Add  a commutator   into the exponent  vector.   In  this  case the first
**  generator in the conjugate has to be skipped.  Here we can use the global
**  exponent.
**
**  Add part  of a word  into the  exponent vector.   Here  we cannot use the
**  global exponent because the beginning of  the word might not commute with
**  the rest.
**/
static void libGAP_C32Bits_AddWordIntoExpVec( libGAP_Int *v, libGAP_UInt4 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt4 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C32Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt4*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

static void libGAP_C32Bits_AddPartIntoExpVec( libGAP_Int *v, libGAP_UInt4 *w, libGAP_UInt4 *wend,
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_Int        i;
    libGAP_Int        ex;

    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm);     /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C32Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt4*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

static void libGAP_C32Bits_AddCommIntoExpVec( libGAP_Int *v, libGAP_UInt4 *w, libGAP_Int e, 
                           libGAP_Int ebits, libGAP_UInt expm, 
                           libGAP_Int p, libGAP_Obj *pow, libGAP_Int lpow ) {

    libGAP_UInt4 *    wend = w + (libGAP_INT_INTOBJ((((libGAP_Obj*)(w))[-1])) - 1);
    libGAP_Int        i;
    libGAP_Int        ex;

    /* Skip the first generator because we need the commutator here.  */
    w++;
    for( ; w <= wend; w++ ) {
        i = ((*w) >> ebits) + 1; 
        v[ i ] += ((*w) & expm) * e;      /* overflow check necessary? */
        if ( p <= v[i] ) {
            ex = v[i] / p;
            v[i] -= ex * p;
            if ( i <= lpow && pow[i] && 0 < libGAP_NPAIRS_WORD(pow[i]) ) {
                libGAP_C32Bits_AddWordIntoExpVec( 
                    v, (libGAP_UInt4*)libGAP_DATA_WORD(pow[i]), ex,
                    ebits, expm, p, pow, lpow  );
            }
        }
    }
}

libGAP_Int libGAP_C32Bits_CombiCollectWord ( libGAP_Obj sc, libGAP_Obj vv, libGAP_Obj w )
{
    libGAP_Int         ebits;      /* number of bits in the exponent              */
    libGAP_UInt        expm;       /* unsigned exponent mask                      */
    libGAP_UInt        exps;       /* sign exponent mask                          */

    libGAP_Obj         vnw;        /* word stack                                  */
    libGAP_UInt4 **    nw;         /* address of <vnw>                            */
    libGAP_Obj         vlw;        /* last syllable stack                         */
    libGAP_UInt4 **    lw;         /* address of <vlw>                            */
    libGAP_Obj         vpw;        /* current syllable stack                      */
    libGAP_UInt4 **    pw;         /* address of <vpw>                            */
    libGAP_Obj         vew;        /* unprocessed exponent stack                  */
    libGAP_UInt4 *     ew;         /* address of <vew>                            */
    libGAP_Obj         vge;        /* global exponent stack                       */
    libGAP_Int *       ge;         /* address of <vge>                            */

    libGAP_Obj         vpow;       /* rhs of power relations                      */
    libGAP_Int         lpow;       /* length of <vpow>                            */
    libGAP_Obj *       pow;        /* address of <vpow>                           */

    libGAP_Obj         vcnj;       /* rhs of conjugate relations                  */
    libGAP_Int         lcnj;       /* length of <vcnj>                            */
    libGAP_Obj *       cnj;        /* address of <vcnj>                           */

    libGAP_Obj *       avc;        /* address of the avector                      */
    libGAP_Obj *       avc2;       /* address of the avector 2                    */
    libGAP_Obj *       wt;         /* address of the weights array                */
    libGAP_Obj *       gns;        /* address of the list of generators           */
    libGAP_Obj *       ro;         /* address of the list of relative orders      */
    libGAP_Obj *       inv;        /* address of the list of inverses             */

    libGAP_Int *       v;          /* address of <vv>                             */

    libGAP_Int         max;        /* maximal stack size                          */
    libGAP_Int         sp;         /* stack pointer                               */
    libGAP_Int         i, j;       /* loop variable                               */
    libGAP_Int         gn;         /* current generator number                    */
    libGAP_Int         ex;         /* current exponent                            */
    libGAP_Int         cl;         /* p-class of the collector                    */
    libGAP_Int         p;          /* the prime                                   */

    libGAP_Obj         tmp;        /* temporary obj for power                     */

    libGAP_Int         resized = 0;/* indicates whether a Resize() happend        */

    /* if <w> is the identity return now                                   */
    if ( libGAP_NPAIRS_WORD(w) == 0 ) {
        return libGAP_SC_NUMBER_RWS_GENERATORS(sc);
    }

    /* get the number of bits for exponents                                */
    ebits = libGAP_EBITS_WORDTYPE( libGAP_SC_DEFAULT_TYPE(sc) );

    /* get the exponent mask                                               */
    expm = (1UL << ebits) - 1;

    /* get the exponent sign masks                                         */
    exps = 1UL << (ebits-1);

    /* <nw> contains the stack of words to insert                          */
    vnw = libGAP_SC_NW_STACK(sc);

    /* <lw> contains the word end of the word in <nw>                      */
    vlw = libGAP_SC_LW_STACK(sc);

    /* <pw> contains the position of the word in <nw> to look at           */
    vpw = libGAP_SC_PW_STACK(sc);

    /* <ew> contains the unprocessed exponents at position <pw>            */
    vew = libGAP_SC_EW_STACK(sc);

    /* <ge> contains the global exponent of the word                       */
    vge = libGAP_SC_GE_STACK(sc);

    /* get the maximal stack size                                          */
    max = libGAP_SC_MAX_STACK_SIZE(sc);

    /* ensure that the stacks are large enough                             */
    if ( libGAP_SIZE_OBJ(vnw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vnw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vnw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vlw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vlw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vlw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vpw)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vpw, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vpw, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vew)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vew, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vew, libGAP_T_STRING );
        resized = 1;
    }
    if ( libGAP_SIZE_OBJ(vge)/sizeof(libGAP_Obj) < max+1 ) {
        libGAP_ResizeBag( vge, sizeof(libGAP_Obj)*(max+1) );
        libGAP_RetypeBag( vge, libGAP_T_STRING );
        resized = 1;
    }
    if( resized ) return -1;

    /* from now on we use addresses instead of handles most of the time    */
    v  = (libGAP_Int*)libGAP_ADDR_OBJ(vv);
    nw = (libGAP_UInt4**)libGAP_ADDR_OBJ(vnw);
    lw = (libGAP_UInt4**)libGAP_ADDR_OBJ(vlw);
    pw = (libGAP_UInt4**)libGAP_ADDR_OBJ(vpw);
    ew = (libGAP_UInt4*)libGAP_ADDR_OBJ(vew);
    ge = (libGAP_Int*)libGAP_ADDR_OBJ(vge);

    /* conjugates, powers, order, generators, avector, inverses            */
    vpow = libGAP_SC_POWERS(sc);
    lpow = libGAP_LEN_PLIST(vpow);
    pow  = libGAP_ADDR_OBJ(vpow);

    vcnj = libGAP_SC_CONJUGATES(sc);
    lcnj = libGAP_LEN_PLIST(vcnj);
    (void) lcnj; /* please compiler -- lcnj not actually used */
    cnj  = libGAP_ADDR_OBJ(vcnj);

    avc = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR(sc) );
    gns = libGAP_ADDR_OBJ( libGAP_SC_RWS_GENERATORS(sc) );

    cl   = libGAP_INT_INTOBJ( libGAP_SC_CLASS(sc) );
    wt   = libGAP_ADDR_OBJ( libGAP_SC_WEIGHTS(sc) );
    avc2 = libGAP_ADDR_OBJ( libGAP_SC_AVECTOR2(sc) );

    ro  = libGAP_ADDR_OBJ( libGAP_SC_RELATIVE_ORDERS(sc) );
    p   = libGAP_INT_INTOBJ(ro[1]);
    inv = libGAP_ADDR_OBJ( libGAP_SC_INVERSES(sc) );

    /* initalize the stack with <w>                                        */
    sp = 0;
    libGAP_SC_PUSH_WORD( w, 1 );

    /* run until the stack is empty                                        */
    while ( 0 < sp ) {

        /* if <ew> is negative use inverse                                 */
        if ( *ew & exps ) {
            gn = ((**pw) >> ebits) + 1;
            ex = ( *ew & (exps-1) ) - exps;
            *ew = 0;
            libGAP_SC_PUSH_WORD( inv[gn], -ex );
        }

        /* if <ew> is zero get next syllable                               */
        else if ( 0 == *ew ) {

            /* if <pw> has reached <lw> get next & reduce globale exponent */
            if ( *pw == *lw ) {

                /* if the globale exponent is greater one reduce it        */
                if ( 1 < *ge ) {
                    (*ge)--;
                    *pw = *nw;
                    *ew = (**pw) & expm;
                }

                /* otherwise get the next word from the stack              */
                else {
                    libGAP_SC_POP_WORD();

                }
            }

            /* otherwise set <ew> to exponent of next syllable             */
            else {
                (*pw)++;
                *ew = (**pw) & expm;
            }
        }

        /* now move the next generator/word to the correct position        */
        else {

            /* get generator number                                        */
            gn = ((**pw) >> ebits) + 1;

            /* collect a single generator on the stack                     */
            if( *lw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              v[gn] += *ew * *ge;
              *ew = *ge = 0;
              if ( p <= v[gn] ) {
                ex = v[gn] / p;
                v[gn] -= ex * p;
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_C32Bits_AddWordIntoExpVec( 
                      v, (libGAP_UInt4*)libGAP_DATA_WORD(pow[gn]), ex, 
                      ebits, expm, p, pow, lpow  );
                }
              }

              continue;
            }

            /* collect a whole word exponent pair                          */
            else if( sp > 1 && *pw == *nw && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              libGAP_C32Bits_AddWordIntoExpVec( 
                   v, *pw, *ge, ebits, expm, p, pow, lpow  );
              *pw = *lw;
              *ew = *ge = 0;

              continue;
            }

            /* collect the rest of a word                                  */
            else if( sp > 1 && libGAP_INT_INTOBJ(avc[gn]) == gn ) {
              libGAP_C32Bits_AddPartIntoExpVec( 
                   v, *pw, *lw, ebits, expm, p, pow, lpow  );
              *pw = *lw;
              *ew = 0;

              continue;
            }

            else if( sp > 1 && 3*libGAP_INT_INTOBJ(wt[gn]) > cl ) {
                /* Collect <gn>^<ew> without stacking commutators.  
                   This is step 6 in (Vaughan-Lee 1990).                   */
                i = libGAP_INT_INTOBJ(avc[gn]);
                for ( ;  gn < i;  i-- ) {
                    if ( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) ) {
                            libGAP_C32Bits_AddCommIntoExpVec( 
                                v, (libGAP_UInt4*)libGAP_DATA_WORD(tmp), (libGAP_Int)(v[i] * (*ew)),
                                ebits, expm, p, pow, lpow );
                        }
                    }
                }
                
                v[gn] += (*ew);
                (*ew) = 0;

                /* If the exponent is too big, we have to stack up the 
                   entries in the exponent vector.                         */
                if ( p <= v[gn] ) {
                    ex  = v[gn] / p;
                    v[gn] -= ex * p;
                    if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                        /* stack the exponent vector first. */
                        i = libGAP_INT_INTOBJ(avc[gn]);
                        for ( ; gn < i;  i-- ) {
                            if ( v[i] ) {
                                libGAP_SC_PUSH_GEN( gns[i], v[i] );
                                v[i] = 0;
                            }
                        }
                        libGAP_C32Bits_AddWordIntoExpVec( 
                             v, (libGAP_UInt4*)libGAP_DATA_WORD(pow[i]), ex,
                             ebits, expm, p, pow, lpow  );
                    }
                }
            }
            /* we have to move <gn> step by step                           */
            else {
              
              (*ew)--;
              v[gn]++;

              i = libGAP_INT_INTOBJ(avc[gn]);
              j = libGAP_INT_INTOBJ(avc2[gn]);
              if( sp > 1 ) {
                  /* Do combinatorial collection as far as possible.       */
                  for( ; j < i; i-- )
                      if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                          tmp = libGAP_ELM_PLIST( cnj[i], gn );
                          if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                              libGAP_C32Bits_AddCommIntoExpVec( 
                                 v, (libGAP_UInt4*)libGAP_DATA_WORD(tmp), v[i],
                                 ebits, expm, p, pow, lpow );
                      }
              }

              /* Find the first position in v from where on ordinary
                 collection  has to be applied.                            */
              for( ; gn < i; i-- )
                  if( v[i] && gn <= libGAP_LEN_PLIST(cnj[i]) ) {
                      tmp = libGAP_ELM_PLIST( cnj[i], gn );
                      if ( tmp != 0 && 0 < libGAP_NPAIRS_WORD(tmp) )
                        break;
                  }

              /* Stack up this part of v if we run through the next 
                 for-loop or if a power relation will be applied           */ 
              if( gn < i || (p <= v[gn] &&
                  gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn])) ) {
                j = libGAP_INT_INTOBJ(avc[gn]);
                for( ; i < j; j-- )
                  if( v[j] ) {
                    libGAP_SC_PUSH_GEN( gns[j], v[j] );
                    v[j] = 0;
                  }
              }

                /* We finish with ordinary collection from the left        */
                for ( ;  gn < i;  i-- ) {
                    if ( v[i] ) {
                      if ( libGAP_LEN_PLIST(cnj[i]) < gn ) {
                        libGAP_SC_PUSH_GEN( gns[i], v[i] );
                      }
                      else {
                        tmp = libGAP_ELM_PLIST( cnj[i], gn );
                        if ( tmp == 0 || libGAP_NPAIRS_WORD(tmp) == 0 ) {
                          libGAP_SC_PUSH_GEN( gns[i], v[i] );
                        }
                        else {
                          libGAP_SC_PUSH_WORD( tmp, v[i] );
                        }
                      }
                      v[i] = 0;
                    }
                }
            }

            /* check that the exponent is not too big                      */
            if ( p <= v[gn] ) {
                i = v[gn] / p;
                v[gn] -= i * p;
                if ( gn <= lpow && pow[gn] && 0 < libGAP_NPAIRS_WORD(pow[gn]) ) {
                    libGAP_SC_PUSH_WORD( pow[gn], i );
                }
            }
        }
    }
    return libGAP_SC_NUMBER_RWS_GENERATORS(sc);
}
#undef libGAP_SC_PUSH_WORD
#undef libGAP_SC_PUSH_GEN
#undef libGAP_SC_POP_WORD


/****************************************************************************
**
*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
**
**  This module does  not   need much initialisation  because  all  necessary
**  initialisations are done in the single collector module.
*/


/****************************************************************************
**
*F  InitInfoCombiCollector()  . . . . . . . . . . . . table of init functions
**
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "objccoll",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    0,                                  /* initKernel                     */
    0,                                  /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoCombiCollector ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  objccoll.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
