/* Copyright (C) 2005-2008 Damien Stehle.
Copyright (C) 2007 David Cade.

This file is part of the fplll Library.

The fplll Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The fplll Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the fplll Library; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */

#ifndef NR_H
#define NR_H

#include "defs.h"

using namespace std;

template<class F> class FP_NR;

/* Integers
   ======== */

template<class Z>
class Z_NR
{
  Z data;
 public:
  inline Z_NR<Z>();
  inline Z_NR<Z>(const Z_NR<Z>& z);
  inline ~Z_NR<Z>();

  inline void print() const;
  inline void printerr() const;
  inline void read();

  inline double get_d() const;
  inline double get_d_2exp(signed long int* expo) const;
  inline long int get_si() const;
  inline void set(const Z_NR<Z>& s);
  inline void set(/*const*/ Z& s);
  inline void set(unsigned long int s);
  inline void set_si(signed long int s);
  /* F=mpfr_t -> rounding to the nearest
     F=double/dpe_t -> undefined rounding direction */
  template<class F> inline void set_f(const FP_NR<F>& f);
  inline void operator=(const Z_NR<Z>& z);
  inline void operator=(long int i);

  inline int cmp(const Z_NR<Z>& m) const;
  inline int sgn() const;
  inline bool operator<(const Z_NR<Z>& a) const;
  inline bool operator<(long int a) const;
  inline bool operator>(const Z_NR<Z>& a) const;
  inline bool operator>(long int a) const;
  inline bool operator<=(const Z_NR<Z>& a) const;
  inline bool operator<=(long int a) const;
  inline bool operator>=(const Z_NR<Z>& a) const;
  inline bool operator>=(long int a) const;
  inline bool operator==(const Z_NR<Z>& a) const;
  inline bool operator==(long int a) const;
  inline bool operator!=(const Z_NR<Z>& a) const;
  inline bool operator!=(long int a) const;

  inline void add(const Z_NR<Z>& a, const Z_NR<Z>& b);
  inline void add_ui(const Z_NR<Z>& a, unsigned int b);
  inline void sub(const Z_NR<Z>& a, const Z_NR<Z>& b);
  template<class T> inline void mul(const Z_NR<Z>& a, const T& b);
  inline void mul_si(const Z_NR<Z>& a, signed long int b);
  inline void mul_si(const Z_NR<Z>& a, const Z_NR<long int>& b);
  inline void mul_ui(const Z_NR<Z>& a, unsigned long int b);
  inline void mul_2exp(const Z_NR<Z>& a, long int expo);
  inline void div_2exp(const Z_NR<Z>& a, long int expo);

  template<class T> inline void addmul(const Z_NR<Z>& a, const T& b);
  inline void addmul_ui(const Z_NR<Z>& a, unsigned long int b);
  inline void addmul_si(const Z_NR<Z>& a, signed long int b);
  template<class T> inline void submul(const Z_NR<Z>& a, const T& b);
  inline void submul_ui(const Z_NR<Z>& a, unsigned long int b);

  inline void abs(const Z_NR<Z>& a);
  inline void randb(int bits);
  inline void randm(const Z_NR<Z>& max);

  inline Z& GetData();
  inline const Z& GetData() const;
};

template<class Z>
inline ostream& operator<<(ostream& os, const Z_NR<Z>& a) {
  if (&os == &cout)
    a.print();
  else if (&os == &cerr)
    a.printerr();
  return os;
}

/* Floats
   ====== */

template<class F>
struct assoc_int_struct {
  typedef Z_NR<long int> assoc_int;
};

template<>
struct assoc_int_struct<mpfr_t> {
  typedef Z_NR<mpz_t> assoc_int;
};

template<class F>
class FP_NR
{
  F data;
 public:
  typedef struct assoc_int_struct<F>::assoc_int assoc_int;
  inline FP_NR<F>();
  inline FP_NR<F>(const FP_NR<F>& f);
  inline ~FP_NR<F>();

  inline void print() const;
  inline void printerr() const;

  inline double get() const;
  inline double get_d(mp_rnd_t rnd = GMP_RNDN) const;
  inline signed long int get_si() const;
  inline void set(const FP_NR<F>& s);
  inline void set(double s);
  inline void set(unsigned int s);
  template<class Z> inline void set_z(const Z_NR<Z>& z, mp_rnd_t rnd = GMP_RNDN);
  inline void operator=(const FP_NR<F>& a);
  inline void operator=(double a);

  inline void add(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd = GMP_RNDN);
  inline void sub(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd = GMP_RNDN);
  inline void neg(const FP_NR<F>& b);
  inline void mul(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd = GMP_RNDN);
  inline void mul_2ui(const FP_NR<F>& b, unsigned int c);
  inline void div(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd = GMP_RNDN);
  inline void div_2ui(const FP_NR<F>& b, unsigned int c);

  inline int cmp(const FP_NR<F>& b) const;
  inline int cmp(double d) const;
  inline int sgn() const;
  inline bool operator<=(const FP_NR<F>& a) const;
  inline bool operator<=(double a) const;
  inline bool operator>=(const FP_NR<F>& a) const;
  inline bool operator>=(double a) const;

  inline void abs(const FP_NR<F>& b);
  inline void rnd(const FP_NR<F>& b);
  inline void floor(const FP_NR<F>& b);
  inline int exp() const;
  inline int zero_p() const;
  inline void set_nan();
  inline int is_nan() const;
  inline void sqrt(const FP_NR<F>& b, mp_rnd_t rnd = GMP_RNDN);
  inline void exponential(const FP_NR<F>& a, mp_rnd_t rnd = GMP_RNDN);
  inline void log(const FP_NR<F>& a, mp_rnd_t rnd = GMP_RNDN);

  static inline void setprec(unsigned int prec);

  inline F& GetData();
  inline const F& GetData() const;
  inline void submul(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd = GMP_RNDN);
  inline void addmul(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd = GMP_RNDN);
};

template<class F>
inline ostream& operator<<(ostream& os, const FP_NR<F>& a) {
  if (&os == &cout)
    a.print();
  else if (&os == &cerr)
    a.printerr();
  return os;
}

#include "nr.cpp"

#endif
