/* Copyright (C) 2005-2009 Damien Stehle.
Copyright (C) 2007 David Cade.

This file is part of the fplll Library.

The fplll Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The fplll Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the fplll Library; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */


#ifndef MATRIX_CPP
#define MATRIX_CPP

#include "matrix.h"

template<class ZT> void ZZ_mat<ZT>::SetNumCols(int cols)
{
  INTERNAL_CHECK(r == 0, "SetNumCols called on an non-empty matrix");
  clear();
  c = cols;
}

template<class ZT> void ZZ_mat<ZT>::SetNumRows(int rows)
{
  int oldSize = matrix.size();
  if (rows > oldSize) {
    matrix.resize(rows);
    for (int i = oldSize; i < rows; i++)
      matrix[i] = new Z_NR<ZT>[c];
  }
  r = rows;
}

template<class ZT> void ZZ_mat<ZT>::clear()
{
  for (unsigned int i = 0; i < matrix.size(); i++) {
    delete[] matrix[i];
  }
  matrix.clear();
  r = 0;
  c = 0;
}

template<class ZT> int ZZ_mat<ZT>::getMaxExp()
{
  int max=0;
  for (int i=0;i<r;i++)
    for (int j=0;j<c;j++)
      {
	long int x;
	matrix[i][j].get_d_2exp(&x);
	if (max<x) max=x;
      }
  return max;
}

inline char nextNonBlankChar(char& ch)
{
  
  ch=getchar();
  while (ch==' '||ch=='\t'||ch=='\r'||ch=='\n')
    ch=getchar();
  return ch;
}

template<class ZT> int ZZ_mat<ZT>::read()
{
  char ch;
  
  nextNonBlankChar(ch);

  if (ch!='[')
    {
      cerr<< "[ expected instead of "<<ch<<"\n";
      return 1;
    }
  for (int i=0; i<r; i++)
    {
      nextNonBlankChar(ch);
      if (ch != '[')
	{
	  cerr <<"Error at row "<<i<< " '[' expected instead of "<<ch<<"\n";
	  return 1;
	}
      for (int j=0; j<c; j++)
	{
	  matrix[i][j].read();
	}
      
      nextNonBlankChar(ch);
      if (ch != ']')
	{
	  cerr<<"Error: ']' expected at line "<<i<<"\n";
	  return 1;
	}
    }

  nextNonBlankChar(ch);
  if (ch != ']')
    {
      cerr<<"Error: ']' expected\n";
      return 1;
    }

  return 0;
}


template<class ZT> inline Z_NR<ZT>& ZZ_mat<ZT>::Get (int i, int j)
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(z_nr) " << i << "," << j);
  return matrix[i][j];
}

template<class ZT>
inline Z_NR<ZT>& ZZ_mat<ZT>::operator()(int i, int j)
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(z_nr) " << i << "," << j);
  return matrix[i][j];
}

template<class ZT>
inline const Z_NR<ZT>& ZZ_mat<ZT>::operator()(int i, int j) const
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(z_nr) " << i << "," << j);
  return matrix[i][j];
}

template<class ZT> inline Z_NR<ZT>*& ZZ_mat<ZT>::GetVec(int i)
{
  BOUND_CHECK(i >= 0 && i < r, "(z_nr) i=" << i);
  return matrix[i];
}

template<class ZT> inline void ZZ_mat<ZT>::Set (int i, int j, Z_NR<ZT>& s)
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(z_nr) " << i << "," << j);
  matrix[i][j].set(s);
}


template<class ZT> int ZZ_mat<ZT>::getShift()
{
  int n = GetNumCols();
  int shift=0;
  for (int i=0;i<GetNumRows();i++)
    {
      int j;
      for (j=n-1;j>=0 && Get(i,j).sgn()==0;j--);  
      
      if (shift<j-i) shift=j-i;
      
    }

#ifdef DEBUG
  cerr << "Shift  =  " << shift <<", ";
#endif
  return shift;
}


template<class ZT> void ZZ_mat<ZT>::print (int d,int n)
{
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();
  cout << "[";
  for (int i=0;i<d;i++) 
    {
      cout<< "[";
      for (int j=0;j<n;j++)
	{
	  matrix[i][j].print();
	  cout<<" ";
	}
      cout << "]" << endl;
    }
  cout << "]" << endl; 
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();

}
template<class ZT> void ZZ_mat<ZT>::print ()
{
  print(r,c);
}



/*fp*/

template<class FT> void FP_mat<FT>::SetNumCols(int cols)
{
  INTERNAL_CHECK(r == 0, "SetNumCols called on an non-empty matrix");
  clear();
  c = cols;
}

template<class FT> void FP_mat<FT>::SetNumRows(int rows)
{
  int oldSize = matrix.size();
  if (rows > oldSize) {
    matrix.resize(rows);
    for (int i = oldSize; i < rows; i++)
      matrix[i] = new FP_NR<FT>[c];
  }
  r = rows;
}

template<class FT> void FP_mat<FT>::clear()
{
  for (unsigned int i = 0; i < matrix.size(); i++) {
    delete[] matrix[i];
  }
  matrix.clear();
  r = 0;
  c = 0;
}

template<class FT> inline FP_NR<FT>& FP_mat<FT>::Get (int i, int j)
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(fp_nr) " << i << "," << j);
  return matrix[i][j];
}

template<class FT>
inline FP_NR<FT>& FP_mat<FT>::operator()(int i, int j)
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(fp_nr) " << i << "," << j);
  return matrix[i][j];
}

template<class FT>
inline const FP_NR<FT>& FP_mat<FT>::operator()(int i, int j) const
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(fp_nr) " << i << "," << j);
  return matrix[i][j];
}

template<class FT> inline FP_NR<FT>*& FP_mat<FT>::GetVec(int i)
{
  BOUND_CHECK(i >= 0 && i < r, "(fp_nr) i=" << i);
  return matrix[i];
}

template<class FT> inline void FP_mat<FT>::Set (int i, int j, FP_NR<FT>& s)
{
  BOUND_CHECK(i >= 0 && i < r && j >= 0 && j < c, "(fp_nr) " << i << "," << j);
  matrix[i][j].set(s);
}

template<class FT> void FP_mat<FT>::print (int d,int n)
{
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();
  cout<< "[";
  for (int i=0;i<d;i++) 
    {
      cout<< "[";
      for (int j=0;j<n;j++)
	{
	  matrix[i][j].print();
	  cout<<" ";
	}
      cout << "]" << endl;
    }
  cout << "]" << endl; 
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();
}
template<class FT> void FP_mat<FT>::print ()
{
  print(r,c);
}




template<class ZT> inline void ZZ_mat<ZT>::gen_intrel(int bits)
{
  if (c!=r+1)
    {
      cerr<<"gen_intrel called on an ill-formed matrix"<<endl;
      return;
    }
  int i,j;
  for (i=0;i<r;i++)
    {
      matrix[i][0].randb(bits);
      for (j=1; j<=i; j++)
	matrix[i][j].set(0);
      matrix[i][i+1].set(1);
      for (j=i+2; j<c; j++)
	matrix[i][j].set(0);      
    }
}


template<class ZT> inline void ZZ_mat<ZT>::gen_simdioph(int bits,int bits2)
{
  if (c!=r)
    {
      cerr<<"gen_simdioph called on an ill-formed matrix"<<endl;
      return;
    }
  int i, j;

  matrix[0][0].set(1);
  matrix[0][0].mul_2exp(matrix[0][0], bits2);
  for (i=1; i<r; i++)
    matrix[0][i].randb(bits);
  for (i=1; i<r; i++)
    {
      for (j=1; j<i; j++)
	matrix[j][i].set(0);
      matrix[i][i].set(1);
      matrix[i][i].mul_2exp(matrix[i][i], bits);
      for (j=i+1; j<c; j++)
	matrix[j][i].set(0);
    }
}

template<class ZT> inline void ZZ_mat<ZT>::gen_uniform(int bits)
{
  for (int i=0;i<r;i++)for(int j=0;j<c;j++)matrix[i][j].randb(bits);
}

template<class ZT> inline void ZZ_mat<ZT>::gen_ntrulike(int bits,int q)
{
  int i, j, k;
  int d=r/2;
  if (c!=r || c!=2*d) 
    {
      cerr<<"gen_ntrulike called on an ill-formed matrix"<<endl;
      return;
    }
  Z_NR<ZT> * h=new Z_NR<ZT>[d];

  for (i=0; i<d; i++)
    h[i].randb(bits);
  
  for (i=0; i<d; i++)
    {
      for (j=0; j<i; j++)
	matrix[i][j].set(0);
      matrix[i][i].set(1);
      for (j=i+1; j<d; j++)
	matrix[i][j].set(0);
    }

  for (i=d; i<r; i++)
    for (j=0; j<d; j++)
      matrix[i][j].set(0);

  for (i=d; i<r; i++)
    {
      for (j=d; j<i; j++)
	matrix[i][j].set(0);
      matrix[i][i].set(q);
      for (j=i+1; j<c; j++)
	matrix[i][j].set(0);
    }

  for (i=0; i<d; i++)
    for (j=d; j<c; j++)
      { 
	k = j+i;
	while (k>=d)k-=d;
	matrix[i][j].set(h[k]);
      }

  delete[] h;
}

template<class ZT> inline void ZZ_mat<ZT>::gen_ntrulike2(int bits,int q)
{

  int i, j, k;
  
  int d=r/2;
  if (c!=r || c!=2*d) 
    {
      cerr<<"gen_ntrulike2 called on an ill-formed matrix"<<endl;
      return;
    }
  Z_NR<ZT> * h=new Z_NR<ZT>[d];
   
  for (i=0; i<d; i++)
    h[i].randb(bits);
  
  for (i=0; i<d; i++)
    for (j=0; j<c; j++)
      matrix[i][j].set(0);

  for (i=0; i<d; i++)
    matrix[i][i].set(q);


  for (i=d; i<r; i++)
    for (j=d; j<c; j++)
      matrix[i][j].set(0);
      
  for (i=d; i<c; i++)
    matrix[i][i].set(1);

  for (i=d; i<r; i++)
    for (j=0; j<d; j++)
      { 
	k = i+j;
	while (k>=d)k-=d;
	matrix[i][j].set(h[k]);
      }

  delete[] h;
}

template<class ZT> inline void ZZ_mat<ZT>::gen_ajtai(double alpha)
{
  int i, j, bits;
  Z_NR<ZT> ztmp, ztmp2, zone;
  
  ztmp2.set(0);
  zone.set(1);

  int d=r;
  if (c!=r) 
    {
      cerr<<"gen_ajtai called on an ill-formed matrix"<<endl;
      return;
    }

  for (i=0; i<d; i++)
    {
      bits = (int) pow((double) (2*d-i), alpha);
      ztmp.set(1);
      ztmp.mul_2exp(ztmp, bits);	  
      ztmp.sub(ztmp,zone); 
      matrix[i][i].randm(ztmp);
      matrix[i][i].add_ui(matrix[i][i], 2);
      ztmp.div_2exp(matrix[i][i], 1);
      for (j=i+1; j<d; j++)
	{
	  matrix[j][i].randm(ztmp);
	  if (rand()%2==1)
	    matrix[j][i].sub(ztmp2, matrix[j][i]);
	  matrix[i][j].set(0);
	}
    }
}


template<class ZT> inline void ZZ_mat<ZT>::gen_ajtai2(FP_NR<mpfr_t> *w)
{
  int i, j;
  Z_NR<ZT> ztmp, ztmp2;
  
  int d=r;
  if (c!=r) 
    {
      cerr<<"gen_ajtai2 called on an ill-formed matrix"<<endl;
      return;
    }

  for (i=0; i<d; i++)
    {
      matrix[i][i].set_f(w[i]);
      ztmp.div_2exp(matrix[i][i], 1);
      ztmp2.set(1);
      ztmp.add(ztmp, ztmp2);
      for (j=i+1; j<d; j++)
	{ 
	  ztmp2.set(0);
	  matrix[j][i].randm(ztmp);
	  if (rand()%2==1)
	    matrix[j][i].sub(ztmp2, matrix[j][i]);
	  matrix[i][j].set(0);
	}
    }

}




#endif
