/* Copyright (C) 2008 Xavier Pujol.

This file is part of the fplll Library.

The fplll Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The fplll Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the fplll Library; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */

#include "evaluator.h"
#include <float.h>

void Evaluator::evalSol(const FloatVect& x,
        const double& newDist, double& maxDist) {
  // Assumes that the solution is valid and stores it
  solCoord = x;
  maxDist = newDist;
  lastPartialDist = newDist; // Exact conversion
  newSolFlag = true;
}

bool Evaluator::getMaxError(Float& maxError) {
  Float maxE, maxDE, maxOptDE, minOptE, one;

  if (solCoord.empty() || !inputErrorDefined) return false;
  if (!getMaxErrorAux(lastPartialDist, false, maxDE)) return false;

  // Exact norm of an optimal solution <= Exact norm of the result <= maxE
  maxE.add(lastPartialDist, maxDE, GMP_RNDU);
  // Error on the norm of an optimal solution <= maxOptDE
  if (!getMaxErrorAux(maxE, true, maxOptDE)) return false;
  // Approx. norm of an optimal solution >= minOptE
  minOptE.sub(lastPartialDist, maxOptDE, GMP_RNDD);

  one = 1.0;
  maxError.div(maxE, minOptE, GMP_RNDU);
  maxError.sub(maxE, one, GMP_RNDU);
  return true;
}

bool Evaluator::certifiedSol() {
  return false;
}

bool SmartEvaluator::certifiedSol() {
  return certified;
}

bool SmartEvaluator::getMaxError(Float& maxError) {
  if (certified) {
    maxError = 0.0;
    return true;
  }
  else
    return Evaluator::getMaxError(maxError);
}

void SmartEvaluator::evalSol(const FloatVect& newSolCoord,
        const double& newPartialDist, double& maxDist) {
  Float fNewPartialDist, maxE, maxDE, maxNextDE, maxNextETilde;
  fNewPartialDist = newPartialDist;

  if (!solCoord.empty() && fNewPartialDist >= lastPartialDist) {
    /* Approx. norm of the prev. solution <= newPartialDist <= current bound
       The two vectors have nearly the same length, the shortest is unknown */
    certified = false;
    return;
  }

  // Error on the norm of the new solution <= maxDE
  bool result = getMaxErrorAux(fNewPartialDist, false, maxDE);
  INTERNAL_CHECK(result, "(SmartEvaluator) error cannot be bounded");

  // Exact norm of the new solution <= maxE
  maxE.add(fNewPartialDist, maxDE, GMP_RNDU);

  // Error on the norm of vectors shorter than the new solution <= maxNextDE
  result = getMaxErrorAux(maxE, true, maxNextDE);
  INTERNAL_CHECK(result, "(SmartEvaluator) error cannot be bounded");

  // Approx. norm of a vector shortest than the new solution <= maxNextETilde
  maxNextETilde.add(maxE, maxNextDE, GMP_RNDU);

  /* Fact: evalSol was called on all vectors shorter than lastPartialDist
     which come before the new solution in the enumeration order of the
     fp algorithm, and lastPartialDist is the minimum approx. norm of
     those vectors. If maxNextETilde <= lastPartialDist, then all previous
     vectors are longer than the new solution */
  certified = solCoord.empty() || maxNextETilde <= lastPartialDist;

  // Converts the new bound to a double (no need to round towards +infinity)
  maxDist = maxNextETilde.get_d();
  lastPartialDist = newPartialDist;
  solCoord = newSolCoord;
  newSolFlag = true;
}

bool ExactEvaluator::certifiedSol() {
  return true;
}

bool ExactEvaluator::getMaxError(Float& maxError) {
  maxError = 0.0;
  return true;
}

void ExactEvaluator::evalSol(const FloatVect& newSolCoord,
        const double& newPartialDist, double& maxDist) {
  int d = matrix.GetNumRows(), n = matrix.GetNumCols();
  Integer newSolDist, coord;
  IntVect newSol;

  zeroVect(newSol, n);
  newSolDist = 0;

  // Computes the distance between x and zero
  for (int i = 0; i < d; i++) {
    coord.set_f(newSolCoord[i]);
    for (int j = 0; j < n; j++)
      newSol[j].addmul(coord, matrix(i, j));
  }
  for (int i = 0; i < n; i++) {
    coord = newSol[i];
    //if (!target.empty()) coord.sub(coord, target[i]);
    newSolDist.addmul(coord, coord);
  }

  // Updates the stored solution
  if (solCoord.empty() || newSolDist < solDist) {
    lastPartialDist = newPartialDist;
    solCoord = newSolCoord;
    solDist.set(newSolDist);
    updateMaxDist(maxDist);
    newSolFlag = true;
  }
}

// Decreases the bound of the algorithm when a solution is found
void ExactEvaluator::updateMaxDist(double& maxDist) {
  Float fSolDist, maxDE;
  fSolDist.set_z(solDist, GMP_RNDU);
  fSolDist.mul(fSolDist, normFactor); // Exact
  bool result = getMaxErrorAux(fSolDist, true, maxDE);
  INTERNAL_CHECK(result, "(ExactEvaluator) Error cannot be bounded");
  INTERNAL_CHECK(maxDE <= rdiag[0], "(ExactEvaluator) Max error is too large");
  fSolDist.add(fSolDist, maxDE);
  maxDist = fSolDist.get_d();
}

bool Evaluator::getMaxErrorAux(const Float& maxDist,
  bool boundOnExactVal, Float& maxDE) {

  STD_CHECK(targetCoord.empty(), "(Evaluator) Error evaluation is not implemented for CVP");
  STD_CHECK(inputErrorDefined, "(Evaluator) Error evaluation failed because the input error is undefined");

  int d = rdiag.size();
  Float ulp, halfULP, K, tmp1, tmp2;
  Float rdiagTilde, minRDiag, maxRDiag, muTilde, maxMu, maxMuTildeX;
  Float maxC, maxCTilde, maxY, maxYTilde, maxY2Tilde, maxRY2Tilde;
  Float maxDC, maxDY, maxDY2, maxDRY2;
  FloatVect maxX(d);

  ulp = numeric_limits<double>::epsilon();
  halfULP.div_2ui(ulp, 1);
  K = 1.0;
  K.add(K, halfULP, GMP_RNDU);
  maxDE = 0.0;

  for (int i = d - 1; i >= 0; i--) {
    maxC = 0.0;
    maxCTilde = 0.0;
    maxDC = 0.0;

    for (int j = d - 1; j > i; j--) {
      maxMu.abs(mu(j, i));
      maxMu.add(maxMu, maxDMu[i], GMP_RNDU);
      maxC.addmul(maxMu, maxX[j], GMP_RNDU);

      muTilde = fabs(mu(j, i).get_d());             // Exact value of |mu_j,i~|
      maxMuTildeX.mul(muTilde, maxX[j], GMP_RNDU);
      maxDC.addmul(maxMuTildeX, halfULP, GMP_RNDU);
      maxCTilde.addmul(maxMuTildeX, K, GMP_RNDU);
      maxDC.addmul(maxCTilde, halfULP, GMP_RNDU);
      maxCTilde.mul(maxCTilde, K, GMP_RNDU);
    }

    if (boundOnExactVal) {
      minRDiag.sub(rdiag[i], maxDRdiag[i], GMP_RNDD);
      if (minRDiag <= 0.0) return false;
      tmp1.div(maxDist, minRDiag, GMP_RNDU);
      maxY.sqrt(tmp1, GMP_RNDU);
      maxDY.mul(maxY, halfULP, GMP_RNDU);
      maxDY.addmul(maxDC, K, GMP_RNDU);
      maxYTilde.add(maxY, maxDY, GMP_RNDU);
      tmp1.add(maxY, maxC, GMP_RNDD);
      maxX[i].floor(tmp1);
      tmp1 = maxY;
    }
    else {
      rdiagTilde = rdiag[i].get_d();                // Exact value of r_i~
      tmp1.mul(maxDist, K, GMP_RNDU);
      tmp1.div(tmp1, rdiagTilde, GMP_RNDU);
      tmp1.mul(tmp1, K, GMP_RNDU);
      maxYTilde.sqrt(tmp1, GMP_RNDU);
      maxDY.mul(maxYTilde, halfULP, GMP_RNDU);
      maxDY.add(maxDY, maxDC, GMP_RNDU);
      tmp1 = maxCTilde;
      tmp1.addmul(maxYTilde, K, GMP_RNDD);
      maxX[i].floor(tmp1);
      tmp1 = maxYTilde;
    }

    maxDY2.mul(maxDY, tmp1);
    maxDY2.mul_2ui(maxDY2, 1);
    maxDY2.addmul(maxDY, maxDY, GMP_RNDU);
    maxY2Tilde.mul(maxYTilde, maxYTilde, GMP_RNDU);
    maxDY2.addmul(maxY2Tilde, halfULP, GMP_RNDU);
    maxY2Tilde.mul(maxY2Tilde, K, GMP_RNDU);
    maxRDiag.add(rdiag[i], maxDRdiag[i], GMP_RNDU);
    maxDRY2.mul(maxRDiag, maxDY2, GMP_RNDU);
    maxRY2Tilde.mul(maxRDiag, maxY2Tilde, GMP_RNDU);
    maxDRY2.addmul(maxRY2Tilde, halfULP, GMP_RNDU);
    maxDE.add(maxDE, maxDRY2, GMP_RNDU);
    maxDE.mul(maxDE, K, GMP_RNDU);
    maxDE.addmul(maxDist, halfULP, GMP_RNDU);
  }

  return true;
}
