// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include <awali/dyn.hh>
#include<cora/edit.hh>
// #include <awali/dyn/loading/locations.hh>
#include <awali/sttc/misc/ato.hh>
#include <awali/sttc/misc/json.hh>
#include<cora/cora.hh>

#include<dirent.h>
#include<fstream>
#include<map>
#include<stdexcept>

namespace awali {
  namespace cora {

    dyn::automaton_t arg1;
    dyn::automaton_t arg2;
    dyn::automaton_t res;
    dyn::ratexp_t exp1;
    dyn::ratexp_t exp_res;
    std::string weight;
    int count;
    bool boolean;
    bool history;

    bool first_cmd=true; //to know whether - is the real standard input

arg_kind_t final_output;
void help() {
  std::cout << "\nBasic commands :" << std::endl;
  std::cout << "-----------------" << std::endl;
  for(auto c : commands_base) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nGeneric commands for automata and transducers :" << std::endl;
  std::cout << "------------------------------------------------" << std::endl;
  for(auto c : commands_aut_tdc) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nCommands for (weighted) automata :" << std::endl;
  std::cout << "-----------------------------------" << std::endl;
  for(auto c : commands_aut) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nCommands for standard automata :" << std::endl;
  std::cout << "---------------------------------" << std::endl;
  for(auto c : commands_std) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nCommands for NFA :" << std::endl;
  std::cout << "-------------------" << std::endl;
  for(auto c : commands_nfa) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nFactories :" << std::endl;
  std::cout << "------------" << std::endl;
  for(auto c : commands_fact) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nCommands for transducers :" << std::endl;
  std::cout << "---------------------------" << std::endl;
  for(auto c : commands_tdc) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nCommands for rational expressions :" << std::endl;
  std::cout << "------------------------------------" << std::endl;
  for(auto c : commands_exp) {
    std::cout << "  " << c.first;
    for(unsigned i= c.first.length(); i<22; ++i)
      std::cout << ' ';
    std::cout << c.second.desc << std::endl;
  }
  std::cout << "\nOptions :  (-Xvalue or --long-option=value)" << std::endl;
  std::cout << "---------" << std::endl;
  for(auto c : options) {
    std::cout << "  -" << c.first;
    for(unsigned i= c.first.length(); i<3; ++i)
      std::cout << ' ';
    std::cout << " --";
    std::cout << c.second.desc;
    for(unsigned i= c.second.desc.length(); i<10; ++i)
      std::cout << ' ';
    std::cout << c.second.longdesc <<  std::endl;
  }
  std::cout << "\nMisc :" << std::endl;
  std::cout << "------" << std::endl;
  std::cout << "* To read the argument on standard input, use -" << std::endl;
  std::cout << "* You can pipe commands with \\| :\n   $> cora cmd1 arg1 \\| cmd2 -" << std::endl;
  std::cout << "  In this case, '-' refers to the result of the last command" << std::endl;
  std::cout << "  When an option is set, its value remains for the next commands on the line" << std::endl;
  std::cout << "* To load stored automata, do not write the suffix .json" << std::endl;
  std::cout << std::endl;
  final_output=NONE;
}

void help(std::string s) {
  auto it = commands.find(s);
  if(it == commands.end())
    std::cout << s << " : unknown Awali command" << std::endl;
  else {
    std::cout << '\n' << it->second.longdesc << std::endl;
    std::cout << "Usage : " << it->first << ' ' << it->second.args << '\n' << std::endl;
  }
  final_output=NONE;
}

void output() {
  switch(final_output) {
  case AUT:
  case TDC:
    if(output_format == "json" || output_format == "default") {
      dyn::json(res,std::cout) << std::endl;
    }
    else if (output_format == "dot") {
      dyn::dot(res,std::cout,history) << std::endl;
    }
    else if (output_format == "daut") {
      dyn::daut(res,std::cout) << std::endl;
    }
    else if (output_format == "fado") {
      dyn::fado(res,std::cout) << std::endl;
    }
    else if (output_format == "grail") {
      dyn::grail(res,std::cout) << std::endl;
    }
    else if (output_format == "pdf") {
      dyn::pdf(res,std::cout) << std::endl;
    }
    else if (output_format == "svg") {
      dyn::svg(res,std::cout) << std::endl;
    }
    else
      std::cerr << output_format << " : unknown automata format" << std::endl;
    return;
  case EXP:
    if(output_format == "json") {
      exp_res->json(std::cout) << std::endl;
    }
    else if (output_format == "text" || output_format == "default") {
      exp_res->print(std::cout) << std::endl;
    }
    else
      std::cerr << output_format << " : unknown rational expression format" << std::endl;
    return;
  case BOOL:
      if(verbose)
	std::cout << std::boolalpha << boolean << std::endl;
      exit(!boolean);
  case INT:
    std::cout << count << std::endl;
    exit(count);
  case WEIGHT:
    std::cout << weight << std::endl;
    return;
  case WORD: //No such command right now
    return;
  case CMD: //No such command right now
    return;
  case SMR: //No such command right now
    return;
  case LBL: //No such command right now
    return;
  case ALPH: //No such command right now
    return;
  case FMT: //No such command right now
    return;
  case MTD: //No such command right now
    return;
  case NM: //No such command right now
    return;
  case NONE: return;
  }
}

void list() {
  std::cout << "Automata library" << std::endl;
  std::cout << "================" << std::endl;
  std::vector<dyn::loading::file_loc_t> ex =
      dyn::loading::example_automata();

  for (const dyn::loading::file_loc_t& p: ex) {
    std::cout << p.name;
    size_t u = p.name.size();
    for(unsigned i=u; i<15; ++i)
      std::cout << ' ';
    std::cout << "| ";
    std::ifstream fic(p.dir+'/'+p.name+"."+p.ext);
    std::string desc = sttc::get_first_attr(fic);
    if(desc=="Description") {
      desc = sttc::parsestring(fic);
      std::cout << desc;
    }
    fic.close();
    std::cout << std::endl;
  }
  final_output=NONE;
}

dyn::ratexp_t load_exp(std::string s) {
  if(s=="-") {
    if(first_cmd)
      return dyn::parse_ratexp(std::cin);
    if(final_output!=EXP)
      throw std::runtime_error("Rational expression expected on pseudo standard input");
    return exp_res;
  }
  if(input_format=="json") {
    std::ifstream fic(s);
    if(!fic.is_open()) {
      throw std::runtime_error(s+" : expression not found");
    }
    dyn::ratexp_t e = dyn::parse_ratexp(fic);
    fic.close();
    return e;
  }
  else if(input_format=="text" || input_format=="default") {
    if(letter_type=="char")
      return dyn::make_ratexp_with_alphabet(s,alphabet,semiring);
    else {
      int n=stoi(alphabet);
      return dyn::make_int_ratexp_with_range(s,0,n-1,semiring);
    }
  }
  throw std::runtime_error(input_format+" : unknown input format");
}

dyn::automaton_t load(std::string s) {
  if(s=="-") {
    if(first_cmd)
      return dyn::parse_automaton(std::cin, input_format);
    if(final_output!=AUT)
      throw std::runtime_error("Automaton expected on pseudo standard input");
    return res;
  }
  return dyn::load(s, input_format);
}


void process_option(const char* opt) {
  coms_t optcst;
  const char* arg=nullptr;
  if(opt[1]!='-') {
    auto it = options.find(std::string(1,opt[1]));
    if(it == options.end())
      throw std::runtime_error(std::string("Unknown option : -")+(opt+1));
    optcst=it->second.cst;
    arg=opt+2;
    if(it->second.argc==0 && *arg!='\0')
      throw std::runtime_error(std::string("Unexpected argument with option : ")+(opt[1]));
    if(it->second.argc==1 && *arg=='\0')
      throw std::runtime_error(std::string("Missing argument with option : ")+(opt[1]));
  }
  else {
    std::string sopt(opt);
    for(auto s : options) {
	std::string d=s.second.desc;
	unsigned l = d.length();
	if(sopt.compare(2,l,d)==0) {
	  if(opt[l+2]!='\0' && opt[l+2]!='=')
	    throw std::runtime_error(std::string("Unknown option : --")+(opt+2));
	  if(s.second.argc==0 && opt[l+2]=='=')
	    throw std::runtime_error(std::string("Unexpected argument with option : ")+(opt+2));
	  if(s.second.argc==1 && opt[l+2]=='\0')
	    throw std::runtime_error(std::string("Missing argument with option : ")+(opt+2));
	  optcst=s.second.cst;
	  arg=opt+3+l;
	  break;
	}
    }
    if(arg == nullptr)
      throw std::runtime_error(std::string("Unknown option : --")+(opt+2));
  }
  switch(optcst) {
  case INPUT_FMT:
    input_format=arg; break;
  case OUTPUT_FMT:
    output_format=arg; break;
  case ALPHABET:
    alphabet=arg; break;
  case ALPHABET_B:
    alphabetb=arg; break;
  case METHOD:
    algo=arg; break;
  case LABELS:
    letter_type=arg; alphabet="2"; break;
  case WEIGHTS:
    semiring=arg; break;
  case QUIET:
    verbose=false; break;
  case VERBOSE:
    verbose=true; break;
  case HISTORY:
    history=true; break;
  case NAME:
    name=arg; break;
  default: break;
  }
}

int main(int argc, const char** argv) {
  init();
  if(argc==1) {
    std::cout << "Please enter a Awali command:" << std::endl;
    help();
    return 1;
  }
  int i=1;
  do {
    unsigned j=0;
    for(; i<argc && argv[i][0]!='|' ; ++i) {
      if(argv[i][0]=='-' && argv[i][1]!='\0' && (argv[i][1]<'0' || argv[i][1]>'9')) {
        process_option(argv[i]);
      }
      else
         args[j++]=argv[i];
    }
    if(i<argc)
      ++i;
    if (j == 0) {
      std::cout << "Please enter a Awali command:" << std::endl;
      help();
      return 1;
    }
    auto it = commands.find(args[0]);
    if(it==commands.end()) {
      std::cout << args[0] << " : illegal Awali command" << std::endl;
      return 1;
    }

    if(it->second.cst == HELP) {
      if(j==1)
	help();
      else{
	if(j!=2) {
	  std::cerr << "Usage : help [cmd]" << std::endl;
	  return 1;
	}
	help(args[1]);
      }
      return 0;
    }

    //Check arg card
    if(j!=it->second.argc+1) {
      std::cerr << "Usage : " << it->first << ' ' << it->second.args << std::endl;
      return 1;
    }

    switch(it->second.cst) {
      //utilities
    case LIST :
      list(); break;
      //make-automatom
    case CAT :
      arg1=load(args[1]);
      res=arg1;
      final_output=AUT;
      break;
    case DISPLAY :
      arg1=load(args[1]);
      if ( output_format == "dot" )
        dyn::display(arg1,name,history);
      else
        dyn::pdfdisplay(arg1,name,history);
      final_output=NONE;
      break;
    case DISPLAY_EXP :
      exp1=load_exp(args[1]);
      dyn::display(dyn::draw_exp(exp1),name, true, false);
      final_output=NONE;
      break;
    case INFO :
      arg1=load(args[1]);
      std::cout << "\n Weights : " << arg1->get_context()->weightname() << '\t';
      std::cout << "Alphabet : ";
      {
	char sep='{';
	for(auto l: arg1->alphabet()) {
	  std::cout << sep << ' ' << l;
	  sep=',';
	}
	std::cout << " }\n" << std::endl;
      }
      std::cout << arg1->num_states() << " state(s)        \t";
      std::cout << arg1->num_transitions() << " transition(s)" << std::endl;
      std::cout << arg1->num_initials() << " initial state(s)\t";
      std::cout << arg1->num_finals() << " final state(s)\n" << std::endl;
      final_output=NONE;
      break;
    case EMPTY_AUT :
      if(letter_type=="char")
	res=dyn::make_automaton(alphabet,semiring);
      if(letter_type=="int") {
	int n= std::stoi(alphabet);
	res=dyn::make_int_automaton(n,semiring);
      }
      if(output_format=="default") {
	dyn::edit(res);
	final_output=NONE;
	break;
      }
      final_output=AUT;
      break;
    case STRIP :
      arg1=load(args[1]);
      arg1->strip_history();
      res=arg1;
      final_output=AUT;
      break;
      // aut -> int
    case NUM_STATE :
    case NUM_INIT :
    case NUM_FIN :
    case NUM_TRANS :
      arg1=load(args[1]);
      switch(it->second.cst) {
      case NUM_STATE :
	count = arg1->num_states();
	break;
      case NUM_INIT :
	count = arg1->num_initials();
	break;
      case NUM_FIN :
	count = arg1->num_finals();
	break;
      case NUM_TRANS :
	count = arg1->num_transitions();
	break;
      default: {}
      }
      final_output=INT;
      break;
    case SUPPORT :
      arg1=load(args[1]);
      res = dyn::support(arg1);
      final_output=AUT;
      break;
    case CHARACTERISTIC :
      arg1=load(args[1]);
      res = dyn::characteristic(arg1,semiring);
      final_output=AUT;
      break;
    case ALLOW_EPS :
      arg1=load(args[1]);
      res = dyn::allow_eps_transition(arg1);
      final_output=AUT;
      break;
      //make-ratexp
    case AUT_TO_EXP :
      arg1=load(args[1]);
      exp_res = dyn::aut_to_exp(arg1);
      final_output=EXP;
      break;
      // exp -> exp
    case CAT_EXP :
    case EXPAND :
    case SNF :
      exp1=load_exp(args[1]);
      switch(it->second.cst) {
      case CAT_EXP :
	exp_res=exp1;
	break;
      case EXPAND :
	exp_res = dyn::expand(exp1);
	break;
      case SNF :
	exp_res = dyn::star_normal_form(exp1);
	break;
      default: {}
      }
      final_output=EXP;
      break;
     case VALID_EXP:
      exp1 =load_exp(args[1]);
      boolean = dyn::is_valid(exp1);
      final_output=BOOL;
      break;
    case CST_TERM :
      {
	exp1=load_exp(args[1]);
	auto ctx = exp1->get_context();
	weight = ctx->weight_to_string(dyn::constant_term(exp1));
	final_output=WEIGHT;
      }
      break;

    case STAR_HEIGHT :
      exp1=load_exp(args[1]);
      count = dyn::star_height(exp1);
      final_output=INT;
      break;
      //accessible
    case NUM_ACC :
      arg1=load(args[1]);
      count = dyn::num_accessible_states(arg1);
      final_output=INT;
      break;
    case NUM_COACC :
      arg1=load(args[1]);
      count = dyn::num_coaccessible_states(arg1);
      final_output=INT;
      break;
    case ACCESSIBLE :
      arg1=load(args[1]);
      dyn::accessible_here(arg1);
      res=arg1;
      final_output=AUT;
      break;
    case COACC :
      arg1=load(args[1]);
      dyn::coaccessible_here(arg1);
      res=arg1;
      final_output=AUT;
      break;
    case TRIM :
      arg1=load(args[1]);
      dyn::trim_here(arg1);
      res=arg1;
      final_output=AUT;
      break;
    case IS_TRIM :
      arg1=load(args[1]);
      boolean = dyn::is_trim(arg1);
      final_output=BOOL;
      break;
    case IS_USELESS :
      arg1=load(args[1]);
      boolean = dyn::is_useless(arg1);
      final_output=BOOL;
      break;
    case IS_ACC :
      arg1=load(args[1]);
      boolean = dyn::is_accessible(arg1);
      final_output=BOOL;
      break;
    case IS_COACC :
      arg1=load(args[1]);
      boolean = dyn::is_coaccessible(arg1);
      final_output=BOOL;
      break;
    case IS_EMPTY :
      arg1=load(args[1]);
      boolean = dyn::is_empty(arg1);
      final_output=BOOL;
      break;
      //factor
    case PREFIX : case SUFFIX : case FACTOR :
      arg1=load(args[1]);
      switch(it->second.cst) {
      case PREFIX : dyn::prefix_here(arg1); break;
      case SUFFIX : dyn::suffix_here(arg1); break;
      case FACTOR : dyn::factor_here(arg1); break;
      default : break;
      }
      res=arg1;
      final_output=AUT;
      break;
      //derivation
    case EXP_TO_AUT :
      exp1=load_exp(args[1]);
      if(algo=="standard")
	res = dyn::standard(exp1);
      else if(algo=="breaking")
	res = dyn::derived_term(exp1, true);
      else if(algo=="thompson")
	res = dyn::thompson(exp1);
      else if(algo=="zpc")
	res = dyn::zpc(exp1);
      else
	res = dyn::derived_term(exp1);
      final_output=AUT;
      break;
    case DERIVATION :
      exp1=load_exp(args[1]);
      {
	std::map<dyn::ratexp_t, dyn::any_t> map;
	if(algo=="breaking")
	  map = dyn::derivation(exp1, args[2], true);
	else
	  map = dyn::derivation(exp1, args[2]);
	for(auto p: map) {
	  std::cout << '<' << p.second << "> ";
	  p.first->print(std::cout) << std::endl;
	}
      }
      final_output=NONE;
      break;
      //determinize
    case DETERMINIZE :
      arg1=load(args[1]);
      res = dyn::determinize(arg1);
      final_output=AUT;
      break;
    case COMPLEMENT :
      arg1=load(args[1]);
      dyn::complement_here(arg1);
      res=arg1;
      final_output=AUT;
      break;
    case COMPLETE :
      arg1=load(args[1]);
      dyn::complete_here(arg1);
      res=arg1;
      final_output=AUT;
      break;
    case REDUCE :
      arg1=load(args[1]);
      res = dyn::reduce(arg1);
      final_output=AUT;
      break;
    case L_REDUCE :
      arg1=load(args[1]);
      res = dyn::left_reduce(arg1);
      final_output=AUT;
      break;
    case R_REDUCE :
      arg1=load(args[1]);
      res = dyn::right_reduce(arg1);
      final_output=AUT;
      break;
    case IS_DET :
      arg1=load(args[1]);
      boolean = dyn::is_deterministic(arg1);
      final_output=BOOL;
      break;
    case IS_COMPLETE :
      arg1=load(args[1]);
      boolean = dyn::is_complete(arg1);
      final_output=BOOL;
      break;
    case IS_AMB :
      arg1=load(args[1]);
      boolean = dyn::is_ambiguous(arg1);
      final_output=BOOL;
      break;
    case ARE_EQVT :
      arg1=load(args[1]);
      arg2=load(args[2]);
      boolean = dyn::are_equivalent(arg1, arg2);
      final_output=BOOL;
      break;
      //eval
    case EVAL :
      {
	arg1=load(args[1]);
	auto ctx = arg1->get_context();
	weight = ctx->weight_to_string(dyn::eval(arg1, args[2]));
	final_output=WEIGHT;
      }
      break;
    case ENUMERATE :
    case SHORTEST :
      {
	arg1 =load(args[1]);
	unsigned n = sttc::atou(args[2]);
	std::map<dyn::any_t,dyn::weight_t> map;
	switch(it->second.cst) {
	case ENUMERATE :
	  map= dyn::enumerate(arg1,n);
	  break;
	case SHORTEST :
	  map= dyn::shortest(arg1,n);
	  break;
	default : {}
	}
	for(auto p: map) {
	  std::cout << '<' << p.second << "> " << p.first << std::endl;
	}
	final_output=NONE;
      }
      break;
      //factories
    case DE_BRUIJN:
    case DOUBLERING:
    case LADYBIRD:
    case CERNY:
    case WITNESS:
      {
	unsigned n=sttc::atou(args[1]);
	switch(it->second.cst) {
	case DE_BRUIJN:
	  res = dyn::de_bruijn(n,alphabet[0]);
	  break;
	case DOUBLERING:
	  res = dyn::double_ring(n,{0},alphabet[0]);
	  break;
	case LADYBIRD:
	  res = dyn::ladybird(n,alphabet[0]);
	  break;
	case CERNY:
	  res = dyn::cerny(n,alphabet[0]);
	  break;
	case WITNESS:
	  res = dyn::witness(n,alphabet[0]);
	  break;
	default : {}
	}
	final_output=AUT;
      }
      break;
    case DIVKBASEB:
      {
	unsigned k=sttc::atou(args[1]);
	unsigned b=sttc::atou(args[2]);
	res = dyn::divkbaseb(k,b,alphabet[0]);
	final_output=AUT;
      }
      break;
    case MINIMALE:
      exp1 =load_exp(args[1]);
      if(algo=="brzozowski")
	res = dyn::minimal_automaton(exp1,BRZOZOWSKI);
      else
      if(algo=="hopcroft")
	res = dyn::minimal_automaton(exp1,HOPCROFT);
      else
	res = dyn::minimal_automaton(exp1,MOORE);
      final_output=AUT;
      break;
    case MINIMAL:
      arg1 =load(args[1]);
      if(algo=="brzozowski")
	res = dyn::minimal_automaton(arg1,BRZOZOWSKI);
      else
      if(algo=="hopcroft")
	res = dyn::minimal_automaton(arg1,HOPCROFT);
      else
	res = dyn::minimal_automaton(arg1,MOORE);
      final_output=AUT;
      break;
    case MINQUOTIENT:
      arg1 =load(args[1]);
      if(algo=="hopcroft")
	res = dyn::min_quotient(arg1, HOPCROFT);
      else
	res = dyn::min_quotient(arg1, MOORE);
      final_output=AUT;
      break;
    case MINCOQUOTIENT:
        arg1 =load(args[1]);
      if(algo=="hopcroft")
	res = dyn::min_coquotient(arg1, HOPCROFT);
      else
	res = dyn::min_coquotient(arg1, MOORE);
      final_output=AUT;
      break;
    case PRODUCT:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      res = dyn::product(arg1, arg2);
      final_output=AUT;
      break;
    case POWER:
      {
	arg1 =load(args[1]);
	unsigned n =sttc::atou(args[2]);
	res = dyn::power(arg1, n);
	final_output=AUT;
      }
      break;
    case SHUFFLE:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      res = dyn::shuffle(arg1, arg2);
      final_output=AUT;
      break;
    case INFILTRATION:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      res = dyn::infiltration(arg1, arg2);
      final_output=AUT;
      break;
    case UNION:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      res = dyn::Union(arg1, arg2);
      final_output=AUT;
      break;
    case ARE_ISOMORPHIC:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      boolean = dyn::are_isomorphic(arg1, arg2);
      final_output=BOOL;
      break;
      //proper
    case PROPER:
      arg1 =load(args[1]);
      if(algo=="forward")
	res=dyn::proper(arg1, param_t::FORWARD);
      else
	res=dyn::proper(arg1);
      final_output=AUT;
      break;
    case IS_PROPER:
      arg1 =load(args[1]);
      boolean = dyn::is_proper(arg1);
      final_output=BOOL;
      break;
    case IS_VALID:
      arg1 =load(args[1]);
      boolean = dyn::is_valid(arg1);
      final_output=BOOL;
      break;
      // standard
    case IS_STANDARD:
      arg1 =load(args[1]);
      boolean = dyn::is_standard(arg1);
      final_output=BOOL;
      break;
    case STANDARD:
      arg1 =load(args[1]);
      dyn::standard_here(arg1);
      res=arg1;
      final_output=AUT;
      break;
    case CONCATENATION:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      res = dyn::concatenate(arg1,arg2);
      final_output=AUT;
      break;
    case SUM:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      if(arg1==arg2)
	res = dyn::sum(arg1,arg2);
      else {
	dyn::sum_here(arg1,arg2);
	res=arg1;
      }
      final_output=AUT;
      break;
    case STAR:
      arg1 =load(args[1]);
      dyn::star_here(arg1);
      res = arg1;
      final_output=AUT;
      break;
    case LEFT_MULT:
      arg1 =load(args[2]);
      dyn::left_mult_here(arg1, arg1->get_context()->get_weight(args[1]));
      res = arg1;
      final_output=AUT;
      break;
    case RIGHT_MULT:
      arg1 =load(args[1]);
      dyn::right_mult_here(arg1, arg1->get_context()->get_weight(args[2]));
      res = arg1;
      final_output=AUT;
      break;
      //transpose
    case TRANSPOSE:
      arg1 =load(args[1]);
      dyn::transpose_here(arg1);
      res = arg1;
      final_output=AUT;
      break;
      //transducer
    case TDC_DOMAIN:
      arg1 =load(args[1]);
      res = dyn::domain(arg1);
      final_output=AUT;
      break;
    case IMAGE:
      arg1 =load(args[1]);
      res = dyn::image(arg1);
      final_output=AUT;
      break;
    case INVERSE:
      arg1 =load(args[1]);
      res = dyn::inverse(arg1);
      final_output=AUT;
      break;
    case COMPOSE:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      res = dyn::compose(arg1, arg2);
      final_output=AUT;
      break;
    case PARTIAL_ID:
      arg1 =load(args[1]);
      res = dyn::partial_identity(arg1);
      final_output=AUT;
      break;
    case EVAL_T:
      arg1 =load(args[1]);
      arg2 =load(args[2]);
      res = dyn::eval(arg1, arg2);
      final_output=AUT;
      break;
    case EVAL_TW:
      arg1 =load(args[1]);
      exp_res = dyn::eval_word(arg1, args[2]);
      final_output=EXP;
      break;
    case IS_FUNC:
      arg1 =load(args[1]);
      boolean = dyn::is_functional(arg1);
      final_output=BOOL;
      break;
    case IS_SYNCHRONIZABLE:
      arg1 =load(args[1]);
      boolean = dyn::is_synchronizable(arg1);
      final_output=BOOL;
      break;
    case SYNCHRONIZE:
      arg1 =load(args[1]);
      res = dyn::synchronize(arg1);
      final_output=AUT;
      break;
    case REALTIME:
      arg1 =load(args[1]);
      res = dyn::realtime(arg1);
      final_output=AUT;
      break;
      /*
    case TDC_AUT:
      arg1 =load(args[1]);
      res = dyn::lift_tdc(arg1);
      final_output=AUT;
      break;
      */
    case EMPTY_TDC :
      res=dyn::make_transducer({alphabet,alphabetb},semiring);
      if(output_format=="default") {
	dyn::edit(res);
	final_output=NONE;
	break;
      }
      final_output=AUT;
      break;
    case RAND_DFA :
      {
	unsigned n=sttc::atou(args[1]);
	res=dyn::generateDFA(n,alphabet);
	final_output=AUT;
      }
      break;
    case EDIT :
      arg1 =load(args[1]);
      dyn::edit(arg1);
      final_output=NONE;
      break;
    default : break;// Does not happen ! Just here to avoid warnings
    }
    first_cmd=false;
  } while(i!=argc);
  output();
  return 0;
}

}}//end of ns awali::cora


int main(int argc, const char** argv) {
  return awali::cora::main(argc, argv);
}
