# This file is part of Awali.
# Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
#
# Awali is a free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

# distutils: language = c++
import os, tempfile, subprocess
from libcpp.string cimport string
from libcpp.vector cimport vector
from libcpp cimport bool
#from cython.operator cimport dereference as deref
from libcpp.map cimport map
from awalip import run_from_ipython_notebook
 
cpdef dict get_map_s_s(map[string, string] m):
    pymap = {}
    for it in m: #Iterate through the c++ map
        pymap[it.first] = it.second
    return pymap

cdef extern from "<set>" namespace "std" nogil:
    cdef cppclass set[T]:
        set() except +
        set(set&) except +

cdef extern from "<memory>" namespace "std":
    cdef cppclass shared_ptr[T]:
        shared_ptr()
        shared_ptr(T*p)
        shared_ptr(const shared_ptr&)
        long use_count()
        T operator*()

cdef extern from "<iostream>" namespace "std":
    cdef cppclass ostream:
        ostream& write(const char*, int) except +

cdef extern from "<sstream>" namespace "std":
    cdef cppclass ostringstream:
        ostream& operator<<(any&) except +
        ostream& operator<<(ostream& (*pf)(ostream&)) except +
        string str()

cdef extern from "<iostream>" namespace "std::ios_base":
    cdef cppclass open_mode:
        pass
    cdef open_mode binary

cdef extern from "<memory>" namespace "std":
    cdef cppclass shared_ptr[T]:
        shared_ptr()
        shared_ptr(T*p)
        shared_ptr(const shared_ptr&)
        long use_count()
        T operator*()

cdef extern from "<fstream>" namespace "std":
    cdef cppclass ofstream(ostream):
        ofstream(const char*) except +
        ofstream(const char*, open_mode) except +

cdef extern from "automaton.h" namespace "dyn":    
    ctypedef unsigned int size_t
    ctypedef unsigned int transition_t

    cdef cppclass any[T]:
        any(any)
        any(string)
        any(T)
        ostream& operator<<(ostream& o, const any& a)

    cdef struct automatondyn:
        automatondyn()

    cdef shared_ptr[automatondyn] make_automaton(string, string)

    cdef simple_automaton_t make_simple_automaton1(string)

    cdef simple_automaton_t make_simple_automaton2(string, string)
    
    cdef simple_automaton_t make_simple_transducer1(vector[string])
    
    cdef simple_ratexp_t make_simple_ratexp1(string)
    cdef simple_ratexp_t make_simple_ratexp2(string, string)

    cdef struct simple_ratexp_t:
        string json()
        string print_()

    cdef struct simple_automaton_t:
        size_t num_states()
        size_t num_initials()
        size_t num_finals()
        size_t num_transitions()
        bool has_state(int s)
        bool is_initial(int s)
        bool is_final(int)
        string get_initial_weight(int s)
        string get_final_weight(int s)
        transition_t get_transition(int src, int dst, string label)
        bool has_transition3(int src, int dst, string label)
        bool has_transition1(transition_t t)
        int src_of(transition_t t)
        int dst_of(transition_t t)
        string label_of(transition_t t)
        string weight_of(transition_t t)
        int add_state()
        void del_state(int s)
        void set_initial2(int s, string w)
        void set_initial1(int s)
        string add_initial(int s, string w)
        void unset_initial(int s)
        void set_final2(int s, string w)
        void set_final1(int s)
        string add_final(int s, string w)
        void unset_final(int s)
        void del_transition1(transition_t tr)
        void del_transition3(int src, int dst, string label)
        void del_transition2(int src, int dst)
        transition_t set_transition4(int src, int dst, string label, string weight)
        transition_t set_transition3(int src, int dst, string label)
        void set_tdc_transition3(int src, int dsl, vector[string] labels)
        transition_t set_eps_transition2(int src, int dst)
        transition_t set_eps_transition3(int src, int dst, string weight)
        string add_transition4(int src, int dst, string label, string weight)
        string add_transition3(int src, int dst, string label)
        string add_eps_transition2(int src, int dst)
        string add_eps_transition3(int src, int dst, string weight)
        string set_weight(transition_t tr, string weight)
        string add_weight(transition_t tr, string weight)
        string lmul_weight(transition_t tr, string weight)
        string rmul_weight(transition_t tr, string weight)
        vector[int] states()
        vector[transition_t] transitions()
        vector[int] initial_states()
        vector[transition_t] initial_transitions()
        vector[int] final_states()
        vector[transition_t] final_transitions()
        vector[int] successors1(int src, string label)
        vector[int] successors2(int s)
        vector[int] predecessors1(int src, string label)
        vector[int] predecessors2(int s)
        vector[transition_t] outin(int src, int dst)
        vector[transition_t] incoming1(int s)
        vector[transition_t] incoming2(int s, string label)
        vector[transition_t] outgoing1(int s)
        vector[transition_t] outgoing2(int s, string label)
        vector[transition_t] in1(int s)
        vector[transition_t] in2(int s, string label)
        vector[transition_t] out1(int s)
        vector[transition_t] out2(int s, string label)
        int pre()
        int post()
        vector[int] all_states()
        vector[transition_t] all_transitions()
        vector[transition_t] all_out(int s)
        vector[string] alphabet()
        vector[vector[string]] alphabets()
        bool has_letter(string l)
        bool is_transducer()
        string get_static_context()
        void strip_history()
        string get_state_name(int s)
        string weight_one()
        string weight_zero()
        string get_weightset()
        string json()
        void release()
        
    cdef struct basic_automaton_t:
            #basic_automaton_t() : aut_() {}
    #
    #basic_automaton_t(const std::shared_ptr<automatondyn>& a) : aut_(a) {}
    #
    #basic_automaton_t(const automaton_t& a) : aut_((std::shared_ptr<automatondyn>)a) {}
        #cdef operator automaton_t()
        size_t num_states()
        size_t num_initials()
        size_t num_finals()
        size_t num_transitions()
        bool has_state(int s)
        bool is_initial(int s)
        bool is_final(int s)
        string get_initial_weight(int s)
        string get_final_weight(int s)
        bool has_transition1(transition_t t)
        int src_of(transition_t t) 
        int dst_of(transition_t t)
        string weight_of(transition_t t)
        int add_state()
        void del_state(int s)
        set_initial2(int s, string w)
        void set_initial1(int s)
        string add_initial(int s, string w)
        void unset_initial(int s)
        void set_final2(int s, string w)
        void set_final1(int s)
        string add_final(int s, string w)
        void unset_final(int s)
        void del_transition1(transition_t tr)
        void del_transition2(int src, int dst)
        transition_t set_eps_transition2(int src, int dst)
        transition_t set_eps_transition3(int src, int dst, string weight)
        string add_eps_transition2(int src, int dst)
        string add_eps_transition3(int src, int dst, string weight)
        string set_weight(transition_t tr, string weight)
        string add_weight(transition_t tr, string weight)
        string lmul_weight(transition_t tr, string weight)
        string rmul_weight(transition_t tr, string weight)
        vector[int] states()
        vector[transition_t] transitions()
        vector[int] initial_states()
        vector[transition_t] initial_transitions()
        vector[int] final_states()
        vector[transition_t] final_transitions()
        vector[int] successors2(int src)
        vector[int] predecessors2(int src)
        vector[transition_t] outin(int src, int dst)
        vector[transition_t] incoming1(int s)
        vector[transition_t] outgoing1(int s)
        vector[transition_t] in1(int s)
        vector[transition_t] out1(int s)
        int pre()
        int post()
        vector[int] all_states()
        vector[transition_t] all_transitions()
        vector[transition_t] all_out(int s)
        vector[transition_t] all_in(int s)
        bool is_transducer()
        string get_static_context()
        void strip_history()
        string get_state_name(int s)
        string weight_one()
        string weight_zero()
        string get_weightset()
        string json()
        void release()
      
      
    #cdef struct simple_transducer_t:


    cdef struct simple_ratexp_t:
        string print_()
        string json()

    cdef vector[int] accessible_states_(simple_automaton_t aut)
    cdef vector[int] coaccessible_states_(simple_automaton_t aut)
    cdef vector[int] useful_states_(simple_automaton_t aut)
    cdef size_t num_accessible_states_(simple_automaton_t aut)
    cdef size_t num_coaccessible_states_(simple_automaton_t aut)
    cdef simple_automaton_t accessible_(simple_automaton_t aut)
    cdef simple_automaton_t coaccessible_(simple_automaton_t aut)
    cdef simple_automaton_t trim_(simple_automaton_t aut)
    cdef void accessible_here_(simple_automaton_t aut)
    cdef void coaccessible_here_(simple_automaton_t aut)
    cdef void trim_here_(simple_automaton_t aut)
    cdef bool is_trim_(simple_automaton_t aut)
    cdef bool is_useless_(simple_automaton_t aut)
    cdef bool is_accessible_(simple_automaton_t aut)
    cdef bool is_coaccessible_(simple_automaton_t aut)
    cdef bool is_empty_(simple_automaton_t aut)
    cdef simple_automaton_t make_weighted_automaton_(string alphabet, string semiring)
    cdef simple_automaton_t make_NFA_(string alphabet)
    cdef simple_automaton_t make_weighted_automaton_with_eps_(string alphabet, string semiring)
    cdef simple_automaton_t make_NFA_with_eps_(string alphabet)
    cdef simple_automaton_t parse_json_automaton_(string s)
    cdef simple_automaton_t copy_(simple_automaton_t aut)
    cdef simple_automaton_t support_(simple_automaton_t aut)
    cdef simple_automaton_t characteristic_(simple_automaton_t aut, string semiring)
    cdef int add_state_(simple_automaton_t aut, string s)
    cdef simple_automaton_t determinize_(simple_automaton_t aut, bool history)
    cdef bool are_equivalent_(simple_automaton_t aut, simple_automaton_t aut2)
    cdef simple_automaton_t complement_(simple_automaton_t aut)
    cdef void complement_here_(simple_automaton_t aut)
    cdef simple_automaton_t complete_(simple_automaton_t aut)
    cdef void complete_here_(simple_automaton_t aut)
    cdef simple_automaton_t reduce_(simple_automaton_t aut)
    cdef bool is_deterministic_(simple_automaton_t aut)
    cdef bool is_complete_(simple_automaton_t aut)
    cdef bool is_ambiguous_(simple_automaton_t aut)
    cdef string eval_(simple_automaton_t aut, string word)
    cdef map[string, string] enumerate_(simple_automaton_t aut, unsigned m)
    cdef map[string, string] shortest_(simple_automaton_t aut, unsigned m)
    cdef void prefix_here_(simple_automaton_t aut)
    cdef simple_automaton_t prefix_(simple_automaton_t aut)
    cdef void suffix_here_(simple_automaton_t aut)
    cdef simple_automaton_t suffix_(simple_automaton_t aut)
    cdef void factor_here_(simple_automaton_t aut)
    cdef simple_automaton_t factor_(simple_automaton_t aut)
    cdef simple_automaton_t make_de_bruijn_(unsigned n, char first_letter)
    cdef simple_automaton_t make_divkbaseb_(unsigned k, unsigned b, char first_letter)
    cdef simple_automaton_t make_double_ring_(unsigned n, vector[unsigned] finals, char first_letter)
    cdef simple_automaton_t make_ladybird_(unsigned n, char first_letter)
    cdef simple_automaton_t make_cerny_(unsigned n, char first_letter)
    cdef simple_automaton_t make_witness_(unsigned n, char first_letter)
    cdef simple_automaton_t minimize_brzozowski_(simple_automaton_t aut)
    cdef simple_automaton_t minimize_moore_(simple_automaton_t aut)
    cdef simple_automaton_t minimize_(simple_automaton_t aut)
    cdef simple_automaton_t minimal_(simple_automaton_t aut)
    cdef simple_automaton_t minimal_quotient_(simple_automaton_t aut)
    cdef simple_automaton_t quotient_(simple_automaton_t aut, vector[vector[int]] equiv)
    cdef string dot_(simple_automaton_t aut, bool history, bool dot2tex)
    cdef string fado_(simple_automaton_t aut)
    cdef string grail_(simple_automaton_t aut)
    cdef string json_(simple_automaton_t aut)
    cdef simple_automaton_t fado_parse_(string s)
    cdef simple_automaton_t grail_parse_(string s)
    cdef simple_automaton_t product_(simple_automaton_t aut1, simple_automaton_t aut2)
    cdef simple_automaton_t power_(simple_automaton_t aut, unsigned n)
    cdef simple_automaton_t shuffle_(simple_automaton_t aut1, simple_automaton_t aut2)
    cdef simple_automaton_t infiltration_(simple_automaton_t aut1, simple_automaton_t aut2)
    cdef simple_automaton_t Union_(simple_automaton_t aut1, simple_automaton_t aut2)
    cdef bool are_isomorphic_(simple_automaton_t aut1, simple_automaton_t aut2)
    cdef void proper_here_(simple_automaton_t aut, bool backward, bool prune)
    cdef simple_automaton_t proper_(simple_automaton_t aut, bool backward, bool prune)
    cdef bool is_proper_(simple_automaton_t aut)
    cdef bool is_valid_(simple_automaton_t aut)
    cdef simple_automaton_t allow_eps_transition_(simple_automaton_t aut)
    cdef simple_automaton_t thompson_(simple_ratexp_t exp)
    cdef simple_automaton_t exp_to_aut_(simple_ratexp_t exp)
    cdef simple_automaton_t generate_DFA_(unsigned size, string alphabet)
    cdef simple_ratexp_t make_weighted_ratexp_with_alphabet_(string exp, string alph, string sr)
    cdef simple_ratexp_t make_regexp_with_alphabet_(string exp, string alph)
    cdef simple_ratexp_t make_weighted_ratexp_(string exp, string sr)
    cdef simple_ratexp_t make_regexp_(string exp)
    cdef simple_ratexp_t aut_to_exp_(simple_automaton_t aut)
    cdef simple_automaton_t lift_(simple_automaton_t aut)
    cdef simple_ratexp_t expand_(simple_ratexp_t exp)
    cdef simple_ratexp_t star_normal_form_(simple_ratexp_t exp)
    cdef unsigned star_height_(simple_ratexp_t exp)
    cdef simple_ratexp_t parse_json_ratexp_(string s)
    cdef bool is_standard_(simple_automaton_t aut)
    cdef simple_automaton_t standard_(simple_automaton_t aut)
    cdef void standard_here_(simple_automaton_t aut)
    cdef simple_automaton_t concatenate_(simple_automaton_t aut, simple_automaton_t aut2)
    cdef void concatenate_here_(simple_automaton_t aut, simple_automaton_t aut2)
    cdef simple_automaton_t sum_(simple_automaton_t aut, simple_automaton_t aut2)
    cdef void sum_here_(simple_automaton_t aut, simple_automaton_t aut2)
    cdef simple_automaton_t star_(simple_automaton_t aut)
    cdef void star_here_(simple_automaton_t aut)
    cdef simple_automaton_t standard_(simple_ratexp_t exp)
    cdef simple_automaton_t synchronize_(simple_automaton_t tdc)
    cdef bool is_synchronizable_(simple_automaton_t tdc)
    cdef void display_(simple_automaton_t aut)
    cdef simple_automaton_t load_automaton_(string s, string fmt)
    cdef void save_automaton_(simple_automaton_t aut, string s, string fmt)
    cdef simple_automaton_t copy_(simple_automaton_t aut)
    cdef simple_automaton_t compose_(simple_automaton_t tdc1, simple_automaton_t tdc2);

cdef automaton refurbish_automaton(simple_automaton_t aut):
    a = automaton("","hack!!")
    a._this = aut
    return a
#cdef transducer refurbish_transducer(simple_automaton_t tdc):
    #t = transducer("","hack!!")
    #t._this = tdc
    #return t

cpdef automaton accessible(automaton aut):
    return refurbish_automaton(accessible_(aut._this))

cpdef bool is_standard(automaton aut):
    """
    Usage:  test = is_standard(automaton)
    
    Description:  Returns true if <automaton> is standard.
    
    Parameter: 
      - automaton: a awali.automaton
      
    Returns:  a bool
    
    Same as:  test = automaton.is_standard()
    """
    return is_standard_(aut._this)

cpdef automaton standard(automaton aut):
    """
    Usage:  automaton1 = standard(automaton2)
    
    Description:  Return the standard <automaton1> corresponding to <automaton2>.
    
    Parameter:
      - automaton: a awali.automaton
      
    Returns:  a awali.automaton
    
    Same as:  automaton1 = automaton2.standard()
    """
    return refurbish_automaton(standard_(aut._this))

cpdef void standard_here(automaton aut):
    """
    Usage:  standard_here(automaton)
    
    Description:  Modify <automaton> in order the make it standard.
    
    Parameters:
      - automaton: a awali.automaton
      
    Returns:  void
    
    Same as:  automaton.standard_here()
    """
    standard_here_(aut._this)

cpdef string weight_of(automaton aut, transition_t t):
    """
    Usage:  weight = weight_of(automaton, transition)
    
    Description:  returns the <weight> of <transition> within <automaton>
    
    Parameters:
      - automaton: a awali.automaton
      - transition: an int, must refer to an existing transition in <automaton>
    
    Returns:  a str, representing a value within the weightset of <automaton>.
    
    Same as:  weight = automaton.weight_of(transition)
    """
    return aut.weight_of(t)

cpdef automaton concatenate(automaton aut, automaton aut2):
    return refurbish_automaton(concatenate_(aut._this, aut2._this))

cpdef void concatenate_here(automaton aut, automaton aut2):
    concatenate_here_(aut._this, aut2._this)

cpdef automaton sum(automaton aut, automaton aut2):
    """
    Usage:  automaton3 = sum(automaton1, automaton2)
    
    Description:  returns the sum of <automaton1> and <automaton2>, that is <automaton3> 
    is the union of <automaton1> and <automaton2> working in parallel.
    
    Parameters:
      - automaton1: a awali.automaton
      - automaton2: a awali.automaton
    
    Returns:  a awali.automaton
    
    Same as: automton3 = automaton1.sum(automaton2)
    """
    return refurbish_automaton(sum_(aut._this, aut2._this))

cpdef void sum_here(automaton aut, automaton aut2):
    sum_here_(aut._this, aut2._this)

cpdef automaton star(automaton aut):
    return refurbish_automaton(star_(aut._this))

cpdef void star_here(automaton aut):
    star_here_(aut._this)

# cpdef automaton standard(ratexp exp):
#     return refurbish_automaton(standard_(exp._this))

cpdef vector[int] states(automaton aut):
    """
    Usage:  state_set = states(automaton)
    
    Description:  Returns the <state_set> of <automaton>.
    
    Parameter:
      - automaton:  a awali.automaton
    
    Returns:  a vector of int
    
    Same as:  state_set = automaton.states()
    """
    return aut.states()

cpdef automaton synchronize(automaton tdc):
    return refurbish_automaton(synchronize_(tdc._this))

cpdef bool is_synchronizable(automaton tdc):
    return is_synchronizable_(tdc._this)

cpdef void display(automaton aut, filename=None):
    """
    Usage:  display(automaton [, filename ] )
        
    Description:  displays <automaton> (require the program graphviz). If the optional argument is present, the automaton representation is in addition written to file named "<filename>.gv" . 
        
    Parameters:
      - automaton: a vancansonr.automaton 
      - filename: a str, optional 
    
    Returns:  void
    
    Same as:  automaton.display(filename)
    """
    aut.display(filename)

cpdef automaton load_automaton(string s, string fmt='json'):
    """
    Usage:  automaton = load_automaton( filename [, format='json'] )
                                    
    Description: Reads file <filename> assuming it has the given <format>, and returns the parsed <automaton>.
    
    Parameters:
      - filename: a str
      - format: a str of { 'dot', 'fado', 'grail', 'json' } , optional, default is 'json'
    
    Returns: a awali.automaton
    """
    return refurbish_automaton(load_automaton_(s, fmt))

cpdef void save_automaton(automaton aut, string s, string fmt = 'json'):
    """
    Usage:  save_automaton(automaton, filename [, format='json' ] )
                                    
    Description:  Writes <automaton> in file <filename> using <format>.
    
    Parameters:
    - automaton: a awali.automaton
    - filename: a str
    - format: a str of { 'dot', 'fado', 'grail', 'json' } , optional, default is 'json'
    
    Returns:  void
    """
    save_automaton_(aut._this, s, fmt)

cpdef ratexp parse_json_ratexp(string s):
    return refurbish_ratexp(parse_json_ _(s))

cpdef unsigned star_height(ratexp exp):
    return star_height_(exp._this)

cpdef ratexp star_normal_form(ratexp exp):
    return refurbish_ratexp(star_normal_form_(exp._this))

cpdef ratexp expand(ratexp exp):
    return refurbish_ratexp(expand_(exp._this))

cpdef automaton lift(automaton aut):
    return refurbish_automaton(lift_(aut._this))

cpdef string lmul_weight(automaton aut, transition_t tr, str weight):
    return aut.lmul_weight(tr, weight)

cpdef ratexp aut_to_exp(automaton aut):
    return refurbish_ratexp(aut_to_exp_(aut._this))

cpdef ratexp make_regexp(string exp):
    return refurbish_ratexp(make_regexp_(exp))

cpdef ratexp make_weighted_ratexp(string exp, string sr):
    return refurbish_ratexp(make_weighted_ratexp_(exp, sr))

cpdef ratexp make_regexp_with_alphabet(string exp, string alph):
    return refurbish_ratexp(make_regexp_with_alphabet_(exp, alph))

cpdef ratexp make_weighted_ratexp_with_alphabet(string exp, string alph, string sr):
    return refurbish_ratexp(make_weighted_ratexp_with_alphabet_(exp, alph, sr))

cpdef automaton generate_DFA(unsigned size, string alphabet):
    return refurbish_automaton(generate_DFA_(size, alphabet))

cpdef automaton thompson(ratexp exp):
    return refurbish_automaton(thompson_(exp._this))
  
cpdef automaton exp_to_aut(ratexp exp):
    return refurbish_automaton(exp_to_aut_(exp._this))

cpdef automaton allow_eps_transition(automaton aut):
    return refurbish_automaton(allow_eps_transition_(aut._this))

cpdef bool is_valid(automaton aut):
    return is_valid_(aut._this)

cpdef bool is_proper(automaton aut):
    return is_proper_(aut._this)

cpdef automaton proper(automaton aut, bool backward=True, bool prune=True):
    return refurbish_automaton(proper_(aut._this, backward, prune))

cpdef void proper_here(automaton aut, bool backward=True, bool prune=True):
    proper_here_(aut._this, backward, prune)

cpdef bool are_isomorphic(automaton aut1, automaton aut2):
    return are_isomorphic_(aut1._this, aut2._this)

cpdef automaton Union(automaton aut1, automaton aut2):
    return refurbish_automaton(Union_(aut1._this, aut2._this))

cpdef automaton infiltration(automaton aut1, automaton aut2):
    return refurbish_automaton(infiltration_(aut1._this, aut2._this))

cpdef automaton shuffle(automaton aut1, automaton aut2):
    return refurbish_automaton(shuffle_(aut1._this, aut2._this))

cpdef automaton power(automaton aut, unsigned n):
    return refurbish_automaton(power_(aut._this, n))

cpdef automaton product(automaton aut1, automaton aut2):
    """
    Usage:  automaton3 = product(automaton1, automaton2)
    
    Description:  Returns the automaton resulting of the Hadamard product of <automaton1> and <automaton2>. The weight of a word in <automaton3> is the product of its respective weights in <automaton1> and <automaton2>.
    
    Parameters:
      - automaton1 and automaton2: two awali.automaton whose semiring may be joined.
      
    Returns: a awali.automaton
    
    Same as: automaton1.product(automaton2)
    """
    return refurbish_automaton(product_(aut1._this, aut2._this))

cpdef string fado(automaton aut):
    """
    Usage:  str = fado(automaton)
    
    Description:  Returns the description of <automaton> using the format fado.
    
    Parameter:
      - automaton: a awali.automaton
    
    Returns: a str
    
    Same as:  str = automaton.fado()

    See also:  grail, json, dot 
    """
    return fado_(aut._this)

cpdef string grail(automaton aut):
    """
    Usage:  str = grail(automaton)
    
    Description:  Returns the description of <automaton> using the format grail.
    
    Parameter:
      - automaton: a awali.automaton
    
    Returns: a str
    
    Same as:  str = automaton.grail()

    See also:  fado, json, dot 
    """
    return grail_(aut._this)

cpdef string json(automaton aut):
    """
    Usage:  str = json(automaton)
    
    Description:  Returns the description of <automaton> using the format json, which is the default format of Awali.
    
    Parameter:
      - automaton: a awali.automaton
    
    Returns: a str
    
    Same as:  str = automaton.json()

    See also:  fado, grail, dot 
    """
    return json_(aut._this)

cpdef automaton fado_parse(string s):
    return refurbish_automaton(fado_parse_(s))

cpdef automaton grail_parse(string s):
    return refurbish_automaton(grail_parse_(s))

cpdef string dot(automaton aut, bool history=False, bool dot2tex=False):
    """
    Usage:  str = dot(automaton)
    
    Description:  Returns the description of <automaton> using the format dot.
    
    Parameter:
      - automaton: a awali.automaton
    
    Returns: a str
    
    Same as:  str = automaton.dot()

    See also:  fado, grail, json 
    """
    return dot_(aut._this, history, dot2tex)

cpdef automaton quotient(automaton aut, vector[vector[int]] equiv):
    return refurbish_automaton(quotient_(aut._this, equiv))

cpdef automaton minimal_quotient(automaton aut):
    return refurbish_automaton(minimal_quotient_(aut._this))

cpdef automaton minimal(automaton aut):
    return refurbish_automaton(minimal_(aut._this))

cpdef automaton minimize(automaton aut):
    return refurbish_automaton(minimize_(aut._this))

cpdef automaton minimize_moore(automaton aut):
    return refurbish_automaton(minimize_moore_(aut._this))

cpdef automaton minimize_brzozowski(automaton aut):
    return refurbish_automaton(minimize_brzozowski_(aut._this))

cpdef automaton make_witness(int n, string a='a'):
    return refurbish_automaton(make_witness_(n, a[0]))

cpdef automaton make_cerny(int n, string first_letter='a'):
    """ Generates a Cerny automaton
    :param n: numbers of states
    :type n: int
    :param first_letter: first letter
    :type first_letter: str
    :rtype: automaton """
    return refurbish_automaton(make_cerny_(n, first_letter[0]))

cpdef automaton make_ladybird(int n, string first_letter='a'):
    return refurbish_automaton(make_ladybird_(n, first_letter[0]))

cpdef automaton make_double_ring(int n, vector[unsigned int] finals, string first_letter="a"):
    return refurbish_automaton(make_double_ring_(n, finals, first_letter[0]))

cpdef automaton make_divkbaseb(int k, int b, string first_letter='a'):
    return refurbish_automaton(make_divkbaseb_(k, b, first_letter[0]))

cpdef automaton make_de_bruijn(int n, string a):
    return refurbish_automaton(make_de_bruijn_(n, a[0]))

cpdef automaton factor(automaton aut):
    return refurbish_automaton(factor_(aut._this))

cpdef void factor_here(automaton aut):
    factor_here_(aut._this)

cpdef automaton suffix(automaton aut):
    return refurbish_automaton(suffix_(aut._this))

cpdef void suffix_here(automaton aut):
    prefix_here_(aut._this)

cpdef automaton prefix(automaton aut):
    return refurbish_automaton(prefix_(aut._this))

cpdef void prefix_here(automaton aut):
    prefix_here_(aut._this)

cpdef dict shortest(automaton aut, int maxim):
    return get_map_s_s(shortest_(aut._this, maxim))

cpdef dict enumerate(automaton aut, int maxim):
    "enumerate all words with their weight up to length maximum : enumerate(<aut>,<lmax>)"
    return get_map_s_s(enumerate_(aut._this, maxim))

cpdef string eval(automaton aut, string word):
    return eval_(aut._this, word)

cpdef bool is_ambiguous(automaton aut):
    return is_ambiguous_(aut._this)

cpdef bool is_complete(automaton aut):
    return is_complete_(aut._this)

cpdef bool is_deterministic(automaton aut):
    return is_deterministic_(aut._this)

cpdef automaton reduce(automaton aut):
    return refurbish_automaton(reduce_(aut._this))

cpdef string rmul_weight(automaton aut, transition_t tr, str weight):
    return aut.rmul_weight(tr, weight)

cpdef void complete_here(automaton aut):
    complete_here_(aut._this)

cpdef automaton complete(automaton aut):
    return refurbish_automaton(complete_(aut._this))

cpdef void complement_here(automaton aut):
    complement_here_(aut._this)

cpdef automaton complement(automaton aut):
    return refurbish_automaton(complement_(aut._this))

cpdef bool are_equivalent(automaton aut, automaton aut2):
    return are_equivalent_(aut._this, aut2._this)

cpdef automaton determinize(automaton aut, bool history=True):
    return refurbish_automaton(determinize_(aut._this, history))

cpdef automaton make_weighted_automaton(string alphabet, string semiring):
    return refurbish_automaton(make_weighted_automaton_(alphabet, semiring))

cpdef  automaton make_NFA(string alphabet):
    return refurbish_automaton(make_NFA_(alphabet))

cpdef automaton make_weighted_automaton_with_eps(string alphabet, string semiring):
    return refurbish_automaton(make_weighted_automaton_with_eps_(alphabet, semiring))

cpdef automaton make_NFA_with_eps(string alphabet):
    return refurbish_automaton(make_NFA_with_eps_(alphabet))

cpdef automaton parse_json_automaton(string s):
    return refurbish_automaton(parse_json_automaton_(s))

cpdef automaton copy(automaton aut):
    return refurbish_automaton(copy_(aut._this))

cpdef automaton support(automaton aut):
    return refurbish_automaton(support_(aut._this))

cpdef automaton characteristic(automaton aut, string semiring):
    return refurbish_automaton(characteristic_(aut._this, semiring))

cpdef int add_state(automaton aut, string s):
    return add_state_(aut._this, s)

cpdef void del_state(automaton aut, int s):
    aut.del_state(s)

cpdef void del_explicit_transition(automaton aut, int src, int dst, str label):
    aut.del_explicit_transition(src, dst, label)

cpdef void del_transition(automaton aut, transition_t t):
    aut.del_transition(t)

cpdef transition_t set_eps_transition(automaton aut, int src, int dst, str weight=""):
    return aut.set_eps_transition(src, dst, weight)

cpdef void set_initial(automaton aut, int s, str w=None):
    aut.set_initial(s, w)

cpdef string set_weight(automaton aut, transition_t tr, str weight):
    return aut.set_weight(tr, weight)

cpdef transition_t set_transition(automaton aut, int src, int dst, str label, str weight=""):
    """
    Usage:  new_transition= set_transition(automaton, source, destination, label [, weight ] ) 
    
    Description:  Set the <weight> of the transition of <automaton> going from <source> to <destination> and labelled by <label>. If no such a transition previously existed, a new one is created, otherwise, the previous weight is overridden.
    
    Returns: an int, referring to a transition of <automaton>
    
    Same as:  new_transition= automaton.set_transition(source, destination, label, weight)
    
    See also: add_transition
    """
    return aut.set_transition(src, dst, label, weight)

cpdef string add_eps_transition(automaton aut, int src, int dst, str weight=""):
    """
    Usage: new_weight = add_eps_transition( automaton, source, destination [, extra_weight ] )
    
    Description:  Increase by <extra_weight> the current weight of the epsilon-transition of <automaton> going from <source> to <destination>.  Note that if a transition does not exists, it has a weight equal to the additive neutral element of the semiring; in this case this function is equivalent to "set_eps_transition".
    
    Parameters:
      - automaton: a awali.automaton that must allow epsilon transitions
      - source: int, must refer to a state of <automaton>
      - destination: int, must refer to a state of <automaton>
      - extra_weight: str, must belong to the weightset of <automaton>, optional, default is the additive unit element of the monoid.
    
    Returns: a str, representing the new weight of the transition.
    
    Same as:  automaton.add_eps_transition(source, destination, weight)
    
    See also:  allow_eps_transition, set_eps_transition
    """
    return aut.add_eps_transition(src, dst, weight)

cpdef void add_initial(automaton aut, int s, str w):
    aut.add_initial(s, w)

cpdef void add_final(automaton aut, int s, str w):
    aut.add_final(s, w)

cpdef string add_transition(automaton aut, int src, int dst, str label, str weight=""):
    """
    Usage:  add_transition( automaton, source, destination, label [, extra_weight ] )
    
    Description:  Increase by <extra_weight> the current weight of the transition of <automaton> going from <source> to <destination> labelled by <label>.  Note that if a transition does not exists, it has a weight equal to the additive neutral element of the semiring; in this case this function is equivalent to "set_transition".
    
    Parameters:
      - automaton: a (awali) automaton
      - source: int, must refer to a state of <automaton>
      - destination: int, must refer to a state that exists in <automaton>
      - label: str, must belong to the alphabet of <automaton>.
      - extra_weight: str, must belong to the semiring of <automaton>, optional, default is the additive unit element of the monoid.
    
    Returns: a str, representing the new weight of the transition.
    
    Same as: automaton.add_transition(source, destination, label, extra_weight)
    
    See also: set_transition.
    """
    return aut.add_transition(src, dst, label, weight)

cpdef string add_weight(automaton aut, transition_t tr, str weight=""):
    return aut.add_weight(tr, weight)

cpdef void unset_initial(automaton aut, int s):
    aut.unset_final(s)

cpdef void unset_final(automaton aut, int s):
    aut.unset_final(s)

cpdef void set_final(automaton aut, int s, str w=""):
    aut.set_final(s, w)

cpdef vector[int] accessible_states(automaton aut):
    return accessible_states_(aut._this)

cpdef vector[int] coaccessible_states(automaton aut):
    return coaccessible_states_(aut._this)

cpdef vector[transition_t] transitions(automaton aut):
    return aut.transitions()

cpdef vector[int] initial_states(automaton aut):
    return aut.initial_states()

cpdef vector[transition_t] initial_transitions(automaton aut):
    return aut.initial_transitions()

cpdef vector[int] final_states(automaton aut):
    return aut.final_states()

cpdef vector[transition_t] final_transitions(automaton aut):
    return aut.final_transitions()

cpdef vector[transition_t] outin(automaton aut, int src, int dst):
    return aut.outin(src, dst)

cpdef vector[transition_t] incoming(automaton aut, int s, str label=""):
    return aut.incoming(s, label)

cpdef vector[transition_t] outgoing(automaton aut, int s, str label=""):
    return aut.outgoing(s, label)

cpdef vector[int] successors(automaton aut, int s, str label=""):
    return aut.successors(s, label)

cpdef vector[int] predecessors(automaton aut, int s, str label=""):
    return aut.predecessors(s, label)

cpdef int pre(automaton aut):
    return aut.pre()

cpdef int post(automaton aut):
    return aut.post()

cpdef vector[int] all_states(automaton aut):
    return aut.all_states()

cpdef vector[transition_t] all_transitions(automaton aut):
    return aut.all_transitions()

cpdef vector[transition_t] all_out(automaton aut, int s):
    return aut.all_out(s)

cpdef vector[string] alphabet(automaton aut):
    return aut.alphabet()

cpdef bool has_letter(automaton aut, str l):
    return aut.has_letter(l)

cpdef bool is_transducer(automaton aut):
    return aut.is_transducer()

cpdef string get_static_context(automaton aut):
    return aut.get_static_context()

cpdef void strip_history(automaton aut):
    aut.strip_history()

cpdef string get_state_index(automaton aut, int s):
    return aut.get_state_name(s)

cpdef vector[int] useful_states(automaton aut):
    return useful_states_(aut._this)

cpdef string weight_one(automaton aut):
    return aut.weight_one()

cpdef string weight_zero(automaton aut):
    return aut.weight_zero()

cpdef string get_weightset(automaton aut):
    return aut.get_weightset()

cpdef size_t num_accessible_states(automaton aut):
    return num_accessible_states_(aut._this)

cpdef size_t num_states(automaton aut):
    return aut.num_states()

cpdef size_t num_initials(automaton aut):
    return aut.num_initials()

cpdef size_t num_finals(automaton aut):
    return aut.num_finals()

cpdef size_t num_transitions(automaton aut):
    return aut.num_transitions()

cpdef size_t num_coaccessible_states(automaton aut):
    return num_coaccessible_states_(aut._this)

cpdef bool has_state(automaton aut, int i):
    return aut.has_state(i)

cpdef bool is_initial(automaton aut, int s):
    return aut.is_initial(s)

cpdef bool is_final(automaton aut, int s):
    return aut.is_final(s)

cpdef string get_initial_weight(automaton aut, int i):
    return aut.get_initial_weight(i)

cpdef string get_final_weight(automaton aut, int i):
    return aut.get_final_weight(i)

cpdef transition_t get_transition(automaton aut, int src, int dst, str label):
    return aut.get_transition(src,dst, label)

cpdef bool has_explicit_transition(automaton aut, int src, int dst, str label):
    return aut.has_explicit_transition(src, dst, label)

cpdef bool has_transition(automaton aut, transition_t t):
    return aut.has_transition(t)

cpdef int src_of(automaton aut, transition_t t):
    return aut.src_of(t)

cpdef int dst_of(automaton aut, transition_t t):
    """
    Usage:  destination = dst_of(automaton, transition)
    
    Description:  returns the <destination> of <transition> within <automaton>
    
    Parameters:
      - automaton: a awali.automaton
      - transition: an int, must refer to an existing transition in <automaton>
    
    Returns:  an int, referring to a state of <automaton>.
    
    Same as:  destination = automaton.dst_of(transition)
    """
    return aut.dst_of(t)

cpdef string label_of(automaton aut, transition_t t):
    """
    Usage:  label = label_of(automaton, transition)
    
    Description:  returns the <label> of <transition> within <automaton>
    
    Parameters:
      - automaton: a awali.automaton
      - transition: an int, must refer to an existing transition in <automaton>
    
    Returns:  a str, representing a letter of the alphabet of <automaton>.
    
    Same as:  label = automaton.label_of(transition)
    """
    return aut.label_of(t)

cpdef automaton coaccessible(automaton aut):
    return refurbish_automaton(coaccessible_(aut._this))

cpdef automaton trim(automaton aut):
    return refurbish_automaton(trim_(aut._this))

cpdef void accessible_here(automaton aut):
    accessible_here_(aut._this)

cpdef void coaccessible_here(automaton aut):
    coaccessible_here_(aut._this)

cpdef void trim_here(automaton aut):
    trim_here_(aut._this)

cpdef bool is_trim(automaton aut):
    return is_trim_(aut._this)

cpdef bool is_useless(automaton aut):
    return is_useless_(aut._this)

cpdef bool is_accessible(automaton aut):
    return is_accessible_(aut._this)

cpdef bool is_coaccessible(automaton aut):
    return is_coaccessible_(aut._this)

cpdef bool is_empty(automaton aut):
    return is_empty_(aut._this)

cdef ratexp refurbish_ratexp(simple_ratexp_t re):
    foo = ratexp('hack')
    foo._this = re
    return foo
  
#cpdef transducer compose(transducer tdc1, transducer tdc2):
  #"""
  #Usage:  transducer3 = compose(transducer, transducer2)
  
  #Description:  Composes <transducer> with <transducer2>.
  
  #Parameters:
    #- transducer: a awali.transducer
    #- transducer2: a awali.transducer whose input alphabet is the output alphabet of <transducer2>.
    
  #Returns:  a awali.transducer
  
  #Same as:  transducer3 = transducer.compose(transducer2)
  #"""
    #return refurbish_transducer(compose_tdc1._this, tdc2._this))
  
#cdef class ratexp:
    #""" 
    #Python wrapper-class  for Awali automaton_t class
    #"""
    #cdef simple_ratexp_t _this
    
    #def __init__(self, string_to_parse, semiring='B', alphabet=None):
        #"""
        #Usage: ratexp(expression [, semiring='B' [, alphabet]])
        
        #Description: builds an empty (weighted) automaton whose transitions are labelled in <alphabet> and weighted in <semiring>
        
        #Parameters:
          #- expression: str
          #- semiring: str of {'B', 'Z', 'Z-min-plus', 'Z-max-plus', 'Q', 'R', 'R-max-prod', 'F2' }; optional: default is 'B'
          #- alphabet: str  or  list of char; optional: if not given the alphabet is the set of letters appearing in <expression>
          
        #See also: make__weighted_ratexp ,  make_weighted_ratexp_with_alphabet
        #"""
        #if type(alphabet) is list:
            #new_alph=''
            #for let in alphabet:
                #new_alph += let
            #alphabet = new_alph
        #if (string_to_parse != 'hack'):
            #if (alphabet == None):
                #self._this= make_weighted_ratexp_(string_to_parse, semiring)
            #else:
                #self._this= make_weighted_ratexp_with_alphabet_(string_to_parse, alphabet, semiring)
        ##else nothing but that's okay 
  
    #cpdef string json(self):
        #return self._this.json()

    #def  __str__(self):
        #return self._this.print_()

    #def __repr__(self):
        #return "ratexp: "+self.__str__()

    #cpdef automaton thompson(self):
        #return refurbish_automaton(thompson_(self._this))

    #cpdef ratexp expand(self):
        #return refurbish_ratexp(expand_(self._this))

    #cpdef ratexp star_normal_form(self):
        #return refurbish_ratexp(star_normal_form_(self._this))

    #cpdef unsigned star_height(self):
        #return star_height_(self._this)

    #cpdef automaton standard(self):
        #return refurbish_automaton(standard_(self._this))
      
    #cpdef automaton exp_to_aut(self):
        #return refurbish_automaton(exp_to_aut_(self._this))

cdef class automaton:
    """ 
    Python wrapper for Awali automaton_t class
    """
    cdef simple_automaton_t _this

    def __init__(self, alphabet, semiring=None):
        """
        Usage: automaton(alphabet [, semiring='B'])
        
        Description: builds an empty (weighted) automaton whose transitions are labelled in <alphabet> and weighted in <semiring>
        
        Parameters:
         - alphabet: str  or  list of char
         - semiring: str of {'B', 'Z', 'Z-min-plus', 'Z-max-plus', 'Q', 'R', 'R-max-prod', 'F2' }, default is 'B'
        """
        if type(alphabet) is list:
            new_alph=''
            for let in alphabet:
                new_alph += let
            alphabet = new_alph
        if semiring is None:
            self._this = make_simple_automaton1(alphabet)
        else:
            if semiring != "hack!!":
                self._this = make_simple_automaton2(alphabet, semiring)
            else:
                pass # returning an empty shell but it's OK!

    def __del__(self):
        self._this.release()

    def __str__(self):
        return repr(self._structure())

    def __repr__(self):
        return "automaton: "+self.__str__()

    cpdef _structure(self):
        cdef str ring
        cdef list s, t, t1, a, i, f
        t = []
        ring = self.get_static_context().split('_')[2]
        s = self.states()
        t1 = self.transitions()
        cdef int j
        if ring == 'b':
            for j in range(len(t1)):
                t.append((self.src_of(t1[j]),self.label_of(t1[j]), self.dst_of(t1[j])))
        else:
            for j in range(len(t1)):
                t.append((self.src_of(t1[j]),self.label_of(t1[j]), self.dst_of(t1[j]),self.weight_of(t1[j])))
        a = self.alphabet()
        i = self.initial_states()
        f = self.final_states()
        return (s,a,t,i,f)

    cpdef size_t num_states(self):
        return self._this.num_states()

    cpdef size_t num_initials(self):
        return self._this.num_initials()

    cpdef size_t num_finals(self):
        return self._this.num_finals()

    cpdef size_t num_transitions(self):
        return self._this.num_transitions()

    cpdef bool has_state(self, int i):
        return self._this.has_state(i)

    cpdef bool is_initial(self, int s):
        return self._this.is_initial(s)

    cpdef bool is_final(self, int s):
        return self._this.is_final(s)

    cpdef string get_initial_weight(self, int i):
        return self._this.get_initial_weight(i)

    cpdef string get_final_weight(self, int i):
        return self._this.get_final_weight(i)

    cpdef transition_t get_transition(self, int src, int dst, str label):
        return self._this.get_transition(src, dst, label)

    cpdef bool has_explicit_transition(self, int src, int dst, str label):
        return self._this.has_transition3(src, dst, label)

    cpdef bool has_transition(self, transition_t t):
        return self._this.has_transition1(t)

    cpdef int src_of(self, transition_t t):
        return self._this.src_of(t)

    cpdef int dst_of(self, transition_t t):
        return self._this.dst_of(t)

    cpdef string label_of(self, transition_t t):
        return self._this.label_of(t)

    cpdef string weight_of(self, transition_t t):
        return self._this.weight_of(t)

    cpdef int add_state(self):
        return self._this.add_state()

    cpdef del_state(self, int s):
        self._this.del_state(s)

    cpdef set_initial(self, int s, str w=None):
        if w is None:
            self._this.set_initial1(s)
        else:
            self._this.set_initial2(s, w)

    cpdef add_initial(self, int s, str w):
        return self._this.add_initial(s, w)

    cpdef unset_initial(self, int s):
        self._this.unset_initial(s)

    cpdef set_final(self, int s, str w=""):
        if w == "":
            self._this.set_final1(s)
        else:
            self._this.set_final2(s, w)

    cpdef add_final(self, int s, str w):
        return self._this.add_final(s, w)

    cpdef unset_final(self, int s):
        self._this.unset_final(s)
    
    cpdef del_explicit_transition(self, int src, int dst, str label=""):
        if label == "" :
            self._this.del_transition2(src, dst)
        else:
            self._this.del_transition3(src, dst, label)

    cpdef del_transition(self, transition_t t):
        self._this.del_transition1(t) # this is actually a transition number

    def set_transition(self, src, dst, label, weight=""):
        if (weight == ""):
            return self._this.set_transition3(src, dst, label)
        else:
            return self._this.set_transition4(src, dst, label, weight)
  
    cpdef transition_t set_eps_transition(self, int src, int dst, str weight=""):
        if weight == "":
            return self._this.set_eps_transition2(src, dst)
        else:
            return self._this.set_eps_transition3(src, dst, weight)

    cpdef string add_transition(self, int src, int dst, str label, str weight=""):
        if weight == "":
            return self._this.add_transition3(src, dst, label)
        else:
            return self._this.add_transition4(src, dst, label, weight)
  
    cpdef string add_eps_transition(self, int src, int dst, str weight=""):
        if weight == "":
            return self._this.add_eps_transition2(src, dst)
        else:
            return self._this.add_eps_transition3(src, dst, weight)

    cpdef string set_weight(self, transition_t tr, str weight):
        return self._this.set_weight(tr, weight)
  
    cpdef string add_weight(self, transition_t tr, str weight):
        return self._this.add_weight(tr, weight)

    cpdef string lmul_weight(self, transition_t tr, str weight):
        return self._this.lmul_weight(tr, weight)

    cpdef string rmul_weight(self, transition_t tr, str weight):
        return self._this.rmul_weight(tr, weight)
  
    cpdef vector[int] states(self):
        return self._this.states()
  
    cpdef vector[transition_t] transitions(self):
        return self._this.transitions()
  
    cpdef vector[int] initial_states(self):
        return self._this.initial_states()
  
    cpdef vector[transition_t] initial_transitions(self):
        return self._this.initial_transitions()
  
    cpdef vector[int] final_states(self):
        return self._this.final_states()
  
    cpdef vector[transition_t] final_transitions(self):
        return self._this.final_transitions()
  
    cpdef vector[transition_t] outin(self, int src, int dst):
        return self._this.outin(src, dst)
  
    cpdef vector[transition_t] incoming(self, int s, str label=""):
        if label == "":
            return self._this.incoming1(s)
        else:
            return self._this.incoming2(s, label)
  
    cpdef vector[transition_t] outgoing(self, int s, str label=""):
        if label == "":
            return self._this.outgoing1(s)
        else:
            return self._this.outgoing2(s, label)

    cpdef vector[int] successors(self, int s, str label=""):
        if label == "":
            return self._this.successors2(s)
        else:
            return self._this.successors1(s, label)

    cpdef vector[int] predecessors(self, int s, str label=""):
        if label == "":
            return self._this.predecessors2(s)
        else:
            return self._this.predecessors1(s, label)

    cpdef int pre(self):
        return self._this.pre()
  
    cpdef int post(self):
        return self._this.post()
  
    cpdef vector[int] all_states(self):
        return self._this.all_states()
  
    cpdef vector[transition_t] all_transitions(self):
        return self._this.all_transitions()
  
    cpdef vector[transition_t] all_out(self, int s):
        return self._this.all_out(s)
  
    cpdef vector[string] alphabet(self):
        return self._this.alphabet()
  
    cpdef bool has_letter(self, str l):
        return self._this.has_letter(l)
  
    cpdef bool is_transducer(self):
        return self._this.is_transducer()
  
    cpdef string get_static_context(self):
        return self._this.get_static_context()
  
    cpdef strip_history(self):
        return self._this.strip_history()
  
    cpdef string get_state_name(self, int s):
        return self._this.get_state_name(s)
  
    cpdef string weight_one(self):
        return self._this.weight_one()
  
    cpdef string weight_zero(self):
        return self._this.weight_zero()
  
    cpdef string get_weightset(self):
        return self._this.get_weightset()

    cpdef string json(self):
        return self._this.json()

    cpdef vector[int] accessible_states(self):
        return accessible_states_(self._this)

    cpdef vector[int] coaccessible_states(self):
        return coaccessible_states_(self._this)

    cpdef vector[int] useful_states(self):
        return useful_states_(self._this)

    cpdef bool is_empty(self):
        return is_empty_(self._this)

    cpdef size_t num_accessible_states_(self):
        return num_accessible_states_(self._this)

    cpdef size_t num_coaccessible_states(self):
        return num_coaccessible_states_(self._this)

    cpdef automaton accessible(self):
        return refurbish_automaton(accessible_(self._this))

    cpdef automaton coaccessible(self):
        return refurbish_automaton(coaccessible_(self._this))

    cpdef automaton trim(self):
        return refurbish_automaton(trim_(self._this))

    cpdef accessible_here(self):
        accessible_here_(self._this)

    cpdef coaccessible_here(self):
        coaccessible_here_(self._this)

    cpdef trim_here(self):
        trim_here_(self._this)

    cpdef bool is_trim(self):
        return is_trim_(self._this)

    cpdef bool is_useless(self):
        return is_useless_(self._this)

    cpdef bool is_accessible(self):
        return is_accessible_(self._this)

    cpdef bool is_coaccessible(self):
        return is_coaccessible_(self._this)

    cpdef bool is_ambiguous(self):
        return is_ambiguous_(self._this)

    cpdef bool is_complete(self):
        return is_complete_(self._this)

    cpdef bool is_deterministic(self):
        return is_deterministic_(self._this)

    cpdef automaton reduce(self):
        return refurbish_automaton(reduce_(self._this))

    cpdef complete_here(self):
        complete_here_(self._this)

    cpdef automaton complete(self):
        return refurbish_automaton(complete_(self._this))

    cpdef complement_here(self):
        complement_here_(self._this)

    cpdef automaton complement(self):
        return refurbish_automaton(complement_(self._this))

    cpdef bool are_equivalent(self, automaton aut2):
        return are_equivalent_(self._this, aut2._this)

    cpdef automaton determinize(self, bool history=True):
        return refurbish_automaton(determinize_(self._this, history))

    cpdef string eval(self, string word):
        return eval_(self._this, word)

    cpdef dict shortest(self, maxim):
        return get_map_s_s(shortest_(self._this, maxim))

    cpdef dict enumerate(self, int maxim):
        return get_map_s_s(enumerate_(self._this, maxim))

    cpdef prefix_here(self):
        prefix_here_(self._this)

    cpdef automaton prefix(self):
        return refurbish_automaton(prefix_(self._this))

    cpdef suffix_here(self):
        prefix_here_(self._this)

    cpdef automaton suffix(self):
        return refurbish_automaton(suffix_(self._this))

    cpdef factor_here(self):
        factor_here_(self._this)

    cpdef automaton factor(self):
        return refurbish_automaton(factor_(self._this))

    cpdef automaton minimize_brzozowski(self):
        return refurbish_automaton(minimize_brzozowski_(self._this))

    cpdef automaton minimize_moore(self):
        return refurbish_automaton(minimize_moore_(self._this))

    cpdef automaton minimize(self):
        return refurbish_automaton(minimize_(self._this))

    cpdef automaton minimal(self):
        return refurbish_automaton(minimal_(self._this))

    cpdef string dot(self, bool history=False, bool dot2tex=False):
        return dot_(self._this, history, dot2tex)

    cpdef string fado(self):
        return fado_(self._this)

    cpdef string grail(self):
        return grail_(self._this)

    cpdef bool are_isomorphic(self, automaton aut2):
        return are_isomorphic_(self._this, aut2._this)

    cpdef automaton Union(self, automaton aut2):
        return refurbish_automaton(Union_(self._this, aut2._this))

    cpdef automaton infiltration(self, automaton aut2):
        return refurbish_automaton(infiltration_(self._this, aut2._this))

    cpdef automaton shuffle(self, automaton aut2):
        return refurbish_automaton(shuffle_(self._this, aut2._this))

    cpdef automaton power(self, unsigned n):
        return refurbish_automaton(power_(self._this, n))

    cpdef automaton product(self, automaton aut2):
        return refurbish_automaton(product_(self._this, aut2._this))

    cpdef automaton quotient(self, vector[vector[int]] equiv):
        return refurbish_automaton(quotient_(self._this, equiv))

    cpdef automaton minimal_quotient(self):
        return refurbish_automaton(minimal_quotient_(self._this))

    cpdef automaton allow_eps_transition(self):
        return refurbish_automaton(allow_eps_transition_(self._this))

    cpdef bool is_valid(self):
        return is_valid_(self._this)

    cpdef bool is_proper(self):
        return is_proper_(self._this)

    cpdef automaton proper(self, bool backward=True, bool prune=True):
        return refurbish_automaton(proper_(self._this, backward, prune))

    cpdef proper_here(self, bool backward=True, bool prune=True):
        proper_here_(self._this, backward, prune)

    cpdef ratexp aut_to_exp(self):
        return refurbish_ratexp(aut_to_exp_(self._this))

    cpdef automaton lift(self):
        return refurbish_automaton(lift_(self._this))

    cpdef bool is_standard(self):
        return is_standard_(self._this)

    cpdef automaton standard(self):
        return refurbish_automaton(standard_(self._this))

    cpdef standard_here(self):
        standard_here_(self._this)

    cpdef automaton concatenate(self, automaton aut2):
        return refurbish_automaton(concatenate_(self._this, aut2._this))

    cpdef concatenate_here(self, automaton aut2):
        concatenate_here_(self._this, aut2._this)

    cpdef automaton sum(self, automaton aut2):
        return refurbish_automaton(sum_(self._this, aut2._this))

    cpdef sum_here(self, automaton aut2):
        sum_here_(self._this, aut2._this)

    cpdef automaton star(self):
        return refurbish_automaton(star_(self._this))

    cpdef star_here(self):
        star_here_(self._this)

    cpdef automaton synchronize(self):
        return refurbish_automaton(synchronize_(self._this))

    cpdef bool is_synchronizable(self):
        return is_synchronizable_(self._this)

    cpdef automaton copy(self):
        """
        Usage:  automaton2 = automaton1.copy()
        
        Description: builds a copy of <automaton1>.
        
        Returns:
          - automaton2: a (awali) automaton
        """
        return refurbish_automaton(copy_(self._this))

    cpdef display(self, fileName=None):
        """
        Usage: automaton.display([filename])
        
        Description: displays <automaton> (require the program graphviz). If the optional argument 
        is present, the automaton representation is in addition written to file <filename>.gv. 
        
        Parameters:
         - filename: str 
        
        Same as: display(automaton, filename)
        """
        if fileName is not None:
            fnameGV = fileName + ".gv"
            if run_from_ipython_notebook():
                filenameOut = fileName + ".svg"
            else:
                filenameOut = fileName + ".pdf"
        else:
            f = tempfile.NamedTemporaryFile(suffix=".gv")
            f.close()
            fnameGV = f.name
            fname, _ = os.path.splitext(fnameGV)
            if run_from_ipython_notebook():
                filenameOut = fname + ".svg"
            else:
                filenameOut = fname + ".pdf"
        foo = open(fnameGV, "w")
        foo.write(self.dot())
        foo.close()
        if run_from_ipython_notebook():
            from IPython.display import Image, display, SVG
            callstr = "dot -Tsvg %s -o %s" % (fnameGV, filenameOut)
        else:
            callstr = "dot -Tpdf %s -o %s" % (fnameGV, filenameOut)
        result = subprocess.call(callstr, shell=True)
        if result:
            print "Need graphviz to visualize objects"
            return
        if run_from_ipython_notebook():
            display(SVG(filename=filenameOut))
        elif os.name == 'nt':
            os.system("start %s" % filenameOut)
        else:
            os.system("open %s" % filenameOut)


    cpdef save(self, string s, string fmt = 'json'):
        """
        Usage: automaton.save( filename [, format='json' ] )
                                        
        Description: writes <automaton> in file <filename> using <format>.
        
        Parameters:
        - filename: a str
        - format: a str of { 'dot', 'fado', 'grail', 'json' } , optional, default is 'json'
        """
        save_automaton_(self._this, s, fmt)
        
#cdef class transducer(automaton): 
  
  #def __init__(self, alphabet_list, semiring=None):
      #if semiring != "hack!!":
          #self._this= make_simple_transducer1(alphabet_list)
      #else:
        #pass

  #def __repr__(self):
      #return "transducer: "+self.__str__()
    
  #cpdef vector[vector[string]] alphabets(self):
        #return self._this.alphabets()
      
  #cpdef _structure(self):
    ##cdef str ring
    #cdef list s, t, t1, a, i, f
    ##t = []
    ##ring = self.get_static_context().split('_')[-1]
    ##s = self.states()
    ##t1 = self.transitions()
    ##cdef int j
    ##if ring == 'b':
        ##for j in range(len(t1)):
            ##t.append((self.src_of(t1[j]),self.label_of(t1[j]), self.dst_of(t1[j])))
    ##else:
        ##for j in range(len(t1)):
            ##t.append((self.src_of(t1[j]),self.label_of(t1[j]), self.dst_of(t1[j]),self.weight_of(t1[j])))
    #a = self.alphabets()
    ##i = self.initial_states()
    ##f = self.final_states()
    #return a #(s,a,t,i,f)
  
  #def set_transition(self, src, dst,  labels, weight=None):
    #self._this.set_tdc_transition3(src, dst, labels)
    
  #cpdef transducer compose(self, transducer other):
  #"""
  #Usage:  transducer3 = transducer.compose(transducer2)
  
  #Description:  Composes <transducer> with <transducer2>
  
  #Parameters:
    #- transducer: a awali.transducer
    #- transducer2: a awali.transducer whose input alphabet is the output alphabet of <transducer2>.
    
  #Returns:  a awali.transducer
  #"""
    #return refurbish_transducer(compose_(self._this, other._this))
    
