// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_WEIGHTSET_B_HH
# define AWALI_WEIGHTSET_B_HH

# include <cassert>
# include <ostream>
# include <string>

#include <awali/sttc/misc/escape.hh>
#include <awali/utils/hash.hh>
#include <awali/sttc/misc/parse_exception.hh>
#include <awali/sttc/misc/raise.hh>
#include <awali/sttc/misc/stream.hh>
#include <awali/common/star_status.hh>
#include <awali/sttc/weightset/fwd.hh>
#include <awali/sttc/weightset/weightset.hh>

namespace awali {
  namespace sttc {

    namespace internal {
      class b_impl {
      public:
	using self_type = b;

	static std::string sname() {
	  return "b";
	}

	static std::string vname(bool = true) {
	  return sname();
	}

	/// Build from the description in \a is.
	static b make(std::istream& is) {
	  eat(is, sname());
	  return {};
	}

	using value_t = bool;

	static value_t zero() {
	  return false;
	}

	static value_t one() {
	  return true;
	}

	static value_t add(value_t l, value_t r) {
	  return l || r;
	}

	static value_t mul(value_t l, value_t r) {
	  return l && r;
	}

	static value_t rdiv(value_t l, value_t r) {
	  require(!is_zero(r), "div: division by zero");
	  return l;
	}

	static value_t ldiv(value_t l, value_t r) {
	  return rdiv(r, l);
	}

	static value_t star(value_t) {
	  return one();
	}

	static bool equals(value_t l, value_t r)
	{
	  return l == r;
	}

	/// Whether \a lhs < \a rhs.
	static bool less_than(value_t lhs, value_t rhs)
	{
	  return lhs < rhs;
	}

	constexpr static bool is_special(value_t) {
	  return false;
	}

	static bool is_zero(value_t v) {
	  return !v;
	}

	static bool is_one(value_t v) {
	  return v;
	}

	constexpr static bool is_commutative_semiring() { return true; }

	constexpr static bool show_one() { return false; }

	constexpr static star_status_t star_status() {
	  return star_status_t::STARRABLE;
	}

	static value_t transpose(value_t v) {
	  return v;
	}

	static size_t hash(value_t v) {
	  return utils::hash_value(v);
	}

	static value_t conv(self_type, value_t v) {
	  return v;
	}

	static value_t conv(std::istream& is) {
	  int i;
	  if (is >> i)
	    {
	      if (i == 0 || i == 1)
		return i;
	      else
		fail_reading(is,
			     sname() + ": invalid value: " + std::to_string(i));
	    }
	  else
	    fail_reading(is, sname() + ": invalid value");
	}

	static value_t parse(const std::string & s, size_t& p) {
	  if(s[p--] == '0')
	    return false;
	  if(s[p+1] == '1')
	    return true;
	  throw parse_exception("Wrong Boolean value");
	}

	static std::ostream& print(value_t v, std::ostream& o,
			    const std::string& format = "text") {
	  if (format == "latex")
	    o << (v ? "\\top" : "\\bot");
	  else
	    o << (v ? '1' : '0');
	  return o;
	}

	static std::ostream& print_set(std::ostream& o, const std::string& format = "text") {
	  if (format == "latex")
	    o << "\\mathbb{B}";
	  else if (format == "text")
	    o << "B";
	  else
	    raise("invalid format: ", format);
	  return o;
	}

	static std::ostream& js_print(std::ostream& o) {
	  o << "{\"Semiring\":\"B\"}";
	  return o;
	}

	static value_t js_parse(std::istream& i) {
	  char c;
	  i >> c;
	  if(c != '"')
	    throw std::runtime_error("json parser B");
	  value_t r;
	  i >> r;
	  i >> c;
	  if(c != '"')
	    throw std::runtime_error("json parser B");
	  return r;
	}

      };
    } // internal::

    AWALI_WEIGHTS_BINARY(b, b, b);
  }
}//end of ns awali::stc

#endif // !AWALI_WEIGHTSET_B_HH
