// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALPHABETS_BASIC_HH
# define AWALI_ALPHABETS_BASIC_HH

# include <cassert>
# include <string>
# include <iostream>
# include <limits>
# include <sstream>

#include <awali/sttc/misc/escape.hh>

namespace awali {
  namespace sttc {

    template<typename T>
    class basic_letters {
    public:
      using letter_t = T;
      using word_t = std::basic_string<T>;
      static const char delimiter=':';

      static std::string sname()
      {
	return "basic_letter";
      }

      virtual std::string vname(bool = true) const
      {
	return sname();
      }

      word_t
      to_word(const letter_t l) const
      {
	return {l};
      }

      const word_t&
      to_word(const word_t& l) const
      {
	return l;
      }

      word_t
      concat(const letter_t l, const letter_t r) const
      {
	return {l, r};
      }

      word_t
      concat(const word_t& l, const letter_t r) const
      {
	return l + r;
      }

      word_t
      concat(const letter_t l, const word_t& r) const
      {
	return l + r;
      }

      word_t
      concat(const word_t& l, const word_t& r) const
      {
	return l + r;
      }

      /// Add the special character first and last.
      word_t delimit(const word_t& w) const
      {
	return concat(concat(special_letter(), w), special_letter());
      }

      /// Remove first and last characters, that must be "special".
      word_t undelimit(const word_t& w) const
      {
	size_t s = w.size();
	assert(2 <= s);
	assert(w[0] == special_letter());
	assert(w[s-1] == special_letter());
	return w.substr(1, s-2);
      }

      static word_t
      empty_word()
      {
	return {};
      }

      static bool
      is_empty_word(const word_t& w)
      {
	return w.empty();
      }

      word_t
      transpose(const word_t& w) const
      {
	// C++11 lacks std::rbegin/rend...
	return {w.rbegin(), w.rend()};
      }

      letter_t
      transpose(letter_t l) const
      {
	return l;
      }

      bool
      equals(const letter_t& l1, const letter_t& l2) const
      {
	return l1 == l2;
      }

      bool
      equals(const word_t& w1, const word_t& w2) const
      {
	return w1 == w2;
      }

      bool
      is_letter(const letter_t&) const
      {
	return true;
      }

      bool
      is_letter(const word_t& w) const
      {
	return w.size() == 1;
      }

      /// The reserved letter used to forge the "one" label (the unit,
      /// the identity).
      static constexpr letter_t one_letter() { return std::numeric_limits<letter_t>::min(); }

    private:
      /// The reserved letter used to forge the labels for initial and
      /// final transitions.
      ///
      /// Use the public special() interface.
      static constexpr letter_t special_letter() { return std::numeric_limits<letter_t>::max(); }

    public:
      std::ostream&
      print(const letter_t& l, std::ostream& o) const
      {
	if (l != one_letter() && l != special_letter())
	  o << l;
	return o;
      }

      std::ostream&
      print(const word_t& w, std::ostream& o) const
      {
	return o << format(w);
      }

      letter_t
      conv_one_letter(std::istream& i) const
      {
	int c = i.peek();
	int pos=1;
	if (c=='-') {
	  i.get();
	  pos=-1;
	}
	int r=0;
	while((c=i.peek())>='0' && c<='9') {
	  r=(10*r)+(c-'0');
	  i.get();
	}
	return r;
      }

      letter_t
      parse_one_letter(const std::string& s, size_t& p) const {
	--p;
	int a=p, b=p;
	while(s[a]>='0' && s[a]<='9')
	  --a;
	if(a==b)
	  throw parse_exception("Invalid letter");
	if(s[a+1]=='0') {
	  if(a+1==b)
	    return 0;
	  throw parse_exception("Invalid letter");
	}
	//int v=1;
	if(s[a]=='-') {
	  int r=0;
	  for(int i=a+1; i<=b; ++i)
	    r=r*10+(s[i]-'0');
	  p=a;
	  return -r;
	}
	int r=0;
	for(int i=a+1; i<=b; ++i)
	  r=r*10+(s[i]-'0');
	p=a+1;
	return r;
      }

      std::string
      format(const letter_t l) const
      {
	std::ostringstream o;
	if (l == one_letter() || l == special_letter())
	  return {};
	else {
	  o << l;
	  return o.str();
	}
      }

      std::string
      format(const word_t& w) const
      {
	size_t s = w.size();

	std::string res;
	if (s == 0
	    || (s == 1 && w[0] == one_letter()))
	  res = "\\e";

	// If the string starts or ends with the special letter, skip
	// it.  If the resulting string is empty, format it this way.
	// (We DON'T want to format it as "\\e".)
	std::string delim{""};
	for(auto& l : w) {
	  if (l == special_letter())
	    continue;
	  res+=delim;
	  delim=std::string{delimiter};
	  res+=format(l);
	}
	return res;
      }

      /// Special character, used to label transitions from pre() and to
      /// post().
      template <typename U = letter_t>
      static U special();
    };


  }
}//end of ns awali::stc

#endif // !AWALI_ALPHABETS_BASIC_HH
