// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_STAR_HH
# define AWALI_ALGOS_STAR_HH

#include <awali/sttc/algos/copy.hh>
#include <awali/sttc/algos/standard.hh> // is_standard
#include <awali/sttc/ctx/traits.hh>
#include <awali/sttc/misc/raise.hh> // require

namespace awali { namespace sttc {

  /*------.
  | star  |
  `------*/

  /// In place star of a standard automaton.
  ///
  /// See standard_visitor::visit(star).
  template <typename Aut>
  Aut&
  star_here(Aut& res)
  {
    require(is_standard(res), __func__, ": input must be standard");

    using automaton_t = Aut;
    using context_t = context_t_of<automaton_t>;
    using weightset_t = weightset_t_of<context_t>;
    using weight_t = weight_t_of<context_t>;

    weightset_t ws(*res->context().weightset());

    state_t initial = res->dst_of(*(res->initial_transitions().begin()));
    // The "final weight of the initial state", starred.
    weight_t w = ws.star(res->get_final_weight(initial));
    // Branch all the final states (but initial) to the successors
    // of initial.
    for (auto ti: res->out(initial))
      {
        res->lmul_weight(ti, w);
        for (auto tf: res->final_transitions())
          if (res->src_of(tf) != initial)
            // The weight of ti has already been multiplied, on the
            // left, by w.
            res->add_transition
              (res->src_of(tf),
               res->dst_of(ti),
               res->label_of(ti),
               ws.mul(res->weight_of(tf), res->weight_of(ti)));
      }
    for (auto tf: res->final_transitions())
      res->rmul_weight(tf, w);
    res->set_final(initial, w);
    return res;
  }


  /// Star of a standard automaton.
  template <typename Aut>
  typename Aut::element_type::automaton_nocv_t
  star(const Aut& aut)
  {
    auto res = copy(aut);
    star_here(res);
    return res;
  }
}}//end of ns awali::stc

#endif // !AWALI_ALGOS_STAR_HH
