// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_JS_PRINT_EXP_HH
# define AWALI_ALGOS_JS_PRINT_EXP_HH

#include <awali/sttc/core/rat/visitor.hh>
#include <awali/sttc/history/history.hh>
#include <awali/sttc/ctx/fwd.hh>
#include <awali/sttc/misc/raise.hh>
#include <awali/sttc/misc/json.hh>
#include <awali/common/tuple.hh>
# include <stack>
# include <iostream>
# include <sstream>

namespace awali { namespace sttc {


  /*---------------------.
  | Json export(ratexp).  |
  `---------------------*/

  namespace rat
  {
    template <typename RatExpSet>
    class js_print_visitor
      : public RatExpSet::const_visitor
    {
    public:
      using ratexpset_t = RatExpSet;
      using context_t = context_t_of<ratexpset_t>;
      using labelset_t = labelset_t_of<context_t>;
      using ratexp_t = typename ratexpset_t::value_t;
      using weightset_t = weightset_t_of<ratexpset_t>;
      using super_type = typename ratexpset_t::const_visitor;
      using node_t = typename super_type::node_t;

     constexpr static const char* me() { return "js-print-exp"; }

      js_print_visitor(const ratexpset_t& rs, std::ostream& o)
        : rs_(rs), stream_(o)
      {}

      std::ostream& operator()(const ratexp_t& v)
      {
        v->accept(*this);
	return stream_;
      }

      AWALI_RAT_UNSUPPORTED(ldiv)
      AWALI_RAT_UNSUPPORTED(transposition)

      AWALI_RAT_VISIT(zero,)
      {
        stream_ << "\"Zero\":null" ;
      }

      AWALI_RAT_VISIT(one,)
      {
       stream_ << "\"One\":null";
      }

      AWALI_RAT_VISIT(atom, e)
      {
        stream_ << "\"Label\":" ;
	ls_.js_print(stream_, e.value());
      }

      AWALI_RAT_VISIT(sum, e)
      {
        stream_ << "\"Sum\":[";
        for (unsigned i = 0, n = e.size(); i < n; ++i) {
	  const auto& v = e[i];
	  stream_ << '{';
	  v->accept(*this);
	  stream_ << ((i<n-1)?"},":"}");
	}
        stream_ << ']';
      }

      AWALI_RAT_VISIT(prod, e)
      {
        stream_ << "\"Prod\":[";
        for (unsigned i = 0, n = e.size(); i < n; ++i) {
	  const auto& v = e[i];
	  stream_ << '{';
	  v->accept(*this);
	  stream_ << ((i<n-1)?"},":"}");
	}
        stream_ << ']';
      }

      AWALI_RAT_VISIT(conjunction, e)
      {
        stream_ << "\"Conjunction\":[";
        for (unsigned i = 0, n = e.size(); i < n; ++i) {
	  const auto& v = e[i];
	  stream_ << '{';
	  v->accept(*this);
	  stream_ << ((i<n-1)?"},":"}");
	}
        stream_ << ']';
     }

      AWALI_RAT_VISIT(shuffle, e)
      {
        stream_ << "\"Shuffle\":[";
        for (unsigned i = 0, n = e.size(); i < n; ++i) {
	  const auto& v = e[i];
	  stream_ << '{';
	  v->accept(*this);
	  stream_ << ((i<n-1)?"},":"}");
	}
        stream_ << ']';
      }

      AWALI_RAT_VISIT(complement, e)
      {
        stream_ << "\"Node\":\"Complement\" \"value\":";
        stream_ << '{';
       e.sub()->accept(*this);
	stream_ << '}';
      }

      AWALI_RAT_VISIT(star, e)
      {
        stream_ << "\"Star\":";
        stream_ << '{';
       e.sub()->accept(*this);
	stream_ << '}';
      }

      AWALI_RAT_VISIT(lweight, e)
      {
        stream_ << "\"LWeight\":\"";
	ws_.print(e.weight(), stream_);
	stream_ << "\" ";
        e.sub()->accept(*this);
      }

      AWALI_RAT_VISIT(rweight, e)
      {
        stream_ << "\"RWeight\":\"";
	ws_.print(e.weight(), stream_);
	stream_ << "\" ";
        e.sub()->accept(*this);
     }

    private:
      ratexpset_t rs_;
      std::ostream& stream_;
      /// Shorthand to the weightset.
      weightset_t ws_ = *rs_.weightset();
      labelset_t ls_ = *rs_.labelset();
    };

  } // rat::

  template <typename RatExpSet>
  inline
  std::ostream&
  js_print(const RatExpSet& rs,
             const typename RatExpSet::ratexp_t& e,
	   std::ostream& org)
  {
  	std::ostringstream o;
    o << "{\"Rational Expression\":[";
    rs.context().js_print(o) <<",{";
    rat::js_print_visitor<RatExpSet> printer{rs,o};
    printer(e);
    o << "}]}";
    std::istringstream i(o.str());
    js_formate(org, i);
    return org;
  }

  template <typename Automaton>
  inline
  std::ostream&
  js_print(Automaton aut,
	   std::ostream& org, bool full=false)
  {
    std::ostringstream o;
    auto ws=aut->context().weightset();
    auto ls=aut->context().labelset();
    o << "{\"Automaton\":[";
    aut->context().js_print(o) <<',';
    o << "{\"Content\":[";
    o << "{\"States\":[";
    bool first=true;
    for(unsigned i: aut->states()) {
      if(first)
	first=false;
      else
	o << ",";
      o << "{\"Id\":" << (full?i:i-2) << " \"Name\":\"";
      aut->print_state(i, o);
      if(aut->has_history(i)){
      	o<< "\" \"History\":\"";
      	aut->print_state_history(i, o);
      }
      o << "\"}";
    }
    o << "]},";
    if(aut->num_initials()>0) {
    o << "{\"Initial States\":[";
    first=true;
    for(unsigned t: aut->initial_transitions()) {
      if(first)
	first=false;
      else
	o << ",";
      if(ws->is_one(aut->weight_of(t)))
	o << (full?aut->dst_of(t):aut->dst_of(t)-2);
      else {
	o << "{\"Id\":" << (full?aut->dst_of(t):aut->dst_of(t)-2) << " \"Weight\":";
	ws-> print(aut->weight_of(t),o,"json") << "}";
      }
    }
    o << "]},";
    }
    if(aut->num_finals()>0) {
    o << "{\"Final States\":[";
    first=true;
     for(unsigned t: aut->final_transitions()) {
      if(first)
	first=false;
      else
	o << ",";
      if(!ls->is_special(aut->label_of(t))) {
	o << "{\"Id\":" << (full?aut->src_of(t):aut->src_of(t)-2) << " \"Label\":";
	ls-> js_print(o,aut->label_of(t));
	if(!ws->is_one(aut->weight_of(t))) {
	  o << " \"Weight\":";
	  ws-> print(aut->weight_of(t),o,"json");
	}
	o << "}";
      }
      else if(ws->is_one(aut->weight_of(t)))
	o <<  (full?aut->src_of(t):aut->src_of(t)-2);
      else {
	o << "{\"Id\":" << (full?aut->src_of(t):aut->src_of(t)-2) << " \"Weight\":";
	ws-> print(aut->weight_of(t),o,"json") << "}";
      }
     }
    o  << "]},";
    }
    o << "{\"Transitions\":[";
    first=true;
    for(unsigned i: aut->transitions()) {
      if(first)
	first=false;
      else
	o << ",";
      o << '{';
      if(full)
	o <<"\"Id\":" << i << ' ';
      o	<< "\"Src\":" << (full?aut->src_of(i):aut->src_of(i)-2)
	<< " \"Dst\":" << (full?aut->dst_of(i):aut->dst_of(i)-2)
	<< " \"Label\":";
      ls->js_print(o, aut->label_of(i));
      if(!ws->is_one(aut->weight_of(i))) {
	o << " \"Weight\":";
	ws-> print(aut->weight_of(i),o,"json") ;
      }
      o  << "}";
    }
    o << "]}";
    o << "]}";
    o << "]}";
    std::istringstream i(o.str());
    js_formate(org, i);
    return org;
  }

}}//end of ns awali::stc

#endif // !AWALI_ALGOS_JS_PRINT_EXP_HH
