// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_DE_BRUIJN_HH
# define AWALI_ALGOS_DE_BRUIJN_HH

# include <iterator> // std::distance
# include <stdexcept>

#include <awali/sttc/alphabets/char.hh>
#include <awali/sttc/alphabets/setalpha.hh>
#include <awali/sttc/core/mutable-automaton.hh>
#include <awali/sttc/misc/raise.hh>

namespace awali { namespace sttc {

  // (a+b)*a(a+b)^n.
  template <typename Context>
  mutable_automaton<Context>
  de_bruijn(const Context& ctx, unsigned n)
  {
    const auto& gens = ctx.labelset()->genset();
    size_t sz = std::distance(std::begin(gens), std::end(gens));
    require(2 <= sz, "de_bruijn: the alphabet needs at least 2 letters");
    require(n > 0, "de_bruijn: automaton defined for n>0");
    using context_t = Context;
    using automaton_t = mutable_automaton<context_t>;
    automaton_t res = make_shared_ptr<automaton_t>(ctx);

    auto init = res->add_state();
    res->set_initial(init);
    for (auto l: gens)
      res->new_transition(init, init, l);

    auto prev = res->add_state();
    res->new_transition(init, prev, *std::begin(gens));

    while (--n)
      {
        auto next = res->add_state();
        for (auto l: gens)
          res->new_transition(prev, next, l);
        prev = next;
      }
    res->set_final(prev);
    return res;
  }

}}//end of ns awali::stc

#endif // !AWALI_ALGOS_DE_BRUIJN_HH
