// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_EXPLICIT_CONTEXT_CC
#define DYN_EXPLICIT_CONTEXT_CC

#include <memory>

#include <awali/sttc/ctx/context.hh>
#include <awali/sttc/labelset/tupleset.hh>

#include <awali/dyn/core/abstract-context.hh>

namespace awali { namespace dyn {

  template <typename Context>
  struct explicit_context_t : public dyn::abstract_context_t {
    std::shared_ptr<Context> context_ptr;

    explicit_context_t(const Context& c) :
    context_ptr(std::make_shared<Context>(c)) {}

    std::string vname(bool full) const
    {
      return context_ptr->vname(full);
    }

    std::string sname() const
    {
      return context_ptr->sname();
    }



    std::string labelname() const {
      return context_ptr->labelset()->sname();
    }

    std::string weightname(std::string format) const {
      std::ostringstream os;
      context_ptr->weightset()->print_set(os, format);
      return os.str();
    }

    const Context& get_context() {
      return *context_ptr;
    }

    std::vector<dyn::any_t> alphabet() const {
      std::vector<dyn::any_t> res;
      for(auto l : context_ptr->labelset()->genset())
        res.emplace_back(l);
      return res;
    }

    any_t epsilon() const {
      return epsilon_();
    }



  private:
    template <typename C = Context>
    typename std::enable_if<C::labelset_t::has_one(),any_t>::type 
    epsilon_() const {
      return context_ptr->labelset()->one();
    }
    template <typename C = Context>
    typename std::enable_if<!C::labelset_t::has_one(),any_t>::type 
    epsilon_() const {
      throw std::runtime_error("Labelset does not allows epsilon");
    }

    template<typename Labelset>
    void add_letter_(dyn::any_t l, const Labelset& lab) {
      const_cast<typename Labelset::genset_t&>(lab.genset()).add_letter(l);
    }

    template<typename Labelset>
    bool has_letter_(dyn::any_t l, const Labelset& lab) const {
      return lab.genset().has(l);
    }

    template<typename... T>
    void add_letter_(dyn::any_t l, const sttc::tupleset<T...>& lab) {
      throw std::runtime_error("No genset in tupleset");
    }

    template<typename... T>
    bool has_letter_(dyn::any_t l, const sttc::tupleset<T...>& lab) const {
      throw std::runtime_error("No genset in tupleset");
    }

    void add_letter_(dyn::any_t l, const sttc::oneset& lab) {
      throw std::runtime_error("No genset in oneset");
    }

    bool has_letter_(dyn::any_t l, const sttc::oneset& lab) const {
      return false;
    }

  public:
    void add_letter(dyn::any_t l) {
      add_letter_(l, *(context_ptr->labelset()));
    }

    bool has_letter(dyn::any_t l) const {
      return has_letter_(l, *(context_ptr->labelset()));
    }

    dyn::any_t weight_one() const {
      return context_ptr->weightset()->one();
    }

    dyn::any_t weight_zero() const {
      return context_ptr->weightset()->zero();
    }

    std::string get_weightset() const {
      return context_ptr->weightset()->vname();
    }

    dyn::any_t get_weight(std::string w) const {
      std::istringstream st(w);
      return context_ptr->weightset()->conv(st);
    }

    dyn::any_t get_label(std::string l) const {
      std::istringstream st(l);
      return context_ptr->labelset()->conv(st);
    }

    std::string weight_to_string(any_t w) const {
      std::ostringstream st;
      context_ptr->weightset()->print(w, st);
      return st.str();
    }

    std::string label_to_string(any_t w) const {
      std::ostringstream st;
      context_ptr->labelset()->print(w, st);
      return st.str();
    }

    dyn::any_t add_weights(dyn::any_t l, dyn::any_t r) const {
      return context_ptr->weightset()->add(l,r);
    }

    dyn::any_t mul_weights(dyn::any_t l, dyn::any_t r) const {
      return context_ptr->weightset()->mul(l,r);
    }

    dyn::any_t div_weights(dyn::any_t l, dyn::any_t r) const {
      return context_ptr->weightset()->rdiv(l,r);
    }


  };

  template<typename Context>
  const Context&
  get_stc_context(dyn::context_t ctx) {
    return dynamic_cast<explicit_context_t<Context>&>(*ctx).get_context();
  }
}}//end of ns awali::dyn

#endif
