// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include <stdio.h>
#include <fstream>

#include <awali/dyn/loading/locations.hh>

#include <awali/dyn/algos/sys.hh>
#include <awali/dyn/modules/output.hh>

namespace awali { namespace dyn {

    void display(automaton_t aut, const std::string& name, bool history, bool horizontal) {
    std::string filename= "/tmp/"+name+".gv";
    std::ofstream o(filename);
    dot(aut, o, history, horizontal) << std::endl;
    o.close();
    system(("dotty "+filename).c_str());
  }

  void 
  pdfdisplay
  ( automaton_t aut, const std::string& name, bool history, bool horizontal)
  {
    std::string filename= "/tmp/"+name+".pdf";
    std::ofstream o(filename);
    pdf(aut,o,history,horizontal);
    std::string command;
// __APPLE__ ,  __linux__ , __unix__ , __WIN32
#ifdef __APPLE__
    command = "open ";
#else
    command = "xdg-open ";
#endif
    system((command+filename).c_str());
  }


  automaton_t parse_automaton(std::istream& i, const std::string& fmt) {
    if (fmt=="fado")
      return fado(i);
    if (!fmt.compare("grail"))
      return grail(i);
    if (!fmt.compare("json") || !fmt.compare("default"))
      return parse_automaton(i);

    throw std::runtime_error("Unrecognised input format: "+fmt);
  }

  automaton_t load(const std::string& s, const std::string& fmt) {
    std::vector<loading::file_loc_t> auts = loading::example_automata();
    bool found = false;
    std::string filename (s);

    for (const auto& aut: auts) {
      if (!aut.name.compare(s)) {	
	filename = aut.dir+"/"+aut.name+"."+aut.ext;
	found = true;
	break;
      }
    }
    
    std::ifstream fic(filename);
    if (fic.fail ())
      throw std::runtime_error ("can not load file " + filename);
    if (found)
      return parse_automaton(fic);
    return parse_automaton(fic, fmt);
  }

  
  std::ostream& put(automaton_t aut, std::ostream& o, const std::string& format)
  {
    if (format == "pdf")
      return pdf(aut, o);
    return (o << io::of_string(format) << aut);
  }

  void save(const automaton_t aut, const std::string& filename, const std::string& fmt) {
     std::ofstream o(filename);
     put(aut,o,fmt);
     o.close();
  }


}}//end of ns awali::dyn
