// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef PRIORITY_HH
#define PRIORITY_HH

namespace awali{
namespace priority {


/* Concept: priority::value is of any of the type priority::ONE<T>,
 * priority::TWO<T>, priority::THREE<T>, etc.. for some type T (different each
 * time).
 * Since it is more specific, static linkage will preferentially use a function
 * using FIVE<T>, over FOUR<T>, over THREE<T>, etc...
 * Example: assuming existence of functions:
 *  - template<typename P> f(..., ONE<P>) and  
 *  - template<typename P> f(..., TWO<P>)
 * Then a call to f(..., priority::value) will always call the second function.
 * This is useful when multiple algorithms that are doing the same thing are
 * conditional (eg. works only for some weightsets) and we don't wan't to
 * enforce that only one function f(...) exists at the same time.
 * (Cf algos/are-equivalent.hh)
 */
  template<typename T>
  struct  _p {} ;

  template<typename T>
  using ONE = _p<T>;

  template<typename T>
  using TWO = _p<_p<T>>;

  template<typename T>
  using THREE = _p<_p<_p<T>>>;

  template<typename T>
  using FOUR = _p<_p<_p<_p<T>>>>;

  template<typename T>
  using FIVE = _p<_p<_p<_p<_p<T>>>>>;

  FIVE<int> value;

} //end of ns priority
} //end of ns awali


#endif /* PRIORITY_HH */
