/*  This file is part of the KDE libraries
    Copyright (C) 1998 Jrgen Hochwald (juergen.hochwald@privat.kkf.net

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; see the file COPYING.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.*/



#ifndef KSPLINE_H
#define KSPLINE_H
#include <stdlib.h>
#include <qapp.h>  // only for TRUE and FALSE declaration.....

#define MAXPNTS 50            // max. Anzahl der Sttzstellen
#define SPL_OK 0              // kein Fehler
#define SPL_ERROR -1          // undefinierter Fehler
#define SPL_ERRMAXPOINTS -2   // Maximale Punktzahl erreicht
#define SPL_ERRXDOPPELT -3    // Doppelter X-Wert beim Einfgen
#define SPL_ERRRANGE -4       // Koordinatenindex auerhalb des Bereiches
#define SPL_NOTFOUND -5       // gesuchter Punkt nicht gefunden

/**
* Die Klasse SplineCurve enthlt den eigentlichen Spline. Es wird ein kubischer
* Spline verwendet, d.h. die Polynome, mit denen die Funktion approximiert wird,
* sind von 3. Ordnung.<br>
* SplineCurve ist der zugrunde liegende Spline, der in KSplineEdit verwendet
* wird. Das Einstellen und Konfigurieren des Splines sollte mit KSplineEdit
* erfolgen. Es ist aber auch mglich, SplineCurve alleine ohne KSplineedit zu
* benutzen.
* <ul><li> Nach der Initialisierung wird ein leerer Spline erzeugt
* (0 Sttzstellen). Der Spline ist natrlich. Derselbe Zustand wird durch
* <a href="#clear">clear()</a> erreicht.
* <li> Mit den Funktionen <a href="#add">add()</a>,<a href="#del">del()</a>
* und <a href="#modify">modify()</a> knnen die Sttzstellen verndert werden.
* <li> <a href="#getX">getX()</a>,<a href="#getY">getY()</a>,
* <a href="#numPoints">numPoints()</a> und <a href="#findX">findX()</a> fragen
* Informationen ber die Sttzstellen ab.
* <li><a href="#getY1">getY1()</a>,<a href="#getYn">getYn()</a>,
* <a href="#setY1">setY1()</a> und <a href="#setYn">setYn()</a> beeinflussen
* die Randpunkte.
* <li><a href="#natural">natural()</a> und <a href="#setNatural">setNatural()</a> schalten
* zwischen parametisiertem und natrlichem Spline um.
* <a href="#numPoints">numPoints()</a>
* <a href="#numPoints">numPoints()</a>
* <li> <a href="#calcSpline">calcSpline()</a> berechnet die Polynomkoeffizienten
* des Splines. Die Berechnung erfolgt automatisch, sobald nach einer nderung
* eine Neuberechnung ntig wird.<a href="#spline">spline()</a> berechnet die
* Funktionswerte des Splines.
* <li> Fehlerkonstanten
* <table border=0 cellspacing="1" cellpadding="4">
* <tr><th>Fehlerkode</th><th>Beschreibung</th></tr>
* <tr><td>SPL_OK</td><td>kein Fehler</td></tr>
* <tr><td>SPL_ERROR</td><td>nicht nher definierter Fehler</td></tr>
* <tr><td>SPL_MAXPOINTS</td><td>Max. Anzahl der Sttzstellen erreicht</td></tr>
* <tr><td>SPL_ERRXDOPPELT</td><td>X-Wert bereits enthalten (bei add() und modify())</td></tr>
* <tr><td>SPL_ERRRANGE</td><td>Sttzstellenindex auerhalb des Bereiches 0..numPoints()-1</td></tr>
* <tr><td>SPL_NOTFOUND</td><td>X-Wert nicht gefunden (bei findX())</td></tr>
* </table>
* </ul>
* @short SplineCurve
* @author Jrgen Hochwald
* @version 0.2 (5.7.1998)
*/
class SplineCurve {
public:
   /**
   *  Spline-Objekt initialisieren
   */
   SplineCurve(void);
   /**
   * Splineobjekt freigeben
   */
   ~SplineCurve(void);
   /**
   * Einen Punkt in die Liste der Sttzstellen einfgen.
   * @param x,y  X-und Y-Werte der Sttzstelle
   * @return Einfgeposition (0..MAXPNTS-1), wenn die Sttzstelle
   * erfolgreich in der Liste eingefgt wurde. Im Fehlerfall wird ein
   * negativer Fehlerkode zurckgegeben.
   * <ul><li>SPL_ERRMAXPOINTS : Max. Anzahl der Sttzstellen ist erreicht
   * <li> SPL_ERRXDOPPELT : ein Punkt mit dem angegebenen X-Wert ist bereits
   * enthalten</ul>
   */
   int add(double x, double y);
   /**
   * Berechnet die Polynomkoeffizienten des Splines. Diese Funktion wird
   * automatisch aufgerufen, sobald der Spline ausgewertet werden soll und
   * seit dem letzten Aufruf Werte oder Parameter verndert wurden.
   */
   void calcSpline(void);
   /**
   * Wertet die Splinefunktion aus.
   * @param xarg X-Koordinate, fr die der Spline berechnet werden soll
   * @return Wert der Splinefunktion
   */
   double spline(double xarg);
   /**
   * Ermittelt die Anzahl der Sttzstellen. 
   * @return # der Sttzstellen
   */
   int numPoints(void) { return n; }
   /**
   * Ermittlung des X-Wertes einer Sttzustelle
   * @param Num Nummer der Sttzstelle
   * @return X-Wert
   */
   double getX(int Num);
   /**
   * Ermittlung des Y-Wertes einer Sttzstelle
   * @param Num Nummer der Sttzstelle
   * @return Y-Wert
   */
   double getY(int Num);
   /**
   * Ermittling des linken Randwertes (Ableitung)
   * @return Ableitungswert
   */
   double getY1(void) { return yp1; }
   /**
   * Ermittling des rechten Randwertes (Ableitung)
   * @return Ableitungswert
   */
   double getYn(void) { return ypn; }
   /**
   * Setzen des linken Randwertes (Ableitung)
   * @param d Ableitungswert
   */
   void   setY1(double d) { yp1=d; ReCalc=TRUE; }
   /**
   * Setzen des rechten Randwertes (Ableitung)
   * @param d Ableitungswert
   */
   void   setYn(double d) { ypn=d; }
   /**
   * Abfragen des Splinestypes. TRUE, wenn ein natrlicher Spline, FALSE wenn
   * ein parametisierter Spline. Nur beim parametisierten Spline haben die
   * Randwerte y1 und yn eine Funktion. Sie geben die Steigung der Funktion
   * an den entsprechenden Randpunkten an.
   * @return boolscher Wert
   */
   int    natural(void) { return Natuerlich; }
   /**
   * Setzen des Splinetypes
   * @param c TRUE natrlicher Spline, FALSE parametisierter Spline
   */
   void   setNatural(int c) { Natuerlich=c; ReCalc=TRUE; }
   /**
   * Sttzstelle mit X-Wert X suchen
   * @param X 
   * @return Index der Sttzstelle oder SPL_NOTFOUND, wenn nicht gefunden
   */
   int findX(double X);
   /**
   * Sttzstelle entfernen
   * @param Num Nummer der Sttzstelle, die gelscht werden soll
   * @return SPL_OK wenn erfolgreich gelscht, SPL_ERRRANGE, wenn Fehler,
   * mglicherweise Num auerhalb des definierten Bereiches.
   */
   int del(int Num);
   /**
   * Der Punkt num erhlt die Werte xy. Intern wird der Punkt Num gelscht
   * und dann ein neuer Punkt mit den gegebenen Werten erzeugt.
   * @param Num Punkt, der gendert werden soll
   * @param x,y Neue Koordinaten fr diesen Punkt
   * @return nderungsstatus: Nummer des Punktes (durch nderung
   * der X-Koordinate kann der Punkt eine neue Position in der Liste
   * bekommen) oder negativer Wert fr Fehler. Dann gilt: <ul><li>SPL_ERRRANGE = 
   * Angegebene Nummer auerhalb der definierten Punkte <li>SPL_ERRXDOPPELT =
   * Der X-Wert ist bereits enthalten</ul> Im Fehlerfall erfolgt keine nderung.
   */
   int modify(int Num, double x, double y);
   /**
   * Alle Sttzstellen lschen und Anzahl auf 0 setzen
   */
   void clear(void);
   /**
   * Text zu dem Fehler ermitteln
   * @param ErrorCode Returncode einer Splinefunktion
   * @return Zugeordneter Fehlertext
   */
   char* errorText(int ErrorCode);
   /**
   * Defaultinitialisierung. 
   * Es wird ein Spline mit drei Knotenpunkten bei (0,0)(0.5,0.5) und (1,1)
   * erzeugt
   */
   void defaultInit();
   void TestAus(void);
private:
   double X[MAXPNTS];    // Sttzstellen;
   double Y[MAXPNTS];
   double y2[MAXPNTS];  // 2. Ableitung
   int n;               // Anzahl der Sttzstellen
   int Natuerlich;      // Flag natrlicher Spline
   double yp1;          // Randbedingung links
   double ypn;          // Randbedingung rechts
   int ReCalc;          // Neuberechnung erfoderlich (True)
};
#endif
