/////////////////////////////////////////////////////////////////////////////
//
// Plugin manager
//
// Time-stamp: <97/05/06 04:18:50 vels>
// Copyright (c) VDOnet Corp. 1996
//
/////////////////////////////////////////////////////////////////////////////
  
#ifndef _KPLUGMNG_H_
#define _KPLUGMNG_H_

#include <qobject.h>
#include <qstring.h>
#include <qlist.h>
#include <dlfcn.h>

#include "kplugin.h"

// Helper class - stores information about loaded plugin
class PluginInfo
{
public:
  PluginInfo(KPlugin *plugin, QString pluginPath, void *dl_handle)
  {
    m_plugin = plugin;
    m_path   = pluginPath;
    m_handle = dl_handle;
  }

  ~PluginInfo()
  {
    if (m_plugin)
      delete m_plugin;

    if (m_handle)
      dlclose(m_handle);
  }

  KPlugin *plugin() { return m_plugin; }
  QString path()    { return m_path;   }
  void    *handle() { return m_handle; }

private:
  KPlugin *m_plugin;
  QString m_path;
  void    *m_handle;
};


/// Plugin manager
/**
   Plugins manager - knows how to load/unload plugins and how to
   create a chain from plugins.
   When you load a plugin - the shared object is loaded from file and
   a new plugin instance is created.
   All loaded plugins will be closed when KPluginManager is deleted.
   Closing last instance of a plugin will unload this plugin from memory.

   Chain idea:

    inputPlugin -> filterPlugin -> filterPlugin -> .... -> outputPlugin
  */
class KPluginManager : public QObject
{
  Q_OBJECT

public:
  /// Constructor
  /**
     Initialize class.
  */
  KPluginManager();

  /// Load input plugin.
  /**
     Load input plugin - it will be placed first in the chain. If some
     input plugin is already in the chain - it will be unloaded first.
  */
  KPlugin *loadInputPlugin(const char *path);
  
  /// Load output plugin.
  /**
     Load output plugin - it will be placed last in the chain. If some
     output plugin is already in the chain - it will be unloaded first.
  */
  KPlugin *loadOutputPlugin(const char *path);

  /// Load filter plugin.
  /**
     Load filter plugin. If only one argument is given, plugin will be
     placed right before the output plugin (after the last filter).
     If the second argument is given, the plugin will be placed *after* the
     specified plugin. You can pass the name of input plugin to place
     this filter first.
  */
  KPlugin *loadFilterPlugin(const char *path, const char *afterPath=NULL);

  /// Execute chain.
  /**
     Execute created chain. Input plugin starts passing data through
     the chain.
  */
  int start();

  /// Stop working.
  /**
     Stop the chain. Input and output should stop working immediately.
     Signal finished() is *not* emmited.
    */
  void stop();

  /// Load plugin.
  /**
     Load plugin from file. If path doesn't start with '/' the directory
     $KDEDIR/plugins is searched for this plugin.
  */
  KPlugin *loadPlugin(const char *path);

  /// Unload plugin.
  /**
     Unload given plugin. If this is a last instace of a plugin it's shared
     object is removed from memory.
  */
  int unloadPlugin(KPlugin *plugin);

  /// Input plugin.
  /**
     Return input plugin.
  */
  KPlugin *inputPlugin();

  /// Output plugin.
  /**
     Return output plugin.
  */
  KPlugin *outputPlugin();

  /// Next plugin in chain.
  /**
     Return next plugin in chain.
  */
  KPlugin *next(KPlugin *current);

  /// Chain has finished working.
  /**
     Called from output plugin. Indicates that both input *and* output
     finished working.
  */
  void chainFinished();

signals:
  /// Finished signal.
  /**
     Emmited when the chain has finished it's job. This means that input
     *and* output plugins are both finished.
  */
  void finished();

private:
  QList<PluginInfo> m_pluginList;
  QList<KPlugin>    m_chain;
};

#endif // _KPLUGMNG_H_
