/*
 *  Kaidan - A user-friendly XMPP client for every device!
 *
 *  Copyright (C) 2016-2019 Kaidan developers and contributors
 *  (see the LICENSE file for a full list of copyright authors)
 *
 *  Kaidan is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  In addition, as a special exception, the author of Kaidan gives
 *  permission to link the code of its release with the OpenSSL
 *  project's "OpenSSL" library (or with modified versions of it that
 *  use the same license as the "OpenSSL" library), and distribute the
 *  linked executables. You must obey the GNU General Public License in
 *  all respects for all of the code used other than "OpenSSL". If you
 *  modify this file, you may extend this exception to your version of
 *  the file, but you are not obligated to do so.  If you do not wish to
 *  do so, delete this exception statement from your version.
 *
 *  Kaidan is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Kaidan.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGEMODEL_H
#define MESSAGEMODEL_H

#include <QAbstractListModel>
#include "Message.h"

class QMimeType;
class MessageDb;
class Kaidan;

class MessageModel : public QAbstractListModel
{
	Q_OBJECT
	Q_PROPERTY(QString chatPartner READ chatPartner WRITE setChatPartner
	                               NOTIFY chatPartnerChanged)

public:
	enum MessageRoles {
		Timestamp = Qt::UserRole + 1,
		Id,
		Sender,
		Recipient,
		Body,
		SentByMe,
		MediaType,
		IsEdited,
		IsSent,
		IsDelivered,
		MediaUrl,
		MediaSize,
		MediaContentType,
		MediaLastModified,
		MediaLocation,
		MediaThumb,
		IsSpoiler,
		SpoilerHint
	};
	Q_ENUM(MessageRoles)

	MessageModel(Kaidan *kaidan, MessageDb *msgDb, QObject *parent = nullptr);
	~MessageModel();

	Q_REQUIRED_RESULT bool isEmpty() const;
	Q_REQUIRED_RESULT int rowCount(const QModelIndex &parent = QModelIndex()) const override;
	Q_REQUIRED_RESULT QHash<int, QByteArray> roleNames() const override;
	Q_REQUIRED_RESULT QVariant data(const QModelIndex &index, int role) const override;

	Q_INVOKABLE void fetchMore(const QModelIndex &parent) override;
	Q_INVOKABLE bool canFetchMore(const QModelIndex &parent) const override;

	QString chatPartner();
	void setChatPartner(const QString &chatPartner);

	Q_INVOKABLE bool canCorrectMessage(const QString &msgId) const;

signals:
	void chatPartnerChanged(const QString &chatPartner);

	void addMessageRequested(const Message &msg);
	void updateMessageRequested(const QString &id,
	                            const std::function<void (Message &)> &updateMsg);
	void setMessageAsSentRequested(const QString &msgId);
	void setMessageAsDeliveredRequested(const QString &msgId);

private slots:
	void handleMessagesFetched(const QVector<Message> &m_messages);

	void addMessage(const Message &msg);
	void updateMessage(const QString &id,
	                   const std::function<void (Message &)> &updateMsg);
	void setMessageAsSent(const QString &msgId);
	void setMessageAsDelivered(const QString &msgId);

private:
	void clearAll();
	void insertMessage(int i, const Message &msg);

	Kaidan *kaidan;
	MessageDb *msgDb;

	QVector<Message> m_messages;
	QString m_chatPartner;
	bool m_fetchedAll = false;
};

#endif // MESSAGEMODEL_H
