/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2009-11-13
 * Description : a plugin to blend bracketed images.
 *
 * Copyright (C) 2009-2016 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2012-2015 by Benjamin Girault <benjamin dot girault at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "expoblendingmanager.h"

// Local includes

#include "importwizarddlg.h"
#include "expoblendingdlg.h"
#include "expoblendingthread.h"
#include "alignbinary.h"
#include "enfusebinary.h"

namespace Digikam
{

class ExpoBlendingManager::Private
{
public:

    Private()
        : thread(0),
          wizard(0),
          dlg(0)
    {
    }

    QList<QUrl>            inputUrls;

    ItemUrlsMap            preProcessedUrlsMap;

    ExpoBlendingThread*    thread;

    AlignBinary            alignBinary;
    EnfuseBinary           enfuseBinary;

    ImportWizardDlg*       wizard;
    ExpoBlendingDlg*       dlg;
};

ExpoBlendingManager::ExpoBlendingManager(QObject* const parent)
    : QObject(parent),
      d(new Private)
{
    d->thread = new ExpoBlendingThread(this);

    connect(&d->enfuseBinary, SIGNAL(signalEnfuseVersion(double)),
            this, SLOT(slotSetEnfuseVersion(double)));

    if (d->enfuseBinary.isValid())
    {
        slotSetEnfuseVersion(d->enfuseBinary.getVersion());
    }
}

ExpoBlendingManager::~ExpoBlendingManager()
{
    delete d->thread;
    delete d->wizard;
    delete d->dlg;
    delete d;
}

bool ExpoBlendingManager::checkBinaries()
{
    if (!d->alignBinary.recheckDirectories())
        return false;

    if (!d->enfuseBinary.recheckDirectories())
        return false;

    return true;
}

AlignBinary& ExpoBlendingManager::alignBinary() const
{
    return d->alignBinary;
}

EnfuseBinary& ExpoBlendingManager::enfuseBinary() const
{
    return d->enfuseBinary;
}

void ExpoBlendingManager::setItemsList(const QList<QUrl>& urls)
{
    d->inputUrls = urls;
}

QList<QUrl>& ExpoBlendingManager::itemsList() const
{
    return d->inputUrls;
}

void ExpoBlendingManager::setPreProcessedMap(const ItemUrlsMap& urls)
{
    d->preProcessedUrlsMap = urls;
}

ItemUrlsMap& ExpoBlendingManager::preProcessedMap() const
{
    return d->preProcessedUrlsMap;
}

ExpoBlendingThread* ExpoBlendingManager::thread() const
{
    return d->thread;
}

void ExpoBlendingManager::run()
{
    startWizard();
}

void ExpoBlendingManager::cleanUp()
{
    d->thread->cleanUpResultFiles();
}

void ExpoBlendingManager::startWizard()
{
    delete d->wizard;
    d->wizard = new ImportWizardDlg(this);
    d->wizard->show();

    connect(d->wizard, SIGNAL(accepted()),
            this, SLOT(slotStartDialog()));
}

void ExpoBlendingManager::slotStartDialog()
{
    d->inputUrls = d->wizard->itemUrls();

    delete d->dlg;
    d->dlg = new ExpoBlendingDlg(this);
    d->dlg->show();
}

void ExpoBlendingManager::slotSetEnfuseVersion(double version)
{
    d->thread->setEnfuseVersion(version);
}

} // namespace Digikam
