#include <klocale.h>
#include <kmessagebox.h>
#include <kdialog.h>
#include <knuminput.h>
#include <kurl.h>
#include <kfiledialog.h>
#include <kcolorbutton.h>
#include <ktrader.h>

#include <qvbuttongroup.h>
#include <qhbuttongroup.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qlayout.h>
#include <qvgroupbox.h>
#include <qhgroupbox.h>
#include <qdir.h>
#include <qstring.h>

#include "generalsettings.h"
#include "setupgeneral.h"


SetupGeneral::SetupGeneral( QWidget* parent,  const char* name)
    : QWidget(parent, name)
{

    QVBoxLayout *layout = new QVBoxLayout( this,10);
    layout->setSpacing( KDialog::spacingHint() );

    QGridLayout *albumPathLayout = new QGridLayout(2,3);

    QLabel *albumPathLabel = new QLabel(this);
    albumPathLabel->setText( i18n( "Set Album Path :" ) );
    albumPathLayout->addWidget(albumPathLabel, 0, 0);

    mAlbumPathBrowseComboBox = new QComboBox(false, this);
    mAlbumPathBrowseComboBox->setEditable(true);
    albumPathLayout->addMultiCellWidget(mAlbumPathBrowseComboBox, 1, 1, 0, 1);

    mAlbumPathBrowseButton = new QPushButton( this);
    mAlbumPathBrowseButton->setText( i18n( "Browse" ) );
    albumPathLayout->addWidget( mAlbumPathBrowseButton, 1, 2);

    layout->addLayout(albumPathLayout);


    QVButtonGroup* iconSizeButtonGroup = new QVButtonGroup(this);
    iconSizeButtonGroup->setTitle( i18n( "Set Default Icon Size" ) );
    iconSizeButtonGroup->setRadioButtonExclusive(true);

    mSmallIconSizeButton = new QRadioButton(iconSizeButtonGroup);
    mSmallIconSizeButton->setText( i18n( "Small (64x64)" ) );

    mMediumIconSizeButton = new QRadioButton(iconSizeButtonGroup);
    mMediumIconSizeButton->setText( i18n( "Medium (90x90)" ) );

    mLargeIconSizeButton = new QRadioButton(iconSizeButtonGroup);
    mLargeIconSizeButton->setText( i18n( "Large (120x120)" ) );

    layout->addWidget(iconSizeButtonGroup);

    // -----------------------------------------------------------------------

    QVGroupBox* slideShowGroupBox = new QVGroupBox(i18n("SlideShow Settings"),
                                                   this);

    mSlideShowDelaySpinBox = new KIntNumInput(slideShowGroupBox);
    mSlideShowDelaySpinBox->setMinValue( 1 );
    mSlideShowDelaySpinBox->setLabel(i18n( "Delay in Seconds between SlideShow Images"),
                                     AlignRight|AlignVCenter);

    mSlideShowEffectCheckBox = new QCheckBox(slideShowGroupBox);
    mSlideShowEffectCheckBox->setText( i18n( "Use Transition Effects in SlideShow" ) );

    QVButtonGroup *slideShowFileNameButtonGroup = new QVButtonGroup(slideShowGroupBox);
    slideShowFileNameButtonGroup->setTitle( i18n("Show Filenames in SlideShow") );
    slideShowFileNameButtonGroup->setRadioButtonExclusive(true);

    mSlideShowFileNameOff = new QRadioButton(slideShowFileNameButtonGroup);
    mSlideShowFileNameOff->setText( i18n( "Off" ));

    mSlideShowFileNameAtTop = new QRadioButton(slideShowFileNameButtonGroup);
    mSlideShowFileNameAtTop->setText( i18n( "At Top" ));

    mSlideShowFileNameAtBottom = new QRadioButton(slideShowFileNameButtonGroup);
    mSlideShowFileNameAtBottom->setText( i18n( "At Bottom" ));


    layout->addWidget(slideShowGroupBox);

    // -----------------------------------------------------------------------

    QHGroupBox* colorBox = new QHGroupBox(this);
    (void) new QLabel(i18n("Set Background Color for Display of Camera Images"),
        colorBox);
    mColorButton = new KColorButton(colorBox);

    layout->addWidget(colorBox);


    // -----------------------------------------------------------------------

    QGroupBox* imageViewerGroupBox =
        new QGroupBox(i18n("Image Viewer Settings"),
                      this);
    imageViewerGroupBox->setColumnLayout(0, Qt::Vertical );
    imageViewerGroupBox->layout()->setSpacing(5);
    imageViewerGroupBox->layout()->setMargin(10);
    QGridLayout *imageViewerGroupBoxLayout =
        new QGridLayout(imageViewerGroupBox->layout());

    mUseInternalImageViewer = new QCheckBox(imageViewerGroupBox);
    mUseInternalImageViewer->setText( i18n("Use Inbuilt Image Viewer") );


    mImageViewerName = new QComboBox(false, imageViewerGroupBox);
    mImageViewerName->setEditable(false);

    KTrader::OfferList offers =
        KTrader::self()->query("image/jpeg","Type == 'Application'");

    KTrader::OfferList::Iterator iter;
    KService::Ptr ptr;
    for( iter = offers.begin(); iter != offers.end(); ++iter ) {
        ptr = *iter;
        mImageViewerName->insertItem(ptr->pixmap(KIcon::Small),
                                     ptr->name());
    }

    QLabel *imageViewerLabel = new QLabel(i18n("Set Default Image Viewer (Opens Image When Double-Clicked)"),
                                          imageViewerGroupBox);

    imageViewerGroupBoxLayout->addMultiCellWidget(mUseInternalImageViewer,
                                                  0, 0, 0, 2);
    imageViewerGroupBoxLayout->addMultiCellWidget(imageViewerLabel,
                                                  1, 1, 0, 1);
    imageViewerGroupBoxLayout->addMultiCellWidget(mImageViewerName,
                                                  1, 1, 2, 2);

    layout->addWidget(imageViewerGroupBox);

    // -----------------------------------------------------------------------

    layout->addStretch();


    loadSettings();

    connect(mAlbumPathBrowseButton, SIGNAL( clicked() ),
	   this, SLOT( slot_albumPathBrowse() ) );
    connect(mUseInternalImageViewer, SIGNAL( toggled(bool) ),
            this, SLOT( slot_imageViewerCheckBoxToggled(bool) ) );
}

SetupGeneral::~SetupGeneral()
{
}

void SetupGeneral::loadSettings()
{

    GeneralSettings genSettings;
    genSettings.readSettings();

    QString albumPath = genSettings.getAlbumPath();
    mAlbumPathBrowseComboBox->insertItem(albumPath,0);

    QString iconSize = genSettings.getDefaultIconSize();
    if (iconSize == "Small")
        mSmallIconSizeButton->setChecked(true);
    else if (iconSize == "Large")
        mLargeIconSizeButton->setChecked(true);
    else
        mMediumIconSizeButton->setChecked(true);

    int slideShowDelay = genSettings.getSlideShowDelay()/1000;
    mSlideShowDelaySpinBox->setValue(slideShowDelay);

    bool slideShowTransitionEffect =
        genSettings.getSlideShowTransitionEffect();
    if (slideShowTransitionEffect)
        mSlideShowEffectCheckBox->setChecked(true);
    else
        mSlideShowEffectCheckBox->setChecked(false);

    QString slideShowFileName = genSettings.getSlideShowFileName();
    if (slideShowFileName == "At Top")
        mSlideShowFileNameAtTop->setChecked(true);
    else if (slideShowFileName == "At Bottom")
        mSlideShowFileNameAtBottom->setChecked(true);
    else
        mSlideShowFileNameOff->setChecked(true);

    mColorButton->setColor(
        genSettings.getCameraViewBackgroundColor());

    bool useInternalImageViewer
        = genSettings.getUseInternalImageViewer();
    if (useInternalImageViewer) {
        mUseInternalImageViewer->setChecked(true);
        slot_imageViewerCheckBoxToggled(true);
    }
    else {
        mUseInternalImageViewer->setChecked(false);
        slot_imageViewerCheckBoxToggled(false);
    }

    QString imageViewer(genSettings.getImageViewer());
    if (!imageViewer.isEmpty() && mImageViewerName->isEnabled()) {
        for (int i = 0; i < mImageViewerName->count(); i++) {
            if (imageViewer == mImageViewerName->text(i)) {
                mImageViewerName->setCurrentItem(i);
                break;
            }
        }
    }
}

void SetupGeneral::saveSettings()
{
    GeneralSettings genSettings;

    QString albumPath = mAlbumPathBrowseComboBox->currentText();
    genSettings.setAlbumPath(albumPath);

    QString iconSize;
    if (mSmallIconSizeButton->isChecked())
        iconSize = "Small";
    else if (mLargeIconSizeButton->isChecked())
        iconSize = "Large";
    else
        iconSize = "Medium";
    genSettings.setDefaultIconSize(iconSize);


    int slideShowDelay = mSlideShowDelaySpinBox->value()*1000;
    genSettings.setSlideShowDelay(slideShowDelay);

    genSettings.setSlideShowTransitionEffect(
        mSlideShowEffectCheckBox->isChecked());

    QString slideShowFileName;
    if (mSlideShowFileNameOff->isChecked())
        slideShowFileName = "Off";
    else if (mSlideShowFileNameAtTop->isChecked())
        slideShowFileName = "At Top";
    else
        slideShowFileName = "At Bottom";
    genSettings.setSlideShowFileName(slideShowFileName);


    genSettings.setCameraViewBackgroundColor(
        mColorButton->color());

    genSettings.setUseInternalImageViewer(
        mUseInternalImageViewer->isChecked());

    if (mImageViewerName->isEnabled())
        genSettings.setImageViewer(
            mImageViewerName->currentText());

    genSettings.saveSettings();

}

void SetupGeneral::slot_albumPathBrowse()
{
    KURL result =
        KURL(KFileDialog::getExistingDirectory(
                 mAlbumPathBrowseComboBox->currentText(),
                 this ));

    if (!result.isEmpty()) {
        for (int i = 0; i < mAlbumPathBrowseComboBox->count();
             i++) {
            if (result.cmp(KURL(mAlbumPathBrowseComboBox->text(i)),
                           true)) {
                mAlbumPathBrowseComboBox->setCurrentItem(i);
                return;
            }
        }
        mAlbumPathBrowseComboBox->insertItem(result.path(), 0);
    }
}

void SetupGeneral::slot_imageViewerCheckBoxToggled(bool on)
{
    if (on) {
        mImageViewerName->setEnabled(false);
    }
    else {
        mImageViewerName->setEnabled(true);
    }
}
