#include <qstringlist.h>
#include <qdir.h>

#include <klocale.h>
#include <kstandarddirs.h>
#include <kmdcodec.h>
#include <kurl.h>

#include "gphoto_interface.h"
#include "gphoto_controller.h"
#include "gphoto_events.h"
#include "gphoto_messages.h"

#include "thumbnailmt.h"
#include "thumbnailsize.h"
#include "mtlist.h"
#include "camerafileiteminfo.h"
#include "thumbnailcachelocator.h"

Gphoto_Controller :: Gphoto_Controller(QObject *_parent) : QThread() {

  parent = _parent;
  cmdQueue = new MTQueue<Gphoto_Command>;
  gpInterface = new Gphoto_Interface();

  globalExit = false;

  connect(Gphoto_Messages::gpMessagesWrapper(), SIGNAL(statusChanged(const QString&)),
	  this, SLOT(slot_cameraStatusMsg(const QString&)) );

  connect(Gphoto_Messages::gpMessagesWrapper(), SIGNAL(progressChanged(int)),
     this, SLOT(slot_cameraProgressVal(int)) );

}

Gphoto_Controller :: ~Gphoto_Controller() {

    globalExit = true;
    wait();

    delete cmdQueue;

    delete gpInterface;
    gpInterface = NULL;
    parent = NULL;

    Gphoto_Messages::deleteMessagesWrapper();

}


void Gphoto_Controller :: initializeCamera() {

  cmdQueue->enqueue(new Gphoto_Command(Gphoto_Command::Init));

}

void Gphoto_Controller :: getSubFolders(const QString& folder) {

  cmdQueue->enqueue(new Gphoto_Command_GetSubFolders(folder));

}

void Gphoto_Controller :: makeFolder(const QString& folder,
				    const QString& newFolder) {

  cmdQueue->enqueue(new Gphoto_Command_MakeFolder(folder, newFolder));

}

void Gphoto_Controller :: deleteFolder(const QString& folder) {

  cmdQueue->enqueue(new Gphoto_Command_DeleteFolder(folder));

}

void Gphoto_Controller :: getThumbnail(const QString& folder,
				       const QString& imageName) {

  cmdQueue->enqueue(new Gphoto_Command_GetThumbNail(folder,
                                                    imageName));

}

void Gphoto_Controller::getImagesInfo(const QString& folder)
{
  cmdQueue->enqueue(new Gphoto_Command_GetImagesInfo(folder));
}

void Gphoto_Controller :: downloadImage(const QString& folder,
					const QString& imageName,
					const QString& saveFileName) {

  cmdQueue->enqueue(new Gphoto_Command_DownloadImage(folder, imageName,
						    saveFileName));
}

void Gphoto_Controller :: deleteImage(const QString& folder,
				      const QString& imageName) {

  cmdQueue->enqueue(new Gphoto_Command_DeleteImage(folder, imageName));

}

void Gphoto_Controller::openImage(const QString& folder, const QString& imageName)
{

    cmdQueue->enqueue(new Gphoto_Command_OpenImage(folder, imageName));

}

void Gphoto_Controller::uploadImage(const QString& folder,
                                    const QString& localImagePath,
                                    const QString& uploadName)
{
    cmdQueue->enqueue(new Gphoto_Command_UploadImage(folder,
                                                     localImagePath,
                                                     uploadName));
}

void Gphoto_Controller::getCameraInformation()
{
    cmdQueue->enqueue(new Gphoto_Command_GetInfo());
}

void Gphoto_Controller :: stopOperations() {

  cmdQueue->flush();

}


void Gphoto_Controller :: run() {


  while(true) {

    if (globalExit) return;

    while (cmdQueue->isEmpty()) {
      if (globalExit) return;
      msleep(200);
      showBusyStatus(false);
    }

    showBusyStatus(true);

    Gphoto_Command *cmd = cmdQueue->dequeue();
    if (!cmd) continue;

    switch (cmd->type()) {

    case Gphoto_Command::Init:
      initializeCamera_Real();
      break;

    case Gphoto_Command::GetSubFolders: {

      Gphoto_Command_GetSubFolders*
	command(static_cast<Gphoto_Command_GetSubFolders *>(cmd));

      getSubFolders_Real(command->parentFolder());

      delete command;
      cmd = NULL;
      break;
    }

    case Gphoto_Command::MakeFolder:
      qWarning("Make Folder");
      break;

    case Gphoto_Command::DeleteFolder:
      qWarning("Delete Folder");
      break;

    case Gphoto_Command::GetImagesInfo: {

      Gphoto_Command_GetImagesInfo*
	command(static_cast<Gphoto_Command_GetImagesInfo *>(cmd));

      getImagesInfo_Real(command->parentFolder());

      delete command;
      cmd = NULL;
      break;
    }


    case Gphoto_Command::GetThumbNail: {

      Gphoto_Command_GetThumbNail*
	command(static_cast<Gphoto_Command_GetThumbNail *>(cmd));

      getThumbNail_Real(command->parentFolder(),command->imageName());

      delete command;
      cmd = NULL;

      break;
    }

    case Gphoto_Command::DownloadImage: {

      Gphoto_Command_DownloadImage*
	command(static_cast<Gphoto_Command_DownloadImage *>(cmd));

      downloadImage_Real(command->parentFolder(),
			 command->imageName(),
			 command->saveFileName());
      break;
    }

    case Gphoto_Command::DeleteImage: {
      Gphoto_Command_DeleteImage*
	command(static_cast<Gphoto_Command_DeleteImage *>(cmd));

      deleteImage_Real(command->parentFolder(),
		       command->imageName());
      break;
    }

    case Gphoto_Command::OpenImage: {
      Gphoto_Command_OpenImage*
	command(static_cast<Gphoto_Command_OpenImage *>(cmd));

      openImage_Real(command->parentFolder(),
                     command->imageName());
      break;
    }

    case Gphoto_Command::UploadImage: {
      Gphoto_Command_UploadImage*
	command(static_cast<Gphoto_Command_UploadImage *>(cmd));

      uploadImage_Real(command->parentFolder(),
                       command->localImagePath(),
                       command->uploadName());
      break;
    }

    case Gphoto_Command::GetInfo: {
        getCameraInformation_Real();
        break;
    }

    default:
      qWarning("Unknown Command\n");
      break;

    }

    if (cmd) delete cmd;

  }


}

void Gphoto_Controller :: initializeCamera_Real()
{

    QString errorMsg;

    int retVal = gpInterface->initializeCamera(errorMsg);

    if ( retVal == Gphoto_Interface::Success) {

        QString camModel, camPort, camPath;

        gpInterface->getCameraSettings(camModel, camPort, camPath);
        postEvent(parent, new Gphoto_Event_InitializedCamera(camModel,
                                                             camPort,
                                                             camPath));

    }
    else if ( retVal == Gphoto_Interface::ErrorSetup){

        QString msg(i18n("Camera Model or Port not specified correctly.\n"
                    "Please run Setup"));
        postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));

    }
    else if ( retVal == Gphoto_Interface::ErrorInit) {

        QString msg(i18n("Failed to initialize camera.\n"
                    "Please ensure camera is connected properly and turned on"));
        if (!errorMsg.isEmpty()) {
            msg += "\n";
            msg += errorMsg;
        }
        postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));

    }

}

void Gphoto_Controller :: getSubFolders_Real(const QString& folder) {


  QStringList subFolderList;
  QStringList subFolderNameList;
  int numSubFolders = 0;
  QString errorMsg;

  subFolderList.clear();
  subFolderNameList.clear();

  if (gpInterface->getSubFolders(folder, subFolderList,
                                 subFolderNameList,
				 numSubFolders,errorMsg)
      == Gphoto_Interface::Success) {

    if (numSubFolders > 0) {
      postEvent(parent,
                new Gphoto_Event_NewSubFolders(folder,
                                               subFolderList,
                                               subFolderNameList));
    }

    for (int i=0; i<numSubFolders; i++) {
      getSubFolders_Real(subFolderList[i]);
    }

    return;

  }
  else {

      QString msg(i18n("Failed to get subfolder names from "));
      msg += folder + "\n";
      msg += errorMsg;
      slot_cameraErrorMsg(msg);
      return;
  }

}

void Gphoto_Controller::getImagesInfo_Real(const QString& folder)
{

    QString errorMsg;
    MTList<CameraFileItemInfo> infoList;

    if (gpInterface->getFilesInformation(folder, infoList, errorMsg)
        == Gphoto_Interface::Success)
        postEvent(parent, new Gphoto_Event_NewItems(folder,
                                                    infoList));
    else {
        QString msg(i18n("Failed to get images information from "));
        msg += folder;
        if (!errorMsg.isEmpty()) {
            msg += "\n";
            msg += errorMsg;
        }
        postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));
    }

}



void Gphoto_Controller :: getThumbNail_Real(const QString& folder,
					     const QString& imageName)
{
    ThumbnailMT thumb;
    gpInterface->getThumbNail(folder, imageName, thumb);

    if (!thumb.isNull()) {

        thumb.scale(ThumbnailSize(ThumbnailSize::Large));
        thumb.dropShadow();
        postEvent(parent,
                  new Gphoto_Event_ThumbNailReady(folder, imageName,
                                                  thumb, true));
    }
    else {
        postEvent(parent,
                  new Gphoto_Event_ThumbNailReady(folder, imageName,
                                                  thumb, false));
    }

}

void Gphoto_Controller :: downloadImage_Real(const QString& folder,
					     const QString& imageName,
					     const QString& saveFileName)
{
    QString errorMsg;

    cacheThumbnails(folder,imageName,saveFileName);

    if (gpInterface->downloadImage(folder, imageName,
                                   saveFileName, errorMsg)
        != Gphoto_Interface::Success) {

        QString msg(i18n("Failed to download "));
        msg += imageName;
        msg += i18n(" from ");
        msg += folder;
        if (!errorMsg.isEmpty()) {
            msg += "\n";
            msg += errorMsg;
        }
        slot_cameraErrorMsg(msg);
    }
    else {
        postEvent(parent, new Gphoto_Event_DownloadImage(folder,
                                                       imageName));
    }
}

void Gphoto_Controller::cacheThumbnails(const QString& folder,
                                        const QString& imageName,
                                        const QString& saveFileName)
{
    ThumbnailMT thumb;
    gpInterface->getThumbNail(folder, imageName, thumb);

    if (!thumb.isNull()) {

        ThumbnailMT thumbSmall, thumbMedium, thumbLarge;

        thumbLarge = thumb;
        thumbLarge.scale(ThumbnailSize(ThumbnailSize::Large));
        thumbLarge.dropShadow();

        thumbMedium = thumbLarge;
        thumbMedium.scale(ThumbnailSize(ThumbnailSize::Medium));
        thumbMedium.dropShadow();

        thumbSmall = thumbMedium;
        thumbSmall.scale(ThumbnailSize(ThumbnailSize::Small));
        thumbSmall.dropShadow();

        QString fileFolder = KURL(saveFileName).directory();
        QString thumbFileName = KURL(saveFileName).fileName();

        QString thumbSmallCacheDir,
            thumbMediumCacheDir,
            thumbLargeCacheDir;

        ThumbnailCacheLocator::locateCache(fileFolder,
                                           thumbSmallCacheDir,
                                           thumbMediumCacheDir,
                                           thumbLargeCacheDir);

        thumbSmall.save(thumbSmallCacheDir+thumbFileName);
        thumbMedium.save(thumbMediumCacheDir+thumbFileName);
        thumbLarge.save(thumbLargeCacheDir+thumbFileName);

    }
}

void Gphoto_Controller :: deleteImage_Real(const QString& folder,
					   const QString& imageName)
{
    QString errorMsg;

    if (gpInterface->deleteImage(folder, imageName, errorMsg)
        != Gphoto_Interface::Success) {

        QString msg(i18n("Failed to delete "));
        msg += imageName;
        msg += i18n(" from ");
        msg += folder;
        if (!errorMsg.isEmpty()) {
            msg += "\n";
            msg += errorMsg;
        }
        slot_cameraErrorMsg(msg);
    }
    else {

        postEvent(parent, new Gphoto_Event_DeleteImage(folder,
                                                       imageName));
    }
}

void Gphoto_Controller::openImage_Real(const QString& folder,
                                       const QString& imageName)
{
    QString saveFile;
    KStandardDirs stdDirs;
    QString errorMsg;

    saveFile = stdDirs.resourceDirs("tmp").first();
    saveFile += imageName;

    if ( gpInterface->downloadImage(folder, imageName, saveFile,
                                    errorMsg)
         == Gphoto_Interface::Success) {

        postEvent(parent, new Gphoto_Event_OpenImage(folder,
                                                     imageName));
    }
    else {
        QString msg(i18n("Failed to open "));
        msg += imageName;
        msg += i18n(" from ");
        msg += folder;
        if (!errorMsg.isEmpty()) {
            msg += "\n";
            msg += errorMsg;
        }
        slot_cameraErrorMsg(msg);
    }

}

void Gphoto_Controller::uploadImage_Real(const QString& folder,
                                         const QString& localImagePath,
                                         const QString& uploadName)
{
    QString errorMsg;

    if (gpInterface->uploadImage(folder, localImagePath,
                                 uploadName,errorMsg)
        != Gphoto_Interface::Success) {
        QString msg(i18n("Failed to upload "));
        msg += localImagePath;
        msg += i18n(" to ");
        msg += folder;
        if (!errorMsg.isEmpty()) {
            msg += "\n";
            msg += errorMsg;
        }
        slot_cameraErrorMsg(msg);
    }
    else {

        MTList<CameraFileItemInfo> infoList;
        MTList<CameraFileItemInfo> infoList2;

        if (gpInterface->getFilesInformation(folder, infoList,
                                             errorMsg)
            == Gphoto_Interface::Success) {

            while( !(infoList.isEmpty()) ) {

                CameraFileItemInfo info( infoList.first() );
                infoList.pop_front();

                if (info.getName() == uploadName) {
                    infoList2.push_back(info);
                    break;
                }

            }
            infoList.flush();

            if (!infoList2.isEmpty())
                postEvent(parent,
                          new Gphoto_Event_NewItems(folder,
                                                    infoList2));
        }
        else {
            QString msg(i18n("Failed to get images information from "));
            msg += folder;
            if (!errorMsg.isEmpty()) {
                msg += "\n";
                msg += errorMsg;
            }
            slot_cameraErrorMsg(msg);
        }
    }

}

void Gphoto_Controller::getCameraInformation_Real()
{
    QString summary;
    QString manual;
    QString about;

    gpInterface->getCameraSummary(summary);
    gpInterface->getCameraManual(manual);
    gpInterface->getCameraAbout(about);

    if (summary.isEmpty())
        summary = i18n("No Camera Summary is available");
    if (manual.isEmpty())
        manual = i18n("No Camera Manual is available");
    if (about.isEmpty())
        about = i18n("No Camera About is available");



    postEvent(parent, new Gphoto_Event_Information(summary,
                                                   manual,
                                                   about));

}

void Gphoto_Controller :: showBusyStatus(bool _busy) {

  postEvent(parent, new Gphoto_Event_Busy(_busy));

}


bool Gphoto_Controller :: isCameraInitialised() {

  return (gpInterface->isCameraInitialised());

}

bool Gphoto_Controller :: cameraSupportsThumbNails() {

  return (gpInterface->cameraSupportsThumbNails());

}

bool Gphoto_Controller :: cameraSupportsDelete() {

  return (gpInterface->cameraSupportsDelete());

}

bool Gphoto_Controller :: cameraSupportsUpload() {

  return (gpInterface->cameraSupportsUpload());

}

bool Gphoto_Controller :: cameraSupportsMkDir(){

  return (gpInterface->cameraSupportsMkDir());

}

bool Gphoto_Controller :: cameraSupportsDelDir() {

  return (gpInterface->cameraSupportsDelDir());

}

QString Gphoto_Controller :: cameraModel() const {

  QString camModel, camPort, camPath;
  gpInterface->getCameraSettings(camModel,camPort,camPath);
  return camModel;

}


QString Gphoto_Controller :: cameraPort() const {

  QString camModel, camPort, camPath;
  gpInterface->getCameraSettings(camModel,camPort,camPath);
  return camPort;

}

QString Gphoto_Controller::cameraGlobalPath() const
{
  QString camModel, camPort, camPath;
  gpInterface->getCameraSettings(camModel,camPort,camPath);
  return camPath;
}


void Gphoto_Controller :: slot_cameraStatusMsg(const QString& msg) {

  if (!msg.isEmpty())
    postEvent(parent, new Gphoto_Event_CameraStatusMsg(msg));

}

void Gphoto_Controller :: slot_cameraProgressVal(int val) {


  postEvent(parent, new Gphoto_Event_CameraProgress(val));

}

void Gphoto_Controller :: slot_cameraErrorMsg(const QString& msg) {

  if (!msg.isEmpty())
    postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));

}

