/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef FOLDERVIEW_H
#define FOLDERVIEW_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <qwidget.h>
#include <qstring.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qpopupmenu.h>
#include <qpoint.h>
#include <qevent.h>

#include <kdialog.h>
#include <klistview.h>
#include <kconfig.h>


extern QPopupMenu *moveMenu;
extern QPopupMenu *copyMenu;



class EditFolder: public KDialog
{
	Q_OBJECT
public:
	EditFolder(QWidget *parent = 0, const char *name = 0);
	~EditFolder();

	void setName(const QString &name_);
	QString name();

private:
	QLineEdit *m_name;
};


inline void EditFolder::setName(const QString &name_)
{	m_name->setText(name_); };


inline QString EditFolder::name()
{	return m_name->text(); };



class FolderViewItem: public QListViewItem
{
public:
	enum Type { Root, Inbox, Trash, User };
	
	FolderViewItem(QListView *parent = 0, Type type = User, const char *name = 0);
	FolderViewItem(QListViewItem *parent = 0, Type type = User, const char *name = 0);
	virtual ~FolderViewItem();

	Type type();

	void setName(const QString &name_);
	QString name();

	void setFiles(int files_);
	int files();

	QString path();

	void markCurrent(bool current);

protected:
	void init(Type type);
	
private:
	Type m_type;
	int m_files;
};


inline FolderViewItem::Type FolderViewItem::type()
{ return m_type; }
	

inline int FolderViewItem::files()
{ return m_files; }




/**
 * This widgets implements the list of folders.
 */
class FolderView: public KListView
{
	Q_OBJECT
public:
	/**
	 * Constructor
	 *
	 * The parameters @p parent and @p name are handled by @ref KListView, as usual.
	 */
	FolderView(QWidget *parent = 0, const char *name = 0);
	
	/**
	 * Destructor
	 */
	virtual ~FolderView();

	/**
	 * Reads the saved window settings for this widget.
	 */
	void readOptions();

	/**
	 * Saves the current window settings.
	 */
	void saveOptions();

	/**
	 * @return a pointer to the Inbox folder.
	 */
	FolderViewItem *inboxFolder();

	/**
	 * @return a pointer to the Trash folder.
	 */
	FolderViewItem *trashFolder();

	/**
	 * @return a pointer to the current folder.
	 */
	FolderViewItem *currentFolder();

	/**
	 * Sets the visible number of files of the folder @p item .
	 */
	void setFiles(FolderViewItem *item);

	void createFolder();
	void renameFolder();
	void removeFolder();

private:
	/**
	 * Creates the folder tree.
	 */
	void init();
	
	/**
	 * Creates a subtree of the folder @p parent with the subdir @p dir .
	 */
	void init(FolderViewItem *parent, const QString &dir);
	
	/**
	 * Rebuilds the menu structures of @ref ::moveMenu and @ref ::copyMenu .
	 */
	void rebuildFolderMenus();

	QPopupMenu *rebuildFolderMenu(FolderViewItem *item, QPopupMenu *menu_, bool move);
	
private slots:
	void rightButtonPressedSlot(QListViewItem *item, const QPoint &point, int);
	void currentChangedSlot(QListViewItem *item);
	void moveSlot(int id_);
	void copySlot(int id_);
	void activatedSlot(int id_);

protected:
	void contentsDragEnterEvent(QDragEnterEvent *e);
	void contentsDragMoveEvent(QDragMoveEvent *e);
	void contentsDragLeaveEvent(QDragLeaveEvent *);
	void contentsDropEvent(QDropEvent *e);

signals:
	void hasCurrent(FolderViewItem *folder);
	void move(FolderViewItem *);
	void copy(FolderViewItem *);
	void empty();

private:
	KConfig *config;
	FolderViewItem *inbox_folder;
	FolderViewItem *trash_folder;
	FolderViewItem *current_folder;

	QPopupMenu *context_menu;

	bool dragging;
	QListViewItem *old_item;
};


inline FolderViewItem *FolderView::inboxFolder()
{ return inbox_folder; }


inline FolderViewItem *FolderView::trashFolder()
{ return trash_folder; }


inline FolderViewItem *FolderView::currentFolder()
{ return (FolderViewItem *)currentItem(); }



#endif // FOLDERVIEW_H
