/*
     File: khronotimer.cpp
     Khrono : Watch, timer and countdown utility for KDE 2
     Copyright (C) 1999,2000,2001 Axel von Arnim <vonarnim@club-internet.fr>

     This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either
     version 2 of the License, or (at your option) any later
     version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "khronotimer.h"
#include <stdio.h>

KhronoTimer::KhronoTimer( QObject * parent, const char * name )
    : QObject( parent, name )
{
  mode=TIME;
  start_=FALSE;
  paused_=FALSE;
  seconds=0;
  startDays=0;
  startHours=0;
  startMins=0;
  startSecs=0;
  time = new QDateTime();
  showTime();
  startStop();
}

QString KhronoTimer::toString()
{
  return toString( TIME_ONLY );
}

QString KhronoTimer::toString( int m )
{
  switch( m ) {
    case TIME_ONLY: return time->time().toString();
    case DAYS_AND_TIME: {
      char s[5];
      if( mode==TIME ) {
        sprintf(s, "%02d ", time->date().day());
        return QString(s)+time->time().toString();
      }
      if( mode==CHRONO ) {
        sprintf(s, "%02d ", (int)(seconds/86400));
        return QString(s)+time->time().toString();
      }
      if( mode==COUNTDOWN ) {
        long totSecs=startDays*86400L+startHours*3600L+startMins*60L+startSecs;
        sprintf(s, "%02d ", (int)((totSecs-seconds)/86400));
        return QString(s)+time->time().toString();
      }
    }
  }
  return QString("");
}

void KhronoTimer::setCountdown( int d, int h, int m, int s )
{
  QTime tmp(h, m, s);
  if( !tmp.isValid() ) return;
  startDays=d;
  startHours=h;
  startMins=m;
  startSecs=s;
//  if( mode!=COUNTDOWN ) {
//    mode=COUNTDOWN;
//  }
  if( mode==COUNTDOWN ) {
    seconds=0;
    time->setTime(tmp);
    time->setDate(QDate());
    showCountdown();
  }
}

KhronoMode KhronoTimer::getMode()
{
  return mode;
}

// *** Evenements *** //

void KhronoTimer::timerEvent( QTimerEvent* )
{
  if( mode==TIME ) {
    showTime();
  } else
  if( mode==CHRONO ) {
    seconds++;
    showChrono();
  } else
  if( mode==COUNTDOWN ) {
    seconds++;
    showCountdown();
  }
}

// *** Slots publics *** //

void KhronoTimer::switchMode()
{
  if( mode==TIME ) {
    if( start_ ) startStop();
    mode=CHRONO;
    seconds=0;
    time->setTime(QTime(0, 0, 0));
    showChrono();
  } else
  if( mode==CHRONO ) {
    if( start_ ) startStop();
    mode=COUNTDOWN;
    seconds=0;
    time->setTime(QTime(startHours, startMins, startSecs));
    showCountdown();
  } else
  if( mode==COUNTDOWN ) {
    if( !start_ ) startStop();
    mode=TIME;
    showTime();
  }
  emit modeSwitched(mode);
}

void KhronoTimer::startStop()
{
  if( start_==FALSE ) {
    start_=TRUE;
    timer=startTimer(1000);
  } else {
    start_=FALSE;
    killTimer(timer);
    if( paused_ ) pause();
  }
  emit started( start_ );
}

void KhronoTimer::reset()
{
  if( mode==CHRONO ) {
    seconds=0;
    time->setTime(QTime(0, 0, 0));
    showChrono();
  } else
  if( mode==COUNTDOWN ) {
    seconds=0;
    time->setTime(QTime(startHours, startMins, startSecs));
    showCountdown();
  }
}

void KhronoTimer::pause()
{
  paused_=!paused_;
  emit paused(paused_);
  if( start_==FALSE ) emit tick();
}

// *** Methodes privees *** //

void KhronoTimer::showTime()
{
  QDateTime tmp=QDateTime::currentDateTime();
  QDate date=tmp.date();
  time->setTime(tmp.time());
  time->setDate(date);  // mise a currentTime
  if( !paused_ ) {
    emit tick();
    emit dateChanged(date);
  }
}

//void KhronoTimer::show()
//{
//  switch(mode) {
//  case TIME:
//    showTime();
//    break;
//  case CHRONO:
//    showChrono();
//    break;
//  case COUNTDOWN:
//    showCountdown();
//    break;
//  }
//}

void KhronoTimer::showChrono()
{
  int days=seconds/86400;
  QTime tmp=QTime().addSecs( seconds%86400 );
  time->setTime(tmp);
  if( !paused_ ) {
    emit daysChanged(days);
    emit tick();
  }
}

void KhronoTimer::showCountdown()
{
  long totSecs=startDays*86400L+startHours*3600L+startMins*60L+startSecs;
  int days=0;
  if( (start_==TRUE)&&(seconds>totSecs) ) {
    emit alarm();
    startStop();
  } else {
    days=(totSecs-seconds)/86400;
    long remsecs=(totSecs-seconds)%86400;
    int hrs=remsecs/3600;
    int mns=(remsecs%3600)/60;
    int sds=(remsecs%3600)%60;
    QTime tmp=QTime(hrs, mns, sds);
    time->setTime(tmp);
  }
//  if( seconds>(startMins*60+startSecs) ) {
//    startStop();
//    return;
//  }
//  if( !time->setHMS( 0, startMins-(59-startSecs+seconds)/60, (60+startSecs-seconds)%60 ) ) {
//    printf("invalid time : %d:%d:%d\n", 0, startMins-(59-startSecs+seconds)/60, (60+startSecs-seconds)%60);
//    return;
//  }
  if( !paused_ ) {
    emit daysChanged(days);
    emit tick();
  }
}
