/***************************************************************************
                          kplview.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qscrollbar.h>
#include <qpaintdevicemetrics.h>
#include <qpixmap.h>
#include <qprinter.h>
#include "kplview.h"
#include "kpldoc.h"
#include "kpl.h"
#include "kplwidget.h"
#include "legenditem.h"
#include "frameitem.h"

const int KplView::scrollBarSize = 16;

KplView::KplView(KplApp* parent, double f) :
 QWidget(parent), sHScrollPos(0), sVScrollPos(0), xPict(0), yPict(0),
 scrx(0), scry(0), fZoom(f)
{
  doc = parent->getDocument();
  fxn = fyn = 1.0;
  page = new KplWidget(this);
  horScrollBar = new QScrollBar(QScrollBar::Horizontal, this);
  vertScrollBar = new QScrollBar(QScrollBar::Vertical, this);
  connect(horScrollBar, SIGNAL(valueChanged(int)),
          SLOT(slotHorScroll(int)));
  connect(vertScrollBar, SIGNAL(valueChanged(int)),
          SLOT(slotVertScroll(int)));
  connect(page, SIGNAL(leftButtonPressed(QMouseEvent*)),
          SLOT(leftButtonPressed(QMouseEvent*)));
  connect(page, SIGNAL(leftButtonReleased(QMouseEvent*)),
          SLOT(leftButtonReleased(QMouseEvent*)));
  connect(page, SIGNAL(mouseMoved(QMouseEvent*)),
          SLOT(slotMouseMoved(QMouseEvent*)));
  connect(page, SIGNAL(mouseEntered(const QPoint&)),
          SLOT(slotMouseEntered(const QPoint&)));
}

KplView::~KplView()
{
}

void KplView::print(QPrinter* printer) const
{
  KGraph g;
  QSize sz = printer->margins();
  g.plInit(printer, sz.width(), sz.height());
  for (KplItem* item = doc->itd.first(); item; item = doc->itd.next())
    item->draw(&g);
  g.end();
}

void KplView::layout()
{
  QPaintDeviceMetrics pdm(this);
  xPict = qRound(10.0 * fZoom * doc->aut.xf * pdm.width() / pdm.widthMM());
  yPict = qRound(10.0 * fZoom * doc->aut.yf * pdm.height() / pdm.heightMM());
  scrx = width();
  scry = height();
  bool hScroll = false, vScroll = false;
  if (xPict > scrx) {
    hScroll = true;
    scry -= scrollBarSize;
  }
  if (yPict > scry) {
    vScroll = true;
    scrx -= scrollBarSize;
    if (!hScroll)
      if (xPict > scrx) {
        hScroll = true;
        scry -= scrollBarSize;
      }
  }
  if (hScroll) {
    horScrollBar->setGeometry(0, scry, scrx, scrollBarSize);
    horScrollBar->setRange(0, xPict - scrx);
    horScrollBar->setSteps(10, scrx);
    horScrollBar->setValue(sHScrollPos);
    horScrollBar->show();
  } else {
    horScrollBar->hide();
    horScrollBar->setRange(0, 0);
    sHScrollPos = 0;
  }
  if (vScroll) {
    vertScrollBar->setGeometry(scrx, 0, scrollBarSize, scry);
    vertScrollBar->setRange(0, yPict - scry);
    vertScrollBar->setSteps(10, scry);
    vertScrollBar->setValue(yPict - scry - sVScrollPos);
    vertScrollBar->show();
  } else {
    vertScrollBar->hide();
    vertScrollBar->setRange(0, 0);
    sVScrollPos = 0;
  }
  page->resize(scrx, scry);
}

void KplView::doDraw()
{
  for (KplItem* item = doc->itd.first(); item; item = doc->itd.next())
    item->draw(this);
}

void KplView::eraseWidget()
{
  doPaint(false);
}

void KplView::doPaint(bool paint)
{
  page->setCursor(waitCursor);
  page->pm->resize(page->size());
  plInit(page->pm, sHScrollPos, sVScrollPos);
  if (paint) {
    factor = fZoom;
    format(doc->aut.xf, doc->aut.yf);
    doDraw();
  }
  end();
  bitBlt(page, 0, 0, page->pm);
  page->setCursor(crossCursor);
}

void KplView::slotScrollLeft()
{
  horScrollBar->subtractLine();
}

void KplView::slotScrollRight()
{
  horScrollBar->addLine();
}

void KplView::slotScrollUp()
{
  vertScrollBar->subtractLine();
}

void KplView::slotScrollDown()
{
  vertScrollBar->addLine();
}

void KplView::slotScrollPageUp()
{
  vertScrollBar->subtractPage();
}

void KplView::slotScrollPageDown()
{
  vertScrollBar->addPage();
}

void KplView::slotScrollPageLeft()
{
  horScrollBar->subtractPage();
}

void KplView::slotScrollPageRight()
{
  horScrollBar->addPage();
}

void KplView::slotScrollHomeLeft()
{
  horScrollBar->setValue(0);
}

void KplView::slotScrollHomeDown()
{
  vertScrollBar->setValue(vertScrollBar->maxValue());
}

void KplView::slotScrollEndRight()
{
  horScrollBar->setValue(horScrollBar->maxValue());
}

void KplView::slotScrollEndUp()
{
  vertScrollBar->setValue(0);
}

void KplView::paintWidget()
{
  doPaint(true);
}

void KplView::leftButtonPressed(QMouseEvent* e)
{
  for (KplItem* item = doc->itd.last(); item; item = doc->itd.prev()) {
    if (item->boundingRect().contains(e->pos())) {
      if (sel.contains(item))
        sel.remove(item);
      else
        sel.append(item);
      startPos = dragPos = e->pos();
      drawBoundingRect(item->boundingRect());
      emit mouseMoved(fromcm(item->xPos(), item->yPos()));
      break;
    }
  }
}

void KplView::leftButtonReleased(QMouseEvent* e)
{
  if (sel.count() && (!(e->state() & ControlButton))) {
    QPoint d = e->pos() - startPos;
    double snap = 0.05;
    double dx = snap * qRound((d.x() / (scalx * factor)) / snap);
    double dy = snap * qRound((d.y() / (scaly * factor)) / snap);
    if (dx || dy) {
      for (KplItem* it1 = sel.first(); it1; it1 = sel.next()) {
        it1->setPos(it1->xPos() + dx, it1->yPos() + dy);
        if (it1->iType() == KplItem::Frame)
          if (it1 != doc->itd.getLast())
            for (KplItem* itm = doc->itd.at(doc->itd.find(it1) + 1); itm;
                 itm = doc->itd.next()) {
              if (itm->iType() == KplItem::Frame)
                break;
              if (itm->iType() >= KplItem::Legend)
                itm->setPos(itm->xPos() + dx, itm->yPos() + dy);
            }
      }
      doc->setModified();
      doc->backupItems();
    } else {
      d = dragPos - startPos;
      for (KplItem* it1 = sel.first(); it1; it1 = sel.next()) {
        QRect r = it1->boundingRect();
        r.moveBy(d.x(), d.y());
        drawBoundingRect(r);
      }
    }
    sel.clear();
  }
}

void KplView::slotMouseMoved(QMouseEvent* e)
{
  if (sel.count()) {
    if (!(e->state() & ControlButton)) {
      KplItem* it1 = sel.first();
      for (KplItem* itm = it1; itm; itm = sel.next()) {
        QRect r = itm->boundingRect();
        QPoint d = dragPos - startPos;
        r.moveBy(d.x(), d.y());
        drawBoundingRect(r);
        d = e->pos() - dragPos;
        r.moveBy(d.x(), d.y());
        drawBoundingRect(r);
      }
      dragPos = e->pos();
      QPoint d = e->pos() - startPos;
      emit mouseMoved(fromcm(d.x() / (scalx * factor) + it1->xPos(),
                             d.y() / (scaly * factor) + it1->yPos()));
    }
  } else
    emit mouseMoved(e->pos());
}

void KplView::slotMouseEntered(const QPoint& p)
{
  emit mouseMoved(p);
}

void KplView::resizeEvent(QResizeEvent*)
{
  layout();
  doPaint(((KplApp*) parent())->plot);
}

void KplView::drawBoundingRect(const QRect& r)
{
  begin(page->pm);
  setRasterOp(NotROP);
  setPen(DashLine);
  drawRect(r);
  end();
  bitBlt(page, 0, 0, page->pm);
}

void KplView::slotHorScroll(int pos)
{
  sHScrollPos = pos;
  paintWidget();
}

void KplView::slotVertScroll(int pos)
{
  sVScrollPos = vertScrollBar->maxValue() - pos;
  paintWidget();
}
