    /*

    tspio.C  for  ktsp-0.1.0

    Copyright (C) 1999 Uwe Thiem
                       uwe@kde.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */


#include <iostream.h>
#include <qobject.h>
#include <qstring.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qfileinfo.h>
#include <kapp.h>


#include "tspio.h"
#include "ktsp.h"



// public member functions

TspIO::TspIO()
  {
  _numberOfPoints = -1;
  _numberOfNeighbours = -1;
  random = new Random();
  CHECK_PTR( random );
  }



TspIO::~TspIO()
  {
  delete random;
  }



Point *TspIO::generateTsp( int points, int neighbours )
  {
  if ( points < 10 )
    {
    _numberOfPoints = 10;
    }
  else
    {
    _numberOfPoints = points;
    }
  if (neighbours < 3 )
    {
    _numberOfNeighbours = 3;
    }
  else
    {
    _numberOfNeighbours = neighbours;
    }
  Point *p = new Point[points];
  CHECK_PTR( p );
  for ( int i = 0; i < _numberOfPoints; i++ )
    {
    bool done = false;
    DBL x = 0.0, y = 0.0;
    p[i].setNumberOfNeighbours( _numberOfNeighbours );
    kapp->processEvents();
    while ( !done )
      {
      x = random->drand( 0.0, 400.0);
      y = random->drand( 0.0, 300.0);
      done = true;
      for ( int t = 0; t < i; t++ )
        {
        if ( ( x == p[t].x() ) && ( y == p[t].y() ) )
          {
          done = false;
          }
        } // for t
      } // while
      p[i].setX( x );
      p[i].setY( y );
      if ( i == _numberOfPoints - 1 )
        {
        p[i].setForwardI( 0 );
        p[i].setForwardP( &(p[0]) );
        }
      else
        {
        p[i].setForwardI( i + 1 );
        p[i].setForwardP( &(p[i+1]) );
        }
      if ( i == 0 )
        {
        p[i].setBackwardI( _numberOfPoints - 1 );
        p[i].setBackwardP( &(p[_numberOfPoints-1]) );
        }
      else
        {
        p[i].setBackwardI ( i - 1 );
        p[i].setBackwardP( &(p[i-1]) );
        }
      p[i].setDirectionForward( true );
    } // for i
  return p;
  }


Point *TspIO::loadTsp( QString name, int neighbours, bool remove )
  {
  Point *p = 0;
  Point *tmp = 0;
  QFile file( name );
  QFileInfo info( file );
  if ( info.isDir() ) return p;
  if ( file.open( IO_ReadOnly ) )
    {
    QTextStream stream( &file );
    QString line;
    bool ok;
    int numLines = 0;
    int numPoints = 0;
    while ( !stream.eof() )
      {
      line = stream.readLine();
      numLines++;
      }
    // there can't be more points in that TSP than half the number
    // of lines
    tmp = new Point[numLines / 2];
    CHECK_PTR( tmp );
    numLines = 0;
    file.at( 0 );
    // Check the two lines in the header
    if ( stream.eof() )
      {
      delete [] tmp;
      file.close();
      return p;
      }
    line = stream.readLine();
    if ( !isTsp( line ) )
      {
      delete [] tmp;
      file.close();
      return p;
      }
    if ( stream.eof() )
      {
      delete [] tmp;
      file.close();
      return p;
      }
    line = stream.readLine();
    if ( !isCorrectVersion( line ) )
      {
      delete [] tmp;
      file.close();
      return p;
      }
    // now read the data
    while ( !stream.eof() )
      {
      line = stream.readLine();
      if ( !ignore( line ) )
        {
        if ( ( numLines % 2 ) == 0 )
          {
          // even line number = x value
          tmp[numPoints].setX( line.toDouble( &ok ) );
          }
        else
          {
          // odd line number = y value
          tmp[numPoints].setY( line.toDouble( &ok ) );
          numPoints++;
          }
        numLines++;
        }
      }
    // all points read
    if ( ( numLines % 2 ) == 1 )
      {
      // Oops! File was corrupt
      // p is still a null pointer, we just have to
      // delete tmp
      delete [] tmp;
      }
    else
      {
      // let's cut the data structure down to its real
      // size and prepare everything for returning the TSP
      p = new Point[numPoints];
      CHECK_PTR( p );
      _numberOfNeighbours = neighbours;
      if ( _numberOfNeighbours >= numPoints )
                     _numberOfNeighbours = numPoints - 1;
      for ( int i = 0; i < numPoints; i++ )
        {
        p[i].setX( tmp[i].x() );
        p[i].setY( tmp[i].y() );
        p[i].setForwardI( i + 1 );
        p[i].setForwardP( &(p[i+1]) );
        p[i].setBackwardI( i - 1 );
        p[i].setBackwardP( &(p[i-1]) );
        p[i].setDirectionForward( true );
        p[i].setNumberOfNeighbours( _numberOfNeighbours );
        }
      // correct the first and last one
      p[0].setBackwardI( numPoints - 1 );
      p[0].setBackwardP( &(p[numPoints-1]) );
      p[numPoints-1].setForwardI( 0 );
      p[numPoints-1].setForwardP( &(p[0]) );
      delete[] tmp;
      _numberOfPoints = numPoints;
      _numberOfNeighbours = neighbours;
      }
    file.close();
    if ( remove ) file.remove();
    }
  else
    {
    // could not open file
    // p is still a null pointer => nothing to do
    }
  return p;
  }


bool TspIO::saveTsp( QString name, Point *tsp, int points )
  {
  QFile file( name );
  if ( !file.open( IO_WriteOnly ) ) return false;
  QTextStream stream( &file );
  stream.precision( 60 );
  stream << "# TSP file\n";
  if ( file.status() != IO_Ok )
    {
    file.close();
    return false;
    }
  stream << "# Version 1.0\n";
  if ( file.status() != IO_Ok )
    {
    file.close();
    return false;
    }
  Point *p = tsp;
  do
    {
    stream << p->x() << "\n";
    stream << p->y() << "\n\n";
    if ( file.status() != IO_Ok )
      {
      file.close();
      return false;
      }
    p = p->nextPointer();
    } while ( p != tsp );
  file.close();
  return true;
  }


bool TspIO::ignore( QString s )
  {
  int i = s.find( '#' );
  if ( i >= 0 ) s.truncate( i );
  QString tmp = s.simplifyWhiteSpace();
  // We removed a possible comment and removed all white spaces
  // at the begin and end of the string. All white spaces inside
  // the string have been converted to single spaces.
  // Numbers can't contain spaces so the remaining string is
  // either empty or a number.
  s = tmp;
  return s.isEmpty();
  }


bool TspIO::isTsp( QString s )
  {
  int i = s.find( "TSP file" );
  bool rc = true;
  if ( i < 0 ) rc = false;
  return rc;
  }


bool TspIO::isCorrectVersion( QString s )
  {
  int i = s.find( "Version" );
  if ( i < 0 ) return false;
  s.remove( 0, i + 7 );
  i = s.find( '#' );
  if ( i >= 0 ) s.truncate( i );
  QString tmp = s.simplifyWhiteSpace();
  s = tmp;
  double version = s.toDouble();
  bool rc = false;
  if ( ( version > 0.0) && ( version <= 1.0 ) ) rc = true;
  return rc;
  }


int TspIO::numberOfPoints()
  {
  return _numberOfPoints;
  }


int TspIO::numberOfNeighbours()
  {
  return _numberOfNeighbours;
  }


