#include <config.h>
#include "smbprot.h"
#include <kapp.h>
#include <stdlib.h>
#include <qmessagebox.h>

#if HAVE_SYS_VFSTAB_H
#include <sys/vfstab.h>
#include <sys/mnttab.h>
#else
#include <mntent.h>
#endif

// get definition for "processForShell()"
#include "common.h"

int sambaVersion()
{
	kapp->getConfig()->setGroup("Configuration");
	return kapp->getConfig()->readNumEntry("SambaVersion",1);
}

QString localConfigFile()
{
	QString		path(getenv("HOME"));
	if (path.right(1) != "/") path.append("/");
	path.append(".smb.conf");
	return path;
}

//----------------------------------------------------------------------------------------------------

SmbProtocol::SmbProtocol(const char *mpoint, const char *service, const char *user, const char *passwd, const char *ip)
	: FileProtocol(mpoint)
{
	parseService(service);
	MountPoint = mpoint;
	if (user) {
		User = user;
		Passwd = passwd;
	}
	IP = ip;
	Type = Protocol::Smb;
	Prefix = "file:";
	Prefix += MountPoint;
	if (Prefix.right(1) != "/") Prefix += '/';
	Mounted = false;
}

bool SmbProtocol::mount()
{
	if (isMounted()) return true;
	sendMessage("Mounting device...");
	QString	cmd;
	cmd += "smbmount";
	cmd += (" " + processForShell(Service));
	switch (sambaVersion()) {
	   // 2.0.3
	   case 0:
	   // 2.0.5a
	   case 1:
		if (sambaVersion() == 0) cmd += (" -c 'mount " + processForShell(MountPoint) + "'");
		else cmd += (" " + processForShell(MountPoint));
		if (User.isEmpty()) cmd += " -N";
		else cmd += (" -U " + processForShell(User) + "%" + processForShell(Passwd));
		if (!IP.isEmpty()) cmd += (" -I " + IP);
		if (!Workgroup.isEmpty()) cmd += (" -W " + Workgroup);
		break;
	   case 2:
	   // 2.0.6
		cmd += (" " + processForShell(MountPoint) + " -o ");
		if (User.isEmpty()) cmd += "guest";
		else cmd += ("username=" + processForShell(User) + ",password=" + processForShell(Passwd));
		if (!IP.isEmpty()) cmd += (",ip=" + IP);
		if (!Workgroup.isEmpty()) cmd += (",workgroup=" + Workgroup);
		break;
	}
	// adding extra arguments
cout << cmd.data() << endl;
	int	result = system(cmd.data());
cout << "smbmount result : " << result << endl;
	if (result != 0) goto error;
#if HAVE_SYS_VFSTAB
#else
	if (!isMounted()) goto error;
#endif
	Mounted = true;
	return TRUE;
error:
	ErrorMsg = i18n("Unable to mount device");
	return FALSE;
}

bool SmbProtocol::umount()
{
	if (!isMounted()) return true;
	sendMessage("Unmounting device...");
	QString	cmd;
	cmd += "smbumount ";
	cmd += processForShell(MountPoint);
	int	result = system(cmd.data());
cout << "smbumount result : " << result << endl;
	if (result != 0) goto error;
#if HAVE_SYS_VFSTAB
#else
	if (isMounted()) goto error;
#endif
	Mounted = false;
	return TRUE;
error:
	ErrorMsg = i18n("Unable to unmount device");
	return FALSE;
}

bool SmbProtocol::init()
{ return mount();}

bool SmbProtocol::finish()
{ return umount();}

bool SmbProtocol::shutDown()
{
	if (isMounted()) {
		QString		msg(Service.data());
		msg += i18n(" is mounted : unmount ?");
		if (QMessageBox::information((QWidget*)0,i18n("Confirmation"),msg.data(),i18n("Yes"),i18n("No"),0,0,1) == 0)
			if (!umount()) QMessageBox::critical((QWidget*)0,i18n("Error"),ErrorMsg.data(),QMessageBox::Ok | QMessageBox::Default,0);
	}
	return TRUE;
}

bool SmbProtocol::isAccesible()
{ return isMounted();}

const char* SmbProtocol::menuEntry()
{ return (isMounted() ? i18n("Unmount") : i18n("Mount"));}

bool SmbProtocol::isMounted()
{
#if HAVE_SYS_VFSTAB_H
	return Mounted;
#else
	FILE	*fp = setmntent(MOUNTED,"r");
	if (fp == 0) return FALSE;
	struct mntent	*buf;
	bool	done(FALSE);
	QString service_tmp(Service.data());
	service_tmp.replace(QRegExp(" "),"_");
	while (!done && (buf = getmntent(fp)) != 0) {
		if (buf->mnt_fsname == service_tmp) done = TRUE;
		if (!MountPoint.isEmpty() && MountPoint.find(buf->mnt_dir) == -1) done = FALSE;
	}
	endmntent(fp);
	return done;
#endif
}

bool SmbProtocol::check()
{
	bool	old = Mounted;
	Mounted = isMounted();
	return (old == Mounted);
}

QString SmbProtocol::mntargs()
{
	QString	args("");
	if (!IP.isEmpty()) args += (" -I " + IP);
	if (!Workgroup.isEmpty()) args += (" -W " + Workgroup);
	return args;
}

void SmbProtocol::parseService(const char *service)
{
	QString	Buffer(service);
	int	p1(0),p2(0),p3(0);
	if (Buffer[0] != '/' || Buffer[1] != '/') return;
	if ((p1 = Buffer.find('/',2)) == -1) return;
	Server = Buffer.mid(2,p1-2);
	Share = Buffer.mid(p1+1,Buffer.length());
	if ((p2 = Server.find('(')) != -1) {	// Workgroup specified
		if ((p3 = Server.find(')',p2+1)) == -1) return;
		Workgroup = Server.mid(p2+1,p3-p2-1);
		Server.truncate(p2);
	}
	Service.sprintf("//%s/%s",Server.data(),Share.data());
}
