/***************************************************************************
                          traficview.cpp  -  description                              
                             -------------------                                         
    begin                : Fri Apr 16 1999                                           
    copyright            : (C) 1999 by Norbert Weuster                         
    email                : weuster@uni-duisburg.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "ktrafficview.h"

KTrafficView::KTrafficView(QWidget* parent, KnetdumpDoc* doc=0, const char *name ) : QWidget(parent,name) , pdoc(doc) {
    // an update timer for the widget
    drawTimer = new QTimer( this );
    connect( drawTimer, SIGNAL(timeout()), SLOT(drawAll()) );
    drawing =false;
    actualSize=0;
}

KTrafficView::~KTrafficView(){
    delete (drawTimer);
}

void KTrafficView::startslot () {
    //
    // store each new node in a list and view them
    //

    // get addresses
    QString from,to;
    pdoc->gethostname_from(from);
    pdoc->gethostname_to(to);
    if (!to || !from) {
        emit gotData();
        return;
    }
    // store connection
    connections(from, to).inc(1+ pdoc->byterec() / 20); 
    
    emit gotData();

    // if there is a new node, draw all, else draw connections
    if (pdoc->getNodeNum()>actualSize) {
        actualSize=pdoc->getNodeNum();
        drawAll();
    }
    else 
        drawConns();
    
}

void KTrafficView::paintEvent(QPaintEvent*)  {
    //
    // Draw current nodes and connections
    //
    if (drawing) return;
    drawAll();
} 

void KTrafficView::drawNodes(QPainter& p) { 
    //
    // Draw nodes in a circle
    //
    QBrush brush (blue);
    QPen pen (NoPen);
    p.setBrush (brush);
    p.setPen (pen);
    
    double coordi=0;
    register int r = min((width()-60)/2, (height()-60)/2);    // set radius

    // draw text:
    p.drawText(0,10,i18n("remote Nodes:"));
    p.setPen(red);
    p.drawText(width()-14, 10, "IP");
    p.setPen(green);
    p.drawText(width()-27, 25, "ARP");
    p.setPen(QColor(255,100,0));
    p.drawText(width()-33, 40, "ICMP");
    p.setPen(yellow);
    p.drawText(width()-55, 55, "TCP/UDP");
    p.setPen(black);
    
    // draw nodes
    for (int ii=pdoc->getNodeNum()-1; ii>=0; ii--) {
        QString node=pdoc->getNode(ii);
        if (!pdoc->isNodeSelected(node))
            continue;
        coordi = coordinate[(string)node];           
        if (pdoc->isNodeLocal(node)) {
            // local nodes in a circle
            int x=0,y=0;
            x = -30 + width()/2 + (int)(r * cos(coordi));
            y = 10 + height()/2 + (int)(r * sin(coordi));
            // blue points
            p.drawEllipse(x-2,y-2,5,5);
            // name
            p.drawText(x-2,y-8,pdoc->getNodeDomainName(node));
        }
        else { 
            // remote nodes at the left side
            p.drawText(0, (int) coordi, pdoc->getNodeDomainName(node));
        }
    }
    return;
}   

void KTrafficView::drawConnections (QPainter& p) {
    //
    // Draw connections between hosts. 
    //
    double from_coordi, to_coordi;
    int from_x, to_x, from_y, to_y;
    int r = min((width()-60)/2, (height()-60)/2);   // set radius
    for (ConnList::iterator i=connections.begin(); i!=connections.end(); ++i) {
        from_coordi=0.0;
        to_coordi=0.0;
        if (i->load() > 0) {
            // from node
            from_coordi = coordinate[(string)i->from()];
            if (pdoc->isNodeLocal(i->from())) {
                from_x = -30 + width()/2 + (int)(r * cos(from_coordi));
                from_y = 10 + height()/2 + (int)(r * sin(from_coordi));
            }
            else {
                from_x = 20;
                from_y = (int)from_coordi;
            }
            // if node is deselected, it will be 0
            if (from_coordi == 0.0)
                continue;

            // to node
            to_coordi = coordinate[(string)i->to()];
            if (pdoc->isNodeLocal(i->to())) {
                to_x = -30 + width()/2 + (int)(r * cos(to_coordi));
                to_y = 10 + height()/2 + (int)(r * sin(to_coordi));
            }
            else {
                to_x = 20;
                to_y = (int) to_coordi; 
            }
            // if node is deselected, it will be 0
            if (to_coordi == 0.0)
                continue;
            
            QPen pen (getColor(),i->load());
            p.setPen (pen);
            p.drawLine(from_x,from_y,to_x,to_y);
            
            i->reset();
        }
    }
}

void KTrafficView::drawAll() {
    //
    // Draw current View.
    //
    if (drawing && actualSize) return;
    setUpdatesEnabled( FALSE );
    drawing = true;
    drawTimer->stop();
    // calculate coordinates for nodes
    calccoordi();

    QRect r=rect();
    QPixmap pm(r.size());
    pm.fill(this, r.bottomLeft());
    QPainter p;
    p.begin(&pm);
    // draw nodes
    drawNodes(p);
    // draw connections between nodes
    drawConnections(p);    

    p.end();
    // erase();
    bitBlt( this, 0, 0, &pm); 
    drawing = false;
    setUpdatesEnabled( TRUE );
    drawTimer->start( pdoc->getUpdateTime(), TRUE );     // single-shot
}

void KTrafficView::drawConns() {
    //
    // only draws new connections
    //
    if (drawing) return;
    setUpdatesEnabled( FALSE );
    drawing = true;

    QPainter p;
    p.begin(this);
    // draw connections between nodes
    drawConnections(p);    
    p.end();
    drawing = false;
    setUpdatesEnabled( TRUE );
}

void KTrafficView::calccoordi () {  // generates map, used for Connections and Nodes
    int nNodes=0;
    coordinate.clear();  

    // count local nodes
    for (int ii=pdoc->getNodeNum()-1; ii>=0; ii--) 
        if (pdoc->isNodeSelected(pdoc->getNode(ii)) && pdoc->isNodeLocal(pdoc->getNode(ii)))
            nNodes++;

    // phi=0 is used for all unselected nodes
    double deltaPhi = 2.0*M_PI/nNodes; 
    double phi  = 0.001;
    double yPos = 20.0;

    // generate map
    for (int ii=pdoc->getNodeNum()-1; ii>=0; ii--) {
        QString node=pdoc->getNode(ii);
        if (!pdoc->isNodeSelected(pdoc->getNode(ii)))
            continue;
        if (pdoc->isNodeLocal(node)) {
            coordinate[(string) node]= phi;
            phi += deltaPhi;
        }
        else {
            coordinate[(string)node]= yPos;
            yPos += 10.0;
        }
    }
}
 
QColor KTrafficView::getColor() {
    //
    // get appropriate color for connections
    //
    switch (pdoc->getNetwProtocol()) {
    case ETHERTYPE_IP:
        switch( ( (struct iphdr *)pdoc->getpNetwLayer())->protocol) {
        case IPPROTO_ICMP:
            return(QColor(255,100,0));
            break;
        case IPPROTO_TCP:
        case IPPROTO_UDP:
            return(yellow);
            break;
        default:
            return(red);
            break;
        }
        break;
    case ETHERTYPE_ARP:
    case ETHERTYPE_REVARP:
        return(green);
        break;
    default:
        return(blue);
        break;
    }

}

//////////////////////////////////////////////////////////////////////////////
// Connections
KTrafficView::Connection::Connection (QString n1, QString n2) : a(n1), b(n2) {
    netLoad = 0;
}
 
KTrafficView::Connection::~Connection () {
}

bool KTrafficView::Connection::operator== (const Connection& rhs) const {
    return (from() == rhs.from() && to() == rhs.to());
}

const QString KTrafficView::Connection::from () const {
    // returns connection FROM node
    if ( (string) a > (string)b)
        return a;
    return b;
}

const QString KTrafficView::Connection::to () const {
    // returns connection TO node
    if ( (string) a > (string)b)
        return b;
    return a;
}

inline int KTrafficView::Connection::load () const {
    return netLoad;
}

inline void KTrafficView::Connection::reset () {
       netLoad = 0;
}

inline void KTrafficView::Connection::decload () {
    netLoad-= 10;
    if (netLoad < 0)
        netLoad = 0;
}

void KTrafficView::Connection::inc (u_int load) {
    netLoad=load;
}

///////////////////////////////////////////////////////////////////////////////
//
KTrafficView::Connection& KTrafficView::ConnList::operator() (const QString &n1, const QString &n2) {
    //
    // Return connection between two nodes
    //
    Connection c (n1, n2);
    
    for (ConnList::iterator i=begin() ; i!=end(); i++) {
        if (*i==c) {
            return (*i);
        }
    } 
    push_back(c);
    if ( size() > 100)
        pop_front();
    
    return (*this)(n1,n2); // recursion intended, element does exist now. 
}



