/***************************************************************************
                          knetdumpdoc.cpp  -  description                              
                             -------------------                                         
    begin                : Mon Mar 29 14:39:40 CEST 1999
                                           
    copyright            : (C) 1999 by Norbert Weuster                         
    email                : weuster@uni-duisburg.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "knetdump.h"
#include <knetdumpdoc.h>

KnetdumpDoc::KnetdumpDoc(QObject *parent, const char *filename) : QObject(parent) {
    sniffstart=TRUE; 
    
    readMyOptions();
    
    // setup timer for inactive nodes
    inactiveTimer = new QTimer(this);
    connect( inactiveTimer, SIGNAL(timeout()), SLOT(cleaner()) );
}

KnetdumpDoc::~KnetdumpDoc() {
    resetDevices();
    delete (inactiveTimer);
}

void KnetdumpDoc::saveMyOptions() {
  //KnetdumpApp *win= (KnetdumpApp *) parent();
  
    KConfig *config = kapp->getConfig();

    config->setGroup("Filter");
    config->writeEntry("IEEE802", recIEEE802 );
    config->writeEntry("ARP", recARP);
    config->writeEntry("IP", recIP);
    config->writeEntry("ICMP", recICMP);
    config->writeEntry("TCP", recTCP);
    config->writeEntry("UDP", recUDP);
    config->writeEntry("Port", (u_int) recPort);
    config->writeEntry("X-Server", recX);
    config->writeEntry("Interfaces", activeIf);
    config->setGroup("Timings&Settings");
    config->writeEntry("eth_promisc", promisc);
    config->writeEntry("asciiOutput", asciiOutput);
    config->writeEntry("update timer", updateTime);
    config->writeEntry("inactive timer", inactiveTime);
    config->writeEntry("number of packets", (u_int) buffercount);
    config->writeEntry("resolve hosnames", hostResolve);
    config->writeEntry("filter new Nodes", nodeSelect);
    
    config->sync();
}

void KnetdumpDoc::readMyOptions() {
    //
    // read the config file entries
    //
    KConfig *config = kapp->getConfig();
      
    config->setGroup("Filter");
    recIEEE802= config->readBoolEntry("IEEE802", true);
    recARP    = config->readBoolEntry("ARP", true);
    recIP     = config->readBoolEntry("IP", true);
    recICMP   = config->readBoolEntry("ICMP", true);
    recTCP    = config->readBoolEntry("TCP", true);
    recUDP    = config->readBoolEntry("UDP", true);
    recPort   = config->readUnsignedNumEntry("Port", 0);
    recX      = config->readBoolEntry("X-Server", false);
    activeIf  = config->readEntry("Interfaces", "" );
    config->setGroup("Timings&Settings");
    asciiOutput  = config->readBoolEntry("asciiOutput", false);
    updateTime   = config->readNumEntry("update timer", UPDATETIME);
    inactiveTime = config->readNumEntry("inactive timer", INACTIVETIME);
    buffercount  = config->readUnsignedNumEntry("number of packets", PACKETMAX);
    hostResolve  = config->readBoolEntry("resolve hosnames", true);
    nodeSelect   = config->readBoolEntry("filter new Nodes", true);
    promisc      = config->readBoolEntry("eth_promisc", true);
}

void KnetdumpDoc::start() {  
    //
    // starts the sniffing process
    //
    fdsock=-1;

    // register all devices
    registerDevices();

    // open main socket
    fdsock= socket(PF_INET, SOCK_PACKET, htons(ETH_P_ALL));
    if (fdsock <0) {
        cerr << " Error, couldn't open socket \n";
        stop();
	return;
    }

    emitter= false;  // false == receiver ; true == emitter
    // initiate SocketNotifier and set Signal/Slot
    sn = new QSocketNotifier( fdsock, QSocketNotifier::Read, this );
    sn->setEnabled(false);
    QObject::connect( sn, SIGNAL(activated(int)), this, SLOT(packetNotified()) );

    // set promisc
    setPromisc(promisc);
    if (sniffstart)
        proceed();  // start QSocketNotifier and connect the signal
    else
        pause();
}

void KnetdumpDoc::pause() {
    //
    // pauses sniffing
    // 

    // stop the inactiveTimer
    inactiveTimer->stop();

    // stop the QSocketNotifier
    sn->setEnabled(false);  

    sniffstart=false;
    
    //emit pauseDoc to inactivate toolbar_button
    emit pausedDoc();
}

void KnetdumpDoc::proceed() {
    //
    // proceeds sniffing
    //

    // emit docStarted(); to inactivate toolbar_button
    sn->setEnabled(true);
    // activate single-shot
    inactiveTimer->start( inactiveTime, TRUE );
    sniffstart=true;
    emit startedDoc();
}

void KnetdumpDoc::stop() {
    //
    // stops sniffing and closes the socket
    //
    inactiveTimer->stop();
    //inform other classes
    emit stoppedDoc();
    // delete SocketNotifier and set him to 0
    delete(sn);
    sn=0;  // so delPacket() and ... can ask for it
    // close main socket
    close(fdsock);
}


void KnetdumpDoc::setPromisc(bool value) {
    //
    // set the promisc mode for eth0 and eth1
    // 
    struct ifreq ifr;
    int fdsocket=-1;
    QString eth;

    promisc= value;
    // open socket
    fdsocket= socket(PF_INET, SOCK_PACKET, htons(ETH_P_ALL));
    if (fdsocket <0) {
        cerr << " Error, couldn't open socket \n";
        stop();
	return;
    }
    // set the interfaces eth0 and eth1 to promisc==value
    for (int i=0; i<2 ; i++) { 
        eth.sprintf("eth%i",i);
        if(devIsPresent(eth)) { 
            // set struct ifr to 0
            memset(&ifr, 0, sizeof(ifr));
            // specify the device
            strcpy(ifr.ifr_name, eth);
            // get all interface flags
            // SIOCGIFFLAGS== I/O Controll: Get InterFace FLAGS
            if (ioctl(fdsocket, SIOCGIFFLAGS, &ifr) < 0 ) {
                cerr << "SIOCGIFFLAGS: can't get flags from device "<< eth << endl ;
                stop();
                return;
            }
            if (promisc) 
                // set promisc flag
                ifr.ifr_flags |= IFF_PROMISC;
            else
                // unset promisc flag
                ifr.ifr_flags &= ~IFF_PROMISC;
            // set the changed interface flags (here it's only flag:IFF_PROMISC)
            // SIOCSIFFLAGS== I/O Controll: Set InterFace FLAGS
            if (ioctl(fdsocket, SIOCSIFFLAGS, &ifr) < 0 ) {
                cerr << "SIOCSIFFLAGS: can't set flags to device " << eth << endl;
                stop();
                return;
            }   	
        }
    }
    close(fdsocket);
}

void KnetdumpDoc::packetNotified() {
    //
    // QSocketNotifier activate this member-fkt
    //
    char buffer[MAX_MTU];   // buffer for receiving from socket
    struct sockaddr fromaddr;      // sockaddr of received packet
    int byterecv;                  // bytes of read packet
    // int err; 

    // other classes will have to emit signal, if they finished work
    // routine only for saveness;
    if (emitter) {
        cerr << "sorry, packets comming in too fast, skipping this one" << endl;
        cerr << "perhaps, you should set a filter on packets\n";
        emitter = false;
        return ;
    }
    
    // just read the packet from the socket
    readPacket(buffer, &fromaddr, byterecv);
    
    // do not store packets under a min of 42 bytes and warn!
    if ( byterecv >= 20 ) {       
	struct packet pack={"",0,0,0,0,0};

	switch (storePacket(pack, buffer, fromaddr, byterecv)) {
	case 40: // device isn't set to active
	case 50: // filter rules:
	case 51:
	case 52:
	case 53: // ** for future use ** //
	case 54: 
	case 55:
	    // just do nothing:   
	    break;
	case 0:  // normal
	default:
	    // insert pack into list<packet>, packetList
	    while ( packetList.size() > buffercount ) { 
		delete (packetList.begin()->pbuffer);
		packetList.pop_front();
	    }
	    packetList.push_back(pack);
	    pPacket= --packetList.end();  // set pPacket to the last element
	    
	    emitter = true;
	    // inform view about a new packet
	    emit viewPacket();
	    break;
	}
    }
    else
        cerr << "only " << byterecv << " bytes received, not stored!" << endl ;
}

void KnetdumpDoc::registerDevices() {
    //
    // registers all present devices
    //

    // register unsigned short int offset=0; // at the moment not used
    register int fdsocket1 = -1, fdsocket2;
    register struct ifreq *ifrpointer, *ifrendpointer, *ifnext;
    // struct sockaddr sa; // in combination with bind()
    struct ifconf ifc;
    struct ifreq ifreqbuf[16], ifr;
    struct ifDevices ifDev={""};

    // only necessary if registerDevices is called twice
    if (!ifDevList.empty()) 
        resetDevices();  
    
    // clear all old lists
    ifDevList.clear();
    broadcasts.clear();

    // open socket
    fdsocket1 = socket(AF_INET, SOCK_DGRAM, 0);
    if (fdsocket1 < 0) {
        cerr << "socket: ";
        stop();
        return;
    }
    ifc.ifc_len = sizeof ifreqbuf;
    ifc.ifc_buf = (caddr_t)ifreqbuf;
    
    memset((char *)ifreqbuf, 0, sizeof(ifreqbuf));
    if (ioctl(fdsocket1, SIOCGIFCONF, (char *)&ifc) < 0 || (unsigned int)ifc.ifc_len < (unsigned int) sizeof(struct ifreq)) {
        cerr << "SIOCGIFCONF: can't get ifconfig" << endl;
        close(fdsocket1);
        stop();
        return ;
    }
    ifrendpointer = (struct ifreq *)((char *)ifreqbuf + ifc.ifc_len);

    // walk through all net-devices
    for (ifrpointer = ifreqbuf ; ifrpointer < ifrendpointer; ifrpointer = ifnext) {    
#ifdef HAVE_SOCKADDR_SA_LEN
        n = ifrpointer->ifr_addr.sa_len + sizeof(ifrpointer->ifr_name);
        if (n < sizeof(*ifrpointer))
            ifnext = ifrpointer + 1;
        else
            ifnext = (struct ifreq *)((char *)ifrpointer + n);
        if (ifrpointer->ifr_addr.sa_family != AF_INET)
            continue;
#else
        ifnext = ifrpointer + 1;
#endif
        memset(&ifr, 0, sizeof(ifr));
        strncpy(ifr.ifr_name, ifrpointer->ifr_name, sizeof(ifr.ifr_name));
        if (ioctl(fdsocket1, SIOCGIFFLAGS, (char *)&ifr) < 0) {
            if (errno == ENXIO)
                continue;
            cerr << "SIOCGIFFLAGS: can't get flags from device: " << ifr.ifr_name << endl;
            close(fdsocket1);
            return ;
        }

        /* Must be up */ 
        if ((ifr.ifr_flags & IFF_UP) == 0 )  
            // to exlude the loopback-device 
            // if ((ifr.ifr_flags & IFF_UP) == 0 || (ifr.ifr_flags & IFF_LOOPBACK)  )
            continue;
        
        // added for multiple devices ; thanks to Joseph A. Perrin 
        if (strchr(ifr.ifr_name,':')) {
            cerr << "the device " << ifr.ifr_name << " seems to be a multiple device, skipping\n";
            continue;
        }
	// skip the dummy device, thanks to Stephan Borchert
        if (strstr(ifr.ifr_name,"dummy")) {
            continue;
        }
        // copy ifname to ifDev
        ifDev.ifName = ifrpointer->ifr_name;
        // copy ifflags to ifDev
        ifDev.saved_ifr = ifr;

        // some _old_ Linux kernel bugs, if sa_family isn't set 
        // ifr.ifr_addr.sa_family = AF_INET;

        // get the broadcast address
        memset(&ifr, 0, sizeof(ifr));
        strncpy(ifr.ifr_name, ifrpointer->ifr_name, sizeof(ifr.ifr_name));
        if (ioctl(fdsocket1, SIOCGIFBRDADDR, (char *)&ifr) < 0) {
            cerr << "SIOCGIFBRDADDR: can't get Broadcast address from device: " << ifr.ifr_name << endl;
            close(fdsocket1);
            return ;
        }
        ifDev.broadcast = ntohl(((struct sockaddr_in *) &ifr.ifr_broadaddr)->sin_addr.s_addr);
        if (ifDev.broadcast) 
            broadcasts.push_back(ifDev.broadcast);

        // get the netmask
        memset(&ifr, 0, sizeof(ifr));
        strncpy(ifr.ifr_name, ifrpointer->ifr_name, sizeof(ifr.ifr_name));
        if (ioctl(fdsocket1, SIOCGIFNETMASK, (char *)&ifr) < 0) {
            cerr << "SIOCGIFNETMASK: can't get Netmask from device: " << ifr.ifr_name << endl;
            close(fdsocket1);
            return ;
        }
        ifDev.netmask = ntohl(((struct sockaddr_in *) &ifr.ifr_netmask)->sin_addr.s_addr);
        
        /*
        // get the interface address
        strncpy(ifr.ifr_name, ifrpointer->ifr_name, sizeof(ifr.ifr_name));
        if (ioctl(fdsocket1, SIOCGIFADDR, (char *)&ifr) < 0) {
            cerr << "SIOCGIFADDR: can't get interface adress from device: " << ifr.ifr_name << endl;
            close(fdsocket1);
            return ;
        }    
        ifDev.hostaddr = ntohl(((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr);
        */

        // get link layer type
        // do the rest with a new socket and SOCK_PACKET
        fdsocket2 = -1;
        fdsocket2 = socket(PF_INET, SOCK_PACKET, htons(ETH_P_ALL));
        if (fdsocket2 < 0) {
            cerr << "socket: can't open socket" << endl;
            close(fdsocket1);
            stop();
            return;
        }                   

        /* Bind to the interface name */  // at the moment, not used
        // memset(&sa, 0, sizeof(sa));
        // sa.sa_family = AF_INET;
        // strncpy(sa.sa_data, ifrpointer->ifr_name, sizeof(sa.sa_data));
        // any hints, for what it's good for ??
        // if (bind(fdsocket2, &sa, (ksize_t)sizeof(sa))) {
        // cerr << "bind: " << ifrpointer->ifr_name <<"Error" ;
        // stop();
        // return;
        // }
        
        // Get InterFace HardWare ADDRess
        memset(&ifr, 0, sizeof(ifr));
        strncpy(ifr.ifr_name, ifrpointer->ifr_name, sizeof(ifr.ifr_name)); 
        if (ioctl(fdsocket2, SIOCGIFHWADDR, &ifr) < 0 ) {    // Get InterFace HardWare ADDRess
            cerr << "SIOCGIFHWADDR: can't get Hardware address from device: "<< ifr.ifr_name << endl;
            close(fdsocket1);
            close(fdsocket2);
            stop();
            return;
        }                 
        ifDev.linkLayer= ifr.ifr_hwaddr.sa_family;  // set i.e. ARPHDR_ETHER

	// workaround:
	// normally the linklayer for token ring will be set to ARPHRD_IEEE802
	// I set it to ARPHRD_PRONET (even if this is wrong), so that I can differ 
	// between IEEE 802.2 and IEEE 802.5
	if ( 0 == ifDev.ifName.find("tr")) {
	    ifDev.linkLayer=ARPHRD_PRONET;
	}

        /* get interface MTU */   // at the moment, not used
        // memset(&ifr, 0, sizeof(ifr));
        // strncpy(ifr.ifr_name, ifrpointer->ifr_name, sizeof(ifr.ifr_name));
        // if (ioctl(fdsocket2, SIOCGIFMTU, &ifr) < 0 ) {
        // cerr << "SIOCGIFMTU: cat't get MTU from device: " << ifr.ifr_name << endl;
        // close(fdsocket1);
        // close(fdsocket2);
        // return ;
        // }
	
	// set all counters to NULL
	// 	ifDev.pcounter.arp=0;
	// 	ifDev.pcounter.rarp=0;
	// 	ifDev.pcounter.ip=0;
	// 	ifDev.pcounter.tcp=0;
	// 	ifDev.pcounter.udp=0;
	// 	ifDev.pcounter.icmp=0;
	// 	ifDev.pcounter.otherip=0;
	// 	ifDev.pcounter.badip=0;
	// 	ifDev.pcounter.nonip=0;
	// 	ifDev.pcounter.ipx=0;
	// 	ifDev.pcounter.apple=0;
	// 	ifDev.pcounter.netbios=0;
	// 
	// 	ifDev.bcounter.arp=0;
	//  ...       ^^^^^^^^
	// 	ifDev.bcounter.netbios=0;

        // and finaly add it to ifDeviceList
        ifDevList.push_back(ifDev);
        close(fdsocket2);

	// setup a statistic structure,
	// delete this structure which delete struct (not provided, yes)
	setupStatsstruct(ifDev.ifName);
    }
	close(fdsocket1);

}

void KnetdumpDoc::resetDevices() {
    //
    // resets all devices by flags, e.g. promisc flag
    //
    register int fd;
    struct ifreq ifr;
    
    // open socket
    fd = socket(PF_INET, SOCK_PACKET, htons(0x0003));
    if (fd < 0) {
	cerr << "linux socket: can't open socket\n";
        return ;
    }

    // restore ifr foreach device
    for(LI i= ifDevList.begin(); i != ifDevList.end(); i++) {
        memset(&ifr, 0, sizeof(ifr));
        strncpy(ifr.ifr_name, i->ifName.c_str(), sizeof(ifr.ifr_name));
        if (ioctl(fd, SIOCSIFFLAGS, (char *) &(i->saved_ifr)) < 0) {
            if (errno == ENXIO)
                continue;
            cerr << "SIOCGIFFLAGS: " <<  ifr.ifr_name;
            close(fd);
            return ;
        }
    }
    close(fd);    
}

void KnetdumpDoc::readPacket(char *buf, struct sockaddr *fromaddr, int &byterecv) {
    ksize_t fromlen;
    byterecv = 0;
    
    fromlen = sizeof(struct sockaddr);
    do {      
        byterecv = recvfrom(fdsock, buf, MAX_MTU, 0,fromaddr, &fromlen);
    } while (byterecv < 0 && errno == EINTR);
}

int KnetdumpDoc::storePacket(struct packet &pack, char *buffer, struct sockaddr from, int byterecv) {
    // 
    // reads packet from device, filters packets and store them
    //

    LI pIfDev;   // iterator to the actual receiving device
    const struct iphdr *ip=0;
    struct iphdr ip_hlp={0};
    const struct arphdr *arp=0;
    const struct ether802hdr *pether=0;
    const struct trh_hdr *tokenring=0;
    const struct trllc *tokenllc=0;
    const struct ipxhdr *ipx=0;
    const struct netbioshdr *netbios=0;
    list<stats>::iterator statsI;

    int bytesRemain;

    u_int16_t ipcsum=0, ipck=0;
    QString daddr,saddr;
    const char *cp;

    // receive from which device? if it's not active exit()
    if (-1 == activeIf.find(from.sa_data,0,false))   
        return (40);

    // save buffer to struct
    register char *pp = new char[byterecv];
    if (!memcpy(pp, buffer, byterecv))
        cerr << "memcpy: couldn't copy the buffer" << endl ;
    pack.pbuffer=pp;
    cp = pack.pbuffer;
    pack.byterecv = byterecv;
    bytesRemain= byterecv;

    // the appropriate receiving interface is stored in from.sa_data 
    // so let pIfDev point to the list<ifDevice> with interface name from.sa_data
    pIfDev = find_if(ifDevList.begin(), ifDevList.end(), ifDevices_eq(from.sa_data));
    if (pIfDev == ifDevList.end()) {
        cerr << "Strange error: you got packets from device: " << from.sa_data << ", but this device doesn't exist" << endl ;
        stop();
        return(10);
    }
    pack.ifName = from.sa_data;

    for(statsI= statsList.begin(); statsI != statsList.end() && statsI->ifName!=from.sa_data ; statsI++) { ;}
    if (statsI == statsList.end()) {
	cerr << "strange error: the interface = " << from.sa_data << " has no statslist!" << endl;	
    }

    statsI->pcounter.packet++;
    statsI->bcounter.packet+=bytesRemain;

    // set local link layer
    switch (pIfDev->linkLayer) {
    case ARPHRD_ETHER:
    case ARPHRD_METRICOM:
    case ARPHRD_IEEE802:
	pack.pLinkLayer = cp;
	// ARPHRD_IEEE802 and ARPHRD_ETHER can coexist, so devide them by check 
	bytesRemain=ntohs(((struct ether_header *) cp )->ether_type);
	// *** be carefull, bytesRemain will be only valid, if the next if-clause is true!
	if (bytesRemain <= ETHERMTU) { 
	    // we have an IEEE802.2 header
	    pether=(const struct ether802hdr *) cp;
	    pack.netwProtocol=0;
	    if (pether->dsap == EXTENDED_SAP && pether->ssap == EXTENDED_SAP && pether->cntl == UI_CMD) { 
		// using SNAP
		pack.netwProtocol = ntohs( ((struct ether802hdr *) cp )->ether_type );
		// skip the full ether802hdr (ether_header, llc and snap)
		cp += sizeof(struct ether802hdr);
		bytesRemain-= sizeof(struct ether802hdr);
	    }
	    else {
		// skip only the ether_header and the llc, no snap following
		cp += sizeof(struct ether_header) + 3 * sizeof(char);
		bytesRemain -= ( 3 * sizeof(u_int8_t) + sizeof(struct ether_header) ) ;
		
		// test, what it is?
		if (pether->dsap == 0xe0 && pether->ssap == 0xe0) {
		    // ipx, skip the llc
		    pack.netwProtocol= ETH_P_IPX;
		}
		if (pether->dsap == 0xf0 && pether->ssap == 0xf0) {
		    // netbios, skip the llc
		    pack.netwProtocol= 0xf0;  // no assigned number found in rfc1700
		}
		if (pether->dsap == 6 && pether->ssap == 6) {
		    // IP without SNAP, skip the llc
		    cerr << "IP should be used with SNAP\n";
		    pack.netwProtocol=ETH_P_IP;
		}
	    }
	}
	else {
	    // we have an ARPHRD_ETHER header, so reset bytesRemain!
	    bytesRemain=byterecv;
	    pack.netwProtocol = ntohs( ((struct ether_header *) cp )->ether_type );
	    cp += sizeof(struct ether_header);
	    bytesRemain-= sizeof(struct ether_header);
	}
	pack.pNetwLayer = cp;
	break;
    case ARPHRD_PRONET:  // this isn't really PROnet, it's just token ring. see also
      pack.pLinkLayer = cp;
      tokenring=(const struct trh_hdr *) cp;
      if (tokenring->saddr[0] & TR_RII ) {
	// Routing Information Indicator is ON, RC Routing Control follows
	// the lenght of trh_hdr= (00011111 00000000 & rcf) + ETHLEN
	register int rii_len=( TR_RCF_LEN_MASK & ntohs(tokenring->rcf))>>8;
	tokenllc=(const struct trllc *) (cp + rii_len + ETH_HLEN);
	bytesRemain-= (rii_len + ETH_HLEN);
      }
      else {
	tokenllc=(const struct trllc *) (cp + ETH_HLEN);
	bytesRemain-= ETH_HLEN;
      }
      
      // now working on the llc-header
      cp= (char *) tokenllc;
      if (tokenllc->dsap == EXTENDED_SAP && tokenllc->ssap == EXTENDED_SAP && tokenllc->llc == UI_CMD) { 
	// using SNAP
	pack.netwProtocol = ntohs( tokenllc->ethertype );
	// skip llc and SNAP
	cp += sizeof(struct trllc);
	bytesRemain-= sizeof(struct trllc);
      }
      else {
	// LLC without SNAP:
	cp += 3 * sizeof(char);
	bytesRemain -= 3 * sizeof(u_int8_t) ;
	
	// test, what is it?
	if (tokenllc->dsap == 0xe0 && tokenllc->ssap == 0xe0) {
	  // ipx
	  pack.netwProtocol= ETH_P_IPX;
	}
	if (tokenllc->dsap == 0xf0 && tokenllc->ssap == 0xf0) {
	  // netbios
	  pack.netwProtocol= 0xf0;  // no assigned number found in rfc1700
	}
	if (tokenllc->dsap == 6 && tokenllc->ssap == 6) {
	  // IP without SNAP
	  cerr << "IP should be used with SNAP\n";
	  pack.netwProtocol=ETH_P_IP;
	}
      }
      pack.pNetwLayer = cp;
      break;
    case ARPHRD_SLIP:          // Serial Line IP 
    case ARPHRD_CSLIP:
    case ARPHRD_SLIP6:
    case ARPHRD_CSLIP6:
      // offset von 16 ???
      pack.netwProtocol= ETHERTYPE_IP;   // only IP possible, s. above
      pack.pLinkLayer = cp;
      pack.pNetwLayer = cp;
      break;
    case ARPHRD_PPP:    
        pack.pLinkLayer= cp;
        // unfortunately ppp uses another protocol id than ether_type
	//if (ntohs( ((struct ppphdr *) cp )->protocol ) == 0x0021)
        pack.netwProtocol = ETHERTYPE_IP;
        // else  pack.netwProtocol = 0;
        // cp += sizeof(struct ppphdr);	
 	pack.pNetwLayer =  cp;
	break;
    case ARPHRD_LOOPBACK:     // the link layer is like ether_header, but the addr are zerod
        pack.pLinkLayer=  cp;   // just set it, like the ether_header
        pack.netwProtocol = ntohs( ((struct ether_header *) cp )->ether_type );
 	cp += sizeof(struct ether_header);
 	pack.pNetwLayer = cp;
	break;
#ifdef ARPHRD_FDDI
    case ARPHRD_FDDI:
        // offset of 21?
#endif
    case ARPHRD_EETHER:
    case ARPHRD_ARCNET:
    case ARPHRD_AX25:
    case ARPHRD_CHAOS:
    default:
 	// not implemented
	cerr << "unknown physical layer type: " << hex << pIfDev->linkLayer;
        return (11);
 	break;
    }    
    
    switch (pack.netwProtocol) {  // s. /usr/include/linux/if_ether.h 
    case  ETHERTYPE_IP:
        // set the pointer to the Transport Layer
        statsI->pcounter.ip++;
	ip = (struct iphdr *) pack.pNetwLayer;
        // checksum check
	ip_hlp= (struct iphdr ) *ip;
	ipcsum=ip_hlp.check;
        ip_hlp.check=0;
        ipck = in_cksum((u_short *) &ip_hlp, ip_hlp.ihl *4);
        if ((ipcsum !=ipck) || (!ipck)) {
            statsI->pcounter.badip++;
	    statsI->bcounter.badip+=bytesRemain;
	}
        else {
            if ( (ntohs(ip->frag_off) & 0x1fff) == 0) {
                pack.pTranspLayer = ((char *) pack.pNetwLayer) + ip->ihl * 4; 
            } 
            else {
		bool fragment;
		// have to defrag here
		fragment = true;
            }
        }
#ifdef DEBUG_OUT
	if (bytesRemain !=  ntohs(ip->tot_len)) {
	    printf("DOC: noticed an different lenght between the received Packets and the IP-length header!\n");
	    printf("DOC: bytesRemain=%u ntohs(op->tot_len)=%u \n",bytesRemain, ntohs(ip->tot_len));
	}
#endif
	bytesRemain = ntohs(ip->tot_len);
	statsI->bcounter.ip+=bytesRemain;
        break;
    case ETHERTYPE_ARP:   // op field == request(1) or reply(2)
      arp = (struct arphdr *) pack.pNetwLayer;
      statsI->pcounter.arp++;
      statsI->bcounter.arp+= sizeof(arphdr);
      break;
    case ETHERTYPE_REVARP:  // op field == reqeust(3) or reply(4)
      arp = (struct arphdr *) pack.pNetwLayer;
      statsI->pcounter.rarp++;
      statsI->bcounter.rarp+= sizeof(arphdr);
      break;
    case ETH_P_IPX:
      statsI->pcounter.ipx++;
      ipx = (struct ipxhdr *) pack.pNetwLayer;
      pack.pTranspLayer = ((char *) pack.pNetwLayer) + sizeof (struct ipxhdr);
      
      // statsI->bcounter.ipx+= ipx->length // might be more exact than ...
      statsI->bcounter.ipx+=bytesRemain;

      //cerr << "someone is using the old-fashioned ipx-protocol!\n";
      break;
    case ETH_P_ATALK:
    case ETH_P_AARP:
      statsI->pcounter.apple++;
      statsI->bcounter.apple+=bytesRemain;
      cerr << "listen, an apple is talking :)\n";
      break;
    case 0xf0:
      statsI->pcounter.netbios++;
      statsI->bcounter.netbios+=bytesRemain;
      netbios = (struct netbioshdr *) pack.pNetwLayer;
      // cerr << "protocol: netbios received\n";
      break;
    case 0:
      break;
    default:
      statsI->pcounter.others++;
      statsI->bcounter.others+=bytesRemain;
      cerr << "sorry, but the protocol type: " << hex << pack.netwProtocol <<" is not implemented yet\n" ;
      break;
    }
      
    // store the node
    // delete storeNewNode(pack);
    // qstring daddr, saddr;
    // storeNode(gethostname_from(from));
    // storeNode(gethostname_to(to));
    storeNode(extractIPaddress(daddr, pack, DESTINATIONADDRESS));
    storeNode(extractIPaddress(saddr, pack, SOURCEADDRESS));

    // filter implementation:
    if (!recIEEE802 && pether)
        return 50;
    if ( (!recARP && arp) || (!recIP && ip) ) 
        return 50;  // filter ip or arp
    if (ip) {
        if ((!recTCP && ip->protocol==IPPROTO_TCP)
            || (!recUDP && ip->protocol==IPPROTO_UDP)
            || (!recICMP && ip->protocol==IPPROTO_ICMP))
            return 50;  // filter tcp or udp or icmp
        if (pack.pTranspLayer && (ip->protocol==IPPROTO_TCP || ip->protocol==IPPROTO_UDP)) {
            register struct udphdr *ptmp= (struct udphdr *) pack.pTranspLayer;
            if (recPort && (recPort!=ntohs(ptmp->source) && recPort!=ntohs(ptmp->dest)))
                return 50;  // filter ports from udp and tcp
            //printf("recX=%i RECXPORT=%i \n",recX, RECXPORT);        	
	    if (!recX
                && ((RECXPORT <= ntohs(ptmp->source) && (RECXPORT+500 > ntohs(ptmp->source))) ||
                    (RECXPORT <= ntohs(ptmp->dest) && (RECXPORT+500 > ntohs(ptmp->dest)))))
                return 50;  // filter the X-Server
        }
    }
    if (arp || ip)
	if (!isNodeSelected(extractIPaddress(daddr, pack, DESTINATIONADDRESS)) && !isNodeSelected(extractIPaddress(saddr, pack, SOURCEADDRESS)))
        // if (!isNodeSelected(gethostname_from(daddr)) && !isNodeSelected(gethostname_to(saddr)))
            return 50;

    // packet accounting:
    if (ip) {
      u_int ipDataLen;
      ipDataLen= ntohs(ip->tot_len) - ip->ihl*4;
      switch (ip->protocol) {
      case IPPROTO_TCP:
	statsI->pcounter.tcp++;
	statsI->bcounter.tcp+=ipDataLen;
	break;
      case IPPROTO_UDP:
	statsI->pcounter.udp++;
	statsI->bcounter.udp+=ipDataLen;
	break;
      case IPPROTO_ICMP:
	statsI->pcounter.icmp++;
	statsI->bcounter.icmp+=ipDataLen;
	break;
      default:
	// either tcp, udp, icmp; -> something else
	statsI->pcounter.otherip++;
	statsI->bcounter.otherip+=ipDataLen;
	break;
      }
    }

    // testausgabe der ersten X bytes
    if (asciiOutput) {
	const u_char *tp= (const u_char *) pack.pLinkLayer;
	register int X=80;
	
	if ( byterecv < X)
	    X=byterecv;
	for ( int i=1 ; i<=X ; i++) {
	    printf("%02x", *tp);
	    tp++;
	    if ( (i % 8) == 0 ) 
		if ( (i % 40) == 0 ) 
		    printf("\n");
		else
		    printf("|");
	    else printf(".");
	    
	}
	printf("\n");
    }
    
    // ok, everthing is set, go back and store the packet in the list
    return 0;
}

list<KnetdumpDoc::ifDevices>::iterator KnetdumpDoc::packetFromDev(const string name) {
    //
    // return the device, which received data
    //
    if (ifDevList.end() == find_if(ifDevList.begin(), ifDevList.end(), ifDevices_eq(name))) {
	cerr << "Strange error: the device: " << name << " of the received packet, wasn't registered" << endl ;
	stop();
	return(0);
    }
    else
	return(find_if(ifDevList.begin(), ifDevList.end(), ifDevices_eq(name)));
}

/*void KnetdumpDoc::storeNewNode() 
{
    //
    // stores a new received Node
    // 
    QString from, to;
   
    // store source/destination ip to nodelist
    storeNode(gethostname_from(from));
    storeNode(gethostname_to(to));
    }*/

void KnetdumpDoc::storeNode(QString node) {
  //
  // stores Node; slot to add a Node, without even receiving to/from that Node.
  //
  list<Node>::iterator ni;
  struct in_addr addr;
  
  // valid IP number?
  if (!node || !inet_aton(node, &addr))
    return;
  
  // store node
  // here, I fixed a symantic bug, which increased hostname lookups,
  // thanks to  Fredrik Soerensson
  if (node.size()) {
    ni=find_if(nodeList.begin(), nodeList.end(), node_eq(node));
    if (ni == nodeList.end()) {
	Node tmpNode(node, hostResolve, nodeSelect);
	nodeList.push_back(tmpNode);
    }
    else 
      ni->resetInactiveCounter();
  }
}
    
void KnetdumpDoc::hasShown() {  // slot
    //
    // acknowledge if the emitted packet was processed
    //
    // ok, received SIGNAL
    emitter = false;
}

QString& KnetdumpDoc::extractIPaddress(QString &addr, struct packet &pPacket, bool source) {
    // 
    // extract the IP name/number of an IP or ARP packet
    // if (source) get_sourceaddr; else get_destinationaddr;
    //
    struct in_addr inaddr;
    addr=0;
    
    // if clean IP-packet
    if (pPacket.netwProtocol == ETHERTYPE_IP) {
        struct iphdr *ipPointer= (iphdr *) pPacket.pNetwLayer;
        if (source)
	    inaddr.s_addr=ipPointer->saddr;
	else
	    inaddr.s_addr=ipPointer->daddr;
        addr = inet_ntoa(inaddr);
    }
    // if (r)arp and IP-numbers in ARP
    if (pPacket.netwProtocol == ETHERTYPE_ARP) {
        struct arphdr *arpPointer= (arphdr *) pPacket.pNetwLayer;
        if (ntohs(arpPointer->ar_pro) == ETHERTYPE_IP) {
            u_int32_t saddr;
            u_char *p= (u_char *) arpPointer;
	    if (source)
		p +=  sizeof(struct arphdr) + arpPointer->ar_hln;
	    else
		p +=  sizeof(struct arphdr) + 2*arpPointer->ar_hln + arpPointer->ar_pln;
	    saddr = *((u_int32_t *) p);
            inaddr.s_addr=saddr;
            addr = inet_ntoa(inaddr);
        }
    }
    return(addr);
} 

QString& KnetdumpDoc::gethostname_from(QString &from, bool fromLocal=false) {
    // 
    // extract the source IP number of an IP or ARP packet
    // 
    extractIPaddress(from, (struct packet &) *pPacket, SOURCEADDRESS);
    return(from);

//   struct in_addr inaddr;
//     from=0;

//     // if clean IP-packet
//     if (pPacket->netwProtocol == ETHERTYPE_IP) {
//         struct iphdr *ipPointer= (iphdr *) pPacket->pNetwLayer;
//         inaddr.s_addr=ipPointer->saddr;
//         from = inet_ntoa(inaddr);
//     }
//     // if (r)arp and IP-numbers in ARP
//     if (pPacket->netwProtocol == ETHERTYPE_ARP) {
//         struct arphdr *arpPointer= (arphdr *) pPacket->pNetwLayer;
//         if (ntohs(arpPointer->ar_pro) == ETHERTYPE_IP) {
//             u_int32_t saddr;
//             u_char *p= (u_char *) arpPointer;
//             p +=  sizeof(struct arphdr) + arpPointer->ar_hln;
//             saddr = *((u_int32_t *) p);
//             inaddr.s_addr=saddr;
//             from = inet_ntoa(inaddr);
//         }
//     }
//     return(from);
}

QString& KnetdumpDoc::gethostname_to(QString &to, bool toLocal=false) {
    // 
    // extract the destination IP number of an IP or ARP packet
    // 
    extractIPaddress(to, (struct packet &) *pPacket, DESTINATIONADDRESS);
    return(to);

//  struct in_addr inaddr;
//     to=0;
 
//     // if clean IP-packet
//     if (pPacket->netwProtocol == ETHERTYPE_IP) {
//         struct iphdr *ipPointer= (iphdr *)pPacket->pNetwLayer;
//         inaddr.s_addr=ipPointer->daddr;
//         to = inet_ntoa(inaddr);
//     }
//     // if (r)arp and IP-numbers in ARP
//     if (pPacket->netwProtocol == ETHERTYPE_ARP) {
//         struct arphdr *arpPointer= (arphdr *) pPacket->pNetwLayer;
//         if (ntohs(arpPointer->ar_pro) == ETHERTYPE_IP) {
//             u_int32_t daddr;
//             u_char *p= (u_char *) arpPointer;
//             p +=  sizeof(struct arphdr) + 2*arpPointer->ar_hln + arpPointer->ar_pln;
//             daddr = *((u_int32_t *) p);
//             inaddr.s_addr=daddr;
//             to = inet_ntoa(inaddr);
//         }
//     }
//     return(to);
}   

const int KnetdumpDoc::byterec() const {
    // returns how many bytes were received
    return (pPacket->byterecv);
}

const short int KnetdumpDoc::getLinkLayer() {
    // returns the linklayer type
    return(packetFromDev(pPacket->ifName)->linkLayer);
}

const u_int16_t KnetdumpDoc::getNetwProtocol() const {
    // returns type of protocol in network layer (IP)
    return (pPacket->netwProtocol);
}

const void* const KnetdumpDoc::getpLinkLayer() const {
    // returns pointer to the data-link layer
    return (pPacket->pLinkLayer);
}

const void* const KnetdumpDoc::getpNetwLayer () const {
    // returns pointer to the network protocol
    return (pPacket->pNetwLayer);
}

const void* const KnetdumpDoc::getpTranspLayer() const {
    // returns pointer to the transport layer
    return (pPacket->pTranspLayer);
}

void KnetdumpDoc::getPacket(int num) {
    //
    // set the packet-iterator to the packet "num" and emit signal
    //
    emitter=true;
    list<packet>::iterator listI;
    int i=0;
    
    if (getPacketNum() < (unsigned int)num) {
        cerr << "packet with number= " << num << " was not received yet!" << endl;
        return;
    }
    for(listI= packetList.begin(); listI != packetList.end() && i!=num; listI++, i++) {
    ;
    }
    if (listI != packetList.end())
        pPacket= listI;
    else {
        cerr << "strange error: packet with number= " << num << " not found!" << endl;
        return;
    }
    
    emit viewOldPacket();
} 

const QString KnetdumpDoc::getNode(unsigned int num) { 
    //
    // returns the node at number "num"
    // node starts with num=0
    list<Node>::iterator ni;
    unsigned int i=0;  
    
    // do we have so much nodes stored?
    if (getNodeNum()<num) {
         cerr << "node with number= " << num << " was not received yet!" << endl;
        return 0;
    }

    // walk through nodes
    for(ni= nodeList.begin(); ni != nodeList.end() && i!=num; ni++, i++) {
        ;
    }
    if (ni != nodeList.end())
        return(ni->name());
    else {
        cerr << "strange error: node with number= " << num << " not found!" << endl;
        return 0;
    }
}

const QString KnetdumpDoc::getNodeDomainName(const QString ipAddr) {
    //
    // extract the FQDN (or alias) from the node-class
    //
    list<Node>::iterator ni;
    
    ni=find_if(nodeList.begin(), nodeList.end(), node_eq(ipAddr));
    if (ni != nodeList.end() && ni->domainName()) {
        return(ni->domainName());
    }
    else
        return(ipAddr);
}

const bool KnetdumpDoc::getPromisc() const{
    // is promisc-flag set?
    return(promisc);
}

// update timer
void KnetdumpDoc::setUpdateTime(int value) {
    updateTime = value;
}
const int KnetdumpDoc::getUpdateTime() const {
    return (updateTime);
}

// inactive node timer
void KnetdumpDoc::setInactiveTime(int value) {
    inactiveTime = value;
}
const int KnetdumpDoc::getInactiveTime() const {
    return (inactiveTime);
}
void KnetdumpDoc::restartInactiveTimer() const {
    inactiveTimer->start( inactiveTime, TRUE );     // single-shot
}

void KnetdumpDoc::setPacketMax(int value) {
    // set the number of packets to store
    buffercount = value;
}

const unsigned short int KnetdumpDoc::getPacketMax() const {
    // return the maximum number of packets to store
    return (buffercount);
}

const unsigned short int KnetdumpDoc::getPacketNum() const { 
    //
    // returns the number of actual stored packets
    //
    // 0 <= number of packets <= getPacketMax() == buffercount
    // buffernum is the number of stored packets
    return (packetList.size()); 
}

const unsigned int KnetdumpDoc::getNodeNum() const {
    // returns the number of actual stored nodes
    return (nodeList.size());
}

const bool KnetdumpDoc::devIsPresent(const char *name) const {
    // returns "true", if device "name" is present in device-list
  if (name)
    return( (bool) count_if(ifDevList.begin(), ifDevList.end(), ifDevices_eq(name)));
  else
    return (false);
}

void KnetdumpDoc::resetInterfaces() {
    // disable all device interfaces for storage
    activeIf = "";
}

void KnetdumpDoc::activeInterfaces(const char *name) {
    // activate device "name" for storage
    activeIf += name;
}

const bool KnetdumpDoc::getInterfaces(const char *name) const {
    // returns "true", if interface "name" is active for storage
    return(-1 != activeIf.find(name,0,false));
}

bool KnetdumpDoc::isStopped() {
    // return "true", if doc (socketnotifier) is stopped
    return (!sniffstart);
}

void KnetdumpDoc::delPackets() {
    //
    // deletes all stored packets
    //
    while ( packetList.size() > 0 ) { 
        delete (packetList.begin()->pbuffer);
        packetList.pop_front();
    }
    // delete all stored Nodes
    nodeList.clear();
    emit deletedPackets();
    if (sn)
        //proceed() may only be called, if sn exists. if KOsiView is running, sn doen't exist!
        proceed(); 
}

void KnetdumpDoc::setFilter(bool rIEEE802, bool rARP, bool rIP, bool rTCP, bool rUDP, bool rICMP, bool rX, int rPort) {
    //
    // set filter
    //
    recIEEE802=rIEEE802;
    recARP=rARP;
    recIP=rIP;
    recTCP=rTCP;
    recUDP=rUDP;
    recICMP=rICMP;
		recX=rX;
    recPort=rPort;
}

/*void KnetdumpDoc::sendPing(const char *hostname) {
    register int sockfd;
    register int i;
    register struct icmphdr *icp;
    u_char sendpack[sizeof(struct icmphdr)];
    struct sockaddr_in target;
    struct protoent *proto;
    
    if (hostname==0)
        return;
    //printf("sending ping to %s\n",hostname);

    if ( (proto= getprotobyname("icmp")) == NULL)
        cerr << "unkown protocol: icmp" << endl;
    if ( (sockfd= socket(AF_INET, SOCK_RAW, proto->p_proto)) <0)
        cerr << "cant create raw socket" << endl;
    
    target.sin_family = AF_INET;
    if ((target.sin_addr.s_addr = inet_addr(hostname)) == INADDR_NONE)
        cerr << "hostname" << hostname << " is not a valid hostname" << endl;
    
    icp = (struct icmphdr *) sendpack;
    icp->type=ICMP_ECHOREPLY;
    icp->code=0;
    icp->checksum=0;
    icp->un.echo.id=0;
    icp->un.echo.sequence=0;
    i=sendto(sockfd, sendpack, sizeof(sendpack), 0, (struct sockaddr *) &target, sizeof(target));
    if (i<0 || i!=sizeof(sendpack))
        cerr << "sendto error" << endl;
}
*/

void KnetdumpDoc::delARPcache() {
    //
    // deletes ARP cache table
    // 
    int sockfd=0;

    // open socket
    if ((sockfd = socket(AF_INET,SOCK_DGRAM,0)) <0) { 
        perror("socket"); 
        exit(-1); 
    } 

    // call extern funktion, to delete cache
    arp_show(sockfd);
}

bool KnetdumpDoc::isNodeLocal(QString ipAddr) {
    //
    // is node local to the broadcast-addresses of localhost?
    //
    // node have to be in numbers-and-dots notation
    //
    struct in_addr addr;

    if (!inet_aton(ipAddr, &addr))
        cerr << "error: the ipAddr " << ipAddr << " was not a valid IP-address\n";

    for(int i=broadcasts.size()-1; i>=0 ; i--) {
        if ((ntohl(addr.s_addr) & ~broadcasts[i]) == 0)
            return(true);
    }
    return(false);
}

bool KnetdumpDoc::isNodeSelected(QString name) {
    // returns true, if node is selected in filter
    list<Node>::iterator ni;
    ni=find_if(nodeList.begin(), nodeList.end(), node_eq(name));
    if (ni != nodeList.end()) {
        return(ni->isSelected());
    }
    else
        cerr << "isNodeSelected: the Node with name " << name << " doesn't exist\n";
}

void KnetdumpDoc::setNodeSelected(QString name, bool value) {
    // set a node as (not)selected 
    list<Node>::iterator ni;
    ni=find_if(nodeList.begin(), nodeList.end(), node_eq(name));
    if (ni != nodeList.end())
        ni->setSelected(value);
}

void KnetdumpDoc::setSelectNode(bool value) { 
    // set nodeSelect
    // 
    // if nodeSelect==value
    //  then all new received nodes will be set selected as nodeSelect==value
    nodeSelect=value;
}
bool KnetdumpDoc::isNodeSelected() { 
    // answers if nodeSelect is set
    return (nodeSelect);
}

void KnetdumpDoc::setHostResolve(bool value) {
    // set hostResolve
    //
    // if hostResolve==true
    // IP-numbers get resolved
    hostResolve=value;
}

bool KnetdumpDoc::isHostResolved() {
    // answers if hostResolve is set
    return (hostResolve);
}

void KnetdumpDoc::setasciiOutput(bool value) {
    // set asciiOutput
    //
    // if asciiOutput == true
    // the network packets will appear on console as ascii-text
    asciiOutput=value;
}

bool KnetdumpDoc::isasciiOutput() {
    // answers if asciiOutput is set
    return (asciiOutput);
}

void KnetdumpDoc::cleaner() {
    //
    // deletes all old and inactive nodes
    //
    register bool del=false;
    list<Node>::iterator ni;

    // delete old nodes from Nodelist
    int i=0;
    ni=nodeList.begin();
    for(ni= nodeList.begin(); ni != nodeList.end(); ni++, i++) {
        ni->decInactiveCounter();
        if (ni->inactiveCounter() < 1) {
            del=true;
            nodeList.erase(ni);
            ni=nodeList.begin();
        }
    }

    // inform other classes 
    if (del) {
        emit cleaned();
    }

    // start the inactiveTimer again;
    inactiveTimer->start( inactiveTime, TRUE );     // single-shot
}

const struct stats KnetdumpDoc::getStatstruct(const string interface) {
    //
    // resets all pcounter
    //
    list<stats>::iterator statsI;
    
    for(statsI= statsList.begin(); statsI != statsList.end() && statsI->ifName!=interface; statsI++) {  ;}
    if (statsI != statsList.end()) {
	return (*statsI);
    }
    else {
        cerr << "getStatstruct: strange error: the interface = " << interface << " was not found!" << endl;
	return ((struct stats) {""});
    }
    
}
/*
void KnetdumpDoc::setStatstruct(struct stats stat) {
    //
    // resets all pcounter
    //
    list<stats>::iterator statsI;
    
    for(statsI= statsList.begin(); statsI != statsList.end() && statsI->ifName!=stat.ifName; statsI++) { ;}
    if (statsI != statsList.end()) {
	statsI->bcounter += stat.bcounter;
	statsI->pcounter += stat.pcounter;
	return ;
    }
    else {
        cerr << "setStatstruct: strange error: the interface = " << stat.ifName << " was not found!" << endl;
        return;
    }
    
}
*/
void KnetdumpDoc::setupStatsstruct(string ifName) {
    //
    // setup a Statsstruct for ifName, if it already exists, exit.
    //
    struct stats stat={""};
    list<stats>::iterator statsI;
    
    for(statsI= statsList.begin(); statsI != statsList.end() && statsI->ifName!=ifName; statsI++) {  ;}
    if (statsI == statsList.end()) {
	stat.ifName=ifName;
	statsList.push_back(stat);
    }
}

void KnetdumpDoc::clearStats() {
    register struct stats nullstats={ ""};
    list<stats>::iterator statsI;
    register string ifName;
    
    for(statsI= statsList.begin(); statsI != statsList.end() ; statsI++) {
	ifName=statsI->ifName;
	*statsI=nullstats;
	statsI->ifName=ifName;
    }
}

////////////////////////////////////////////////////////////////////////////////
// everything for class Node (storage class for single nodes)

KnetdumpDoc::Node::Node (QString nName, bool hostResolve, bool select) : nodeName(nName), inactive_cnt(INACTIVECOUNT), selected(select) {
    struct hostent *phostent=0;
    unsigned long int addr=inet_addr((const char *) nName);
    if (hostResolve)
        phostent= gethostbyaddr((char *) &addr, sizeof (addr), AF_INET);
    if (phostent) 
        nodeDomainName= phostent->h_name;
    else 
        nodeDomainName = nName;
}

KnetdumpDoc::Node::~Node () {
}

const QString KnetdumpDoc::Node::name () const {
    return nodeName;
}

const QString KnetdumpDoc::Node::domainName() const {
    // returns FQDN or ip-address
    return nodeDomainName;
}

KnetdumpDoc::Node::operator QString () const {
    return nodeName;
}

bool KnetdumpDoc::Node::operator== (const Node& rhs) const {
    return nodeName == rhs.nodeName;
}

inline int KnetdumpDoc::Node::inactiveCounter () const {
    return inactive_cnt;
}
   
inline void KnetdumpDoc::Node::resetInactiveCounter () {
    inactive_cnt=INACTIVECOUNT;
}
   
inline void KnetdumpDoc::Node::setSelected(bool value) {
    selected=value;
}

inline bool KnetdumpDoc::Node::isSelected() {
    return (selected);
}

inline void KnetdumpDoc::Node::decInactiveCounter () {
    inactive_cnt--;
}



