/*
 *            klm: A lm_sensors front end for the KDE project
 *
 * $Id: Lm80Voltage.cpp,v 1.1 1999/01/21 20:51:48 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>

#include "Lm80Voltage.h"

#define RANGE_INCREMENT 1
#define RANGE_LOWER_LIMIT 3
#define RANGE_UPPER_LIMIT 15

#define SCALE_INCREMENT 0.001L
#define SCALE_LOWER_LIMIT 0.5
#define SCALE_UPPER_LIMIT 5

Lm80Voltage::Lm80Voltage( 
  int sysc1, 
  int sysc2, 
  int sysc3,
  int sysc4,
  int in0Sysc4,
  double in0DefaultConv,
  double grMin, 
  double grMax,
  char  *defaultName,
  double defaultConvFactor ) : Sensor(sysc1, sysc2, sysc3, sysc4, grMin, grMax)
{
  SensorValue val;
  QString sensorName;
  double scaleFactor;
  
  in0 = new StdVoltage( 
      sysc1, 
      sysc2, 
      sysc3,
      in0Sysc4,
      grMin,
      grMax,
      defaultName,
      in0DefaultConv	);
  
  // attempt to read the sensor config from the application config
  // and write the config to the lm_sensors module. 
  // must be root for the write operation to succeed. If it fails
  // the klm will read existing data from the sensors.
  if(Sensor::getSettings( val, sensorName, scaleFactor ))
  {
    setScale( scaleFactor );		
    write( val );
    setName( sensorName );
  }
  else
  {
    setName( defaultName );
    setScale( defaultConvFactor );
  }
}

Lm80Voltage::~Lm80Voltage()
{
}

void Lm80Voltage::name( QString &name )
{
  name = sensorName;
}

bool Lm80Voltage::setName( QString &name )
{
  sensorName = name;
  return true;
}

double Lm80Voltage::scale()
{
  return convFactor;
}

bool Lm80Voltage::setScale( double scaleFactor )
{
  convFactor = scaleFactor;
  return true;
}

bool Lm80Voltage::save()
{
  QString sensorName;
  SensorValue val;
  
  name( sensorName );
  if( read( val ) )
  {
    return Sensor::saveSettings( 
      val, 
      sensorName, 
      scale() );
  }
  
  return false;
}

bool Lm80Voltage::read( SensorValue &val )
{
  SensorValue in0Val;
  
  if ( in0->read( in0Val ) )
  {	
    if( !Sensor::getVolt( val ) )
    {
      val.min = val.value = val.max = 0;
      return false;
    }
    else
    {	
      // knock off the 100 * factor
      val.min = (val.min/100);
      val.max = (val.max/100);
      val.value = (val.value/100);
      
      // calculate differntial value (should this be done to min, max?)
      val.value = convFactor * (val.value - in0Val.value) + val.value; 
      val.min = convFactor * (val.min - in0Val.min) + val.min; 
      val.max = convFactor * (val.max - in0Val.max) + val.max; 
      
      return true;
    }
  }
  
  return false;
}

bool Lm80Voltage::write( SensorValue &val )
{
  SensorValue in0Val;
  
  if ( !in0->read( in0Val ) )
  {	
    return false;
  }
  
  // do conversion (bet this does not work).
  val.value = (val.value + in0Val.value*convFactor)/(1 + convFactor);
  val.min = (val.min + in0Val.min*convFactor)/(1 + convFactor);
  val.max = (val.max + in0Val.max*convFactor)/(1 + convFactor);
  
  // multiply by 100 for storing in kernel.
  val.min = (val.min / convFactor)*100;
  val.max = (val.max / convFactor)*100;
  val.value = (val.value / convFactor)*100;
  
  return Sensor::writeVolt( val );
}

bool Lm80Voltage::displayDecimals()
{
  return true;
}

bool Lm80Voltage::displayUnits()
{
  return true;
}

void Lm80Voltage::units( QString &unitStr )
{
  unitStr="V";
}

bool Lm80Voltage::increaseGraphRange()
{
  double min;
  double max;
  
  Sensor::graphRange( min, max );
  
  max += RANGE_INCREMENT;
  
  if (max > RANGE_UPPER_LIMIT)
  {
    max = RANGE_UPPER_LIMIT;
  }
  
  return Sensor::setGraphRange( min ,max );
}

bool Lm80Voltage::decreaseGraphRange()
{
  SensorValue val;
  double min;
  double max;
  
  Sensor::graphRange( min, max );
  
  max -= RANGE_INCREMENT;
  
  if (max < RANGE_LOWER_LIMIT)
  {
    max = RANGE_LOWER_LIMIT;
  }
  
  if (!read( val ) )
  {
    return false;
  }
  
  if ( fabs(val.value) > max )
  {
    max = fabs(val.value) + (RANGE_INCREMENT * 2);
  }	
  
  return Sensor::setGraphRange( min ,max );	
}

bool Lm80Voltage::increaseScale()
{
  double sc;
  double sign;
  
  sc = scale();
  
  if (sc<0) 
  {
    sign = -1;
  }
  else
  {
    sign = 1;
  }
  
  sc = fabs(sc);
  sc += SCALE_INCREMENT;
  
  if (sc > SCALE_UPPER_LIMIT)
  {
    sc = SCALE_UPPER_LIMIT;
  }
  
  return setScale( sign * sc );
}

bool Lm80Voltage::decreaseScale()
{
  double sc;
  double sign;
  
  sc = scale();
  
  if (sc<0) 
  {
    sign = -1;
  }
  else
  {
    sign = 1;
  }
  
  sc = fabs(sc);
  sc -= SCALE_INCREMENT;
  
  if (sc < SCALE_LOWER_LIMIT)
  {
    sc = SCALE_LOWER_LIMIT;
  }
  
  return setScale( sign * sc );
}

bool Lm80Voltage::isAlarm()
{
  SensorValue val;
  
  read( val );
  val.min = fabs(val.min);
  val.max = fabs(val.max);
  val.value = fabs(val.value);
  
  return ( (val.value < val.min) || (val.value > val.max) );
}
