/*
 *            klm: A lm_sensors front end for the KDE project
 *
 * $Id: Item.cpp,v 1.49 1999/01/14 17:33:05 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <qframe.h>
#include <qpixmap.h>
#include <qlayout.h>
#include <qpalette.h>
#include <qpainter.h>
#include <qrect.h>
#include <qfont.h>
#include <qbrush.h>
#include <qcolor.h>
#include <qevent.h>
#include <qtooltip.h>
#include <kapp.h>
#include <kiconloader.h>
#include <klocale.h>

#include "Item.h"
#include "VertGraph.h"
#include "Sensors.h"

#define VERTITEM_WIDTH 70
#define VERTITEM_HEIGHT 225


VertItem::VertItem( 
  QWidget *parent, 
  const char *name,
  KStatusBar *sb,
  Sensor *aSensor	) : QFrame( parent, name )
{	
  double grMin;
  double grMax;
  
  sensor = aSensor;	
  sensor->units(s_units);
  drawDecimals = sensor->displayDecimals();
  sensor->graphRange( grMin, grMax );
  status = sb;
  configuring = false;
  
  setFrameStyle(Box | Sunken );
  setBackgroundColor(yellow);
  
  lblValue = new QLabel( this, "lblValue" );
  lblValue->setMaximumHeight(15);
  lblValue->setMinimumHeight(15);
  lblValue->setAlignment( 292 );
  lblValue->setMargin( -1 );
  
  graph = new VertGraph( 
      this, 
      "data_graph", 
      grMin, grMax, 
      sensor->lowAlarmSettable(),
      sensor->highAlarmSettable(),
      sensor->scalable(),
      sensor->isOsHystTemp()); 		
  
  graph->setMinimumWidth( 20 );
  
  graph->setFocusPolicy( QWidget::NoFocus );
  graph->setBackgroundMode( QWidget::PaletteMid );
  graph->setFontPropagation( QWidget::NoChildren );
  graph->setPalettePropagation( QWidget::NoChildren );
  
  connect( 
    graph, 
    SIGNAL( newLimits(double, double) ), 
    this, 
    SLOT( newRange( double, double ) ));
  
  connect(
    graph, SIGNAL( configurationRequest() ),
    this, SLOT( configurationRequest() ));
  
  hideButton = new KButton( this, "hideme" );
  hideButton->hide();
  hideButton->setMinimumWidth(15);
  hideButton->setMaximumWidth(15);
  
  QToolTip::add( 
    hideButton, 
    klocale->translate("turns off this sensor") );
  
  KIconLoader *il = kapp->getIconLoader();
  
  hideButton->setPixmap( il->loadIcon("filedel.xpm") );
  
  connect(
    hideButton, SIGNAL( clicked() ),
    this, SLOT( hideButtonPressed() ) );
  
  lblTitle = new QLabel( this, "lblLabel" );
  lblTitle->setAlignment( 292 );
  lblTitle->setMargin( -1 );
  lblTitle->setMaximumHeight(15);
  lblTitle->setMinimumHeight(15);
  
  editTitle = new QLineEdit( this, "lblLabel" );
  editTitle->hide();
  editTitle->setMaxLength(6);
  editTitle->setMaximumHeight(15);
  editTitle->setMinimumHeight(15);	
  QToolTip::add( 
    editTitle, 
    klocale->translate("Enter a new name for the Sensor.") );
  
  connect(
    editTitle, SIGNAL(returnPressed()),
    this, SLOT(titleEnterPressed()));
  
  scale = new UpDownButton( this, "scale" );
  scale->setMinimumWidth(15);
  scale->setMaximumWidth(15);
  QToolTip::add( 
    scale, 
    klocale->translate("Alters the lm_sensor conversion factor (scale).") );
  
  range  = new UpDownButton( this, "range" );
  range->setMinimumWidth(15);
  range->setMaximumWidth(15);
  QToolTip::add( 
    range, 
    klocale->translate("Alters maximum display value of graph") );
  
  bButton  = new UpDownButton( this, "range" );
  bButton->setMinimumWidth(15);
  bButton->setMaximumWidth(15);
  QToolTip::add( 
    bButton, 
    klocale->translate("adds a constant offset to the sensor reading.") );
  
  connect(
    bButton, SIGNAL( upClicked() ),
    this, SLOT( increaseOffset() ));
  
  connect(
    bButton, SIGNAL( downClicked() ),
    this, SLOT( decreaseOffset() ));
  
  connect(
    range, SIGNAL( upClicked() ),
    this, SLOT( increaseGraphMax() ));
  
  connect(
    range, SIGNAL( downClicked() ),
    this, SLOT( decreaseGraphMax() ));
  
  connect(
    scale, SIGNAL( upClicked() ),
    this, SLOT( increaseScale() ));
  
  connect(
    scale, SIGNAL( downClicked() ),
    this, SLOT( decreaseScale() ));
  
  scale->hide();
  range->hide();
  bButton->hide();
  
  QGridLayout* qtarch_layout_1 = new QGridLayout( this, 3, 1, 5, 5, NULL );
  qtarch_layout_1->addColSpacing( 0, 2 );
  qtarch_layout_1->setColStretch( 0, 1 );
  qtarch_layout_1->addRowSpacing( 0, 0 );
  qtarch_layout_1->setRowStretch( 0, 1 );
  qtarch_layout_1->addWidget( lblValue, 0, 0, 12 );
  qtarch_layout_1->addRowSpacing( 1, 0 );
  qtarch_layout_1->setRowStretch( 1, 100 );
  QBoxLayout* qtarch_layout_1_2_1 = new QBoxLayout( QBoxLayout::LeftToRight, 2, NULL );
  qtarch_layout_1->addLayout( qtarch_layout_1_2_1, 1, 0 );
  qtarch_layout_1_2_1->addStrut( 0 );
  QGridLayout* qtarch_layout_1_2_1_1 = new QGridLayout( 2, 1, 2, NULL );
  qtarch_layout_1_2_1->addLayout( qtarch_layout_1_2_1_1, 1 );
  qtarch_layout_1_2_1_1->addColSpacing( 0, 2 );
  qtarch_layout_1_2_1_1->setColStretch( 0, 1 );
  qtarch_layout_1_2_1_1->addRowSpacing( 0, 0 );
  qtarch_layout_1_2_1_1->setRowStretch( 0, 1 );
  qtarch_layout_1_2_1_1->addWidget( hideButton, 0, 0, 9 );
  qtarch_layout_1_2_1_1->addRowSpacing( 1, 0 );
  qtarch_layout_1_2_1_1->setRowStretch( 1, 1 );
  qtarch_layout_1_2_1_1->addWidget( range, 1, 0, 17 );
  qtarch_layout_1_2_1->addWidget( graph, 100, 36 );
  QGridLayout* qtarch_layout_1_2_1_3 = new QGridLayout( 2, 1, 2, NULL );
  qtarch_layout_1_2_1->addLayout( qtarch_layout_1_2_1_3, 1 );
  qtarch_layout_1_2_1_3->addColSpacing( 0, 2 );
  qtarch_layout_1_2_1_3->setColStretch( 0, 1 );
  qtarch_layout_1_2_1_3->addRowSpacing( 0, 0 );
  qtarch_layout_1_2_1_3->setRowStretch( 0, 1 );
  qtarch_layout_1_2_1_3->addWidget( bButton, 0, 0, 10 );
  qtarch_layout_1_2_1_3->addRowSpacing( 1, 0 );
  qtarch_layout_1_2_1_3->setRowStretch( 1, 1 ); 
  qtarch_layout_1_2_1_3->addWidget( scale, 1, 0, 18 );
  qtarch_layout_1->addRowSpacing( 2, 0 );
  qtarch_layout_1->setRowStretch( 2, 1 );
  qtarch_layout_1->addWidget( lblTitle, 2, 0, 20 );
  qtarch_layout_1->addWidget( editTitle, 2, 0, 20 );
  
  setMinimumSize( 0, 0 );
  setMaximumSize( 32767, 32767 );
  setFocusPolicy( QWidget::NoFocus );
  setBackgroundMode( QWidget::PaletteMid );
  setFontPropagation( QWidget::NoChildren );
  setPalettePropagation( QWidget::NoChildren );	
  setGeometry( 0, 0, VERTITEM_WIDTH, VERTITEM_HEIGHT );
}

VertItem::~VertItem()
{
  if(graph) delete graph;
  if(lblValue) delete lblValue;
  if(lblTitle) delete lblTitle;
  if(editTitle) delete editTitle;
  if(range) delete range;
  if(scale) delete scale;
  if(hideButton) delete hideButton;
  if(bButton) delete bButton;
  
}

void VertItem::plot()
{
  if (!configuring)
  {
    forcePlot();
  }
}


void VertItem::forcePlot( )
{
  SensorValue val;
  QString tmp;
  double grMin;
  double grMax;
  bool sensorAlarm;
  
  sensorAlarm = sensor->isAlarm();
  
  sensor->graphRange( grMin, grMax );
  
  if(!sensor->read( val ))
  {
    QString str;
    
    str.sprintf(
      klocale->translate("Error: read failed"));
    status->changeItem( str.data(), 0 );		
    return;
  }	
  
  if (val.value == LM_NO_DATA )
  {
    val.value = 0;
  }
  
  if (drawDecimals)
  {
    tmp.sprintf("%-.2f%s", val.value, s_units.data());
  }
  else
  {
    tmp.sprintf("%-.0f", val.value);
  }
  
  lblValue->setText(tmp.data());	
  sensor->name(tmp);
  lblTitle->setText(tmp.data());
  
  // auto scale the graphs.
  if ( val.value > grMax )
  {
    grMax = val.value * 1.10;
    sensor->setGraphRange( grMin, grMax );		
  }
  
  if ( val.max > grMax )
  {
    grMax = val.max * 1.10;
    sensor->setGraphRange( grMin, grMax );
  }
  
  graph->plot( grMin, grMax, val.min, val.max, val.value, sensorAlarm );
}

void VertItem::paintEvent( QPaintEvent *e )
{
  QPainter paint;
  QColorGroup cg;
  
  cg = colorGroup();
  
  QPixmap pm(width() - 4, height() - 4);
  
  pm.fill(cg.background() );
  bitBlt(this, 2, 2, &pm, 0, 0, pm.width(), pm.height(), CopyROP);
  
  paint.begin( this );
  drawFrame( &paint );
  paint.end();
}

void VertItem::configurationRequest()
{
  if ( configuring )
  {
    // hide the range and scale buttons.		
    scale->hide();
    range->hide();
    bButton->hide();		
    hideButton->hide();
    editTitle->hide();
    lblTitle->show();
    
    // save the new sensor config immediately.
    sensor->save();
    configuring = false;
  }
  else
  {
    // certain configuration items are only available if 
    // we have root access.		
    lblTitle->hide();
    editTitle->show();
    editTitle->setText( lblTitle->text() );
    range->show();		
    hideButton->show();		
    
    if( sensor->isTemp() )
    {
      bButton->show();
    }
    
    if( sensor->isConfigurable() && sensor->scalable() )
    {
      scale->show();
    }
    
    configuring = true;				
  }
  
  if( sensor->isConfigurable() )
  {
    graph->configureYourself( configuring );
  }
}

void VertItem::newRange( double min, double max )
{
  QString str;
  QString title;
  SensorValue val;
  SensorValue polVal;
  double polarity = 1;
  double grMin;
  double grMax;
  
  sensor->graphRange( grMin, grMax );
  
  sensor->name(title);
  // the bar graph widget is a little ignorant as to what is 
  // really happening in the real world - it only plots 
  // positive values - so we need to determine 
  // the polarity of the sensor before writing back
  // into the kernel.
  if(!sensor->read( polVal ))
  {
    QString str;
    
    str.sprintf(
      klocale->translate("Error: read failed."));
    status->changeItem( str.data(), 0 );		
    return;
  }
  else
  {
    if ( polVal.value < 0 )
    {
      polarity = -1;
    }
  }
  
  //  save the new setting into the kernel.
  val.min = polarity * min;
  val.max = polarity * max;
  val.value = 0; 
  if(!sensor->write( val ))
  {
    QString str;
    
    str.sprintf(
      klocale->translate("Error: write failed."));
    status->changeItem( str.data(), 0 );		
    return;
  }
  
  // set the slider to the value selected by the lm_sensors module
  forcePlot();
  
  if(!sensor->read( val ))
  {
    QString str;
    
    str.sprintf(
      klocale->translate("Error: read failed"));
    status->changeItem( str.data(), 0 );		
    return;
  }	
  
  // report the results to the user	
  if ( sensor->isOsHystTemp() )
  {
    str.sprintf(
      klocale->translate("%s hysteresis zone is from %-.2f%s to %-.2f%s."),
      title.data(),
      val.min,
      s_units.data(),
      val.max,
      s_units.data());		
  }
  else
  {	
    str.sprintf(
      klocale->translate("Accepted %s range is from %-.2f%s to %-.2f%s."),
      title.data(),
      val.min,
      s_units.data(),
      val.max,
      s_units.data());
  }
  
  status->changeItem( str.data(), 0 );
  
}

void VertItem::titleEnterPressed()
{
  QString tmp;
  QString oldName;
  QString str;
  
  // set the sensors name, then read back and display
  // the name.
  sensor->name(oldName);
  sensor->setName( editTitle->text() );
  sensor->name(tmp);
  editTitle->setText(tmp);
  lblTitle->setText(tmp);
  
  str.sprintf( 
    klocale->translate("%s was renamed to %s."),
    oldName.data(),
    tmp.data() );
  
  status->changeItem( str.data(), 0 );		
}

void VertItem::increaseScale()
{
  QString str;
  QString title;
  double grMin;
  double grMax;
  
  sensor->name( title );
  sensor->graphRange( grMin, grMax );
  
  // ask the sensor to alter its scale value
  sensor->increaseScale();
  
  // re-plot the slider as things will have changed with
  // the news scale value.
  forcePlot();
  
  // report the results to the user	
  str.sprintf(
    klocale->translate("%s scale factor set to %-.4f"),
    title.data(),
    sensor->scale() );
  
  status->changeItem( str.data(), 0 );	
}

void VertItem::decreaseScale()
{
  QString str;
  QString title;
  double grMin;
  double grMax;
  
  sensor->name( title );
  sensor->graphRange( grMin, grMax );
  
  // ask the sensor to alter its scale value
  sensor->decreaseScale();
  
  // re-plot the slider as things will have changed with
  // the news scale value.
  forcePlot();
  
  // report the results to the user	
  str.sprintf(
    klocale->translate("%s scale factor set to %-.4f"),
    title.data(),
    sensor->scale() );
  
  status->changeItem( str.data(), 0 );	
}

void VertItem::increaseGraphMax()
{
  QString str;
  QString title;	
  double grMin;
  double grMax;
  
  sensor->name( title );
  sensor->increaseGraphRange();
  sensor->graphRange( grMin, grMax );
  
  // re-plot the slider as things will have changed with
  // the new scale value.
  forcePlot();
  
  // report the results to the user	
  str.sprintf(
    klocale->translate("%s graph ranges from %-.2f to %-.2f"),
    title.data(),
    grMin, grMax	);
  
  status->changeItem( str.data(), 0 );		
}

void VertItem::decreaseGraphMax()
{
  QString str;
  QString title;
  double grMin;
  double grMax;
  
  sensor->name( title );
  sensor->decreaseGraphRange();
  sensor->graphRange( grMin, grMax );
  
  // re-plot the slider as things will have changed with
  // the new scale value. 
  forcePlot();
  
  // report the results to the user	
  str.sprintf(
    klocale->translate("%s graph ranges from %-.2f to %-.2f"),
    title.data(),
    grMin, grMax	);
  
  status->changeItem( str.data(), 0 );		
}

void VertItem::hideButtonPressed()
{
  sensor->setDisabled( true );
  emit( hideMe() );
}

void VertItem::increaseOffset()
{
  QString str;
  QString title;
  
  sensor->name( title );
  sensor->increaseOffset();
  
  // re-plot the slider as things will have changed with
  // the new scale value.
  forcePlot();
  
  // report the results to the user	
  str.sprintf(
    klocale->translate("%s offset set to %-.2f"),
    title.data(),
    sensor->getOffset() );
  
  status->changeItem( str.data(), 0 );			
}

void VertItem::decreaseOffset()
{
  QString str;
  QString title;
  
  sensor->name( title );
  sensor->decreaseOffset();
  
  // re-plot the slider as things will have changed with
  // the new scale value.
  forcePlot();
  
  // report the results to the user	
  str.sprintf(
    klocale->translate("%s offset set to %-.2f"),
    title.data(),
    sensor->getOffset() );
  
  status->changeItem( str.data(), 0 );			
}

void VertItem::enterEvent( QEvent *e )
{
  SensorValue val;
  QString str;
  QString title;
  double grMin;
  double grMax;
  
  sensor->name( title );
  sensor->graphRange( grMin, grMax );
  if(!sensor->read( val ))
  {
    QString str;
    
    str.sprintf(
      klocale->translate("Error: read failed."));
    status->changeItem( str.data(), 0 );		
    return;
  }	
  
  if( sensor->isOsHystTemp() )
  {
    str.sprintf(
      klocale->translate("%s reads %-.2f%s, hysteresis zone is %-.2f%s to %-.2f%s, y axis is %-.2f%s to %-.2f%s, dividor is %-.2f"),
      title.data(),
      val.value, s_units.data(),
      val.min, s_units.data(), val.max, s_units.data(),
      grMin, s_units.data(), grMax, s_units.data(),
      sensor->scale());	
  }
  else
  {
    str.sprintf(
      klocale->translate("%s reads %-.2f%s, normal is %-.2f%s to %-.2f%s, y axis is %-.2f%s to %-.2f%s, dividor is %-.2f"),
      title.data(),
      val.value, s_units.data(),
      val.min, s_units.data(), val.max, s_units.data(),
      grMin, s_units.data(), grMax, s_units.data(),
      sensor->scale());	
  }	
  
  status->changeItem( str.data(), 0 );			
}

void VertItem::mousePressEvent( QMouseEvent *e ) 
{
  switch( e->button() )
  {
    case MidButton: // entered/left config mode 
      configurationRequest();
      break;
    case LeftButton:
    case RightButton:
    default:
      QFrame::mousePressEvent( e );			
      break;
  }
}


#include <Item.moc.cpp>

