// This file is part of kseti,
// a utility to manage SETI@home calculations on Un*x boxes..
//
// Copyright (C) 1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.


#ifndef KSETIDOC_H
#define KSETIDOC_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif 

#include <qobject.h>
#include <qstring.h>
#include <qlist.h>

class KsetiView;
class QFile;
class QTextStream;



/** KsetiDoc provides a document object for a document-view model.
 *
 * The KsetiDoc class provides a document object that can be used in conjunction with the classes
 * KsetiApp and KsetiView to create a document-view model for standard KDE applications based on
 * KApplication and KTMainWindow. Thereby, the document object is created by the KsetiApp instance
 * and contains the document structure with the according methods for manipulation of the document
 * data by KsetiView objects. Also, KsetiDoc contains the methods for serialization of the document
 * data from and to files.
 *
 * @author Jochen Kpper
 * @version 1999/08/24
 */
class KsetiDoc : public QObject
{
    Q_OBJECT

public:

    /** Constructor for the fileclass of the application */
    KsetiDoc( QWidget* parent, const char *name=0 );
    
    /** Destructor for the fileclass of the application */
    ~KsetiDoc();
    
    /** adds a view to the document which represents the document contents. Usually this is your main view. */
    void addView( KsetiView* m_pView );
    
    /** removes a view from the list of currently connected views */
    void removeView( KsetiView* m_pView );
	
    /** sets the modified flag for the document after a modifying action on the view connected to the document.*/
    void setModified( bool modified=true ){
	b_modified = modified;
    }
    
    /** returns if the document is modified or not. Use this to determine if your document needs saving by the user on closing.*/
    bool isModified(){
	return b_modified;
    }
    
    /** deletes the document's contents */
    void deleteContents();
	
    /** initializes the document generally */
    bool newDocument();
	
    /** closes the acutal document */
    void closeDocument();
	
    /** loads the document by filename and format and emits the updateViews() signal */
    bool openDocument( const char* filename, const char* format=0 );
    
    /** sets the path to the file connected with the document */
    void pathName( const char* path_name );
	
    /** returns the pathname of the current document file*/
    const QString& getPathName() const;
	
    /** sets the filename of the document */
    void title( const char* title );

    /** returns the title of the document */
    const QString& getTitle() const;
	

    
    /** @return Progress of SETI@home calculation */
    double progress() {
	return progressvalue;
    }

    /** @return Number of packages finished */
    int results() {
	return resultvalue;
    }

    
    /** @return current CPU-time of this SETI@home calculation */
    const QString& cpu() {
	return cpustring;
    }
	
    /** @return date/time of measurement in UnixTime */
    const QString& date() {
	return datestring;
    }
	
    /** @return User's email address */
    const QString& email() {
	return emailstring;
    }
	
    /** @return estimated overall calculation time of this SETI@home calculation */
    const QString& overall() {
	return overallstring;
    }
	
    /** @return date/time of SETI@home start in UnixTime */
    const QString& user() {
	return userstring;
    }
	
	
public slots:
 	
    /**
     * calls repaint() on all views connected to the document object and is called by the view by which the
     * document has been changed.
     * As this view normally repaints itself, it is excluded from the paintEvent.
     */
    void slotUpdateAllViews( KsetiView* pSender );
 	
 	
protected slots:

    /** Read the progress. */
    void readData();

    /**
     * read a entry from the given file
     *
     * @param stream File to read from.
     * @param key    What are we looking for ? This is a QRegExp pattern.
     * @return  value of the key, a Nullstring if not found.
     */
    QString readEntry( QTextStream& stream, const QString& key );

    
public:	
 	
    /** the list of the views currently connected to the document */
    static QList<KsetiView> *viewList;	


protected:
 	
    /** the modified flag of the current document */
    bool b_modified;

    /** Number of packages finished */
    int resultvalue;
    
    /** calculation time and progress values */
    double cpuvalue, progressvalue;

    /** information strings */
    QString cpustring, datestring, emailstring,
	overallstring, progressstring, userstring;

    /** title and path of the document */
    QString m_title, m_path;

    /** calculation files */
    QFile *stateFile, *userFile, *workFile;
	
    /** read progress timer */
    QTimer *timer;
};



#endif // KSETIDOC_H



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
