// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION that comes with the package.
//


#ifndef KROT_TRANSITION_INLINE_H
#define KROT_TRANSITION_INLINE_H



#include <iostream>
#include <string>
#include <strstream>



inline QNum::QNum()
{
    for(QuantumNumber i = begin; i < end; i = static_cast< QuantumNumber >( i + 1 ) )
	qnum[ i ] = 0;
    return;
}



inline QNum::QNum( const QNum& q )
{
    for( QuantumNumber i = begin; i < end; i = static_cast< QuantumNumber >( i + 1 ) )
	qnum[ i ] = q.qnum[ i ];
    return;
}



inline QNum::QNum( const unsigned short q[ end - begin ] )
{
    for(QuantumNumber i = begin; i < end; i = static_cast< QuantumNumber >( i + 1) )
	qnum[ i ] = q[ i ];
    return;
}



inline QNum::QNum( const string& str )
{
    istrstream in( str.c_str(), str.length() );
    in >> *this;
    return;
}



inline QNum::QNum( const char* buf )
{
    QNum( string( buf ) );
    return;
}



inline QNum::operator string() const
{
    char buffer[8];
    string str;
    for( QuantumNumber i = begin; i < end; i = static_cast< QuantumNumber >( i + 1 ) ) {
	sprintf( buffer, "%4d", qnum[ i ] );
	str += buffer;
    }
    return str;
}



inline QNum& QNum::operator=( const QNum& q )
{
    if( this == &q )
	return *this;
    for( QuantumNumber i = begin; i < end; i = static_cast< QuantumNumber >( i + 1 ) )
	qnum[ i ] = q.qnum[ i ];
    return *this;
}

  

inline bool QNum::operator==( const QNum& q ) const
{
    for( QuantumNumber i = begin; i < end; i = static_cast< QuantumNumber >( i + 1 ) )
	if( qnum[ i ] != q.qnum[ i ] )
	    return false;
    return true;
}



inline bool QNum::operator<( const QNum& q ) const
{
    for( QuantumNumber i = begin; i < end; i = static_cast< QuantumNumber >( i + 1 ) ) {
	if( qnum[ i ] < q[ i ] )
	    return true;
	else if( qnum[ i ] > q[ i ] )
	    return false;
    }
    return false;
}



inline unsigned short QNum::operator[]( const QuantumNumber n ) const
{
    return qnum[ n ];
}



inline unsigned short& QNum::operator[]( const QuantumNumber n )
{
    return qnum[ n ];
}



inline AssignedFrequency::AssignedFrequency()
    : QNum(), freq( 0 )
{
    return;
}



inline AssignedFrequency::AssignedFrequency( const AssignedFrequency& ass )
    : QNum( ass.qnum ), freq( ass.freq )
{
    return;
}



inline AssignedFrequency::AssignedFrequency( const string& str )
{
    istrstream in( str.c_str(), str.length() );
    in >> qnum[ QNum::eJ ] >> qnum[ QNum::eKa ] >> qnum[ QNum::eKc ]
       >> qnum[ QNum::gJ ] >> qnum[ QNum::gKa ] >> qnum[ QNum::gKc ]
       >> freq;
    return;
}


    
inline AssignedFrequency::AssignedFrequency( const char* buf )
{
    AssignedFrequency( string( buf ) );
    return;
}



inline AssignedFrequency::AssignedFrequency( const QNum& qnum )
    : QNum( qnum ), freq( 0 )
{
    return;
}



inline AssignedFrequency::AssignedFrequency( const QNum& qnum, double f )
    : QNum( qnum ), freq( f )
{
    return;
}



inline AssignedFrequency::operator string() const
{
    string str = QNum::operator string();
    char buf[16];
    sprintf( buf, " %.2f", freq );
    str += buf;
    return str;
}



inline bool AssignedFrequency::operator<(const AssignedFrequency& x) const
{
    return ( freq < x.freq );
}



inline double AssignedFrequency::frequency() const
{
    return freq;
}



inline double& AssignedFrequency::frequency()
{
    return freq;
}



inline AssignedIntensity::AssignedIntensity()
    : QNum(), inten( 0 )
{
    return;
}


inline AssignedIntensity::AssignedIntensity( const string& str )
{
    istrstream in( str.c_str(), str.length() );
    in >> static_cast< QNum& >( *this ) >> inten;
    return;
}



inline AssignedIntensity::operator string() const
{
    string str = QNum::operator string();
    char buf[16];
    sprintf( buf, "%.2f", inten );
    str += buf;
    return str;
}



inline double AssignedIntensity::intensity() const
{
    return inten;
}



inline double& AssignedIntensity::intensity()
{
    return inten;
}



inline Transition::Transition()
    : QNum()
{
    freq = inten = 0;
    return;
}



inline Transition::Transition( const Transition& q )
    : QNum( q )
{
    freq = q.freq;
    inten = q.inten;
    return;
}



inline Transition::Transition( const AssignedFrequency& asn )
    : QNum( asn )
{
    freq = asn.frequency();
    inten = 0;
    return;
}



inline Transition::Transition( const AssignedIntensity& ass )
    : QNum( ass )
{
    freq = 0;
    inten = ass.intensity();
    return;
}



inline Transition::Transition(const QNum& qnum)
    : QNum( qnum )
{
    freq = 0;
    inten = 0;
    return;
}



inline Transition::Transition( const char* buf )
{
    sscanf( buf, "%hu %hu %hu %hu %hu %hu %lf %lf",
	    &qnum[ eJ ], &qnum[ eKa ], &qnum[ eKc ],
	    &qnum[ gJ ], &qnum[ gKa ], &qnum[ gKc ],
	    &freq, &inten );
    return;
}



inline Transition::operator string() const
{
    string str = AssignedIntensity::operator string();
    char buf[16];
    sprintf( buf, "%.2f", freq );
    str += buf;
    return str;
}



inline Transition& Transition::operator=(const Transition& q)
{
    // check for object identity
    if(this == &q)
	return *this;

    freq = q.freq;
    inten = q.inten;
    for(QuantumNumber i = begin; i < end; i=static_cast< QuantumNumber >( i + 1 ) )
	qnum[ i ] = q.qnum[ i ];

    return *this;
}



// --- I/O functions ---

inline istream& operator>>( istream& stream, QNum& qnum )
{
    stream >> qnum[ QNum::eJ ] >> qnum[ QNum::eKa ] >> qnum[ QNum::eKc ]
	   >> qnum[ QNum::gJ ] >> qnum[ QNum::gKa ] >> qnum[ QNum::gKc ];
    return stream;
}



inline istream& operator>>( istream& stream, AssignedFrequency& asn )
{
    stream >> static_cast< QNum& >( asn ) >> asn.frequency();
    return stream;
}



inline istream& operator>>( istream& stream, AssignedIntensity& asn )
{
    stream >> static_cast< QNum& >( asn ) >> asn.intensity();
    return stream;
}



inline istream& operator>>( istream& stream, Transition& line )
{
    stream >> static_cast< AssignedIntensity& >( line ) >> line.frequency();
    return stream;
}



inline ostream& operator<<( ostream& stream, const QNum& qnum )
{
    stream << string( qnum );
    return stream;
}



inline ostream& operator<<( ostream& stream, const Transition& line )
{
    stream << string( line );
    return stream;
}




inline QTextStream& operator>>( QTextStream& stream, Transition& line )
{
    stream >> line[ QNum::eJ ] >> line[ QNum::eKa ] >> line[ QNum::eKc ]
	   >> line[ QNum::gJ ] >> line[ QNum::gKa ] >> line[ QNum::gKc ]
	   >> line.intensity() >> line.frequency();
    return stream;
}



#endif
