// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION that comes with the package.
//



#ifndef KROT_TRANSITION_H
#define KROT_TRANSITION_H



#include <iostream>
#include <string>



/**
 * @short Set of quantum numbers.
 *
 * Hold one set of quantum numbers to describe a rotational transition.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class QNum
{
    
public:

    enum QuantumNumber { begin=0, inc=1, eJ=0, eKa=1, eKc=2, gJ=3, gKa=4, gKc=5, end=6, num = end - begin };

    /**
     * Standard constructor; initialize all quantum numbers to zero.
     */
    QNum();

    /**
     * Copy constructor.
     *
     * @param qnum Object to copy.
     */
    QNum( const QNum& qnum );

    /**
     * Contruct an object from an 1 dimensional array containing the
     * quantum numbers.
     *
     * @param qnum One dimensional array containing ( end - begin ) quantum numbers.
     */
    explicit QNum( const unsigned short qnum[ num ] );

    /**
     * Contruct an object from an C string containing all quantum numbers.
     */
    explicit QNum( const char *buf );

    /**
     * Contruct an object from an string containing all quantum numbers.
     */
    explicit QNum( const string& str );

    /**
     * Conversion to const string. Used for output purposes.
     */
    virtual operator string() const;

    /**
     * Assignment
     */
    virtual QNum& operator=( const QNum& rval );

    /**
     * Equality.
     * Checks wether all corresponding quantum nubers are equal.
     *
     * @param rval Object to compare to.
     * @return True on Equality.
     */
    virtual bool operator==( const QNum& rval ) const;

    /**
     * Less than.
     *
     * @param rval Object to compare to.
     * @return True if this is smaller than rval.
     */
    virtual bool operator<( const QNum& rval ) const;

    /**
     * Read single quantum number.
     *
     * @param qn Quantum number to access.
     */
    unsigned short operator[]( const QuantumNumber qn ) const;
    
    /**
     * Write single quantum number.
     *
     * @param qn Quantum number to access.
     */
    unsigned short& operator[]( const QuantumNumber qn );

    
protected:

    unsigned short qnum[ num ];
};



/**
 * @short Assigned frequency.
 *
 * Hold one set of quantum numbers to describe a rotational transition and a frequency this
 * transition occurs at.
 *
 * @short Set of quantum numbers and frequency.
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class AssignedFrequency : virtual public QNum
{

public:

    AssignedFrequency();

    AssignedFrequency( const AssignedFrequency& ass );

    explicit AssignedFrequency( const string& str );

    explicit AssignedFrequency( const char* cstr );

    explicit AssignedFrequency( const QNum& qnum );

    explicit AssignedFrequency( const QNum& qnum, double freq );

    /**
     * Conversion to const string. Used for output purposes.
     */
    virtual operator string() const;

    virtual bool operator<( const AssignedFrequency& rval ) const;

    double frequency() const;

    double& frequency();


protected:

    double freq;
};



/**
 * @short Assigned intensity.
 *
 * Hold one set of quantum numbers to describe a rotational transition and an intensity this
 * transition occurs with.
 *
 * @short Set of quantum numbers and intensity.
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class AssignedIntensity : virtual public QNum
{

public:

    /**
     * Construct an object.
     */
    AssignedIntensity();

    /**
     * Construct an object from a string.#
     *
     * @param str String containing data.
     */
    AssignedIntensity( const string& str);

    /**
     * Construct an object from a C string.
     *
     * @param str C string containing data..
     */
    AssignedIntensity( const char *cstr );

    /**
     * Conversion to const string. Used for output purposes.
     */
    virtual operator string() const;

    /**
     * Intensity.
     *
     * @return Give back a copy of the intensity of this object.
     */
    double intensity() const;

    /**
     * Intensity.
     *
     * @return Give back a (writeable) reference of the intensity of this object.
     */
    double& intensity();

    
protected:

    double inten;
};



/**
 * Hold one set of quantum numbers to describe a rotational transition, a frequency this
 * transition occurs at and a intensity this transition occurs with
 *
 * @short Set of quantum numbers and frequency and intensity.
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class Transition : virtual public AssignedFrequency, virtual public AssignedIntensity
{
    
public:

    Transition();

    Transition( const Transition& trans );

    explicit Transition( const AssignedFrequency& asn );

    explicit Transition( const AssignedIntensity& ass );

    explicit Transition( const string& str );

    explicit Transition( const char *cstr );

    explicit Transition( const QNum& qnum );

    /**
     * Conversion to const string. Used for output purposes.
     */
    virtual operator string() const;

    virtual Transition& operator=( const Transition& rval );
};



// I/O functions

/**
 * Read set of quantum numbers from istream.
 *
 * @param stream Where to read from.
 * @param qnum Put read data in here.
 * @return Give back istream reference, so we can concatenate substatements.
 */
inline istream& operator>>( istream& stream, QNum& qnum );

/**
 * Read set of quantum numbers and frequency from istream.
 *
 * @param stream Where to read from.
 * @param qnum Put read data in here.
 * @return Give back istream reference, so we can concatenate substatements.
 */
inline istream& operator>>( istream& stream, AssignedFrequency& asn );

/**
 * Read set of quantum numbers and intensity from istream.
 *
 * @param stream Where to read from.
 * @param qnum Put read data in here.
 * @return Give back istream reference, so we can concatenate substatements.
 */
inline istream& operator>>( istream& stream, AssignedIntensity& asn );

/**
 * Read Transition from istream
 *
 * @param stream Where to read from.
 * @param line Put read data in here.
 * @return Give back istream reference, so we can concatenate substatements.
 */
inline istream& operator>>( istream& stream, Transition& line );


/**
 * Write Transition to ostream
 *
 * @param stream Where to write to.
 * @param line Data to write.
 * @return Give back ostream reference, so we can concatenate substatements.
 */
inline ostream& operator<<( ostream& stream, Transition& line );



#include "transition_inline.h"



#endif
