// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with the package.



#include <qdialog.h>
#include <qlayout.h> 
#include <qlistbox.h> 
#include <qpushbutton.h> 
#include <qstring.h>

#include <kapp.h>
#include <kbuttonbox.h>
#include <kmsgbox.h> 

#include "dataset.h"
#include "krotData.h"
#include "krot.h"
#include "selectSet.h"



DialogSelectSet::DialogSelectSet( KRotData *krotdata, QWidget *parent, const char *name )
    : QDialog(parent, name, true),
      krotData( krotdata )
{
    KROT_LAUNCH( "Launching DialogSelectSet constructor" );
    setCaption( i18n("KRot [Select set]") );
    QBoxLayout *layout = new QVBoxLayout( this, KROT_DIALOGMARGIN, KROT_DIALOGMARGIN / 2 );
    // *** message ***
    label = new QLabel( i18n( "Please select a DataSet:" ), this );
    label->setFixedHeight( label->sizeHint().height() );
    layout->addWidget( label );    
    // *** listbox ***
    list = new QListBox(this, "Listbox", WStyle_Customize | WStyle_DialogBorder | WStyle_SysMenu | WStyle_Title );
    layout->addWidget( list );
    // *** buttons ***
    KButtonBox *bbox = new KButtonBox(this);
    bbox->addStretch( 5 );
    ok = bbox->addButton( i18n( "Select" ) );
    bbox->addStretch( 10 );
    QPushButton *cancel = bbox->addButton( i18n( "Cancel" ) );
    bbox->addStretch( 10 );
    QPushButton *help   = bbox->addButton( i18n( "Help" ) );
    bbox->addStretch( 5 );
    bbox->layout();
    bbox->setFixedHeight( bbox->sizeHint().height() + 10 );

    layout->addWidget( bbox );
    layout->activate();

    // connect signals
    connect( list, SIGNAL( selected(const char *) ), this, SLOT( select(const char *) ) );
    connect( ok, SIGNAL( clicked() ), this, SLOT( accept() ) );
    connect( cancel, SIGNAL( clicked() ), this, SLOT( reject() ) );
    connect( help, SIGNAL( clicked() ), this, SLOT( help() ) );
    
    refresh();
    setGeometry( x(), y(), bbox->sizeHint().width(), height() );
}



int DialogSelectSet::exec( const QString& msg )
{
    if( msg.isEmpty() )
	label->setText( i18n( "Please select a DataSet:" ) );
    else
	label->setText( msg );
    label->setFixedHeight( label->sizeHint().height() );
    refresh();
    return QDialog::exec();
}



DataSet *DialogSelectSet::getSelection()
{
    return krotData->getDataSet( getSelectionString() );
}



const QString DialogSelectSet::getSelectionString() const
{
    KROT_LAUNCH( "Launching DialogSelectSet::getSelection" );
    return list->text( list->currentItem() );
}



void DialogSelectSet::help() const
{
    kapp->invokeHTMLHelp( "", "" );
}



void DialogSelectSet::select(const char *)
{
    KROT_LAUNCH( "Launching DialogSelectSet::select(const char *)" );
    accept();
}



void DialogSelectSet::refresh()
{
    // insert items for all current sets into list
    list->setAutoUpdate( false );
    list->clear();
    mapSpectrum::const_iterator spec = krotData->spectra().begin();
    while( spec != krotData->spectra().end() ) {
	list->inSort( spec->first.data() );
	spec++;
    }
    mapSimulation::const_iterator sims = krotData->simulations().begin();
    while( sims != krotData->simulations().end() ) {
	list->inSort( sims->first.data() );
	sims++;
    }
    list->setAutoUpdate( true );
    if( list->count() == 0 )
	ok->setEnabled( false );
    else
	ok->setEnabled( true );
}



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
