// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_MAINWINDOW_H
#define KROT_MAINWINDOW_H


#include "calculation.h"
#include "transformations.h"
#include "workarea.h"

#include <kapp.h>
#include <kmsgbox.h>
#include <ktmainwindow.h>

#include <sys/types.h>


class DialogCalculationArnirot;
class DialogFoldSet;
class DialogGeometryTransform;
class DialogImportRawData;
class DialogSelectSet;
class KRotData;
class QTimer;
class WorkArea;



/**
 * Provides the application main window.
 *
 * Contains a menubar, a toolbar, a statusbar,
 * and a @ref WorkArea widget.
 *
 * @short MainWindow.
 * @author Jochen Kpper
 * @version 99/02/03
 */
class MainWindow : public KTMainWindow
{
    Q_OBJECT

private:

    /**
     * Ids of StatusBar items.
     */
    enum StatusBarID { ID_FREQUENCY, ID_MESSAGE };
    
    /**
     * Ids of ToolBar items.
     */
    enum ToolBarID   { ID_AVERAGING };

    /**
     * Time to display/keep temporary messages in the statusbar [ms].
     */
    static const int messageTimout = 7500;
    

public:

    /**
     * Keys for KConfig entries of this class
     */
    static const char config_CurrentDir[] = "CurrentDir";

    /**
     * Constructs the widget.
     *
     * @param name Passed to @ref KTMainWindow constructor.
     */
    MainWindow( const char *name=0 );


public slots:

    /**
     * Display actual mouse frequency in the statusbar area.
     *
     * @param freq Mouse frequency to display.
     */
    void setFreq( const int64_t freq );

    /**
     * Inform user by displaying the message in the statusbar message field.
     * This message will be removed to show the normal statusbar contents after
     * a short period of time.
     *
     * Since KStatusBar was broken in KDE 1.0, we provide our own timer.
     * Guess we will keep it since its much nicer in our case not to overwrite
     * the whole statusbar.
     *
     * @param msg Message to display.
     */
    void setMessage( const QString& msg );


protected slots:

    /**
     * Perform auto-correlation of dataset
     */
    void autocorrelate();

    /**
     * Show parameter input to start arnirot calculation.
     */
    void arnirot();

    /**
     * Called whenever the averaging value in the toolbar-combobox was changed.
     *
     * This routine converts the string to an integer number, sets the combobox
     * content accordingly and emits the signal @ref newAveraging.
     *
     * @param val New value string of the combobox.
     */
    void averaging( const char *val );

    /**
     * Delete a temporary message from the stausbar. This slot and the associated
     * \ref QTimer msgTimer are needed due to a bug of KStatusBar in KDE 1.0.
     *
     * We should be able to use the \ref KStatusBar::setMessage( message, timout )
     * function with later KDE versions. At that point, this function and the
     * msgTimer should be removed.
     */
    void clearMessage();

    /**
     * Delete this project.
     */
    void close();

    /**
     * Delete all projects.
     */
    void closeAll();

    /**
     * @ref KTMainWindow (from which this class is derived) has the nice habbit
     * that it will exit the application when the very last @ref KTMainWindow is
     * destroyed.
     *
     * Here we ask the user, wether he really want to do that. If he cancels the
     * action, the window will remain open.
     */
    void closeEvent( QCloseEvent *event );

    /**
     * Perform cross-correlation of two @see DataSet
     */
    void crosscorrelate();

    /**
     * Open a selection list dialog and let the user delete the selected set.
     * This is all done by the @ref DialogDeleteSet dialog itself.
     */
    void deleteSet();

    /**
     * Calculate the difference of two @see DataSet
     */
    void difference();

    /**
     * Open a selection list dialog and let the user change the selected assignment.
     * This is all done by the @ref DialogEditAssignment dialog itself.
     */
    void editAssignments();

    /**
     * Open a selection list dialog and let the user edit the properties of the
     * selected spectra.
     * This is all done by the @ref DialogEditSpectra dialog itself.
     */
    void editSpectra();

    /**
     * Called when the user selects the Export->Assignments item.
     * Should take the approbiate action to save all stored assignments to disk.
     * Should provide a choice which format to save in -> not implemented yet.
     *
     * If the user selects a file to save assignments to, the signal
     * @ref exportAssignments( const char * ) is emitted.
     */
    void exportAssignments();

    /**
     * Called when the user selects the Export->Lines item.
     * Should take the approbiate action to save one @ref Simulation to disk.
     */
    void exportLines();

    /**
     * Called when the user selects the Export->XY item.
     * Should take the approbiate action to save one set in xy format to disk.
     */
    void exportXY();

    /**
     * Open a selection list dialog and let the user specify the folding parameter
     * for selected simulations.
     * This is all done by the @ref DialogEditSimulations dialog itself.
     */
    void foldSimulations();

    /**
     * Open a selection list dialog and let the user specify the folding parameter
     * for selected simulations.
     * This is all done by the @ref DialogGeometryTransform dialog itself.
     */
    void geomTransform();

    /**
     * Show parameter input to start IAR calculation.
     */
    void iar();

    /**
     * Select filename to import lines from.
     */
    void importLines();

    /**
     * Import xy data, one data point each line ( x y ).
     */
    void importXY();

    /**
     * Import raw jba data.
     */
    void importJba();

    /**
     * Import raw KHiDaq data.
     */
    void importKHiDaq();

    /**
     * Show parameter input to start jarot calculation.
     */
    void jarot();

    /**
     * Create new window.
     */
    void newWindow();

    /**
     * Show preferences dialog
     */
    void preferences();

    /**
     * Save all Infomation in native (ASCII) format.
     *
     * First is the experimental data; format is:
     *
     * Start frequency and frequency separation of two points
     * (both are double format, but are rounded to the nearest integer value
     * right now).
     *
     * Number of data points.
     *
     * Data points.
     *
     * Number of assignments, following the actual assignments.
     *
     * Number of simulation/fit parameter to be stored.
     *
     * Simulation Parameter as defined by the Simulation::writeParameter
     * and Simulation::readParameter format/functions.
     */
    void save();



signals:

    /**
     * Emitted whenever the averaging value was changed inside the toolbar.
     *
     * @param avg New averaging value.
     */
    void newAveraging( const unsigned int avg );

    /**
     * Assignments shall be saved to the file specified.
     *
     * @param filename Full name of file the assignments shall be saved to.
     */
    void exportAssignments( const QString& filename );

    /**
     * Specified set shall be saved to the file specified.
     *
     * @param simname  Name of the @ref Simulation that shall be saved.
     * @param filename Full name of file the set shall be saved to.
     */
    void exportLines( const QString& simname, const QString& filename );

    /**
     * Specified set shall be saved to the file specified.
     *
     * @param setname  Name of the set that shall be saved.
     * @param filename Full name of file the set shall be saved to.
     */
    void exportXY( const QString& setname, const QString& filename );

    /**
     * Import lines from specified file.
     *
     * @param filename Full name of file the lines shall be loaded from.
     */
    void importLines( const QString& filename );

    /**
     * Import xy data from specified file.
     *
     * @param filename Full name of file the data shall be loaded from.
     */
    void importXY( const QString& filename );


protected:

    /**
     * Build Menu.
     */
    void menu();

    /**
     * Build StatusBar.
     */
    void status();

    /**
     * Build ToolBar.
     */
    void tool();


protected:

    KConfig *config;

    DialogCalculationArnirot *dialogArnirot;

    DialogFoldSet *dialogFoldSet;

    DialogGeometryTransform *dialogGeomTransform;

    DialogImportRawData *dialogImport;

    // DialogPreferences *dialogPreferences;

    DialogSelectSet *dialogSelectSet;

    /**
     * Data for this instance of the program.
     */
    KRotData *krotData;

    int msgCount;
    
    QTimer *msgTimer;
    
    WorkArea *view;

    /**
     * List of instantiated MainWindows
     */
    static QList<MainWindow> windowList;

    /**
     * KFileDialog filter
     */
    const QString datFilter, krtFilter;
};



#include "mainwindow_inline.h"


#endif
