// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_DIALOG_IMPORT_INLINE_H
#define KROT_DIALOG_IMPORT_INLINE_H



#include "krot.h"
#include "lined.h"
#include "rawData.h"

#include <qradiobutton.h>
#include <qscrollbar.h>

#include <kapp.h>
#include <kconfig.h>


inline void DialogImportRawData::accept()
{
    emit import( data );
    hide();
    return;
}



inline void DialogImportRawData::fsr()
{
    DialogFSR *dialog = new DialogFSR( this, 0 );
    dialog->exec();
    return;
}



inline void DialogImportRawData::linModel()
{
    DialogLinModel *dialog = new DialogLinModel( this, 0 );
    dialog->exec();
    return;
}



inline void DialogImportRawData::reject()
{
    emit cancel();
    hide();
    return;
}



inline void DialogImportRawData::setAveraging( const char *text )
{
    KROT_LAUNCH( "Launching RawDataWorkArea::setAveraging" );
    int avg = atoi( text );
    data.setAveraging( avg );
    view->repaint();
    return;
}



inline void DialogImportRawData::setMarker( const int pos )
{
    emit marker( pos );
    return;
}



inline void DialogImportRawData::setAssignment( const int pos )
{
    DialogIodineFrequency *dialog = new DialogIodineFrequency( pos, this);
    dialog->exec();
    return;
}


inline void DialogImportRawData::toggleEtalon()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( "ShowEtalon", config->readBoolEntry(  "ShowEtalon", true ) ? false : true );
    traces->setItemChecked( id_etalon, config->readBoolEntry( "ShowEtalon", true ) );
    view->repaint();
    return;
}



inline void DialogImportRawData::toggleIodine()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( "ShowIodine", config->readBoolEntry(  "ShowIodine", true ) ? false : true );
    traces->setItemChecked( id_iodine, config->readBoolEntry( "ShowIodine", true ) );
    view->repaint();
    return;
}



inline void DialogImportRawData::toggleMarker()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( "ShowMarker", config->readBoolEntry( "ShowMarker", true ) ? false : true );
    traces->setItemChecked( id_marker, config->readBoolEntry( "ShowMarker", true ) );
    view->repaint();
    return;
}



inline void DialogImportRawData::togglePower()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( "ShowPower", config->readBoolEntry( "ShowPower", true ) ? false : true );
    traces->setItemChecked( id_power, config->readBoolEntry( "ShowPower", true ) );
    view->repaint();
    return;
}



inline void DialogImportRawData::toggleSignal()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( "ShowSignal", config->readBoolEntry( "ShowSignal", true ) ? false : true );
    traces->setItemChecked( id_signal, config->readBoolEntry( "ShowSignal", true ) );
    view->repaint();
    return;
}



inline DialogImportKHiDaq::DialogImportKHiDaq( RawData& data, const char *name )
    : DialogImportRawData( data, name )
{
    setCaption( i18n( "KRot [Import KHiDaq data]" ) );
    config->setGroup( KROT_CONFIG_IMPORT );
    traces->setItemChecked( id_etalon, config->readBoolEntry( "ShowEtalon", true ) );
    traces->setItemChecked( id_iodine, config->readBoolEntry( "ShowIodine", true ) );
    traces->setItemChecked( id_power,  config->readBoolEntry( "ShowPower",  true ) );
    traces->setItemChecked( id_signal, config->readBoolEntry( "ShowSignal", true ) );
    traces->setItemChecked( id_marker, config->readBoolEntry( "ShowMarker", true ) );
    return;
}



inline void RawDataWorkArea::paintEvent( QPaintEvent* )
{
    drawarea->repaint();
    return;
}



inline void RawDataWorkArea::setAssignment( const int pos )
{
    emit assignment( pos );
    return;
}



inline void RawDataWorkArea::setMarker( const int pos )
{
    emit marker( pos );
    return;
}



inline void RawDataDrawArea::setStart( int val )
{
    KROT_LAUNCH_VERBOSE( "Launching RawDataDrawArea::setStart" );
    int beg = 0;
    int end = data.size() - width();
    start = val < beg ? beg : ( val > end  ? end : val );
    repaint();
    return;
}



inline void DialogIodineFrequency::accept()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( config_AbsFreq, input->value() );
    config->writeEntry( config_Double, dbl->isChecked() );
    if( mhz->isChecked() )
	config->writeEntry( config_AbsFreqUnit, static_cast< int >( UNIT_MHZ ) );
    else
	config->writeEntry( config_AbsFreqUnit, static_cast< int >( UNIT_CM ) );
    QDialog::accept();
    return;
}


    
inline void DialogFSR::accept()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( config_FSR, atof( input->text() ) );
    if( cm->isChecked() )
	config->writeEntry( config_UnitFSR, static_cast< int >( UNIT_CM ) );
    else
	config->writeEntry( config_UnitFSR, static_cast< int >( UNIT_MHZ ) );
    QDialog::accept();
    return;
}



inline void DialogLinModel::accept()
{
    config->setGroup( KROT_CONFIG_IMPORT );
    if( linear->isChecked() )
	config->writeEntry( config_LinModel, static_cast< int >( RawData::LINEAR ) );
    else if( quadratic->isChecked() )
	config->writeEntry( config_LinModel, static_cast< int >( RawData::QUADRATIC ) );
    else
	config->writeEntry( config_LinModel, static_cast< int >( RawData::CUBICSPLINE ) );
    QDialog::accept();
    return;
}



#endif
