// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-130, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_DATASET_INLINE_H
#define KROT_DATASET_INLINE_H



inline DataSet::DataSet( KRotData *krotdata )
    : clr( blue ), krotData( krotdata ), nam( i18n( "no name" ) ),
      data( 0 ), realdata( 0 ), data_valid( false ),
      dlta( 1 ), maxmin_valid( false ), mx( 0 ), mn( 0 ),
      xy_shifts( make_pair( 0, 0 ) ), scalg( 1 ), siz( 0 ),
      strt( 0 ), stp( 0 ), realstrt( 0 ), realstp( 0 )
{
}



inline DataSet::DataSet( const DataSet& set )
    : clr( set.clr ), krotData( set.krotData ), nam( set.nam ),
      data( new double[ set.siz ] ), realdata( new double[ set.siz ] ), data_valid( set.data_valid ),
      dlta( set.dlta ), xy_shifts( set.xy_shifts ), scalg( set.scalg ), siz( set.siz ),
      strt( set.strt ), stp( set.stp ), realstrt( set.realstrt ), realstp( set.realstp )
{
    // copy data
    copy( set.data, &( set.data[ siz ] ), data );
    copy( set.realdata, &( set.realdata[ siz ] ), realdata );
    // determine addresses of min/max
    mn = std::min_element( data, &( data[ siz ] ) );
    mx = std::max_element( data, &( data[ siz ] ) );
    maxmin_valid = true;
}




inline DataSet::~DataSet()
{
    if( 0 != data )
	delete [] data;
    if( 0 != realdata )
	delete [] realdata;
}



inline double DataSet::operator[]( int64_t freq ) const
{
    KROT_LAUNCH_VERBOSE( "Launching DataSet::operator[]" );
    if( ! data_valid ) {
	// Hey, this is the sign of my bad design,
	// please let me know any idea you might have (Jochen) !
	const_cast< DataSet * >( this )->recalculateData();
    }
    freq += xy_shifts.first;
    return ( freq >= strt && freq <= stp ) ? data[ freq - strt ] : 0.0;
}	



inline bool DataSet::operator<( const DataSet& set ) const
{ 
    return nam < set.nam;
}



inline bool DataSet::operator==( const DataSet& set ) const
{ 
    return ( this == &set );
}



inline const QColor& DataSet::color() const
{
    return clr;
}



inline unsigned int DataSet::delta() const
{
    return dlta;
}



inline double DataSet::max() const
{
    if( ! data_valid )
	const_cast< DataSet * >( this )->recalculateData();
    return maxmin_valid ?  *mx  :  *( std::max_element( data, &( data[ siz ] ) ) );
}



inline double DataSet::min() const
{
    if( ! data_valid )
	const_cast< DataSet * >( this )->recalculateData();
    return maxmin_valid ?  *mn  :  *( std::min_element( data, &( data[ siz ] ) ) );
}



inline const QString& DataSet::name() const
{
    return nam;
}

 

inline const QString DataSet::shortName() const
{
    // findRev returns the last occurence of '/',
    // we want the short name to start right behind that one.
    return nam.mid( nam.findRev( '/' ) + 1, nam.length() );
}

 

inline const double DataSet::scaling() const
{
    return scalg;
}



inline void DataSet::setColor( const QColor& color )
{
    clr = color;
}



inline void DataSet::setScaling( const double scaling )
{
    scalg = scaling;
    data_valid = false;
}



inline void DataSet::setShifts( pair< int64_t, int64_t > shifts )
{
    KROT_LAUNCH( "Launching DataSet::setShifts" );
    xy_shifts = shifts;
    data_valid = false;
}



inline pair< int64_t, int64_t > DataSet::shifts() const
{
    return xy_shifts;
}


    
inline unsigned int DataSet::size() const
{
    return siz;
}



inline int64_t DataSet::start() const
{
    return strt;
}



inline int64_t DataSet::stop() const
{
    return stp;
}



#endif



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
