// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1999 Jochen Kpper
// Copyright (C) 1999 Chris Schauerte
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "dataset.h"
#include "spectrum.h"

#include <algorithm>

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "constants.h"

static void fourier(double data[], u_int64_t nn, int isign)
{
  u_int64_t n, lmax, m, l, j, istep, i;
  double wtemp, wr, wpr, wpi, wi, theta;
  double swap2, swap3;

  n=nn << 1;
  j=0;                           
  for (i=0;i<n-1;i+=2) 
  {
    if (j > i) 
    {
      double swap = data[j], swap1 = data[j+1];
      data[j] = data[i];
      data[i] = swap;
      data[j+1] = data[i+1];
      data[i+1] = swap1;
    }
    m= nn;
    while (m >= 2 && j >= m)     
    {
      j -= m;
      m >>= 1;
    }
    j += m;
  }
  
  lmax=2;
  while (n > lmax) 
  {
    istep=lmax << 1;
    theta=isign*(2 * Constants::pi() / lmax);
    wtemp=sin(0.5*theta);
    wpr = -2.0*wtemp*wtemp;
    wpi=sin(theta);
    wr=1.0;
    wi=0.0;
    for (l=1;l<lmax;l+=2) 
    {
      for (i=l-1;i<n;i+=istep)
      {
        j=i+lmax;
	swap2=wr*data[j]-wi*data[j+1];
	swap3=wr*data[j+1]+wi*data[j];
	data[j]=data[i]-swap2;
	data[j+1]=data[i+1]-swap3;
	data[i] += swap2;
	data[i+1] += swap3;
      }
      wr=(wtemp=wr)*wpr-wi*wpi+wr;
      wi=wi*wpr+wtemp*wpi+wi;
    }
    lmax=istep;
  }
}

static void realfft (double data[], u_int64_t n, int isign)
{
  u_int64_t i,i1,i2,i3,i4,np3;
  double c1=0.5,c2,h1r,h1i,h2r,h2i;
  double wr,wi,wpr,wpi,wtemp,theta;

  theta = Constants::pi() / (double) (n>>1);

  if (isign == 1) 
  {
    c2 = -0.5;
    fourier(data,n>>1,1);
  } 
  else 
  {
    c2=0.5;
    theta = -theta;
  }

  wtemp=sin(0.5*theta);
  wpr = -2.0*wtemp*wtemp;
  wpi=sin(theta);
  wr=1.0+wpr;
  wi=wpi;
  np3=n+3;

  for (i=2;i<=(n>>2);i++)       
  {
    i4 = 1+(i3 = np3-2-(i2 = 1+(i1 = i+i-2)));
    h1r=c1*(data[i1]+data[i3]);
    h1i=c1*(data[i2]-data[i4]);
    h2r = -c2*(data[i2]+data[i4]);
    h2i=c2*(data[i1]-data[i3]);
    data[i1]=h1r+wr*h2r-wi*h2i;
    data[i2]=h1i+wr*h2i+wi*h2r;
    data[i3]=h1r-wr*h2r+wi*h2i;
    data[i4] = -h1i+wr*h2i+wi*h2r;
    wr=(wtemp=wr)*wpr-wi*wpi+wr;
    wi=wi*wpr+wtemp*wpi+wi;
  }
  
  if (isign == 1) 
  {
      data[0] = (h1r=data[0])+data[1];    
      data[1] = h1r-data[1];              
  } 
  else 
  {
    data[0]=c1*((h1r=data[0])+data[1]);    
    data[1]=c1*(h1r-data[1]);             
    fourier(data,n>>1,-1);
  }
}


static void real2fft(double data1[], double data2[], double fft1[],
              double fft2[], u_int64_t n)  
{
  u_int64_t nn3,nn2,jj,j;
  double rep,rem,aip,aim;

  nn3=1+(nn2=n+n);                 
  for (j=0,jj=1;j<n;j++,jj+=2)      
  {
    fft1[jj-1]=data1[j];
    fft1[jj]=data2[j];
  }
  fourier(fft1,n,1);
  fft2[0]=fft1[1];                  
  fft1[1]=fft2[1]=0.0;             
  for (j=2;j<n+1;j+=2)              
  {
    rep=0.5*(fft1[j]+fft1[nn2-j]);
    rem=0.5*(fft1[j]-fft1[nn2-j]);
    aip=0.5*(fft1[j+1]+fft1[nn3-j]);
    aim=0.5*(fft1[j+1]-fft1[nn3-j]);
    fft1[j]=rep;
    fft1[j+1]=aim;
    fft1[nn2-j]=rep;
    fft1[nn3-j] = -aim;
    fft2[j]=aip;
    fft2[j+1] = -rem;
    fft2[nn2-j]=aip;
    fft2[nn3-j]=rem;
  }
}


static void autocorrel(double data1[], u_int64_t n, double ans[])
{
  u_int64_t no2,i;
  double dum, fft[2*n];

  realfft(data1, n, 1);
  memcpy(fft, data1, n*sizeof(double));
  fft[n]=fft[1];
  fft[n+1]=fft[1]=0;
  for (i=2; i<=n-2; i+=2)
  {
    fft[n+i]=fft[n-i];  
    fft[n+i+1]=fft[n-i+1];
  }
  memcpy(ans, fft, 2*n*sizeof(double));
    
  no2=n>>1;
  for (i=1;i<n+2;i+=2)        
  {
    ans[i-1]=(fft[i-1]*(dum=ans[i-1])+fft[i]*ans[i])/no2;
    ans[i]=(fft[i]*dum-fft[i-1]*ans[i])/no2;
  }
  ans[1]=ans[n];              
  realfft(ans, n, -1);
}

static void crosscorrel(double data1[], double data2[],
			u_int64_t n, double ans[])
{
  u_int64_t no2,i;
  double dum, fft[2*n];

  real2fft(data1,data2,fft,ans,n);
  no2=n>>1;
  for (i=1;i<n+2;i+=2)        
  {
    ans[i-1]=(fft[i-1]*(dum=ans[i-1])+fft[i]*ans[i])/no2;
    ans[i]=(fft[i]*dum-fft[i-1]*ans[i])/no2;
  }
  ans[1]=ans[n];               
  realfft(ans,n,-1);
}


Spectrum& DataSet::autocorrelate() const
{
    KROT_LAUNCH( "Launching DataSet::autocorrelate" );
    Spectrum *corr = new Spectrum( *this );
    // determine size of zero padded data array
    u_int64_t n=1;
    while( n < 2 * siz )
		n <<= 1;
    // generate and initialize zero padded array
    double *dataA = new double[ n ];
    memcpy( dataA, data, siz * sizeof(double) );
    for( u_int64_t i = siz; i < n; i++ )
		dataA[ i ] = 0;
    // allocate space for answer and do the actual calculation
    double *ans1 = new double[ 2 * n + 1 ];
    autocorrel( dataA, n, ans1 );
    // allocate space for the usefull art of the answer and initialize
    double *ans = new double[ 2 * siz - 1 ];
    memcpy( ans, ans1 + ( n - siz ) + 1, ( siz - 1 ) * sizeof( double ) );
    memcpy( ans + siz - 1, ans1, siz * sizeof( double ) );
    // delete the useless helper arrays
    delete dataA;
    delete ans1;
    // Scale by interval width
    if( 1.0 != dlta )
		for( u_int64_t i=0; i < 2 * siz - 1; i++ )
			ans[ i ] *= dlta;
    // delete old and assign the new data
    if( 0 != corr->data )
		delete [] corr->data;
    if( 0 != corr->realdata )
		delete [] corr->realdata;
    corr->realdata = ans;
    corr->data = new double[ 2 * siz - 1 ];
    std::copy( corr->realdata, &( corr->realdata[ 2 * siz - 1 ] ), corr->data );
	corr->data_valid = true;
    corr->mx = max_element( corr->data, &( corr->data[ siz ] ) );
    corr->mn = min_element( corr->data, &( corr->data[ siz ] ) );
    corr->maxmin_valid = true;
    corr->nam.sprintf( i18n( "Autocorrelation of %s" ), shortName().data() );
    corr->siz = 2 * siz - 1;
    corr->strt = corr->realstrt = -siz+1;
    corr->stp  = corr->realstp  =  siz-1;
    corr->scalg = 1.0;
    corr->xy_shifts = make_pair( 0, 0 );
    return *corr;
}



Spectrum& DataSet::crosscorrelate( const DataSet& other ) const
{
  KROT_LAUNCH( "Launching DataSet::autocorrelate" );
  Spectrum *corr = new Spectrum( *this );
    
  u_int64_t n=1;
    
  if (siz >= other.siz)
  {
    while (n < 2 * siz)
      n <<= 1;
  }
  else
  {
    while (n < 2 * other.siz)
      n <<= 1;
  }
  
  double *dataA = new double [ n ];
  memcpy (dataA, data, siz * sizeof( double ));
  for (u_int64_t i = siz; i < n; i++)  
    dataA[ i ] = 0;

  double *dataB = new double [ n ];
  memcpy (dataB, other.data, other.siz * sizeof( double ));
  for (u_int64_t j = other.siz; j < n; j++)
    dataB[ j ] = 0;
  
  double *ans1 = new double[ 2 * n + 1 ];
  
  crosscorrel (dataA, dataB, n, ans1);
  
  double *ans = new double[ siz + other.siz - 1 ];
  memcpy(ans, ans1 + (n - other.siz) + 1, (other.siz - 1) * sizeof( double ));
  memcpy(ans + other.siz - 1, ans1, siz * sizeof( double ));
  
  delete dataA;
  delete dataB;
  delete ans1;
  
  if( 1.0 != dlta )
      for (u_int64_t k = 0; k < siz + other.siz - 1; k++)
	  ans[ k ] *= dlta;

  if( 0 != corr->data )
      delete [] corr->data;
  if( 0 != corr->realdata )
		delete [] corr->realdata;
  corr->realdata = ans;
  corr->data = new double[ siz + other.siz - 1 ];
  std::copy( corr->realdata, &( corr->realdata[ siz + other.siz - 1 ] ), corr->data );
	corr->data_valid = true;
  corr->mx = max_element( corr->data, &( corr->data[ siz ] ) );
  corr->mn = min_element( corr->data, &( corr->data[ siz ] ) );
  corr->maxmin_valid = true;
  corr->nam.sprintf( i18n( "Crosscorrelation of %s and %s" ), shortName().data(), other.shortName().data() );
  corr->siz = siz + other.siz - 1;
  corr->strt = - other.siz + 1;
  corr->stp = siz - 1;
  corr->scalg = 1.0;
  corr->xy_shifts = make_pair( 0, 0 );
  return *corr;
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
