c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1998 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c
c     If you use this program for your scientific work, please cite it according to
c     the file CITATION included with this package.
c
c     krot-arnirot
c     a program to calculate rotational resolved vibrational/vibronic bands


#include "arni.h"


      subroutine svdfit( y, sigma, ndata, a, nparft, u, v, w, mp, np, chisq, funcs, shorti )
c     prepare data for singular value decomposition

c     Given a set of NDATA points I, Y(I) with standard deviation SIGMA, use
c     chi-square minimization to determine the IACPAR coefficients A of the
c     fitting equations using singular value decomposition of the NDATA by
c     NPARFT matrix. Arrays U, V, W provide workspace on input, on output they
c     define the singular value decomposition, and can be used to obtain the
c     covariance matrix. MP, NP are the physical dimensions of the matrices U, V,
c     W, as indicated below. It is necessary that MP>=NDATA, NP>=NPARFT. The
c     program returns values for the NPARFT fit parameters A, and chi-square,
c     CHISQ. The user supplies an array FUNCS that contains the NPARFT basis
c     functions.


      implicit none

      integer        maxnli
      parameter      ( maxnli = ARNIROT_MAXNLI )
      real*8         tol
      parameter      ( tol = 1.d-6 )

      integer        i, j
      integer        mp, np
      integer        ndata
      integer        nparft
      integer        nzeros
      integer        shorti

      real*8         a(np)
      real*8         b(maxnli)
      real*8         chisq
      real*8         funcs(mp,np)
      real*8         sigma
      real*8         sum
      real*8         thresh
      real*8         tmp
      real*8         u(mp,np)
      real*8         v(np,np)
      real*8         w(np)
      real*8         wmax
      real*8         y(mp)

      ARNIROT_LAUNCH ( "Launching svdfit." )

      tmp = 1.d0 / sigma
      do i = 1, ndata, 1
         do j = 1, nparft, 1
            u( i, j ) = funcs( i, j ) * tmp
         end do
         b( i ) = y( i ) * tmp
      end do

#ifdef DEBUG_FIT_OUTPUT
c     output original matrix A which is to decompose (SVD input as U)
      write(*,'(/,''matrix A (coefficients of the set of linear equations):'',/)')
      do i = 1, ndata, 1
         write(*,'(23(d11.3))') (u(i,j), j = 1, nparft, 1)
      end do
c     output right-hand side vector b
      write(*,'(/,''right-hand side vector b:'',/)')
      do i = 1, ndata, 1
         write(*,'(d15.7)') b(i)
      end do
#endif

c     Singular value decomposition of the matrix U
      call svdcmp(u, ndata, nparft, mp, np, w, v)

#ifdef DEBUG_FIT_OUTPUT
c     output matrix U after SVD
      write(*,'(/,''SVD matrix U:'',/)')
      do i = 1, ndata, 1
         write(*,'(23(d11.3))') (u(i,j), j = 1, nparft, 1)
      end do
c     output singular values I
      write(*,'(/,''SVD singular values w (before zeroing):'',/)')
      do j = 1, nparft, 1
         write(*,'(d15.7)') w(j)
      end do
#endif

c     edit the singular values
      wmax = 0.d0
      do j = 1, nparft, 1
         if( w(j) .gt. wmax ) wmax = w(j)
      end do
c     count the number of (nearly) zero singular values
      nzeros = 0
      thresh = tol*wmax
      do j = 1, nparft, 1
         if ( w(j) .lt. thresh ) then
            w(j) = 0.d0
            nzeros = nzeros + 1
         end if
      end do

#ifdef DEBUG_FIT_OUTPUT
c     output singular values II
      write(*,'(/,''SVD singular values w (after zeroing):'',/)')
      do j = 1, nparft, 1
         write(*,'(d15.7)') w(j)
      end do
c     output matrix V after SVD
      write(*,'(/,''SVD matrix V (not V**T):'',/)')
      do i = 1, nparft, 1
         write(*,'(23(d11.3))') (v(i,j), j = 1, nparft, 1)
      end do
      write(*,'(/,''number of zeroed singular values:'',i6)') nzeros
#endif

c     use chisq as indicator for an underdetermined system
      chisq = -1.d0
      if( ( ndata - nzeros ) .ge. nparft ) then
c        enough linear independent data to give a sensible solution
         call svbksb( u, w, v, ndata, nparft, mp, np, b, a )
c        evaluate chi-square
         chisq = 0.d0
         do i = 1, ndata, 1
            sum = 0.d0
            do j = 1, nparft, 1
               sum = sum + a(j) * funcs(i,j)
            end do
            chisq = chisq + ( ( y( i ) - sum ) / sigma )**2
         end do
         if (shorti.eq.0) then
            write(*,'(/,''wmax   = '',d15.7)') wmax
            write(*,'(''thresh = '',d15.7)')   thresh
            write(*,'(''chisq  = '',d15.7)')   chisq
         end if
      else
         if (shorti.eq.0) then
            write(*,'(/,''number of lin. independent data :'',i6)') ndata-nzeros
            write(*,'(''number of parameters to be fit  :'',i6)') nparft
            write(*,'(/,''ARNIROT is not able to solve for '',i6,''parameters'')') nparft
            write(*,'(''with only '',i6,'' linearly independent sets of data'',)') ndata-nzeros
         end if
      end if

      return
      end


c------------------------------------------------------------------------------
      subroutine svdcmp(a, m, n, mp, np, w, v)
c     perform singular value decomposition

c     Given a matrix A, with logical dimensions M by N and physical dimensions
c     MP by NP, this routine computes its singular value decomposition,
c     A = U*W*V^T. The matrix U replaces A on output. The diagonal matrix of
c     singular values W is output as a vector W. The matrix V (not the transpose
c     V^T) is output as V. M must be greater or equal to N; if it is smaller,
c     then A should be filled up to square with zero rows.

      implicit none

      integer        npar, nmax
      parameter      ( npar = ARNIROT_NPAR, nmax = 2*npar+2 )

      integer        i, j, k, l, m, n
      integer        its
      integer        mp, np
      integer        nm

      real*8         a(mp,np)
      real*8         anorm
      real*8         c, f, g, h
      real*8         rv1(nmax)
      real*8         s
      real*8         scale
      real*8         v(np,np)
      real*8         w(np)
      real*8         x, y, z

      ARNIROT_LAUNCH ( "Launching svdcmp." )

c     initialization because of compiler warning or program malfunction
      l = 0
      nm = 0
c     cf. subroutine arnical
c     do i = 1, np, 1
c        w(i) = 0.d0
c     end do

c     Householder reduction to bidiagonal form
      g = 0.d0
      scale = 0.d0
      anorm = 0.d0
      do i = 1, n, 1
         l = i + 1
         rv1(i) = scale*g
         g = 0.d0
         s = 0.d0
         scale = 0.d0
         if (i.le.m) then
            do k = i, m, 1
               scale = scale + dabs(a(k,i))
            end do
            if (scale.ne.0.d0) then
               do k = i, m, 1
                  a(k,i) = a(k,i)/scale
                  s = s + a(k,i)*a(k,i)
               end do
               f = a(i,i)
               g = -dsign(dsqrt(s), f)
               h = f*g - s
               a(i,i) = f - g
               if (i.ne.n) then
                  do j = l, n, 1
                     s = 0.d0
                     do k = i, m, 1
                        s = s + a(k,i) * a(k,j)
                     end do
                     f = s/h
                     do k = i, m, 1
                        a(k,j) = a(k,j) + f*a(k,i)
                     end do
                  end do
               end if
               do k = i, m, 1
                  a(k,i) = scale * a(k,i)
               end do
            end if
         end if
         w(i) = scale*g
         g = 0.d0
         s = 0.d0
         scale = 0.d0
         if ((i.le.m).and.(i.ne.n)) then
            do k = l, n, 1
               scale = scale + dabs( a(i,k) )
            end do
            if (scale.ne.0.d0) then
               do k = l, n, 1
                  a(i,k) = a(i,k)/scale
                  s = s + a(i,k)*a(i,k)
               end do
               f = a(i,l)
               g = -dsign(dsqrt(s), f)
               h = f*g - s
               a(i,l) = f - g
               do k = l, n, 1
                  rv1(k) = a(i,k)/h
               end do
               if( i .ne. m ) then
                  do j = l, m, 1
                     s = 0.d0
                     do k = l, n, 1
                        s = s + a(j,k) * a(i,k)
                     end do
                     do k = l, n, 1
                        a(j,k) = a(j,k) + s * rv1(k)
                     end do
                  end do
               end if
               do k = l, n, 1
                  a(i,k) = scale * a(i,k)
               end do
            end if
         end if
         anorm = dmax1(anorm, (dabs(w(i)) + dabs(rv1(i))))
      end do

c     accumulation of right-hand transformations
      do i = n, 1, -1
         if (i.lt.n) then
            if (g.ne.0.d0) then
               do j = l, n, 1
c                 double division to avoid possible underflow
                  v(j,i) = (a(i,j)/a(i,l))/g
               end do
               do j = l, n, 1
                  s = 0.d0
                  do k = l, n, 1
                     s = s + a(i,k)*v(k,j)
                  end do
                  do k = l, n, 1
                     v(k,j) = v(k,j) + s*v(k,i)
                  end do
               end do
            end if
            do j = l, n, 1
               v(i,j) = 0.d0
               v(j,i) = 0.d0
            end do
         end if
         v(i,i) = 1.d0
         g = rv1(i)
         l = i
      end do

c     accumulation of left-hand transformations
      do i = n, 1, -1
         l = i + 1
         g = w(i)
         if (i.lt.n) then
            do j = l, n, 1
               a(i,j) = 0.d0
            end do
         end if
         if (g.ne.0.d0) then
            g = 1.d0/g
            if (i.ne.n) then
               do j = l, n, 1
                  s = 0.d0
                  do k = l, m, 1
                     s = s + a(k,i)*a(k,j)
                  end do
                  f = (s/a(i,i))*g
                  do k = i, m, 1
                     a(k,j) = a(k,j) + f*a(k,i)
                  end do
               end do
            end if
            do j = i, m, 1
               a(j,i) = a(j,i)*g
            end do
         else
            do j = i, m, 1
               a(j,i) = 0.d0
            end do
         end if
         a(i,i) = a(i,i) + 1.d0
      end do

c     diagonalization of the bidiagonal form
c     loop over singular values
      do k = n, 1, -1
c        loop over allowed iterations
         do its = 1, 30, 1
c           test for splitting: note that rv1(1) is always zero
            do l = k, 1, -1
               nm = l - 1
               if ((dabs(rv1(l)) + anorm).eq.anorm) goto 2
               if ((dabs(w(nm))  + anorm).eq.anorm) goto 1
            end do
c           cancellation of rv1(l), if l > 1
    1       c = 0.d0
            s = 1.d0
            do i = l, k, 1
               f = s*rv1(i)
               if ((dabs(f) + anorm).ne.anorm) then
                  g = w(i)
                  h = dsqrt(f*f + g*g)
                  w(i) = h
                  h = 1.d0/h
                  c =  (g*h)
                  s = -(f*h)
                  do j = 1, m, 1
                     y = a(j,nm)
                     z = a(j,i)
                     a(j,nm) = (y*c) + (z*s)
                     a(j,i) = -(y*s) + (z*c)
                  end do
               end if
            end do
c           convergence
    2       z = w(k)
            if (l.eq.k) then
c              singular value is made nonnegative
               if (z.lt.0.d0) then
                  w(k) = -z
                  do j = 1, n, 1
                     v(j,k) = -v(j,k)
                  end do
               end if
               goto 3
            end if
            if (its.eq.30) pause 'No convergence in 30 svdcmp iterations.'
c           shift from bottom 2-by-2 minor:
            x = w(l)
            nm = k - 1
            y = w(nm)
            g = rv1(nm)
            h = rv1(k)
            f = ((y-z)*(y+z) + (g-h)*(g+h))/(2.d0*h*y)
            g = dsqrt(f*f + 1.d0)
            f = ((x-z)*(x+z) + h*((y/(f + dsign(g,f))) - h))/x
c           next QR transformation
            c = 1.d0
            s = 1.d0
            do j = l, nm, 1
               i = j + 1
               g = rv1(i)
               y = w(i)
               h = s*g
               g = c*g
               z = dsqrt(f*f + h*h)
               rv1(j) = z
               c = f/z
               s = h/z
               f =  (x*c) + (g*s)
               g = -(x*s) + (g*c)
               h = y*s
               y = y*c
               do nm = 1, n, 1
                  x = v(nm,j)
                  z = v(nm,i)
                  v(nm,j) =  (x*c) + (z*s)
                  v(nm,i) = -(x*s) + (z*c)
               end do
               z = dsqrt(f*f + h*h)
               w(j) = z
c              rotation can be arbitrary if z=0
               if (z.ne.0.d0) then
                  z = 1.d0/z
                  c = f*z
                  s = h*z
               end if
               f =  (c*g) + (s*y)
               x = -(s*g) + (c*y)
               do nm = 1, m, 1
                  y = a(nm,j)
                  z = a(nm,i)
                  a(nm,j) =  (y*c) + (z*s)
                  a(nm,i) = -(y*s) + (z*c)
               end do
            end do
            rv1(l) = 0.d0
            rv1(k) = f
            w(k)   = x
         end do
    3    continue
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine svbksb( u, w, v, m, n, mp, np, b, x )
c     singular value back substitution

      implicit none

      integer           m, mp, n, np
      double precision  b( mp ), x( np ), w( np )
      double precision  u( mp, np ), v( np, np )

c     Solves A*x = b for a vector x, where A is specified by the arrays U, W, and V as returned by SVDCMP.
c     m and n are the logical dimensions of A, and will be equal for square matrices.
c     mp and np are the physical dimensions of A.
c     b is the input right-hand side. x is the output solution vector. No input quantities are destroyed,
c     so the routine may be called sequentially with different b's.
c     Input x is only needed to provide vector memory space !

      double precision  ddot
      
      integer           i
      double precision  tmp( n )

      ARNIROT_LAUNCH ( "Launching svbksb." )

c     tmp = W^{-1} * U^T * b
      do i = 1, n
c        nonzero result only if w_j is nonzero
         if( w( i ) .ne. 0.d0 ) then
            tmp( i ) = ddot( m, u( 1, i ), 1, b, 1 ) / w( i )
         else
            tmp( i ) = 0.d0
         end if
      end do
      
c     x = V * tmp
      do i = 1, n
         x( i ) = ddot( n, v( i, 1 ), np, tmp, 1 )
      enddo

      return
      end


c------------------------------------------------------------------------------
      subroutine svdvar( v, ma, np, w, cvm, ncvm )
c     covariance matrix

c     Evaluates the covariance matrix CVM of the fit for MA parameters
c     obtained by SVDFIT. NP, NCVM give the physical dimensions of V, W, CVM
c     as indicated below.

      implicit none

      integer           ma, np, ncvm
      double precision  cvm( ncvm, ncvm )
      double precision  v( np, np ), w( np )

      integer           i, j, k
      double precision  wti( ma )

      ARNIROT_LAUNCH ( "Launching svdvar." )
      ARNIROT_DEBUG_FIT1( "ma = ", ma )

      do j = 1, ma
         ARNIROT_DEBUG_FIT3( "w(", j, " ) = ", w( j ) )
         if( ( j .le. np ) .and. ( w(j) .ne. 0.d0 ) ) then
            wti(j) = 1.d0 / w(j)**2
         else
            wti(j) = 0.d0
         end if
         do i = j, ma
            cvm( i, j ) = 0.0
         end do
      end do
      do k = 1, ma
         do j = 1, ma
            do i = j, ma
               cvm( i, j ) = cvm( i, j ) + v( i, k ) * v( j, k ) * wti( k )
            end do
         end do
      end do
c     copy to symmetric lower half
      do i = 1, ma
         do j = i+1, ma
            cvm( i, j ) = cvm( j, i )
         end do
      end do
      ARNIROT_DEBUG_FIT( 'und geschafft !' )
      
      return
      end


cc Local Variables:
cc mode: FORTRAN
cc End:
