c     -*- mode: FORTRAN -*-
c
c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1998 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c     


#include "arni.h"


      subroutine pqr( Jmx, dmeval, dmevec, maxnli, npar,
     *                ntheli, icqn, lioflw, fcompl,
     *                derivg, derive, derlg, derle,
     *                evalg, evale,
     *                evecgr, evecgi, evecer, evecei,
     *                Jmxcal, dKmax, cutint,
     *                polcal,
     *                shorti,
     *                kind,
     *                cBran, polax1,
     *                lngbar, lbars, ldash,
     *                lqn, lqn2, intens, thefre,
     *                tempf1, tempf2, weight, nuspsw,
     *                nuzero,
     *                Jmxout,
     *                ivpt )
c     calculate all allowed P,Q,R branch transitions for 0 <= J <= Jmxcal

      implicit none

      integer        Jmax, Jmx, dmham, dmeval, dmevec
      integer        maxnli, npar
      parameter      ( Jmax = ARNIROT_JMAX )
      parameter      ( dmham = 2*Jmax + 1 )

c     maximum value of abs(igs - ies), determined by (DeltaK)max
      integer        dismmx
c     flag for complex eigenvectors
      integer        fcompl
c     simple loop variables
      integer        i, m, ml, n, nl, nu
      integer        icqn(dmeval,3)
c     help variable for DeltaJ expression: idj = abs(DeltaJ) + DeltaJ
      integer        idj
c     tau loop variables: igs = tau''+ J''+ 1, ies = tau'+ J'+ 1
      integer        igs, ies, ies1
c     pointer to first eigenstate of J block
      integer        ipgsqn, ipesqn
      integer        ipgv, ipev
c     pointer to first eigenvector coefficient of J block
      integer        ipgvec, ipevec
      integer        isymg
c     type of transition (1=b, 2=c, 3=a)
      integer        itype
      integer        ivpt(0:Jmx)
      integer        Jg, Je, DeltaJ
      integer        Jmxcal, Jmxout
      integer        K
      integer        dKmax
      integer        lioflw
      integer        lqn(maxnli,6), lqn2(maxnli,6)
      integer        ntheli
      integer        nuspsw(0:3)
c     offset 'function' to pick out excited states suitable for a transition
c     1st index: type (a=3, b=1, c=2)
c     2nd index: abs(DeltaJ) (0 or 1)
c     3rd index: mod(igs,2) (0 or 1)
      integer        offset(1:3,0:1,0:1)
      integer        shorti

      real*8         ar, ai, br, bi, cr, ci
      real*8         cfreq
      real*8         cutint
      real*8         derivg(dmeval,npar), derive(dmeval,npar)
      real*8         derlg(maxnli,npar), derle(maxnli,npar)
      real*8         evecgr(dmevec), evecer(dmevec)
      real*8         evecgi(dmevec), evecei(dmevec)
      real*8         evalg(dmeval), evale(dmeval)
      real*8         evr, evrl, evru, evi, evil, eviu
      real*8         help
      real*8         intens(maxnli)
      real*8         nuzero
      real*8         polcal(3)
      real*8         rint, rintnu, rintsu
      real*8         tempf1, tempf2, weight
      real*8         thefre(maxnli)
      real*8         totinr, totini
      real*8         x1, x2, y1, y2
      real*8         xnt(dmham,dmham,-1:1), ynt(dmham,dmham,-1:1), znt(dmham,dmham,-1:1)

      character*1    cBran(-1:1), polax1(3)
      character*9    kind(0:1)
      character*81   lbars, ldash
      character*220  lngbar

#ifdef DEBUG_VERBOSE
c     unit for debug output: 30 (P), 40 (Q), 50 (R)
      integer        iunout
c     line intensity matrices
      real*8         intmat(dmham,dmham,-1:1)
c     line strength matrices
      real*8         lsfmat(dmham,dmham,-1:1)
c     workspace for subroutine imout2
      real*8         sum(dmham)
#endif


      ARNIROT_LAUNCH ( "Launching pqr." )

c     initialize error flag checking for lines overflow
      lioflw = 0

c     define offset 'function'
      offset(1,0,0) = 2
      offset(1,0,1) = 2
      offset(1,1,0) = 3
      offset(1,1,1) = 1
      offset(2,0,0) = 3
      offset(2,0,1) = 1
      offset(2,1,0) = 2
      offset(2,1,1) = 2
      offset(3,0,0) = 1
      offset(3,0,1) = 3
      offset(3,1,0) = 0
      offset(3,1,1) = 0

c     treat separately the following types of transitions not covered by the loop:
c        R type        0 --> 1            (right here)
c        Q type   Jmxcal --> Jmxcal       (v.i.)
c        P type   Jmxcal --> Jmxcal - 1   (v.i.)
      call pqr2( Jmx, dmeval, dmevec, maxnli, npar,
     *           0, 1, ntheli, icqn, lioflw, fcompl,
     *           derivg, derive, derlg, derle,
     *           evalg, evale,
     *           evecgr, evecgi, evecer, evecei,
     *           dKmax, cutint,
     *           polcal,
     *           shorti,
     *           kind,
     *           cBran, polax1,
     *           lngbar, lbars, ldash,
     *           lqn, lqn2, intens, thefre,
     *           tempf1, tempf2, weight, nuspsw,
     *           nuzero,
     *           Jmxout,
     *           ivpt,
     *           offset )

c     loop over all J except 0 and Jmxcal
      do Jg = 1, Jmxcal-1, 1
#ifdef DEBUG_VERBOSE
         if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) ) then
c           zero the cosine matrices work spaces
c           this should only be necessary for output purpose because all
c           matrix elements being accessed later are properly initialized
            call cmat02( xnt, dmham, Jg )
            call cmat02( ynt, dmham, Jg )
            call cmat02( znt, dmham, Jg )

c           zero the line strength factor / intensity matrices workspaces
            call cmat02( lsfmat, dmham, Jg )
            call cmat02( intmat, dmham, Jg )
         end if
#endif

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

c--------P branch, Delta J = -1 -----------------------------------------------
c        set up the <J''K''M''| phi Za |J'K'M'> matrix
c        ground state quantum number counter for row of cosine matrix
         K = -Jg
         do i = 2, 2*Jg, 1
            K = K + 1
c           fill the K' = K'' cosine matrix elements for z type cosine operator
            x1 = ( Jg + K ) * ( Jg - K )
            x2 = Jg
            znt(i,i-1,-1) = dsqrt( x1 / x2 )
         end do

c        set up <J''K''M''| phi Zb |J'K'M'> and
c               <J''K''M''| phi Zc |J'K'M'> matrices
         K = -Jg - 1
c        offset for filling lower diagonal matrix elements
         m = 2*Jg + 2
         n = 2*Jg
         do i = 1, 2*Jg-1
            K = K + 1
            x1 = ( Jg - K ) * ( Jg - K - 1 )
            x2 = 4 * Jg
            help = -dsqrt( x1 / x2 )
            xnt(i,i,-1) = -help
            ynt(i,i,-1) =  help
            m = m - 1
            n = n - 1
            xnt(m,n,-1) =  help
            ynt(m,n,-1) =  help
         end do

c--------Q branch, Delta J =  0 -----------------------------------------------
c        set up the <J''K''M''| phi Za |J'K'M'> matrix
c        ground state quantum number counter for row of cosine matrix
         K = -Jg - 1
c        calculate J dependent factor outside of K loop
         x1 = 2*Jg + 1
         x2 = Jg * ( Jg + 1 )
         y1 = dsqrt( x1 / x2 )
         do i = 1, 2*Jg+1, 1
            K = K + 1
c           fill the diagonal cosine matrix elements for z type cosine operator
            znt(i,i,0) = y1*K
         end do

c        set up <J''K''M''| phi Zb |J'K'M'> and
c               <J''K''M''| phi Zc |J'K'M'> matrices
         K = -Jg - 1
c        evaluate J dependent factor outside of K loop
         y2 = 0.5d0*y1
         do i = 1, 2*Jg, 1
            K = K + 1
            x1 = ( Jg - K ) * ( Jg + K + 1 )
            help = y2*dsqrt( x1 )
            xnt(i  ,i+1,0) =  help
            ynt(i  ,i+1,0) = -help
            xnt(i+1,i  ,0) =  help
            ynt(i+1,i  ,0) =  help
         end do

c--------R branch, Delta J = +1 -----------------------------------------------
c        set up the <J''K''M''| phi Za |J'K'M'> matrix
c        ground state quantum number counter for row of cosine matrix
         K = -Jg - 1
         do i = 1, 2*Jg+1, 1
            K = K + 1
c           fill the K' = K'' cosine matrix elements for z type cosine operator
            x1 = ( Jg + K + 1 ) * ( Jg - K + 1 )
            x2 = Jg + 1
            znt(i,i+1,1) = dsqrt( x1 / x2 )
         end do

c        set up <J''K''M''| phi Zb |J'K'M'> and
c               <J''K''M''| phi Zc |J'K'M'> matrices
         K = -Jg - 1
c        offset for filling lower diagonal matrix elements
         m = 2*Jg + 2
         do i = 1, 2*Jg+1, 1
            K = K + 1
            x1 = ( Jg + K + 1 ) * ( Jg + K + 2 )
            x2 = 4 * ( Jg + 1 )
            help = dsqrt( x1 / x2 )
            xnt(i,i+2,1) = -help
            ynt(i,i+2,1) =  help
            m = m - 1
            xnt(m,m  ,1) =  help
            ynt(m,m  ,1) =  help
         end do

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

#ifdef DEBUG_VERBOSE
c        output the pure cosine matrices for a given J block
         if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) ) then
            do DeltaJ = -1, 1, 1
               call cmout2( xnt, dmham, Jg, DeltaJ, 1, 0, kind, cBran, polax1, ldash )
               call cmout2( ynt, dmham, Jg, DeltaJ, 2, 1, kind, cBran, polax1, ldash )
               call cmout2( znt, dmham, Jg, DeltaJ, 3, 0, kind, cBran, polax1, ldash )
               write(*,'(a)') lbars
            end do
         end if
#endif

c        pointer to first ground state eigenvector coefficient
         ipgvec = ivpt(Jg)
c        pointer to first ground eigenstate
         ipgsqn = Jg*Jg

c        loop through branches
         do DeltaJ = -1, 1, 1
            Je = Jg + DeltaJ
c           pointer to first excited state eigenvector coefficient
            ipevec = ivpt(Je)
c           pointer to first excited eigenstate
            ipesqn = Je*Je
c           determine the maximum difference between loop variables igs and ies (v.i.)
c           taking into account the constraint for DeltaK
            dismmx = 2*dKmax - DeltaJ

#ifdef DEBUG_VERBOSE
c           phase debugging module
            if ( shorti .eq. 0 ) then
               iunout = 40 + 10*DeltaJ
               write(iunout,'(''phase debugging module: '',i3,'' <--'',i3, '' transitions'')') Je, Jg
               write(iunout,'(''  J''''  Ka'''' Kc'''' J'''''''' Ka''''''''Kc''''''''  ai     bi     ci     ar     br     cr'',
     *              ''   totini totinr  rintsu*1E04'')')
            end if
#endif
      
c           loop through all allowed intensity calculations

c           loop through ground states
            do igs = 1, 2*Jg+1, 1
c              evaluate pointers to the first elements of the eigenvectors being considered
               ipgv = ipgvec + (igs - 1)*(2*Jg + 1)

c              loop through b,c,a types of transition
               do itype = 1, 3, 1
c                 check for vanishing dipole component first
                  if ( polcal(itype) .ne. 0 ) then
c                    set correct start position for ies in a do-while construction
                     ies1 = igs - dismmx + mod( ( offset( itype, iabs(DeltaJ), mod(igs,2) ) + dismmx ), 4 ) - 4
   10                continue
                        ies1 = ies1 + 4
                     if ( ies1 .lt. 1 ) goto 10

c                    loop through excited states: pick out only the right partners (note step width!)
                     do ies = ies1, min0( 2*Je + 1, igs + 2*dKmax + DeltaJ ), 4
c                       evaluate pointers to the first elements of the eigenvectors being considered
                        ipev = ipevec + (ies - 1)*(2*Je + 1)
c                       initialize intensity summation terms to 0
                        ar = 0.d0
                        ai = 0.d0
                        br = 0.d0
                        bi = 0.d0
                        cr = 0.d0
                        ci = 0.d0

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

c                       complex eigenvectors (general case)
                        if ( fcompl .ne. 0 ) then
c- - - - - - - - - - - - - phi(Za) matrix   (a = z)- - - - - - - - - - - - - -
c                          <J''K''M''| phi Za |J'K'M'>
                           if ( polcal(3) .ne. 0 ) then
c                             one-lower (P), diagonal (Q) or one-upper (R) diagonal elements
                              idj = iabs(DeltaJ) - DeltaJ
                              m = idj / 2
                              n = m + DeltaJ
                              do i = 1, 2*Jg+1-idj, 1
                                 m = m + 1
                                 n = n + 1
c                                calculate real eigenvector products
                                 evr = evecgr(m+ipgv)*evecer(n+ipev) + evecgi(m+ipgv)*evecei(n+ipev)
c                                calculate imaginary eigenvector products
                                 evi = evecgi(m+ipgv)*evecer(n+ipev) - evecgr(m+ipgv)*evecei(n+ipev)
                                 ar = ar + evr*znt(m,n,DeltaJ)
                                 ai = ai + evi*znt(m,n,DeltaJ)
                              end do
c                             multiply NOW with polarization factor
                              ar = ar * polcal(3)
                              ai = ai * polcal(3)
                           end if

c- - - - - - - - - - - - - phi(Zb) matrix   (b = x)- - - - - - - - - - - - - -
c                          <J''K''M''| phi Zb |J'K'M'>
                           if ( polcal(1) .ne. 0 ) then
c                             ml,nl: two-lower (P), one-lower (Q) and diagonal (R) elements
c                             mu,nu: diagonal (P), one-upper (Q) and two-upper (R) diagonal elements
c                          // nl = 1, ..., 2J'-DeltaJ
c                          // ml = nl - DeltaJ + 1
c                          // nu = nl + DeltaJ + 1
c                          // mu = nu - DeltaJ - 1 = nl !  ==> index mu is not used
                              ml = 1 - DeltaJ
                              nu = 1 + DeltaJ
                              do nl = 1, 2*Je-DeltaJ, 1
                                 ml = ml + 1
                                 nu = nu + 1
c                                calculate real eigenvector products
                                 evrl = evecgr(ml+ipgv)*evecer(nl+ipev) + evecgi(ml+ipgv)*evecei(nl+ipev)
                                 evru = evecgr(nl+ipgv)*evecer(nu+ipev) + evecgi(nl+ipgv)*evecei(nu+ipev)
c                                calculate imaginary eigenvector products
                                 evil = evecgi(ml+ipgv)*evecer(nl+ipev) - evecgr(ml+ipgv)*evecei(nl+ipev)
                                 eviu = evecgi(nl+ipgv)*evecer(nu+ipev) - evecgr(nl+ipgv)*evecei(nu+ipev)
                                 br = br + evrl*xnt(ml,nl,DeltaJ) + evru*xnt(nl,nu,DeltaJ)
                                 bi = bi + evil*xnt(ml,nl,DeltaJ) + eviu*xnt(nl,nu,DeltaJ)
                              end do
                              br = br * polcal(1)
                              bi = bi * polcal(1)
                           end if

c- - - - - - - - - - - - - phi(Zc) matrix   (c = y)- - - - - - - - - - - - - -
c                          <J''K''M''| phi Zc |J'K'M'>
                           if ( polcal(2) .ne. 0 ) then
c                             ml,nl: two-lower (P), one-lower (Q) and diagonal (R) elements
c                             mu,nu: diagonal (P), one-upper (Q) and two-upper (R) diagonal elements
                              ml = 1 - DeltaJ
                              nu = 1 + DeltaJ
                              do nl = 1, 2*Je-DeltaJ, 1
                                 ml = ml + 1
                                 nu = nu + 1
c                                calculate real eigenvector products
                                 evrl = evecgr(ml+ipgv)*evecer(nl+ipev) + evecgi(ml+ipgv)*evecei(nl+ipev)
                                 evru = evecgr(nl+ipgv)*evecer(nu+ipev) + evecgi(nl+ipgv)*evecei(nu+ipev)
c                                calculate imaginary eigenvector products
                                 evil = evecgi(ml+ipgv)*evecer(nl+ipev) - evecgr(ml+ipgv)*evecei(nl+ipev)
                                 eviu = evecgi(nl+ipgv)*evecer(nu+ipev) - evecgr(nl+ipgv)*evecei(nu+ipev)
                                 cr = cr - evil*ynt(ml,nl,DeltaJ) - eviu*ynt(nl,nu,DeltaJ)
                                 ci = ci + evrl*ynt(ml,nl,DeltaJ) + evru*ynt(nl,nu,DeltaJ)
                              end do
                              cr = cr * polcal(2)
                              ci = ci * polcal(2)
                           end if

c                       real eigenvectors (simple rr case)
                        else
c- - - - - - - - - - - - - phi(Za) matrix   (a = z)- - - - - - - - - - - - - -
c                          <J''K''M''| phi Za |J'K'M'>
                           if ( polcal(3) .ne. 0 ) then
c                             one-lower (P), diagonal (Q) or one-upper (R) diagonal elements
                              idj = iabs(DeltaJ) - DeltaJ
                              m = idj / 2
                              n = m + DeltaJ
                              do i = 1, 2*Jg+1-idj, 1
                                 m = m + 1
                                 n = n + 1
c                                calculate real eigenvector products
                                 evr = evecgr(m+ipgv)*evecer(n+ipev)
                                 ar = ar + evr*znt(m,n,DeltaJ)
                              end do
c                             multiply NOW with polarization factor
                              ar = ar * polcal(3)
                           end if

c- - - - - - - - - - - - - phi(Zb) matrix   (b = x)- - - - - - - - - - - - - -
c                          <J''K''M''| phi Zb |J'K'M'>
                           if ( polcal(1) .ne. 0 ) then
                              ml = 1 - DeltaJ
                              nu = 1 + DeltaJ
                              do nl = 1, 2*Je-DeltaJ, 1
                                 ml = ml + 1
                                 nu = nu + 1
c                                calculate real eigenvector products
                                 evrl = evecgr(ml+ipgv)*evecer(nl+ipev)
                                 evru = evecgr(nl+ipgv)*evecer(nu+ipev)
                                 br = br + evrl*xnt(ml,nl,DeltaJ) + evru*xnt(nl,nu,DeltaJ)
                              end do
                              br = br * polcal(1)
                           end if

c- - - - - - - - - - - - - phi(Zc) matrix   (c = y)- - - - - - - - - - - - - -
c                          <J''K''M''| phi Zc |J'K'M'>
                           if ( polcal(2) .ne. 0 ) then
                              ml = 1 - DeltaJ
                              nu = 1 + DeltaJ
                              do nl = 1, 2*Je-DeltaJ, 1
                                 ml = ml + 1
                                 nu = nu + 1
c                                calculate real eigenvector products
                                 evrl = evecgr(ml+ipgv)*evecer(nl+ipev)
                                 evru = evecgr(nl+ipgv)*evecer(nu+ipev)
                                 ci = ci + evrl*ynt(ml,nl,DeltaJ) + evru*ynt(nl,nu,DeltaJ)
                              end do
                              ci = ci * polcal(2)
                           end if

                        end if

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

c                       calculate the total intensity
c                       total real intensity component
                        totinr = ar + br + cr
c                       total imaginary intensity component
                        totini = ai + bi + ci
c                       total intensity squared (= line strength factor)
                        rintsu = totinr*totinr + totini*totini
#ifdef DEBUG_VERBOSE
c                       store line strength factor
                        if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) )
     *                       lsfmat(ies,igs,DeltaJ) = rintsu
#endif
c                       consider the statistical weight of nuclear spin
c                       Ka'' Kc''   bin.   dec.code
c                       even even   0  0      0
c                       even  odd   0  1      1
c                        odd even   1  0      2
c                        odd  odd   1  1      3

                        isymg = 2*mod( icqn( igs + ipgsqn, 2 ), 2 ) + mod( icqn( igs + ipgsqn, 3 ), 2 )
                        rintnu = rintsu * nuspsw(isymg)

c                       apply the Boltzman factor: Boltzman constant in MHz
c                       permit two-temperature model
                        if ( weight .ne. 0 ) then
                           rint = dexp( -(evalg(igs+ipgsqn)*tempf1) )
     *                            + weight * dexp( -(evalg(igs+ipgsqn)*tempf2) )
                        else
                           rint = dexp( -(evalg(igs+ipgsqn)*tempf1) )
                        end if
                        rint = rint * rintnu

c                       intensity calculation done for these two states for this type of transition
#ifdef DEBUG_VERBOSE
c                       fill calculated value into intensity matrix
                        if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) )
     *                       intmat(ies,igs,DeltaJ) = rint
#endif

c                       intensity cutoff; save line arrays if |<..|phi Fg|..>| > cutint
                        if ( rint .gt. cutint ) then
c                          increment line counter by one
                           ntheli = ntheli + 1
c                          leave this subroutine if ntheli > maxnli, pass current value of qn J
                           if ( ntheli .gt. maxnli ) then
                              lioflw = Jg
                              return
                           end if
c                          save excited and ground state qns for theoretical line set
                           do i = 1, 3, 1
                              lqn( ntheli, i+3 ) = icqn( igs+ipgsqn, i )
                              lqn( ntheli, i   ) = icqn( ies+ipesqn, i )
                           end do
#ifdef DEBUG_SUBBRANCH_OUTPUT
                           call lqnset( ntheli, Jmax, dKmax, lqn, lqn2, maxnli )
#endif
c                          actual intensity of this transition
                           intens(ntheli) = rint
c                          pure rotational energy difference
                           cfreq = evale(ies+ipesqn) - evalg(igs+ipgsqn)
c                          actual frequency: add nuzero to the above value
                           thefre(ntheli) = cfreq + nuzero

c                          save the derivative information
                           do i = 1, npar, 1
                              derle(ntheli,i) = derive(ies+ipesqn,i)
                              derlg(ntheli,i) = derivg(igs+ipgsqn,i)
                           end do
#ifdef DEBUG_VERBOSE
c                          phase debugging module
                           if ( shorti .eq. 0 ) then
                              write(iunout,'(6(i4),1x,8(1x,f6.3),1x,i7)')
     *                             (lqn(ntheli,i), i = 1,6), ai, bi, ci, ar, br, cr, totini, totinr, idnint(rintsu*1.d4)
                           end if
#endif
c                       { line storage }
                        end if
c                    { tau' loop }
                     end do
c                 { type check }
                  end if
c              { type loop }
               end do
c           { tau'' loop }
            end do

#ifdef DEBUG_VERBOSE
c           output complete line strength factor / intensity matrices
            if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) ) then
               call imout2( lsfmat, dmham, Jg, DeltaJ, icqn, dmeval, sum, 0, cBran, polax1, lngbar, lbars )
               call imout2( intmat, dmham, Jg, DeltaJ, icqn, dmeval, sum, 1, cBran, polax1, lngbar, lbars )
            end if
#endif
c        { DeltaJ loop }
         end do
c     { J loop }
      end do

c     separate treatment:
c        Q type   Jmxcal --> Jmxcal
c        P type   Jmxcal --> Jmxcal - 1
      call pqr2( Jmx, dmeval, dmevec, maxnli, npar,
     *           Jmxcal, 0, ntheli, icqn, lioflw, fcompl,
     *           derivg, derive, derlg, derle,
     *           evalg, evale,
     *           evecgr, evecgi, evecer, evecei,
     *           dKmax, cutint,
     *           polcal,
     *           shorti,
     *           kind,
     *           cBran, polax1,
     *           lngbar, lbars, ldash,
     *           lqn, lqn2, intens, thefre,
     *           tempf1, tempf2, weight, nuspsw,
     *           nuzero,
     *           Jmxout,
     *           ivpt,
     *           offset )
c     stop calculation of lines if array limit has been reached
      if ( lioflw .ne. 0 )
     *     return

      call pqr2( Jmx, dmeval, dmevec, maxnli, npar,
     *           Jmxcal, -1, ntheli, icqn, lioflw, fcompl,
     *           derivg, derive, derlg, derle,
     *           evalg, evale,
     *           evecgr, evecgi, evecer, evecei,
     *           dKmax, cutint,
     *           polcal,
     *           shorti,
     *           kind,
     *           cBran, polax1,
     *           lngbar, lbars, ldash,
     *           lqn, lqn2, intens, thefre,
     *           tempf1, tempf2, weight, nuspsw,
     *           nuzero,
     *           Jmxout,
     *           ivpt,
     *           offset )

      ARNIROT_LAUNCH ( "Leaving pqr." )

      return
      end


c------------------------------------------------------------------------------
      subroutine pqr2( Jmx, dmeval, dmevec, maxnli, npar,
     *                 J, DeltaJ, ntheli, icqn, lioflw, fcompl,
     *                 derivg, derive, derlg, derle,
     *                 evalg, evale,
     *                 evecgr, evecgi, evecer, evecei,
     *                 dKmax, cutint,
     *                 polcal,
     *                 shorti,
     *                 kind,
     *                 cBran, polax1,
     *                 lngbar, lbars, ldash,
     *                 lqn, lqn2, intens, thefre,
     *                 tempf1, tempf2, weight, nuspsw,
     *                 nuzero,
     *                 Jmxout,
     *                 ivpt,
     *                 offset )
c     calculate (P or Q or R branch) transitions for given J and DeltaJ

      implicit none

      integer        Jmax, Jmx, dmham, dmeval, dmevec
      integer        maxnli, npar
      parameter      ( Jmax = ARNIROT_JMAX )
      parameter      ( dmham = 2*Jmax + 1 )

      integer        dismmx
      integer        fcompl
      integer        i, m, ml, n, nl, nu
      integer        icqn(dmeval,3)
      integer        idj
      integer        igs, ies, ies1
      integer        ipgsqn, ipesqn
      integer        ipgv, ipev
      integer        ipgvec, ipevec
      integer        isymg
      integer        itype
      integer        ivpt(0:Jmx)
      integer        J, Jg, Je, DeltaJ
      integer        Jmxout
      integer        K
      integer        dKmax
      integer        lioflw
      integer        lqn(maxnli,6), lqn2(maxnli,6)
      integer        ntheli
      integer        nuspsw(0:3)
      integer        offset(1:3,0:1,0:1)
      integer        shorti

      real*8         ar, ai, br, bi, cr, ci
      real*8         cfreq
      real*8         cutint
      real*8         derivg(dmeval,npar), derive(dmeval,npar)
      real*8         derlg(maxnli,npar), derle(maxnli,npar)
      real*8         evecgr(dmevec), evecer(dmevec)
      real*8         evecgi(dmevec), evecei(dmevec)
      real*8         evalg(dmeval), evale(dmeval)
      real*8         evr, evrl, evru, evi, evil, eviu
      real*8         help
      real*8         intens(maxnli)
      real*8         nuzero
      real*8         polcal(3)
      real*8         rint, rintnu, rintsu
      real*8         tempf1, tempf2, weight
      real*8         thefre(maxnli)
      real*8         totinr, totini
      real*8         x1, x2, y1, y2
      real*8         xnt(dmham,dmham), ynt(dmham,dmham), znt(dmham,dmham)

      character*1    cBran(-1:1), polax1(3)
      character*9    kind(0:1)
      character*81   lbars, ldash
      character*220  lngbar

#ifdef DEBUG_VERBOSE
      integer        iunout
      real*8         intmat(dmham,dmham)
      real*8         lsfmat(dmham,dmham)
      real*8         sum(dmham)
#endif


      ARNIROT_LAUNCH ( "Launching pqr2." )

c     define J'' and J' quantum numbers from J
      Jg = J
      Je = J + DeltaJ

#ifdef DEBUG_VERBOSE
      if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) ) then
c        zero the cosine matrices work spaces
         call cmat0( xnt, dmham, Jg, Je )
         call cmat0( ynt, dmham, Jg, Je )
         call cmat0( znt, dmham, Jg, Je )

c        zero the line strength factor / intensity matrices workspaces
         call cmat0( lsfmat, dmham, Je, Jg )
         call cmat0( intmat, dmham, Je, Jg )
      end if
#endif

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

c-----P branch, Delta J = -1 --------------------------------------------------
      if ( DeltaJ .eq. -1 ) then
c        set up the <J''K''M''| phi Za |J'K'M'> matrix
c        ground state quantum number counter for row of cosine matrix
         K = -Jg
         do i = 2, 2*Jg, 1
            K = K + 1
c           fill the K' = K'' cosine matrix elements for z type cosine operator
            x1 = ( Jg + K ) * ( Jg - K )
            x2 = Jg
            znt(i,i-1) = dsqrt( x1 / x2 )
         end do

c        set up <J''K''M''| phi Zb |J'K'M'> and
c               <J''K''M''| phi Zc |J'K'M'> matrices
         K = -Jg - 1
c        offset for filling lower diagonal matrix elements
         m = 2*Jg + 2
         n = 2*Jg
         do i = 1, 2*Jg-1
            K = K + 1
            x1 = ( Jg - K ) * ( Jg - K - 1 )
            x2 = 4 * Jg
            help = -dsqrt( x1 / x2 )
            xnt(i,i) = -help
            ynt(i,i) =  help
            m = m - 1
            n = n - 1
            xnt(m,n) =  help
            ynt(m,n) =  help
         end do

c-----Q branch, Delta J =  0 --------------------------------------------------
      else if ( DeltaJ .eq. 0 ) then
c        set up the <J''K''M''| phi Za |J'K'M'> matrix
c        ground state quantum number counter for row of cosine matrix
         K = -Jg - 1
c        calculate J dependent factor outside of K loop
         x1 = 2*Jg + 1
         x2 = Jg * ( Jg + 1 )
         y1 = dsqrt( x1 / x2 )
         do i = 1, 2*Jg+1, 1
            K = K + 1
c           fill the diagonal cosine matrix elements for z type cosine operator
            znt(i,i) = y1*K
         end do

c        set up <J''K''M''| phi Zb |J'K'M'> and
c               <J''K''M''| phi Zc |J'K'M'> matrices
         K = -Jg - 1
c        evaluate J dependent factor outside of K loop
         y2 = 0.5d0*y1
         do i = 1, 2*Jg, 1
            K = K + 1
            x1 = ( Jg - K ) * ( Jg + K + 1 )
            help = y2*dsqrt( x1 )
            xnt(i  ,i+1) =  help
            ynt(i  ,i+1) = -help
            xnt(i+1,i  ) =  help
            ynt(i+1,i  ) =  help
         end do

c-----R branch, Delta J = +1 --------------------------------------------------
      else
c        set up the <J''K''M''| phi Za |J'K'M'> matrix
c        ground state quantum number counter for row of cosine matrix
         K = -Jg - 1
         do i = 1, 2*Jg+1, 1
            K = K + 1
c           fill the K' = K'' cosine matrix elements for z type cosine operator
            x1 = ( Jg + K + 1 ) * ( Jg - K + 1 )
            x2 = Jg + 1
            znt(i,i+1) = dsqrt( x1 / x2 )
         end do

c        set up <J''K''M''| phi Zb |J'K'M'> and
c               <J''K''M''| phi Zc |J'K'M'> matrices
         K = -Jg - 1
c        offset for filling lower diagonal matrix elements
         m = 2*Jg + 2
         do i = 1, 2*Jg+1, 1
            K = K + 1
            x1 = ( Jg + K + 1 ) * ( Jg + K + 2 )
            x2 = 4 * ( Jg + 1 )
            help = dsqrt( x1 / x2 )
            xnt(i,i+2) = -help
            ynt(i,i+2) =  help
            m = m - 1
            xnt(m,m  ) =  help
            ynt(m,m  ) =  help
         end do
      end if

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

#ifdef DEBUG_VERBOSE
c     output the pure cosine matrices for a given J block
      if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) ) then
         call cmout( xnt, dmham, Jg, Je, 1, 0, kind, cBran, polax1, ldash )
         call cmout( ynt, dmham, Jg, Je, 2, 1, kind, cBran, polax1, ldash )
         call cmout( znt, dmham, Jg, Je, 3, 0, kind, cBran, polax1, ldash )
         write(*,'(a)') lbars
      end if
#endif

c     pointer to first ground state eigenvector coefficient
      ipgvec = ivpt(Jg)
c     pointer to first ground eigenstate
      ipgsqn = Jg*Jg

c     pointer to first excited state eigenvector coefficient
      ipevec = ivpt(Je)
c     pointer to first excited eigenstate
      ipesqn = Je*Je
c     determine the maximum difference between loop variables ies and igs (below)
c     taking into account the constraint for DeltaK
      dismmx = 2*dKmax - DeltaJ

#ifdef DEBUG_VERBOSE
c     phase debugging module
      if ( shorti .eq. 0 ) then
         iunout = 40 + 10*DeltaJ
         write(iunout,'(''phase debugging module: '',i3,'' <--'',i3, '' transitions'')') Je, Jg
         write(iunout,'(''  J''''  Ka'''' Kc'''' J'''''''' Ka''''''''Kc''''''''  ai     bi     ci     ar     br     cr'',
     *        ''   totini totinr  rintsu*1E04'')')
      end if
#endif
      
c     loop through all allowed intensity calculations
c     loop through ground states
      do igs = 1, 2*Jg+1, 1
c        evaluate pointers to the first elements of the eigenvectors being considered
         ipgv = ipgvec + (igs - 1)*(2*Jg + 1)

c        loop through b,c,a types of transition
         do itype = 1, 3, 1
c           check for vanishing dipole component first
            if ( polcal(itype) .ne. 0 ) then
c              set correct start position for ies in a do-while construction
               ies1 = igs - dismmx + mod( ( offset( itype, iabs(DeltaJ), mod(igs,2) ) + dismmx ), 4 ) - 4
   10          continue
                  ies1 = ies1 + 4
               if ( ies1 .lt. 1 ) goto 10

c              loop through excited states: pick out only the right partners (note the step width!)
               do ies = ies1, min0( 2*Je + 1, igs + 2*dKmax + DeltaJ ), 4
c                 evaluate pointers to the first elements of the eigenvectors being considered
                  ipev = ipevec + (ies - 1)*(2*Je + 1)
c                 initialize intensity summation terms to 0
                  ar = 0.d0
                  ai = 0.d0
                  br = 0.d0
                  bi = 0.d0
                  cr = 0.d0
                  ci = 0.d0

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

c                 complex eigenvectors (general case)
                  if ( fcompl .ne. 0 ) then
c- - - - - - - - - - phi(Za) matrix   (a = z)- - - - - - - - - - - - - - - - -
c                    <J''K''M''| phi Za |J'K'M'>
                     if ( polcal(3) .ne. 0 ) then
                        idj = iabs(DeltaJ) - DeltaJ
                        m = idj / 2
                        n = m + DeltaJ
                        do i = 1, 2*Jg+1-idj, 1
                           m = m + 1
                           n = n + 1
                           evr = evecgr(m+ipgv)*evecer(n+ipev) + evecgi(m+ipgv)*evecei(n+ipev)
                           evi = evecgi(m+ipgv)*evecer(n+ipev) - evecgr(m+ipgv)*evecei(n+ipev)
                           ar = ar + evr*znt(m,n)
                           ai = ai + evi*znt(m,n)
                        end do
                        ar = ar * polcal(3)
                        ai = ai * polcal(3)
                     end if

c- - - - - - - - - - phi(Zb) matrix   (b = x)- - - - - - - - - - - - - - - - -
c                    <J''K''M''| phi Zb |J'K'M'>
                     if ( polcal(1) .ne. 0 ) then
                        ml = 1 - DeltaJ
                        nu = 1 + DeltaJ
                        do nl = 1, 2*Je-DeltaJ, 1
                           ml = ml + 1
                           nu = nu + 1
                           evrl = evecgr(ml+ipgv)*evecer(nl+ipev) + evecgi(ml+ipgv)*evecei(nl+ipev)
                           evru = evecgr(nl+ipgv)*evecer(nu+ipev) + evecgi(nl+ipgv)*evecei(nu+ipev)
                           evil = evecgi(ml+ipgv)*evecer(nl+ipev) - evecgr(ml+ipgv)*evecei(nl+ipev)
                           eviu = evecgi(nl+ipgv)*evecer(nu+ipev) - evecgr(nl+ipgv)*evecei(nu+ipev)
                           br = br + evrl*xnt(ml,nl) + evru*xnt(nl,nu)
                           bi = bi + evil*xnt(ml,nl) + eviu*xnt(nl,nu)
                        end do
                        br = br * polcal(1)
                        bi = bi * polcal(1)
                     end if

c- - - - - - - - - - phi(Zc) matrix   (c = y)- - - - - - - - - - - - - - - - -
c                    <J''K''M''| phi Zc |J'K'M'>
                     if ( polcal(2) .ne. 0 ) then
                        ml = 1 - DeltaJ
                        nu = 1 + DeltaJ
                        do nl = 1, 2*Je-DeltaJ, 1
                           ml = ml + 1
                           nu = nu + 1
                           evrl = evecgr(ml+ipgv)*evecer(nl+ipev) + evecgi(ml+ipgv)*evecei(nl+ipev)
                           evru = evecgr(nl+ipgv)*evecer(nu+ipev) + evecgi(nl+ipgv)*evecei(nu+ipev)
                           evil = evecgi(ml+ipgv)*evecer(nl+ipev) - evecgr(ml+ipgv)*evecei(nl+ipev)
                           eviu = evecgi(nl+ipgv)*evecer(nu+ipev) - evecgr(nl+ipgv)*evecei(nu+ipev)
                           cr = cr - evil*ynt(ml,nl) - eviu*ynt(nl,nu)
                           ci = ci + evrl*ynt(ml,nl) + evru*ynt(nl,nu)
                        end do
                        cr = cr * polcal(2)
                        ci = ci * polcal(2)
                     end if

c                 real eigenvectors (simple rr case)
                  else
c- - - - - - - - - - phi(Za) matrix   (a = z)- - - - - - - - - - - - - - - - -
c                    <J''K''M''| phi Za |J'K'M'>
                     if ( polcal(3) .ne. 0 ) then
                        idj = iabs(DeltaJ) - DeltaJ
                        m = idj / 2
                        n = m + DeltaJ
                        do i = 1, 2*Jg+1-idj, 1
                           m = m + 1
                           n = n + 1
                           evr = evecgr(m+ipgv)*evecer(n+ipev)
                           ar = ar + evr*znt(m,n)
                        end do
                        ar = ar * polcal(3)
                     end if

c- - - - - - - - - - phi(Zb) matrix   (b = x)- - - - - - - - - - - - - - - - -
c                    <J''K''M''| phi Zb |J'K'M'>
                     if ( polcal(1) .ne. 0 ) then
                        ml = 1 - DeltaJ
                        nu = 1 + DeltaJ
                        do nl = 1, 2*Je-DeltaJ, 1
                           ml = ml + 1
                           nu = nu + 1
                           evrl = evecgr(ml+ipgv)*evecer(nl+ipev)
                           evru = evecgr(nl+ipgv)*evecer(nu+ipev)
                           br = br + evrl*xnt(ml,nl) + evru*xnt(nl,nu)
                        end do
                        br = br * polcal(1)
                     end if

c- - - - - - - - - - phi(Zc) matrix   (c = y)- - - - - - - - - - - - - - - - -
c                    <J''K''M''| phi Zc |J'K'M'>
                     if ( polcal(2) .ne. 0 ) then
                        ml = 1 - DeltaJ
                        nu = 1 + DeltaJ
                        do nl = 1, 2*Je-DeltaJ, 1
                           ml = ml + 1
                           nu = nu + 1
                           evrl = evecgr(ml+ipgv)*evecer(nl+ipev)
                           evru = evecgr(nl+ipgv)*evecer(nu+ipev)
                           ci = ci + evrl*ynt(ml,nl) + evru*ynt(nl,nu)
                        end do
                        ci = ci * polcal(2)
                     end if

                  end if

c/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

c                 calculate the total intensity
c                 total real intensity component
                  totinr = ar + br + cr
c                 total imaginary intensity component
                  totini = ai + bi + ci
c                 total intensity squared (= line strength factor)
                  rintsu = totinr*totinr + totini*totini
#ifdef DEBUG_VERBOSE
c                 store line strength factor
                  if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) )
     *                 lsfmat(ies,igs) = rintsu
#endif
c                 consider the statistical weight of nuclear spin
c                 Ka'' Kc''   bin.   dec.code
c                 even even   0  0      0
c                 even  odd   0  1      1
c                  odd even   1  0      2
c                  odd  odd   1  1      3

                  isymg = 2*mod( icqn( igs + ipgsqn, 2 ), 2 ) + mod( icqn( igs + ipgsqn, 3 ), 2 )
                  rintnu = rintsu * nuspsw(isymg)

c                 apply the Boltzman factor: Boltzman constant in MHz
c                 permit two-temperature model
                  if ( weight .ne. 0 ) then
                     rint = dexp( -(evalg(igs+ipgsqn)*tempf1) )
     *                      + weight * dexp( -(evalg(igs+ipgsqn)*tempf2) )
                  else
                     rint = dexp( -(evalg(igs+ipgsqn)*tempf1) )
                  end if
                  rint = rint * rintnu

c                 intensity calculation done for these two states for this type of transition
#ifdef DEBUG_VERBOSE
c                 fill calculated value into intensity matrix
                  if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) )
     *                 intmat(ies,igs) = rint
#endif

c                 intensity cutoff; save line arrays if |<..|phi Fg|..>| > cutint
                  if ( rint .gt. cutint ) then
c                    increment line counter by one
                     ntheli = ntheli + 1
c                    leave this subroutine if ntheli > maxnli
                     if ( ntheli .gt. maxnli ) then
                        lioflw = Jg
                        return
                     end if
c                    save excited and ground state qns for theoretical line set
                     do i = 1, 3, 1
                        lqn( ntheli, i+3 ) = icqn( igs+ipgsqn, i )
                        lqn( ntheli, i   ) = icqn( ies+ipesqn, i )
                     end do
#ifdef DEBUG_SUBBRANCH_OUTPUT
                     call lqnset( ntheli, Jmax, dKmax, lqn, lqn2, maxnli )
#endif
c                    actual intensity of this transition
                     intens(ntheli) = rint
c                    pure rotational energy difference
                     cfreq = evale(ies+ipesqn) - evalg(igs+ipgsqn)
c                    actual frequency: add nuzero to the above value
                     thefre(ntheli) = cfreq + nuzero

c                    save the derivative information
                     do i = 1, npar, 1
                        derle(ntheli,i) = derive(ies+ipesqn,i)
                        derlg(ntheli,i) = derivg(igs+ipgsqn,i)
                     end do
#ifdef DEBUG_VERBOSE
c                    phase debugging module
                     if ( shorti .eq. 0 ) then
                        write(iunout,'(6(i4),1x,8(1x,f6.3),1x,i7)')
     *                       (lqn(ntheli,i), i = 1,6), ai, bi, ci, ar, br, cr, totini, totinr, idnint(rintsu*1.d4)
                     end if
#endif
c                 { storage of line information }
                  end if
c              { tau' loop }
               end do
c           { type check }
            end if
c        { type loop }
         end do
c     { tau'' loop }
      end do

#ifdef DEBUG_VERBOSE
c     output complete line strength factor / intensity matrices
      if ( ( shorti .eq. 0 ) .and. ( Jg .le. Jmxout ) ) then
         call imout( lsfmat, dmham, Jg, Je, icqn, dmeval, sum, 0, cBran, polax1, lngbar, lbars )
         call imout( intmat, dmham, Jg, Je, icqn, dmeval, sum, 1, cBran, polax1, lngbar, lbars )
      end if
#endif

      return
      end


c------------------------------------------------------------------------------
      subroutine cmat0( cosmat, dmham, Jg, Je )
c     zero the cosine matrices for a single branch and given J

      implicit none

      integer        dmham
      integer        row, col, Jg, Je
      real*8         cosmat(dmham,dmham)

      ARNIROT_LAUNCH ( "Launching cmat0." )

      do col = 1, 2*Je+1, 1
         do row = 1, 2*Jg+1, 1
            cosmat( row, col ) = 0.d0
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine cmat02( cosmat, dmham, Jg )
c     zero the cosine matrices for all three branches of given J

      implicit none

      integer        dmham
      integer        row, col, Jg, Je, DeltaJ
      real*8         cosmat(dmham,dmham,-1:1)

      ARNIROT_LAUNCH ( "Launching cmat02." )

      do DeltaJ = -1, 1, 1
         Je = Jg + DeltaJ
         do col = 1, 2*Je+1, 1
            do row = 1, 2*Jg+1, 1
               cosmat( row, col, DeltaJ ) = 0.d0
            end do
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
#ifdef DEBUG_VERBOSE
      subroutine cmout( cosmat, dmham, Jg, Je, ipol, itype,
     *                  kind, cBran, polax1, ldash )
c     output the cosine matrices; designed for matrices up to J = 10

      implicit none

      integer        dmham
      integer        ipol, itype
      integer        row, col, Je, Jg, DeltaJ
      integer        strlen
      real*8         cosmat(dmham,dmham)
      character*1    cBran(-1:1), polax1(3)
      character*9    kind(0:1)
      character*81   ldash

      ARNIROT_LAUNCH ( "Launching cmout." )

      DeltaJ = Je - Jg
      write(*,'(a)') ldash
      write(*,'(''J'''' ='',i3,'' <-- J'''''''' ='',i3,'': '',a1,'' branch transitions '',a1,'' type'')')
     *     Je, Jg, cBran(DeltaJ), polax1(ipol)
      write(*,'(/,a,'' < J''''''''K''''''''M'''''''' | phi(Z'',a1,'') | J''''K''''M'''' > matrix:'',/)')
     *     kind(itype)(1:strlen(kind(itype))), polax1(ipol)

      do row = 1, 2*Jg+1, 1
         write(*,'(21(f9.5,2x))') (cosmat(row,col), col = 1, 2*Je+1, 1)
      end do

      return
      end
#endif


c------------------------------------------------------------------------------
#ifdef DEBUG_VERBOSE
      subroutine cmout2( cosmat, dmham, Jg, DeltaJ, ipol, itype,
     *                   kind, cBran, polax1, ldash )
c     output the cosine matrices; designed for matrices up to J = 10

      implicit none

      integer        dmham
      integer        ipol, itype
      integer        row, col, Jg, Je, DeltaJ
      integer        strlen
      real*8         cosmat(dmham,dmham,-1:1)
      character*1    cBran(-1:1), polax1(3)
      character*9    kind(0:1)
      character*81   ldash

      ARNIROT_LAUNCH ( "Launching cmout2." )

      Je = Jg + DeltaJ
      write(*,'(a)') ldash
      write(*,'(''J'''' ='',i3,'' <-- J'''''''' ='',i3,'': '',a1,'' branch transitions '',a1,'' type'')')
     *     Je, Jg, cBran(DeltaJ), polax1(ipol)
      write(*,'(/,a,'' < J''''''''K''''''''M'''''''' | phi(Z'',a1,'') | J''''K''''M'''' > matrix:'',/)')
     *     kind(itype)(1:strlen(kind(itype))), polax1(ipol)

      do row = 1, 2*Jg+1, 1
         write(*,'(21(f9.5,2x))') (cosmat(row,col,DeltaJ), col = 1, 2*Je+1, 1)
      end do

      return
      end
#endif


c------------------------------------------------------------------------------
#ifdef DEBUG_VERBOSE
      subroutine imout( mat, dmham, Jg, Je, icqn, dmeval, sum, switch,
     *                   cBran, polax1, lngbar, lbars )
c     matrix output depending on value of switch; designed for J <= 10
c     switch=0 :  line strength factor matrix
c     switch=1 :  intensity matrix

      implicit none

      integer        dmham, dmeval
      integer        i, k, l, m
      integer        ioffg, ioffe
      integer        Jg, Je, DeltaJ
      integer        icqn(dmeval,3)
      integer        switch
      real*8         mat(dmham,dmham), sum(dmham), totsum
      character*1    cBran(-1:1), polax1(3)
      character*81   lbars
      character*220  lngbar

      ARNIROT_LAUNCH ( "Launching imout." )

      DeltaJ = Je - Jg
      ioffg = Jg*Jg
      ioffe = Je*Je

c     initialize sum array
      totsum = 0.d0
      do k = 1, 2*Jg+1, 1
         sum(k) = 0.d0
         do i = 1, 2*Je+1, 1
            sum(k) = sum(k) + mat(i,k)
         end do
         totsum = totsum + sum(k)
      end do

      write(*,'(a)') lbars
      if ( switch .eq. 0 ) then
         write(*,'(''line strength factor matrix for '',a1,'' branch transitions J''''='',i2,'' <-- J''''''''='',i2,/)')
     *        cBran(DeltaJ), Je, Jg
      else
         write(*,'(''intensity matrix for '',a1,'' branch transitions J''''='',i2,'' <-- J''''''''='',i2,/)')
     *        cBran(DeltaJ), Je, Jg
      end if
      write(*,'(''|J''''Ka''''Kc''''>/ |J''''''''Ka''''''''Kc''''''''>'',/)')
      if (Jg.eq. 0) write(*,'(12x, 1(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 1) write(*,'(12x, 3(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 2) write(*,'(12x, 5(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 3) write(*,'(12x, 7(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 4) write(*,'(12x, 9(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 5) write(*,'(12x,11(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 6) write(*,'(12x,13(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 7) write(*,'(12x,15(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 8) write(*,'(12x,17(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 9) write(*,'(12x,19(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq.10) write(*,'(12x,21(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      write(*,'(11x,a)') lngbar(1:(2*Jg+1)*10-1)
      do l = 1, 2*Je+1, 1
         write(*,'(''|'',3(i2),''>  | '',21(f8.5,2x))') (icqn(ioffe+l,k), k = 1,3,1), (mat(l,m), m = 1, 2*Jg+1, 1)
      end do
      write(*,'(a)') lngbar(1:(2*Jg+2)*10)
      write(*,'(/,''Total'',7x,21(f8.5,2x))') (sum(k), k = 1, 2*Jg+1, 1)
      write(*,'(/,''sum over all elements = '',f9.5)') totsum
      if ( switch .eq. 0) then
         write(*,'(''(1/3)(2J''''''''+1)(2J''''+1)  = '',f9.5)') (2*Jg + 1)*(2*Je + 1)/3.d0
      end if

      return
      end
#endif


c------------------------------------------------------------------------------
#ifdef DEBUG_VERBOSE
      subroutine imout2( mat, dmham, Jg, DeltaJ, icqn, dmeval, sum, switch,
     *                   cBran, polax1, lngbar, lbars )
c     matrix output depending on value of switch; designed for J <= 10
c     switch=0 :  line strength factor matrix
c     switch=1 :  intensity matrix

      implicit none

      integer        dmham, dmeval
      integer        i, k, l, m
      integer        ioffg, ioffe
      integer        Jg, Je, DeltaJ
      integer        icqn(dmeval,3)
      integer        switch
      real*8         mat(dmham,dmham,-1:1), sum(dmham), totsum
      character*1    cBran(-1:1), polax1(3)
      character*81   lbars
      character*220  lngbar

      ARNIROT_LAUNCH ( "Launching imout2." )

      Je = Jg + DeltaJ
      ioffg = Jg*Jg
      ioffe = Je*Je

c     initialize sum array
      totsum = 0.d0
      do k = 1, 2*Jg+1, 1
         sum(k) = 0.d0
         do i = 1, 2*Je+1, 1
            sum(k) = sum(k) + mat(i,k,DeltaJ)
         end do
         totsum = totsum + sum(k)
      end do

      write(*,'(a)') lbars
      if ( switch .eq. 0 ) then
         write(*,'(''line strength factor matrix for '',a1,'' branch transitions J''''='',i2,'' <-- J''''''''='',i2,/)')
     *        cBran(DeltaJ), Je, Jg
      else
         write(*,'(''intensity matrix for '',a1,'' branch transitions J''''='',i2,'' <-- J''''''''='',i2,/)')
     *        cBran(DeltaJ), Je, Jg
      end if
      write(*,'(''|J''''Ka''''Kc''''>/ |J''''''''Ka''''''''Kc''''''''>'',/)')
      if (Jg.eq. 0) write(*,'(12x, 1(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 1) write(*,'(12x, 3(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 2) write(*,'(12x, 5(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 3) write(*,'(12x, 7(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 4) write(*,'(12x, 9(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 5) write(*,'(12x,11(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 6) write(*,'(12x,13(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 7) write(*,'(12x,15(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 8) write(*,'(12x,17(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq. 9) write(*,'(12x,19(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      if (Jg.eq.10) write(*,'(12x,21(''|'',3(i2),''>'',2x))') ((icqn(ioffg+i,k), k = 1,3,1), i = 1, 2*Jg+1, 1)
      write(*,'(11x,a)') lngbar(1:(2*Jg+1)*10-1)
      do l = 1, 2*Je+1, 1
         write(*,'(''|'',3(i2),''>  | '',21(f8.5,2x))') (icqn(ioffe+l,k), k = 1,3,1), (mat(l,m,DeltaJ), m = 1, 2*Jg+1, 1)
      end do
      write(*,'(a)') lngbar(1:(2*Jg+2)*10)
      write(*,'(/,''Total'',7x,21(f8.5,2x))') (sum(k), k = 1, 2*Jg+1, 1)
      write(*,'(/,''sum over all elements = '',f9.5)') totsum
      if ( switch .eq. 0 ) then
         write(*,'(''(1/3)(2J''''''''+1)(2J''''+1)  = '',f9.5)') (2*Jg + 1)*(2*Je + 1)/3.d0
      end if

      return
      end
#endif


c------------------------------------------------------------------------------
#ifdef DEBUG_SUBBRANCH_OUTPUT
c     fill lines information set lqn2
      subroutine lqnset( nl, Jmx, dKmx, lqn, lqn2, maxnli )

      implicit none

      integer        maxnli
      integer        nl
      integer        Jg, Je, Kag, Kcg, Jmx
      integer        dKa, dKc, dKmx
      integer        type
      integer        lqn(maxnli,6), lqn2(maxnli,6)

      ARNIROT_LAUNCH ( "Launching lqnset." )

      Jg  = lqn(nl,4)
      Je  = lqn(nl,1)
      Kag = lqn(nl,5)
      Kcg = lqn(nl,6)
      dKa = lqn(nl,2) - Kag
      dKc = lqn(nl,3) - Kcg

      type = mod(iabs(dKa),2) - mod(iabs(dKc),2) + 1

c     original assignment
c     lqn2(nl,1) = type
c     lqn2(nl,2) = Kag
c     lqn2(nl,3) = dKa + dKmx
c     lqn2(nl,4) = Jmx - Jg + Kcg
c     lqn2(nl,5) = Je - Jg + 1
c     lqn2(nl,6) = Kcg

c     new assignment Sep 10, 1998
      lqn2(nl,1) = type
      lqn2(nl,2) = Je - Jg + 1
      lqn2(nl,3) = dKa + dKmx
      lqn2(nl,4) = Kag
      lqn2(nl,5) = Jmx - Jg + Kcg
      lqn2(nl,6) = Kcg

      return
      end
#endif
