#include <qaccel.h>
#include <qkeycode.h>

#include "worksheet.h"

#include "ws_tabledata.h"

#include "ws_tabledata.moc"


// ***********************  Table Input Widget  *******************************


TableInput::TableInput (QWidget* parent, const char* name)
  : QLineEdit(parent,name)
  {
    // necessary for receiving Tab-Press-Events
    setFocusPolicy (QWidget::ClickFocus);

    setFixedSize (CELLWIDTH-1,CELLHEIGHT-1);
  }


void TableInput::keyPressEvent (QKeyEvent* e)
  { 
    QLineEdit::keyPressEvent(e);
  }


//
// ************************  The Main Table  *********************************
//


TableData::TableData( int numRows, int numCols, QWidget *parent, const char *name )
  : QTableView(parent,name)
  {
    setFocusPolicy( StrongFocus );		// we accept keyboard focus
    setBackgroundColor( KApplication::getKApplication()->windowColor);
                                                // set widgets background white
    setNumCols( numCols );			// set number of col's in table
    setNumRows( numRows );			// set number of rows in table
    setCellWidth( CELLWIDTH );			// set width of cell in pixels
    setCellHeight( CELLHEIGHT );		// set height of cell in pixels
    setTableFlags( Tbl_vScrollBar | Tbl_hScrollBar |
		   Tbl_clipCellPainting |	// avoid drawing outside cell
		   Tbl_snapToGrid |
    		   Tbl_cutCellsV |
    		   Tbl_cutCellsH);
      
    Focus = TRUE;

    vertScroll = verticalScrollBar ();
    horizScroll = horizontalScrollBar ();
    infield = new TableInput(this,"lineEdit");

    connect (vertScroll, SIGNAL (sliderMoved (int)), 
	     SLOT (topCellChanged(int)));
    connect (vertScroll, SIGNAL (valueChanged (int)), 
	     SLOT (topCellChanged (int)));

    connect (KApplication::getKApplication(), 
	     SIGNAL (kdisplayPaletteChanged()), 
	     SLOT (colorsChanged ()));

    connect (horizScroll, SIGNAL (sliderMoved (int)), 
	     SLOT (leftCellChanged(int)));
    connect (horizScroll, SIGNAL (valueChanged (int)), 
	     SLOT (leftCellChanged (int)));

    myworksheet = (Worksheet*) parent;

    // the column and row which should be drawn grey, not white
    showHighForPopupCol = -1;
    showHighForPopupRow = -1;
  }


/*
  Destructor: deallocates memory for contents
*/

TableData::~TableData()
  {
    delete infield;
  } 


void TableData::colorsChanged ()
{
  setBackgroundColor( KApplication::getKApplication()->windowColor);
}


// **************************  scrolling up and down  ************************


void TableData::scrollDown ()
  {
    setTopCell (lastRowVisible()+1);

    myworksheet->curRow = lastRowVisible();
    updateCell (myworksheet->curRow, myworksheet->curCol);

    emit (topCellHasChanged ( topCell() ) );    
  }


void TableData::scrollUp ()
  {
    if ( topCell() -(lastRowVisible()-topCell()) < 0 ) 
      {
	myworksheet->curRow = 0;
	setTopCell (0);
      }
    else 
      {
	myworksheet->curRow = topCell() -(lastRowVisible()-topCell());
	setTopCell (topCell() -(lastRowVisible()-topCell()) );
      }

    emit (topCellHasChanged ( topCell() ) );  
  }


// ***************************************************************************


void TableData::resizeTableData (int numRows, int numCols)
  {
    setNumRows (numRows);
    setNumCols (numCols);
  }


// the next signals will be emitted when a slider of the scrollbars moved
// this is not really connected with a change in cell position, but
// the TableView Class is intelligent enough to check this and only redraws
// anything if possible.

void TableData::topCellChanged (int)
  {
    emit ( topCellHasChanged ( topCell () ) );
  }


void TableData::leftCellChanged (int)
  {
    emit (leftCellHasChanged ( leftCell () ) );
  }


// ***************************************************************************


void TableData::showCurrentRowGrey ()
  {
    showHighForPopupRow = myworksheet->curRow;

    int i;

    for (i=leftCell (); i<=lastColVisible (); i++)
      updateCell (myworksheet->curRow, i);
  }  


void TableData::showCurrentRowNormal ()
  {
    showHighForPopupRow = -1;

    int i;

    for (i=leftCell (); i<=lastColVisible (); i++)
      updateCell (myworksheet->curRow, i);
  }


void TableData::showCurrentColumnGrey ()
  {
    showHighForPopupCol = myworksheet->curCol;

    int i;

    for (i=topCell (); i<=lastRowVisible (); i++)
      updateCell (i, myworksheet->curCol);
  }


void TableData::showCurrentColumnNormal ()
  {
    int i;

    showHighForPopupCol = -1;
    
    for (i=topCell (); i<=lastRowVisible (); i++)
      updateCell (i, myworksheet->curCol);
  }


// ****************************************************************************


/*
  Return content of cell
*/

const char* TableData::getCellContent( int row, int col ) const
  {
    return myworksheet->T->cell (col, row);
  }


/*
  Set content of cell
*/

void TableData::setCellContent( int row, int col, const char* c )
  {
    myworksheet->T->setCell (col, row, c);
  
    updateCell( row, col );
    return;
  }


void TableData::setFocus (bool f)
  {
    if ( (Focus == TRUE) && (f == FALSE) )
      {
	Focus = f;
	infield->hide ();
	updateCell( myworksheet->curRow, myworksheet->curCol );
      }
    else if ( (Focus == FALSE) && (f == TRUE) ) 
      {
	Focus = f;
	infield->show ();
	updateCell( myworksheet->curRow, myworksheet->curCol );
      }
  }


bool TableData::hasFocus ()
  {
    return (Focus);
  }


/*
  Handles cell painting for the Table widget.
*/

void TableData::paintCell( QPainter* p, int row, int col )
  {
    int w = cellWidth( col );			// width of cell in pixels
    int h = cellHeight( row );			// height of cell in pixels
    int x2 = w - 1;
    int y2 = h - 1;
    int x;
    int y;
    

    if ( (row==myworksheet->curRow && col==myworksheet->curCol ) 
      && Focus )
      {
	//
	// Active cell, LineEdit inside
	//
	p->drawLine( x2, 0, x2, y2 );
	p->drawLine( 0, y2, x2, y2 );
	if ( strcmp (getCellContent (row, col), "-") == 0 )
	  infield->setText ("");
	else infield->setText(getCellContent(row,col));
	//	infield->selectAll();
	
	if ( (colXPos (col, &x) == TRUE) && rowYPos (row, &y) )
	  {
	    infield->move(x ,y);
	    infield->show();
	  }
	  
	infield->setEnabled(TRUE);
      }
    else
      {
	//
	// Normal, inactive cell, no LineEdit inside
	//

	if ( (col == showHighForPopupCol) || (row == showHighForPopupRow) )
	  {
	    p->setPen (NoPen);
	    p->setBrush (KApplication::getKApplication()->backgroundColor);
	    p->drawRect (1, 1, x2, y2);
	    p->setPen (SolidLine);
	    p->setPen (KApplication::getKApplication()->selectTextColor);
	  }
	else if ( myworksheet->high->isHighlighted (row, col) == FALSE )
	  {
	    p->setPen (SolidLine);
	    p->setPen (KApplication::getKApplication()->windowTextColor);
	    p->setBrush (KApplication::getKApplication()->windowColor);
	    p->drawLine( x2, 0, x2, y2 );
	    p->drawLine( 0, y2, x2, y2 );
	  }
	else
	  {
	    p->setPen (NoPen);
	    p->setBrush (KApplication::getKApplication()->selectColor);
	    p->drawRect (1, 1, x2, y2);
	    p->setPen (SolidLine);
	    p->setPen (KApplication::getKApplication()->selectTextColor);	    
	  }
	p->drawText( 0, 0, w, h, AlignCenter, getCellContent(row,col));
      }
  }


/*
  Handles mouse press events for the Table widget.
  The current cell marker is set to the cell the mouse is clicked in.
*/

void TableData::mousePressEvent( QMouseEvent* e )
  {
    // emit a signal to get the active WS if mousepress inside
    if (activeWS != myworksheet)
      {
	emit (changeActiveWS ());
	cout << "activeWSChanged" << endl;
      }

    if ( e->button () == RightButton )
      return;

    QPoint clickedPos = e->pos();		    // extract pointer position

    if ( e->button () == LeftButton )
      {
	myworksheet->hideHighlighted ();

	myworksheet->high->setMouseStartPos (QPoint (findCol(clickedPos.x()), 
				         findRow(clickedPos.y())), "Tdata");
	myworksheet->high->setMouseEndPos (QPoint (findCol(clickedPos.x()), 
				   	   findRow(clickedPos.y()) ) ); 
	infield->hide ();
	Focus = FALSE;
	
	update (findRow(clickedPos.y()), findCol(clickedPos.x()));
	mouseOldPos = myworksheet->high->getMouseStartPos ();
      }
    
    if ( e->button () == MidButton )
      {
	myworksheet->curRow = findRow ( clickedPos.y() );
	myworksheet->curCol = findCol ( clickedPos.x() );

	myworksheet->copyFromClipboard ();
	myworksheet->hideHighlighted();
	repaint ();
      }
  }


void TableData::mouseReleaseEvent (QMouseEvent* e)
  {
    QPoint p = e->pos ();
    int x, y;

    if ( e->button () != LeftButton ) return;


    if ( p.x() >= (lastColVisible()-leftCell()+1)*CELLWIDTH )
      {
	setLeftCell (leftCell ()+1);
	myworksheet->Thead->changeLeftCell (leftCell ());
	x = lastColVisible ();
      }
    else if (p.x() < 0)
      {
	setLeftCell (leftCell ()-1);
	myworksheet->Thead->changeLeftCell (leftCell ());
	x = leftCell ();
      }
    else
      x = findCol (p.x());


    if ( p.y () >= (lastRowVisible()-topCell()+1)*CELLHEIGHT )
      {
	setTopCell (topCell ()+1);
	myworksheet->TfiCol->changeTopCell (topCell ());
	y = lastRowVisible ();
      }
    else if (p.y() < 0)
      {
	setTopCell (topCell ()-1);
	myworksheet->TfiCol->changeTopCell (topCell ());
	y = topCell ();
      }
    else
      y = findRow (p.y());


    myworksheet->high->setMouseEndPos (QPoint (x, y));


    if ( myworksheet->high->getMouseEndPos () == 
	 myworksheet->high->getMouseStartPos () )
      {
	// the same cell -> show the infield line else do nothing what
	// means to highlight the selected region
	myworksheet->high->setNoHighlight ();
	//	myworksheet->high->setMouseStartPos (QPoint (-2, -2), "Tdata");
	//	myworksheet->high->setMouseEndPos (QPoint (-2, -2));

	updateCell ( myworksheet->curRow, myworksheet->curCol );

	myworksheet->curRow = findRow( p.y() );
	myworksheet->curCol = findCol( p.x() );
    
	// this will insert the infield again
	Focus = TRUE;
	updateCell ( myworksheet->curRow, myworksheet->curCol );

	return;
      }	
    else 
      {
	//	for (i=mouseOldPos.x(); i<; i++);
      }

    if (AUTOCOPY) myworksheet->copyToClipboard ();
  }


void TableData::mouseMoveEvent (QMouseEvent* e)
  {
    QPoint p = e->pos ();
    QPoint a;
    int x, y, i, j;

    // only left button is interesting, he causes highlightning
    if ( myworksheet->high->isHighlighted () == FALSE) return;

    if ( p.x() >= (lastColVisible()-leftCell()+1)*CELLWIDTH )
      {
	setLeftCell (leftCell ()+1);
	myworksheet->Thead->changeLeftCell (leftCell ());
	x = lastColVisible ();
      }
    else if (p.x() < 0)
      {
	setLeftCell (leftCell ()-1);
	myworksheet->Thead->changeLeftCell (leftCell ());
	x = leftCell ();
      }
    else
      x = findCol (p.x());


    if ( p.y () >= (lastRowVisible()-topCell()+1)*CELLHEIGHT )
      {
	setTopCell (topCell ()+1);
	myworksheet->TfiCol->changeTopCell (topCell ());
	y = lastRowVisible ();
      }
    else if (p.y() < 0)
      {
	setTopCell (topCell ()-1);
	myworksheet->TfiCol->changeTopCell (topCell ());
	y = topCell ();
      }
    else
      y = findRow (p.y());


    myworksheet->high->setMouseEndPos (QPoint (x, y));

    a = QPoint (x,y);


    if ( (mouseOldPos.y() == myworksheet->high->getMouseEndPos().y()) &&
	 (mouseOldPos.x() == myworksheet->high->getMouseEndPos().x()) ) ;
    else
      {
	int luy = (a.y() < mouseOldPos.y() )? a.y() : mouseOldPos.y();
	int rly = (a.y () > mouseOldPos.y() )? a.y() : mouseOldPos.y();
	int lux = (a.x() < mouseOldPos.x() )? a.x() : mouseOldPos.x();
	int rlx = (a.x() > mouseOldPos.x() )? a.x() : mouseOldPos.x();

	if (myworksheet->high->getMouseStartPos().x() <= 
	    myworksheet->high->getMouseEndPos().x() ) 
	  {
	
	    if (myworksheet->high->getMouseStartPos().y() <=
		myworksheet->high->getMouseEndPos().y() )
	      {
		for (i=myworksheet->high->getMouseStartPos().x(); i<=rlx; i++)
		  for (j=luy; j<=rly; j++)
		    updateCell (j,i);

		for (i=lux; i<=rlx; i++)
		  for (j=myworksheet->high->getMouseStartPos().y();j<=rly; j++)
		    updateCell (j,i);
	      }
	    else
	      {
		for (i=myworksheet->high->getMouseStartPos().x(); i<=rlx; i++)
		  for (j=luy; j<=rly; j++)
		    updateCell (j,i);

		for (i=lux; i<=rlx; i++)
		  for (j=rly;j<=myworksheet->high->getMouseStartPos().y(); j++)
		    updateCell (j,i);
	      }

	  }
	else
	  {

	    if (myworksheet->high->getMouseStartPos().y() <=
		myworksheet->high->getMouseEndPos().y() )
	      {
		for (i=lux; i<=myworksheet->high->getMouseStartPos().x(); i++)
		  for (j=luy; j<=rly; j++)
		    updateCell (j,i);

		for (i=lux; i<=rlx; i++)
		  for (j=myworksheet->high->getMouseStartPos().y();j<=rly; j++)
		    updateCell (j,i);
	      }
	    else
	      {
		for (i=lux; i<=myworksheet->high->getMouseStartPos().x(); i++)
		  for (j=luy; j<=rly; j++)
		    updateCell (j,i);

		for (i=lux; i<=rlx; i++)
		  for (j=luy;j<=myworksheet->high->getMouseStartPos().y(); j++)
		    updateCell (j,i);
	      }

	  }
      }

    mouseOldPos = a;    

  }


void TableData::update (int row, int col)
  {
    updateCell (row, col);
  }


void TableData::paintEvent (QPaintEvent* e)
  {
    if (leftCell () > myworksheet->curCol ) infield->hide ();
    if (lastColVisible () < myworksheet->curCol ) infield->hide ();
    if (topCell () > myworksheet->curRow ) infield->hide ();
    if (lastRowVisible () < myworksheet->curRow ) infield->hide ();

    QTableView::paintEvent (e);
  }


void TableData::keyPressEvent( QKeyEvent* e )
  {
    int oldRow = myworksheet->curRow;
    int oldCol = myworksheet->curCol;
    int curRow = myworksheet->curRow;
    int curCol = myworksheet->curCol;
    int edge = 0;
    switch( e->key() ) 
      {
// 	case Key_Left:
// 	    if( curCol > 0 ) {
// 		curCol--;
// 		edge = leftCell();
// 		if ( curCol < edge )
// 		    setLeftCell( edge - 1 );
// 	    }
// 	    break; 
// 	case Key_Right:
// 	    if( curCol < numCols()-1 ) {
// 		curCol++;
// 		edge = lastColVisible();
// 		if ( curCol >= edge )
// 		    setLeftCell( leftCell() + 1 );
// 	    }
// 	    break;
      case Key_Tab:
	if (curCol < numCols()-1 )
	  {
	    setCellContent(curRow, curCol, infield->text());
	    curCol++;
	    edge = lastColVisible();
	    if ( curCol >= edge ) 
	      {
		setLeftCell( leftCell() + 1 );
	      }
	  }
	break;
      case Key_Backtab:    
	if ( curCol > 0 ) 
	  {
	    setCellContent(curRow, curCol,infield->text());
	    curCol--;
	    edge = leftCell();
	    if ( curCol < edge ) setLeftCell( edge - 1 );
	  }
	break;
      case Key_Up:
	if ( curRow > 0 ) 
	  {
	    //cerr << "UP" << endl;
	    setCellContent(curRow, curCol, infield->text());
	    curRow--;
	    edge = topCell();
	    if ( curRow < edge ) setTopCell( edge - 1 );
	  }
	break;
      case Key_Down:
	if ( curRow < numRows()-1 ) 
	  {
	    //cerr << "DN" << endl;
	    setCellContent(curRow,curCol,infield->text());
	    curRow++;
	    edge = lastRowVisible();
	    if ( curRow >= edge ) setTopCell( topCell() + 1 );
	  }
	break;
      case Key_Return:
	if ( curRow < numRows()-1 ) 
	  {
	    //cerr << "Return" << endl;
	    setCellContent(curRow,curCol,infield->text());
	    curRow++;
	    edge = lastRowVisible();
	    if ( curRow >= edge ) setTopCell( topCell() + 1 );
	  }
	break;

	//	default:
	//	  infield->sendKeyPressEvent(e);
	return;
      }

    // this must come before updateCell because in updateCell the row
    // and col will be compared with the settings in Worksheet
    myworksheet->curRow = curRow;
    myworksheet->curCol = curCol;

    if ( (curRow != oldRow) 			// if current cell has moved,
	 || (curCol != oldCol)  ) 
      {
	updateCell( oldRow, oldCol );		// erase previous marking
	updateCell( curRow, curCol );		// show new current cell
      }
    
  }


void TableData::changeTopCell (int row)
  {
    setTopCell (row);
  }


void TableData::changeLeftCell (int col)
  {
    setLeftCell (col);
  }


void TableData::returnInput()
  {
    int oldRow = myworksheet->curRow;
    int oldCol = myworksheet->curCol;
    int curRow = oldRow;
    int curCol = oldCol;
    int edge = 0;

    setCellContent ( curRow, curCol, infield->text () );

    //  updateCell( curRow, curCol );
    if( curRow < numRows()-1 ) {
      curRow++;
      edge = lastRowVisible();
      if ( curRow >= edge ) setTopCell( topCell() + 1 );
    }

    updateCell( oldRow, oldCol );
    updateCell( curRow, curCol );

    myworksheet->curCol = curCol;
    myworksheet->curRow = curRow;
  }



/*
  Handles focus reception events for the Table widget.
  Repaint only the current cell; to avoid flickering
*/

void TableData::focusInEvent( QFocusEvent* )
  {
    infield->setFocus ();                         // sehr wichtig

    //draw current cell
    updateCell( myworksheet->curRow, myworksheet->curCol );
  }    


/*
  Handles focus loss events for the Table widget.
  Repaint only the current cell; to avoid flickering
*/

void TableData::focusOutEvent( QFocusEvent* )
  {
    infield->clearFocus ();

    // draw current cell
    updateCell( myworksheet->curRow, myworksheet->curCol );
  }    



