/* Emacs, this is -*- C++ -*- */

#ifndef _DEBUG_H_
#define _DEBUG_H_


/** Query debug level for a module.
	Most likely, you want to use IFDEBUG() instead.

	This function simply returns the debug level for the given part.
	A program part definition is just a string of the form "program.module"
	or "program.class". See debSetLevel() for more details.

	@param part the module description, e.g. "korigin.centre".
	@return old debug level of the module.
	@see debSetLevel() for detailed description.
	@see IFDEBUG()
*/
int debLevel (const char* part);


/** Set debug level for a module.

	Debug Module Philosophy:

	With this debug module, you can define a debug level for
	arbitrary program parts, e.g. for classes. Suppose you have a
	class "MyClass" in a program called "MyProgram". You would
	set the debug level for this class on top of the main() function,
	using "debSetLevel("MyProg.MyClass",3);". In the classes member
	function, you would use "IFDEBUG("MyProg.MyClass",2) cout << "Hallo";
	ENDDEBUG;". This would print cout the word "Hallo" if the debug
	level for "MyProg.MyClass" is >=2. If it is less than 2, the output
	command is not executed.

	You should use different debug levels for different verbosity of
	your classes. A verbosity of zero is silence. A verbosity of 1 usually
	prints only severe warnings. Debug level 3 and above floods the console.

	@param part the module description, e.g. "korigin.centre".
	@param level the new debug level for the module.
	@return old debug level of the module.
	@see IFDEBUG()
*/
int debSetLevel (const char* part, int level);


/** Set all debug levels to zero.
	This disables debug output.

	@see debSetLevel() for detailed description.
*/
int debDefaults ();


/** Overture for debug conditional block.
	Statements between IFDEBUG() and ENDDEBUG; are executed only if the
	debug level of the "part" is higher than the givel "level".
	
	The conditional block *must* be terminated by "ENDDEBUG;". Do not
	omit the trailing semicolon, or Emacs indentation will be confused!

	@param part the module description, e.g. "korigin.centre".
	@param level the minimum debug level for the module to enter the 
	       conditional block.
	@see debLevel() is silently called by this macro.
	@see debSetLevel() for detailed description.
	@see ENDDEBUG to terminate conditional block.
*/
#define IFDEBUG(part,level) if(debLevel((part))>=(level)) {


/** Terminates debug conditional block.

	@see debSetLevel() for detailed description.
	@see IFDEBUG()
*/
#define ENDDEBUG            }


/*
  When doing production-level builds, one would use these definitions
  to get rid of all debug conditional blocks:

  #define IFDEBUG(part,level) if(0) {
  #define ENDDEBUG            }
*/


#endif  /* _DEBUG_H_ */

