#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>

#include "debug.h"
#include "column.h"



char* newStr (const char* string)
{
  char* new_string = new char [strlen(string)+1];
  strcpy(new_string,string);
  return new_string;
}


class ColumnBase
{
  friend class Column;

protected:
  ColumnBase ();
  virtual ~ColumnBase ();

  virtual const char* getValue (int) = 0;
  virtual void setValue (int, const char*) = 0;
  virtual double& directAccess (int) = 0;

protected:
  char* the_title;
  char* the_format;

private:
  int ref_count;
  int the_dimension;
};



class DoubleColumn : public ColumnBase
{
public:
  DoubleColumn ();
  ~DoubleColumn ();
  const char* getValue (int);
  void setValue (int, const char*);
  double& directAccess (int);
private:
  double** data;
  const int n_entries = 1024;
  const int n_banks = 1024;
  int bank_cnt;
};



class StringColumn : public ColumnBase
{
public:
  StringColumn ();
  ~StringColumn ();
  const char* getValue (int);
  void setValue (int, const char*);
  double& directAccess (int);
private:
  char*** data;
  const int n_entries = 1024;
  const int n_banks = 1024;
  int bank_cnt;
  double dummy;
};


//
// ColumnBase Implementation
//

ColumnBase::ColumnBase ()
{
  ref_count = 1;
  the_dimension = 0;
  the_format = 0;
  the_title = 0;
}

ColumnBase::~ColumnBase ()
{
  delete [] the_title;
  delete [] the_format;
  if (ref_count) {
	cout << "WARNING: column destroyed was still in use!" << endl;
  }
}


//
// DoubleColumn implementation
//

DoubleColumn::DoubleColumn ()
  : ColumnBase ()
{
  int i;
  the_format = new char [10];
  strcpy(the_format,"%.2lf");
  data = new double* [n_banks];
  for (i=0; i<n_banks; i++)
	data[i] = 0;
  bank_cnt = 0;
}

DoubleColumn::~DoubleColumn ()
{
  int i;
  for (i=0; i<bank_cnt; i++)
	delete [] data[i];
  delete [] data;
}

const char* DoubleColumn::getValue (int index)
{
  static char val [128];
  int bank = index / n_entries;
  int entry = index % n_entries;
  if (bank >= bank_cnt)
	strcpy(val,"-");
  else if (isnan(data[bank][entry]))
	strcpy(val,"-");
  else
	sprintf(val,the_format,data[bank][entry]);
  return val;
}

void DoubleColumn::setValue (int index, const char* val)
{
  double tmp;
  double* vec;
  char* end;
  int bank = index / n_entries;
  int entry = index % n_entries;
  int i;

  while (bank >= bank_cnt) {
	vec = data[bank_cnt] = new double [n_entries];
	for (i=0; i<n_entries; i++)
	  vec[i] = NAN;
	bank_cnt++;
  }
  tmp = strtod(val,&end);
  data[bank][entry] = tmp;
  if (val==0 || val==end || *val==0) data[bank][entry] = NAN;
  IFDEBUG("korigin.column",3)
	cout << "DoubleColumn::setValue("<<index<<",\""<<val<<"\") checks:" << endl
		 << "   val==0    " << (val==0?"y":"n") << endl
		 << "   val==end  " << (val==end?"y":"n") << endl
		 << "   *val==0   " << (*val==0?"y":"n") << endl
		 << "   value   = " << data[bank][entry] << endl;
  ENDDEBUG;
}

double& DoubleColumn::directAccess (int index)
{
  double* vec;
  int bank = index / n_entries;
  int entry = index % n_entries;
  int i;

  while (bank >= bank_cnt) {
	vec = data[bank_cnt] = new double [n_entries];
	for (i=0; i<n_entries; i++)
	  vec[i] = NAN;
	bank_cnt++;
  }
  return data [bank][entry];
}


//
// StringColumn implementation
//

StringColumn::StringColumn ()
  : ColumnBase ()
{
  int i;
  the_format = new char [10];
  strcpy(the_format,"%s");
  data = new char** [n_banks];
  for (i=0; i<n_banks; i++)
	data[i] = 0;
  bank_cnt = 0;
}

StringColumn::~StringColumn ()
{
  int i, j;
  for (i=0; i<bank_cnt; i++) {
	for (j=0; j<n_entries; j++)
	  delete [] data[i][j];
	delete [] data[i];
  }
  delete [] data;
}

const char* StringColumn::getValue (int index)
{
  static char val [128];
  int bank = index / n_entries;
  int entry = index % n_entries;
  if (bank >= bank_cnt)
	strcpy(val,"");
  else if (data[bank][entry] == 0)
	strcpy(val,"");
  else
	return data[bank][entry];
  return val;
}

void StringColumn::setValue (int index, const char* val)
{
  char** vec;
  int bank = index / n_entries;
  int entry = index % n_entries;
  int i;

  while (bank >= bank_cnt) {
	vec = data[bank_cnt] = new char* [n_entries];
	for (i=0; i<n_entries; i++)
	  vec[i] = 0;
	bank_cnt++;
  }
  delete [] data[bank][entry];
  data[bank][entry] = newStr(val);
}

double& StringColumn::directAccess (int)
{
  dummy = NAN;
  return dummy;
}


//
// Column implementation
//

ostream& operator<< (ostream& os, Column& column)
{
  switch (column.the_type) {
  case Column::columnEmpty:
	return os << "columnEmpty";
  case Column::columnString:
	os << "columnString,title=";
	break;
  case Column::columnDouble:
	os << "columnDouble,title=";
	break;
  default:
	return os << "frogshit=" << column.the_type;
  }
  if (column.column)
	return os << column.title();
  return os << "n/a";
}


Column::Column (ColumnType t)
{
  the_type = t;
  switch(the_type) {
  case columnEmpty:
	column = 0;
	break;
  case columnString:
	column = new StringColumn();
	break;
  case columnDouble:
	column = new DoubleColumn();
	break;
  }
  IFDEBUG("korigin.column",3)
	cout << "constructed column " << *this << " from scratch." << endl;
  ENDDEBUG;
}

Column::Column (const Column& rhs)
{
  the_type = rhs.the_type;
  column = rhs.column;
  ++(column->ref_count);
  IFDEBUG("korigin.column",3)
	cout << "constructed column " << *this << " from "
		 << *(const_cast<Column*>(&rhs)) << endl;
  ENDDEBUG;
}

Column::~Column ()
{
  IFDEBUG("korigin.column",3)
	cout << "destructing column " << *this << endl;
  ENDDEBUG;
  clear();
}

Column& Column::operator= (const Column& rhs)
{
  IFDEBUG("korigin.column",3)
	cout << "assign " << *(const_cast<Column*>(&rhs)) << " to " 
		 << *this << endl;
  ENDDEBUG;
  clear();
  the_type = rhs.the_type;
  column = rhs.column;
  if (column)
	++(column->ref_count);
  return *this;
}

void Column::clear ()
{
  IFDEBUG("korigin.column",3)
	cout << "clearing column " << *this << endl;
  ENDDEBUG;
  if (column) {
	if (--(column->ref_count) == 0) {
	  IFDEBUG("korigin.column",3)
		cout << "   really destructing " << column->the_title << endl;
	  ENDDEBUG;
	  delete column;
	  column = 0;
	}
  }
  the_type = columnEmpty;
}

Column::ColumnType Column::type ()
{
  return the_type;
}

int Column::dim ()
{
  if (column == 0) return 0;
  return column->the_dimension;
}

const char* Column::value (int index)
{
  if (column == 0) return 0;
  return column->getValue(index);
}

void Column::setValue (int index, const char* val)
{
  if (column == 0) return;
  if (val == 0) {
	if (index == column->the_dimension) column->the_dimension--;
	return;
  }
  if (index > column->the_dimension) column->the_dimension = index;
  column->setValue(index,val);
}

double& Column::operator[] (int index)
{
  if (column == 0) return (dummy=NAN);
  if (index > column->the_dimension) column->the_dimension = index;
  return column->directAccess(index);
}

const char* Column::title ()
{
  if (column == 0) return 0;
  return column->the_title;
}

void Column::setTitle (const char* t)
{
  if (column == 0) return;
  delete [] column->the_title;
  column->the_title = newStr(t);
}

const char* Column::format ()
{
  if (column == 0) return 0;
  return column->the_format;
}

void Column::setFormat (const char* f)
{
  if (column == 0) return;
  delete [] column->the_format;
  column->the_format = newStr(f);
}

